#
#
#           The Nim Compiler
#        (c) Copyright 2018 Andreas Rumpf
#
#    See the file "copying.txt", included in this
#    distribution, for details about the copyright.
#

## Plugin to transform a proc into an differently typed proc.

import ".." / [idents, ast, astalgo, semdata, msgs, lookups, renderer]
import intsets, tables

type
  Con = object
    symMap: Table[int, PSym]
    toAdapt: PSym
    stmtList: PNode
    ids: IntSet
    replacements: seq[(PIdent, PIdent)]

const runtimeRoutine = routineKinds - {skTemplate, skMacro}

proc isDependent(n: PNode; q: Con): bool =
  if n.kind == nkSym and n.sym.kind == skParam and n.sym.owner == q.toAdapt:
    return true
  for i in 0 ..< safeLen(n):
    if isDependent(n[i], q): return true
  return false

proc identReplace(ident: PIdent; q: Con): PIdent =
  if ident.id in q.ids:
    for r in q.replacements:
      if r[0].id == ident.id:
        return r[1]
  else:
    result = ident

proc dupNode(n: PNode): PNode =
  result = copyNode(n)
  result.typ = nil
  excl result.flags, nfSem

proc p(c: PContext; n: PNode; q: var Con): PNode =
  when false:
    case n.kind
    of nkNone..nkNilLit, nkTypeSection, nkProcDef, nkConverterDef, nkMethodDef,
        nkIteratorDef, nkMacroDef, nkTemplateDef, nkLambda, nkDo, nkFuncDef:
      # only transform imperative sections:
      result = copyTree(n)

  proc generateInstance(c: PContext; s: PSym; q: var Con): PSym =
    echo "generateInstance ", s.name.s
    result = copySym(s)
    q.symMap[s.id] = result
    let oldAdapt = q.toAdapt
    q.toAdapt = s
    result.ast = p(c, s.ast, q)
    q.toAdapt = oldAdapt

  case n.kind
  of nkCallKinds:
    # adapt n[0] if 'n' is a dependent expression:
    if n[0].kind == nkSym and n[0].sym.kind in runtimeRoutine and
        n[0].sym.magic == mNone and sfSystemModule notin n[0].sym.owner.flags and
        isDependent(n, q):
      let helper = n[0].sym
      var inst = q.symMap.getOrDefault(helper.id)
      if inst == nil:
        inst = generateInstance(c, helper, q)
      if q.stmtList == nil: q.stmtList = newTree(nkStmtList, inst.ast)
      else: q.stmtList = newTree(nkStmtList, inst.ast, q.stmtList)
    result = dupNode(n)
    for i in 0 ..< n.len:
      result.add p(c, n[i], q)

  of nkSym:
    result = newIdentNode(identReplace(n.sym.name, q), n.info)
    # XXX gensym'ed symbols get their ID to ensure correctness somewhat...
    #vm.evalMacroCall(c.module, c.graph, call, call, q.adapter)
  of nkOpenSymChoice, nkClosedSymChoice:
    result = newIdentNode(identReplace(n[0].sym.name, q), n.info)
  of nkIdent:
    result = newIdentNode(identReplace(n.ident, q), n.info)
  of nkProcDef, nkConverterDef, nkMethodDef,
     nkIteratorDef, nkMacroDef, nkFuncDef:
    # do not copy the crazy hidden 'result' node:
    result = dupNode(n)
    for i in 0 ..< min(resultPos, n.len):
      if i == pragmasPos: result.add newNodeI(nkEmpty, n[i].info)
      else: result.add p(c, n[i], q)
  of nkHiddenStdConv, nkHiddenSubConv:
    result = p(c, n[1], q)
  of nkConv:
    result = newNode(nkCall, n.info)
    result.add p(c, n[0], q)
    result.add p(c, n[1], q)
  of nkHiddenAddr, nkHiddenDeref, nkObjUpConv, nkObjDownConv, nkCheckedFieldExpr:
    result = p(c, n[0], q)
  of nkRange:
    result = newNode(nkInfix, n.info)
    result.add newIdentNode(c.cache.getIdent(".."), n.info)
    result.add p(c, n[0], q)
    result.add p(c, n[1], q)
  of nkReturnStmt:
    result = dupNode(n)
    if n[0].kind in {nkAsgn, nkFastAsgn}:
      result.add p(c, n[0][1], q)
    else:
      for i in 0 ..< n.safeLen:
        result.add p(c, n[i], q)
  else:
    result = dupNode(n)
    for i in 0 ..< n.safeLen:
      result.add p(c, n[i], q)

proc semAdapt*(c: PContext, n: PNode): PNode =
  # adapt(procEx, t)
  if n[1].kind != nkSym:
    localError(c.config, n.info, "adapter takes a proc sym")
    return n

  let toAdapt = n[1].sym
  if toAdapt.kind notin runtimeRoutine:
    localError(c.config, n.info, "adapter takes a proc sym")
    return n

  var q = Con(symMap: initTable[int, PSym](), toAdapt: toAdapt, stmtList: nil,
    ids: initIntSet(), replacements: @[])
  let r = n[2]
  if r.kind notin {nkTableConstr}:
    localError(c.config, n.info, "adapter takes a curly expression describing the replacements")
    return n
  for a in r:
    doAssert(a.kind == nkExprColonExpr)
    let key = considerQuotedIdent(c, a[0])
    let val = considerQuotedIdent(c, a[1])
    q.ids.incl key.id
    q.replacements.add((key, val))

  result = p(c, toAdapt.ast, q)
  if q.stmtList != nil:
    result = newTree(nkStmtList, q.stmtList, result)

  result = c.semExpr(c, result)

#[

proc recSort(a: openArray[int]) =
  if a[0] < a[1]: swap(a[0], a[1])

proc sort(a: openArray[int]) =
  recSort(toOpenArray(a, 0, 3))
  recSort(toOpenArray(a, 4, 9))

adapt(sort, (int), (string))

adapt(sort, openArray[string])

We call an expression *dependent* if it depends on an input parameter:

Examples:

  p
  p[i]
  p.foo

Real field accesses cannot produce an invalid instantiation.
Likewise, non-overloaded private symbols are never generic and need
to be duplicated.

]#
