/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/
#include <vdb/extern.h>

#include "v0-decompress.h"
#include "v0-decompress-local.h"
#include <sra/sradb.h> /* SRA_PLATFORM_... */
#include <klib/sort.h>
#include <sysalloc.h>

/* globals
 */
extern SRALocal     gv_local;
extern SRALookup_t  gp_lookup;

/* local prototypes
 */

static void _s_init_sig_454_default (void);
static void _s_init_prb_454_default (void);
static void _s_init_pos_454_default (void);

static void _s_init_seq_default     (void);

static void _s_init_sig_slx_default (void);
static void _s_init_int_slx_default (void);
static void _s_init_nse_slx_default (void);
static void _s_init_prb_slx_default (void);

static void _s_init_slx_convqs4to1_default (void);


/* initialize tables with default values
 */
void sra_decompress_init (void)
{ 
  memset (&gv_local, 0, sizeof(struct SRALocal));

  _s_init_sig_454_default (); 
  _s_init_prb_454_default ();
  _s_init_pos_454_default ();

  _s_init_seq_default     ();

  _s_init_sig_slx_default ();
  _s_init_int_slx_default ();
  _s_init_nse_slx_default ();
  _s_init_prb_slx_default ();

  _s_init_slx_convqs4to1_default ();
}   

/* local routines
 */
static int64_t CC _s_cmp_indexes (const void* a1, const void* a2, void *data)
{
  SRALookup_t plook = data;
  uint32_t bits1    = plook[*(uint16_t*)a1].bits;
  uint32_t bits2    = plook[*(uint16_t*)a2].bits;

  return (int64_t)bits1 - (int64_t)bits2;
}
/*
 */
static void _s_sort_lookup (const SRALookup_t plook, uint16_t* idx, uint16_t size)
{
  uint16_t i;
  for (i=0; i<size; i++) *(idx+i) = i;

  /* make sure it will sort a corect table
   */
  ksort ( idx, size, sizeof(uint16_t), _s_cmp_indexes, plook );
}
/*
 */
/*
 */
static void _s_init_slx_convqs4to1_default (void)
{
  gv_local.convqs4to1 [ 0] =  0;  gv_local.convqs4to1 [ 1] =  0;
  gv_local.convqs4to1 [ 2] =  0;  gv_local.convqs4to1 [ 3] =  0;
  gv_local.convqs4to1 [ 4] =  0;  gv_local.convqs4to1 [ 5] =  0;
  gv_local.convqs4to1 [ 6] =  0;  gv_local.convqs4to1 [ 7] =  0;
  gv_local.convqs4to1 [ 8] =  0;  gv_local.convqs4to1 [ 9] =  0;
  gv_local.convqs4to1 [10] =  0;  gv_local.convqs4to1 [11] =  0;
  gv_local.convqs4to1 [12] =  0;  gv_local.convqs4to1 [13] =  0;
  gv_local.convqs4to1 [14] =  0;  gv_local.convqs4to1 [15] =  0;
  gv_local.convqs4to1 [16] =  0;  gv_local.convqs4to1 [17] =  0;
  gv_local.convqs4to1 [18] =  0;  gv_local.convqs4to1 [19] =  0;
  gv_local.convqs4to1 [20] =  0;  gv_local.convqs4to1 [21] =  0;
  gv_local.convqs4to1 [22] =  0;  gv_local.convqs4to1 [23] =  0;
  gv_local.convqs4to1 [24] =  0;  gv_local.convqs4to1 [25] =  0;
  gv_local.convqs4to1 [26] =  0;  gv_local.convqs4to1 [27] =  0;
  gv_local.convqs4to1 [28] =  0;  gv_local.convqs4to1 [29] =  0;
  gv_local.convqs4to1 [30] =  0;  gv_local.convqs4to1 [31] =  1;
  gv_local.convqs4to1 [32] =  1;  gv_local.convqs4to1 [33] =  1;
  gv_local.convqs4to1 [34] =  1;  gv_local.convqs4to1 [35] =  1;
  gv_local.convqs4to1 [36] =  1;  gv_local.convqs4to1 [37] =  2;
  gv_local.convqs4to1 [38] =  2;  gv_local.convqs4to1 [39] =  3;
  gv_local.convqs4to1 [40] =  3;  gv_local.convqs4to1 [41] =  4;
  gv_local.convqs4to1 [42] =  4;  gv_local.convqs4to1 [43] =  5;
  gv_local.convqs4to1 [44] =  5;  gv_local.convqs4to1 [45] =  6;
  gv_local.convqs4to1 [46] =  7;  gv_local.convqs4to1 [47] =  8;
  gv_local.convqs4to1 [48] =  9;  gv_local.convqs4to1 [49] = 10;
  gv_local.convqs4to1 [50] = 10;  gv_local.convqs4to1 [51] = 11;
  gv_local.convqs4to1 [52] = 12;  gv_local.convqs4to1 [53] = 13;
  gv_local.convqs4to1 [54] = 14;  gv_local.convqs4to1 [55] = 15;
  gv_local.convqs4to1 [56] = 16;  gv_local.convqs4to1 [57] = 17;
  gv_local.convqs4to1 [58] = 18;  gv_local.convqs4to1 [59] = 19;
  gv_local.convqs4to1 [60] = 20;  gv_local.convqs4to1 [61] = 21;
  gv_local.convqs4to1 [62] = 22;  gv_local.convqs4to1 [63] = 23;
  gv_local.convqs4to1 [64] = 24;  gv_local.convqs4to1 [65] = 25;
  gv_local.convqs4to1 [66] = 26;  gv_local.convqs4to1 [67] = 27;
  gv_local.convqs4to1 [68] = 28;  gv_local.convqs4to1 [69] = 29;
  gv_local.convqs4to1 [70] = 30;  gv_local.convqs4to1 [71] = 31;
  gv_local.convqs4to1 [72] = 32;  gv_local.convqs4to1 [73] = 33;
  gv_local.convqs4to1 [74] = 34;  gv_local.convqs4to1 [75] = 35;
  gv_local.convqs4to1 [76] = 36;  gv_local.convqs4to1 [77] = 37;
  gv_local.convqs4to1 [78] = 38;  gv_local.convqs4to1 [79] = 39;
  gv_local.convqs4to1 [80] = 40;
}
/*
 */
static void _s_init_prb_slx_default (void)
{
#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  int      i, j;
  uchar_t  val;
#endif

  memset (gv_local.lookup_slx_prb, 0, sizeof(SRALookup)*PRB_SLX_HUFF_TREESIZE);

  /*
  // each bit is represented by two slashes or backslashes
  //
  // where / is 0
  //      /
  //
  // and  \  is 1
  //       \
  //             1                   0 - 0000
  //              \  /\              1 - 0100
  //               \/  \    6        4 - 1000
  //               /   /\  /         5 - 1010
  //              /   /  \/   14     6 - 1100
  //             0   /\   \  /      14 - 1110
  //                /  \   \/       15 - 1111
  //               4    5   \
  //                         \
  //                         15
  //
  */

  gv_local.lookup_slx_prb[0].nbits = 2; gv_local.lookup_slx_prb[0].bits = 0x00000000;  /* 0: m_qmax  m_qmin  m_qmin m_qmin */
  gv_local.lookup_slx_prb[1].nbits = 2; gv_local.lookup_slx_prb[1].bits = 0x40000000;  /* 1:     -5      -5      -5     -5 */
  gv_local.lookup_slx_prb[2].nbits = 3; gv_local.lookup_slx_prb[2].bits = 0x80000000;  /* 2:      A      -A  m_qmin m_qmin */
  gv_local.lookup_slx_prb[3].nbits = 3; gv_local.lookup_slx_prb[3].bits = 0xA0000000;  /* 3:      A  m_qmin      -A m_qmin */
  gv_local.lookup_slx_prb[4].nbits = 3; gv_local.lookup_slx_prb[4].bits = 0xC0000000;  /* 4:      A  m_qmin  m_qmin     -A */
  gv_local.lookup_slx_prb[5].nbits = 4; gv_local.lookup_slx_prb[5].bits = 0xE0000000;  /* 5: escape                        */
  gv_local.lookup_slx_prb[6].nbits = 4; gv_local.lookup_slx_prb[6].bits = 0xF0000000;  /* 6: UNUSED                        */

   _s_sort_lookup (gv_local.lookup_slx_prb, gv_local.idx_slx_prb, PRB_SLX_HUFF_TREESIZE);

#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  /* reverse huffman lookup table
   */
  memset (gv_local.rlookup_slx_prb, 0, sizeof(uchar_t)*REV_HUFF_TREESIZE_08);

  for (i=0; i<PRB_SLX_HUFF_TREESIZE; i++)
  {
    val = gv_local.lookup_slx_prb[i].bits >> 24;
    gv_local.rlookup_slx_prb [val] = (uchar_t) i;

    for (j=1; j<=(0xFF>>gv_local.lookup_slx_prb[i].nbits); j++)
      gv_local.rlookup_slx_prb [val+j] = (uchar_t) i;
  }
#endif

  gv_local.lookup_slx_prb_initialized = 1;
}
/*
 */
static void _s_init_sig_slx_default (void)
{
#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  int      i, j;
  uint16_t val;
#endif

  memset (gv_local.lookup_slx_sig,  0, sizeof(SRALookup)*SIG_HUFF_TREESIZE);

  gv_local.lookup_slx_sig[  0].nbits = 13; gv_local.lookup_slx_sig[  0].bits = 0xB5180000;
  gv_local.lookup_slx_sig[  1].nbits = 13; gv_local.lookup_slx_sig[  1].bits = 0x3D680000;
  gv_local.lookup_slx_sig[  2].nbits = 13; gv_local.lookup_slx_sig[  2].bits = 0xBAA00000;
  gv_local.lookup_slx_sig[  3].nbits = 13; gv_local.lookup_slx_sig[  3].bits = 0xB7D80000;
  gv_local.lookup_slx_sig[  4].nbits = 13; gv_local.lookup_slx_sig[  4].bits = 0x59500000;
  gv_local.lookup_slx_sig[  5].nbits = 13; gv_local.lookup_slx_sig[  5].bits = 0xB7080000;
  gv_local.lookup_slx_sig[  6].nbits = 13; gv_local.lookup_slx_sig[  6].bits = 0xB8180000;
  gv_local.lookup_slx_sig[  7].nbits = 13; gv_local.lookup_slx_sig[  7].bits = 0x8B980000;
  gv_local.lookup_slx_sig[  8].nbits = 13; gv_local.lookup_slx_sig[  8].bits = 0xB7880000;
  gv_local.lookup_slx_sig[  9].nbits = 13; gv_local.lookup_slx_sig[  9].bits = 0x5EE80000;
  gv_local.lookup_slx_sig[ 10].nbits = 13; gv_local.lookup_slx_sig[ 10].bits = 0xC0A00000;
  gv_local.lookup_slx_sig[ 11].nbits = 13; gv_local.lookup_slx_sig[ 11].bits = 0xB7980000;
  gv_local.lookup_slx_sig[ 12].nbits = 13; gv_local.lookup_slx_sig[ 12].bits = 0x8B880000;
  gv_local.lookup_slx_sig[ 13].nbits = 13; gv_local.lookup_slx_sig[ 13].bits = 0x8B900000;
  gv_local.lookup_slx_sig[ 14].nbits = 13; gv_local.lookup_slx_sig[ 14].bits = 0xB7900000;
  gv_local.lookup_slx_sig[ 15].nbits = 13; gv_local.lookup_slx_sig[ 15].bits = 0xC4400000;
  gv_local.lookup_slx_sig[ 16].nbits = 13; gv_local.lookup_slx_sig[ 16].bits = 0xBA480000;
  gv_local.lookup_slx_sig[ 17].nbits = 13; gv_local.lookup_slx_sig[ 17].bits = 0xC0A80000;
  gv_local.lookup_slx_sig[ 18].nbits = 13; gv_local.lookup_slx_sig[ 18].bits = 0xBD700000;
  gv_local.lookup_slx_sig[ 19].nbits = 13; gv_local.lookup_slx_sig[ 19].bits = 0xB8100000;
  gv_local.lookup_slx_sig[ 20].nbits = 13; gv_local.lookup_slx_sig[ 20].bits = 0xC1A00000;
  gv_local.lookup_slx_sig[ 21].nbits = 13; gv_local.lookup_slx_sig[ 21].bits = 0xBC700000;
  gv_local.lookup_slx_sig[ 22].nbits = 13; gv_local.lookup_slx_sig[ 22].bits = 0xC4500000;
  gv_local.lookup_slx_sig[ 23].nbits = 13; gv_local.lookup_slx_sig[ 23].bits = 0xC5200000;
  gv_local.lookup_slx_sig[ 24].nbits = 13; gv_local.lookup_slx_sig[ 24].bits = 0xC5780000;
  gv_local.lookup_slx_sig[ 25].nbits = 13; gv_local.lookup_slx_sig[ 25].bits = 0xC5380000;
  gv_local.lookup_slx_sig[ 26].nbits = 13; gv_local.lookup_slx_sig[ 26].bits = 0xC9A00000;
  gv_local.lookup_slx_sig[ 27].nbits = 13; gv_local.lookup_slx_sig[ 27].bits = 0xCBA00000;
  gv_local.lookup_slx_sig[ 28].nbits = 13; gv_local.lookup_slx_sig[ 28].bits = 0xC5300000;
  gv_local.lookup_slx_sig[ 29].nbits = 13; gv_local.lookup_slx_sig[ 29].bits = 0xC0B80000;
  gv_local.lookup_slx_sig[ 30].nbits = 13; gv_local.lookup_slx_sig[ 30].bits = 0xC5700000;
  gv_local.lookup_slx_sig[ 31].nbits = 13; gv_local.lookup_slx_sig[ 31].bits = 0xD6280000;
  gv_local.lookup_slx_sig[ 32].nbits = 13; gv_local.lookup_slx_sig[ 32].bits = 0xC4480000;
  gv_local.lookup_slx_sig[ 33].nbits = 13; gv_local.lookup_slx_sig[ 33].bits = 0xD6180000;
  gv_local.lookup_slx_sig[ 34].nbits = 13; gv_local.lookup_slx_sig[ 34].bits = 0xC9C00000;
  gv_local.lookup_slx_sig[ 35].nbits = 13; gv_local.lookup_slx_sig[ 35].bits = 0xCA680000;
  gv_local.lookup_slx_sig[ 36].nbits = 13; gv_local.lookup_slx_sig[ 36].bits = 0xCEF80000;
  gv_local.lookup_slx_sig[ 37].nbits = 13; gv_local.lookup_slx_sig[ 37].bits = 0xCCD00000;
  gv_local.lookup_slx_sig[ 38].nbits = 12; gv_local.lookup_slx_sig[ 38].bits = 0x1AA00000;
  gv_local.lookup_slx_sig[ 39].nbits = 12; gv_local.lookup_slx_sig[ 39].bits = 0x18800000;
  gv_local.lookup_slx_sig[ 40].nbits = 13; gv_local.lookup_slx_sig[ 40].bits = 0xD6300000;
  gv_local.lookup_slx_sig[ 41].nbits = 12; gv_local.lookup_slx_sig[ 41].bits = 0x18900000;
  gv_local.lookup_slx_sig[ 42].nbits = 13; gv_local.lookup_slx_sig[ 42].bits = 0xCBA80000;
  gv_local.lookup_slx_sig[ 43].nbits = 13; gv_local.lookup_slx_sig[ 43].bits = 0xD7E80000;
  gv_local.lookup_slx_sig[ 44].nbits = 13; gv_local.lookup_slx_sig[ 44].bits = 0xC9C80000;
  gv_local.lookup_slx_sig[ 45].nbits = 13; gv_local.lookup_slx_sig[ 45].bits = 0xD4F80000;
  gv_local.lookup_slx_sig[ 46].nbits = 12; gv_local.lookup_slx_sig[ 46].bits = 0x1AB00000;
  gv_local.lookup_slx_sig[ 47].nbits = 12; gv_local.lookup_slx_sig[ 47].bits = 0x2A800000;
  gv_local.lookup_slx_sig[ 48].nbits = 12; gv_local.lookup_slx_sig[ 48].bits = 0x2F100000;
  gv_local.lookup_slx_sig[ 49].nbits = 12; gv_local.lookup_slx_sig[ 49].bits = 0x1B600000;
  gv_local.lookup_slx_sig[ 50].nbits = 12; gv_local.lookup_slx_sig[ 50].bits = 0x1C500000;
  gv_local.lookup_slx_sig[ 51].nbits = 12; gv_local.lookup_slx_sig[ 51].bits = 0x29D00000;
  gv_local.lookup_slx_sig[ 52].nbits = 12; gv_local.lookup_slx_sig[ 52].bits = 0x2A200000;
  gv_local.lookup_slx_sig[ 53].nbits = 12; gv_local.lookup_slx_sig[ 53].bits = 0x39400000;
  gv_local.lookup_slx_sig[ 54].nbits = 12; gv_local.lookup_slx_sig[ 54].bits = 0x33100000;
  gv_local.lookup_slx_sig[ 55].nbits = 12; gv_local.lookup_slx_sig[ 55].bits = 0x29C00000;
  gv_local.lookup_slx_sig[ 56].nbits = 12; gv_local.lookup_slx_sig[ 56].bits = 0x28700000;
  gv_local.lookup_slx_sig[ 57].nbits = 12; gv_local.lookup_slx_sig[ 57].bits = 0x32200000;
  gv_local.lookup_slx_sig[ 58].nbits = 12; gv_local.lookup_slx_sig[ 58].bits = 0x39700000;
  gv_local.lookup_slx_sig[ 59].nbits = 12; gv_local.lookup_slx_sig[ 59].bits = 0x39500000;
  gv_local.lookup_slx_sig[ 60].nbits = 12; gv_local.lookup_slx_sig[ 60].bits = 0x32800000;
  gv_local.lookup_slx_sig[ 61].nbits = 12; gv_local.lookup_slx_sig[ 61].bits = 0x34300000;
  gv_local.lookup_slx_sig[ 62].nbits = 12; gv_local.lookup_slx_sig[ 62].bits = 0x32300000;
  gv_local.lookup_slx_sig[ 63].nbits = 12; gv_local.lookup_slx_sig[ 63].bits = 0x2A900000;
  gv_local.lookup_slx_sig[ 64].nbits = 12; gv_local.lookup_slx_sig[ 64].bits = 0x3FA00000;
  gv_local.lookup_slx_sig[ 65].nbits = 12; gv_local.lookup_slx_sig[ 65].bits = 0x39B00000;
  gv_local.lookup_slx_sig[ 66].nbits = 12; gv_local.lookup_slx_sig[ 66].bits = 0x3F900000;
  gv_local.lookup_slx_sig[ 67].nbits = 12; gv_local.lookup_slx_sig[ 67].bits = 0x3C300000;
  gv_local.lookup_slx_sig[ 68].nbits = 12; gv_local.lookup_slx_sig[ 68].bits = 0x5E300000;
  gv_local.lookup_slx_sig[ 69].nbits = 12; gv_local.lookup_slx_sig[ 69].bits = 0x3E800000;
  gv_local.lookup_slx_sig[ 70].nbits = 12; gv_local.lookup_slx_sig[ 70].bits = 0x39E00000;
  gv_local.lookup_slx_sig[ 71].nbits = 12; gv_local.lookup_slx_sig[ 71].bits = 0x3E500000;
  gv_local.lookup_slx_sig[ 72].nbits = 12; gv_local.lookup_slx_sig[ 72].bits = 0x88F00000;
  gv_local.lookup_slx_sig[ 73].nbits = 12; gv_local.lookup_slx_sig[ 73].bits = 0x88000000;
  gv_local.lookup_slx_sig[ 74].nbits = 12; gv_local.lookup_slx_sig[ 74].bits = 0x88D00000;
  gv_local.lookup_slx_sig[ 75].nbits = 12; gv_local.lookup_slx_sig[ 75].bits = 0x5EB00000;
  gv_local.lookup_slx_sig[ 76].nbits = 12; gv_local.lookup_slx_sig[ 76].bits = 0x3FB00000;
  gv_local.lookup_slx_sig[ 77].nbits = 12; gv_local.lookup_slx_sig[ 77].bits = 0x59100000;
  gv_local.lookup_slx_sig[ 78].nbits = 12; gv_local.lookup_slx_sig[ 78].bits = 0x88C00000;
  gv_local.lookup_slx_sig[ 79].nbits = 12; gv_local.lookup_slx_sig[ 79].bits = 0x88E00000;
  gv_local.lookup_slx_sig[ 80].nbits = 12; gv_local.lookup_slx_sig[ 80].bits = 0x88100000;
  gv_local.lookup_slx_sig[ 81].nbits = 12; gv_local.lookup_slx_sig[ 81].bits = 0xB6800000;
  gv_local.lookup_slx_sig[ 82].nbits = 12; gv_local.lookup_slx_sig[ 82].bits = 0xB5200000;
  gv_local.lookup_slx_sig[ 83].nbits = 12; gv_local.lookup_slx_sig[ 83].bits = 0x5E200000;
  gv_local.lookup_slx_sig[ 84].nbits = 12; gv_local.lookup_slx_sig[ 84].bits = 0xB7C00000;
  gv_local.lookup_slx_sig[ 85].nbits = 12; gv_local.lookup_slx_sig[ 85].bits = 0xB8000000;
  gv_local.lookup_slx_sig[ 86].nbits = 12; gv_local.lookup_slx_sig[ 86].bits = 0xB9200000;
  gv_local.lookup_slx_sig[ 87].nbits = 12; gv_local.lookup_slx_sig[ 87].bits = 0xB7300000;
  gv_local.lookup_slx_sig[ 88].nbits = 12; gv_local.lookup_slx_sig[ 88].bits = 0xC1100000;
  gv_local.lookup_slx_sig[ 89].nbits = 12; gv_local.lookup_slx_sig[ 89].bits = 0xBA500000;
  gv_local.lookup_slx_sig[ 90].nbits = 12; gv_local.lookup_slx_sig[ 90].bits = 0xC0900000;
  gv_local.lookup_slx_sig[ 91].nbits = 12; gv_local.lookup_slx_sig[ 91].bits = 0xC1900000;
  gv_local.lookup_slx_sig[ 92].nbits = 12; gv_local.lookup_slx_sig[ 92].bits = 0xB8500000;
  gv_local.lookup_slx_sig[ 93].nbits = 12; gv_local.lookup_slx_sig[ 93].bits = 0xC1800000;
  gv_local.lookup_slx_sig[ 94].nbits = 12; gv_local.lookup_slx_sig[ 94].bits = 0xBAE00000;
  gv_local.lookup_slx_sig[ 95].nbits = 12; gv_local.lookup_slx_sig[ 95].bits = 0xC9B00000;
  gv_local.lookup_slx_sig[ 96].nbits = 12; gv_local.lookup_slx_sig[ 96].bits = 0xC0800000;
  gv_local.lookup_slx_sig[ 97].nbits = 12; gv_local.lookup_slx_sig[ 97].bits = 0xC5D00000;
  gv_local.lookup_slx_sig[ 98].nbits = 12; gv_local.lookup_slx_sig[ 98].bits = 0xC1000000;
  gv_local.lookup_slx_sig[ 99].nbits = 12; gv_local.lookup_slx_sig[ 99].bits = 0xC0500000;
  gv_local.lookup_slx_sig[100].nbits = 12; gv_local.lookup_slx_sig[100].bits = 0xC8100000;
  gv_local.lookup_slx_sig[101].nbits = 12; gv_local.lookup_slx_sig[101].bits = 0xC1C00000;
  gv_local.lookup_slx_sig[102].nbits = 12; gv_local.lookup_slx_sig[102].bits = 0xC8600000;
  gv_local.lookup_slx_sig[103].nbits = 12; gv_local.lookup_slx_sig[103].bits = 0xCED00000;
  gv_local.lookup_slx_sig[104].nbits = 12; gv_local.lookup_slx_sig[104].bits = 0xCC900000;
  gv_local.lookup_slx_sig[105].nbits = 12; gv_local.lookup_slx_sig[105].bits = 0xCE400000;
  gv_local.lookup_slx_sig[106].nbits = 12; gv_local.lookup_slx_sig[106].bits = 0xCC500000;
  gv_local.lookup_slx_sig[107].nbits = 12; gv_local.lookup_slx_sig[107].bits = 0xCC400000;
  gv_local.lookup_slx_sig[108].nbits = 12; gv_local.lookup_slx_sig[108].bits = 0xCBB00000;
  gv_local.lookup_slx_sig[109].nbits = 12; gv_local.lookup_slx_sig[109].bits = 0xCA700000;
  gv_local.lookup_slx_sig[110].nbits = 12; gv_local.lookup_slx_sig[110].bits = 0xD6000000;
  gv_local.lookup_slx_sig[111].nbits = 12; gv_local.lookup_slx_sig[111].bits = 0xD5D00000;
  gv_local.lookup_slx_sig[112].nbits = 12; gv_local.lookup_slx_sig[112].bits = 0xD4E00000;
  gv_local.lookup_slx_sig[113].nbits = 12; gv_local.lookup_slx_sig[113].bits = 0xD5C00000;
  gv_local.lookup_slx_sig[114].nbits = 12; gv_local.lookup_slx_sig[114].bits = 0xC9E00000;
  gv_local.lookup_slx_sig[115].nbits = 12; gv_local.lookup_slx_sig[115].bits = 0xD5700000;
  gv_local.lookup_slx_sig[116].nbits = 11; gv_local.lookup_slx_sig[116].bits = 0x1B400000;
  gv_local.lookup_slx_sig[117].nbits = 12; gv_local.lookup_slx_sig[117].bits = 0xD7F00000;
  gv_local.lookup_slx_sig[118].nbits = 11; gv_local.lookup_slx_sig[118].bits = 0x1CC00000;
  gv_local.lookup_slx_sig[119].nbits = 11; gv_local.lookup_slx_sig[119].bits = 0x18A00000;
  gv_local.lookup_slx_sig[120].nbits = 11; gv_local.lookup_slx_sig[120].bits = 0x19000000;
  gv_local.lookup_slx_sig[121].nbits = 11; gv_local.lookup_slx_sig[121].bits = 0x1C600000;
  gv_local.lookup_slx_sig[122].nbits = 11; gv_local.lookup_slx_sig[122].bits = 0x1E000000;
  gv_local.lookup_slx_sig[123].nbits = 11; gv_local.lookup_slx_sig[123].bits = 0x1E200000;
  gv_local.lookup_slx_sig[124].nbits = 11; gv_local.lookup_slx_sig[124].bits = 0x32A00000;
  gv_local.lookup_slx_sig[125].nbits = 11; gv_local.lookup_slx_sig[125].bits = 0x32E00000;
  gv_local.lookup_slx_sig[126].nbits = 11; gv_local.lookup_slx_sig[126].bits = 0x33400000;
  gv_local.lookup_slx_sig[127].nbits = 11; gv_local.lookup_slx_sig[127].bits = 0x2E200000;
  gv_local.lookup_slx_sig[128].nbits = 11; gv_local.lookup_slx_sig[128].bits = 0x2F200000;
  gv_local.lookup_slx_sig[129].nbits = 11; gv_local.lookup_slx_sig[129].bits = 0x2E400000;
  gv_local.lookup_slx_sig[130].nbits = 11; gv_local.lookup_slx_sig[130].bits = 0x2F400000;
  gv_local.lookup_slx_sig[131].nbits = 11; gv_local.lookup_slx_sig[131].bits = 0x35A00000;
  gv_local.lookup_slx_sig[132].nbits = 11; gv_local.lookup_slx_sig[132].bits = 0x3B400000;
  gv_local.lookup_slx_sig[133].nbits = 11; gv_local.lookup_slx_sig[133].bits = 0x33200000;
  gv_local.lookup_slx_sig[134].nbits = 11; gv_local.lookup_slx_sig[134].bits = 0x34A00000;
  gv_local.lookup_slx_sig[135].nbits = 11; gv_local.lookup_slx_sig[135].bits = 0x33A00000;
  gv_local.lookup_slx_sig[136].nbits = 11; gv_local.lookup_slx_sig[136].bits = 0x35C00000;
  gv_local.lookup_slx_sig[137].nbits = 11; gv_local.lookup_slx_sig[137].bits = 0x3DA00000;
  gv_local.lookup_slx_sig[138].nbits = 11; gv_local.lookup_slx_sig[138].bits = 0x39800000;
  gv_local.lookup_slx_sig[139].nbits = 11; gv_local.lookup_slx_sig[139].bits = 0x3E600000;
  gv_local.lookup_slx_sig[140].nbits = 11; gv_local.lookup_slx_sig[140].bits = 0x3F000000;
  gv_local.lookup_slx_sig[141].nbits = 11; gv_local.lookup_slx_sig[141].bits = 0x3D000000;
  gv_local.lookup_slx_sig[142].nbits = 11; gv_local.lookup_slx_sig[142].bits = 0x59200000;
  gv_local.lookup_slx_sig[143].nbits = 11; gv_local.lookup_slx_sig[143].bits = 0x3D400000;
  gv_local.lookup_slx_sig[144].nbits = 11; gv_local.lookup_slx_sig[144].bits = 0x88200000;
  gv_local.lookup_slx_sig[145].nbits = 11; gv_local.lookup_slx_sig[145].bits = 0x5E000000;
  gv_local.lookup_slx_sig[146].nbits = 11; gv_local.lookup_slx_sig[146].bits = 0x5E800000;
  gv_local.lookup_slx_sig[147].nbits = 11; gv_local.lookup_slx_sig[147].bits = 0x8BA00000;
  gv_local.lookup_slx_sig[148].nbits = 11; gv_local.lookup_slx_sig[148].bits = 0x5EC00000;
  gv_local.lookup_slx_sig[149].nbits = 11; gv_local.lookup_slx_sig[149].bits = 0x5F400000;
  gv_local.lookup_slx_sig[150].nbits = 11; gv_local.lookup_slx_sig[150].bits = 0x5E600000;
  gv_local.lookup_slx_sig[151].nbits = 11; gv_local.lookup_slx_sig[151].bits = 0x5F600000;
  gv_local.lookup_slx_sig[152].nbits = 11; gv_local.lookup_slx_sig[152].bits = 0xB6C00000;
  gv_local.lookup_slx_sig[153].nbits = 11; gv_local.lookup_slx_sig[153].bits = 0xB6400000;
  gv_local.lookup_slx_sig[154].nbits = 11; gv_local.lookup_slx_sig[154].bits = 0xB8200000;
  gv_local.lookup_slx_sig[155].nbits = 11; gv_local.lookup_slx_sig[155].bits = 0xB6600000;
  gv_local.lookup_slx_sig[156].nbits = 11; gv_local.lookup_slx_sig[156].bits = 0xBAC00000;
  gv_local.lookup_slx_sig[157].nbits = 11; gv_local.lookup_slx_sig[157].bits = 0xBA600000;
  gv_local.lookup_slx_sig[158].nbits = 11; gv_local.lookup_slx_sig[158].bits = 0xBCA00000;
  gv_local.lookup_slx_sig[159].nbits = 11; gv_local.lookup_slx_sig[159].bits = 0xC0600000;
  gv_local.lookup_slx_sig[160].nbits = 11; gv_local.lookup_slx_sig[160].bits = 0xC0200000;
  gv_local.lookup_slx_sig[161].nbits = 11; gv_local.lookup_slx_sig[161].bits = 0xBD400000;
  gv_local.lookup_slx_sig[162].nbits = 11; gv_local.lookup_slx_sig[162].bits = 0xC1200000;
  gv_local.lookup_slx_sig[163].nbits = 11; gv_local.lookup_slx_sig[163].bits = 0xBBC00000;
  gv_local.lookup_slx_sig[164].nbits = 11; gv_local.lookup_slx_sig[164].bits = 0xC1600000;
  gv_local.lookup_slx_sig[165].nbits = 11; gv_local.lookup_slx_sig[165].bits = 0xC0000000;
  gv_local.lookup_slx_sig[166].nbits = 11; gv_local.lookup_slx_sig[166].bits = 0xC5000000;
  gv_local.lookup_slx_sig[167].nbits = 11; gv_local.lookup_slx_sig[167].bits = 0xC4600000;
  gv_local.lookup_slx_sig[168].nbits = 11; gv_local.lookup_slx_sig[168].bits = 0xBCE00000;
  gv_local.lookup_slx_sig[169].nbits = 11; gv_local.lookup_slx_sig[169].bits = 0xC0E00000;
  gv_local.lookup_slx_sig[170].nbits = 11; gv_local.lookup_slx_sig[170].bits = 0xC5E00000;
  gv_local.lookup_slx_sig[171].nbits = 11; gv_local.lookup_slx_sig[171].bits = 0xC9600000;
  gv_local.lookup_slx_sig[172].nbits = 11; gv_local.lookup_slx_sig[172].bits = 0xC8400000;
  gv_local.lookup_slx_sig[173].nbits = 11; gv_local.lookup_slx_sig[173].bits = 0xC9800000;
  gv_local.lookup_slx_sig[174].nbits = 11; gv_local.lookup_slx_sig[174].bits = 0xCC600000;
  gv_local.lookup_slx_sig[175].nbits = 11; gv_local.lookup_slx_sig[175].bits = 0xCB000000;
  gv_local.lookup_slx_sig[176].nbits = 11; gv_local.lookup_slx_sig[176].bits = 0xCA400000;
  gv_local.lookup_slx_sig[177].nbits = 11; gv_local.lookup_slx_sig[177].bits = 0xD4C00000;
  gv_local.lookup_slx_sig[178].nbits = 11; gv_local.lookup_slx_sig[178].bits = 0xCCA00000;
  gv_local.lookup_slx_sig[179].nbits = 11; gv_local.lookup_slx_sig[179].bits = 0xD5400000;
  gv_local.lookup_slx_sig[180].nbits = 11; gv_local.lookup_slx_sig[180].bits = 0xD4800000;
  gv_local.lookup_slx_sig[181].nbits = 11; gv_local.lookup_slx_sig[181].bits = 0xD5200000;
  gv_local.lookup_slx_sig[182].nbits = 11; gv_local.lookup_slx_sig[182].bits = 0xD7600000;
  gv_local.lookup_slx_sig[183].nbits = 10; gv_local.lookup_slx_sig[183].bits = 0x18400000;
  gv_local.lookup_slx_sig[184].nbits = 11; gv_local.lookup_slx_sig[184].bits = 0xD5800000;
  gv_local.lookup_slx_sig[185].nbits = 11; gv_local.lookup_slx_sig[185].bits = 0xD5E00000;
  gv_local.lookup_slx_sig[186].nbits = 11; gv_local.lookup_slx_sig[186].bits = 0xD7C00000;
  gv_local.lookup_slx_sig[187].nbits = 10; gv_local.lookup_slx_sig[187].bits = 0x1C000000;
  gv_local.lookup_slx_sig[188].nbits = 10; gv_local.lookup_slx_sig[188].bits = 0x18C00000;
  gv_local.lookup_slx_sig[189].nbits = 10; gv_local.lookup_slx_sig[189].bits = 0x1A400000;
  gv_local.lookup_slx_sig[190].nbits = 10; gv_local.lookup_slx_sig[190].bits = 0x1B000000;
  gv_local.lookup_slx_sig[191].nbits = 10; gv_local.lookup_slx_sig[191].bits = 0x28000000;
  gv_local.lookup_slx_sig[192].nbits = 10; gv_local.lookup_slx_sig[192].bits = 0x29400000;
  gv_local.lookup_slx_sig[193].nbits = 10; gv_local.lookup_slx_sig[193].bits = 0x29000000;
  gv_local.lookup_slx_sig[194].nbits = 10; gv_local.lookup_slx_sig[194].bits = 0x1F400000;
  gv_local.lookup_slx_sig[195].nbits = 10; gv_local.lookup_slx_sig[195].bits = 0x2EC00000;
  gv_local.lookup_slx_sig[196].nbits = 10; gv_local.lookup_slx_sig[196].bits = 0x1F000000;
  gv_local.lookup_slx_sig[197].nbits = 10; gv_local.lookup_slx_sig[197].bits = 0x29800000;
  gv_local.lookup_slx_sig[198].nbits = 10; gv_local.lookup_slx_sig[198].bits = 0x32400000;
  gv_local.lookup_slx_sig[199].nbits = 10; gv_local.lookup_slx_sig[199].bits = 0x33C00000;
  gv_local.lookup_slx_sig[200].nbits = 10; gv_local.lookup_slx_sig[200].bits = 0x2E800000;
  gv_local.lookup_slx_sig[201].nbits = 10; gv_local.lookup_slx_sig[201].bits = 0x34C00000;
  gv_local.lookup_slx_sig[202].nbits = 10; gv_local.lookup_slx_sig[202].bits = 0x37C00000;
  gv_local.lookup_slx_sig[203].nbits = 10; gv_local.lookup_slx_sig[203].bits = 0x37800000;
  gv_local.lookup_slx_sig[204].nbits = 10; gv_local.lookup_slx_sig[204].bits = 0x37400000;
  gv_local.lookup_slx_sig[205].nbits = 10; gv_local.lookup_slx_sig[205].bits = 0x39000000;
  gv_local.lookup_slx_sig[206].nbits = 10; gv_local.lookup_slx_sig[206].bits = 0x3C400000;
  gv_local.lookup_slx_sig[207].nbits = 10; gv_local.lookup_slx_sig[207].bits = 0x3FC00000;
  gv_local.lookup_slx_sig[208].nbits = 10; gv_local.lookup_slx_sig[208].bits = 0x58400000;
  gv_local.lookup_slx_sig[209].nbits = 10; gv_local.lookup_slx_sig[209].bits = 0x3EC00000;
  gv_local.lookup_slx_sig[210].nbits = 10; gv_local.lookup_slx_sig[210].bits = 0x3E000000;
  gv_local.lookup_slx_sig[211].nbits = 10; gv_local.lookup_slx_sig[211].bits = 0x3F400000;
  gv_local.lookup_slx_sig[212].nbits = 10; gv_local.lookup_slx_sig[212].bits = 0x59800000;
  gv_local.lookup_slx_sig[213].nbits = 10; gv_local.lookup_slx_sig[213].bits = 0x59C00000;
  gv_local.lookup_slx_sig[214].nbits = 10; gv_local.lookup_slx_sig[214].bits = 0x8B400000;
  gv_local.lookup_slx_sig[215].nbits = 10; gv_local.lookup_slx_sig[215].bits = 0x88400000;
  gv_local.lookup_slx_sig[216].nbits = 10; gv_local.lookup_slx_sig[216].bits = 0xB4000000;
  gv_local.lookup_slx_sig[217].nbits = 10; gv_local.lookup_slx_sig[217].bits = 0xB6000000;
  gv_local.lookup_slx_sig[218].nbits = 10; gv_local.lookup_slx_sig[218].bits = 0xB9800000;
  gv_local.lookup_slx_sig[219].nbits = 10; gv_local.lookup_slx_sig[219].bits = 0xB9C00000;
  gv_local.lookup_slx_sig[220].nbits = 10; gv_local.lookup_slx_sig[220].bits = 0xBC000000;
  gv_local.lookup_slx_sig[221].nbits = 10; gv_local.lookup_slx_sig[221].bits = 0xBDC00000;
  gv_local.lookup_slx_sig[222].nbits = 10; gv_local.lookup_slx_sig[222].bits = 0xBD000000;
  gv_local.lookup_slx_sig[223].nbits = 10; gv_local.lookup_slx_sig[223].bits = 0xC9000000;
  gv_local.lookup_slx_sig[224].nbits = 10; gv_local.lookup_slx_sig[224].bits = 0xC5800000;
  gv_local.lookup_slx_sig[225].nbits = 10; gv_local.lookup_slx_sig[225].bits = 0xCA000000;
  gv_local.lookup_slx_sig[226].nbits = 10; gv_local.lookup_slx_sig[226].bits = 0xCB400000;
  gv_local.lookup_slx_sig[227].nbits = 10; gv_local.lookup_slx_sig[227].bits = 0xCE000000;
  gv_local.lookup_slx_sig[228].nbits = 10; gv_local.lookup_slx_sig[228].bits = 0xD7000000;
  gv_local.lookup_slx_sig[229].nbits =  9; gv_local.lookup_slx_sig[229].bits = 0x1B800000;
  gv_local.lookup_slx_sig[230].nbits =  9; gv_local.lookup_slx_sig[230].bits = 0x28800000;
  gv_local.lookup_slx_sig[231].nbits =  9; gv_local.lookup_slx_sig[231].bits = 0x2F800000;
  gv_local.lookup_slx_sig[232].nbits =  9; gv_local.lookup_slx_sig[232].bits = 0x36000000;
  gv_local.lookup_slx_sig[233].nbits =  9; gv_local.lookup_slx_sig[233].bits = 0x3C800000;
  gv_local.lookup_slx_sig[234].nbits =  9; gv_local.lookup_slx_sig[234].bits = 0x58800000;
  gv_local.lookup_slx_sig[235].nbits =  9; gv_local.lookup_slx_sig[235].bits = 0xB4800000;
  gv_local.lookup_slx_sig[236].nbits =  9; gv_local.lookup_slx_sig[236].bits = 0xB8800000;
  gv_local.lookup_slx_sig[237].nbits =  9; gv_local.lookup_slx_sig[237].bits = 0xC4800000;
  gv_local.lookup_slx_sig[238].nbits =  9; gv_local.lookup_slx_sig[238].bits = 0xCA800000;
  gv_local.lookup_slx_sig[239].nbits =  9; gv_local.lookup_slx_sig[239].bits = 0xD6800000;
  gv_local.lookup_slx_sig[240].nbits =  8; gv_local.lookup_slx_sig[240].bits = 0x2B000000;
  gv_local.lookup_slx_sig[241].nbits =  8; gv_local.lookup_slx_sig[241].bits = 0x3A000000;
  gv_local.lookup_slx_sig[242].nbits =  8; gv_local.lookup_slx_sig[242].bits = 0x8A000000;
  gv_local.lookup_slx_sig[243].nbits =  8; gv_local.lookup_slx_sig[243].bits = 0xBE000000;
  gv_local.lookup_slx_sig[244].nbits =  8; gv_local.lookup_slx_sig[244].bits = 0xCF000000;
  gv_local.lookup_slx_sig[245].nbits =  7; gv_local.lookup_slx_sig[245].bits = 0x30000000;
  gv_local.lookup_slx_sig[246].nbits =  7; gv_local.lookup_slx_sig[246].bits = 0x5A000000;
  gv_local.lookup_slx_sig[247].nbits =  7; gv_local.lookup_slx_sig[247].bits = 0xC2000000;
  gv_local.lookup_slx_sig[248].nbits =  6; gv_local.lookup_slx_sig[248].bits = 0x20000000;
  gv_local.lookup_slx_sig[249].nbits =  6; gv_local.lookup_slx_sig[249].bits = 0x8C000000;
  gv_local.lookup_slx_sig[250].nbits =  6; gv_local.lookup_slx_sig[250].bits = 0xD0000000;
  gv_local.lookup_slx_sig[251].nbits =  5; gv_local.lookup_slx_sig[251].bits = 0x50000000;
  gv_local.lookup_slx_sig[252].nbits =  5; gv_local.lookup_slx_sig[252].bits = 0xD8000000;
  gv_local.lookup_slx_sig[253].nbits =  4; gv_local.lookup_slx_sig[253].bits = 0x90000000;
  gv_local.lookup_slx_sig[254].nbits =  4; gv_local.lookup_slx_sig[254].bits = 0xF0000000;
  gv_local.lookup_slx_sig[255].nbits =  3; gv_local.lookup_slx_sig[255].bits = 0x60000000;
  gv_local.lookup_slx_sig[256].nbits =  4; gv_local.lookup_slx_sig[256].bits = 0xE0000000;
  gv_local.lookup_slx_sig[257].nbits =  4; gv_local.lookup_slx_sig[257].bits = 0xA0000000;
  gv_local.lookup_slx_sig[258].nbits =  4; gv_local.lookup_slx_sig[258].bits = 0x00000000;
  gv_local.lookup_slx_sig[259].nbits =  5; gv_local.lookup_slx_sig[259].bits = 0x80000000;
  gv_local.lookup_slx_sig[260].nbits =  5; gv_local.lookup_slx_sig[260].bits = 0x10000000;
  gv_local.lookup_slx_sig[261].nbits =  6; gv_local.lookup_slx_sig[261].bits = 0xB0000000;
  gv_local.lookup_slx_sig[262].nbits =  6; gv_local.lookup_slx_sig[262].bits = 0x24000000;
  gv_local.lookup_slx_sig[263].nbits =  7; gv_local.lookup_slx_sig[263].bits = 0xC6000000;
  gv_local.lookup_slx_sig[264].nbits =  7; gv_local.lookup_slx_sig[264].bits = 0x5C000000;
  gv_local.lookup_slx_sig[265].nbits =  7; gv_local.lookup_slx_sig[265].bits = 0x2C000000;
  gv_local.lookup_slx_sig[266].nbits =  8; gv_local.lookup_slx_sig[266].bits = 0xCD000000;
  gv_local.lookup_slx_sig[267].nbits =  8; gv_local.lookup_slx_sig[267].bits = 0xBF000000;
  gv_local.lookup_slx_sig[268].nbits =  8; gv_local.lookup_slx_sig[268].bits = 0x89000000;
  gv_local.lookup_slx_sig[269].nbits =  8; gv_local.lookup_slx_sig[269].bits = 0x38000000;
  gv_local.lookup_slx_sig[270].nbits =  8; gv_local.lookup_slx_sig[270].bits = 0x1D000000;
  gv_local.lookup_slx_sig[271].nbits =  9; gv_local.lookup_slx_sig[271].bits = 0xD4000000;
  gv_local.lookup_slx_sig[272].nbits =  9; gv_local.lookup_slx_sig[272].bits = 0xC8800000;
  gv_local.lookup_slx_sig[273].nbits =  9; gv_local.lookup_slx_sig[273].bits = 0xBB000000;
  gv_local.lookup_slx_sig[274].nbits =  9; gv_local.lookup_slx_sig[274].bits = 0xB5800000;
  gv_local.lookup_slx_sig[275].nbits =  9; gv_local.lookup_slx_sig[275].bits = 0x5F800000;
  gv_local.lookup_slx_sig[276].nbits =  9; gv_local.lookup_slx_sig[276].bits = 0x3B800000;
  gv_local.lookup_slx_sig[277].nbits =  9; gv_local.lookup_slx_sig[277].bits = 0x35000000;
  gv_local.lookup_slx_sig[278].nbits =  9; gv_local.lookup_slx_sig[278].bits = 0x1F800000;
  gv_local.lookup_slx_sig[279].nbits =  9; gv_local.lookup_slx_sig[279].bits = 0x19800000;
  gv_local.lookup_slx_sig[280].nbits = 10; gv_local.lookup_slx_sig[280].bits = 0xD6400000;
  gv_local.lookup_slx_sig[281].nbits = 10; gv_local.lookup_slx_sig[281].bits = 0xCE800000;
  gv_local.lookup_slx_sig[282].nbits = 10; gv_local.lookup_slx_sig[282].bits = 0xCBC00000;
  gv_local.lookup_slx_sig[283].nbits = 10; gv_local.lookup_slx_sig[283].bits = 0xC4000000;
  gv_local.lookup_slx_sig[284].nbits = 10; gv_local.lookup_slx_sig[284].bits = 0xBD800000;
  gv_local.lookup_slx_sig[285].nbits = 10; gv_local.lookup_slx_sig[285].bits = 0xBB800000;
  gv_local.lookup_slx_sig[286].nbits = 10; gv_local.lookup_slx_sig[286].bits = 0xB9400000;
  gv_local.lookup_slx_sig[287].nbits = 10; gv_local.lookup_slx_sig[287].bits = 0xB5400000;
  gv_local.lookup_slx_sig[288].nbits = 10; gv_local.lookup_slx_sig[288].bits = 0x88800000;
  gv_local.lookup_slx_sig[289].nbits = 10; gv_local.lookup_slx_sig[289].bits = 0x5F000000;
  gv_local.lookup_slx_sig[290].nbits = 10; gv_local.lookup_slx_sig[290].bits = 0x3DC00000;
  gv_local.lookup_slx_sig[291].nbits = 10; gv_local.lookup_slx_sig[291].bits = 0x3B000000;
  gv_local.lookup_slx_sig[292].nbits = 10; gv_local.lookup_slx_sig[292].bits = 0x36C00000;
  gv_local.lookup_slx_sig[293].nbits = 10; gv_local.lookup_slx_sig[293].bits = 0x34400000;
  gv_local.lookup_slx_sig[294].nbits = 10; gv_local.lookup_slx_sig[294].bits = 0x36800000;
  gv_local.lookup_slx_sig[295].nbits = 10; gv_local.lookup_slx_sig[295].bits = 0x2A400000;
  gv_local.lookup_slx_sig[296].nbits = 10; gv_local.lookup_slx_sig[296].bits = 0x1EC00000;
  gv_local.lookup_slx_sig[297].nbits = 10; gv_local.lookup_slx_sig[297].bits = 0x2AC00000;
  gv_local.lookup_slx_sig[298].nbits = 10; gv_local.lookup_slx_sig[298].bits = 0x1AC00000;
  gv_local.lookup_slx_sig[299].nbits = 10; gv_local.lookup_slx_sig[299].bits = 0x1E400000;
  gv_local.lookup_slx_sig[300].nbits = 11; gv_local.lookup_slx_sig[300].bits = 0xD7400000;
  gv_local.lookup_slx_sig[301].nbits = 11; gv_local.lookup_slx_sig[301].bits = 0xD5A00000;
  gv_local.lookup_slx_sig[302].nbits = 11; gv_local.lookup_slx_sig[302].bits = 0xD7A00000;
  gv_local.lookup_slx_sig[303].nbits = 10; gv_local.lookup_slx_sig[303].bits = 0x18000000;
  gv_local.lookup_slx_sig[304].nbits = 11; gv_local.lookup_slx_sig[304].bits = 0xCCE00000;
  gv_local.lookup_slx_sig[305].nbits = 11; gv_local.lookup_slx_sig[305].bits = 0xC9400000;
  gv_local.lookup_slx_sig[306].nbits = 11; gv_local.lookup_slx_sig[306].bits = 0xCB800000;
  gv_local.lookup_slx_sig[307].nbits = 11; gv_local.lookup_slx_sig[307].bits = 0xCC200000;
  gv_local.lookup_slx_sig[308].nbits = 11; gv_local.lookup_slx_sig[308].bits = 0xCC000000;
  gv_local.lookup_slx_sig[309].nbits = 11; gv_local.lookup_slx_sig[309].bits = 0xC8200000;
  gv_local.lookup_slx_sig[310].nbits = 11; gv_local.lookup_slx_sig[310].bits = 0xC5400000;
  gv_local.lookup_slx_sig[311].nbits = 11; gv_local.lookup_slx_sig[311].bits = 0xC0C00000;
  gv_local.lookup_slx_sig[312].nbits = 11; gv_local.lookup_slx_sig[312].bits = 0xC1E00000;
  gv_local.lookup_slx_sig[313].nbits = 11; gv_local.lookup_slx_sig[313].bits = 0xBA800000;
  gv_local.lookup_slx_sig[314].nbits = 11; gv_local.lookup_slx_sig[314].bits = 0xBCC00000;
  gv_local.lookup_slx_sig[315].nbits = 11; gv_local.lookup_slx_sig[315].bits = 0xBC400000;
  gv_local.lookup_slx_sig[316].nbits = 11; gv_local.lookup_slx_sig[316].bits = 0xBC800000;
  gv_local.lookup_slx_sig[317].nbits = 11; gv_local.lookup_slx_sig[317].bits = 0xB9000000;
  gv_local.lookup_slx_sig[318].nbits = 11; gv_local.lookup_slx_sig[318].bits = 0xB8600000;
  gv_local.lookup_slx_sig[319].nbits = 11; gv_local.lookup_slx_sig[319].bits = 0xB7E00000;
  gv_local.lookup_slx_sig[320].nbits = 11; gv_local.lookup_slx_sig[320].bits = 0xBA200000;
  gv_local.lookup_slx_sig[321].nbits = 11; gv_local.lookup_slx_sig[321].bits = 0xB7600000;
  gv_local.lookup_slx_sig[322].nbits = 11; gv_local.lookup_slx_sig[322].bits = 0xB7400000;
  gv_local.lookup_slx_sig[323].nbits = 11; gv_local.lookup_slx_sig[323].bits = 0xB4600000;
  gv_local.lookup_slx_sig[324].nbits = 11; gv_local.lookup_slx_sig[324].bits = 0x8BC00000;
  gv_local.lookup_slx_sig[325].nbits = 11; gv_local.lookup_slx_sig[325].bits = 0xB6E00000;
  gv_local.lookup_slx_sig[326].nbits = 11; gv_local.lookup_slx_sig[326].bits = 0xB4400000;
  gv_local.lookup_slx_sig[327].nbits = 11; gv_local.lookup_slx_sig[327].bits = 0x8BE00000;
  gv_local.lookup_slx_sig[328].nbits = 11; gv_local.lookup_slx_sig[328].bits = 0x58000000;
  gv_local.lookup_slx_sig[329].nbits = 11; gv_local.lookup_slx_sig[329].bits = 0x58200000;
  gv_local.lookup_slx_sig[330].nbits = 11; gv_local.lookup_slx_sig[330].bits = 0x3D800000;
  gv_local.lookup_slx_sig[331].nbits = 11; gv_local.lookup_slx_sig[331].bits = 0x3EA00000;
  gv_local.lookup_slx_sig[332].nbits = 11; gv_local.lookup_slx_sig[332].bits = 0x3F200000;
  gv_local.lookup_slx_sig[333].nbits = 11; gv_local.lookup_slx_sig[333].bits = 0x59600000;
  gv_local.lookup_slx_sig[334].nbits = 11; gv_local.lookup_slx_sig[334].bits = 0x37000000;
  gv_local.lookup_slx_sig[335].nbits = 11; gv_local.lookup_slx_sig[335].bits = 0x3D200000;
  gv_local.lookup_slx_sig[336].nbits = 11; gv_local.lookup_slx_sig[336].bits = 0x33800000;
  gv_local.lookup_slx_sig[337].nbits = 11; gv_local.lookup_slx_sig[337].bits = 0x2E000000;
  gv_local.lookup_slx_sig[338].nbits = 11; gv_local.lookup_slx_sig[338].bits = 0x3C000000;
  gv_local.lookup_slx_sig[339].nbits = 11; gv_local.lookup_slx_sig[339].bits = 0x2AA00000;
  gv_local.lookup_slx_sig[340].nbits = 11; gv_local.lookup_slx_sig[340].bits = 0x34800000;
  gv_local.lookup_slx_sig[341].nbits = 11; gv_local.lookup_slx_sig[341].bits = 0x32C00000;
  gv_local.lookup_slx_sig[342].nbits = 11; gv_local.lookup_slx_sig[342].bits = 0x34000000;
  gv_local.lookup_slx_sig[343].nbits = 11; gv_local.lookup_slx_sig[343].bits = 0x2E600000;
  gv_local.lookup_slx_sig[344].nbits = 11; gv_local.lookup_slx_sig[344].bits = 0x35E00000;
  gv_local.lookup_slx_sig[345].nbits = 11; gv_local.lookup_slx_sig[345].bits = 0x32000000;
  gv_local.lookup_slx_sig[346].nbits = 11; gv_local.lookup_slx_sig[346].bits = 0x2A000000;
  gv_local.lookup_slx_sig[347].nbits = 11; gv_local.lookup_slx_sig[347].bits = 0x29E00000;
  gv_local.lookup_slx_sig[348].nbits = 11; gv_local.lookup_slx_sig[348].bits = 0x1CA00000;
  gv_local.lookup_slx_sig[349].nbits = 11; gv_local.lookup_slx_sig[349].bits = 0x1A200000;
  gv_local.lookup_slx_sig[350].nbits = 11; gv_local.lookup_slx_sig[350].bits = 0x19400000;
  gv_local.lookup_slx_sig[351].nbits = 12; gv_local.lookup_slx_sig[351].bits = 0xD5100000;
  gv_local.lookup_slx_sig[352].nbits = 11; gv_local.lookup_slx_sig[352].bits = 0x1C800000;
  gv_local.lookup_slx_sig[353].nbits = 11; gv_local.lookup_slx_sig[353].bits = 0x19600000;
  gv_local.lookup_slx_sig[354].nbits = 12; gv_local.lookup_slx_sig[354].bits = 0xD7900000;
  gv_local.lookup_slx_sig[355].nbits = 12; gv_local.lookup_slx_sig[355].bits = 0xD7800000;
  gv_local.lookup_slx_sig[356].nbits = 12; gv_local.lookup_slx_sig[356].bits = 0xCEE00000;
  gv_local.lookup_slx_sig[357].nbits = 12; gv_local.lookup_slx_sig[357].bits = 0xD5000000;
  gv_local.lookup_slx_sig[358].nbits = 12; gv_local.lookup_slx_sig[358].bits = 0xCC800000;
  gv_local.lookup_slx_sig[359].nbits = 12; gv_local.lookup_slx_sig[359].bits = 0xD4A00000;
  gv_local.lookup_slx_sig[360].nbits = 12; gv_local.lookup_slx_sig[360].bits = 0xCB300000;
  gv_local.lookup_slx_sig[361].nbits = 12; gv_local.lookup_slx_sig[361].bits = 0xCE700000;
  gv_local.lookup_slx_sig[362].nbits = 12; gv_local.lookup_slx_sig[362].bits = 0xCE600000;
  gv_local.lookup_slx_sig[363].nbits = 12; gv_local.lookup_slx_sig[363].bits = 0xCCC00000;
  gv_local.lookup_slx_sig[364].nbits = 12; gv_local.lookup_slx_sig[364].bits = 0xC1D00000;
  gv_local.lookup_slx_sig[365].nbits = 12; gv_local.lookup_slx_sig[365].bits = 0xC5600000;
  gv_local.lookup_slx_sig[366].nbits = 12; gv_local.lookup_slx_sig[366].bits = 0xC8000000;
  gv_local.lookup_slx_sig[367].nbits = 12; gv_local.lookup_slx_sig[367].bits = 0xC9F00000;
  gv_local.lookup_slx_sig[368].nbits = 12; gv_local.lookup_slx_sig[368].bits = 0xC9D00000;
  gv_local.lookup_slx_sig[369].nbits = 12; gv_local.lookup_slx_sig[369].bits = 0xC5C00000;
  gv_local.lookup_slx_sig[370].nbits = 12; gv_local.lookup_slx_sig[370].bits = 0xBC600000;
  gv_local.lookup_slx_sig[371].nbits = 12; gv_local.lookup_slx_sig[371].bits = 0xC1B00000;
  gv_local.lookup_slx_sig[372].nbits = 12; gv_local.lookup_slx_sig[372].bits = 0xBBE00000;
  gv_local.lookup_slx_sig[373].nbits = 12; gv_local.lookup_slx_sig[373].bits = 0xC1500000;
  gv_local.lookup_slx_sig[374].nbits = 12; gv_local.lookup_slx_sig[374].bits = 0xBAB00000;
  gv_local.lookup_slx_sig[375].nbits = 12; gv_local.lookup_slx_sig[375].bits = 0xC1400000;
  gv_local.lookup_slx_sig[376].nbits = 12; gv_local.lookup_slx_sig[376].bits = 0xB8400000;
  gv_local.lookup_slx_sig[377].nbits = 12; gv_local.lookup_slx_sig[377].bits = 0xBA000000;
  gv_local.lookup_slx_sig[378].nbits = 12; gv_local.lookup_slx_sig[378].bits = 0xB7B00000;
  gv_local.lookup_slx_sig[379].nbits = 12; gv_local.lookup_slx_sig[379].bits = 0xC0400000;
  gv_local.lookup_slx_sig[380].nbits = 12; gv_local.lookup_slx_sig[380].bits = 0x59400000;
  gv_local.lookup_slx_sig[381].nbits = 12; gv_local.lookup_slx_sig[381].bits = 0x8B300000;
  gv_local.lookup_slx_sig[382].nbits = 12; gv_local.lookup_slx_sig[382].bits = 0xB6B00000;
  gv_local.lookup_slx_sig[383].nbits = 12; gv_local.lookup_slx_sig[383].bits = 0xBA100000;
  gv_local.lookup_slx_sig[384].nbits = 12; gv_local.lookup_slx_sig[384].bits = 0x5EF00000;
  gv_local.lookup_slx_sig[385].nbits = 12; gv_local.lookup_slx_sig[385].bits = 0xB5000000;
  gv_local.lookup_slx_sig[386].nbits = 12; gv_local.lookup_slx_sig[386].bits = 0xB7200000;
  gv_local.lookup_slx_sig[387].nbits = 12; gv_local.lookup_slx_sig[387].bits = 0xB7A00000;
  gv_local.lookup_slx_sig[388].nbits = 12; gv_local.lookup_slx_sig[388].bits = 0x8B200000;
  gv_local.lookup_slx_sig[389].nbits = 12; gv_local.lookup_slx_sig[389].bits = 0x3F800000;
  gv_local.lookup_slx_sig[390].nbits = 12; gv_local.lookup_slx_sig[390].bits = 0xB7100000;
  gv_local.lookup_slx_sig[391].nbits = 12; gv_local.lookup_slx_sig[391].bits = 0x3E400000;
  gv_local.lookup_slx_sig[392].nbits = 12; gv_local.lookup_slx_sig[392].bits = 0x5EA00000;
  gv_local.lookup_slx_sig[393].nbits = 12; gv_local.lookup_slx_sig[393].bits = 0xB6A00000;
  gv_local.lookup_slx_sig[394].nbits = 12; gv_local.lookup_slx_sig[394].bits = 0x39C00000;
  gv_local.lookup_slx_sig[395].nbits = 12; gv_local.lookup_slx_sig[395].bits = 0x3E900000;
  gv_local.lookup_slx_sig[396].nbits = 12; gv_local.lookup_slx_sig[396].bits = 0x8B100000;
  gv_local.lookup_slx_sig[397].nbits = 12; gv_local.lookup_slx_sig[397].bits = 0x39D00000;
  gv_local.lookup_slx_sig[398].nbits = 12; gv_local.lookup_slx_sig[398].bits = 0x3B600000;
  gv_local.lookup_slx_sig[399].nbits = 12; gv_local.lookup_slx_sig[399].bits = 0x3C200000;
  gv_local.lookup_slx_sig[400].nbits = 12; gv_local.lookup_slx_sig[400].bits = 0x39600000;
  gv_local.lookup_slx_sig[401].nbits = 12; gv_local.lookup_slx_sig[401].bits = 0x3D700000;
  gv_local.lookup_slx_sig[402].nbits = 12; gv_local.lookup_slx_sig[402].bits = 0x35900000;
  gv_local.lookup_slx_sig[403].nbits = 12; gv_local.lookup_slx_sig[403].bits = 0x1E900000;
  gv_local.lookup_slx_sig[404].nbits = 12; gv_local.lookup_slx_sig[404].bits = 0x34200000;
  gv_local.lookup_slx_sig[405].nbits = 12; gv_local.lookup_slx_sig[405].bits = 0x2F700000;
  gv_local.lookup_slx_sig[406].nbits = 12; gv_local.lookup_slx_sig[406].bits = 0x33700000;
  gv_local.lookup_slx_sig[407].nbits = 12; gv_local.lookup_slx_sig[407].bits = 0x33000000;
  gv_local.lookup_slx_sig[408].nbits = 12; gv_local.lookup_slx_sig[408].bits = 0x2F000000;
  gv_local.lookup_slx_sig[409].nbits = 12; gv_local.lookup_slx_sig[409].bits = 0x37200000;
  gv_local.lookup_slx_sig[410].nbits = 12; gv_local.lookup_slx_sig[410].bits = 0x1C400000;
  gv_local.lookup_slx_sig[411].nbits = 12; gv_local.lookup_slx_sig[411].bits = 0x28500000;
  gv_local.lookup_slx_sig[412].nbits = 12; gv_local.lookup_slx_sig[412].bits = 0x1CE00000;
  gv_local.lookup_slx_sig[413].nbits = 12; gv_local.lookup_slx_sig[413].bits = 0x1A800000;
  gv_local.lookup_slx_sig[414].nbits = 12; gv_local.lookup_slx_sig[414].bits = 0x1EB00000;
  gv_local.lookup_slx_sig[415].nbits = 13; gv_local.lookup_slx_sig[415].bits = 0xCEC80000;
  gv_local.lookup_slx_sig[416].nbits = 13; gv_local.lookup_slx_sig[416].bits = 0xD6100000;
  gv_local.lookup_slx_sig[417].nbits = 13; gv_local.lookup_slx_sig[417].bits = 0xD7E00000;
  gv_local.lookup_slx_sig[418].nbits = 12; gv_local.lookup_slx_sig[418].bits = 0x1A100000;
  gv_local.lookup_slx_sig[419].nbits = 12; gv_local.lookup_slx_sig[419].bits = 0x19300000;
  gv_local.lookup_slx_sig[420].nbits = 12; gv_local.lookup_slx_sig[420].bits = 0x1CF00000;
  gv_local.lookup_slx_sig[421].nbits = 12; gv_local.lookup_slx_sig[421].bits = 0x2A300000;
  gv_local.lookup_slx_sig[422].nbits = 13; gv_local.lookup_slx_sig[422].bits = 0xD4B00000;
  gv_local.lookup_slx_sig[423].nbits = 13; gv_local.lookup_slx_sig[423].bits = 0xD6380000;
  gv_local.lookup_slx_sig[424].nbits = 12; gv_local.lookup_slx_sig[424].bits = 0x1A000000;
  gv_local.lookup_slx_sig[425].nbits = 13; gv_local.lookup_slx_sig[425].bits = 0xCB280000;
  gv_local.lookup_slx_sig[426].nbits = 13; gv_local.lookup_slx_sig[426].bits = 0xD4F00000;
  gv_local.lookup_slx_sig[427].nbits = 13; gv_local.lookup_slx_sig[427].bits = 0xD4B80000;
  gv_local.lookup_slx_sig[428].nbits = 12; gv_local.lookup_slx_sig[428].bits = 0x1B700000;
  gv_local.lookup_slx_sig[429].nbits = 13; gv_local.lookup_slx_sig[429].bits = 0xD5680000;
  gv_local.lookup_slx_sig[430].nbits = 13; gv_local.lookup_slx_sig[430].bits = 0xCCD80000;
  gv_local.lookup_slx_sig[431].nbits = 13; gv_local.lookup_slx_sig[431].bits = 0xCE500000;
  gv_local.lookup_slx_sig[432].nbits = 13; gv_local.lookup_slx_sig[432].bits = 0xCEC00000;
  gv_local.lookup_slx_sig[433].nbits = 13; gv_local.lookup_slx_sig[433].bits = 0xC9A80000;
  gv_local.lookup_slx_sig[434].nbits = 13; gv_local.lookup_slx_sig[434].bits = 0xCB200000;
  gv_local.lookup_slx_sig[435].nbits = 13; gv_local.lookup_slx_sig[435].bits = 0xBD780000;
  gv_local.lookup_slx_sig[436].nbits = 13; gv_local.lookup_slx_sig[436].bits = 0xC8780000;
  gv_local.lookup_slx_sig[437].nbits = 13; gv_local.lookup_slx_sig[437].bits = 0xBBF00000;
  gv_local.lookup_slx_sig[438].nbits = 13; gv_local.lookup_slx_sig[438].bits = 0xBD600000;
  gv_local.lookup_slx_sig[439].nbits = 13; gv_local.lookup_slx_sig[439].bits = 0xC8700000;
  gv_local.lookup_slx_sig[440].nbits = 13; gv_local.lookup_slx_sig[440].bits = 0xB7D00000;
  gv_local.lookup_slx_sig[441].nbits = 13; gv_local.lookup_slx_sig[441].bits = 0xC4580000;
  gv_local.lookup_slx_sig[442].nbits = 13; gv_local.lookup_slx_sig[442].bits = 0xC5280000;
  gv_local.lookup_slx_sig[443].nbits = 13; gv_local.lookup_slx_sig[443].bits = 0xC0B00000;
  gv_local.lookup_slx_sig[444].nbits = 13; gv_local.lookup_slx_sig[444].bits = 0xB9300000;
  gv_local.lookup_slx_sig[445].nbits = 13; gv_local.lookup_slx_sig[445].bits = 0xBC780000;
  gv_local.lookup_slx_sig[446].nbits = 13; gv_local.lookup_slx_sig[446].bits = 0xB7000000;
  gv_local.lookup_slx_sig[447].nbits = 13; gv_local.lookup_slx_sig[447].bits = 0xBD680000;
  gv_local.lookup_slx_sig[448].nbits = 13; gv_local.lookup_slx_sig[448].bits = 0xC1A80000;
  gv_local.lookup_slx_sig[449].nbits = 13; gv_local.lookup_slx_sig[449].bits = 0xBAA80000;
  gv_local.lookup_slx_sig[450].nbits = 13; gv_local.lookup_slx_sig[450].bits = 0xB7800000;
  gv_local.lookup_slx_sig[451].nbits = 13; gv_local.lookup_slx_sig[451].bits = 0xB9380000;
  gv_local.lookup_slx_sig[452].nbits = 13; gv_local.lookup_slx_sig[452].bits = 0x5E480000;
  gv_local.lookup_slx_sig[453].nbits = 13; gv_local.lookup_slx_sig[453].bits = 0xB6980000;
  gv_local.lookup_slx_sig[454].nbits = 13; gv_local.lookup_slx_sig[454].bits = 0xBBF80000;
  gv_local.lookup_slx_sig[455].nbits = 13; gv_local.lookup_slx_sig[455].bits = 0x8B000000;
  gv_local.lookup_slx_sig[456].nbits = 13; gv_local.lookup_slx_sig[456].bits = 0xB6900000;
  gv_local.lookup_slx_sig[457].nbits = 13; gv_local.lookup_slx_sig[457].bits = 0x8B800000;
  gv_local.lookup_slx_sig[458].nbits = 13; gv_local.lookup_slx_sig[458].bits = 0x5E400000;
  gv_local.lookup_slx_sig[459].nbits = 13; gv_local.lookup_slx_sig[459].bits = 0xB5100000;
  gv_local.lookup_slx_sig[460].nbits = 13; gv_local.lookup_slx_sig[460].bits = 0xB5380000;
  gv_local.lookup_slx_sig[461].nbits = 13; gv_local.lookup_slx_sig[461].bits = 0xBA400000;
  gv_local.lookup_slx_sig[462].nbits = 13; gv_local.lookup_slx_sig[462].bits = 0xBAF00000;
  gv_local.lookup_slx_sig[463].nbits = 13; gv_local.lookup_slx_sig[463].bits = 0xB5300000;
  gv_local.lookup_slx_sig[464].nbits = 13; gv_local.lookup_slx_sig[464].bits = 0x3B700000;
  gv_local.lookup_slx_sig[465].nbits = 13; gv_local.lookup_slx_sig[465].bits = 0xBAF80000;
  gv_local.lookup_slx_sig[466].nbits = 13; gv_local.lookup_slx_sig[466].bits = 0x8B080000;
  gv_local.lookup_slx_sig[467].nbits = 13; gv_local.lookup_slx_sig[467].bits = 0x5EE00000;
  gv_local.lookup_slx_sig[468].nbits = 13; gv_local.lookup_slx_sig[468].bits = 0x59580000;
  gv_local.lookup_slx_sig[469].nbits = 13; gv_local.lookup_slx_sig[469].bits = 0x5E580000;
  gv_local.lookup_slx_sig[470].nbits = 13; gv_local.lookup_slx_sig[470].bits = 0x59080000;
  gv_local.lookup_slx_sig[471].nbits = 13; gv_local.lookup_slx_sig[471].bits = 0x39A00000;
  gv_local.lookup_slx_sig[472].nbits = 13; gv_local.lookup_slx_sig[472].bits = 0x39F00000;
  gv_local.lookup_slx_sig[473].nbits = 13; gv_local.lookup_slx_sig[473].bits = 0x37300000;
  gv_local.lookup_slx_sig[474].nbits = 13; gv_local.lookup_slx_sig[474].bits = 0x3D600000;
  gv_local.lookup_slx_sig[475].nbits = 13; gv_local.lookup_slx_sig[475].bits = 0x39F80000;
  gv_local.lookup_slx_sig[476].nbits = 13; gv_local.lookup_slx_sig[476].bits = 0x28480000;
  gv_local.lookup_slx_sig[477].nbits = 13; gv_local.lookup_slx_sig[477].bits = 0x5E500000;
  gv_local.lookup_slx_sig[478].nbits = 13; gv_local.lookup_slx_sig[478].bits = 0x28680000;
  gv_local.lookup_slx_sig[479].nbits = 13; gv_local.lookup_slx_sig[479].bits = 0x3B780000;
  gv_local.lookup_slx_sig[480].nbits = 13; gv_local.lookup_slx_sig[480].bits = 0x32980000;
  gv_local.lookup_slx_sig[481].nbits = 13; gv_local.lookup_slx_sig[481].bits = 0x2F600000;
  gv_local.lookup_slx_sig[482].nbits = 13; gv_local.lookup_slx_sig[482].bits = 0x37380000;
  gv_local.lookup_slx_sig[483].nbits = 13; gv_local.lookup_slx_sig[483].bits = 0x39A80000;
  gv_local.lookup_slx_sig[484].nbits = 13; gv_local.lookup_slx_sig[484].bits = 0x35880000;
  gv_local.lookup_slx_sig[485].nbits = 13; gv_local.lookup_slx_sig[485].bits = 0x35800000;
  gv_local.lookup_slx_sig[486].nbits = 13; gv_local.lookup_slx_sig[486].bits = 0x1E800000;
  gv_local.lookup_slx_sig[487].nbits = 13; gv_local.lookup_slx_sig[487].bits = 0x1EA80000;
  gv_local.lookup_slx_sig[488].nbits = 13; gv_local.lookup_slx_sig[488].bits = 0x33680000;
  gv_local.lookup_slx_sig[489].nbits = 13; gv_local.lookup_slx_sig[489].bits = 0x1E880000;
  gv_local.lookup_slx_sig[490].nbits = 13; gv_local.lookup_slx_sig[490].bits = 0x2F680000;
  gv_local.lookup_slx_sig[491].nbits = 13; gv_local.lookup_slx_sig[491].bits = 0x33600000;
  gv_local.lookup_slx_sig[492].nbits = 14; gv_local.lookup_slx_sig[492].bits = 0xCEF40000;
  gv_local.lookup_slx_sig[493].nbits = 13; gv_local.lookup_slx_sig[493].bits = 0x59000000;
  gv_local.lookup_slx_sig[494].nbits = 13; gv_local.lookup_slx_sig[494].bits = 0x19200000;
  gv_local.lookup_slx_sig[495].nbits = 13; gv_local.lookup_slx_sig[495].bits = 0x1EA00000;
  gv_local.lookup_slx_sig[496].nbits = 13; gv_local.lookup_slx_sig[496].bits = 0x1A900000;
  gv_local.lookup_slx_sig[497].nbits = 13; gv_local.lookup_slx_sig[497].bits = 0x32900000;
  gv_local.lookup_slx_sig[498].nbits = 13; gv_local.lookup_slx_sig[498].bits = 0x28600000;
  gv_local.lookup_slx_sig[499].nbits = 14; gv_local.lookup_slx_sig[499].bits = 0xCE580000;
  gv_local.lookup_slx_sig[500].nbits = 13; gv_local.lookup_slx_sig[500].bits = 0x1A980000;
  gv_local.lookup_slx_sig[501].nbits = 14; gv_local.lookup_slx_sig[501].bits = 0xCA640000;
  gv_local.lookup_slx_sig[502].nbits = 13; gv_local.lookup_slx_sig[502].bits = 0x19280000;
  gv_local.lookup_slx_sig[503].nbits = 14; gv_local.lookup_slx_sig[503].bits = 0xD6240000;
  gv_local.lookup_slx_sig[504].nbits = 14; gv_local.lookup_slx_sig[504].bits = 0xD5600000;
  gv_local.lookup_slx_sig[505].nbits = 13; gv_local.lookup_slx_sig[505].bits = 0x28400000;
  gv_local.lookup_slx_sig[506].nbits = 14; gv_local.lookup_slx_sig[506].bits = 0xD6200000;
  gv_local.lookup_slx_sig[507].nbits = 14; gv_local.lookup_slx_sig[507].bits = 0xCE5C0000;
  gv_local.lookup_slx_sig[508].nbits = 14; gv_local.lookup_slx_sig[508].bits = 0xD5640000;
  gv_local.lookup_slx_sig[509].nbits = 14; gv_local.lookup_slx_sig[509].bits = 0xCEF00000;
  gv_local.lookup_slx_sig[510].nbits = 14; gv_local.lookup_slx_sig[510].bits = 0xCA600000;
  gv_local.lookup_slx_sig[511].nbits =  4; gv_local.lookup_slx_sig[511].bits = 0x40000000;

#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  /* reverse huffman lookup table
   */
  memset (gv_local.rlookup_slx_sig, 0, sizeof(uint16_t)*REV_HUFF_TREESIZE_16);

  for (i=0; i<SIG_HUFF_TREESIZE; i++)
  {
    val = gv_local.lookup_slx_sig[i].bits >> 16;
    gv_local.rlookup_slx_sig [val] = (uint16_t) i;
    for (j=1; j<=(0xFFFF>>gv_local.lookup_slx_sig[i].nbits); j++)
      gv_local.rlookup_slx_sig [val+j] = (uint16_t) i;
  }
#endif

  _s_sort_lookup (gv_local.lookup_slx_sig, gv_local.idx_slx_sig, SIG_HUFF_TREESIZE);

  gv_local.lookup_slx_sig_initialized = 1;
}
/*
 */
static void _s_init_int_slx_default (void)
{
#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  int      i, j;
  uint16_t val;
#endif

  memset (gv_local.lookup_slx_int,  0, sizeof(SRALookup)*INT_HUFF_TREESIZE);

  gv_local.lookup_slx_int[   0].nbits = 13; gv_local.lookup_slx_int[   0].bits = 0x9DF80000;
  gv_local.lookup_slx_int[   1].nbits = 13; gv_local.lookup_slx_int[   1].bits = 0x9DF00000;
  gv_local.lookup_slx_int[   2].nbits = 13; gv_local.lookup_slx_int[   2].bits = 0xA4A00000;
  gv_local.lookup_slx_int[   3].nbits = 13; gv_local.lookup_slx_int[   3].bits = 0xA7C00000;
  gv_local.lookup_slx_int[   4].nbits = 13; gv_local.lookup_slx_int[   4].bits = 0xA4A80000;
  gv_local.lookup_slx_int[   5].nbits = 13; gv_local.lookup_slx_int[   5].bits = 0xA7C80000;
  gv_local.lookup_slx_int[   6].nbits = 13; gv_local.lookup_slx_int[   6].bits = 0xB6000000;
  gv_local.lookup_slx_int[   7].nbits = 13; gv_local.lookup_slx_int[   7].bits = 0xAF600000;
  gv_local.lookup_slx_int[   8].nbits = 13; gv_local.lookup_slx_int[   8].bits = 0xAF680000;
  gv_local.lookup_slx_int[   9].nbits = 13; gv_local.lookup_slx_int[   9].bits = 0xBBA00000;
  gv_local.lookup_slx_int[  10].nbits = 13; gv_local.lookup_slx_int[  10].bits = 0xB6080000;
  gv_local.lookup_slx_int[  11].nbits = 13; gv_local.lookup_slx_int[  11].bits = 0xBBA80000;
  gv_local.lookup_slx_int[  12].nbits = 13; gv_local.lookup_slx_int[  12].bits = 0xBDC80000;
  gv_local.lookup_slx_int[  13].nbits = 13; gv_local.lookup_slx_int[  13].bits = 0xBDC00000;
  gv_local.lookup_slx_int[  14].nbits = 13; gv_local.lookup_slx_int[  14].bits = 0xCA400000;
  gv_local.lookup_slx_int[  15].nbits = 13; gv_local.lookup_slx_int[  15].bits = 0xC1F80000;
  gv_local.lookup_slx_int[  16].nbits = 13; gv_local.lookup_slx_int[  16].bits = 0xC1F00000;
  gv_local.lookup_slx_int[  17].nbits = 13; gv_local.lookup_slx_int[  17].bits = 0xCA480000;
  gv_local.lookup_slx_int[  18].nbits = 13; gv_local.lookup_slx_int[  18].bits = 0xCF600000;
  gv_local.lookup_slx_int[  19].nbits = 13; gv_local.lookup_slx_int[  19].bits = 0xD2C00000;
  gv_local.lookup_slx_int[  20].nbits = 13; gv_local.lookup_slx_int[  20].bits = 0xCF680000;
  gv_local.lookup_slx_int[  21].nbits = 13; gv_local.lookup_slx_int[  21].bits = 0xD2C80000;
  gv_local.lookup_slx_int[  22].nbits = 13; gv_local.lookup_slx_int[  22].bits = 0xDA200000;
  gv_local.lookup_slx_int[  23].nbits = 13; gv_local.lookup_slx_int[  23].bits = 0xDE900000;
  gv_local.lookup_slx_int[  24].nbits = 13; gv_local.lookup_slx_int[  24].bits = 0xDA280000;
  gv_local.lookup_slx_int[  25].nbits = 12; gv_local.lookup_slx_int[  25].bits = 0x02C00000;
  gv_local.lookup_slx_int[  26].nbits = 13; gv_local.lookup_slx_int[  26].bits = 0xDE980000;
  gv_local.lookup_slx_int[  27].nbits = 12; gv_local.lookup_slx_int[  27].bits = 0x02D00000;
  gv_local.lookup_slx_int[  28].nbits = 12; gv_local.lookup_slx_int[  28].bits = 0x0C400000;
  gv_local.lookup_slx_int[  29].nbits = 12; gv_local.lookup_slx_int[  29].bits = 0x0C500000;
  gv_local.lookup_slx_int[  30].nbits = 12; gv_local.lookup_slx_int[  30].bits = 0x1A800000;
  gv_local.lookup_slx_int[  31].nbits = 12; gv_local.lookup_slx_int[  31].bits = 0x1A900000;
  gv_local.lookup_slx_int[  32].nbits = 12; gv_local.lookup_slx_int[  32].bits = 0x21300000;
  gv_local.lookup_slx_int[  33].nbits = 12; gv_local.lookup_slx_int[  33].bits = 0x21100000;
  gv_local.lookup_slx_int[  34].nbits = 12; gv_local.lookup_slx_int[  34].bits = 0x29900000;
  gv_local.lookup_slx_int[  35].nbits = 12; gv_local.lookup_slx_int[  35].bits = 0x3C200000;
  gv_local.lookup_slx_int[  36].nbits = 12; gv_local.lookup_slx_int[  36].bits = 0x3C000000;
  gv_local.lookup_slx_int[  37].nbits = 12; gv_local.lookup_slx_int[  37].bits = 0x3E600000;
  gv_local.lookup_slx_int[  38].nbits = 12; gv_local.lookup_slx_int[  38].bits = 0x4C000000;
  gv_local.lookup_slx_int[  39].nbits = 12; gv_local.lookup_slx_int[  39].bits = 0x4F000000;
  gv_local.lookup_slx_int[  40].nbits = 12; gv_local.lookup_slx_int[  40].bits = 0x4C400000;
  gv_local.lookup_slx_int[  41].nbits = 12; gv_local.lookup_slx_int[  41].bits = 0x4F300000;
  gv_local.lookup_slx_int[  42].nbits = 12; gv_local.lookup_slx_int[  42].bits = 0x58D00000;
  gv_local.lookup_slx_int[  43].nbits = 12; gv_local.lookup_slx_int[  43].bits = 0x4F100000;
  gv_local.lookup_slx_int[  44].nbits = 12; gv_local.lookup_slx_int[  44].bits = 0x5BA00000;
  gv_local.lookup_slx_int[  45].nbits = 12; gv_local.lookup_slx_int[  45].bits = 0x6D000000;
  gv_local.lookup_slx_int[  46].nbits = 12; gv_local.lookup_slx_int[  46].bits = 0x6F200000;
  gv_local.lookup_slx_int[  47].nbits = 12; gv_local.lookup_slx_int[  47].bits = 0x7D800000;
  gv_local.lookup_slx_int[  48].nbits = 12; gv_local.lookup_slx_int[  48].bits = 0x84800000;
  gv_local.lookup_slx_int[  49].nbits = 12; gv_local.lookup_slx_int[  49].bits = 0x86600000;
  gv_local.lookup_slx_int[  50].nbits = 12; gv_local.lookup_slx_int[  50].bits = 0x8DC00000;
  gv_local.lookup_slx_int[  51].nbits = 12; gv_local.lookup_slx_int[  51].bits = 0x8DD00000;
  gv_local.lookup_slx_int[  52].nbits = 12; gv_local.lookup_slx_int[  52].bits = 0x91100000;
  gv_local.lookup_slx_int[  53].nbits = 12; gv_local.lookup_slx_int[  53].bits = 0x95000000;
  gv_local.lookup_slx_int[  54].nbits = 12; gv_local.lookup_slx_int[  54].bits = 0x9CA00000;
  gv_local.lookup_slx_int[  55].nbits = 12; gv_local.lookup_slx_int[  55].bits = 0x9DE00000;
  gv_local.lookup_slx_int[  56].nbits = 12; gv_local.lookup_slx_int[  56].bits = 0x9CB00000;
  gv_local.lookup_slx_int[  57].nbits = 12; gv_local.lookup_slx_int[  57].bits = 0xA3200000;
  gv_local.lookup_slx_int[  58].nbits = 12; gv_local.lookup_slx_int[  58].bits = 0xA5E00000;
  gv_local.lookup_slx_int[  59].nbits = 12; gv_local.lookup_slx_int[  59].bits = 0xA9500000;
  gv_local.lookup_slx_int[  60].nbits = 12; gv_local.lookup_slx_int[  60].bits = 0xAF400000;
  gv_local.lookup_slx_int[  61].nbits = 12; gv_local.lookup_slx_int[  61].bits = 0xAF700000;
  gv_local.lookup_slx_int[  62].nbits = 12; gv_local.lookup_slx_int[  62].bits = 0xB6900000;
  gv_local.lookup_slx_int[  63].nbits = 12; gv_local.lookup_slx_int[  63].bits = 0xB6100000;
  gv_local.lookup_slx_int[  64].nbits = 12; gv_local.lookup_slx_int[  64].bits = 0xBBB00000;
  gv_local.lookup_slx_int[  65].nbits = 12; gv_local.lookup_slx_int[  65].bits = 0xBD900000;
  gv_local.lookup_slx_int[  66].nbits = 12; gv_local.lookup_slx_int[  66].bits = 0xC1A00000;
  gv_local.lookup_slx_int[  67].nbits = 12; gv_local.lookup_slx_int[  67].bits = 0xC1E00000;
  gv_local.lookup_slx_int[  68].nbits = 12; gv_local.lookup_slx_int[  68].bits = 0xC9400000;
  gv_local.lookup_slx_int[  69].nbits = 12; gv_local.lookup_slx_int[  69].bits = 0xCA500000;
  gv_local.lookup_slx_int[  70].nbits = 12; gv_local.lookup_slx_int[  70].bits = 0xCB600000;
  gv_local.lookup_slx_int[  71].nbits = 12; gv_local.lookup_slx_int[  71].bits = 0xCB700000;
  gv_local.lookup_slx_int[  72].nbits = 12; gv_local.lookup_slx_int[  72].bits = 0xCFF00000;
  gv_local.lookup_slx_int[  73].nbits = 12; gv_local.lookup_slx_int[  73].bits = 0xD9900000;
  gv_local.lookup_slx_int[  74].nbits = 12; gv_local.lookup_slx_int[  74].bits = 0xD9800000;
  gv_local.lookup_slx_int[  75].nbits = 12; gv_local.lookup_slx_int[  75].bits = 0xDA300000;
  gv_local.lookup_slx_int[  76].nbits = 11; gv_local.lookup_slx_int[  76].bits = 0x02800000;
  gv_local.lookup_slx_int[  77].nbits = 11; gv_local.lookup_slx_int[  77].bits = 0x02A00000;
  gv_local.lookup_slx_int[  78].nbits = 11; gv_local.lookup_slx_int[  78].bits = 0x0C600000;
  gv_local.lookup_slx_int[  79].nbits = 11; gv_local.lookup_slx_int[  79].bits = 0x09800000;
  gv_local.lookup_slx_int[  80].nbits = 11; gv_local.lookup_slx_int[  80].bits = 0x0D600000;
  gv_local.lookup_slx_int[  81].nbits = 11; gv_local.lookup_slx_int[  81].bits = 0x11200000;
  gv_local.lookup_slx_int[  82].nbits = 11; gv_local.lookup_slx_int[  82].bits = 0x1AA00000;
  gv_local.lookup_slx_int[  83].nbits = 11; gv_local.lookup_slx_int[  83].bits = 0x27800000;
  gv_local.lookup_slx_int[  84].nbits = 11; gv_local.lookup_slx_int[  84].bits = 0x29400000;
  gv_local.lookup_slx_int[  85].nbits = 11; gv_local.lookup_slx_int[  85].bits = 0x29A00000;
  gv_local.lookup_slx_int[  86].nbits = 11; gv_local.lookup_slx_int[  86].bits = 0x32800000;
  gv_local.lookup_slx_int[  87].nbits = 11; gv_local.lookup_slx_int[  87].bits = 0x3C600000;
  gv_local.lookup_slx_int[  88].nbits = 11; gv_local.lookup_slx_int[  88].bits = 0x42000000;
  gv_local.lookup_slx_int[  89].nbits = 11; gv_local.lookup_slx_int[  89].bits = 0x42E00000;
  gv_local.lookup_slx_int[  90].nbits = 11; gv_local.lookup_slx_int[  90].bits = 0x4C200000;
  gv_local.lookup_slx_int[  91].nbits = 11; gv_local.lookup_slx_int[  91].bits = 0x4E400000;
  gv_local.lookup_slx_int[  92].nbits = 11; gv_local.lookup_slx_int[  92].bits = 0x58600000;
  gv_local.lookup_slx_int[  93].nbits = 11; gv_local.lookup_slx_int[  93].bits = 0x58400000;
  gv_local.lookup_slx_int[  94].nbits = 11; gv_local.lookup_slx_int[  94].bits = 0x58A00000;
  gv_local.lookup_slx_int[  95].nbits = 11; gv_local.lookup_slx_int[  95].bits = 0x63E00000;
  gv_local.lookup_slx_int[  96].nbits = 11; gv_local.lookup_slx_int[  96].bits = 0x65800000;
  gv_local.lookup_slx_int[  97].nbits = 11; gv_local.lookup_slx_int[  97].bits = 0x6D200000;
  gv_local.lookup_slx_int[  98].nbits = 11; gv_local.lookup_slx_int[  98].bits = 0x6D400000;
  gv_local.lookup_slx_int[  99].nbits = 11; gv_local.lookup_slx_int[  99].bits = 0x75C00000;
  gv_local.lookup_slx_int[ 100].nbits = 11; gv_local.lookup_slx_int[ 100].bits = 0x75E00000;
  gv_local.lookup_slx_int[ 101].nbits = 11; gv_local.lookup_slx_int[ 101].bits = 0x84A00000;
  gv_local.lookup_slx_int[ 102].nbits = 11; gv_local.lookup_slx_int[ 102].bits = 0x87200000;
  gv_local.lookup_slx_int[ 103].nbits = 11; gv_local.lookup_slx_int[ 103].bits = 0x87E00000;
  gv_local.lookup_slx_int[ 104].nbits = 11; gv_local.lookup_slx_int[ 104].bits = 0x8DE00000;
  gv_local.lookup_slx_int[ 105].nbits = 11; gv_local.lookup_slx_int[ 105].bits = 0x90400000;
  gv_local.lookup_slx_int[ 106].nbits = 11; gv_local.lookup_slx_int[ 106].bits = 0x91E00000;
  gv_local.lookup_slx_int[ 107].nbits = 11; gv_local.lookup_slx_int[ 107].bits = 0x9CC00000;
  gv_local.lookup_slx_int[ 108].nbits = 11; gv_local.lookup_slx_int[ 108].bits = 0x9CE00000;
  gv_local.lookup_slx_int[ 109].nbits = 11; gv_local.lookup_slx_int[ 109].bits = 0xA3000000;
  gv_local.lookup_slx_int[ 110].nbits = 11; gv_local.lookup_slx_int[ 110].bits = 0xA3A00000;
  gv_local.lookup_slx_int[ 111].nbits = 11; gv_local.lookup_slx_int[ 111].bits = 0xA4800000;
  gv_local.lookup_slx_int[ 112].nbits = 11; gv_local.lookup_slx_int[ 112].bits = 0xA4E00000;
  gv_local.lookup_slx_int[ 113].nbits = 11; gv_local.lookup_slx_int[ 113].bits = 0xA7E00000;
  gv_local.lookup_slx_int[ 114].nbits = 11; gv_local.lookup_slx_int[ 114].bits = 0xAE000000;
  gv_local.lookup_slx_int[ 115].nbits = 11; gv_local.lookup_slx_int[ 115].bits = 0xB1A00000;
  gv_local.lookup_slx_int[ 116].nbits = 11; gv_local.lookup_slx_int[ 116].bits = 0xB3A00000;
  gv_local.lookup_slx_int[ 117].nbits = 11; gv_local.lookup_slx_int[ 117].bits = 0xB6A00000;
  gv_local.lookup_slx_int[ 118].nbits = 11; gv_local.lookup_slx_int[ 118].bits = 0xBD000000;
  gv_local.lookup_slx_int[ 119].nbits = 11; gv_local.lookup_slx_int[ 119].bits = 0xBA600000;
  gv_local.lookup_slx_int[ 120].nbits = 11; gv_local.lookup_slx_int[ 120].bits = 0xBF000000;
  gv_local.lookup_slx_int[ 121].nbits = 11; gv_local.lookup_slx_int[ 121].bits = 0xBF200000;
  gv_local.lookup_slx_int[ 122].nbits = 11; gv_local.lookup_slx_int[ 122].bits = 0xC1800000;
  gv_local.lookup_slx_int[ 123].nbits = 11; gv_local.lookup_slx_int[ 123].bits = 0xC1C00000;
  gv_local.lookup_slx_int[ 124].nbits = 11; gv_local.lookup_slx_int[ 124].bits = 0xC9600000;
  gv_local.lookup_slx_int[ 125].nbits = 11; gv_local.lookup_slx_int[ 125].bits = 0xCA600000;
  gv_local.lookup_slx_int[ 126].nbits = 11; gv_local.lookup_slx_int[ 126].bits = 0xCD000000;
  gv_local.lookup_slx_int[ 127].nbits = 11; gv_local.lookup_slx_int[ 127].bits = 0xCF800000;
  gv_local.lookup_slx_int[ 128].nbits = 11; gv_local.lookup_slx_int[ 128].bits = 0xCF400000;
  gv_local.lookup_slx_int[ 129].nbits = 11; gv_local.lookup_slx_int[ 129].bits = 0xD2200000;
  gv_local.lookup_slx_int[ 130].nbits = 11; gv_local.lookup_slx_int[ 130].bits = 0xD6800000;
  gv_local.lookup_slx_int[ 131].nbits = 11; gv_local.lookup_slx_int[ 131].bits = 0xD6A00000;
  gv_local.lookup_slx_int[ 132].nbits = 11; gv_local.lookup_slx_int[ 132].bits = 0xD9A00000;
  gv_local.lookup_slx_int[ 133].nbits = 11; gv_local.lookup_slx_int[ 133].bits = 0xDBA00000;
  gv_local.lookup_slx_int[ 134].nbits = 11; gv_local.lookup_slx_int[ 134].bits = 0xDBE00000;
  gv_local.lookup_slx_int[ 135].nbits = 10; gv_local.lookup_slx_int[ 135].bits = 0x08400000;
  gv_local.lookup_slx_int[ 136].nbits = 10; gv_local.lookup_slx_int[ 136].bits = 0x08C00000;
  gv_local.lookup_slx_int[ 137].nbits = 10; gv_local.lookup_slx_int[ 137].bits = 0x0A000000;
  gv_local.lookup_slx_int[ 138].nbits = 10; gv_local.lookup_slx_int[ 138].bits = 0x0C000000;
  gv_local.lookup_slx_int[ 139].nbits = 10; gv_local.lookup_slx_int[ 139].bits = 0x0D000000;
  gv_local.lookup_slx_int[ 140].nbits = 10; gv_local.lookup_slx_int[ 140].bits = 0x18000000;
  gv_local.lookup_slx_int[ 141].nbits = 10; gv_local.lookup_slx_int[ 141].bits = 0x1AC00000;
  gv_local.lookup_slx_int[ 142].nbits = 10; gv_local.lookup_slx_int[ 142].bits = 0x21400000;
  gv_local.lookup_slx_int[ 143].nbits = 10; gv_local.lookup_slx_int[ 143].bits = 0x21800000;
  gv_local.lookup_slx_int[ 144].nbits = 10; gv_local.lookup_slx_int[ 144].bits = 0x26000000;
  gv_local.lookup_slx_int[ 145].nbits = 10; gv_local.lookup_slx_int[ 145].bits = 0x2B800000;
  gv_local.lookup_slx_int[ 146].nbits = 10; gv_local.lookup_slx_int[ 146].bits = 0x32C00000;
  gv_local.lookup_slx_int[ 147].nbits = 10; gv_local.lookup_slx_int[ 147].bits = 0x34800000;
  gv_local.lookup_slx_int[ 148].nbits = 10; gv_local.lookup_slx_int[ 148].bits = 0x3CC00000;
  gv_local.lookup_slx_int[ 149].nbits = 10; gv_local.lookup_slx_int[ 149].bits = 0x3E800000;
  gv_local.lookup_slx_int[ 150].nbits = 10; gv_local.lookup_slx_int[ 150].bits = 0x49C00000;
  gv_local.lookup_slx_int[ 151].nbits = 10; gv_local.lookup_slx_int[ 151].bits = 0x48C00000;
  gv_local.lookup_slx_int[ 152].nbits = 10; gv_local.lookup_slx_int[ 152].bits = 0x49800000;
  gv_local.lookup_slx_int[ 153].nbits = 10; gv_local.lookup_slx_int[ 153].bits = 0x4E800000;
  gv_local.lookup_slx_int[ 154].nbits = 10; gv_local.lookup_slx_int[ 154].bits = 0x56400000;
  gv_local.lookup_slx_int[ 155].nbits = 10; gv_local.lookup_slx_int[ 155].bits = 0x5BC00000;
  gv_local.lookup_slx_int[ 156].nbits = 10; gv_local.lookup_slx_int[ 156].bits = 0x59400000;
  gv_local.lookup_slx_int[ 157].nbits = 10; gv_local.lookup_slx_int[ 157].bits = 0x63800000;
  gv_local.lookup_slx_int[ 158].nbits = 10; gv_local.lookup_slx_int[ 158].bits = 0x64C00000;
  gv_local.lookup_slx_int[ 159].nbits = 10; gv_local.lookup_slx_int[ 159].bits = 0x6F400000;
  gv_local.lookup_slx_int[ 160].nbits = 10; gv_local.lookup_slx_int[ 160].bits = 0x6FC00000;
  gv_local.lookup_slx_int[ 161].nbits = 10; gv_local.lookup_slx_int[ 161].bits = 0x75400000;
  gv_local.lookup_slx_int[ 162].nbits = 10; gv_local.lookup_slx_int[ 162].bits = 0x84C00000;
  gv_local.lookup_slx_int[ 163].nbits = 10; gv_local.lookup_slx_int[ 163].bits = 0x86800000;
  gv_local.lookup_slx_int[ 164].nbits = 10; gv_local.lookup_slx_int[ 164].bits = 0x87400000;
  gv_local.lookup_slx_int[ 165].nbits = 10; gv_local.lookup_slx_int[ 165].bits = 0x87800000;
  gv_local.lookup_slx_int[ 166].nbits = 10; gv_local.lookup_slx_int[ 166].bits = 0x8D800000;
  gv_local.lookup_slx_int[ 167].nbits = 10; gv_local.lookup_slx_int[ 167].bits = 0x93C00000;
  gv_local.lookup_slx_int[ 168].nbits = 10; gv_local.lookup_slx_int[ 168].bits = 0x95400000;
  gv_local.lookup_slx_int[ 169].nbits = 10; gv_local.lookup_slx_int[ 169].bits = 0x9C000000;
  gv_local.lookup_slx_int[ 170].nbits = 10; gv_local.lookup_slx_int[ 170].bits = 0xA2400000;
  gv_local.lookup_slx_int[ 171].nbits = 10; gv_local.lookup_slx_int[ 171].bits = 0xA3C00000;
  gv_local.lookup_slx_int[ 172].nbits = 10; gv_local.lookup_slx_int[ 172].bits = 0xA5000000;
  gv_local.lookup_slx_int[ 173].nbits = 10; gv_local.lookup_slx_int[ 173].bits = 0xA7800000;
  gv_local.lookup_slx_int[ 174].nbits = 10; gv_local.lookup_slx_int[ 174].bits = 0xAF000000;
  gv_local.lookup_slx_int[ 175].nbits = 10; gv_local.lookup_slx_int[ 175].bits = 0xAF800000;
  gv_local.lookup_slx_int[ 176].nbits = 10; gv_local.lookup_slx_int[ 176].bits = 0xB3C00000;
  gv_local.lookup_slx_int[ 177].nbits = 10; gv_local.lookup_slx_int[ 177].bits = 0xB6C00000;
  gv_local.lookup_slx_int[ 178].nbits = 10; gv_local.lookup_slx_int[ 178].bits = 0xBA000000;
  gv_local.lookup_slx_int[ 179].nbits = 10; gv_local.lookup_slx_int[ 179].bits = 0xBF400000;
  gv_local.lookup_slx_int[ 180].nbits = 10; gv_local.lookup_slx_int[ 180].bits = 0xC1000000;
  gv_local.lookup_slx_int[ 181].nbits = 10; gv_local.lookup_slx_int[ 181].bits = 0xC5000000;
  gv_local.lookup_slx_int[ 182].nbits = 10; gv_local.lookup_slx_int[ 182].bits = 0xCA000000;
  gv_local.lookup_slx_int[ 183].nbits = 10; gv_local.lookup_slx_int[ 183].bits = 0xCB000000;
  gv_local.lookup_slx_int[ 184].nbits = 10; gv_local.lookup_slx_int[ 184].bits = 0xD2400000;
  gv_local.lookup_slx_int[ 185].nbits = 10; gv_local.lookup_slx_int[ 185].bits = 0xD3800000;
  gv_local.lookup_slx_int[ 186].nbits = 10; gv_local.lookup_slx_int[ 186].bits = 0xD6C00000;
  gv_local.lookup_slx_int[ 187].nbits = 10; gv_local.lookup_slx_int[ 187].bits = 0xDB000000;
  gv_local.lookup_slx_int[ 188].nbits = 10; gv_local.lookup_slx_int[ 188].bits = 0xDEC00000;
  gv_local.lookup_slx_int[ 189].nbits =  9; gv_local.lookup_slx_int[ 189].bits = 0x09000000;
  gv_local.lookup_slx_int[ 190].nbits =  9; gv_local.lookup_slx_int[ 190].bits = 0x0C800000;
  gv_local.lookup_slx_int[ 191].nbits =  9; gv_local.lookup_slx_int[ 191].bits = 0x18800000;
  gv_local.lookup_slx_int[ 192].nbits =  9; gv_local.lookup_slx_int[ 192].bits = 0x1B800000;
  gv_local.lookup_slx_int[ 193].nbits =  9; gv_local.lookup_slx_int[ 193].bits = 0x27000000;
  gv_local.lookup_slx_int[ 194].nbits =  9; gv_local.lookup_slx_int[ 194].bits = 0x32000000;
  gv_local.lookup_slx_int[ 195].nbits =  9; gv_local.lookup_slx_int[ 195].bits = 0x3D000000;
  gv_local.lookup_slx_int[ 196].nbits =  9; gv_local.lookup_slx_int[ 196].bits = 0x49000000;
  gv_local.lookup_slx_int[ 197].nbits =  9; gv_local.lookup_slx_int[ 197].bits = 0x4C800000;
  gv_local.lookup_slx_int[ 198].nbits =  9; gv_local.lookup_slx_int[ 198].bits = 0x56800000;
  gv_local.lookup_slx_int[ 199].nbits =  9; gv_local.lookup_slx_int[ 199].bits = 0x5B000000;
  gv_local.lookup_slx_int[ 200].nbits =  9; gv_local.lookup_slx_int[ 200].bits = 0x64000000;
  gv_local.lookup_slx_int[ 201].nbits =  9; gv_local.lookup_slx_int[ 201].bits = 0x74800000;
  gv_local.lookup_slx_int[ 202].nbits =  9; gv_local.lookup_slx_int[ 202].bits = 0x7D000000;
  gv_local.lookup_slx_int[ 203].nbits =  9; gv_local.lookup_slx_int[ 203].bits = 0x89000000;
  gv_local.lookup_slx_int[ 204].nbits =  9; gv_local.lookup_slx_int[ 204].bits = 0x90800000;
  gv_local.lookup_slx_int[ 205].nbits =  9; gv_local.lookup_slx_int[ 205].bits = 0x95800000;
  gv_local.lookup_slx_int[ 206].nbits =  9; gv_local.lookup_slx_int[ 206].bits = 0xA2800000;
  gv_local.lookup_slx_int[ 207].nbits =  9; gv_local.lookup_slx_int[ 207].bits = 0xA7000000;
  gv_local.lookup_slx_int[ 208].nbits =  9; gv_local.lookup_slx_int[ 208].bits = 0xAE800000;
  gv_local.lookup_slx_int[ 209].nbits =  9; gv_local.lookup_slx_int[ 209].bits = 0xB3000000;
  gv_local.lookup_slx_int[ 210].nbits =  9; gv_local.lookup_slx_int[ 210].bits = 0xBB000000;
  gv_local.lookup_slx_int[ 211].nbits =  9; gv_local.lookup_slx_int[ 211].bits = 0xBF800000;
  gv_local.lookup_slx_int[ 212].nbits =  9; gv_local.lookup_slx_int[ 212].bits = 0xC5800000;
  gv_local.lookup_slx_int[ 213].nbits =  9; gv_local.lookup_slx_int[ 213].bits = 0xCB800000;
  gv_local.lookup_slx_int[ 214].nbits =  9; gv_local.lookup_slx_int[ 214].bits = 0xD3000000;
  gv_local.lookup_slx_int[ 215].nbits =  9; gv_local.lookup_slx_int[ 215].bits = 0xD9000000;
  gv_local.lookup_slx_int[ 216].nbits =  9; gv_local.lookup_slx_int[ 216].bits = 0xDE000000;
  gv_local.lookup_slx_int[ 217].nbits =  8; gv_local.lookup_slx_int[ 217].bits = 0x0B000000;
  gv_local.lookup_slx_int[ 218].nbits =  8; gv_local.lookup_slx_int[ 218].bits = 0x19000000;
  gv_local.lookup_slx_int[ 219].nbits =  8; gv_local.lookup_slx_int[ 219].bits = 0x28000000;
  gv_local.lookup_slx_int[ 220].nbits =  8; gv_local.lookup_slx_int[ 220].bits = 0x33000000;
  gv_local.lookup_slx_int[ 221].nbits =  8; gv_local.lookup_slx_int[ 221].bits = 0x3F000000;
  gv_local.lookup_slx_int[ 222].nbits =  8; gv_local.lookup_slx_int[ 222].bits = 0x4D000000;
  gv_local.lookup_slx_int[ 223].nbits =  8; gv_local.lookup_slx_int[ 223].bits = 0x5A000000;
  gv_local.lookup_slx_int[ 224].nbits =  8; gv_local.lookup_slx_int[ 224].bits = 0x6C000000;
  gv_local.lookup_slx_int[ 225].nbits =  8; gv_local.lookup_slx_int[ 225].bits = 0x7C000000;
  gv_local.lookup_slx_int[ 226].nbits =  8; gv_local.lookup_slx_int[ 226].bits = 0x88000000;
  gv_local.lookup_slx_int[ 227].nbits =  8; gv_local.lookup_slx_int[ 227].bits = 0x94000000;
  gv_local.lookup_slx_int[ 228].nbits =  8; gv_local.lookup_slx_int[ 228].bits = 0xA1000000;
  gv_local.lookup_slx_int[ 229].nbits =  8; gv_local.lookup_slx_int[ 229].bits = 0xA8000000;
  gv_local.lookup_slx_int[ 230].nbits =  8; gv_local.lookup_slx_int[ 230].bits = 0xB2000000;
  gv_local.lookup_slx_int[ 231].nbits =  8; gv_local.lookup_slx_int[ 231].bits = 0xBC000000;
  gv_local.lookup_slx_int[ 232].nbits =  8; gv_local.lookup_slx_int[ 232].bits = 0xC4000000;
  gv_local.lookup_slx_int[ 233].nbits =  8; gv_local.lookup_slx_int[ 233].bits = 0xCC000000;
  gv_local.lookup_slx_int[ 234].nbits =  8; gv_local.lookup_slx_int[ 234].bits = 0xD7000000;
  gv_local.lookup_slx_int[ 235].nbits =  7; gv_local.lookup_slx_int[ 235].bits = 0x00000000;
  gv_local.lookup_slx_int[ 236].nbits =  7; gv_local.lookup_slx_int[ 236].bits = 0x12000000;
  gv_local.lookup_slx_int[ 237].nbits =  7; gv_local.lookup_slx_int[ 237].bits = 0x24000000;
  gv_local.lookup_slx_int[ 238].nbits =  7; gv_local.lookup_slx_int[ 238].bits = 0x36000000;
  gv_local.lookup_slx_int[ 239].nbits =  7; gv_local.lookup_slx_int[ 239].bits = 0x4A000000;
  gv_local.lookup_slx_int[ 240].nbits =  7; gv_local.lookup_slx_int[ 240].bits = 0x60000000;
  gv_local.lookup_slx_int[ 241].nbits =  7; gv_local.lookup_slx_int[ 241].bits = 0x76000000;
  gv_local.lookup_slx_int[ 242].nbits =  7; gv_local.lookup_slx_int[ 242].bits = 0x8A000000;
  gv_local.lookup_slx_int[ 243].nbits =  7; gv_local.lookup_slx_int[ 243].bits = 0x96000000;
  gv_local.lookup_slx_int[ 244].nbits =  7; gv_local.lookup_slx_int[ 244].bits = 0xAA000000;
  gv_local.lookup_slx_int[ 245].nbits =  7; gv_local.lookup_slx_int[ 245].bits = 0xB4000000;
  gv_local.lookup_slx_int[ 246].nbits =  7; gv_local.lookup_slx_int[ 246].bits = 0xC2000000;
  gv_local.lookup_slx_int[ 247].nbits =  7; gv_local.lookup_slx_int[ 247].bits = 0xD0000000;
  gv_local.lookup_slx_int[ 248].nbits =  7; gv_local.lookup_slx_int[ 248].bits = 0xDC000000;
  gv_local.lookup_slx_int[ 249].nbits =  6; gv_local.lookup_slx_int[ 249].bits = 0x14000000;
  gv_local.lookup_slx_int[ 250].nbits =  6; gv_local.lookup_slx_int[ 250].bits = 0x2C000000;
  gv_local.lookup_slx_int[ 251].nbits =  6; gv_local.lookup_slx_int[ 251].bits = 0x44000000;
  gv_local.lookup_slx_int[ 252].nbits =  6; gv_local.lookup_slx_int[ 252].bits = 0x5C000000;
  gv_local.lookup_slx_int[ 253].nbits =  6; gv_local.lookup_slx_int[ 253].bits = 0x70000000;
  gv_local.lookup_slx_int[ 254].nbits =  6; gv_local.lookup_slx_int[ 254].bits = 0x80000000;
  gv_local.lookup_slx_int[ 255].nbits =  6; gv_local.lookup_slx_int[ 255].bits = 0x98000000;
  gv_local.lookup_slx_int[ 256].nbits =  6; gv_local.lookup_slx_int[ 256].bits = 0x78000000;
  gv_local.lookup_slx_int[ 257].nbits =  6; gv_local.lookup_slx_int[ 257].bits = 0x68000000;
  gv_local.lookup_slx_int[ 258].nbits =  6; gv_local.lookup_slx_int[ 258].bits = 0x50000000;
  gv_local.lookup_slx_int[ 259].nbits =  6; gv_local.lookup_slx_int[ 259].bits = 0x38000000;
  gv_local.lookup_slx_int[ 260].nbits =  6; gv_local.lookup_slx_int[ 260].bits = 0x1C000000;
  gv_local.lookup_slx_int[ 261].nbits =  6; gv_local.lookup_slx_int[ 261].bits = 0x04000000;
  gv_local.lookup_slx_int[ 262].nbits =  7; gv_local.lookup_slx_int[ 262].bits = 0xD4000000;
  gv_local.lookup_slx_int[ 263].nbits =  7; gv_local.lookup_slx_int[ 263].bits = 0xC6000000;
  gv_local.lookup_slx_int[ 264].nbits =  7; gv_local.lookup_slx_int[ 264].bits = 0xB8000000;
  gv_local.lookup_slx_int[ 265].nbits =  7; gv_local.lookup_slx_int[ 265].bits = 0xAC000000;
  gv_local.lookup_slx_int[ 266].nbits =  7; gv_local.lookup_slx_int[ 266].bits = 0x9E000000;
  gv_local.lookup_slx_int[ 267].nbits =  7; gv_local.lookup_slx_int[ 267].bits = 0x8E000000;
  gv_local.lookup_slx_int[ 268].nbits =  7; gv_local.lookup_slx_int[ 268].bits = 0x7E000000;
  gv_local.lookup_slx_int[ 269].nbits =  7; gv_local.lookup_slx_int[ 269].bits = 0x66000000;
  gv_local.lookup_slx_int[ 270].nbits =  7; gv_local.lookup_slx_int[ 270].bits = 0x54000000;
  gv_local.lookup_slx_int[ 271].nbits =  7; gv_local.lookup_slx_int[ 271].bits = 0x40000000;
  gv_local.lookup_slx_int[ 272].nbits =  7; gv_local.lookup_slx_int[ 272].bits = 0x30000000;
  gv_local.lookup_slx_int[ 273].nbits =  7; gv_local.lookup_slx_int[ 273].bits = 0x22000000;
  gv_local.lookup_slx_int[ 274].nbits =  7; gv_local.lookup_slx_int[ 274].bits = 0x0E000000;
  gv_local.lookup_slx_int[ 275].nbits =  8; gv_local.lookup_slx_int[ 275].bits = 0xDF000000;
  gv_local.lookup_slx_int[ 276].nbits =  8; gv_local.lookup_slx_int[ 276].bits = 0xD8000000;
  gv_local.lookup_slx_int[ 277].nbits =  8; gv_local.lookup_slx_int[ 277].bits = 0xCE000000;
  gv_local.lookup_slx_int[ 278].nbits =  8; gv_local.lookup_slx_int[ 278].bits = 0xC8000000;
  gv_local.lookup_slx_int[ 279].nbits =  8; gv_local.lookup_slx_int[ 279].bits = 0xBE000000;
  gv_local.lookup_slx_int[ 280].nbits =  8; gv_local.lookup_slx_int[ 280].bits = 0xB7000000;
  gv_local.lookup_slx_int[ 281].nbits =  8; gv_local.lookup_slx_int[ 281].bits = 0xB0000000;
  gv_local.lookup_slx_int[ 282].nbits =  8; gv_local.lookup_slx_int[ 282].bits = 0xA6000000;
  gv_local.lookup_slx_int[ 283].nbits =  8; gv_local.lookup_slx_int[ 283].bits = 0xA0000000;
  gv_local.lookup_slx_int[ 284].nbits =  8; gv_local.lookup_slx_int[ 284].bits = 0x92000000;
  gv_local.lookup_slx_int[ 285].nbits =  8; gv_local.lookup_slx_int[ 285].bits = 0x8C000000;
  gv_local.lookup_slx_int[ 286].nbits =  8; gv_local.lookup_slx_int[ 286].bits = 0x85000000;
  gv_local.lookup_slx_int[ 287].nbits =  8; gv_local.lookup_slx_int[ 287].bits = 0x6E000000;
  gv_local.lookup_slx_int[ 288].nbits =  8; gv_local.lookup_slx_int[ 288].bits = 0x62000000;
  gv_local.lookup_slx_int[ 289].nbits =  8; gv_local.lookup_slx_int[ 289].bits = 0x57000000;
  gv_local.lookup_slx_int[ 290].nbits =  8; gv_local.lookup_slx_int[ 290].bits = 0x43000000;
  gv_local.lookup_slx_int[ 291].nbits =  8; gv_local.lookup_slx_int[ 291].bits = 0x35000000;
  gv_local.lookup_slx_int[ 292].nbits =  8; gv_local.lookup_slx_int[ 292].bits = 0x2A000000;
  gv_local.lookup_slx_int[ 293].nbits =  8; gv_local.lookup_slx_int[ 293].bits = 0x20000000;
  gv_local.lookup_slx_int[ 294].nbits =  8; gv_local.lookup_slx_int[ 294].bits = 0x10000000;
  gv_local.lookup_slx_int[ 295].nbits =  8; gv_local.lookup_slx_int[ 295].bits = 0x03000000;
  gv_local.lookup_slx_int[ 296].nbits =  9; gv_local.lookup_slx_int[ 296].bits = 0xDA800000;
  gv_local.lookup_slx_int[ 297].nbits =  9; gv_local.lookup_slx_int[ 297].bits = 0xD6000000;
  gv_local.lookup_slx_int[ 298].nbits =  9; gv_local.lookup_slx_int[ 298].bits = 0xCD800000;
  gv_local.lookup_slx_int[ 299].nbits =  9; gv_local.lookup_slx_int[ 299].bits = 0xC9800000;
  gv_local.lookup_slx_int[ 300].nbits =  9; gv_local.lookup_slx_int[ 300].bits = 0xC0800000;
  gv_local.lookup_slx_int[ 301].nbits =  9; gv_local.lookup_slx_int[ 301].bits = 0xBA800000;
  gv_local.lookup_slx_int[ 302].nbits =  9; gv_local.lookup_slx_int[ 302].bits = 0xB1000000;
  gv_local.lookup_slx_int[ 303].nbits =  9; gv_local.lookup_slx_int[ 303].bits = 0xA9800000;
  gv_local.lookup_slx_int[ 304].nbits =  9; gv_local.lookup_slx_int[ 304].bits = 0xA4000000;
  gv_local.lookup_slx_int[ 305].nbits =  9; gv_local.lookup_slx_int[ 305].bits = 0x9D000000;
  gv_local.lookup_slx_int[ 306].nbits =  9; gv_local.lookup_slx_int[ 306].bits = 0x93000000;
  gv_local.lookup_slx_int[ 307].nbits =  9; gv_local.lookup_slx_int[ 307].bits = 0x8D000000;
  gv_local.lookup_slx_int[ 308].nbits =  9; gv_local.lookup_slx_int[ 308].bits = 0x84000000;
  gv_local.lookup_slx_int[ 309].nbits =  9; gv_local.lookup_slx_int[ 309].bits = 0x74000000;
  gv_local.lookup_slx_int[ 310].nbits =  9; gv_local.lookup_slx_int[ 310].bits = 0x65000000;
  gv_local.lookup_slx_int[ 311].nbits =  9; gv_local.lookup_slx_int[ 311].bits = 0x59800000;
  gv_local.lookup_slx_int[ 312].nbits =  9; gv_local.lookup_slx_int[ 312].bits = 0x4F800000;
  gv_local.lookup_slx_int[ 313].nbits =  9; gv_local.lookup_slx_int[ 313].bits = 0x48000000;
  gv_local.lookup_slx_int[ 314].nbits =  9; gv_local.lookup_slx_int[ 314].bits = 0x3D800000;
  gv_local.lookup_slx_int[ 315].nbits =  9; gv_local.lookup_slx_int[ 315].bits = 0x34000000;
  gv_local.lookup_slx_int[ 316].nbits =  9; gv_local.lookup_slx_int[ 316].bits = 0x26800000;
  gv_local.lookup_slx_int[ 317].nbits =  9; gv_local.lookup_slx_int[ 317].bits = 0x1B000000;
  gv_local.lookup_slx_int[ 318].nbits =  9; gv_local.lookup_slx_int[ 318].bits = 0x11800000;
  gv_local.lookup_slx_int[ 319].nbits =  9; gv_local.lookup_slx_int[ 319].bits = 0x0A800000;
  gv_local.lookup_slx_int[ 320].nbits =  9; gv_local.lookup_slx_int[ 320].bits = 0x02000000;
  gv_local.lookup_slx_int[ 321].nbits = 10; gv_local.lookup_slx_int[ 321].bits = 0xDB400000;
  gv_local.lookup_slx_int[ 322].nbits = 10; gv_local.lookup_slx_int[ 322].bits = 0xD9C00000;
  gv_local.lookup_slx_int[ 323].nbits = 10; gv_local.lookup_slx_int[ 323].bits = 0xD2800000;
  gv_local.lookup_slx_int[ 324].nbits = 10; gv_local.lookup_slx_int[ 324].bits = 0xCF000000;
  gv_local.lookup_slx_int[ 325].nbits = 10; gv_local.lookup_slx_int[ 325].bits = 0xCD400000;
  gv_local.lookup_slx_int[ 326].nbits = 10; gv_local.lookup_slx_int[ 326].bits = 0xCA800000;
  gv_local.lookup_slx_int[ 327].nbits = 10; gv_local.lookup_slx_int[ 327].bits = 0xC5400000;
  gv_local.lookup_slx_int[ 328].nbits = 10; gv_local.lookup_slx_int[ 328].bits = 0xC0000000;
  gv_local.lookup_slx_int[ 329].nbits = 10; gv_local.lookup_slx_int[ 329].bits = 0xBD400000;
  gv_local.lookup_slx_int[ 330].nbits = 10; gv_local.lookup_slx_int[ 330].bits = 0xBBC00000;
  gv_local.lookup_slx_int[ 331].nbits = 10; gv_local.lookup_slx_int[ 331].bits = 0xB6400000;
  gv_local.lookup_slx_int[ 332].nbits = 10; gv_local.lookup_slx_int[ 332].bits = 0xB1C00000;
  gv_local.lookup_slx_int[ 333].nbits = 10; gv_local.lookup_slx_int[ 333].bits = 0xAFC00000;
  gv_local.lookup_slx_int[ 334].nbits = 10; gv_local.lookup_slx_int[ 334].bits = 0xAE400000;
  gv_local.lookup_slx_int[ 335].nbits = 10; gv_local.lookup_slx_int[ 335].bits = 0xA9000000;
  gv_local.lookup_slx_int[ 336].nbits = 10; gv_local.lookup_slx_int[ 336].bits = 0xA5800000;
  gv_local.lookup_slx_int[ 337].nbits = 10; gv_local.lookup_slx_int[ 337].bits = 0xA3400000;
  gv_local.lookup_slx_int[ 338].nbits = 10; gv_local.lookup_slx_int[ 338].bits = 0xA2000000;
  gv_local.lookup_slx_int[ 339].nbits = 10; gv_local.lookup_slx_int[ 339].bits = 0x9C400000;
  gv_local.lookup_slx_int[ 340].nbits = 10; gv_local.lookup_slx_int[ 340].bits = 0x93800000;
  gv_local.lookup_slx_int[ 341].nbits = 10; gv_local.lookup_slx_int[ 341].bits = 0x91800000;
  gv_local.lookup_slx_int[ 342].nbits = 10; gv_local.lookup_slx_int[ 342].bits = 0x91400000;
  gv_local.lookup_slx_int[ 343].nbits = 10; gv_local.lookup_slx_int[ 343].bits = 0x90000000;
  gv_local.lookup_slx_int[ 344].nbits = 10; gv_local.lookup_slx_int[ 344].bits = 0x89800000;
  gv_local.lookup_slx_int[ 345].nbits = 10; gv_local.lookup_slx_int[ 345].bits = 0x86C00000;
  gv_local.lookup_slx_int[ 346].nbits = 10; gv_local.lookup_slx_int[ 346].bits = 0x86000000;
  gv_local.lookup_slx_int[ 347].nbits = 10; gv_local.lookup_slx_int[ 347].bits = 0x7DC00000;
  gv_local.lookup_slx_int[ 348].nbits = 10; gv_local.lookup_slx_int[ 348].bits = 0x75000000;
  gv_local.lookup_slx_int[ 349].nbits = 10; gv_local.lookup_slx_int[ 349].bits = 0x6F800000;
  gv_local.lookup_slx_int[ 350].nbits = 10; gv_local.lookup_slx_int[ 350].bits = 0x6DC00000;
  gv_local.lookup_slx_int[ 351].nbits = 10; gv_local.lookup_slx_int[ 351].bits = 0x6D800000;
  gv_local.lookup_slx_int[ 352].nbits = 10; gv_local.lookup_slx_int[ 352].bits = 0x65C00000;
  gv_local.lookup_slx_int[ 353].nbits = 10; gv_local.lookup_slx_int[ 353].bits = 0x64800000;
  gv_local.lookup_slx_int[ 354].nbits = 10; gv_local.lookup_slx_int[ 354].bits = 0x63000000;
  gv_local.lookup_slx_int[ 355].nbits = 10; gv_local.lookup_slx_int[ 355].bits = 0x59000000;
  gv_local.lookup_slx_int[ 356].nbits = 10; gv_local.lookup_slx_int[ 356].bits = 0x58000000;
  gv_local.lookup_slx_int[ 357].nbits = 10; gv_local.lookup_slx_int[ 357].bits = 0x4F400000;
  gv_local.lookup_slx_int[ 358].nbits = 10; gv_local.lookup_slx_int[ 358].bits = 0x4EC00000;
  gv_local.lookup_slx_int[ 359].nbits = 10; gv_local.lookup_slx_int[ 359].bits = 0x56000000;
  gv_local.lookup_slx_int[ 360].nbits = 10; gv_local.lookup_slx_int[ 360].bits = 0x4E000000;
  gv_local.lookup_slx_int[ 361].nbits = 10; gv_local.lookup_slx_int[ 361].bits = 0x42800000;
  gv_local.lookup_slx_int[ 362].nbits = 10; gv_local.lookup_slx_int[ 362].bits = 0x42400000;
  gv_local.lookup_slx_int[ 363].nbits = 10; gv_local.lookup_slx_int[ 363].bits = 0x3EC00000;
  gv_local.lookup_slx_int[ 364].nbits = 10; gv_local.lookup_slx_int[ 364].bits = 0x3E000000;
  gv_local.lookup_slx_int[ 365].nbits = 10; gv_local.lookup_slx_int[ 365].bits = 0x3C800000;
  gv_local.lookup_slx_int[ 366].nbits = 10; gv_local.lookup_slx_int[ 366].bits = 0x34C00000;
  gv_local.lookup_slx_int[ 367].nbits = 10; gv_local.lookup_slx_int[ 367].bits = 0x2BC00000;
  gv_local.lookup_slx_int[ 368].nbits = 10; gv_local.lookup_slx_int[ 368].bits = 0x2B400000;
  gv_local.lookup_slx_int[ 369].nbits = 10; gv_local.lookup_slx_int[ 369].bits = 0x29000000;
  gv_local.lookup_slx_int[ 370].nbits = 10; gv_local.lookup_slx_int[ 370].bits = 0x26400000;
  gv_local.lookup_slx_int[ 371].nbits = 10; gv_local.lookup_slx_int[ 371].bits = 0x27C00000;
  gv_local.lookup_slx_int[ 372].nbits = 10; gv_local.lookup_slx_int[ 372].bits = 0x21C00000;
  gv_local.lookup_slx_int[ 373].nbits = 10; gv_local.lookup_slx_int[ 373].bits = 0x18400000;
  gv_local.lookup_slx_int[ 374].nbits = 10; gv_local.lookup_slx_int[ 374].bits = 0x1A000000;
  gv_local.lookup_slx_int[ 375].nbits = 10; gv_local.lookup_slx_int[ 375].bits = 0x11400000;
  gv_local.lookup_slx_int[ 376].nbits = 10; gv_local.lookup_slx_int[ 376].bits = 0x0DC00000;
  gv_local.lookup_slx_int[ 377].nbits = 10; gv_local.lookup_slx_int[ 377].bits = 0x0D800000;
  gv_local.lookup_slx_int[ 378].nbits = 10; gv_local.lookup_slx_int[ 378].bits = 0x0A400000;
  gv_local.lookup_slx_int[ 379].nbits = 10; gv_local.lookup_slx_int[ 379].bits = 0x09C00000;
  gv_local.lookup_slx_int[ 380].nbits = 10; gv_local.lookup_slx_int[ 380].bits = 0x08800000;
  gv_local.lookup_slx_int[ 381].nbits = 10; gv_local.lookup_slx_int[ 381].bits = 0x08000000;
  gv_local.lookup_slx_int[ 382].nbits = 11; gv_local.lookup_slx_int[ 382].bits = 0xDEA00000;
  gv_local.lookup_slx_int[ 383].nbits = 11; gv_local.lookup_slx_int[ 383].bits = 0xDBC00000;
  gv_local.lookup_slx_int[ 384].nbits = 11; gv_local.lookup_slx_int[ 384].bits = 0xDA400000;
  gv_local.lookup_slx_int[ 385].nbits = 11; gv_local.lookup_slx_int[ 385].bits = 0xDA600000;
  gv_local.lookup_slx_int[ 386].nbits = 11; gv_local.lookup_slx_int[ 386].bits = 0xDA000000;
  gv_local.lookup_slx_int[ 387].nbits = 11; gv_local.lookup_slx_int[ 387].bits = 0xD3C00000;
  gv_local.lookup_slx_int[ 388].nbits = 11; gv_local.lookup_slx_int[ 388].bits = 0xD2E00000;
  gv_local.lookup_slx_int[ 389].nbits = 11; gv_local.lookup_slx_int[ 389].bits = 0xD2000000;
  gv_local.lookup_slx_int[ 390].nbits = 11; gv_local.lookup_slx_int[ 390].bits = 0xCFC00000;
  gv_local.lookup_slx_int[ 391].nbits = 11; gv_local.lookup_slx_int[ 391].bits = 0xCD200000;
  gv_local.lookup_slx_int[ 392].nbits = 11; gv_local.lookup_slx_int[ 392].bits = 0xCFA00000;
  gv_local.lookup_slx_int[ 393].nbits = 11; gv_local.lookup_slx_int[ 393].bits = 0xCB400000;
  gv_local.lookup_slx_int[ 394].nbits = 11; gv_local.lookup_slx_int[ 394].bits = 0xCAC00000;
  gv_local.lookup_slx_int[ 395].nbits = 11; gv_local.lookup_slx_int[ 395].bits = 0xC9000000;
  gv_local.lookup_slx_int[ 396].nbits = 11; gv_local.lookup_slx_int[ 396].bits = 0xC9200000;
  gv_local.lookup_slx_int[ 397].nbits = 11; gv_local.lookup_slx_int[ 397].bits = 0xC1600000;
  gv_local.lookup_slx_int[ 398].nbits = 11; gv_local.lookup_slx_int[ 398].bits = 0xC1400000;
  gv_local.lookup_slx_int[ 399].nbits = 11; gv_local.lookup_slx_int[ 399].bits = 0xC0600000;
  gv_local.lookup_slx_int[ 400].nbits = 11; gv_local.lookup_slx_int[ 400].bits = 0xBDE00000;
  gv_local.lookup_slx_int[ 401].nbits = 11; gv_local.lookup_slx_int[ 401].bits = 0xBDA00000;
  gv_local.lookup_slx_int[ 402].nbits = 11; gv_local.lookup_slx_int[ 402].bits = 0xBD200000;
  gv_local.lookup_slx_int[ 403].nbits = 11; gv_local.lookup_slx_int[ 403].bits = 0xBB800000;
  gv_local.lookup_slx_int[ 404].nbits = 11; gv_local.lookup_slx_int[ 404].bits = 0xBA400000;
  gv_local.lookup_slx_int[ 405].nbits = 11; gv_local.lookup_slx_int[ 405].bits = 0xB3800000;
  gv_local.lookup_slx_int[ 406].nbits = 11; gv_local.lookup_slx_int[ 406].bits = 0xB1800000;
  gv_local.lookup_slx_int[ 407].nbits = 11; gv_local.lookup_slx_int[ 407].bits = 0xAE200000;
  gv_local.lookup_slx_int[ 408].nbits = 11; gv_local.lookup_slx_int[ 408].bits = 0xA9600000;
  gv_local.lookup_slx_int[ 409].nbits = 11; gv_local.lookup_slx_int[ 409].bits = 0xA5400000;
  gv_local.lookup_slx_int[ 410].nbits = 11; gv_local.lookup_slx_int[ 410].bits = 0xA5C00000;
  gv_local.lookup_slx_int[ 411].nbits = 11; gv_local.lookup_slx_int[ 411].bits = 0xA5600000;
  gv_local.lookup_slx_int[ 412].nbits = 11; gv_local.lookup_slx_int[ 412].bits = 0xA4C00000;
  gv_local.lookup_slx_int[ 413].nbits = 11; gv_local.lookup_slx_int[ 413].bits = 0xA3800000;
  gv_local.lookup_slx_int[ 414].nbits = 11; gv_local.lookup_slx_int[ 414].bits = 0x9DA00000;
  gv_local.lookup_slx_int[ 415].nbits = 11; gv_local.lookup_slx_int[ 415].bits = 0x9D800000;
  gv_local.lookup_slx_int[ 416].nbits = 11; gv_local.lookup_slx_int[ 416].bits = 0x95200000;
  gv_local.lookup_slx_int[ 417].nbits = 11; gv_local.lookup_slx_int[ 417].bits = 0x9C800000;
  gv_local.lookup_slx_int[ 418].nbits = 11; gv_local.lookup_slx_int[ 418].bits = 0x90600000;
  gv_local.lookup_slx_int[ 419].nbits = 11; gv_local.lookup_slx_int[ 419].bits = 0x91C00000;
  gv_local.lookup_slx_int[ 420].nbits = 11; gv_local.lookup_slx_int[ 420].bits = 0x89E00000;
  gv_local.lookup_slx_int[ 421].nbits = 11; gv_local.lookup_slx_int[ 421].bits = 0x89C00000;
  gv_local.lookup_slx_int[ 422].nbits = 11; gv_local.lookup_slx_int[ 422].bits = 0x87000000;
  gv_local.lookup_slx_int[ 423].nbits = 11; gv_local.lookup_slx_int[ 423].bits = 0x86400000;
  gv_local.lookup_slx_int[ 424].nbits = 11; gv_local.lookup_slx_int[ 424].bits = 0x7DA00000;
  gv_local.lookup_slx_int[ 425].nbits = 11; gv_local.lookup_slx_int[ 425].bits = 0x75A00000;
  gv_local.lookup_slx_int[ 426].nbits = 11; gv_local.lookup_slx_int[ 426].bits = 0x75800000;
  gv_local.lookup_slx_int[ 427].nbits = 11; gv_local.lookup_slx_int[ 427].bits = 0x6F000000;
  gv_local.lookup_slx_int[ 428].nbits = 11; gv_local.lookup_slx_int[ 428].bits = 0x65A00000;
  gv_local.lookup_slx_int[ 429].nbits = 11; gv_local.lookup_slx_int[ 429].bits = 0x63600000;
  gv_local.lookup_slx_int[ 430].nbits = 11; gv_local.lookup_slx_int[ 430].bits = 0x63C00000;
  gv_local.lookup_slx_int[ 431].nbits = 11; gv_local.lookup_slx_int[ 431].bits = 0x5B800000;
  gv_local.lookup_slx_int[ 432].nbits = 11; gv_local.lookup_slx_int[ 432].bits = 0x58E00000;
  gv_local.lookup_slx_int[ 433].nbits = 11; gv_local.lookup_slx_int[ 433].bits = 0x58800000;
  gv_local.lookup_slx_int[ 434].nbits = 11; gv_local.lookup_slx_int[ 434].bits = 0x4E600000;
  gv_local.lookup_slx_int[ 435].nbits = 11; gv_local.lookup_slx_int[ 435].bits = 0x4C600000;
  gv_local.lookup_slx_int[ 436].nbits = 11; gv_local.lookup_slx_int[ 436].bits = 0x48A00000;
  gv_local.lookup_slx_int[ 437].nbits = 11; gv_local.lookup_slx_int[ 437].bits = 0x42200000;
  gv_local.lookup_slx_int[ 438].nbits = 11; gv_local.lookup_slx_int[ 438].bits = 0x42C00000;
  gv_local.lookup_slx_int[ 439].nbits = 11; gv_local.lookup_slx_int[ 439].bits = 0x3E400000;
  gv_local.lookup_slx_int[ 440].nbits = 11; gv_local.lookup_slx_int[ 440].bits = 0x3C400000;
  gv_local.lookup_slx_int[ 441].nbits = 11; gv_local.lookup_slx_int[ 441].bits = 0x32A00000;
  gv_local.lookup_slx_int[ 442].nbits = 11; gv_local.lookup_slx_int[ 442].bits = 0x29E00000;
  gv_local.lookup_slx_int[ 443].nbits = 11; gv_local.lookup_slx_int[ 443].bits = 0x29C00000;
  gv_local.lookup_slx_int[ 444].nbits = 11; gv_local.lookup_slx_int[ 444].bits = 0x29600000;
  gv_local.lookup_slx_int[ 445].nbits = 11; gv_local.lookup_slx_int[ 445].bits = 0x27A00000;
  gv_local.lookup_slx_int[ 446].nbits = 11; gv_local.lookup_slx_int[ 446].bits = 0x1A600000;
  gv_local.lookup_slx_int[ 447].nbits = 11; gv_local.lookup_slx_int[ 447].bits = 0x1A400000;
  gv_local.lookup_slx_int[ 448].nbits = 11; gv_local.lookup_slx_int[ 448].bits = 0x11000000;
  gv_local.lookup_slx_int[ 449].nbits = 11; gv_local.lookup_slx_int[ 449].bits = 0x0D400000;
  gv_local.lookup_slx_int[ 450].nbits = 11; gv_local.lookup_slx_int[ 450].bits = 0x09A00000;
  gv_local.lookup_slx_int[ 451].nbits = 11; gv_local.lookup_slx_int[ 451].bits = 0x02E00000;
  gv_local.lookup_slx_int[ 452].nbits = 12; gv_local.lookup_slx_int[ 452].bits = 0xDE800000;
  gv_local.lookup_slx_int[ 453].nbits = 12; gv_local.lookup_slx_int[ 453].bits = 0xDB900000;
  gv_local.lookup_slx_int[ 454].nbits = 12; gv_local.lookup_slx_int[ 454].bits = 0xDB800000;
  gv_local.lookup_slx_int[ 455].nbits = 12; gv_local.lookup_slx_int[ 455].bits = 0xD3E00000;
  gv_local.lookup_slx_int[ 456].nbits = 12; gv_local.lookup_slx_int[ 456].bits = 0xD3F00000;
  gv_local.lookup_slx_int[ 457].nbits = 12; gv_local.lookup_slx_int[ 457].bits = 0xCFE00000;
  gv_local.lookup_slx_int[ 458].nbits = 12; gv_local.lookup_slx_int[ 458].bits = 0xD2D00000;
  gv_local.lookup_slx_int[ 459].nbits = 12; gv_local.lookup_slx_int[ 459].bits = 0xCF700000;
  gv_local.lookup_slx_int[ 460].nbits = 12; gv_local.lookup_slx_int[ 460].bits = 0xCAF00000;
  gv_local.lookup_slx_int[ 461].nbits = 12; gv_local.lookup_slx_int[ 461].bits = 0xCAE00000;
  gv_local.lookup_slx_int[ 462].nbits = 12; gv_local.lookup_slx_int[ 462].bits = 0xC9500000;
  gv_local.lookup_slx_int[ 463].nbits = 12; gv_local.lookup_slx_int[ 463].bits = 0xC1B00000;
  gv_local.lookup_slx_int[ 464].nbits = 12; gv_local.lookup_slx_int[ 464].bits = 0xC0500000;
  gv_local.lookup_slx_int[ 465].nbits = 12; gv_local.lookup_slx_int[ 465].bits = 0xC0400000;
  gv_local.lookup_slx_int[ 466].nbits = 12; gv_local.lookup_slx_int[ 466].bits = 0xBDD00000;
  gv_local.lookup_slx_int[ 467].nbits = 12; gv_local.lookup_slx_int[ 467].bits = 0xBD800000;
  gv_local.lookup_slx_int[ 468].nbits = 12; gv_local.lookup_slx_int[ 468].bits = 0xB6800000;
  gv_local.lookup_slx_int[ 469].nbits = 12; gv_local.lookup_slx_int[ 469].bits = 0xB6300000;
  gv_local.lookup_slx_int[ 470].nbits = 12; gv_local.lookup_slx_int[ 470].bits = 0xB6200000;
  gv_local.lookup_slx_int[ 471].nbits = 12; gv_local.lookup_slx_int[ 471].bits = 0xAF500000;
  gv_local.lookup_slx_int[ 472].nbits = 12; gv_local.lookup_slx_int[ 472].bits = 0xA7D00000;
  gv_local.lookup_slx_int[ 473].nbits = 12; gv_local.lookup_slx_int[ 473].bits = 0xA5F00000;
  gv_local.lookup_slx_int[ 474].nbits = 12; gv_local.lookup_slx_int[ 474].bits = 0xA9400000;
  gv_local.lookup_slx_int[ 475].nbits = 12; gv_local.lookup_slx_int[ 475].bits = 0xA4B00000;
  gv_local.lookup_slx_int[ 476].nbits = 12; gv_local.lookup_slx_int[ 476].bits = 0xA3300000;
  gv_local.lookup_slx_int[ 477].nbits = 12; gv_local.lookup_slx_int[ 477].bits = 0x9DD00000;
  gv_local.lookup_slx_int[ 478].nbits = 12; gv_local.lookup_slx_int[ 478].bits = 0x9DC00000;
  gv_local.lookup_slx_int[ 479].nbits = 12; gv_local.lookup_slx_int[ 479].bits = 0x95100000;
  gv_local.lookup_slx_int[ 480].nbits = 12; gv_local.lookup_slx_int[ 480].bits = 0x91300000;
  gv_local.lookup_slx_int[ 481].nbits = 12; gv_local.lookup_slx_int[ 481].bits = 0x91000000;
  gv_local.lookup_slx_int[ 482].nbits = 12; gv_local.lookup_slx_int[ 482].bits = 0x91200000;
  gv_local.lookup_slx_int[ 483].nbits = 12; gv_local.lookup_slx_int[ 483].bits = 0x87D00000;
  gv_local.lookup_slx_int[ 484].nbits = 12; gv_local.lookup_slx_int[ 484].bits = 0x87C00000;
  gv_local.lookup_slx_int[ 485].nbits = 12; gv_local.lookup_slx_int[ 485].bits = 0x86700000;
  gv_local.lookup_slx_int[ 486].nbits = 12; gv_local.lookup_slx_int[ 486].bits = 0x7D900000;
  gv_local.lookup_slx_int[ 487].nbits = 12; gv_local.lookup_slx_int[ 487].bits = 0x84900000;
  gv_local.lookup_slx_int[ 488].nbits = 12; gv_local.lookup_slx_int[ 488].bits = 0x6F300000;
  gv_local.lookup_slx_int[ 489].nbits = 12; gv_local.lookup_slx_int[ 489].bits = 0x6D700000;
  gv_local.lookup_slx_int[ 490].nbits = 12; gv_local.lookup_slx_int[ 490].bits = 0x6D100000;
  gv_local.lookup_slx_int[ 491].nbits = 12; gv_local.lookup_slx_int[ 491].bits = 0x6D600000;
  gv_local.lookup_slx_int[ 492].nbits = 12; gv_local.lookup_slx_int[ 492].bits = 0x5BB00000;
  gv_local.lookup_slx_int[ 493].nbits = 12; gv_local.lookup_slx_int[ 493].bits = 0x63500000;
  gv_local.lookup_slx_int[ 494].nbits = 12; gv_local.lookup_slx_int[ 494].bits = 0x63400000;
  gv_local.lookup_slx_int[ 495].nbits = 12; gv_local.lookup_slx_int[ 495].bits = 0x4C500000;
  gv_local.lookup_slx_int[ 496].nbits = 12; gv_local.lookup_slx_int[ 496].bits = 0x58C00000;
  gv_local.lookup_slx_int[ 497].nbits = 12; gv_local.lookup_slx_int[ 497].bits = 0x4F200000;
  gv_local.lookup_slx_int[ 498].nbits = 12; gv_local.lookup_slx_int[ 498].bits = 0x48900000;
  gv_local.lookup_slx_int[ 499].nbits = 12; gv_local.lookup_slx_int[ 499].bits = 0x48800000;
  gv_local.lookup_slx_int[ 500].nbits = 12; gv_local.lookup_slx_int[ 500].bits = 0x3E700000;
  gv_local.lookup_slx_int[ 501].nbits = 12; gv_local.lookup_slx_int[ 501].bits = 0x4C100000;
  gv_local.lookup_slx_int[ 502].nbits = 12; gv_local.lookup_slx_int[ 502].bits = 0x3C100000;
  gv_local.lookup_slx_int[ 503].nbits = 12; gv_local.lookup_slx_int[ 503].bits = 0x2B200000;
  gv_local.lookup_slx_int[ 504].nbits = 12; gv_local.lookup_slx_int[ 504].bits = 0x3C300000;
  gv_local.lookup_slx_int[ 505].nbits = 12; gv_local.lookup_slx_int[ 505].bits = 0x2B300000;
  gv_local.lookup_slx_int[ 506].nbits = 12; gv_local.lookup_slx_int[ 506].bits = 0x2B000000;
  gv_local.lookup_slx_int[ 507].nbits = 12; gv_local.lookup_slx_int[ 507].bits = 0x2B100000;
  gv_local.lookup_slx_int[ 508].nbits = 12; gv_local.lookup_slx_int[ 508].bits = 0x29800000;
  gv_local.lookup_slx_int[ 509].nbits = 12; gv_local.lookup_slx_int[ 509].bits = 0x21000000;
  gv_local.lookup_slx_int[ 510].nbits = 12; gv_local.lookup_slx_int[ 510].bits = 0x21200000;
  gv_local.lookup_slx_int[ 511].nbits =  3; gv_local.lookup_slx_int[ 511].bits = 0xE0000000;

#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  /* reverse huffman lookup table
   */
  memset (gv_local.rlookup_slx_int, 0, sizeof(uint16_t)*REV_HUFF_TREESIZE_16);

  for (i=0; i<INT_HUFF_TREESIZE; i++)
  {
    val = gv_local.lookup_slx_int[i].bits >> 16;
    gv_local.rlookup_slx_int [val] = (uint16_t) i;
    for (j=1; j<=(0xFFFF>>gv_local.lookup_slx_int[i].nbits); j++)
      gv_local.rlookup_slx_int [val+j] = (uint16_t) i;
  }
#endif

  _s_sort_lookup (gv_local.lookup_slx_int, gv_local.idx_slx_int, INT_HUFF_TREESIZE);

  gv_local.lookup_slx_int_initialized = 1;
}
/*
 */
static void _s_init_nse_slx_default (void)
{
#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  int      i, j;
  uint16_t val;
#endif

  memset (gv_local.lookup_slx_nse,  0, sizeof(SRALookup)*NSE_HUFF_TREESIZE);

  gv_local.lookup_slx_nse[   0].nbits = 11; gv_local.lookup_slx_nse[   0].bits = 0x6FE00000;
  gv_local.lookup_slx_nse[   1].nbits = 11; gv_local.lookup_slx_nse[   1].bits = 0x7AC00000;
  gv_local.lookup_slx_nse[   2].nbits = 11; gv_local.lookup_slx_nse[   2].bits = 0x6FC00000;
  gv_local.lookup_slx_nse[   3].nbits = 11; gv_local.lookup_slx_nse[   3].bits = 0x7AE00000;
  gv_local.lookup_slx_nse[   4].nbits = 11; gv_local.lookup_slx_nse[   4].bits = 0xC6A00000;
  gv_local.lookup_slx_nse[   5].nbits = 11; gv_local.lookup_slx_nse[   5].bits = 0xDFC00000;
  gv_local.lookup_slx_nse[   6].nbits = 11; gv_local.lookup_slx_nse[   6].bits = 0xD7200000;
  gv_local.lookup_slx_nse[   7].nbits = 11; gv_local.lookup_slx_nse[   7].bits = 0xE4E00000;
  gv_local.lookup_slx_nse[   8].nbits = 11; gv_local.lookup_slx_nse[   8].bits = 0xDA000000;
  gv_local.lookup_slx_nse[   9].nbits = 11; gv_local.lookup_slx_nse[   9].bits = 0xD0000000;
  gv_local.lookup_slx_nse[  10].nbits = 11; gv_local.lookup_slx_nse[  10].bits = 0xE8000000;
  gv_local.lookup_slx_nse[  11].nbits = 11; gv_local.lookup_slx_nse[  11].bits = 0xCE400000;
  gv_local.lookup_slx_nse[  12].nbits = 11; gv_local.lookup_slx_nse[  12].bits = 0xD7000000;
  gv_local.lookup_slx_nse[  13].nbits = 11; gv_local.lookup_slx_nse[  13].bits = 0xD5000000;
  gv_local.lookup_slx_nse[  14].nbits = 11; gv_local.lookup_slx_nse[  14].bits = 0xDDE00000;
  gv_local.lookup_slx_nse[  15].nbits = 11; gv_local.lookup_slx_nse[  15].bits = 0xF2C00000;
  gv_local.lookup_slx_nse[  16].nbits = 11; gv_local.lookup_slx_nse[  16].bits = 0xF6600000;
  gv_local.lookup_slx_nse[  17].nbits = 11; gv_local.lookup_slx_nse[  17].bits = 0xEE400000;
  gv_local.lookup_slx_nse[  18].nbits = 11; gv_local.lookup_slx_nse[  18].bits = 0xE4C00000;
  gv_local.lookup_slx_nse[  19].nbits = 10; gv_local.lookup_slx_nse[  19].bits = 0x0F000000;
  gv_local.lookup_slx_nse[  20].nbits = 11; gv_local.lookup_slx_nse[  20].bits = 0xEBC00000;
  gv_local.lookup_slx_nse[  21].nbits = 11; gv_local.lookup_slx_nse[  21].bits = 0xF7E00000;
  gv_local.lookup_slx_nse[  22].nbits = 11; gv_local.lookup_slx_nse[  22].bits = 0xF7C00000;
  gv_local.lookup_slx_nse[  23].nbits = 10; gv_local.lookup_slx_nse[  23].bits = 0x18000000;
  gv_local.lookup_slx_nse[  24].nbits = 10; gv_local.lookup_slx_nse[  24].bits = 0x22800000;
  gv_local.lookup_slx_nse[  25].nbits = 10; gv_local.lookup_slx_nse[  25].bits = 0x0A000000;
  gv_local.lookup_slx_nse[  26].nbits = 10; gv_local.lookup_slx_nse[  26].bits = 0x0F400000;
  gv_local.lookup_slx_nse[  27].nbits = 10; gv_local.lookup_slx_nse[  27].bits = 0x18400000;
  gv_local.lookup_slx_nse[  28].nbits = 10; gv_local.lookup_slx_nse[  28].bits = 0x1D800000;
  gv_local.lookup_slx_nse[  29].nbits = 10; gv_local.lookup_slx_nse[  29].bits = 0x3CC00000;
  gv_local.lookup_slx_nse[  30].nbits = 10; gv_local.lookup_slx_nse[  30].bits = 0x46C00000;
  gv_local.lookup_slx_nse[  31].nbits = 10; gv_local.lookup_slx_nse[  31].bits = 0x1DC00000;
  gv_local.lookup_slx_nse[  32].nbits = 10; gv_local.lookup_slx_nse[  32].bits = 0x42C00000;
  gv_local.lookup_slx_nse[  33].nbits = 10; gv_local.lookup_slx_nse[  33].bits = 0x3F800000;
  gv_local.lookup_slx_nse[  34].nbits = 10; gv_local.lookup_slx_nse[  34].bits = 0x3FC00000;
  gv_local.lookup_slx_nse[  35].nbits = 10; gv_local.lookup_slx_nse[  35].bits = 0x4B800000;
  gv_local.lookup_slx_nse[  36].nbits = 10; gv_local.lookup_slx_nse[  36].bits = 0x4BC00000;
  gv_local.lookup_slx_nse[  37].nbits = 10; gv_local.lookup_slx_nse[  37].bits = 0x27C00000;
  gv_local.lookup_slx_nse[  38].nbits = 10; gv_local.lookup_slx_nse[  38].bits = 0x46400000;
  gv_local.lookup_slx_nse[  39].nbits = 10; gv_local.lookup_slx_nse[  39].bits = 0x52000000;
  gv_local.lookup_slx_nse[  40].nbits = 10; gv_local.lookup_slx_nse[  40].bits = 0x58400000;
  gv_local.lookup_slx_nse[  41].nbits = 10; gv_local.lookup_slx_nse[  41].bits = 0x50800000;
  gv_local.lookup_slx_nse[  42].nbits = 10; gv_local.lookup_slx_nse[  42].bits = 0x56800000;
  gv_local.lookup_slx_nse[  43].nbits = 10; gv_local.lookup_slx_nse[  43].bits = 0x56000000;
  gv_local.lookup_slx_nse[  44].nbits = 10; gv_local.lookup_slx_nse[  44].bits = 0x57000000;
  gv_local.lookup_slx_nse[  45].nbits = 10; gv_local.lookup_slx_nse[  45].bits = 0x62400000;
  gv_local.lookup_slx_nse[  46].nbits = 10; gv_local.lookup_slx_nse[  46].bits = 0x56C00000;
  gv_local.lookup_slx_nse[  47].nbits = 10; gv_local.lookup_slx_nse[  47].bits = 0x57400000;
  gv_local.lookup_slx_nse[  48].nbits = 10; gv_local.lookup_slx_nse[  48].bits = 0x63800000;
  gv_local.lookup_slx_nse[  49].nbits = 10; gv_local.lookup_slx_nse[  49].bits = 0x62000000;
  gv_local.lookup_slx_nse[  50].nbits = 10; gv_local.lookup_slx_nse[  50].bits = 0x5C000000;
  gv_local.lookup_slx_nse[  51].nbits = 10; gv_local.lookup_slx_nse[  51].bits = 0x60800000;
  gv_local.lookup_slx_nse[  52].nbits = 10; gv_local.lookup_slx_nse[  52].bits = 0x6B000000;
  gv_local.lookup_slx_nse[  53].nbits = 10; gv_local.lookup_slx_nse[  53].bits = 0x6B400000;
  gv_local.lookup_slx_nse[  54].nbits = 10; gv_local.lookup_slx_nse[  54].bits = 0x67C00000;
  gv_local.lookup_slx_nse[  55].nbits = 10; gv_local.lookup_slx_nse[  55].bits = 0x66400000;
  gv_local.lookup_slx_nse[  56].nbits = 10; gv_local.lookup_slx_nse[  56].bits = 0x6A400000;
  gv_local.lookup_slx_nse[  57].nbits = 10; gv_local.lookup_slx_nse[  57].bits = 0x6F800000;
  gv_local.lookup_slx_nse[  58].nbits = 10; gv_local.lookup_slx_nse[  58].bits = 0x77000000;
  gv_local.lookup_slx_nse[  59].nbits = 10; gv_local.lookup_slx_nse[  59].bits = 0x76400000;
  gv_local.lookup_slx_nse[  60].nbits = 10; gv_local.lookup_slx_nse[  60].bits = 0x69800000;
  gv_local.lookup_slx_nse[  61].nbits = 10; gv_local.lookup_slx_nse[  61].bits = 0x70000000;
  gv_local.lookup_slx_nse[  62].nbits = 10; gv_local.lookup_slx_nse[  62].bits = 0x6A000000;
  gv_local.lookup_slx_nse[  63].nbits = 10; gv_local.lookup_slx_nse[  63].bits = 0x76000000;
  gv_local.lookup_slx_nse[  64].nbits = 10; gv_local.lookup_slx_nse[  64].bits = 0x79400000;
  gv_local.lookup_slx_nse[  65].nbits = 10; gv_local.lookup_slx_nse[  65].bits = 0x7CC00000;
  gv_local.lookup_slx_nse[  66].nbits = 10; gv_local.lookup_slx_nse[  66].bits = 0x74800000;
  gv_local.lookup_slx_nse[  67].nbits = 10; gv_local.lookup_slx_nse[  67].bits = 0x78C00000;
  gv_local.lookup_slx_nse[  68].nbits = 10; gv_local.lookup_slx_nse[  68].bits = 0x7B000000;
  gv_local.lookup_slx_nse[  69].nbits = 10; gv_local.lookup_slx_nse[  69].bits = 0x6C000000;
  gv_local.lookup_slx_nse[  70].nbits = 10; gv_local.lookup_slx_nse[  70].bits = 0xC2000000;
  gv_local.lookup_slx_nse[  71].nbits = 10; gv_local.lookup_slx_nse[  71].bits = 0x79000000;
  gv_local.lookup_slx_nse[  72].nbits = 10; gv_local.lookup_slx_nse[  72].bits = 0x77800000;
  gv_local.lookup_slx_nse[  73].nbits = 10; gv_local.lookup_slx_nse[  73].bits = 0xC4C00000;
  gv_local.lookup_slx_nse[  74].nbits = 10; gv_local.lookup_slx_nse[  74].bits = 0x78800000;
  gv_local.lookup_slx_nse[  75].nbits = 10; gv_local.lookup_slx_nse[  75].bits = 0x7C400000;
  gv_local.lookup_slx_nse[  76].nbits = 10; gv_local.lookup_slx_nse[  76].bits = 0xC7C00000;
  gv_local.lookup_slx_nse[  77].nbits = 10; gv_local.lookup_slx_nse[  77].bits = 0x7A800000;
  gv_local.lookup_slx_nse[  78].nbits = 10; gv_local.lookup_slx_nse[  78].bits = 0xC1400000;
  gv_local.lookup_slx_nse[  79].nbits = 10; gv_local.lookup_slx_nse[  79].bits = 0x7F400000;
  gv_local.lookup_slx_nse[  80].nbits = 10; gv_local.lookup_slx_nse[  80].bits = 0xC9C00000;
  gv_local.lookup_slx_nse[  81].nbits = 10; gv_local.lookup_slx_nse[  81].bits = 0xCC800000;
  gv_local.lookup_slx_nse[  82].nbits = 10; gv_local.lookup_slx_nse[  82].bits = 0xC6C00000;
  gv_local.lookup_slx_nse[  83].nbits = 10; gv_local.lookup_slx_nse[  83].bits = 0xC0C00000;
  gv_local.lookup_slx_nse[  84].nbits = 10; gv_local.lookup_slx_nse[  84].bits = 0xC9000000;
  gv_local.lookup_slx_nse[  85].nbits = 10; gv_local.lookup_slx_nse[  85].bits = 0xD2800000;
  gv_local.lookup_slx_nse[  86].nbits = 10; gv_local.lookup_slx_nse[  86].bits = 0xD0C00000;
  gv_local.lookup_slx_nse[  87].nbits = 10; gv_local.lookup_slx_nse[  87].bits = 0xC6400000;
  gv_local.lookup_slx_nse[  88].nbits = 10; gv_local.lookup_slx_nse[  88].bits = 0xC1800000;
  gv_local.lookup_slx_nse[  89].nbits = 10; gv_local.lookup_slx_nse[  89].bits = 0xCE000000;
  gv_local.lookup_slx_nse[  90].nbits = 10; gv_local.lookup_slx_nse[  90].bits = 0xC3800000;
  gv_local.lookup_slx_nse[  91].nbits = 10; gv_local.lookup_slx_nse[  91].bits = 0xCD400000;
  gv_local.lookup_slx_nse[  92].nbits = 10; gv_local.lookup_slx_nse[  92].bits = 0xD3000000;
  gv_local.lookup_slx_nse[  93].nbits = 10; gv_local.lookup_slx_nse[  93].bits = 0xD7800000;
  gv_local.lookup_slx_nse[  94].nbits = 10; gv_local.lookup_slx_nse[  94].bits = 0xD8800000;
  gv_local.lookup_slx_nse[  95].nbits = 10; gv_local.lookup_slx_nse[  95].bits = 0xCB400000;
  gv_local.lookup_slx_nse[  96].nbits = 10; gv_local.lookup_slx_nse[  96].bits = 0xCE800000;
  gv_local.lookup_slx_nse[  97].nbits = 10; gv_local.lookup_slx_nse[  97].bits = 0xCB800000;
  gv_local.lookup_slx_nse[  98].nbits = 10; gv_local.lookup_slx_nse[  98].bits = 0xE1400000;
  gv_local.lookup_slx_nse[  99].nbits = 10; gv_local.lookup_slx_nse[  99].bits = 0xD8400000;
  gv_local.lookup_slx_nse[ 100].nbits = 10; gv_local.lookup_slx_nse[ 100].bits = 0xE1800000;
  gv_local.lookup_slx_nse[ 101].nbits = 10; gv_local.lookup_slx_nse[ 101].bits = 0xDD400000;
  gv_local.lookup_slx_nse[ 102].nbits = 10; gv_local.lookup_slx_nse[ 102].bits = 0xDF400000;
  gv_local.lookup_slx_nse[ 103].nbits = 10; gv_local.lookup_slx_nse[ 103].bits = 0xCEC00000;
  gv_local.lookup_slx_nse[ 104].nbits = 10; gv_local.lookup_slx_nse[ 104].bits = 0xDA800000;
  gv_local.lookup_slx_nse[ 105].nbits = 10; gv_local.lookup_slx_nse[ 105].bits = 0xD7C00000;
  gv_local.lookup_slx_nse[ 106].nbits = 10; gv_local.lookup_slx_nse[ 106].bits = 0xE3800000;
  gv_local.lookup_slx_nse[ 107].nbits = 10; gv_local.lookup_slx_nse[ 107].bits = 0xDB000000;
  gv_local.lookup_slx_nse[ 108].nbits = 10; gv_local.lookup_slx_nse[ 108].bits = 0xE0400000;
  gv_local.lookup_slx_nse[ 109].nbits = 10; gv_local.lookup_slx_nse[ 109].bits = 0xE3C00000;
  gv_local.lookup_slx_nse[ 110].nbits = 10; gv_local.lookup_slx_nse[ 110].bits = 0xDF800000;
  gv_local.lookup_slx_nse[ 111].nbits = 10; gv_local.lookup_slx_nse[ 111].bits = 0xD9800000;
  gv_local.lookup_slx_nse[ 112].nbits = 10; gv_local.lookup_slx_nse[ 112].bits = 0xDE000000;
  gv_local.lookup_slx_nse[ 113].nbits = 10; gv_local.lookup_slx_nse[ 113].bits = 0xE5800000;
  gv_local.lookup_slx_nse[ 114].nbits = 10; gv_local.lookup_slx_nse[ 114].bits = 0xD2C00000;
  gv_local.lookup_slx_nse[ 115].nbits = 10; gv_local.lookup_slx_nse[ 115].bits = 0xED400000;
  gv_local.lookup_slx_nse[ 116].nbits = 10; gv_local.lookup_slx_nse[ 116].bits = 0xE7C00000;
  gv_local.lookup_slx_nse[ 117].nbits = 10; gv_local.lookup_slx_nse[ 117].bits = 0xD3800000;
  gv_local.lookup_slx_nse[ 118].nbits = 10; gv_local.lookup_slx_nse[ 118].bits = 0xE9800000;
  gv_local.lookup_slx_nse[ 119].nbits = 10; gv_local.lookup_slx_nse[ 119].bits = 0xDBC00000;
  gv_local.lookup_slx_nse[ 120].nbits = 10; gv_local.lookup_slx_nse[ 120].bits = 0xEC800000;
  gv_local.lookup_slx_nse[ 121].nbits = 10; gv_local.lookup_slx_nse[ 121].bits = 0xED000000;
  gv_local.lookup_slx_nse[ 122].nbits = 10; gv_local.lookup_slx_nse[ 122].bits = 0xEF800000;
  gv_local.lookup_slx_nse[ 123].nbits = 10; gv_local.lookup_slx_nse[ 123].bits = 0xEC000000;
  gv_local.lookup_slx_nse[ 124].nbits = 10; gv_local.lookup_slx_nse[ 124].bits = 0xF0400000;
  gv_local.lookup_slx_nse[ 125].nbits = 10; gv_local.lookup_slx_nse[ 125].bits = 0xE1000000;
  gv_local.lookup_slx_nse[ 126].nbits = 10; gv_local.lookup_slx_nse[ 126].bits = 0xEE000000;
  gv_local.lookup_slx_nse[ 127].nbits = 10; gv_local.lookup_slx_nse[ 127].bits = 0xEAC00000;
  gv_local.lookup_slx_nse[ 128].nbits = 10; gv_local.lookup_slx_nse[ 128].bits = 0xF8800000;
  gv_local.lookup_slx_nse[ 129].nbits = 10; gv_local.lookup_slx_nse[ 129].bits = 0xE6800000;
  gv_local.lookup_slx_nse[ 130].nbits = 10; gv_local.lookup_slx_nse[ 130].bits = 0xE6C00000;
  gv_local.lookup_slx_nse[ 131].nbits = 10; gv_local.lookup_slx_nse[ 131].bits = 0xEB800000;
  gv_local.lookup_slx_nse[ 132].nbits = 10; gv_local.lookup_slx_nse[ 132].bits = 0xECC00000;
  gv_local.lookup_slx_nse[ 133].nbits = 10; gv_local.lookup_slx_nse[ 133].bits = 0xF2400000;
  gv_local.lookup_slx_nse[ 134].nbits = 10; gv_local.lookup_slx_nse[ 134].bits = 0xF0000000;
  gv_local.lookup_slx_nse[ 135].nbits = 10; gv_local.lookup_slx_nse[ 135].bits = 0xF5C00000;
  gv_local.lookup_slx_nse[ 136].nbits = 10; gv_local.lookup_slx_nse[ 136].bits = 0xF7000000;
  gv_local.lookup_slx_nse[ 137].nbits = 10; gv_local.lookup_slx_nse[ 137].bits = 0xEDC00000;
  gv_local.lookup_slx_nse[ 138].nbits = 10; gv_local.lookup_slx_nse[ 138].bits = 0xF2800000;
  gv_local.lookup_slx_nse[ 139].nbits = 10; gv_local.lookup_slx_nse[ 139].bits = 0xE9000000;
  gv_local.lookup_slx_nse[ 140].nbits = 10; gv_local.lookup_slx_nse[ 140].bits = 0xF9400000;
  gv_local.lookup_slx_nse[ 141].nbits = 10; gv_local.lookup_slx_nse[ 141].bits = 0xFF400000;
  gv_local.lookup_slx_nse[ 142].nbits = 10; gv_local.lookup_slx_nse[ 142].bits = 0xFA800000;
  gv_local.lookup_slx_nse[ 143].nbits = 10; gv_local.lookup_slx_nse[ 143].bits = 0xF8400000;
  gv_local.lookup_slx_nse[ 144].nbits = 10; gv_local.lookup_slx_nse[ 144].bits = 0xEFC00000;
  gv_local.lookup_slx_nse[ 145].nbits = 10; gv_local.lookup_slx_nse[ 145].bits = 0xEF400000;
  gv_local.lookup_slx_nse[ 146].nbits = 10; gv_local.lookup_slx_nse[ 146].bits = 0xF9000000;
  gv_local.lookup_slx_nse[ 147].nbits =  9; gv_local.lookup_slx_nse[ 147].bits = 0x07000000;
  gv_local.lookup_slx_nse[ 148].nbits = 10; gv_local.lookup_slx_nse[ 148].bits = 0xF7800000;
  gv_local.lookup_slx_nse[ 149].nbits = 10; gv_local.lookup_slx_nse[ 149].bits = 0xFF000000;
  gv_local.lookup_slx_nse[ 150].nbits =  9; gv_local.lookup_slx_nse[ 150].bits = 0x04800000;
  gv_local.lookup_slx_nse[ 151].nbits =  9; gv_local.lookup_slx_nse[ 151].bits = 0x08800000;
  gv_local.lookup_slx_nse[ 152].nbits = 10; gv_local.lookup_slx_nse[ 152].bits = 0xFEC00000;
  gv_local.lookup_slx_nse[ 153].nbits = 10; gv_local.lookup_slx_nse[ 153].bits = 0xFC000000;
  gv_local.lookup_slx_nse[ 154].nbits =  9; gv_local.lookup_slx_nse[ 154].bits = 0x16000000;
  gv_local.lookup_slx_nse[ 155].nbits = 10; gv_local.lookup_slx_nse[ 155].bits = 0xFB400000;
  gv_local.lookup_slx_nse[ 156].nbits =  9; gv_local.lookup_slx_nse[ 156].bits = 0x01000000;
  gv_local.lookup_slx_nse[ 157].nbits =  9; gv_local.lookup_slx_nse[ 157].bits = 0x08000000;
  gv_local.lookup_slx_nse[ 158].nbits =  9; gv_local.lookup_slx_nse[ 158].bits = 0x18800000;
  gv_local.lookup_slx_nse[ 159].nbits =  9; gv_local.lookup_slx_nse[ 159].bits = 0x0C000000;
  gv_local.lookup_slx_nse[ 160].nbits =  9; gv_local.lookup_slx_nse[ 160].bits = 0x10800000;
  gv_local.lookup_slx_nse[ 161].nbits =  9; gv_local.lookup_slx_nse[ 161].bits = 0x15000000;
  gv_local.lookup_slx_nse[ 162].nbits =  9; gv_local.lookup_slx_nse[ 162].bits = 0x0A800000;
  gv_local.lookup_slx_nse[ 163].nbits =  9; gv_local.lookup_slx_nse[ 163].bits = 0x04000000;
  gv_local.lookup_slx_nse[ 164].nbits =  9; gv_local.lookup_slx_nse[ 164].bits = 0x22000000;
  gv_local.lookup_slx_nse[ 165].nbits =  9; gv_local.lookup_slx_nse[ 165].bits = 0x17000000;
  gv_local.lookup_slx_nse[ 166].nbits =  9; gv_local.lookup_slx_nse[ 166].bits = 0x1E800000;
  gv_local.lookup_slx_nse[ 167].nbits =  9; gv_local.lookup_slx_nse[ 167].bits = 0x26000000;
  gv_local.lookup_slx_nse[ 168].nbits =  9; gv_local.lookup_slx_nse[ 168].bits = 0x0D800000;
  gv_local.lookup_slx_nse[ 169].nbits =  9; gv_local.lookup_slx_nse[ 169].bits = 0x10000000;
  gv_local.lookup_slx_nse[ 170].nbits =  9; gv_local.lookup_slx_nse[ 170].bits = 0x31800000;
  gv_local.lookup_slx_nse[ 171].nbits =  9; gv_local.lookup_slx_nse[ 171].bits = 0x29000000;
  gv_local.lookup_slx_nse[ 172].nbits =  9; gv_local.lookup_slx_nse[ 172].bits = 0x15800000;
  gv_local.lookup_slx_nse[ 173].nbits =  9; gv_local.lookup_slx_nse[ 173].bits = 0x29800000;
  gv_local.lookup_slx_nse[ 174].nbits =  9; gv_local.lookup_slx_nse[ 174].bits = 0x26800000;
  gv_local.lookup_slx_nse[ 175].nbits =  9; gv_local.lookup_slx_nse[ 175].bits = 0x1F800000;
  gv_local.lookup_slx_nse[ 176].nbits =  9; gv_local.lookup_slx_nse[ 176].bits = 0x2E800000;
  gv_local.lookup_slx_nse[ 177].nbits =  9; gv_local.lookup_slx_nse[ 177].bits = 0x3E000000;
  gv_local.lookup_slx_nse[ 178].nbits =  9; gv_local.lookup_slx_nse[ 178].bits = 0x1F000000;
  gv_local.lookup_slx_nse[ 179].nbits =  9; gv_local.lookup_slx_nse[ 179].bits = 0x34000000;
  gv_local.lookup_slx_nse[ 180].nbits =  9; gv_local.lookup_slx_nse[ 180].bits = 0x2E000000;
  gv_local.lookup_slx_nse[ 181].nbits =  9; gv_local.lookup_slx_nse[ 181].bits = 0x40800000;
  gv_local.lookup_slx_nse[ 182].nbits =  9; gv_local.lookup_slx_nse[ 182].bits = 0x28000000;
  gv_local.lookup_slx_nse[ 183].nbits =  9; gv_local.lookup_slx_nse[ 183].bits = 0x49800000;
  gv_local.lookup_slx_nse[ 184].nbits =  9; gv_local.lookup_slx_nse[ 184].bits = 0x4A000000;
  gv_local.lookup_slx_nse[ 185].nbits =  9; gv_local.lookup_slx_nse[ 185].bits = 0x4A800000;
  gv_local.lookup_slx_nse[ 186].nbits =  9; gv_local.lookup_slx_nse[ 186].bits = 0x45000000;
  gv_local.lookup_slx_nse[ 187].nbits =  9; gv_local.lookup_slx_nse[ 187].bits = 0x51000000;
  gv_local.lookup_slx_nse[ 188].nbits =  9; gv_local.lookup_slx_nse[ 188].bits = 0x48000000;
  gv_local.lookup_slx_nse[ 189].nbits =  9; gv_local.lookup_slx_nse[ 189].bits = 0x49000000;
  gv_local.lookup_slx_nse[ 190].nbits =  9; gv_local.lookup_slx_nse[ 190].bits = 0x4E000000;
  gv_local.lookup_slx_nse[ 191].nbits =  9; gv_local.lookup_slx_nse[ 191].bits = 0x4E800000;
  gv_local.lookup_slx_nse[ 192].nbits =  9; gv_local.lookup_slx_nse[ 192].bits = 0x53000000;
  gv_local.lookup_slx_nse[ 193].nbits =  9; gv_local.lookup_slx_nse[ 193].bits = 0x4C800000;
  gv_local.lookup_slx_nse[ 194].nbits =  9; gv_local.lookup_slx_nse[ 194].bits = 0x57800000;
  gv_local.lookup_slx_nse[ 195].nbits =  9; gv_local.lookup_slx_nse[ 195].bits = 0x4F800000;
  gv_local.lookup_slx_nse[ 196].nbits =  9; gv_local.lookup_slx_nse[ 196].bits = 0x54800000;
  gv_local.lookup_slx_nse[ 197].nbits =  9; gv_local.lookup_slx_nse[ 197].bits = 0x52800000;
  gv_local.lookup_slx_nse[ 198].nbits =  9; gv_local.lookup_slx_nse[ 198].bits = 0x59800000;
  gv_local.lookup_slx_nse[ 199].nbits =  9; gv_local.lookup_slx_nse[ 199].bits = 0x5A000000;
  gv_local.lookup_slx_nse[ 200].nbits =  9; gv_local.lookup_slx_nse[ 200].bits = 0x58800000;
  gv_local.lookup_slx_nse[ 201].nbits =  9; gv_local.lookup_slx_nse[ 201].bits = 0x5E000000;
  gv_local.lookup_slx_nse[ 202].nbits =  9; gv_local.lookup_slx_nse[ 202].bits = 0x5E800000;
  gv_local.lookup_slx_nse[ 203].nbits =  9; gv_local.lookup_slx_nse[ 203].bits = 0x61800000;
  gv_local.lookup_slx_nse[ 204].nbits =  9; gv_local.lookup_slx_nse[ 204].bits = 0x6A800000;
  gv_local.lookup_slx_nse[ 205].nbits =  9; gv_local.lookup_slx_nse[ 205].bits = 0x65800000;
  gv_local.lookup_slx_nse[ 206].nbits =  9; gv_local.lookup_slx_nse[ 206].bits = 0x68000000;
  gv_local.lookup_slx_nse[ 207].nbits =  9; gv_local.lookup_slx_nse[ 207].bits = 0x64000000;
  gv_local.lookup_slx_nse[ 208].nbits =  9; gv_local.lookup_slx_nse[ 208].bits = 0x5D800000;
  gv_local.lookup_slx_nse[ 209].nbits =  9; gv_local.lookup_slx_nse[ 209].bits = 0x6F000000;
  gv_local.lookup_slx_nse[ 210].nbits =  9; gv_local.lookup_slx_nse[ 210].bits = 0x6D000000;
  gv_local.lookup_slx_nse[ 211].nbits =  9; gv_local.lookup_slx_nse[ 211].bits = 0x6D800000;
  gv_local.lookup_slx_nse[ 212].nbits =  9; gv_local.lookup_slx_nse[ 212].bits = 0x71000000;
  gv_local.lookup_slx_nse[ 213].nbits =  9; gv_local.lookup_slx_nse[ 213].bits = 0x76800000;
  gv_local.lookup_slx_nse[ 214].nbits =  9; gv_local.lookup_slx_nse[ 214].bits = 0x74000000;
  gv_local.lookup_slx_nse[ 215].nbits =  9; gv_local.lookup_slx_nse[ 215].bits = 0x72000000;
  gv_local.lookup_slx_nse[ 216].nbits =  9; gv_local.lookup_slx_nse[ 216].bits = 0x78000000;
  gv_local.lookup_slx_nse[ 217].nbits =  9; gv_local.lookup_slx_nse[ 217].bits = 0x7D800000;
  gv_local.lookup_slx_nse[ 218].nbits =  9; gv_local.lookup_slx_nse[ 218].bits = 0x7E800000;
  gv_local.lookup_slx_nse[ 219].nbits =  9; gv_local.lookup_slx_nse[ 219].bits = 0xC2800000;
  gv_local.lookup_slx_nse[ 220].nbits =  9; gv_local.lookup_slx_nse[ 220].bits = 0x7D000000;
  gv_local.lookup_slx_nse[ 221].nbits =  9; gv_local.lookup_slx_nse[ 221].bits = 0xC4000000;
  gv_local.lookup_slx_nse[ 222].nbits =  9; gv_local.lookup_slx_nse[ 222].bits = 0xC3000000;
  gv_local.lookup_slx_nse[ 223].nbits =  9; gv_local.lookup_slx_nse[ 223].bits = 0xCD800000;
  gv_local.lookup_slx_nse[ 224].nbits =  9; gv_local.lookup_slx_nse[ 224].bits = 0xE2000000;
  gv_local.lookup_slx_nse[ 225].nbits =  9; gv_local.lookup_slx_nse[ 225].bits = 0xD4000000;
  gv_local.lookup_slx_nse[ 226].nbits =  9; gv_local.lookup_slx_nse[ 226].bits = 0xDC800000;
  gv_local.lookup_slx_nse[ 227].nbits =  9; gv_local.lookup_slx_nse[ 227].bits = 0xD6000000;
  gv_local.lookup_slx_nse[ 228].nbits =  9; gv_local.lookup_slx_nse[ 228].bits = 0xDE800000;
  gv_local.lookup_slx_nse[ 229].nbits =  9; gv_local.lookup_slx_nse[ 229].bits = 0xE2800000;
  gv_local.lookup_slx_nse[ 230].nbits =  9; gv_local.lookup_slx_nse[ 230].bits = 0xE0800000;
  gv_local.lookup_slx_nse[ 231].nbits =  9; gv_local.lookup_slx_nse[ 231].bits = 0xE7000000;
  gv_local.lookup_slx_nse[ 232].nbits =  9; gv_local.lookup_slx_nse[ 232].bits = 0xEB000000;
  gv_local.lookup_slx_nse[ 233].nbits =  9; gv_local.lookup_slx_nse[ 233].bits = 0xF3000000;
  gv_local.lookup_slx_nse[ 234].nbits =  9; gv_local.lookup_slx_nse[ 234].bits = 0xF5000000;
  gv_local.lookup_slx_nse[ 235].nbits =  9; gv_local.lookup_slx_nse[ 235].bits = 0xF0800000;
  gv_local.lookup_slx_nse[ 236].nbits =  9; gv_local.lookup_slx_nse[ 236].bits = 0xFA000000;
  gv_local.lookup_slx_nse[ 237].nbits =  9; gv_local.lookup_slx_nse[ 237].bits = 0xF4000000;
  gv_local.lookup_slx_nse[ 238].nbits =  9; gv_local.lookup_slx_nse[ 238].bits = 0xFF800000;
  gv_local.lookup_slx_nse[ 239].nbits =  9; gv_local.lookup_slx_nse[ 239].bits = 0xFC800000;
  gv_local.lookup_slx_nse[ 240].nbits =  8; gv_local.lookup_slx_nse[ 240].bits = 0x03000000;
  gv_local.lookup_slx_nse[ 241].nbits =  8; gv_local.lookup_slx_nse[ 241].bits = 0x0E000000;
  gv_local.lookup_slx_nse[ 242].nbits =  8; gv_local.lookup_slx_nse[ 242].bits = 0x11000000;
  gv_local.lookup_slx_nse[ 243].nbits =  8; gv_local.lookup_slx_nse[ 243].bits = 0x1B000000;
  gv_local.lookup_slx_nse[ 244].nbits =  8; gv_local.lookup_slx_nse[ 244].bits = 0x1A000000;
  gv_local.lookup_slx_nse[ 245].nbits =  8; gv_local.lookup_slx_nse[ 245].bits = 0x19000000;
  gv_local.lookup_slx_nse[ 246].nbits =  8; gv_local.lookup_slx_nse[ 246].bits = 0x2C000000;
  gv_local.lookup_slx_nse[ 247].nbits =  8; gv_local.lookup_slx_nse[ 247].bits = 0x35000000;
  gv_local.lookup_slx_nse[ 248].nbits =  8; gv_local.lookup_slx_nse[ 248].bits = 0x37000000;
  gv_local.lookup_slx_nse[ 249].nbits =  8; gv_local.lookup_slx_nse[ 249].bits = 0x32000000;
  gv_local.lookup_slx_nse[ 250].nbits =  8; gv_local.lookup_slx_nse[ 250].bits = 0x39000000;
  gv_local.lookup_slx_nse[ 251].nbits =  8; gv_local.lookup_slx_nse[ 251].bits = 0x47000000;
  gv_local.lookup_slx_nse[ 252].nbits =  8; gv_local.lookup_slx_nse[ 252].bits = 0x2F000000;
  gv_local.lookup_slx_nse[ 253].nbits =  8; gv_local.lookup_slx_nse[ 253].bits = 0x3D000000;
  gv_local.lookup_slx_nse[ 254].nbits =  8; gv_local.lookup_slx_nse[ 254].bits = 0x21000000;
  gv_local.lookup_slx_nse[ 255].nbits =  8; gv_local.lookup_slx_nse[ 255].bits = 0xD1000000;
  gv_local.lookup_slx_nse[ 256].nbits =  8; gv_local.lookup_slx_nse[ 256].bits = 0x25000000;
  gv_local.lookup_slx_nse[ 257].nbits =  8; gv_local.lookup_slx_nse[ 257].bits = 0x2D000000;
  gv_local.lookup_slx_nse[ 258].nbits =  8; gv_local.lookup_slx_nse[ 258].bits = 0x30000000;
  gv_local.lookup_slx_nse[ 259].nbits =  8; gv_local.lookup_slx_nse[ 259].bits = 0x36000000;
  gv_local.lookup_slx_nse[ 260].nbits =  8; gv_local.lookup_slx_nse[ 260].bits = 0x33000000;
  gv_local.lookup_slx_nse[ 261].nbits =  8; gv_local.lookup_slx_nse[ 261].bits = 0x41000000;
  gv_local.lookup_slx_nse[ 262].nbits =  8; gv_local.lookup_slx_nse[ 262].bits = 0x2B000000;
  gv_local.lookup_slx_nse[ 263].nbits =  8; gv_local.lookup_slx_nse[ 263].bits = 0x43000000;
  gv_local.lookup_slx_nse[ 264].nbits =  8; gv_local.lookup_slx_nse[ 264].bits = 0x3B000000;
  gv_local.lookup_slx_nse[ 265].nbits =  8; gv_local.lookup_slx_nse[ 265].bits = 0x2A000000;
  gv_local.lookup_slx_nse[ 266].nbits =  8; gv_local.lookup_slx_nse[ 266].bits = 0x38000000;
  gv_local.lookup_slx_nse[ 267].nbits =  8; gv_local.lookup_slx_nse[ 267].bits = 0x20000000;
  gv_local.lookup_slx_nse[ 268].nbits =  8; gv_local.lookup_slx_nse[ 268].bits = 0x24000000;
  gv_local.lookup_slx_nse[ 269].nbits =  8; gv_local.lookup_slx_nse[ 269].bits = 0x09000000;
  gv_local.lookup_slx_nse[ 270].nbits =  8; gv_local.lookup_slx_nse[ 270].bits = 0x13000000;
  gv_local.lookup_slx_nse[ 271].nbits =  8; gv_local.lookup_slx_nse[ 271].bits = 0x02000000;
  gv_local.lookup_slx_nse[ 272].nbits =  8; gv_local.lookup_slx_nse[ 272].bits = 0x12000000;
  gv_local.lookup_slx_nse[ 273].nbits =  8; gv_local.lookup_slx_nse[ 273].bits = 0x1C000000;
  gv_local.lookup_slx_nse[ 274].nbits =  8; gv_local.lookup_slx_nse[ 274].bits = 0x05000000;
  gv_local.lookup_slx_nse[ 275].nbits =  8; gv_local.lookup_slx_nse[ 275].bits = 0x06000000;
  gv_local.lookup_slx_nse[ 276].nbits =  9; gv_local.lookup_slx_nse[ 276].bits = 0xFD000000;
  gv_local.lookup_slx_nse[ 277].nbits =  8; gv_local.lookup_slx_nse[ 277].bits = 0x00000000;
  gv_local.lookup_slx_nse[ 278].nbits =  9; gv_local.lookup_slx_nse[ 278].bits = 0xFB800000;
  gv_local.lookup_slx_nse[ 279].nbits =  9; gv_local.lookup_slx_nse[ 279].bits = 0xF9800000;
  gv_local.lookup_slx_nse[ 280].nbits =  9; gv_local.lookup_slx_nse[ 280].bits = 0xF6800000;
  gv_local.lookup_slx_nse[ 281].nbits =  9; gv_local.lookup_slx_nse[ 281].bits = 0xEE800000;
  gv_local.lookup_slx_nse[ 282].nbits =  9; gv_local.lookup_slx_nse[ 282].bits = 0xF4800000;
  gv_local.lookup_slx_nse[ 283].nbits =  9; gv_local.lookup_slx_nse[ 283].bits = 0xE5000000;
  gv_local.lookup_slx_nse[ 284].nbits =  9; gv_local.lookup_slx_nse[ 284].bits = 0xEA000000;
  gv_local.lookup_slx_nse[ 285].nbits =  9; gv_local.lookup_slx_nse[ 285].bits = 0xE4000000;
  gv_local.lookup_slx_nse[ 286].nbits =  9; gv_local.lookup_slx_nse[ 286].bits = 0xE6000000;
  gv_local.lookup_slx_nse[ 287].nbits =  9; gv_local.lookup_slx_nse[ 287].bits = 0xDC000000;
  gv_local.lookup_slx_nse[ 288].nbits =  9; gv_local.lookup_slx_nse[ 288].bits = 0xD5800000;
  gv_local.lookup_slx_nse[ 289].nbits =  9; gv_local.lookup_slx_nse[ 289].bits = 0xD4800000;
  gv_local.lookup_slx_nse[ 290].nbits =  9; gv_local.lookup_slx_nse[ 290].bits = 0xC8800000;
  gv_local.lookup_slx_nse[ 291].nbits =  9; gv_local.lookup_slx_nse[ 291].bits = 0xCA800000;
  gv_local.lookup_slx_nse[ 292].nbits =  9; gv_local.lookup_slx_nse[ 292].bits = 0xCA000000;
  gv_local.lookup_slx_nse[ 293].nbits =  9; gv_local.lookup_slx_nse[ 293].bits = 0xCF000000;
  gv_local.lookup_slx_nse[ 294].nbits =  9; gv_local.lookup_slx_nse[ 294].bits = 0xC7000000;
  gv_local.lookup_slx_nse[ 295].nbits =  9; gv_local.lookup_slx_nse[ 295].bits = 0xC0000000;
  gv_local.lookup_slx_nse[ 296].nbits =  9; gv_local.lookup_slx_nse[ 296].bits = 0x7F800000;
  gv_local.lookup_slx_nse[ 297].nbits =  9; gv_local.lookup_slx_nse[ 297].bits = 0x7B800000;
  gv_local.lookup_slx_nse[ 298].nbits =  9; gv_local.lookup_slx_nse[ 298].bits = 0x75800000;
  gv_local.lookup_slx_nse[ 299].nbits =  9; gv_local.lookup_slx_nse[ 299].bits = 0x75000000;
  gv_local.lookup_slx_nse[ 300].nbits =  9; gv_local.lookup_slx_nse[ 300].bits = 0x79800000;
  gv_local.lookup_slx_nse[ 301].nbits =  9; gv_local.lookup_slx_nse[ 301].bits = 0x70800000;
  gv_local.lookup_slx_nse[ 302].nbits =  9; gv_local.lookup_slx_nse[ 302].bits = 0x73800000;
  gv_local.lookup_slx_nse[ 303].nbits =  9; gv_local.lookup_slx_nse[ 303].bits = 0x72800000;
  gv_local.lookup_slx_nse[ 304].nbits =  9; gv_local.lookup_slx_nse[ 304].bits = 0x64800000;
  gv_local.lookup_slx_nse[ 305].nbits =  9; gv_local.lookup_slx_nse[ 305].bits = 0x6C800000;
  gv_local.lookup_slx_nse[ 306].nbits =  9; gv_local.lookup_slx_nse[ 306].bits = 0x6B800000;
  gv_local.lookup_slx_nse[ 307].nbits =  9; gv_local.lookup_slx_nse[ 307].bits = 0x6E000000;
  gv_local.lookup_slx_nse[ 308].nbits =  9; gv_local.lookup_slx_nse[ 308].bits = 0x67000000;
  gv_local.lookup_slx_nse[ 309].nbits =  9; gv_local.lookup_slx_nse[ 309].bits = 0x69000000;
  gv_local.lookup_slx_nse[ 310].nbits =  9; gv_local.lookup_slx_nse[ 310].bits = 0x5F800000;
  gv_local.lookup_slx_nse[ 311].nbits =  9; gv_local.lookup_slx_nse[ 311].bits = 0x61000000;
  gv_local.lookup_slx_nse[ 312].nbits =  9; gv_local.lookup_slx_nse[ 312].bits = 0x62800000;
  gv_local.lookup_slx_nse[ 313].nbits =  9; gv_local.lookup_slx_nse[ 313].bits = 0x66800000;
  gv_local.lookup_slx_nse[ 314].nbits =  9; gv_local.lookup_slx_nse[ 314].bits = 0x60000000;
  gv_local.lookup_slx_nse[ 315].nbits =  9; gv_local.lookup_slx_nse[ 315].bits = 0x5F000000;
  gv_local.lookup_slx_nse[ 316].nbits =  9; gv_local.lookup_slx_nse[ 316].bits = 0x5C800000;
  gv_local.lookup_slx_nse[ 317].nbits =  9; gv_local.lookup_slx_nse[ 317].bits = 0x5D000000;
  gv_local.lookup_slx_nse[ 318].nbits =  9; gv_local.lookup_slx_nse[ 318].bits = 0x5B800000;
  gv_local.lookup_slx_nse[ 319].nbits =  9; gv_local.lookup_slx_nse[ 319].bits = 0x5B000000;
  gv_local.lookup_slx_nse[ 320].nbits =  9; gv_local.lookup_slx_nse[ 320].bits = 0x55800000;
  gv_local.lookup_slx_nse[ 321].nbits =  9; gv_local.lookup_slx_nse[ 321].bits = 0x59000000;
  gv_local.lookup_slx_nse[ 322].nbits =  9; gv_local.lookup_slx_nse[ 322].bits = 0x54000000;
  gv_local.lookup_slx_nse[ 323].nbits =  9; gv_local.lookup_slx_nse[ 323].bits = 0x50000000;
  gv_local.lookup_slx_nse[ 324].nbits =  9; gv_local.lookup_slx_nse[ 324].bits = 0x4F000000;
  gv_local.lookup_slx_nse[ 325].nbits =  9; gv_local.lookup_slx_nse[ 325].bits = 0x51800000;
  gv_local.lookup_slx_nse[ 326].nbits =  9; gv_local.lookup_slx_nse[ 326].bits = 0x53800000;
  gv_local.lookup_slx_nse[ 327].nbits =  9; gv_local.lookup_slx_nse[ 327].bits = 0x4D800000;
  gv_local.lookup_slx_nse[ 328].nbits =  9; gv_local.lookup_slx_nse[ 328].bits = 0x45800000;
  gv_local.lookup_slx_nse[ 329].nbits =  9; gv_local.lookup_slx_nse[ 329].bits = 0x48800000;
  gv_local.lookup_slx_nse[ 330].nbits =  9; gv_local.lookup_slx_nse[ 330].bits = 0x4B000000;
  gv_local.lookup_slx_nse[ 331].nbits =  9; gv_local.lookup_slx_nse[ 331].bits = 0x3E800000;
  gv_local.lookup_slx_nse[ 332].nbits =  9; gv_local.lookup_slx_nse[ 332].bits = 0x44000000;
  gv_local.lookup_slx_nse[ 333].nbits =  9; gv_local.lookup_slx_nse[ 333].bits = 0x3A800000;
  gv_local.lookup_slx_nse[ 334].nbits =  9; gv_local.lookup_slx_nse[ 334].bits = 0x40000000;
  gv_local.lookup_slx_nse[ 335].nbits =  9; gv_local.lookup_slx_nse[ 335].bits = 0x3C000000;
  gv_local.lookup_slx_nse[ 336].nbits =  9; gv_local.lookup_slx_nse[ 336].bits = 0x27000000;
  gv_local.lookup_slx_nse[ 337].nbits =  9; gv_local.lookup_slx_nse[ 337].bits = 0x3F000000;
  gv_local.lookup_slx_nse[ 338].nbits =  9; gv_local.lookup_slx_nse[ 338].bits = 0x1E000000;
  gv_local.lookup_slx_nse[ 339].nbits =  9; gv_local.lookup_slx_nse[ 339].bits = 0x3A000000;
  gv_local.lookup_slx_nse[ 340].nbits =  9; gv_local.lookup_slx_nse[ 340].bits = 0x31000000;
  gv_local.lookup_slx_nse[ 341].nbits =  9; gv_local.lookup_slx_nse[ 341].bits = 0x28800000;
  gv_local.lookup_slx_nse[ 342].nbits =  9; gv_local.lookup_slx_nse[ 342].bits = 0x1D000000;
  gv_local.lookup_slx_nse[ 343].nbits =  9; gv_local.lookup_slx_nse[ 343].bits = 0x34800000;
  gv_local.lookup_slx_nse[ 344].nbits =  9; gv_local.lookup_slx_nse[ 344].bits = 0x14000000;
  gv_local.lookup_slx_nse[ 345].nbits =  9; gv_local.lookup_slx_nse[ 345].bits = 0x23000000;
  gv_local.lookup_slx_nse[ 346].nbits =  9; gv_local.lookup_slx_nse[ 346].bits = 0x23800000;
  gv_local.lookup_slx_nse[ 347].nbits =  9; gv_local.lookup_slx_nse[ 347].bits = 0x0F800000;
  gv_local.lookup_slx_nse[ 348].nbits =  9; gv_local.lookup_slx_nse[ 348].bits = 0x0B000000;
  gv_local.lookup_slx_nse[ 349].nbits =  9; gv_local.lookup_slx_nse[ 349].bits = 0x0D000000;
  gv_local.lookup_slx_nse[ 350].nbits = 10; gv_local.lookup_slx_nse[ 350].bits = 0xFE400000;
  gv_local.lookup_slx_nse[ 351].nbits = 10; gv_local.lookup_slx_nse[ 351].bits = 0xFD800000;
  gv_local.lookup_slx_nse[ 352].nbits =  9; gv_local.lookup_slx_nse[ 352].bits = 0x16800000;
  gv_local.lookup_slx_nse[ 353].nbits =  9; gv_local.lookup_slx_nse[ 353].bits = 0x17800000;
  gv_local.lookup_slx_nse[ 354].nbits = 10; gv_local.lookup_slx_nse[ 354].bits = 0xF7400000;
  gv_local.lookup_slx_nse[ 355].nbits =  9; gv_local.lookup_slx_nse[ 355].bits = 0x0C800000;
  gv_local.lookup_slx_nse[ 356].nbits =  9; gv_local.lookup_slx_nse[ 356].bits = 0x07800000;
  gv_local.lookup_slx_nse[ 357].nbits = 10; gv_local.lookup_slx_nse[ 357].bits = 0xFB000000;
  gv_local.lookup_slx_nse[ 358].nbits = 10; gv_local.lookup_slx_nse[ 358].bits = 0xFC400000;
  gv_local.lookup_slx_nse[ 359].nbits =  9; gv_local.lookup_slx_nse[ 359].bits = 0x0B800000;
  gv_local.lookup_slx_nse[ 360].nbits = 10; gv_local.lookup_slx_nse[ 360].bits = 0xFAC00000;
  gv_local.lookup_slx_nse[ 361].nbits = 10; gv_local.lookup_slx_nse[ 361].bits = 0xFE800000;
  gv_local.lookup_slx_nse[ 362].nbits = 10; gv_local.lookup_slx_nse[ 362].bits = 0xF3800000;
  gv_local.lookup_slx_nse[ 363].nbits = 10; gv_local.lookup_slx_nse[ 363].bits = 0xF5800000;
  gv_local.lookup_slx_nse[ 364].nbits = 10; gv_local.lookup_slx_nse[ 364].bits = 0xF3C00000;
  gv_local.lookup_slx_nse[ 365].nbits = 10; gv_local.lookup_slx_nse[ 365].bits = 0xF1400000;
  gv_local.lookup_slx_nse[ 366].nbits = 10; gv_local.lookup_slx_nse[ 366].bits = 0xF8C00000;
  gv_local.lookup_slx_nse[ 367].nbits = 10; gv_local.lookup_slx_nse[ 367].bits = 0xF8000000;
  gv_local.lookup_slx_nse[ 368].nbits = 10; gv_local.lookup_slx_nse[ 368].bits = 0xF2000000;
  gv_local.lookup_slx_nse[ 369].nbits = 10; gv_local.lookup_slx_nse[ 369].bits = 0xED800000;
  gv_local.lookup_slx_nse[ 370].nbits = 10; gv_local.lookup_slx_nse[ 370].bits = 0xF1C00000;
  gv_local.lookup_slx_nse[ 371].nbits = 10; gv_local.lookup_slx_nse[ 371].bits = 0xF1800000;
  gv_local.lookup_slx_nse[ 372].nbits = 10; gv_local.lookup_slx_nse[ 372].bits = 0xEC400000;
  gv_local.lookup_slx_nse[ 373].nbits = 10; gv_local.lookup_slx_nse[ 373].bits = 0xDE400000;
  gv_local.lookup_slx_nse[ 374].nbits = 10; gv_local.lookup_slx_nse[ 374].bits = 0xF1000000;
  gv_local.lookup_slx_nse[ 375].nbits = 10; gv_local.lookup_slx_nse[ 375].bits = 0xE9400000;
  gv_local.lookup_slx_nse[ 376].nbits = 10; gv_local.lookup_slx_nse[ 376].bits = 0xE8800000;
  gv_local.lookup_slx_nse[ 377].nbits = 10; gv_local.lookup_slx_nse[ 377].bits = 0xEA800000;
  gv_local.lookup_slx_nse[ 378].nbits = 10; gv_local.lookup_slx_nse[ 378].bits = 0xEF000000;
  gv_local.lookup_slx_nse[ 379].nbits = 10; gv_local.lookup_slx_nse[ 379].bits = 0xE1C00000;
  gv_local.lookup_slx_nse[ 380].nbits = 10; gv_local.lookup_slx_nse[ 380].bits = 0xE8C00000;
  gv_local.lookup_slx_nse[ 381].nbits = 10; gv_local.lookup_slx_nse[ 381].bits = 0xE7800000;
  gv_local.lookup_slx_nse[ 382].nbits = 10; gv_local.lookup_slx_nse[ 382].bits = 0xF6000000;
  gv_local.lookup_slx_nse[ 383].nbits = 10; gv_local.lookup_slx_nse[ 383].bits = 0xDB800000;
  gv_local.lookup_slx_nse[ 384].nbits = 10; gv_local.lookup_slx_nse[ 384].bits = 0xDAC00000;
  gv_local.lookup_slx_nse[ 385].nbits = 10; gv_local.lookup_slx_nse[ 385].bits = 0xD9C00000;
  gv_local.lookup_slx_nse[ 386].nbits = 10; gv_local.lookup_slx_nse[ 386].bits = 0xDD000000;
  gv_local.lookup_slx_nse[ 387].nbits = 10; gv_local.lookup_slx_nse[ 387].bits = 0xE8400000;
  gv_local.lookup_slx_nse[ 388].nbits = 10; gv_local.lookup_slx_nse[ 388].bits = 0xE5C00000;
  gv_local.lookup_slx_nse[ 389].nbits = 10; gv_local.lookup_slx_nse[ 389].bits = 0xE0000000;
  gv_local.lookup_slx_nse[ 390].nbits = 10; gv_local.lookup_slx_nse[ 390].bits = 0xE3000000;
  gv_local.lookup_slx_nse[ 391].nbits = 10; gv_local.lookup_slx_nse[ 391].bits = 0xD8C00000;
  gv_local.lookup_slx_nse[ 392].nbits = 10; gv_local.lookup_slx_nse[ 392].bits = 0xCCC00000;
  gv_local.lookup_slx_nse[ 393].nbits = 10; gv_local.lookup_slx_nse[ 393].bits = 0xD9000000;
  gv_local.lookup_slx_nse[ 394].nbits = 10; gv_local.lookup_slx_nse[ 394].bits = 0xE4800000;
  gv_local.lookup_slx_nse[ 395].nbits = 10; gv_local.lookup_slx_nse[ 395].bits = 0xE9C00000;
  gv_local.lookup_slx_nse[ 396].nbits = 10; gv_local.lookup_slx_nse[ 396].bits = 0xE3400000;
  gv_local.lookup_slx_nse[ 397].nbits = 10; gv_local.lookup_slx_nse[ 397].bits = 0xD7400000;
  gv_local.lookup_slx_nse[ 398].nbits = 10; gv_local.lookup_slx_nse[ 398].bits = 0xD6800000;
  gv_local.lookup_slx_nse[ 399].nbits = 10; gv_local.lookup_slx_nse[ 399].bits = 0xC9800000;
  gv_local.lookup_slx_nse[ 400].nbits = 10; gv_local.lookup_slx_nse[ 400].bits = 0xCF800000;
  gv_local.lookup_slx_nse[ 401].nbits = 10; gv_local.lookup_slx_nse[ 401].bits = 0xD0400000;
  gv_local.lookup_slx_nse[ 402].nbits = 10; gv_local.lookup_slx_nse[ 402].bits = 0xD5400000;
  gv_local.lookup_slx_nse[ 403].nbits = 10; gv_local.lookup_slx_nse[ 403].bits = 0xD0800000;
  gv_local.lookup_slx_nse[ 404].nbits = 10; gv_local.lookup_slx_nse[ 404].bits = 0xDA400000;
  gv_local.lookup_slx_nse[ 405].nbits = 10; gv_local.lookup_slx_nse[ 405].bits = 0xD2000000;
  gv_local.lookup_slx_nse[ 406].nbits = 10; gv_local.lookup_slx_nse[ 406].bits = 0xD3400000;
  gv_local.lookup_slx_nse[ 407].nbits = 10; gv_local.lookup_slx_nse[ 407].bits = 0xD3C00000;
  gv_local.lookup_slx_nse[ 408].nbits = 10; gv_local.lookup_slx_nse[ 408].bits = 0xD8000000;
  gv_local.lookup_slx_nse[ 409].nbits = 10; gv_local.lookup_slx_nse[ 409].bits = 0xC5000000;
  gv_local.lookup_slx_nse[ 410].nbits = 10; gv_local.lookup_slx_nse[ 410].bits = 0xC9400000;
  gv_local.lookup_slx_nse[ 411].nbits = 10; gv_local.lookup_slx_nse[ 411].bits = 0xD6C00000;
  gv_local.lookup_slx_nse[ 412].nbits = 10; gv_local.lookup_slx_nse[ 412].bits = 0xCD000000;
  gv_local.lookup_slx_nse[ 413].nbits = 10; gv_local.lookup_slx_nse[ 413].bits = 0xDF000000;
  gv_local.lookup_slx_nse[ 414].nbits = 10; gv_local.lookup_slx_nse[ 414].bits = 0xC8400000;
  gv_local.lookup_slx_nse[ 415].nbits = 10; gv_local.lookup_slx_nse[ 415].bits = 0xC4800000;
  gv_local.lookup_slx_nse[ 416].nbits = 10; gv_local.lookup_slx_nse[ 416].bits = 0xD2400000;
  gv_local.lookup_slx_nse[ 417].nbits = 10; gv_local.lookup_slx_nse[ 417].bits = 0xCC400000;
  gv_local.lookup_slx_nse[ 418].nbits = 10; gv_local.lookup_slx_nse[ 418].bits = 0xC5800000;
  gv_local.lookup_slx_nse[ 419].nbits = 10; gv_local.lookup_slx_nse[ 419].bits = 0xDD800000;
  gv_local.lookup_slx_nse[ 420].nbits = 10; gv_local.lookup_slx_nse[ 420].bits = 0xC8000000;
  gv_local.lookup_slx_nse[ 421].nbits = 10; gv_local.lookup_slx_nse[ 421].bits = 0xCFC00000;
  gv_local.lookup_slx_nse[ 422].nbits = 10; gv_local.lookup_slx_nse[ 422].bits = 0xC3C00000;
  gv_local.lookup_slx_nse[ 423].nbits = 10; gv_local.lookup_slx_nse[ 423].bits = 0xC6000000;
  gv_local.lookup_slx_nse[ 424].nbits = 10; gv_local.lookup_slx_nse[ 424].bits = 0xD9400000;
  gv_local.lookup_slx_nse[ 425].nbits = 10; gv_local.lookup_slx_nse[ 425].bits = 0xCB000000;
  gv_local.lookup_slx_nse[ 426].nbits = 10; gv_local.lookup_slx_nse[ 426].bits = 0xCC000000;
  gv_local.lookup_slx_nse[ 427].nbits = 10; gv_local.lookup_slx_nse[ 427].bits = 0x7C000000;
  gv_local.lookup_slx_nse[ 428].nbits = 10; gv_local.lookup_slx_nse[ 428].bits = 0xCBC00000;
  gv_local.lookup_slx_nse[ 429].nbits = 10; gv_local.lookup_slx_nse[ 429].bits = 0x77400000;
  gv_local.lookup_slx_nse[ 430].nbits = 10; gv_local.lookup_slx_nse[ 430].bits = 0xC2400000;
  gv_local.lookup_slx_nse[ 431].nbits = 10; gv_local.lookup_slx_nse[ 431].bits = 0xC7800000;
  gv_local.lookup_slx_nse[ 432].nbits = 10; gv_local.lookup_slx_nse[ 432].bits = 0xC1C00000;
  gv_local.lookup_slx_nse[ 433].nbits = 10; gv_local.lookup_slx_nse[ 433].bits = 0xC0800000;
  gv_local.lookup_slx_nse[ 434].nbits = 10; gv_local.lookup_slx_nse[ 434].bits = 0xC5400000;
  gv_local.lookup_slx_nse[ 435].nbits = 10; gv_local.lookup_slx_nse[ 435].bits = 0x7E400000;
  gv_local.lookup_slx_nse[ 436].nbits = 10; gv_local.lookup_slx_nse[ 436].bits = 0x7E000000;
  gv_local.lookup_slx_nse[ 437].nbits = 10; gv_local.lookup_slx_nse[ 437].bits = 0xC1000000;
  gv_local.lookup_slx_nse[ 438].nbits = 10; gv_local.lookup_slx_nse[ 438].bits = 0xC5C00000;
  gv_local.lookup_slx_nse[ 439].nbits = 10; gv_local.lookup_slx_nse[ 439].bits = 0x7A400000;
  gv_local.lookup_slx_nse[ 440].nbits = 10; gv_local.lookup_slx_nse[ 440].bits = 0x7A000000;
  gv_local.lookup_slx_nse[ 441].nbits = 10; gv_local.lookup_slx_nse[ 441].bits = 0x70400000;
  gv_local.lookup_slx_nse[ 442].nbits = 10; gv_local.lookup_slx_nse[ 442].bits = 0x7F000000;
  gv_local.lookup_slx_nse[ 443].nbits = 10; gv_local.lookup_slx_nse[ 443].bits = 0x73400000;
  gv_local.lookup_slx_nse[ 444].nbits = 10; gv_local.lookup_slx_nse[ 444].bits = 0x7C800000;
  gv_local.lookup_slx_nse[ 445].nbits = 10; gv_local.lookup_slx_nse[ 445].bits = 0x7B400000;
  gv_local.lookup_slx_nse[ 446].nbits = 10; gv_local.lookup_slx_nse[ 446].bits = 0x74C00000;
  gv_local.lookup_slx_nse[ 447].nbits = 10; gv_local.lookup_slx_nse[ 447].bits = 0x6E800000;
  gv_local.lookup_slx_nse[ 448].nbits = 10; gv_local.lookup_slx_nse[ 448].bits = 0x77C00000;
  gv_local.lookup_slx_nse[ 449].nbits = 10; gv_local.lookup_slx_nse[ 449].bits = 0x73000000;
  gv_local.lookup_slx_nse[ 450].nbits = 10; gv_local.lookup_slx_nse[ 450].bits = 0x71800000;
  gv_local.lookup_slx_nse[ 451].nbits = 10; gv_local.lookup_slx_nse[ 451].bits = 0x6C400000;
  gv_local.lookup_slx_nse[ 452].nbits = 10; gv_local.lookup_slx_nse[ 452].bits = 0x68C00000;
  gv_local.lookup_slx_nse[ 453].nbits = 10; gv_local.lookup_slx_nse[ 453].bits = 0x71C00000;
  gv_local.lookup_slx_nse[ 454].nbits = 10; gv_local.lookup_slx_nse[ 454].bits = 0x68800000;
  gv_local.lookup_slx_nse[ 455].nbits = 10; gv_local.lookup_slx_nse[ 455].bits = 0x69C00000;
  gv_local.lookup_slx_nse[ 456].nbits = 10; gv_local.lookup_slx_nse[ 456].bits = 0x6EC00000;
  gv_local.lookup_slx_nse[ 457].nbits = 10; gv_local.lookup_slx_nse[ 457].bits = 0x65000000;
  gv_local.lookup_slx_nse[ 458].nbits = 10; gv_local.lookup_slx_nse[ 458].bits = 0x67800000;
  gv_local.lookup_slx_nse[ 459].nbits = 10; gv_local.lookup_slx_nse[ 459].bits = 0x5C400000;
  gv_local.lookup_slx_nse[ 460].nbits = 10; gv_local.lookup_slx_nse[ 460].bits = 0x56400000;
  gv_local.lookup_slx_nse[ 461].nbits = 10; gv_local.lookup_slx_nse[ 461].bits = 0x63400000;
  gv_local.lookup_slx_nse[ 462].nbits = 10; gv_local.lookup_slx_nse[ 462].bits = 0x60C00000;
  gv_local.lookup_slx_nse[ 463].nbits = 10; gv_local.lookup_slx_nse[ 463].bits = 0x65400000;
  gv_local.lookup_slx_nse[ 464].nbits = 10; gv_local.lookup_slx_nse[ 464].bits = 0x66000000;
  gv_local.lookup_slx_nse[ 465].nbits = 10; gv_local.lookup_slx_nse[ 465].bits = 0x63C00000;
  gv_local.lookup_slx_nse[ 466].nbits = 10; gv_local.lookup_slx_nse[ 466].bits = 0x5AC00000;
  gv_local.lookup_slx_nse[ 467].nbits = 10; gv_local.lookup_slx_nse[ 467].bits = 0x55400000;
  gv_local.lookup_slx_nse[ 468].nbits = 10; gv_local.lookup_slx_nse[ 468].bits = 0x63000000;
  gv_local.lookup_slx_nse[ 469].nbits = 10; gv_local.lookup_slx_nse[ 469].bits = 0x5A800000;
  gv_local.lookup_slx_nse[ 470].nbits = 10; gv_local.lookup_slx_nse[ 470].bits = 0x55000000;
  gv_local.lookup_slx_nse[ 471].nbits = 10; gv_local.lookup_slx_nse[ 471].bits = 0x58000000;
  gv_local.lookup_slx_nse[ 472].nbits = 10; gv_local.lookup_slx_nse[ 472].bits = 0x4D000000;
  gv_local.lookup_slx_nse[ 473].nbits = 10; gv_local.lookup_slx_nse[ 473].bits = 0x50C00000;
  gv_local.lookup_slx_nse[ 474].nbits = 10; gv_local.lookup_slx_nse[ 474].bits = 0x46800000;
  gv_local.lookup_slx_nse[ 475].nbits = 10; gv_local.lookup_slx_nse[ 475].bits = 0x44800000;
  gv_local.lookup_slx_nse[ 476].nbits = 10; gv_local.lookup_slx_nse[ 476].bits = 0x4C000000;
  gv_local.lookup_slx_nse[ 477].nbits = 10; gv_local.lookup_slx_nse[ 477].bits = 0x52400000;
  gv_local.lookup_slx_nse[ 478].nbits = 10; gv_local.lookup_slx_nse[ 478].bits = 0x4D400000;
  gv_local.lookup_slx_nse[ 479].nbits = 10; gv_local.lookup_slx_nse[ 479].bits = 0x42400000;
  gv_local.lookup_slx_nse[ 480].nbits = 10; gv_local.lookup_slx_nse[ 480].bits = 0x42000000;
  gv_local.lookup_slx_nse[ 481].nbits = 10; gv_local.lookup_slx_nse[ 481].bits = 0x42800000;
  gv_local.lookup_slx_nse[ 482].nbits = 10; gv_local.lookup_slx_nse[ 482].bits = 0x27800000;
  gv_local.lookup_slx_nse[ 483].nbits = 10; gv_local.lookup_slx_nse[ 483].bits = 0x46000000;
  gv_local.lookup_slx_nse[ 484].nbits = 10; gv_local.lookup_slx_nse[ 484].bits = 0x44C00000;
  gv_local.lookup_slx_nse[ 485].nbits = 10; gv_local.lookup_slx_nse[ 485].bits = 0x3C800000;
  gv_local.lookup_slx_nse[ 486].nbits = 10; gv_local.lookup_slx_nse[ 486].bits = 0x4C400000;
  gv_local.lookup_slx_nse[ 487].nbits = 10; gv_local.lookup_slx_nse[ 487].bits = 0x14800000;
  gv_local.lookup_slx_nse[ 488].nbits = 10; gv_local.lookup_slx_nse[ 488].bits = 0x22C00000;
  gv_local.lookup_slx_nse[ 489].nbits = 11; gv_local.lookup_slx_nse[ 489].bits = 0xFE000000;
  gv_local.lookup_slx_nse[ 490].nbits = 10; gv_local.lookup_slx_nse[ 490].bits = 0x0A400000;
  gv_local.lookup_slx_nse[ 491].nbits = 11; gv_local.lookup_slx_nse[ 491].bits = 0xFDC00000;
  gv_local.lookup_slx_nse[ 492].nbits = 10; gv_local.lookup_slx_nse[ 492].bits = 0x14C00000;
  gv_local.lookup_slx_nse[ 493].nbits = 10; gv_local.lookup_slx_nse[ 493].bits = 0x01C00000;
  gv_local.lookup_slx_nse[ 494].nbits = 11; gv_local.lookup_slx_nse[ 494].bits = 0xFE200000;
  gv_local.lookup_slx_nse[ 495].nbits = 11; gv_local.lookup_slx_nse[ 495].bits = 0xFDE00000;
  gv_local.lookup_slx_nse[ 496].nbits = 10; gv_local.lookup_slx_nse[ 496].bits = 0x01800000;
  gv_local.lookup_slx_nse[ 497].nbits = 11; gv_local.lookup_slx_nse[ 497].bits = 0xF6400000;
  gv_local.lookup_slx_nse[ 498].nbits = 11; gv_local.lookup_slx_nse[ 498].bits = 0xF2E00000;
  gv_local.lookup_slx_nse[ 499].nbits = 11; gv_local.lookup_slx_nse[ 499].bits = 0xEE600000;
  gv_local.lookup_slx_nse[ 500].nbits = 11; gv_local.lookup_slx_nse[ 500].bits = 0xEBE00000;
  gv_local.lookup_slx_nse[ 501].nbits = 11; gv_local.lookup_slx_nse[ 501].bits = 0xCE600000;
  gv_local.lookup_slx_nse[ 502].nbits = 11; gv_local.lookup_slx_nse[ 502].bits = 0xE8200000;
  gv_local.lookup_slx_nse[ 503].nbits = 11; gv_local.lookup_slx_nse[ 503].bits = 0xDA200000;
  gv_local.lookup_slx_nse[ 504].nbits = 11; gv_local.lookup_slx_nse[ 504].bits = 0xD0200000;
  gv_local.lookup_slx_nse[ 505].nbits = 11; gv_local.lookup_slx_nse[ 505].bits = 0xDFE00000;
  gv_local.lookup_slx_nse[ 506].nbits = 11; gv_local.lookup_slx_nse[ 506].bits = 0xDB600000;
  gv_local.lookup_slx_nse[ 507].nbits = 11; gv_local.lookup_slx_nse[ 507].bits = 0xD5200000;
  gv_local.lookup_slx_nse[ 508].nbits = 11; gv_local.lookup_slx_nse[ 508].bits = 0xDB400000;
  gv_local.lookup_slx_nse[ 509].nbits = 11; gv_local.lookup_slx_nse[ 509].bits = 0xDDC00000;
  gv_local.lookup_slx_nse[ 510].nbits = 11; gv_local.lookup_slx_nse[ 510].bits = 0xC6800000;
  gv_local.lookup_slx_nse[ 511].nbits =  2; gv_local.lookup_slx_nse[ 511].bits = 0x80000000;

#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  /* reverse huffman lookup table
   */
  memset (gv_local.rlookup_slx_nse, 0, sizeof(uint16_t)*REV_HUFF_TREESIZE_16);

  for (i=0; i<NSE_HUFF_TREESIZE; i++)
  {
    val = gv_local.lookup_slx_nse[i].bits >> 16;
    gv_local.rlookup_slx_nse [val] = (uint16_t) i;
    for (j=1; j<=(0xFFFF>>gv_local.lookup_slx_nse[i].nbits); j++)
      gv_local.rlookup_slx_nse [val+j] = (uint16_t) i;
  }
#endif

  _s_sort_lookup (gv_local.lookup_slx_nse, gv_local.idx_slx_nse, NSE_HUFF_TREESIZE);

  gv_local.lookup_slx_nse_initialized = 1;
}

/*
 */
static void _s_init_seq_default (void)
{
  memcpy (gv_local.conv2to4na[  0], "AAAA", 4);  memcpy (gv_local.conv2to4na[  1], "AAAC", 4);
  memcpy (gv_local.conv2to4na[  2], "AAAG", 4);  memcpy (gv_local.conv2to4na[  3], "AAAT", 4);
  memcpy (gv_local.conv2to4na[  4], "AACA", 4);  memcpy (gv_local.conv2to4na[  5], "AACC", 4);
  memcpy (gv_local.conv2to4na[  6], "AACG", 4);  memcpy (gv_local.conv2to4na[  7], "AACT", 4);
  memcpy (gv_local.conv2to4na[  8], "AAGA", 4);  memcpy (gv_local.conv2to4na[  9], "AAGC", 4);
  memcpy (gv_local.conv2to4na[ 10], "AAGG", 4);  memcpy (gv_local.conv2to4na[ 11], "AAGT", 4);
  memcpy (gv_local.conv2to4na[ 12], "AATA", 4);  memcpy (gv_local.conv2to4na[ 13], "AATC", 4);
  memcpy (gv_local.conv2to4na[ 14], "AATG", 4);  memcpy (gv_local.conv2to4na[ 15], "AATT", 4);
  memcpy (gv_local.conv2to4na[ 16], "ACAA", 4);  memcpy (gv_local.conv2to4na[ 17], "ACAC", 4);
  memcpy (gv_local.conv2to4na[ 18], "ACAG", 4);  memcpy (gv_local.conv2to4na[ 19], "ACAT", 4);
  memcpy (gv_local.conv2to4na[ 20], "ACCA", 4);  memcpy (gv_local.conv2to4na[ 21], "ACCC", 4);
  memcpy (gv_local.conv2to4na[ 22], "ACCG", 4);  memcpy (gv_local.conv2to4na[ 23], "ACCT", 4);
  memcpy (gv_local.conv2to4na[ 24], "ACGA", 4);  memcpy (gv_local.conv2to4na[ 25], "ACGC", 4);
  memcpy (gv_local.conv2to4na[ 26], "ACGG", 4);  memcpy (gv_local.conv2to4na[ 27], "ACGT", 4);
  memcpy (gv_local.conv2to4na[ 28], "ACTA", 4);  memcpy (gv_local.conv2to4na[ 29], "ACTC", 4);
  memcpy (gv_local.conv2to4na[ 30], "ACTG", 4);  memcpy (gv_local.conv2to4na[ 31], "ACTT", 4);
  memcpy (gv_local.conv2to4na[ 32], "AGAA", 4);  memcpy (gv_local.conv2to4na[ 33], "AGAC", 4);
  memcpy (gv_local.conv2to4na[ 34], "AGAG", 4);  memcpy (gv_local.conv2to4na[ 35], "AGAT", 4);
  memcpy (gv_local.conv2to4na[ 36], "AGCA", 4);  memcpy (gv_local.conv2to4na[ 37], "AGCC", 4);
  memcpy (gv_local.conv2to4na[ 38], "AGCG", 4);  memcpy (gv_local.conv2to4na[ 39], "AGCT", 4);
  memcpy (gv_local.conv2to4na[ 40], "AGGA", 4);  memcpy (gv_local.conv2to4na[ 41], "AGGC", 4);
  memcpy (gv_local.conv2to4na[ 42], "AGGG", 4);  memcpy (gv_local.conv2to4na[ 43], "AGGT", 4);
  memcpy (gv_local.conv2to4na[ 44], "AGTA", 4);  memcpy (gv_local.conv2to4na[ 45], "AGTC", 4);
  memcpy (gv_local.conv2to4na[ 46], "AGTG", 4);  memcpy (gv_local.conv2to4na[ 47], "AGTT", 4);
  memcpy (gv_local.conv2to4na[ 48], "ATAA", 4);  memcpy (gv_local.conv2to4na[ 49], "ATAC", 4);
  memcpy (gv_local.conv2to4na[ 50], "ATAG", 4);  memcpy (gv_local.conv2to4na[ 51], "ATAT", 4);
  memcpy (gv_local.conv2to4na[ 52], "ATCA", 4);  memcpy (gv_local.conv2to4na[ 53], "ATCC", 4);
  memcpy (gv_local.conv2to4na[ 54], "ATCG", 4);  memcpy (gv_local.conv2to4na[ 55], "ATCT", 4);
  memcpy (gv_local.conv2to4na[ 56], "ATGA", 4);  memcpy (gv_local.conv2to4na[ 57], "ATGC", 4);
  memcpy (gv_local.conv2to4na[ 58], "ATGG", 4);  memcpy (gv_local.conv2to4na[ 59], "ATGT", 4);
  memcpy (gv_local.conv2to4na[ 60], "ATTA", 4);  memcpy (gv_local.conv2to4na[ 61], "ATTC", 4);
  memcpy (gv_local.conv2to4na[ 62], "ATTG", 4);  memcpy (gv_local.conv2to4na[ 63], "ATTT", 4);
  memcpy (gv_local.conv2to4na[ 64], "CAAA", 4);  memcpy (gv_local.conv2to4na[ 65], "CAAC", 4);
  memcpy (gv_local.conv2to4na[ 66], "CAAG", 4);  memcpy (gv_local.conv2to4na[ 67], "CAAT", 4);
  memcpy (gv_local.conv2to4na[ 68], "CACA", 4);  memcpy (gv_local.conv2to4na[ 69], "CACC", 4);
  memcpy (gv_local.conv2to4na[ 70], "CACG", 4);  memcpy (gv_local.conv2to4na[ 71], "CACT", 4);
  memcpy (gv_local.conv2to4na[ 72], "CAGA", 4);  memcpy (gv_local.conv2to4na[ 73], "CAGC", 4);
  memcpy (gv_local.conv2to4na[ 74], "CAGG", 4);  memcpy (gv_local.conv2to4na[ 75], "CAGT", 4);
  memcpy (gv_local.conv2to4na[ 76], "CATA", 4);  memcpy (gv_local.conv2to4na[ 77], "CATC", 4);
  memcpy (gv_local.conv2to4na[ 78], "CATG", 4);  memcpy (gv_local.conv2to4na[ 79], "CATT", 4);
  memcpy (gv_local.conv2to4na[ 80], "CCAA", 4);  memcpy (gv_local.conv2to4na[ 81], "CCAC", 4);
  memcpy (gv_local.conv2to4na[ 82], "CCAG", 4);  memcpy (gv_local.conv2to4na[ 83], "CCAT", 4);
  memcpy (gv_local.conv2to4na[ 84], "CCCA", 4);  memcpy (gv_local.conv2to4na[ 85], "CCCC", 4);
  memcpy (gv_local.conv2to4na[ 86], "CCCG", 4);  memcpy (gv_local.conv2to4na[ 87], "CCCT", 4);
  memcpy (gv_local.conv2to4na[ 88], "CCGA", 4);  memcpy (gv_local.conv2to4na[ 89], "CCGC", 4);
  memcpy (gv_local.conv2to4na[ 90], "CCGG", 4);  memcpy (gv_local.conv2to4na[ 91], "CCGT", 4);
  memcpy (gv_local.conv2to4na[ 92], "CCTA", 4);  memcpy (gv_local.conv2to4na[ 93], "CCTC", 4);
  memcpy (gv_local.conv2to4na[ 94], "CCTG", 4);  memcpy (gv_local.conv2to4na[ 95], "CCTT", 4);
  memcpy (gv_local.conv2to4na[ 96], "CGAA", 4);  memcpy (gv_local.conv2to4na[ 97], "CGAC", 4);
  memcpy (gv_local.conv2to4na[ 98], "CGAG", 4);  memcpy (gv_local.conv2to4na[ 99], "CGAT", 4);
  memcpy (gv_local.conv2to4na[100], "CGCA", 4);  memcpy (gv_local.conv2to4na[101], "CGCC", 4);
  memcpy (gv_local.conv2to4na[102], "CGCG", 4);  memcpy (gv_local.conv2to4na[103], "CGCT", 4);
  memcpy (gv_local.conv2to4na[104], "CGGA", 4);  memcpy (gv_local.conv2to4na[105], "CGGC", 4);
  memcpy (gv_local.conv2to4na[106], "CGGG", 4);  memcpy (gv_local.conv2to4na[107], "CGGT", 4);
  memcpy (gv_local.conv2to4na[108], "CGTA", 4);  memcpy (gv_local.conv2to4na[109], "CGTC", 4);
  memcpy (gv_local.conv2to4na[110], "CGTG", 4);  memcpy (gv_local.conv2to4na[111], "CGTT", 4);
  memcpy (gv_local.conv2to4na[112], "CTAA", 4);  memcpy (gv_local.conv2to4na[113], "CTAC", 4);
  memcpy (gv_local.conv2to4na[114], "CTAG", 4);  memcpy (gv_local.conv2to4na[115], "CTAT", 4);
  memcpy (gv_local.conv2to4na[116], "CTCA", 4);  memcpy (gv_local.conv2to4na[117], "CTCC", 4);
  memcpy (gv_local.conv2to4na[118], "CTCG", 4);  memcpy (gv_local.conv2to4na[119], "CTCT", 4);
  memcpy (gv_local.conv2to4na[120], "CTGA", 4);  memcpy (gv_local.conv2to4na[121], "CTGC", 4);
  memcpy (gv_local.conv2to4na[122], "CTGG", 4);  memcpy (gv_local.conv2to4na[123], "CTGT", 4);
  memcpy (gv_local.conv2to4na[124], "CTTA", 4);  memcpy (gv_local.conv2to4na[125], "CTTC", 4);
  memcpy (gv_local.conv2to4na[126], "CTTG", 4);  memcpy (gv_local.conv2to4na[127], "CTTT", 4);
  memcpy (gv_local.conv2to4na[128], "GAAA", 4);  memcpy (gv_local.conv2to4na[129], "GAAC", 4);
  memcpy (gv_local.conv2to4na[130], "GAAG", 4);  memcpy (gv_local.conv2to4na[131], "GAAT", 4);
  memcpy (gv_local.conv2to4na[132], "GACA", 4);  memcpy (gv_local.conv2to4na[133], "GACC", 4);
  memcpy (gv_local.conv2to4na[134], "GACG", 4);  memcpy (gv_local.conv2to4na[135], "GACT", 4);
  memcpy (gv_local.conv2to4na[136], "GAGA", 4);  memcpy (gv_local.conv2to4na[137], "GAGC", 4);
  memcpy (gv_local.conv2to4na[138], "GAGG", 4);  memcpy (gv_local.conv2to4na[139], "GAGT", 4);
  memcpy (gv_local.conv2to4na[140], "GATA", 4);  memcpy (gv_local.conv2to4na[141], "GATC", 4);
  memcpy (gv_local.conv2to4na[142], "GATG", 4);  memcpy (gv_local.conv2to4na[143], "GATT", 4);
  memcpy (gv_local.conv2to4na[144], "GCAA", 4);  memcpy (gv_local.conv2to4na[145], "GCAC", 4);
  memcpy (gv_local.conv2to4na[146], "GCAG", 4);  memcpy (gv_local.conv2to4na[147], "GCAT", 4);
  memcpy (gv_local.conv2to4na[148], "GCCA", 4);  memcpy (gv_local.conv2to4na[149], "GCCC", 4);
  memcpy (gv_local.conv2to4na[150], "GCCG", 4);  memcpy (gv_local.conv2to4na[151], "GCCT", 4);
  memcpy (gv_local.conv2to4na[152], "GCGA", 4);  memcpy (gv_local.conv2to4na[153], "GCGC", 4);
  memcpy (gv_local.conv2to4na[154], "GCGG", 4);  memcpy (gv_local.conv2to4na[155], "GCGT", 4);
  memcpy (gv_local.conv2to4na[156], "GCTA", 4);  memcpy (gv_local.conv2to4na[157], "GCTC", 4);
  memcpy (gv_local.conv2to4na[158], "GCTG", 4);  memcpy (gv_local.conv2to4na[159], "GCTT", 4);
  memcpy (gv_local.conv2to4na[160], "GGAA", 4);  memcpy (gv_local.conv2to4na[161], "GGAC", 4);
  memcpy (gv_local.conv2to4na[162], "GGAG", 4);  memcpy (gv_local.conv2to4na[163], "GGAT", 4);
  memcpy (gv_local.conv2to4na[164], "GGCA", 4);  memcpy (gv_local.conv2to4na[165], "GGCC", 4);
  memcpy (gv_local.conv2to4na[166], "GGCG", 4);  memcpy (gv_local.conv2to4na[167], "GGCT", 4);
  memcpy (gv_local.conv2to4na[168], "GGGA", 4);  memcpy (gv_local.conv2to4na[169], "GGGC", 4);
  memcpy (gv_local.conv2to4na[170], "GGGG", 4);  memcpy (gv_local.conv2to4na[171], "GGGT", 4);
  memcpy (gv_local.conv2to4na[172], "GGTA", 4);  memcpy (gv_local.conv2to4na[173], "GGTC", 4);
  memcpy (gv_local.conv2to4na[174], "GGTG", 4);  memcpy (gv_local.conv2to4na[175], "GGTT", 4);
  memcpy (gv_local.conv2to4na[176], "GTAA", 4);  memcpy (gv_local.conv2to4na[177], "GTAC", 4);
  memcpy (gv_local.conv2to4na[178], "GTAG", 4);  memcpy (gv_local.conv2to4na[179], "GTAT", 4);
  memcpy (gv_local.conv2to4na[180], "GTCA", 4);  memcpy (gv_local.conv2to4na[181], "GTCC", 4);
  memcpy (gv_local.conv2to4na[182], "GTCG", 4);  memcpy (gv_local.conv2to4na[183], "GTCT", 4);
  memcpy (gv_local.conv2to4na[184], "GTGA", 4);  memcpy (gv_local.conv2to4na[185], "GTGC", 4);
  memcpy (gv_local.conv2to4na[186], "GTGG", 4);  memcpy (gv_local.conv2to4na[187], "GTGT", 4);
  memcpy (gv_local.conv2to4na[188], "GTTA", 4);  memcpy (gv_local.conv2to4na[189], "GTTC", 4);
  memcpy (gv_local.conv2to4na[190], "GTTG", 4);  memcpy (gv_local.conv2to4na[191], "GTTT", 4);
  memcpy (gv_local.conv2to4na[192], "TAAA", 4);  memcpy (gv_local.conv2to4na[193], "TAAC", 4);
  memcpy (gv_local.conv2to4na[194], "TAAG", 4);  memcpy (gv_local.conv2to4na[195], "TAAT", 4);
  memcpy (gv_local.conv2to4na[196], "TACA", 4);  memcpy (gv_local.conv2to4na[197], "TACC", 4);
  memcpy (gv_local.conv2to4na[198], "TACG", 4);  memcpy (gv_local.conv2to4na[199], "TACT", 4);
  memcpy (gv_local.conv2to4na[200], "TAGA", 4);  memcpy (gv_local.conv2to4na[201], "TAGC", 4);
  memcpy (gv_local.conv2to4na[202], "TAGG", 4);  memcpy (gv_local.conv2to4na[203], "TAGT", 4);
  memcpy (gv_local.conv2to4na[204], "TATA", 4);  memcpy (gv_local.conv2to4na[205], "TATC", 4);
  memcpy (gv_local.conv2to4na[206], "TATG", 4);  memcpy (gv_local.conv2to4na[207], "TATT", 4);
  memcpy (gv_local.conv2to4na[208], "TCAA", 4);  memcpy (gv_local.conv2to4na[209], "TCAC", 4);
  memcpy (gv_local.conv2to4na[210], "TCAG", 4);  memcpy (gv_local.conv2to4na[211], "TCAT", 4);
  memcpy (gv_local.conv2to4na[212], "TCCA", 4);  memcpy (gv_local.conv2to4na[213], "TCCC", 4);
  memcpy (gv_local.conv2to4na[214], "TCCG", 4);  memcpy (gv_local.conv2to4na[215], "TCCT", 4);
  memcpy (gv_local.conv2to4na[216], "TCGA", 4);  memcpy (gv_local.conv2to4na[217], "TCGC", 4);
  memcpy (gv_local.conv2to4na[218], "TCGG", 4);  memcpy (gv_local.conv2to4na[219], "TCGT", 4);
  memcpy (gv_local.conv2to4na[220], "TCTA", 4);  memcpy (gv_local.conv2to4na[221], "TCTC", 4);
  memcpy (gv_local.conv2to4na[222], "TCTG", 4);  memcpy (gv_local.conv2to4na[223], "TCTT", 4);
  memcpy (gv_local.conv2to4na[224], "TGAA", 4);  memcpy (gv_local.conv2to4na[225], "TGAC", 4);
  memcpy (gv_local.conv2to4na[226], "TGAG", 4);  memcpy (gv_local.conv2to4na[227], "TGAT", 4);
  memcpy (gv_local.conv2to4na[228], "TGCA", 4);  memcpy (gv_local.conv2to4na[229], "TGCC", 4);
  memcpy (gv_local.conv2to4na[230], "TGCG", 4);  memcpy (gv_local.conv2to4na[231], "TGCT", 4);
  memcpy (gv_local.conv2to4na[232], "TGGA", 4);  memcpy (gv_local.conv2to4na[233], "TGGC", 4);
  memcpy (gv_local.conv2to4na[234], "TGGG", 4);  memcpy (gv_local.conv2to4na[235], "TGGT", 4);
  memcpy (gv_local.conv2to4na[236], "TGTA", 4);  memcpy (gv_local.conv2to4na[237], "TGTC", 4);
  memcpy (gv_local.conv2to4na[238], "TGTG", 4);  memcpy (gv_local.conv2to4na[239], "TGTT", 4);
  memcpy (gv_local.conv2to4na[240], "TTAA", 4);  memcpy (gv_local.conv2to4na[241], "TTAC", 4);
  memcpy (gv_local.conv2to4na[242], "TTAG", 4);  memcpy (gv_local.conv2to4na[243], "TTAT", 4);
  memcpy (gv_local.conv2to4na[244], "TTCA", 4);  memcpy (gv_local.conv2to4na[245], "TTCC", 4);
  memcpy (gv_local.conv2to4na[246], "TTCG", 4);  memcpy (gv_local.conv2to4na[247], "TTCT", 4);
  memcpy (gv_local.conv2to4na[248], "TTGA", 4);  memcpy (gv_local.conv2to4na[249], "TTGC", 4);
  memcpy (gv_local.conv2to4na[250], "TTGG", 4);  memcpy (gv_local.conv2to4na[251], "TTGT", 4);
  memcpy (gv_local.conv2to4na[252], "TTTA", 4);  memcpy (gv_local.conv2to4na[253], "TTTC", 4);
  memcpy (gv_local.conv2to4na[254], "TTTG", 4);  memcpy (gv_local.conv2to4na[255], "TTTT", 4);

  memcpy (gv_local.conv2to4cs[  0], "0000", 4);  memcpy (gv_local.conv2to4cs[  1], "0001", 4);
  memcpy (gv_local.conv2to4cs[  2], "0002", 4);  memcpy (gv_local.conv2to4cs[  3], "0003", 4);
  memcpy (gv_local.conv2to4cs[  4], "0010", 4);  memcpy (gv_local.conv2to4cs[  5], "0011", 4);
  memcpy (gv_local.conv2to4cs[  6], "0012", 4);  memcpy (gv_local.conv2to4cs[  7], "0013", 4);
  memcpy (gv_local.conv2to4cs[  8], "0020", 4);  memcpy (gv_local.conv2to4cs[  9], "0021", 4);
  memcpy (gv_local.conv2to4cs[ 10], "0022", 4);  memcpy (gv_local.conv2to4cs[ 11], "0023", 4);
  memcpy (gv_local.conv2to4cs[ 12], "0030", 4);  memcpy (gv_local.conv2to4cs[ 13], "0031", 4);
  memcpy (gv_local.conv2to4cs[ 14], "0032", 4);  memcpy (gv_local.conv2to4cs[ 15], "0033", 4);
  memcpy (gv_local.conv2to4cs[ 16], "0100", 4);  memcpy (gv_local.conv2to4cs[ 17], "0101", 4);
  memcpy (gv_local.conv2to4cs[ 18], "0102", 4);  memcpy (gv_local.conv2to4cs[ 19], "0103", 4);
  memcpy (gv_local.conv2to4cs[ 20], "0110", 4);  memcpy (gv_local.conv2to4cs[ 21], "0111", 4);
  memcpy (gv_local.conv2to4cs[ 22], "0112", 4);  memcpy (gv_local.conv2to4cs[ 23], "0113", 4);
  memcpy (gv_local.conv2to4cs[ 24], "0120", 4);  memcpy (gv_local.conv2to4cs[ 25], "0121", 4);
  memcpy (gv_local.conv2to4cs[ 26], "0122", 4);  memcpy (gv_local.conv2to4cs[ 27], "0123", 4);
  memcpy (gv_local.conv2to4cs[ 28], "0130", 4);  memcpy (gv_local.conv2to4cs[ 29], "0131", 4);
  memcpy (gv_local.conv2to4cs[ 30], "0132", 4);  memcpy (gv_local.conv2to4cs[ 31], "0133", 4);
  memcpy (gv_local.conv2to4cs[ 32], "0200", 4);  memcpy (gv_local.conv2to4cs[ 33], "0201", 4);
  memcpy (gv_local.conv2to4cs[ 34], "0202", 4);  memcpy (gv_local.conv2to4cs[ 35], "0203", 4);
  memcpy (gv_local.conv2to4cs[ 36], "0210", 4);  memcpy (gv_local.conv2to4cs[ 37], "0211", 4);
  memcpy (gv_local.conv2to4cs[ 38], "0212", 4);  memcpy (gv_local.conv2to4cs[ 39], "0213", 4);
  memcpy (gv_local.conv2to4cs[ 40], "0220", 4);  memcpy (gv_local.conv2to4cs[ 41], "0221", 4);
  memcpy (gv_local.conv2to4cs[ 42], "0222", 4);  memcpy (gv_local.conv2to4cs[ 43], "0223", 4);
  memcpy (gv_local.conv2to4cs[ 44], "0230", 4);  memcpy (gv_local.conv2to4cs[ 45], "0231", 4);
  memcpy (gv_local.conv2to4cs[ 46], "0232", 4);  memcpy (gv_local.conv2to4cs[ 47], "0233", 4);
  memcpy (gv_local.conv2to4cs[ 48], "0300", 4);  memcpy (gv_local.conv2to4cs[ 49], "0301", 4);
  memcpy (gv_local.conv2to4cs[ 50], "0302", 4);  memcpy (gv_local.conv2to4cs[ 51], "0303", 4);
  memcpy (gv_local.conv2to4cs[ 52], "0310", 4);  memcpy (gv_local.conv2to4cs[ 53], "0311", 4);
  memcpy (gv_local.conv2to4cs[ 54], "0312", 4);  memcpy (gv_local.conv2to4cs[ 55], "0313", 4);
  memcpy (gv_local.conv2to4cs[ 56], "0320", 4);  memcpy (gv_local.conv2to4cs[ 57], "0321", 4);
  memcpy (gv_local.conv2to4cs[ 58], "0322", 4);  memcpy (gv_local.conv2to4cs[ 59], "0323", 4);
  memcpy (gv_local.conv2to4cs[ 60], "0330", 4);  memcpy (gv_local.conv2to4cs[ 61], "0331", 4);
  memcpy (gv_local.conv2to4cs[ 62], "0332", 4);  memcpy (gv_local.conv2to4cs[ 63], "0333", 4);
  memcpy (gv_local.conv2to4cs[ 64], "1000", 4);  memcpy (gv_local.conv2to4cs[ 65], "1001", 4);
  memcpy (gv_local.conv2to4cs[ 66], "1002", 4);  memcpy (gv_local.conv2to4cs[ 67], "1003", 4);
  memcpy (gv_local.conv2to4cs[ 68], "1010", 4);  memcpy (gv_local.conv2to4cs[ 69], "1011", 4);
  memcpy (gv_local.conv2to4cs[ 70], "1012", 4);  memcpy (gv_local.conv2to4cs[ 71], "1013", 4);
  memcpy (gv_local.conv2to4cs[ 72], "1020", 4);  memcpy (gv_local.conv2to4cs[ 73], "1021", 4);
  memcpy (gv_local.conv2to4cs[ 74], "1022", 4);  memcpy (gv_local.conv2to4cs[ 75], "1023", 4);
  memcpy (gv_local.conv2to4cs[ 76], "1030", 4);  memcpy (gv_local.conv2to4cs[ 77], "1031", 4);
  memcpy (gv_local.conv2to4cs[ 78], "1032", 4);  memcpy (gv_local.conv2to4cs[ 79], "1033", 4);
  memcpy (gv_local.conv2to4cs[ 80], "1100", 4);  memcpy (gv_local.conv2to4cs[ 81], "1101", 4);
  memcpy (gv_local.conv2to4cs[ 82], "1102", 4);  memcpy (gv_local.conv2to4cs[ 83], "1103", 4);
  memcpy (gv_local.conv2to4cs[ 84], "1110", 4);  memcpy (gv_local.conv2to4cs[ 85], "1111", 4);
  memcpy (gv_local.conv2to4cs[ 86], "1112", 4);  memcpy (gv_local.conv2to4cs[ 87], "1113", 4);
  memcpy (gv_local.conv2to4cs[ 88], "1120", 4);  memcpy (gv_local.conv2to4cs[ 89], "1121", 4);
  memcpy (gv_local.conv2to4cs[ 90], "1122", 4);  memcpy (gv_local.conv2to4cs[ 91], "1123", 4);
  memcpy (gv_local.conv2to4cs[ 92], "1130", 4);  memcpy (gv_local.conv2to4cs[ 93], "1131", 4);
  memcpy (gv_local.conv2to4cs[ 94], "1132", 4);  memcpy (gv_local.conv2to4cs[ 95], "1133", 4);
  memcpy (gv_local.conv2to4cs[ 96], "1200", 4);  memcpy (gv_local.conv2to4cs[ 97], "1201", 4);
  memcpy (gv_local.conv2to4cs[ 98], "1202", 4);  memcpy (gv_local.conv2to4cs[ 99], "1203", 4);
  memcpy (gv_local.conv2to4cs[100], "1210", 4);  memcpy (gv_local.conv2to4cs[101], "1211", 4);
  memcpy (gv_local.conv2to4cs[102], "1212", 4);  memcpy (gv_local.conv2to4cs[103], "1213", 4);
  memcpy (gv_local.conv2to4cs[104], "1220", 4);  memcpy (gv_local.conv2to4cs[105], "1221", 4);
  memcpy (gv_local.conv2to4cs[106], "1222", 4);  memcpy (gv_local.conv2to4cs[107], "1223", 4);
  memcpy (gv_local.conv2to4cs[108], "1230", 4);  memcpy (gv_local.conv2to4cs[109], "1231", 4);
  memcpy (gv_local.conv2to4cs[110], "1232", 4);  memcpy (gv_local.conv2to4cs[111], "1233", 4);
  memcpy (gv_local.conv2to4cs[112], "1300", 4);  memcpy (gv_local.conv2to4cs[113], "1301", 4);
  memcpy (gv_local.conv2to4cs[114], "1302", 4);  memcpy (gv_local.conv2to4cs[115], "1303", 4);
  memcpy (gv_local.conv2to4cs[116], "1310", 4);  memcpy (gv_local.conv2to4cs[117], "1311", 4);
  memcpy (gv_local.conv2to4cs[118], "1312", 4);  memcpy (gv_local.conv2to4cs[119], "1313", 4);
  memcpy (gv_local.conv2to4cs[120], "1320", 4);  memcpy (gv_local.conv2to4cs[121], "1321", 4);
  memcpy (gv_local.conv2to4cs[122], "1322", 4);  memcpy (gv_local.conv2to4cs[123], "1323", 4);
  memcpy (gv_local.conv2to4cs[124], "1330", 4);  memcpy (gv_local.conv2to4cs[125], "1331", 4);
  memcpy (gv_local.conv2to4cs[126], "1332", 4);  memcpy (gv_local.conv2to4cs[127], "1333", 4);
  memcpy (gv_local.conv2to4cs[128], "2000", 4);  memcpy (gv_local.conv2to4cs[129], "2001", 4);
  memcpy (gv_local.conv2to4cs[130], "2002", 4);  memcpy (gv_local.conv2to4cs[131], "2003", 4);
  memcpy (gv_local.conv2to4cs[132], "2010", 4);  memcpy (gv_local.conv2to4cs[133], "2011", 4);
  memcpy (gv_local.conv2to4cs[134], "2012", 4);  memcpy (gv_local.conv2to4cs[135], "2013", 4);
  memcpy (gv_local.conv2to4cs[136], "2020", 4);  memcpy (gv_local.conv2to4cs[137], "2021", 4);
  memcpy (gv_local.conv2to4cs[138], "2022", 4);  memcpy (gv_local.conv2to4cs[139], "2023", 4);
  memcpy (gv_local.conv2to4cs[140], "2030", 4);  memcpy (gv_local.conv2to4cs[141], "2031", 4);
  memcpy (gv_local.conv2to4cs[142], "2032", 4);  memcpy (gv_local.conv2to4cs[143], "2033", 4);
  memcpy (gv_local.conv2to4cs[144], "2100", 4);  memcpy (gv_local.conv2to4cs[145], "2101", 4);
  memcpy (gv_local.conv2to4cs[146], "2102", 4);  memcpy (gv_local.conv2to4cs[147], "2103", 4);
  memcpy (gv_local.conv2to4cs[148], "2110", 4);  memcpy (gv_local.conv2to4cs[149], "2111", 4);
  memcpy (gv_local.conv2to4cs[150], "2112", 4);  memcpy (gv_local.conv2to4cs[151], "2113", 4);
  memcpy (gv_local.conv2to4cs[152], "2120", 4);  memcpy (gv_local.conv2to4cs[153], "2121", 4);
  memcpy (gv_local.conv2to4cs[154], "2122", 4);  memcpy (gv_local.conv2to4cs[155], "2123", 4);
  memcpy (gv_local.conv2to4cs[156], "2130", 4);  memcpy (gv_local.conv2to4cs[157], "2131", 4);
  memcpy (gv_local.conv2to4cs[158], "2132", 4);  memcpy (gv_local.conv2to4cs[159], "2133", 4);
  memcpy (gv_local.conv2to4cs[160], "2200", 4);  memcpy (gv_local.conv2to4cs[161], "2201", 4);
  memcpy (gv_local.conv2to4cs[162], "2202", 4);  memcpy (gv_local.conv2to4cs[163], "2203", 4);
  memcpy (gv_local.conv2to4cs[164], "2210", 4);  memcpy (gv_local.conv2to4cs[165], "2211", 4);
  memcpy (gv_local.conv2to4cs[166], "2212", 4);  memcpy (gv_local.conv2to4cs[167], "2213", 4);
  memcpy (gv_local.conv2to4cs[168], "2220", 4);  memcpy (gv_local.conv2to4cs[169], "2221", 4);
  memcpy (gv_local.conv2to4cs[170], "2222", 4);  memcpy (gv_local.conv2to4cs[171], "2223", 4);
  memcpy (gv_local.conv2to4cs[172], "2230", 4);  memcpy (gv_local.conv2to4cs[173], "2231", 4);
  memcpy (gv_local.conv2to4cs[174], "2232", 4);  memcpy (gv_local.conv2to4cs[175], "2233", 4);
  memcpy (gv_local.conv2to4cs[176], "2300", 4);  memcpy (gv_local.conv2to4cs[177], "2301", 4);
  memcpy (gv_local.conv2to4cs[178], "2302", 4);  memcpy (gv_local.conv2to4cs[179], "2303", 4);
  memcpy (gv_local.conv2to4cs[180], "2310", 4);  memcpy (gv_local.conv2to4cs[181], "2311", 4);
  memcpy (gv_local.conv2to4cs[182], "2312", 4);  memcpy (gv_local.conv2to4cs[183], "2313", 4);
  memcpy (gv_local.conv2to4cs[184], "2320", 4);  memcpy (gv_local.conv2to4cs[185], "2321", 4);
  memcpy (gv_local.conv2to4cs[186], "2322", 4);  memcpy (gv_local.conv2to4cs[187], "2323", 4);
  memcpy (gv_local.conv2to4cs[188], "2330", 4);  memcpy (gv_local.conv2to4cs[189], "2331", 4);
  memcpy (gv_local.conv2to4cs[190], "2332", 4);  memcpy (gv_local.conv2to4cs[191], "2333", 4);
  memcpy (gv_local.conv2to4cs[192], "3000", 4);  memcpy (gv_local.conv2to4cs[193], "3001", 4);
  memcpy (gv_local.conv2to4cs[194], "3002", 4);  memcpy (gv_local.conv2to4cs[195], "3003", 4);
  memcpy (gv_local.conv2to4cs[196], "3010", 4);  memcpy (gv_local.conv2to4cs[197], "3011", 4);
  memcpy (gv_local.conv2to4cs[198], "3012", 4);  memcpy (gv_local.conv2to4cs[199], "3013", 4);
  memcpy (gv_local.conv2to4cs[200], "3020", 4);  memcpy (gv_local.conv2to4cs[201], "3021", 4);
  memcpy (gv_local.conv2to4cs[202], "3022", 4);  memcpy (gv_local.conv2to4cs[203], "3023", 4);
  memcpy (gv_local.conv2to4cs[204], "3030", 4);  memcpy (gv_local.conv2to4cs[205], "3031", 4);
  memcpy (gv_local.conv2to4cs[206], "3032", 4);  memcpy (gv_local.conv2to4cs[207], "3033", 4);
  memcpy (gv_local.conv2to4cs[208], "3100", 4);  memcpy (gv_local.conv2to4cs[209], "3101", 4);
  memcpy (gv_local.conv2to4cs[210], "3102", 4);  memcpy (gv_local.conv2to4cs[211], "3103", 4);
  memcpy (gv_local.conv2to4cs[212], "3110", 4);  memcpy (gv_local.conv2to4cs[213], "3111", 4);
  memcpy (gv_local.conv2to4cs[214], "3112", 4);  memcpy (gv_local.conv2to4cs[215], "3113", 4);
  memcpy (gv_local.conv2to4cs[216], "3120", 4);  memcpy (gv_local.conv2to4cs[217], "3121", 4);
  memcpy (gv_local.conv2to4cs[218], "3122", 4);  memcpy (gv_local.conv2to4cs[219], "3123", 4);
  memcpy (gv_local.conv2to4cs[220], "3130", 4);  memcpy (gv_local.conv2to4cs[221], "3131", 4);
  memcpy (gv_local.conv2to4cs[222], "3132", 4);  memcpy (gv_local.conv2to4cs[223], "3133", 4);
  memcpy (gv_local.conv2to4cs[224], "3200", 4);  memcpy (gv_local.conv2to4cs[225], "3201", 4);
  memcpy (gv_local.conv2to4cs[226], "3202", 4);  memcpy (gv_local.conv2to4cs[227], "3203", 4);
  memcpy (gv_local.conv2to4cs[228], "3210", 4);  memcpy (gv_local.conv2to4cs[229], "3211", 4);
  memcpy (gv_local.conv2to4cs[230], "3212", 4);  memcpy (gv_local.conv2to4cs[231], "3213", 4);
  memcpy (gv_local.conv2to4cs[232], "3220", 4);  memcpy (gv_local.conv2to4cs[233], "3221", 4);
  memcpy (gv_local.conv2to4cs[234], "3222", 4);  memcpy (gv_local.conv2to4cs[235], "3223", 4);
  memcpy (gv_local.conv2to4cs[236], "3230", 4);  memcpy (gv_local.conv2to4cs[237], "3231", 4);
  memcpy (gv_local.conv2to4cs[238], "3232", 4);  memcpy (gv_local.conv2to4cs[239], "3233", 4);
  memcpy (gv_local.conv2to4cs[240], "3300", 4);  memcpy (gv_local.conv2to4cs[241], "3301", 4);
  memcpy (gv_local.conv2to4cs[242], "3302", 4);  memcpy (gv_local.conv2to4cs[243], "3303", 4);
  memcpy (gv_local.conv2to4cs[244], "3310", 4);  memcpy (gv_local.conv2to4cs[245], "3311", 4);
  memcpy (gv_local.conv2to4cs[246], "3312", 4);  memcpy (gv_local.conv2to4cs[247], "3313", 4);
  memcpy (gv_local.conv2to4cs[248], "3320", 4);  memcpy (gv_local.conv2to4cs[249], "3321", 4);
  memcpy (gv_local.conv2to4cs[250], "3322", 4);  memcpy (gv_local.conv2to4cs[251], "3323", 4);
  memcpy (gv_local.conv2to4cs[252], "3330", 4);  memcpy (gv_local.conv2to4cs[253], "3331", 4);
  memcpy (gv_local.conv2to4cs[254], "3332", 4);  memcpy (gv_local.conv2to4cs[255], "3333", 4);
  gv_local.lookup_454_seq_initialized = 1;
}
/*
 */
static void _s_init_pos_454_default (void)
{
  memset (gv_local.lookup_454_pos, 0, sizeof(SRALookup)*POS_454_HUFF_TREESIZE);

  gv_local.lookup_454_pos[  0].nbits =  2; gv_local.lookup_454_pos[  0].bits = 0xC0000000;
  gv_local.lookup_454_pos[  1].nbits =  3; gv_local.lookup_454_pos[  1].bits = 0x20000000;
  gv_local.lookup_454_pos[  2].nbits =  2; gv_local.lookup_454_pos[  2].bits = 0x40000000;
  gv_local.lookup_454_pos[  3].nbits =  2; gv_local.lookup_454_pos[  3].bits = 0x80000000;
  gv_local.lookup_454_pos[  4].nbits =  4; gv_local.lookup_454_pos[  4].bits = 0x10000000;
  gv_local.lookup_454_pos[  5].nbits =  6; gv_local.lookup_454_pos[  5].bits = 0x04000000;
  gv_local.lookup_454_pos[  6].nbits = 11; gv_local.lookup_454_pos[  6].bits = 0x0E400000;
  gv_local.lookup_454_pos[  7].nbits =  8; gv_local.lookup_454_pos[  7].bits = 0x0F000000;
  gv_local.lookup_454_pos[  8].nbits =  7; gv_local.lookup_454_pos[  8].bits = 0x0C000000;
  gv_local.lookup_454_pos[  9].nbits =  9; gv_local.lookup_454_pos[  9].bits = 0x08800000;
  gv_local.lookup_454_pos[ 10].nbits = 12; gv_local.lookup_454_pos[ 10].bits = 0x0A000000;
  gv_local.lookup_454_pos[ 11].nbits =  6; gv_local.lookup_454_pos[ 11].bits = 0x00000000;
  gv_local.lookup_454_pos[ 12].nbits =  8; gv_local.lookup_454_pos[ 12].bits = 0x09000000;
  gv_local.lookup_454_pos[ 13].nbits = 11; gv_local.lookup_454_pos[ 13].bits = 0x0AC00000;
  gv_local.lookup_454_pos[ 14].nbits = 14; gv_local.lookup_454_pos[ 14].bits = 0x0E240000;
  gv_local.lookup_454_pos[ 15].nbits =  9; gv_local.lookup_454_pos[ 15].bits = 0x0E800000;
  gv_local.lookup_454_pos[ 16].nbits =  9; gv_local.lookup_454_pos[ 16].bits = 0x0B000000;
  gv_local.lookup_454_pos[ 17].nbits = 12; gv_local.lookup_454_pos[ 17].bits = 0x0A300000;
  gv_local.lookup_454_pos[ 18].nbits = 13; gv_local.lookup_454_pos[ 18].bits = 0x08580000;
  gv_local.lookup_454_pos[ 19].nbits = 12; gv_local.lookup_454_pos[ 19].bits = 0x0E600000;
  gv_local.lookup_454_pos[ 20].nbits = 10; gv_local.lookup_454_pos[ 20].bits = 0x0B800000;
  gv_local.lookup_454_pos[ 21].nbits = 13; gv_local.lookup_454_pos[ 21].bits = 0x08400000;
  gv_local.lookup_454_pos[ 22].nbits = 14; gv_local.lookup_454_pos[ 22].bits = 0x08540000;
  gv_local.lookup_454_pos[ 23].nbits = 13; gv_local.lookup_454_pos[ 23].bits = 0x0E280000;
  gv_local.lookup_454_pos[ 24].nbits = 10; gv_local.lookup_454_pos[ 24].bits = 0x0A400000;
  gv_local.lookup_454_pos[ 25].nbits = 13; gv_local.lookup_454_pos[ 25].bits = 0x0E000000;
  gv_local.lookup_454_pos[ 26].nbits = 14; gv_local.lookup_454_pos[ 26].bits = 0x0E180000;
  gv_local.lookup_454_pos[ 27].nbits = 13; gv_local.lookup_454_pos[ 27].bits = 0x0E380000;
  gv_local.lookup_454_pos[ 28].nbits = 10; gv_local.lookup_454_pos[ 28].bits = 0x08000000;
  gv_local.lookup_454_pos[ 29].nbits = 14; gv_local.lookup_454_pos[ 29].bits = 0x0E0C0000;
  gv_local.lookup_454_pos[ 30].nbits = 15; gv_local.lookup_454_pos[ 30].bits = 0x0AB40000;
  gv_local.lookup_454_pos[ 31].nbits = 13; gv_local.lookup_454_pos[ 31].bits = 0x0AA00000;
  gv_local.lookup_454_pos[ 32].nbits = 11; gv_local.lookup_454_pos[ 32].bits = 0x0A800000;
  gv_local.lookup_454_pos[ 33].nbits = 14; gv_local.lookup_454_pos[ 33].bits = 0x084C0000;
  gv_local.lookup_454_pos[ 34].nbits = 16; gv_local.lookup_454_pos[ 34].bits = 0x0AB60000;
  gv_local.lookup_454_pos[ 35].nbits = 14; gv_local.lookup_454_pos[ 35].bits = 0x0E200000;
  gv_local.lookup_454_pos[ 36].nbits = 12; gv_local.lookup_454_pos[ 36].bits = 0x0E700000;
  gv_local.lookup_454_pos[ 37].nbits = 14; gv_local.lookup_454_pos[ 37].bits = 0x0ABC0000;
  gv_local.lookup_454_pos[ 38].nbits = 15; gv_local.lookup_454_pos[ 38].bits = 0x0E080000;
  gv_local.lookup_454_pos[ 39].nbits = 13; gv_local.lookup_454_pos[ 39].bits = 0x0AA80000;
  gv_local.lookup_454_pos[ 40].nbits = 11; gv_local.lookup_454_pos[ 40].bits = 0x08600000;
  gv_local.lookup_454_pos[ 41].nbits = 14; gv_local.lookup_454_pos[ 41].bits = 0x0AB00000;
  gv_local.lookup_454_pos[ 42].nbits = 16; gv_local.lookup_454_pos[ 42].bits = 0x0E1E0000;
  gv_local.lookup_454_pos[ 43].nbits = 14; gv_local.lookup_454_pos[ 43].bits = 0x08500000;
  gv_local.lookup_454_pos[ 44].nbits = 12; gv_local.lookup_454_pos[ 44].bits = 0x0AF00000;
  gv_local.lookup_454_pos[ 45].nbits = 14; gv_local.lookup_454_pos[ 45].bits = 0x0A1C0000;
  gv_local.lookup_454_pos[ 46].nbits = 14; gv_local.lookup_454_pos[ 46].bits = 0x08480000;
  gv_local.lookup_454_pos[ 47].nbits = 13; gv_local.lookup_454_pos[ 47].bits = 0x0E300000;
  gv_local.lookup_454_pos[ 48].nbits = 12; gv_local.lookup_454_pos[ 48].bits = 0x0AE00000;
  gv_local.lookup_454_pos[ 49].nbits = 14; gv_local.lookup_454_pos[ 49].bits = 0x0AB80000;
  gv_local.lookup_454_pos[ 50].nbits = 16; gv_local.lookup_454_pos[ 50].bits = 0x0A1B0000;
  gv_local.lookup_454_pos[ 51].nbits = 15; gv_local.lookup_454_pos[ 51].bits = 0x0A180000;
  gv_local.lookup_454_pos[ 52].nbits = 12; gv_local.lookup_454_pos[ 52].bits = 0x0A200000;
  gv_local.lookup_454_pos[ 53].nbits = 15; gv_local.lookup_454_pos[ 53].bits = 0x0E1C0000;
  gv_local.lookup_454_pos[ 54].nbits = 16; gv_local.lookup_454_pos[ 54].bits = 0x0A1A0000;
  gv_local.lookup_454_pos[ 55].nbits = 16; gv_local.lookup_454_pos[ 55].bits = 0x0E1F0000;
  gv_local.lookup_454_pos[ 56].nbits = 13; gv_local.lookup_454_pos[ 56].bits = 0x0E100000;
  gv_local.lookup_454_pos[ 57].nbits = 17; gv_local.lookup_454_pos[ 57].bits = 0x0AB78000;
  gv_local.lookup_454_pos[ 58].nbits = 18; gv_local.lookup_454_pos[ 58].bits = 0x0AB70000;
  gv_local.lookup_454_pos[ 59].nbits = 15; gv_local.lookup_454_pos[ 59].bits = 0x0E0A0000;
  gv_local.lookup_454_pos[ 60].nbits = 13; gv_local.lookup_454_pos[ 60].bits = 0x0A100000;
  gv_local.lookup_454_pos[ 61].nbits = 19; gv_local.lookup_454_pos[ 61].bits = 0x0AB76000;
  gv_local.lookup_454_pos[ 62].nbits = 19; gv_local.lookup_454_pos[ 62].bits = 0x0AB74000;
  gv_local.lookup_454_pos[ 63].nbits = 10; gv_local.lookup_454_pos[ 63].bits = 0x0BC00000;

  _s_sort_lookup (gv_local.lookup_454_pos, gv_local.idx_454_pos, POS_454_HUFF_TREESIZE);

  gv_local.lookup_454_pos_initialized = 1;
}
/*
 */
static void _s_init_prb_454_default (void)
{
#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  int      i, j;
  uint16_t val;
#endif

  memset (gv_local.lookup_454_prb, 0, sizeof(SRALookup)*PRB_454_HUFF_TREESIZE);
  
  gv_local.lookup_454_prb[  0].nbits = 10; gv_local.lookup_454_prb[  0].bits = 0xA0400000;
  gv_local.lookup_454_prb[  1].nbits =  9; gv_local.lookup_454_prb[  1].bits = 0xA0800000;
  gv_local.lookup_454_prb[  2].nbits =  9; gv_local.lookup_454_prb[  2].bits = 0xA8800000;
  gv_local.lookup_454_prb[  3].nbits =  9; gv_local.lookup_454_prb[  3].bits = 0xA8000000;
  gv_local.lookup_454_prb[  4].nbits =  9; gv_local.lookup_454_prb[  4].bits = 0xDA000000;
  gv_local.lookup_454_prb[  5].nbits =  9; gv_local.lookup_454_prb[  5].bits = 0xDA800000;
  gv_local.lookup_454_prb[  6].nbits =  8; gv_local.lookup_454_prb[  6].bits = 0xA4000000;
  gv_local.lookup_454_prb[  7].nbits =  8; gv_local.lookup_454_prb[  7].bits = 0xBA000000;
  gv_local.lookup_454_prb[  8].nbits =  8; gv_local.lookup_454_prb[  8].bits = 0xA5000000;
  gv_local.lookup_454_prb[  9].nbits =  8; gv_local.lookup_454_prb[  9].bits = 0xD4000000;
  gv_local.lookup_454_prb[ 10].nbits =  8; gv_local.lookup_454_prb[ 10].bits = 0xA9000000;
  gv_local.lookup_454_prb[ 11].nbits =  8; gv_local.lookup_454_prb[ 11].bits = 0xBB000000;
  gv_local.lookup_454_prb[ 12].nbits =  8; gv_local.lookup_454_prb[ 12].bits = 0xDB000000;
  gv_local.lookup_454_prb[ 13].nbits =  8; gv_local.lookup_454_prb[ 13].bits = 0xD5000000;
  gv_local.lookup_454_prb[ 14].nbits =  7; gv_local.lookup_454_prb[ 14].bits = 0x2A000000;
  gv_local.lookup_454_prb[ 15].nbits =  7; gv_local.lookup_454_prb[ 15].bits = 0xD0000000;
  gv_local.lookup_454_prb[ 16].nbits =  7; gv_local.lookup_454_prb[ 16].bits = 0xD6000000;
  gv_local.lookup_454_prb[ 17].nbits =  6; gv_local.lookup_454_prb[ 17].bits = 0xAC000000;
  gv_local.lookup_454_prb[ 18].nbits =  7; gv_local.lookup_454_prb[ 18].bits = 0xAA000000;
  gv_local.lookup_454_prb[ 19].nbits =  7; gv_local.lookup_454_prb[ 19].bits = 0xA2000000;
  gv_local.lookup_454_prb[ 20].nbits =  6; gv_local.lookup_454_prb[ 20].bits = 0x80000000;
  gv_local.lookup_454_prb[ 21].nbits =  6; gv_local.lookup_454_prb[ 21].bits = 0x84000000;
  gv_local.lookup_454_prb[ 22].nbits =  6; gv_local.lookup_454_prb[ 22].bits = 0xBC000000;
  gv_local.lookup_454_prb[ 23].nbits =  5; gv_local.lookup_454_prb[ 23].bits = 0x88000000;
  gv_local.lookup_454_prb[ 24].nbits =  5; gv_local.lookup_454_prb[ 24].bits = 0xB0000000;
  gv_local.lookup_454_prb[ 25].nbits =  4; gv_local.lookup_454_prb[ 25].bits = 0x90000000;
  gv_local.lookup_454_prb[ 26].nbits =  3; gv_local.lookup_454_prb[ 26].bits = 0x00000000;
  gv_local.lookup_454_prb[ 27].nbits =  2; gv_local.lookup_454_prb[ 27].bits = 0x40000000;
  gv_local.lookup_454_prb[ 28].nbits =  3; gv_local.lookup_454_prb[ 28].bits = 0xE0000000;
  gv_local.lookup_454_prb[ 29].nbits =  8; gv_local.lookup_454_prb[ 29].bits = 0xA1000000;
  gv_local.lookup_454_prb[ 30].nbits =  7; gv_local.lookup_454_prb[ 30].bits = 0xD2000000;
  gv_local.lookup_454_prb[ 31].nbits =  6; gv_local.lookup_454_prb[ 31].bits = 0x2C000000;
  gv_local.lookup_454_prb[ 32].nbits =  5; gv_local.lookup_454_prb[ 32].bits = 0x20000000;
  gv_local.lookup_454_prb[ 33].nbits =  4; gv_local.lookup_454_prb[ 33].bits = 0x30000000;
  gv_local.lookup_454_prb[ 34].nbits =  4; gv_local.lookup_454_prb[ 34].bits = 0xC0000000;
  gv_local.lookup_454_prb[ 35].nbits =  7; gv_local.lookup_454_prb[ 35].bits = 0xD8000000;
  gv_local.lookup_454_prb[ 36].nbits =  7; gv_local.lookup_454_prb[ 36].bits = 0xB8000000;
  gv_local.lookup_454_prb[ 37].nbits =  6; gv_local.lookup_454_prb[ 37].bits = 0xDC000000;
  gv_local.lookup_454_prb[ 38].nbits =  7; gv_local.lookup_454_prb[ 38].bits = 0x28000000;
  gv_local.lookup_454_prb[ 39].nbits =  7; gv_local.lookup_454_prb[ 39].bits = 0xA6000000;
  gv_local.lookup_454_prb[ 40].nbits = 15; gv_local.lookup_454_prb[ 40].bits = 0xA0000000;
  gv_local.lookup_454_prb[ 41].nbits = 14; gv_local.lookup_454_prb[ 41].bits = 0xA03C0000;
  gv_local.lookup_454_prb[ 42].nbits = 16; gv_local.lookup_454_prb[ 42].bits = 0xA0030000;
  gv_local.lookup_454_prb[ 43].nbits = 16; gv_local.lookup_454_prb[ 43].bits = 0xA0020000;
  gv_local.lookup_454_prb[ 44].nbits = 16; gv_local.lookup_454_prb[ 44].bits = 0xA0390000;
  gv_local.lookup_454_prb[ 45].nbits = 16; gv_local.lookup_454_prb[ 45].bits = 0xA0380000;
  gv_local.lookup_454_prb[ 46].nbits = 16; gv_local.lookup_454_prb[ 46].bits = 0xA03B0000;
  gv_local.lookup_454_prb[ 47].nbits = 16; gv_local.lookup_454_prb[ 47].bits = 0xA03A0000;
  gv_local.lookup_454_prb[ 48].nbits = 16; gv_local.lookup_454_prb[ 48].bits = 0xA00D0000;
  gv_local.lookup_454_prb[ 49].nbits = 16; gv_local.lookup_454_prb[ 49].bits = 0xA00C0000;
  gv_local.lookup_454_prb[ 50].nbits = 16; gv_local.lookup_454_prb[ 50].bits = 0xA00F0000;
  gv_local.lookup_454_prb[ 51].nbits = 16; gv_local.lookup_454_prb[ 51].bits = 0xA00E0000;
  gv_local.lookup_454_prb[ 52].nbits = 16; gv_local.lookup_454_prb[ 52].bits = 0xA0090000;
  gv_local.lookup_454_prb[ 53].nbits = 16; gv_local.lookup_454_prb[ 53].bits = 0xA0080000;
  gv_local.lookup_454_prb[ 54].nbits = 16; gv_local.lookup_454_prb[ 54].bits = 0xA00B0000;
  gv_local.lookup_454_prb[ 55].nbits = 16; gv_local.lookup_454_prb[ 55].bits = 0xA00A0000;
  gv_local.lookup_454_prb[ 56].nbits = 16; gv_local.lookup_454_prb[ 56].bits = 0xA0250000;
  gv_local.lookup_454_prb[ 57].nbits = 16; gv_local.lookup_454_prb[ 57].bits = 0xA0240000;
  gv_local.lookup_454_prb[ 58].nbits = 16; gv_local.lookup_454_prb[ 58].bits = 0xA0270000;
  gv_local.lookup_454_prb[ 59].nbits = 16; gv_local.lookup_454_prb[ 59].bits = 0xA0260000;
  gv_local.lookup_454_prb[ 60].nbits = 16; gv_local.lookup_454_prb[ 60].bits = 0xA0210000;
  gv_local.lookup_454_prb[ 61].nbits = 16; gv_local.lookup_454_prb[ 61].bits = 0xA0200000;
  gv_local.lookup_454_prb[ 62].nbits = 16; gv_local.lookup_454_prb[ 62].bits = 0xA0230000;
  gv_local.lookup_454_prb[ 63].nbits = 16; gv_local.lookup_454_prb[ 63].bits = 0xA0220000;
  gv_local.lookup_454_prb[ 64].nbits = 16; gv_local.lookup_454_prb[ 64].bits = 0xA02D0000;
  gv_local.lookup_454_prb[ 65].nbits = 16; gv_local.lookup_454_prb[ 65].bits = 0xA02C0000;
  gv_local.lookup_454_prb[ 66].nbits = 16; gv_local.lookup_454_prb[ 66].bits = 0xA02F0000;
  gv_local.lookup_454_prb[ 67].nbits = 16; gv_local.lookup_454_prb[ 67].bits = 0xA02E0000;
  gv_local.lookup_454_prb[ 68].nbits = 16; gv_local.lookup_454_prb[ 68].bits = 0xA0290000;
  gv_local.lookup_454_prb[ 69].nbits = 16; gv_local.lookup_454_prb[ 69].bits = 0xA0280000;
  gv_local.lookup_454_prb[ 70].nbits = 16; gv_local.lookup_454_prb[ 70].bits = 0xA02B0000;
  gv_local.lookup_454_prb[ 71].nbits = 16; gv_local.lookup_454_prb[ 71].bits = 0xA02A0000;
  gv_local.lookup_454_prb[ 72].nbits = 16; gv_local.lookup_454_prb[ 72].bits = 0xA0150000;
  gv_local.lookup_454_prb[ 73].nbits = 16; gv_local.lookup_454_prb[ 73].bits = 0xA0140000;
  gv_local.lookup_454_prb[ 74].nbits = 16; gv_local.lookup_454_prb[ 74].bits = 0xA0170000;
  gv_local.lookup_454_prb[ 75].nbits = 16; gv_local.lookup_454_prb[ 75].bits = 0xA0160000;
  gv_local.lookup_454_prb[ 76].nbits = 16; gv_local.lookup_454_prb[ 76].bits = 0xA0110000;
  gv_local.lookup_454_prb[ 77].nbits = 16; gv_local.lookup_454_prb[ 77].bits = 0xA0100000;
  gv_local.lookup_454_prb[ 78].nbits = 16; gv_local.lookup_454_prb[ 78].bits = 0xA0130000;
  gv_local.lookup_454_prb[ 79].nbits = 16; gv_local.lookup_454_prb[ 79].bits = 0xA0120000;
  gv_local.lookup_454_prb[ 80].nbits = 16; gv_local.lookup_454_prb[ 80].bits = 0xA01D0000;
  gv_local.lookup_454_prb[ 81].nbits = 16; gv_local.lookup_454_prb[ 81].bits = 0xA01C0000;
  gv_local.lookup_454_prb[ 82].nbits = 16; gv_local.lookup_454_prb[ 82].bits = 0xA01F0000;
  gv_local.lookup_454_prb[ 83].nbits = 16; gv_local.lookup_454_prb[ 83].bits = 0xA01E0000;
  gv_local.lookup_454_prb[ 84].nbits = 16; gv_local.lookup_454_prb[ 84].bits = 0xA0190000;
  gv_local.lookup_454_prb[ 85].nbits = 16; gv_local.lookup_454_prb[ 85].bits = 0xA0180000;
  gv_local.lookup_454_prb[ 86].nbits = 16; gv_local.lookup_454_prb[ 86].bits = 0xA01B0000;
  gv_local.lookup_454_prb[ 87].nbits = 16; gv_local.lookup_454_prb[ 87].bits = 0xA01A0000;
  gv_local.lookup_454_prb[ 88].nbits = 16; gv_local.lookup_454_prb[ 88].bits = 0xA0350000;
  gv_local.lookup_454_prb[ 89].nbits = 16; gv_local.lookup_454_prb[ 89].bits = 0xA0340000;
  gv_local.lookup_454_prb[ 90].nbits = 16; gv_local.lookup_454_prb[ 90].bits = 0xA0370000;
  gv_local.lookup_454_prb[ 91].nbits = 16; gv_local.lookup_454_prb[ 91].bits = 0xA0360000;
  gv_local.lookup_454_prb[ 92].nbits = 16; gv_local.lookup_454_prb[ 92].bits = 0xA0310000;
  gv_local.lookup_454_prb[ 93].nbits = 16; gv_local.lookup_454_prb[ 93].bits = 0xA0300000;
  gv_local.lookup_454_prb[ 94].nbits = 16; gv_local.lookup_454_prb[ 94].bits = 0xA0330000;
  gv_local.lookup_454_prb[ 95].nbits = 16; gv_local.lookup_454_prb[ 95].bits = 0xA0320000;
  gv_local.lookup_454_prb[ 96].nbits = 16; gv_local.lookup_454_prb[ 96].bits = 0xA0050000;
  gv_local.lookup_454_prb[ 97].nbits = 16; gv_local.lookup_454_prb[ 97].bits = 0xA0040000;
  gv_local.lookup_454_prb[ 98].nbits = 16; gv_local.lookup_454_prb[ 98].bits = 0xA0070000;
  gv_local.lookup_454_prb[ 99].nbits = 16; gv_local.lookup_454_prb[ 99].bits = 0xA0060000;

#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  /* reverse huffman lookup table
   */
  memset (gv_local.rlookup_454_prb, 0, sizeof(uint16_t)*REV_HUFF_TREESIZE_16);

  for (i=0; i<PRB_454_HUFF_TREESIZE; i++)
  {
    val = gv_local.lookup_454_prb[i].bits >> 16;
    gv_local.rlookup_454_prb [val] = (uint16_t) i;
    for (j=1; j<=(0xFFFF>>gv_local.lookup_454_prb[i].nbits); j++)
      gv_local.rlookup_454_prb [val+j] = (uint16_t) i;
  }
#endif
  _s_sort_lookup (gv_local.lookup_454_prb, gv_local.idx_454_prb, PRB_454_HUFF_TREESIZE);

  gv_local.lookup_454_prb_initialized = 1;
}
/*
 */
static void _s_init_sig_454_default (void)
{
#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  int      i, j;
  uint16_t val;
#endif

  memset (gv_local.lookup_454_sig, 0, sizeof(SRALookup)*SIG_HUFF_TREESIZE);

  gv_local.lookup_454_sig[  0].nbits =  7; gv_local.lookup_454_sig[  0].bits = 0x8C000000;
  gv_local.lookup_454_sig[  1].nbits =  8; gv_local.lookup_454_sig[  1].bits = 0x65000000;
  gv_local.lookup_454_sig[  2].nbits =  8; gv_local.lookup_454_sig[  2].bits = 0xCA000000;
  gv_local.lookup_454_sig[  3].nbits =  7; gv_local.lookup_454_sig[  3].bits = 0x5A000000;
  gv_local.lookup_454_sig[  4].nbits =  7; gv_local.lookup_454_sig[  4].bits = 0xBC000000;
  gv_local.lookup_454_sig[  5].nbits =  6; gv_local.lookup_454_sig[  5].bits = 0x48000000;
  gv_local.lookup_454_sig[  6].nbits =  6; gv_local.lookup_454_sig[  6].bits = 0x98000000;
  gv_local.lookup_454_sig[  7].nbits =  6; gv_local.lookup_454_sig[  7].bits = 0xFC000000;
  gv_local.lookup_454_sig[  8].nbits =  5; gv_local.lookup_454_sig[  8].bits = 0x68000000;
  gv_local.lookup_454_sig[  9].nbits =  5; gv_local.lookup_454_sig[  9].bits = 0xA8000000;
  gv_local.lookup_454_sig[ 10].nbits =  5; gv_local.lookup_454_sig[ 10].bits = 0xE0000000;
  gv_local.lookup_454_sig[ 11].nbits =  4; gv_local.lookup_454_sig[ 11].bits = 0x00000000;
  gv_local.lookup_454_sig[ 12].nbits =  4; gv_local.lookup_454_sig[ 12].bits = 0x10000000;
  gv_local.lookup_454_sig[ 13].nbits =  5; gv_local.lookup_454_sig[ 13].bits = 0xE8000000;
  gv_local.lookup_454_sig[ 14].nbits =  5; gv_local.lookup_454_sig[ 14].bits = 0xC0000000;
  gv_local.lookup_454_sig[ 15].nbits =  5; gv_local.lookup_454_sig[ 15].bits = 0x78000000;
  gv_local.lookup_454_sig[ 16].nbits =  5; gv_local.lookup_454_sig[ 16].bits = 0x38000000;
  gv_local.lookup_454_sig[ 17].nbits =  6; gv_local.lookup_454_sig[ 17].bits = 0xCC000000;
  gv_local.lookup_454_sig[ 18].nbits =  6; gv_local.lookup_454_sig[ 18].bits = 0x80000000;
  gv_local.lookup_454_sig[ 19].nbits =  6; gv_local.lookup_454_sig[ 19].bits = 0x4C000000;
  gv_local.lookup_454_sig[ 20].nbits =  7; gv_local.lookup_454_sig[ 20].bits = 0xDE000000;
  gv_local.lookup_454_sig[ 21].nbits =  7; gv_local.lookup_454_sig[ 21].bits = 0xB0000000;
  gv_local.lookup_454_sig[ 22].nbits =  7; gv_local.lookup_454_sig[ 22].bits = 0x76000000;
  gv_local.lookup_454_sig[ 23].nbits =  7; gv_local.lookup_454_sig[ 23].bits = 0x52000000;
  gv_local.lookup_454_sig[ 24].nbits =  8; gv_local.lookup_454_sig[ 24].bits = 0xF4000000;
  gv_local.lookup_454_sig[ 25].nbits =  8; gv_local.lookup_454_sig[ 25].bits = 0xBB000000;
  gv_local.lookup_454_sig[ 26].nbits =  8; gv_local.lookup_454_sig[ 26].bits = 0x8F000000;
  gv_local.lookup_454_sig[ 27].nbits =  8; gv_local.lookup_454_sig[ 27].bits = 0x64000000;
  gv_local.lookup_454_sig[ 28].nbits =  8; gv_local.lookup_454_sig[ 28].bits = 0x31000000;
  gv_local.lookup_454_sig[ 29].nbits =  9; gv_local.lookup_454_sig[ 29].bits = 0xDD000000;
  gv_local.lookup_454_sig[ 30].nbits =  9; gv_local.lookup_454_sig[ 30].bits = 0xC8000000;
  gv_local.lookup_454_sig[ 31].nbits =  9; gv_local.lookup_454_sig[ 31].bits = 0x95000000;
  gv_local.lookup_454_sig[ 32].nbits =  9; gv_local.lookup_454_sig[ 32].bits = 0x74800000;
  gv_local.lookup_454_sig[ 33].nbits =  9; gv_local.lookup_454_sig[ 33].bits = 0x57800000;
  gv_local.lookup_454_sig[ 34].nbits =  9; gv_local.lookup_454_sig[ 34].bits = 0x30000000;
  gv_local.lookup_454_sig[ 35].nbits = 10; gv_local.lookup_454_sig[ 35].bits = 0xF0C00000;
  gv_local.lookup_454_sig[ 36].nbits = 10; gv_local.lookup_454_sig[ 36].bits = 0xCB000000;
  gv_local.lookup_454_sig[ 37].nbits = 10; gv_local.lookup_454_sig[ 37].bits = 0xA5400000;
  gv_local.lookup_454_sig[ 38].nbits = 10; gv_local.lookup_454_sig[ 38].bits = 0x8E000000;
  gv_local.lookup_454_sig[ 39].nbits = 10; gv_local.lookup_454_sig[ 39].bits = 0x70C00000;
  gv_local.lookup_454_sig[ 40].nbits = 10; gv_local.lookup_454_sig[ 40].bits = 0x56400000;
  gv_local.lookup_454_sig[ 41].nbits = 10; gv_local.lookup_454_sig[ 41].bits = 0x30C00000;
  gv_local.lookup_454_sig[ 42].nbits = 11; gv_local.lookup_454_sig[ 42].bits = 0xF5400000;
  gv_local.lookup_454_sig[ 43].nbits = 11; gv_local.lookup_454_sig[ 43].bits = 0xD0600000;
  gv_local.lookup_454_sig[ 44].nbits = 11; gv_local.lookup_454_sig[ 44].bits = 0xB3800000;
  gv_local.lookup_454_sig[ 45].nbits = 11; gv_local.lookup_454_sig[ 45].bits = 0x97400000;
  gv_local.lookup_454_sig[ 46].nbits = 11; gv_local.lookup_454_sig[ 46].bits = 0x86200000;
  gv_local.lookup_454_sig[ 47].nbits = 11; gv_local.lookup_454_sig[ 47].bits = 0x70600000;
  gv_local.lookup_454_sig[ 48].nbits = 11; gv_local.lookup_454_sig[ 48].bits = 0x66400000;
  gv_local.lookup_454_sig[ 49].nbits = 11; gv_local.lookup_454_sig[ 49].bits = 0x5DE00000;
  gv_local.lookup_454_sig[ 50].nbits = 12; gv_local.lookup_454_sig[ 50].bits = 0xF0000000;
  gv_local.lookup_454_sig[ 51].nbits = 12; gv_local.lookup_454_sig[ 51].bits = 0x75000000;
  gv_local.lookup_454_sig[ 52].nbits = 12; gv_local.lookup_454_sig[ 52].bits = 0x74500000;
  gv_local.lookup_454_sig[ 53].nbits = 12; gv_local.lookup_454_sig[ 53].bits = 0x74200000;
  gv_local.lookup_454_sig[ 54].nbits = 12; gv_local.lookup_454_sig[ 54].bits = 0x70A00000;
  gv_local.lookup_454_sig[ 55].nbits = 12; gv_local.lookup_454_sig[ 55].bits = 0x70500000;
  gv_local.lookup_454_sig[ 56].nbits = 12; gv_local.lookup_454_sig[ 56].bits = 0x66700000;
  gv_local.lookup_454_sig[ 57].nbits = 12; gv_local.lookup_454_sig[ 57].bits = 0x66100000;
  gv_local.lookup_454_sig[ 58].nbits = 12; gv_local.lookup_454_sig[ 58].bits = 0x5DB00000;
  gv_local.lookup_454_sig[ 59].nbits = 12; gv_local.lookup_454_sig[ 59].bits = 0x5D400000;
  gv_local.lookup_454_sig[ 60].nbits = 12; gv_local.lookup_454_sig[ 60].bits = 0x5D100000;
  gv_local.lookup_454_sig[ 61].nbits = 12; gv_local.lookup_454_sig[ 61].bits = 0x58600000;
  gv_local.lookup_454_sig[ 62].nbits = 12; gv_local.lookup_454_sig[ 62].bits = 0x58300000;
  gv_local.lookup_454_sig[ 63].nbits = 12; gv_local.lookup_454_sig[ 63].bits = 0x58700000;
  gv_local.lookup_454_sig[ 64].nbits = 12; gv_local.lookup_454_sig[ 64].bits = 0x59800000;
  gv_local.lookup_454_sig[ 65].nbits = 12; gv_local.lookup_454_sig[ 65].bits = 0x5D900000;
  gv_local.lookup_454_sig[ 66].nbits = 12; gv_local.lookup_454_sig[ 66].bits = 0x70400000;
  gv_local.lookup_454_sig[ 67].nbits = 12; gv_local.lookup_454_sig[ 67].bits = 0x84100000;
  gv_local.lookup_454_sig[ 68].nbits = 12; gv_local.lookup_454_sig[ 68].bits = 0x94C00000;
  gv_local.lookup_454_sig[ 69].nbits = 12; gv_local.lookup_454_sig[ 69].bits = 0xB3F00000;
  gv_local.lookup_454_sig[ 70].nbits = 12; gv_local.lookup_454_sig[ 70].bits = 0xDBD00000;
  gv_local.lookup_454_sig[ 71].nbits = 11; gv_local.lookup_454_sig[ 71].bits = 0x2A400000;
  gv_local.lookup_454_sig[ 72].nbits = 11; gv_local.lookup_454_sig[ 72].bits = 0x5D600000;
  gv_local.lookup_454_sig[ 73].nbits = 11; gv_local.lookup_454_sig[ 73].bits = 0x87200000;
  gv_local.lookup_454_sig[ 74].nbits = 11; gv_local.lookup_454_sig[ 74].bits = 0xB2E00000;
  gv_local.lookup_454_sig[ 75].nbits = 11; gv_local.lookup_454_sig[ 75].bits = 0xDA400000;
  gv_local.lookup_454_sig[ 76].nbits = 10; gv_local.lookup_454_sig[ 76].bits = 0x28C00000;
  gv_local.lookup_454_sig[ 77].nbits = 10; gv_local.lookup_454_sig[ 77].bits = 0x59C00000;
  gv_local.lookup_454_sig[ 78].nbits = 10; gv_local.lookup_454_sig[ 78].bits = 0x84400000;
  gv_local.lookup_454_sig[ 79].nbits = 10; gv_local.lookup_454_sig[ 79].bits = 0xB2800000;
  gv_local.lookup_454_sig[ 80].nbits = 10; gv_local.lookup_454_sig[ 80].bits = 0xDA000000;
  gv_local.lookup_454_sig[ 81].nbits =  9; gv_local.lookup_454_sig[ 81].bits = 0x28000000;
  gv_local.lookup_454_sig[ 82].nbits =  9; gv_local.lookup_454_sig[ 82].bits = 0x59000000;
  gv_local.lookup_454_sig[ 83].nbits =  9; gv_local.lookup_454_sig[ 83].bits = 0x84800000;
  gv_local.lookup_454_sig[ 84].nbits =  9; gv_local.lookup_454_sig[ 84].bits = 0xA7800000;
  gv_local.lookup_454_sig[ 85].nbits =  9; gv_local.lookup_454_sig[ 85].bits = 0xDA800000;
  gv_local.lookup_454_sig[ 86].nbits =  8; gv_local.lookup_454_sig[ 86].bits = 0x29000000;
  gv_local.lookup_454_sig[ 87].nbits =  8; gv_local.lookup_454_sig[ 87].bits = 0x5C000000;
  gv_local.lookup_454_sig[ 88].nbits =  8; gv_local.lookup_454_sig[ 88].bits = 0x85000000;
  gv_local.lookup_454_sig[ 89].nbits =  8; gv_local.lookup_454_sig[ 89].bits = 0xA6000000;
  gv_local.lookup_454_sig[ 90].nbits =  8; gv_local.lookup_454_sig[ 90].bits = 0xD3000000;
  gv_local.lookup_454_sig[ 91].nbits =  7; gv_local.lookup_454_sig[ 91].bits = 0x22000000;
  gv_local.lookup_454_sig[ 92].nbits =  7; gv_local.lookup_454_sig[ 92].bits = 0x54000000;
  gv_local.lookup_454_sig[ 93].nbits =  7; gv_local.lookup_454_sig[ 93].bits = 0x72000000;
  gv_local.lookup_454_sig[ 94].nbits =  7; gv_local.lookup_454_sig[ 94].bits = 0x92000000;
  gv_local.lookup_454_sig[ 95].nbits =  7; gv_local.lookup_454_sig[ 95].bits = 0xB8000000;
  gv_local.lookup_454_sig[ 96].nbits =  7; gv_local.lookup_454_sig[ 96].bits = 0xD8000000;
  gv_local.lookup_454_sig[ 97].nbits =  7; gv_local.lookup_454_sig[ 97].bits = 0xF6000000;
  gv_local.lookup_454_sig[ 98].nbits =  6; gv_local.lookup_454_sig[ 98].bits = 0x2C000000;
  gv_local.lookup_454_sig[ 99].nbits =  6; gv_local.lookup_454_sig[ 99].bits = 0x44000000;
  gv_local.lookup_454_sig[100].nbits =  6; gv_local.lookup_454_sig[100].bits = 0xA0000000;
  gv_local.lookup_454_sig[101].nbits =  6; gv_local.lookup_454_sig[101].bits = 0xF8000000;
  gv_local.lookup_454_sig[102].nbits =  6; gv_local.lookup_454_sig[102].bits = 0xD4000000;
  gv_local.lookup_454_sig[103].nbits =  6; gv_local.lookup_454_sig[103].bits = 0xB4000000;
  gv_local.lookup_454_sig[104].nbits =  6; gv_local.lookup_454_sig[104].bits = 0x9C000000;
  gv_local.lookup_454_sig[105].nbits =  6; gv_local.lookup_454_sig[105].bits = 0x88000000;
  gv_local.lookup_454_sig[106].nbits =  6; gv_local.lookup_454_sig[106].bits = 0x60000000;
  gv_local.lookup_454_sig[107].nbits =  6; gv_local.lookup_454_sig[107].bits = 0x40000000;
  gv_local.lookup_454_sig[108].nbits =  7; gv_local.lookup_454_sig[108].bits = 0xF2000000;
  gv_local.lookup_454_sig[109].nbits =  7; gv_local.lookup_454_sig[109].bits = 0xBE000000;
  gv_local.lookup_454_sig[110].nbits =  7; gv_local.lookup_454_sig[110].bits = 0x90000000;
  gv_local.lookup_454_sig[111].nbits =  7; gv_local.lookup_454_sig[111].bits = 0x5E000000;
  gv_local.lookup_454_sig[112].nbits =  7; gv_local.lookup_454_sig[112].bits = 0x26000000;
  gv_local.lookup_454_sig[113].nbits =  8; gv_local.lookup_454_sig[113].bits = 0xD1000000;
  gv_local.lookup_454_sig[114].nbits =  8; gv_local.lookup_454_sig[114].bits = 0x96000000;
  gv_local.lookup_454_sig[115].nbits =  8; gv_local.lookup_454_sig[115].bits = 0x67000000;
  gv_local.lookup_454_sig[116].nbits =  8; gv_local.lookup_454_sig[116].bits = 0x34000000;
  gv_local.lookup_454_sig[117].nbits =  9; gv_local.lookup_454_sig[117].bits = 0xDC000000;
  gv_local.lookup_454_sig[118].nbits =  9; gv_local.lookup_454_sig[118].bits = 0xA5800000;
  gv_local.lookup_454_sig[119].nbits =  9; gv_local.lookup_454_sig[119].bits = 0x75800000;
  gv_local.lookup_454_sig[120].nbits =  9; gv_local.lookup_454_sig[120].bits = 0x51000000;
  gv_local.lookup_454_sig[121].nbits = 10; gv_local.lookup_454_sig[121].bits = 0xF5800000;
  gv_local.lookup_454_sig[122].nbits = 10; gv_local.lookup_454_sig[122].bits = 0xBA000000;
  gv_local.lookup_454_sig[123].nbits = 10; gv_local.lookup_454_sig[123].bits = 0x86400000;
  gv_local.lookup_454_sig[124].nbits = 10; gv_local.lookup_454_sig[124].bits = 0x56000000;
  gv_local.lookup_454_sig[125].nbits = 11; gv_local.lookup_454_sig[125].bits = 0xF5200000;
  gv_local.lookup_454_sig[126].nbits = 11; gv_local.lookup_454_sig[126].bits = 0xB3400000;
  gv_local.lookup_454_sig[127].nbits = 11; gv_local.lookup_454_sig[127].bits = 0x87000000;
  gv_local.lookup_454_sig[128].nbits = 11; gv_local.lookup_454_sig[128].bits = 0x58400000;
  gv_local.lookup_454_sig[129].nbits = 11; gv_local.lookup_454_sig[129].bits = 0x2AA00000;
  gv_local.lookup_454_sig[130].nbits = 12; gv_local.lookup_454_sig[130].bits = 0xF1800000;
  gv_local.lookup_454_sig[131].nbits = 12; gv_local.lookup_454_sig[131].bits = 0xDBB00000;
  gv_local.lookup_454_sig[132].nbits = 12; gv_local.lookup_454_sig[132].bits = 0xCBA00000;
  gv_local.lookup_454_sig[133].nbits = 12; gv_local.lookup_454_sig[133].bits = 0xC9900000;
  gv_local.lookup_454_sig[134].nbits = 12; gv_local.lookup_454_sig[134].bits = 0xB3700000;
  gv_local.lookup_454_sig[135].nbits = 12; gv_local.lookup_454_sig[135].bits = 0xA5000000;
  gv_local.lookup_454_sig[136].nbits = 12; gv_local.lookup_454_sig[136].bits = 0x94500000;
  gv_local.lookup_454_sig[137].nbits = 12; gv_local.lookup_454_sig[137].bits = 0x8E500000;
  gv_local.lookup_454_sig[138].nbits = 12; gv_local.lookup_454_sig[138].bits = 0x75700000;
  gv_local.lookup_454_sig[139].nbits = 12; gv_local.lookup_454_sig[139].bits = 0x70B00000;
  gv_local.lookup_454_sig[140].nbits = 12; gv_local.lookup_454_sig[140].bits = 0x66D00000;
  gv_local.lookup_454_sig[141].nbits = 12; gv_local.lookup_454_sig[141].bits = 0x5D300000;
  gv_local.lookup_454_sig[142].nbits = 12; gv_local.lookup_454_sig[142].bits = 0x51F00000;
  gv_local.lookup_454_sig[143].nbits = 12; gv_local.lookup_454_sig[143].bits = 0x36B00000;
  gv_local.lookup_454_sig[144].nbits = 12; gv_local.lookup_454_sig[144].bits = 0x33A00000;
  gv_local.lookup_454_sig[145].nbits = 12; gv_local.lookup_454_sig[145].bits = 0x2A700000;
  gv_local.lookup_454_sig[146].nbits = 12; gv_local.lookup_454_sig[146].bits = 0x24D00000;
  gv_local.lookup_454_sig[147].nbits = 13; gv_local.lookup_454_sig[147].bits = 0xF5D00000;
  gv_local.lookup_454_sig[148].nbits = 13; gv_local.lookup_454_sig[148].bits = 0xF1980000;
  gv_local.lookup_454_sig[149].nbits = 13; gv_local.lookup_454_sig[149].bits = 0xF0600000;
  gv_local.lookup_454_sig[150].nbits = 13; gv_local.lookup_454_sig[150].bits = 0xCB480000;
  gv_local.lookup_454_sig[151].nbits = 13; gv_local.lookup_454_sig[151].bits = 0xA5300000;
  gv_local.lookup_454_sig[152].nbits = 13; gv_local.lookup_454_sig[152].bits = 0xA7600000;
  gv_local.lookup_454_sig[153].nbits = 13; gv_local.lookup_454_sig[153].bits = 0xA7680000;
  gv_local.lookup_454_sig[154].nbits = 13; gv_local.lookup_454_sig[154].bits = 0xB3600000;
  gv_local.lookup_454_sig[155].nbits = 13; gv_local.lookup_454_sig[155].bits = 0xA7700000;
  gv_local.lookup_454_sig[156].nbits = 13; gv_local.lookup_454_sig[156].bits = 0xB3E00000;
  gv_local.lookup_454_sig[157].nbits = 13; gv_local.lookup_454_sig[157].bits = 0xC9800000;
  gv_local.lookup_454_sig[158].nbits = 13; gv_local.lookup_454_sig[158].bits = 0xC9E80000;
  gv_local.lookup_454_sig[159].nbits = 13; gv_local.lookup_454_sig[159].bits = 0xCBB00000;
  gv_local.lookup_454_sig[160].nbits = 13; gv_local.lookup_454_sig[160].bits = 0xDA680000;
  gv_local.lookup_454_sig[161].nbits = 13; gv_local.lookup_454_sig[161].bits = 0xF0680000;
  gv_local.lookup_454_sig[162].nbits = 13; gv_local.lookup_454_sig[162].bits = 0xF5E80000;
  gv_local.lookup_454_sig[163].nbits = 12; gv_local.lookup_454_sig[163].bits = 0x28B00000;
  gv_local.lookup_454_sig[164].nbits = 12; gv_local.lookup_454_sig[164].bits = 0x33B00000;
  gv_local.lookup_454_sig[165].nbits = 12; gv_local.lookup_454_sig[165].bits = 0x51C00000;
  gv_local.lookup_454_sig[166].nbits = 12; gv_local.lookup_454_sig[166].bits = 0x5DA00000;
  gv_local.lookup_454_sig[167].nbits = 12; gv_local.lookup_454_sig[167].bits = 0x74600000;
  gv_local.lookup_454_sig[168].nbits = 12; gv_local.lookup_454_sig[168].bits = 0x94000000;
  gv_local.lookup_454_sig[169].nbits = 12; gv_local.lookup_454_sig[169].bits = 0x97700000;
  gv_local.lookup_454_sig[170].nbits = 12; gv_local.lookup_454_sig[170].bits = 0xB3C00000;
  gv_local.lookup_454_sig[171].nbits = 12; gv_local.lookup_454_sig[171].bits = 0xD0400000;
  gv_local.lookup_454_sig[172].nbits = 12; gv_local.lookup_454_sig[172].bits = 0xF0200000;
  gv_local.lookup_454_sig[173].nbits = 11; gv_local.lookup_454_sig[173].bits = 0x24E00000;
  gv_local.lookup_454_sig[174].nbits = 11; gv_local.lookup_454_sig[174].bits = 0x36C00000;
  gv_local.lookup_454_sig[175].nbits = 11; gv_local.lookup_454_sig[175].bits = 0x5DC00000;
  gv_local.lookup_454_sig[176].nbits = 11; gv_local.lookup_454_sig[176].bits = 0x75400000;
  gv_local.lookup_454_sig[177].nbits = 11; gv_local.lookup_454_sig[177].bits = 0x94A00000;
  gv_local.lookup_454_sig[178].nbits = 11; gv_local.lookup_454_sig[178].bits = 0xB3200000;
  gv_local.lookup_454_sig[179].nbits = 11; gv_local.lookup_454_sig[179].bits = 0xD0200000;
  gv_local.lookup_454_sig[180].nbits = 11; gv_local.lookup_454_sig[180].bits = 0xF5000000;
  gv_local.lookup_454_sig[181].nbits = 10; gv_local.lookup_454_sig[181].bits = 0x2AC00000;
  gv_local.lookup_454_sig[182].nbits = 10; gv_local.lookup_454_sig[182].bits = 0x56800000;
  gv_local.lookup_454_sig[183].nbits = 10; gv_local.lookup_454_sig[183].bits = 0x70000000;
  gv_local.lookup_454_sig[184].nbits = 10; gv_local.lookup_454_sig[184].bits = 0x87800000;
  gv_local.lookup_454_sig[185].nbits = 10; gv_local.lookup_454_sig[185].bits = 0xA4400000;
  gv_local.lookup_454_sig[186].nbits = 10; gv_local.lookup_454_sig[186].bits = 0xC8C00000;
  gv_local.lookup_454_sig[187].nbits = 10; gv_local.lookup_454_sig[187].bits = 0xDB400000;
  gv_local.lookup_454_sig[188].nbits =  9; gv_local.lookup_454_sig[188].bits = 0x24000000;
  gv_local.lookup_454_sig[189].nbits =  9; gv_local.lookup_454_sig[189].bits = 0x36000000;
  gv_local.lookup_454_sig[190].nbits =  9; gv_local.lookup_454_sig[190].bits = 0x58800000;
  gv_local.lookup_454_sig[191].nbits =  9; gv_local.lookup_454_sig[191].bits = 0x71000000;
  gv_local.lookup_454_sig[192].nbits =  9; gv_local.lookup_454_sig[192].bits = 0x86800000;
  gv_local.lookup_454_sig[193].nbits =  9; gv_local.lookup_454_sig[193].bits = 0x95800000;
  gv_local.lookup_454_sig[194].nbits =  9; gv_local.lookup_454_sig[194].bits = 0xB2000000;
  gv_local.lookup_454_sig[195].nbits =  9; gv_local.lookup_454_sig[195].bits = 0xC9000000;
  gv_local.lookup_454_sig[196].nbits =  9; gv_local.lookup_454_sig[196].bits = 0xD2000000;
  gv_local.lookup_454_sig[197].nbits =  9; gv_local.lookup_454_sig[197].bits = 0xDC800000;
  gv_local.lookup_454_sig[198].nbits =  9; gv_local.lookup_454_sig[198].bits = 0xF1000000;
  gv_local.lookup_454_sig[199].nbits =  8; gv_local.lookup_454_sig[199].bits = 0x20000000;
  gv_local.lookup_454_sig[200].nbits =  8; gv_local.lookup_454_sig[200].bits = 0x2B000000;
  gv_local.lookup_454_sig[201].nbits =  8; gv_local.lookup_454_sig[201].bits = 0x37000000;
  gv_local.lookup_454_sig[202].nbits =  8; gv_local.lookup_454_sig[202].bits = 0x50000000;
  gv_local.lookup_454_sig[203].nbits =  8; gv_local.lookup_454_sig[203].bits = 0x35000000;
  gv_local.lookup_454_sig[204].nbits =  8; gv_local.lookup_454_sig[204].bits = 0x32000000;
  gv_local.lookup_454_sig[205].nbits =  8; gv_local.lookup_454_sig[205].bits = 0x25000000;
  gv_local.lookup_454_sig[206].nbits =  8; gv_local.lookup_454_sig[206].bits = 0x21000000;
  gv_local.lookup_454_sig[207].nbits =  9; gv_local.lookup_454_sig[207].bits = 0xDD800000;
  gv_local.lookup_454_sig[208].nbits =  9; gv_local.lookup_454_sig[208].bits = 0xD0800000;
  gv_local.lookup_454_sig[209].nbits =  9; gv_local.lookup_454_sig[209].bits = 0xBA800000;
  gv_local.lookup_454_sig[210].nbits =  9; gv_local.lookup_454_sig[210].bits = 0xA4800000;
  gv_local.lookup_454_sig[211].nbits =  9; gv_local.lookup_454_sig[211].bits = 0x8E800000;
  gv_local.lookup_454_sig[212].nbits =  9; gv_local.lookup_454_sig[212].bits = 0x71800000;
  gv_local.lookup_454_sig[213].nbits =  9; gv_local.lookup_454_sig[213].bits = 0x57000000;
  gv_local.lookup_454_sig[214].nbits =  9; gv_local.lookup_454_sig[214].bits = 0x33000000;
  gv_local.lookup_454_sig[215].nbits = 10; gv_local.lookup_454_sig[215].bits = 0xF1C00000;
  gv_local.lookup_454_sig[216].nbits = 10; gv_local.lookup_454_sig[216].bits = 0xCBC00000;
  gv_local.lookup_454_sig[217].nbits = 10; gv_local.lookup_454_sig[217].bits = 0xA7000000;
  gv_local.lookup_454_sig[218].nbits = 10; gv_local.lookup_454_sig[218].bits = 0x87C00000;
  gv_local.lookup_454_sig[219].nbits = 10; gv_local.lookup_454_sig[219].bits = 0x66800000;
  gv_local.lookup_454_sig[220].nbits = 10; gv_local.lookup_454_sig[220].bits = 0x51800000;
  gv_local.lookup_454_sig[221].nbits = 10; gv_local.lookup_454_sig[221].bits = 0x24800000;
  gv_local.lookup_454_sig[222].nbits = 11; gv_local.lookup_454_sig[222].bits = 0xD2E00000;
  gv_local.lookup_454_sig[223].nbits = 11; gv_local.lookup_454_sig[223].bits = 0xB3A00000;
  gv_local.lookup_454_sig[224].nbits = 11; gv_local.lookup_454_sig[224].bits = 0x94600000;
  gv_local.lookup_454_sig[225].nbits = 11; gv_local.lookup_454_sig[225].bits = 0x74000000;
  gv_local.lookup_454_sig[226].nbits = 11; gv_local.lookup_454_sig[226].bits = 0x56E00000;
  gv_local.lookup_454_sig[227].nbits = 11; gv_local.lookup_454_sig[227].bits = 0x2A800000;
  gv_local.lookup_454_sig[228].nbits = 12; gv_local.lookup_454_sig[228].bits = 0xF0700000;
  gv_local.lookup_454_sig[229].nbits = 12; gv_local.lookup_454_sig[229].bits = 0xCB800000;
  gv_local.lookup_454_sig[230].nbits = 12; gv_local.lookup_454_sig[230].bits = 0xA5100000;
  gv_local.lookup_454_sig[231].nbits = 12; gv_local.lookup_454_sig[231].bits = 0x8E600000;
  gv_local.lookup_454_sig[232].nbits = 12; gv_local.lookup_454_sig[232].bits = 0x70900000;
  gv_local.lookup_454_sig[233].nbits = 12; gv_local.lookup_454_sig[233].bits = 0x58200000;
  gv_local.lookup_454_sig[234].nbits = 12; gv_local.lookup_454_sig[234].bits = 0x30A00000;
  gv_local.lookup_454_sig[235].nbits = 13; gv_local.lookup_454_sig[235].bits = 0xF5D80000;
  gv_local.lookup_454_sig[236].nbits = 13; gv_local.lookup_454_sig[236].bits = 0xDA600000;
  gv_local.lookup_454_sig[237].nbits = 13; gv_local.lookup_454_sig[237].bits = 0xC8980000;
  gv_local.lookup_454_sig[238].nbits = 13; gv_local.lookup_454_sig[238].bits = 0xA4200000;
  gv_local.lookup_454_sig[239].nbits = 13; gv_local.lookup_454_sig[239].bits = 0x8E480000;
  gv_local.lookup_454_sig[240].nbits = 13; gv_local.lookup_454_sig[240].bits = 0x75600000;
  gv_local.lookup_454_sig[241].nbits = 13; gv_local.lookup_454_sig[241].bits = 0x5D200000;
  gv_local.lookup_454_sig[242].nbits = 13; gv_local.lookup_454_sig[242].bits = 0x36F00000;
  gv_local.lookup_454_sig[243].nbits = 13; gv_local.lookup_454_sig[243].bits = 0x30880000;
  gv_local.lookup_454_sig[244].nbits = 13; gv_local.lookup_454_sig[244].bits = 0x28800000;
  gv_local.lookup_454_sig[245].nbits = 14; gv_local.lookup_454_sig[245].bits = 0xF5780000;
  gv_local.lookup_454_sig[246].nbits = 14; gv_local.lookup_454_sig[246].bits = 0xF1900000;
  gv_local.lookup_454_sig[247].nbits = 14; gv_local.lookup_454_sig[247].bits = 0xDA700000;
  gv_local.lookup_454_sig[248].nbits = 14; gv_local.lookup_454_sig[248].bits = 0xDA780000;
  gv_local.lookup_454_sig[249].nbits = 14; gv_local.lookup_454_sig[249].bits = 0xD2C80000;
  gv_local.lookup_454_sig[250].nbits = 14; gv_local.lookup_454_sig[250].bits = 0xCB9C0000;
  gv_local.lookup_454_sig[251].nbits = 14; gv_local.lookup_454_sig[251].bits = 0xC8B80000;
  gv_local.lookup_454_sig[252].nbits = 14; gv_local.lookup_454_sig[252].bits = 0xCB500000;
  gv_local.lookup_454_sig[253].nbits = 14; gv_local.lookup_454_sig[253].bits = 0xC9E40000;
  gv_local.lookup_454_sig[254].nbits = 14; gv_local.lookup_454_sig[254].bits = 0xC8BC0000;
  gv_local.lookup_454_sig[255].nbits = 14; gv_local.lookup_454_sig[255].bits = 0xCB740000;
  gv_local.lookup_454_sig[256].nbits = 14; gv_local.lookup_454_sig[256].bits = 0xD0540000;
  gv_local.lookup_454_sig[257].nbits = 14; gv_local.lookup_454_sig[257].bits = 0xD2C40000;
  gv_local.lookup_454_sig[258].nbits = 14; gv_local.lookup_454_sig[258].bits = 0xDA7C0000;
  gv_local.lookup_454_sig[259].nbits = 14; gv_local.lookup_454_sig[259].bits = 0xF0380000;
  gv_local.lookup_454_sig[260].nbits = 14; gv_local.lookup_454_sig[260].bits = 0xF5700000;
  gv_local.lookup_454_sig[261].nbits = 14; gv_local.lookup_454_sig[261].bits = 0xF5F00000;
  gv_local.lookup_454_sig[262].nbits = 13; gv_local.lookup_454_sig[262].bits = 0x2A680000;
  gv_local.lookup_454_sig[263].nbits = 13; gv_local.lookup_454_sig[263].bits = 0x33C00000;
  gv_local.lookup_454_sig[264].nbits = 13; gv_local.lookup_454_sig[264].bits = 0x36F80000;
  gv_local.lookup_454_sig[265].nbits = 13; gv_local.lookup_454_sig[265].bits = 0x5D280000;
  gv_local.lookup_454_sig[266].nbits = 13; gv_local.lookup_454_sig[266].bits = 0x66680000;
  gv_local.lookup_454_sig[267].nbits = 13; gv_local.lookup_454_sig[267].bits = 0x75100000;
  gv_local.lookup_454_sig[268].nbits = 13; gv_local.lookup_454_sig[268].bits = 0x86180000;
  gv_local.lookup_454_sig[269].nbits = 13; gv_local.lookup_454_sig[269].bits = 0x97200000;
  gv_local.lookup_454_sig[270].nbits = 13; gv_local.lookup_454_sig[270].bits = 0xA7780000;
  gv_local.lookup_454_sig[271].nbits = 13; gv_local.lookup_454_sig[271].bits = 0xB3E80000;
  gv_local.lookup_454_sig[272].nbits = 13; gv_local.lookup_454_sig[272].bits = 0xCBB80000;
  gv_local.lookup_454_sig[273].nbits = 13; gv_local.lookup_454_sig[273].bits = 0xF0100000;
  gv_local.lookup_454_sig[274].nbits = 13; gv_local.lookup_454_sig[274].bits = 0xF5F80000;
  gv_local.lookup_454_sig[275].nbits = 12; gv_local.lookup_454_sig[275].bits = 0x30B00000;
  gv_local.lookup_454_sig[276].nbits = 12; gv_local.lookup_454_sig[276].bits = 0x51E00000;
  gv_local.lookup_454_sig[277].nbits = 12; gv_local.lookup_454_sig[277].bits = 0x5D800000;
  gv_local.lookup_454_sig[278].nbits = 12; gv_local.lookup_454_sig[278].bits = 0x74400000;
  gv_local.lookup_454_sig[279].nbits = 12; gv_local.lookup_454_sig[279].bits = 0x87700000;
  gv_local.lookup_454_sig[280].nbits = 12; gv_local.lookup_454_sig[280].bits = 0x97600000;
  gv_local.lookup_454_sig[281].nbits = 12; gv_local.lookup_454_sig[281].bits = 0xB3100000;
  gv_local.lookup_454_sig[282].nbits = 12; gv_local.lookup_454_sig[282].bits = 0xC9F00000;
  gv_local.lookup_454_sig[283].nbits = 12; gv_local.lookup_454_sig[283].bits = 0xDBA00000;
  gv_local.lookup_454_sig[284].nbits = 12; gv_local.lookup_454_sig[284].bits = 0xF5600000;
  gv_local.lookup_454_sig[285].nbits = 11; gv_local.lookup_454_sig[285].bits = 0x2A200000;
  gv_local.lookup_454_sig[286].nbits = 11; gv_local.lookup_454_sig[286].bits = 0x33E00000;
  gv_local.lookup_454_sig[287].nbits = 11; gv_local.lookup_454_sig[287].bits = 0x58000000;
  gv_local.lookup_454_sig[288].nbits = 11; gv_local.lookup_454_sig[288].bits = 0x66200000;
  gv_local.lookup_454_sig[289].nbits = 11; gv_local.lookup_454_sig[289].bits = 0x75200000;
  gv_local.lookup_454_sig[290].nbits = 11; gv_local.lookup_454_sig[290].bits = 0x87400000;
  gv_local.lookup_454_sig[291].nbits = 11; gv_local.lookup_454_sig[291].bits = 0x94800000;
  gv_local.lookup_454_sig[292].nbits = 11; gv_local.lookup_454_sig[292].bits = 0xA4000000;
  gv_local.lookup_454_sig[293].nbits = 11; gv_local.lookup_454_sig[293].bits = 0xA7400000;
  gv_local.lookup_454_sig[294].nbits = 11; gv_local.lookup_454_sig[294].bits = 0xBA600000;
  gv_local.lookup_454_sig[295].nbits = 11; gv_local.lookup_454_sig[295].bits = 0xC9A00000;
  gv_local.lookup_454_sig[296].nbits = 11; gv_local.lookup_454_sig[296].bits = 0xD0000000;
  gv_local.lookup_454_sig[297].nbits = 11; gv_local.lookup_454_sig[297].bits = 0xD2A00000;
  gv_local.lookup_454_sig[298].nbits = 11; gv_local.lookup_454_sig[298].bits = 0xDB000000;
  gv_local.lookup_454_sig[299].nbits = 11; gv_local.lookup_454_sig[299].bits = 0xDB800000;
  gv_local.lookup_454_sig[300].nbits = 11; gv_local.lookup_454_sig[300].bits = 0xF0400000;
  gv_local.lookup_454_sig[301].nbits = 11; gv_local.lookup_454_sig[301].bits = 0xF1A00000;
  gv_local.lookup_454_sig[302].nbits = 11; gv_local.lookup_454_sig[302].bits = 0xF0A00000;
  gv_local.lookup_454_sig[303].nbits = 11; gv_local.lookup_454_sig[303].bits = 0xF0800000;
  gv_local.lookup_454_sig[304].nbits = 11; gv_local.lookup_454_sig[304].bits = 0xDBE00000;
  gv_local.lookup_454_sig[305].nbits = 11; gv_local.lookup_454_sig[305].bits = 0xDB200000;
  gv_local.lookup_454_sig[306].nbits = 11; gv_local.lookup_454_sig[306].bits = 0xD2800000;
  gv_local.lookup_454_sig[307].nbits = 11; gv_local.lookup_454_sig[307].bits = 0xC9C00000;
  gv_local.lookup_454_sig[308].nbits = 11; gv_local.lookup_454_sig[308].bits = 0xBA400000;
  gv_local.lookup_454_sig[309].nbits = 11; gv_local.lookup_454_sig[309].bits = 0xB2C00000;
  gv_local.lookup_454_sig[310].nbits = 11; gv_local.lookup_454_sig[310].bits = 0x97000000;
  gv_local.lookup_454_sig[311].nbits = 11; gv_local.lookup_454_sig[311].bits = 0x94200000;
  gv_local.lookup_454_sig[312].nbits = 11; gv_local.lookup_454_sig[312].bits = 0x84200000;
  gv_local.lookup_454_sig[313].nbits = 11; gv_local.lookup_454_sig[313].bits = 0x66E00000;
  gv_local.lookup_454_sig[314].nbits = 11; gv_local.lookup_454_sig[314].bits = 0x59A00000;
  gv_local.lookup_454_sig[315].nbits = 11; gv_local.lookup_454_sig[315].bits = 0x36800000;
  gv_local.lookup_454_sig[316].nbits = 11; gv_local.lookup_454_sig[316].bits = 0x2A000000;
  gv_local.lookup_454_sig[317].nbits = 12; gv_local.lookup_454_sig[317].bits = 0xF5C00000;
  gv_local.lookup_454_sig[318].nbits = 12; gv_local.lookup_454_sig[318].bits = 0xD2D00000;
  gv_local.lookup_454_sig[319].nbits = 12; gv_local.lookup_454_sig[319].bits = 0xC8800000;
  gv_local.lookup_454_sig[320].nbits = 12; gv_local.lookup_454_sig[320].bits = 0xA4300000;
  gv_local.lookup_454_sig[321].nbits = 12; gv_local.lookup_454_sig[321].bits = 0x8E700000;
  gv_local.lookup_454_sig[322].nbits = 12; gv_local.lookup_454_sig[322].bits = 0x74700000;
  gv_local.lookup_454_sig[323].nbits = 12; gv_local.lookup_454_sig[323].bits = 0x5D000000;
  gv_local.lookup_454_sig[324].nbits = 12; gv_local.lookup_454_sig[324].bits = 0x33D00000;
  gv_local.lookup_454_sig[325].nbits = 12; gv_local.lookup_454_sig[325].bits = 0x24C00000;
  gv_local.lookup_454_sig[326].nbits = 13; gv_local.lookup_454_sig[326].bits = 0xF0300000;
  gv_local.lookup_454_sig[327].nbits = 13; gv_local.lookup_454_sig[327].bits = 0xCB400000;
  gv_local.lookup_454_sig[328].nbits = 13; gv_local.lookup_454_sig[328].bits = 0xA5280000;
  gv_local.lookup_454_sig[329].nbits = 13; gv_local.lookup_454_sig[329].bits = 0x94E80000;
  gv_local.lookup_454_sig[330].nbits = 13; gv_local.lookup_454_sig[330].bits = 0x84080000;
  gv_local.lookup_454_sig[331].nbits = 13; gv_local.lookup_454_sig[331].bits = 0x66600000;
  gv_local.lookup_454_sig[332].nbits = 13; gv_local.lookup_454_sig[332].bits = 0x51D80000;
  gv_local.lookup_454_sig[333].nbits = 13; gv_local.lookup_454_sig[333].bits = 0x33900000;
  gv_local.lookup_454_sig[334].nbits = 14; gv_local.lookup_454_sig[334].bits = 0xF5E40000;
  gv_local.lookup_454_sig[335].nbits = 14; gv_local.lookup_454_sig[335].bits = 0xD2CC0000;
  gv_local.lookup_454_sig[336].nbits = 14; gv_local.lookup_454_sig[336].bits = 0xC98C0000;
  gv_local.lookup_454_sig[337].nbits = 14; gv_local.lookup_454_sig[337].bits = 0xB3000000;
  gv_local.lookup_454_sig[338].nbits = 14; gv_local.lookup_454_sig[338].bits = 0x94F00000;
  gv_local.lookup_454_sig[339].nbits = 14; gv_local.lookup_454_sig[339].bits = 0x87600000;
  gv_local.lookup_454_sig[340].nbits = 14; gv_local.lookup_454_sig[340].bits = 0x66080000;
  gv_local.lookup_454_sig[341].nbits = 14; gv_local.lookup_454_sig[341].bits = 0x66040000;
  gv_local.lookup_454_sig[342].nbits = 14; gv_local.lookup_454_sig[342].bits = 0x51D40000;
  gv_local.lookup_454_sig[343].nbits = 14; gv_local.lookup_454_sig[343].bits = 0x33CC0000;
  gv_local.lookup_454_sig[344].nbits = 14; gv_local.lookup_454_sig[344].bits = 0x28AC0000;
  gv_local.lookup_454_sig[345].nbits = 14; gv_local.lookup_454_sig[345].bits = 0x28A00000;
  gv_local.lookup_454_sig[346].nbits = 15; gv_local.lookup_454_sig[346].bits = 0xF03E0000;
  gv_local.lookup_454_sig[347].nbits = 15; gv_local.lookup_454_sig[347].bits = 0xF1940000;
  gv_local.lookup_454_sig[348].nbits = 15; gv_local.lookup_454_sig[348].bits = 0xDBCA0000;
  gv_local.lookup_454_sig[349].nbits = 15; gv_local.lookup_454_sig[349].bits = 0xDBC00000;
  gv_local.lookup_454_sig[350].nbits = 15; gv_local.lookup_454_sig[350].bits = 0xC9E20000;
  gv_local.lookup_454_sig[351].nbits = 15; gv_local.lookup_454_sig[351].bits = 0xA42C0000;
  gv_local.lookup_454_sig[352].nbits = 15; gv_local.lookup_454_sig[352].bits = 0xA42A0000;
  gv_local.lookup_454_sig[353].nbits = 15; gv_local.lookup_454_sig[353].bits = 0xA53E0000;
  gv_local.lookup_454_sig[354].nbits = 15; gv_local.lookup_454_sig[354].bits = 0xB3D80000;
  gv_local.lookup_454_sig[355].nbits = 15; gv_local.lookup_454_sig[355].bits = 0xA53C0000;
  gv_local.lookup_454_sig[356].nbits = 15; gv_local.lookup_454_sig[356].bits = 0xC8900000;
  gv_local.lookup_454_sig[357].nbits = 15; gv_local.lookup_454_sig[357].bits = 0xC9880000;
  gv_local.lookup_454_sig[358].nbits = 15; gv_local.lookup_454_sig[358].bits = 0xC8A80000;
  gv_local.lookup_454_sig[359].nbits = 15; gv_local.lookup_454_sig[359].bits = 0xCB700000;
  gv_local.lookup_454_sig[360].nbits = 15; gv_local.lookup_454_sig[360].bits = 0xD2C00000;
  gv_local.lookup_454_sig[361].nbits = 15; gv_local.lookup_454_sig[361].bits = 0xD2C20000;
  gv_local.lookup_454_sig[362].nbits = 15; gv_local.lookup_454_sig[362].bits = 0xF01E0000;
  gv_local.lookup_454_sig[363].nbits = 15; gv_local.lookup_454_sig[363].bits = 0xF01A0000;
  gv_local.lookup_454_sig[364].nbits = 15; gv_local.lookup_454_sig[364].bits = 0xF5E00000;
  gv_local.lookup_454_sig[365].nbits = 14; gv_local.lookup_454_sig[365].bits = 0x28A40000;
  gv_local.lookup_454_sig[366].nbits = 14; gv_local.lookup_454_sig[366].bits = 0x33800000;
  gv_local.lookup_454_sig[367].nbits = 14; gv_local.lookup_454_sig[367].bits = 0x36EC0000;
  gv_local.lookup_454_sig[368].nbits = 14; gv_local.lookup_454_sig[368].bits = 0x51D00000;
  gv_local.lookup_454_sig[369].nbits = 14; gv_local.lookup_454_sig[369].bits = 0x56D40000;
  gv_local.lookup_454_sig[370].nbits = 14; gv_local.lookup_454_sig[370].bits = 0x5D540000;
  gv_local.lookup_454_sig[371].nbits = 14; gv_local.lookup_454_sig[371].bits = 0x70840000;
  gv_local.lookup_454_sig[372].nbits = 14; gv_local.lookup_454_sig[372].bits = 0x86080000;
  gv_local.lookup_454_sig[373].nbits = 14; gv_local.lookup_454_sig[373].bits = 0x87640000;
  gv_local.lookup_454_sig[374].nbits = 14; gv_local.lookup_454_sig[374].bits = 0x94180000;
  gv_local.lookup_454_sig[375].nbits = 14; gv_local.lookup_454_sig[375].bits = 0x972C0000;
  gv_local.lookup_454_sig[376].nbits = 14; gv_local.lookup_454_sig[376].bits = 0xB3040000;
  gv_local.lookup_454_sig[377].nbits = 14; gv_local.lookup_454_sig[377].bits = 0xC8940000;
  gv_local.lookup_454_sig[378].nbits = 14; gv_local.lookup_454_sig[378].bits = 0xCB980000;
  gv_local.lookup_454_sig[379].nbits = 14; gv_local.lookup_454_sig[379].bits = 0xDA740000;
  gv_local.lookup_454_sig[380].nbits = 14; gv_local.lookup_454_sig[380].bits = 0xDBC40000;
  gv_local.lookup_454_sig[381].nbits = 14; gv_local.lookup_454_sig[381].bits = 0xF57C0000;
  gv_local.lookup_454_sig[382].nbits = 13; gv_local.lookup_454_sig[382].bits = 0x28900000;
  gv_local.lookup_454_sig[383].nbits = 13; gv_local.lookup_454_sig[383].bits = 0x28880000;
  gv_local.lookup_454_sig[384].nbits = 13; gv_local.lookup_454_sig[384].bits = 0x30900000;
  gv_local.lookup_454_sig[385].nbits = 13; gv_local.lookup_454_sig[385].bits = 0x36E00000;
  gv_local.lookup_454_sig[386].nbits = 13; gv_local.lookup_454_sig[386].bits = 0x56C00000;
  gv_local.lookup_454_sig[387].nbits = 13; gv_local.lookup_454_sig[387].bits = 0x56D80000;
  gv_local.lookup_454_sig[388].nbits = 13; gv_local.lookup_454_sig[388].bits = 0x5D580000;
  gv_local.lookup_454_sig[389].nbits = 13; gv_local.lookup_454_sig[389].bits = 0x66C00000;
  gv_local.lookup_454_sig[390].nbits = 13; gv_local.lookup_454_sig[390].bits = 0x70880000;
  gv_local.lookup_454_sig[391].nbits = 13; gv_local.lookup_454_sig[391].bits = 0x75680000;
  gv_local.lookup_454_sig[392].nbits = 13; gv_local.lookup_454_sig[392].bits = 0x84000000;
  gv_local.lookup_454_sig[393].nbits = 13; gv_local.lookup_454_sig[393].bits = 0x86100000;
  gv_local.lookup_454_sig[394].nbits = 13; gv_local.lookup_454_sig[394].bits = 0x8E400000;
  gv_local.lookup_454_sig[395].nbits = 13; gv_local.lookup_454_sig[395].bits = 0x94400000;
  gv_local.lookup_454_sig[396].nbits = 13; gv_local.lookup_454_sig[396].bits = 0x94E00000;
  gv_local.lookup_454_sig[397].nbits = 13; gv_local.lookup_454_sig[397].bits = 0x94F80000;
  gv_local.lookup_454_sig[398].nbits = 13; gv_local.lookup_454_sig[398].bits = 0x94480000;
  gv_local.lookup_454_sig[399].nbits = 13; gv_local.lookup_454_sig[399].bits = 0x97300000;
  gv_local.lookup_454_sig[400].nbits = 13; gv_local.lookup_454_sig[400].bits = 0xB3D00000;
  gv_local.lookup_454_sig[401].nbits = 13; gv_local.lookup_454_sig[401].bits = 0xD0580000;
  gv_local.lookup_454_sig[402].nbits = 13; gv_local.lookup_454_sig[402].bits = 0xCB900000;
  gv_local.lookup_454_sig[403].nbits = 13; gv_local.lookup_454_sig[403].bits = 0xCB680000;
  gv_local.lookup_454_sig[404].nbits = 13; gv_local.lookup_454_sig[404].bits = 0xCB780000;
  gv_local.lookup_454_sig[405].nbits = 13; gv_local.lookup_454_sig[405].bits = 0xCB580000;
  gv_local.lookup_454_sig[406].nbits = 13; gv_local.lookup_454_sig[406].bits = 0xCB600000;
  gv_local.lookup_454_sig[407].nbits = 13; gv_local.lookup_454_sig[407].bits = 0xC8B00000;
  gv_local.lookup_454_sig[408].nbits = 13; gv_local.lookup_454_sig[408].bits = 0xC8A00000;
  gv_local.lookup_454_sig[409].nbits = 13; gv_local.lookup_454_sig[409].bits = 0xB3680000;
  gv_local.lookup_454_sig[410].nbits = 13; gv_local.lookup_454_sig[410].bits = 0xB3080000;
  gv_local.lookup_454_sig[411].nbits = 13; gv_local.lookup_454_sig[411].bits = 0xA5200000;
  gv_local.lookup_454_sig[412].nbits = 13; gv_local.lookup_454_sig[412].bits = 0x97380000;
  gv_local.lookup_454_sig[413].nbits = 13; gv_local.lookup_454_sig[413].bits = 0x94D80000;
  gv_local.lookup_454_sig[414].nbits = 13; gv_local.lookup_454_sig[414].bits = 0x94100000;
  gv_local.lookup_454_sig[415].nbits = 13; gv_local.lookup_454_sig[415].bits = 0x86000000;
  gv_local.lookup_454_sig[416].nbits = 13; gv_local.lookup_454_sig[416].bits = 0x74380000;
  gv_local.lookup_454_sig[417].nbits = 13; gv_local.lookup_454_sig[417].bits = 0x66C80000;
  gv_local.lookup_454_sig[418].nbits = 13; gv_local.lookup_454_sig[418].bits = 0x59980000;
  gv_local.lookup_454_sig[419].nbits = 13; gv_local.lookup_454_sig[419].bits = 0x56C80000;
  gv_local.lookup_454_sig[420].nbits = 13; gv_local.lookup_454_sig[420].bits = 0x36A00000;
  gv_local.lookup_454_sig[421].nbits = 13; gv_local.lookup_454_sig[421].bits = 0x30800000;
  gv_local.lookup_454_sig[422].nbits = 13; gv_local.lookup_454_sig[422].bits = 0x28980000;
  gv_local.lookup_454_sig[423].nbits = 14; gv_local.lookup_454_sig[423].bits = 0xF5740000;
  gv_local.lookup_454_sig[424].nbits = 14; gv_local.lookup_454_sig[424].bits = 0xDBCC0000;
  gv_local.lookup_454_sig[425].nbits = 14; gv_local.lookup_454_sig[425].bits = 0xCB540000;
  gv_local.lookup_454_sig[426].nbits = 14; gv_local.lookup_454_sig[426].bits = 0xB3DC0000;
  gv_local.lookup_454_sig[427].nbits = 14; gv_local.lookup_454_sig[427].bits = 0xA5380000;
  gv_local.lookup_454_sig[428].nbits = 14; gv_local.lookup_454_sig[428].bits = 0x97280000;
  gv_local.lookup_454_sig[429].nbits = 14; gv_local.lookup_454_sig[429].bits = 0x876C0000;
  gv_local.lookup_454_sig[430].nbits = 14; gv_local.lookup_454_sig[430].bits = 0x74340000;
  gv_local.lookup_454_sig[431].nbits = 14; gv_local.lookup_454_sig[431].bits = 0x74300000;
  gv_local.lookup_454_sig[432].nbits = 14; gv_local.lookup_454_sig[432].bits = 0x59900000;
  gv_local.lookup_454_sig[433].nbits = 14; gv_local.lookup_454_sig[433].bits = 0x36A80000;
  gv_local.lookup_454_sig[434].nbits = 14; gv_local.lookup_454_sig[434].bits = 0x30980000;
  gv_local.lookup_454_sig[435].nbits = 15; gv_local.lookup_454_sig[435].bits = 0xF5F40000;
  gv_local.lookup_454_sig[436].nbits = 15; gv_local.lookup_454_sig[436].bits = 0xF03C0000;
  gv_local.lookup_454_sig[437].nbits = 15; gv_local.lookup_454_sig[437].bits = 0xF0180000;
  gv_local.lookup_454_sig[438].nbits = 15; gv_local.lookup_454_sig[438].bits = 0xD0500000;
  gv_local.lookup_454_sig[439].nbits = 15; gv_local.lookup_454_sig[439].bits = 0xC8AA0000;
  gv_local.lookup_454_sig[440].nbits = 15; gv_local.lookup_454_sig[440].bits = 0xA4280000;
  gv_local.lookup_454_sig[441].nbits = 15; gv_local.lookup_454_sig[441].bits = 0x94D20000;
  gv_local.lookup_454_sig[442].nbits = 15; gv_local.lookup_454_sig[442].bits = 0x94D00000;
  gv_local.lookup_454_sig[443].nbits = 15; gv_local.lookup_454_sig[443].bits = 0x860E0000;
  gv_local.lookup_454_sig[444].nbits = 15; gv_local.lookup_454_sig[444].bits = 0x751A0000;
  gv_local.lookup_454_sig[445].nbits = 15; gv_local.lookup_454_sig[445].bits = 0x70820000;
  gv_local.lookup_454_sig[446].nbits = 15; gv_local.lookup_454_sig[446].bits = 0x660C0000;
  gv_local.lookup_454_sig[447].nbits = 15; gv_local.lookup_454_sig[447].bits = 0x59940000;
  gv_local.lookup_454_sig[448].nbits = 15; gv_local.lookup_454_sig[448].bits = 0x56D20000;
  gv_local.lookup_454_sig[449].nbits = 15; gv_local.lookup_454_sig[449].bits = 0x36E80000;
  gv_local.lookup_454_sig[450].nbits = 16; gv_local.lookup_454_sig[450].bits = 0xF1970000;
  gv_local.lookup_454_sig[451].nbits = 16; gv_local.lookup_454_sig[451].bits = 0xD0520000;
  gv_local.lookup_454_sig[452].nbits = 16; gv_local.lookup_454_sig[452].bits = 0xC8AC0000;
  gv_local.lookup_454_sig[453].nbits = 16; gv_local.lookup_454_sig[453].bits = 0xC8930000;
  gv_local.lookup_454_sig[454].nbits = 16; gv_local.lookup_454_sig[454].bits = 0xC9E00000;
  gv_local.lookup_454_sig[455].nbits = 16; gv_local.lookup_454_sig[455].bits = 0xC98A0000;
  gv_local.lookup_454_sig[456].nbits = 16; gv_local.lookup_454_sig[456].bits = 0xC8AE0000;
  gv_local.lookup_454_sig[457].nbits = 16; gv_local.lookup_454_sig[457].bits = 0xC8920000;
  gv_local.lookup_454_sig[458].nbits = 16; gv_local.lookup_454_sig[458].bits = 0xC8AD0000;
  gv_local.lookup_454_sig[459].nbits = 16; gv_local.lookup_454_sig[459].bits = 0xC8AF0000;
  gv_local.lookup_454_sig[460].nbits = 16; gv_local.lookup_454_sig[460].bits = 0xC98B0000;
  gv_local.lookup_454_sig[461].nbits = 16; gv_local.lookup_454_sig[461].bits = 0xB3DA0000;
  gv_local.lookup_454_sig[462].nbits = 16; gv_local.lookup_454_sig[462].bits = 0xCB720000;
  gv_local.lookup_454_sig[463].nbits = 16; gv_local.lookup_454_sig[463].bits = 0xC9E10000;
  gv_local.lookup_454_sig[464].nbits = 16; gv_local.lookup_454_sig[464].bits = 0xD0530000;
  gv_local.lookup_454_sig[465].nbits = 16; gv_local.lookup_454_sig[465].bits = 0xB3DB0000;
  gv_local.lookup_454_sig[466].nbits = 16; gv_local.lookup_454_sig[466].bits = 0xDBC80000;
  gv_local.lookup_454_sig[467].nbits = 16; gv_local.lookup_454_sig[467].bits = 0xCB730000;
  gv_local.lookup_454_sig[468].nbits = 16; gv_local.lookup_454_sig[468].bits = 0xDBC90000;
  gv_local.lookup_454_sig[469].nbits = 16; gv_local.lookup_454_sig[469].bits = 0xF01D0000;
  gv_local.lookup_454_sig[470].nbits = 16; gv_local.lookup_454_sig[470].bits = 0xF01C0000;
  gv_local.lookup_454_sig[471].nbits = 16; gv_local.lookup_454_sig[471].bits = 0xF5E20000;
  gv_local.lookup_454_sig[472].nbits = 16; gv_local.lookup_454_sig[472].bits = 0xF1960000;
  gv_local.lookup_454_sig[473].nbits = 16; gv_local.lookup_454_sig[473].bits = 0xF5E30000;
  gv_local.lookup_454_sig[474].nbits = 15; gv_local.lookup_454_sig[474].bits = 0x2A600000;
  gv_local.lookup_454_sig[475].nbits = 15; gv_local.lookup_454_sig[475].bits = 0x309C0000;
  gv_local.lookup_454_sig[476].nbits = 15; gv_local.lookup_454_sig[476].bits = 0x309E0000;
  gv_local.lookup_454_sig[477].nbits = 15; gv_local.lookup_454_sig[477].bits = 0x2A620000;
  gv_local.lookup_454_sig[478].nbits = 15; gv_local.lookup_454_sig[478].bits = 0x36EA0000;
  gv_local.lookup_454_sig[479].nbits = 15; gv_local.lookup_454_sig[479].bits = 0x56D00000;
  gv_local.lookup_454_sig[480].nbits = 15; gv_local.lookup_454_sig[480].bits = 0x59960000;
  gv_local.lookup_454_sig[481].nbits = 15; gv_local.lookup_454_sig[481].bits = 0x5D520000;
  gv_local.lookup_454_sig[482].nbits = 15; gv_local.lookup_454_sig[482].bits = 0x5D500000;
  gv_local.lookup_454_sig[483].nbits = 15; gv_local.lookup_454_sig[483].bits = 0x66000000;
  gv_local.lookup_454_sig[484].nbits = 15; gv_local.lookup_454_sig[484].bits = 0x66020000;
  gv_local.lookup_454_sig[485].nbits = 15; gv_local.lookup_454_sig[485].bits = 0x660E0000;
  gv_local.lookup_454_sig[486].nbits = 15; gv_local.lookup_454_sig[486].bits = 0x75180000;
  gv_local.lookup_454_sig[487].nbits = 15; gv_local.lookup_454_sig[487].bits = 0x70800000;
  gv_local.lookup_454_sig[488].nbits = 15; gv_local.lookup_454_sig[488].bits = 0x751C0000;
  gv_local.lookup_454_sig[489].nbits = 15; gv_local.lookup_454_sig[489].bits = 0x860C0000;
  gv_local.lookup_454_sig[490].nbits = 15; gv_local.lookup_454_sig[490].bits = 0x751E0000;
  gv_local.lookup_454_sig[491].nbits = 15; gv_local.lookup_454_sig[491].bits = 0x876A0000;
  gv_local.lookup_454_sig[492].nbits = 15; gv_local.lookup_454_sig[492].bits = 0x87680000;
  gv_local.lookup_454_sig[493].nbits = 15; gv_local.lookup_454_sig[493].bits = 0x941C0000;
  gv_local.lookup_454_sig[494].nbits = 15; gv_local.lookup_454_sig[494].bits = 0x94D60000;
  gv_local.lookup_454_sig[495].nbits = 15; gv_local.lookup_454_sig[495].bits = 0x941E0000;
  gv_local.lookup_454_sig[496].nbits = 15; gv_local.lookup_454_sig[496].bits = 0x94F60000;
  gv_local.lookup_454_sig[497].nbits = 15; gv_local.lookup_454_sig[497].bits = 0x94F40000;
  gv_local.lookup_454_sig[498].nbits = 15; gv_local.lookup_454_sig[498].bits = 0x94D40000;
  gv_local.lookup_454_sig[499].nbits = 15; gv_local.lookup_454_sig[499].bits = 0xA42E0000;
  gv_local.lookup_454_sig[500].nbits = 15; gv_local.lookup_454_sig[500].bits = 0xDBC20000;
  gv_local.lookup_454_sig[501].nbits = 14; gv_local.lookup_454_sig[501].bits = 0x36AC0000;
  gv_local.lookup_454_sig[502].nbits = 14; gv_local.lookup_454_sig[502].bits = 0x339C0000;
  gv_local.lookup_454_sig[503].nbits = 14; gv_local.lookup_454_sig[503].bits = 0x33980000;
  gv_local.lookup_454_sig[504].nbits = 14; gv_local.lookup_454_sig[504].bits = 0x33C80000;
  gv_local.lookup_454_sig[505].nbits = 14; gv_local.lookup_454_sig[505].bits = 0x33840000;
  gv_local.lookup_454_sig[506].nbits = 14; gv_local.lookup_454_sig[506].bits = 0x33880000;
  gv_local.lookup_454_sig[507].nbits = 14; gv_local.lookup_454_sig[507].bits = 0x2A640000;
  gv_local.lookup_454_sig[508].nbits = 14; gv_local.lookup_454_sig[508].bits = 0x338C0000;
  gv_local.lookup_454_sig[509].nbits = 14; gv_local.lookup_454_sig[509].bits = 0x28A80000;
  gv_local.lookup_454_sig[510].nbits = 15; gv_local.lookup_454_sig[510].bits = 0xF5F60000;
  gv_local.lookup_454_sig[511].nbits =  9; gv_local.lookup_454_sig[511].bits = 0x97800000;

#if SRA_DECOMPRESS_USE_REVERSE_LOOKUP
  /* reverse huffman lookup table
   */
  memset (gv_local.rlookup_454_sig, 0, sizeof(uint16_t)*REV_HUFF_TREESIZE_16);

  for (i=0; i<SIG_HUFF_TREESIZE; i++)
  {
    val = gv_local.lookup_454_sig[i].bits >> 16;
    gv_local.rlookup_454_sig [val] = (uint16_t) i;
    for (j=1; j<=(0xFFFF>>gv_local.lookup_454_sig[i].nbits); j++)
      gv_local.rlookup_454_sig [val+j] = (uint16_t) i;
  }
#endif
  _s_sort_lookup (gv_local.lookup_454_sig, gv_local.idx_454_sig, SIG_HUFF_TREESIZE);

  gv_local.lookup_454_sig_initialized = 1;
}
