(*
 * Copyright (c) 2017, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the "hack" directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 *)

(**
 * HTML5 special entity decoding
 *
 * HHVM decodes certain HTML entities present in input strings before
 * generating bytecode. In order to generate bytecode identical to HHVM's,
 * this module performs the same HTML entity decoding as HHVM.
 *
 * The list of entities tested was taken from
 * https://dev.w3.org/html5/html-author/charref on 09/27/2017.
 *)
let entity_regex = Str.regexp "&[^;]+;"

let decode_entity s = match s with
| "&#x00023;" -> "#"
| "&#35;" -> "#"
| "&#x02198;" -> "\o342\o206\o230"
| "&#8600;" -> "\o342\o206\o230"
| "&#x02199;" -> "\o342\o206\o231"
| "&#8601;" -> "\o342\o206\o231"
| "&#x0219A;" -> "\o342\o206\o232"
| "&#8602;" -> "\o342\o206\o232"
| "&sup2;" -> "\o302\o262"
| "&#x000B2;" -> "\o302\o262"
| "&#178;" -> "\o302\o262"
| "&#x0219B;" -> "\o342\o206\o233"
| "&#8603;" -> "\o342\o206\o233"
| "&rarrw;" -> "\o342\o207\o235"
| "&#x021DD;" -> "\o342\o207\o235"
| "&#8669;" -> "\o342\o207\o235"
| "&#x0219D;" -> "\o342\o206\o235"
| "&#8605;" -> "\o342\o206\o235"
| "&#x0219E;" -> "\o342\o206\o236"
| "&#8606;" -> "\o342\o206\o236"
| "&#x0219F;" -> "\o342\o206\o237"
| "&#8607;" -> "\o342\o206\o237"
| "&sup3;" -> "\o302\o263"
| "&#x000B3;" -> "\o302\o263"
| "&#179;" -> "\o302\o263"
| "&#x021A0;" -> "\o342\o206\o240"
| "&#8608;" -> "\o342\o206\o240"
| "&#x021A1;" -> "\o342\o206\o241"
| "&#8609;" -> "\o342\o206\o241"
| "&#x021A2;" -> "\o342\o206\o242"
| "&#8610;" -> "\o342\o206\o242"
| "&#x021A3;" -> "\o342\o206\o243"
| "&#8611;" -> "\o342\o206\o243"
| "&acute;" -> "\o302\o264"
| "&#x000B4;" -> "\o302\o264"
| "&#180;" -> "\o302\o264"
| "&#x021A4;" -> "\o342\o206\o244"
| "&#8612;" -> "\o342\o206\o244"
| "&#x021A5;" -> "\o342\o206\o245"
| "&#8613;" -> "\o342\o206\o245"
| "&#x021A6;" -> "\o342\o206\o246"
| "&#8614;" -> "\o342\o206\o246"
| "&#x021A7;" -> "\o342\o206\o247"
| "&#8615;" -> "\o342\o206\o247"
| "&#x021A9;" -> "\o342\o206\o251"
| "&#8617;" -> "\o342\o206\o251"
| "&#x021AA;" -> "\o342\o206\o252"
| "&#8618;" -> "\o342\o206\o252"
| "&micro;" -> "\o302\o265"
| "&#x000B5;" -> "\o302\o265"
| "&#181;" -> "\o302\o265"
| "&#x021AB;" -> "\o342\o206\o253"
| "&#8619;" -> "\o342\o206\o253"
| "&#x021AC;" -> "\o342\o206\o254"
| "&#8620;" -> "\o342\o206\o254"
| "&#x021AD;" -> "\o342\o206\o255"
| "&#8621;" -> "\o342\o206\o255"
| "&para;" -> "\o302\o266"
| "&#x000B6;" -> "\o302\o266"
| "&#182;" -> "\o302\o266"
| "&#x021AE;" -> "\o342\o206\o256"
| "&#8622;" -> "\o342\o206\o256"
| "&#x021B0;" -> "\o342\o206\o260"
| "&#8624;" -> "\o342\o206\o260"
| "&#x021B1;" -> "\o342\o206\o261"
| "&#8625;" -> "\o342\o206\o261"
| "&#x021B2;" -> "\o342\o206\o262"
| "&#8626;" -> "\o342\o206\o262"
| "&middot;" -> "\o302\o267"
| "&#x000B7;" -> "\o302\o267"
| "&#183;" -> "\o302\o267"
| "&#x021B3;" -> "\o342\o206\o263"
| "&#8627;" -> "\o342\o206\o263"
| "&crarr;" -> "\o342\o206\o265"
| "&#x021B5;" -> "\o342\o206\o265"
| "&#8629;" -> "\o342\o206\o265"
| "&#x021B6;" -> "\o342\o206\o266"
| "&#8630;" -> "\o342\o206\o266"
| "&#x021B7;" -> "\o342\o206\o267"
| "&#8631;" -> "\o342\o206\o267"
| "&#x021BA;" -> "\o342\o206\o272"
| "&#8634;" -> "\o342\o206\o272"
| "&#x021BB;" -> "\o342\o206\o273"
| "&#8635;" -> "\o342\o206\o273"
| "&#x021BC;" -> "\o342\o206\o274"
| "&#8636;" -> "\o342\o206\o274"
| "&#x021BD;" -> "\o342\o206\o275"
| "&#8637;" -> "\o342\o206\o275"
| "&#x021BE;" -> "\o342\o206\o276"
| "&#8638;" -> "\o342\o206\o276"
| "&#x021BF;" -> "\o342\o206\o277"
| "&#8639;" -> "\o342\o206\o277"
| "&cedil;" -> "\o302\o270"
| "&#x000B8;" -> "\o302\o270"
| "&#184;" -> "\o302\o270"
| "&#x021C0;" -> "\o342\o207\o200"
| "&#8640;" -> "\o342\o207\o200"
| "&#x021C1;" -> "\o342\o207\o201"
| "&#8641;" -> "\o342\o207\o201"
| "&#x021C2;" -> "\o342\o207\o202"
| "&#8642;" -> "\o342\o207\o202"
| "&#x021C3;" -> "\o342\o207\o203"
| "&#8643;" -> "\o342\o207\o203"
| "&#x021C4;" -> "\o342\o207\o204"
| "&#8644;" -> "\o342\o207\o204"
| "&sup1;" -> "\o302\o271"
| "&#x000B9;" -> "\o302\o271"
| "&#185;" -> "\o302\o271"
| "&#x021C5;" -> "\o342\o207\o205"
| "&#8645;" -> "\o342\o207\o205"
| "&#x021C6;" -> "\o342\o207\o206"
| "&#8646;" -> "\o342\o207\o206"
| "&#x021C7;" -> "\o342\o207\o207"
| "&#8647;" -> "\o342\o207\o207"
| "&ordm;" -> "\o302\o272"
| "&#x000BA;" -> "\o302\o272"
| "&#186;" -> "\o302\o272"
| "&#x021C8;" -> "\o342\o207\o210"
| "&#8648;" -> "\o342\o207\o210"
| "&#x021C9;" -> "\o342\o207\o211"
| "&#8649;" -> "\o342\o207\o211"
| "&#x021CA;" -> "\o342\o207\o212"
| "&#8650;" -> "\o342\o207\o212"
| "&raquo;" -> "\o302\o273"
| "&#x000BB;" -> "\o302\o273"
| "&#187;" -> "\o302\o273"
| "&#x021CB;" -> "\o342\o207\o213"
| "&#8651;" -> "\o342\o207\o213"
| "&#x021CC;" -> "\o342\o207\o214"
| "&#8652;" -> "\o342\o207\o214"
| "&#x021CD;" -> "\o342\o207\o215"
| "&#8653;" -> "\o342\o207\o215"
| "&frac14;" -> "\o302\o274"
| "&#x000BC;" -> "\o302\o274"
| "&#188;" -> "\o302\o274"
| "&#x021CE;" -> "\o342\o207\o216"
| "&#8654;" -> "\o342\o207\o216"
| "&#x021CF;" -> "\o342\o207\o217"
| "&#8655;" -> "\o342\o207\o217"
| "&lArr;" -> "\o342\o207\o220"
| "&#x021D0;" -> "\o342\o207\o220"
| "&#8656;" -> "\o342\o207\o220"
| "&uArr;" -> "\o342\o207\o221"
| "&#x021D1;" -> "\o342\o207\o221"
| "&#8657;" -> "\o342\o207\o221"
| "&frac12;" -> "\o302\o275"
| "&#x000BD;" -> "\o302\o275"
| "&#189;" -> "\o302\o275"
| "&rArr;" -> "\o342\o207\o222"
| "&#x021D2;" -> "\o342\o207\o222"
| "&#8658;" -> "\o342\o207\o222"
| "&dArr;" -> "\o342\o207\o223"
| "&#x021D3;" -> "\o342\o207\o223"
| "&#8659;" -> "\o342\o207\o223"
| "&hArr;" -> "\o342\o207\o224"
| "&#x021D4;" -> "\o342\o207\o224"
| "&#8660;" -> "\o342\o207\o224"
| "&vArr;" -> "\o342\o207\o225"
| "&#x021D5;" -> "\o342\o207\o225"
| "&#8661;" -> "\o342\o207\o225"
| "&#x021D6;" -> "\o342\o207\o226"
| "&#8662;" -> "\o342\o207\o226"
| "&#x021D7;" -> "\o342\o207\o227"
| "&#8663;" -> "\o342\o207\o227"
| "&frac34;" -> "\o302\o276"
| "&#x000BE;" -> "\o302\o276"
| "&#190;" -> "\o302\o276"
| "&#x021D8;" -> "\o342\o207\o230"
| "&#8664;" -> "\o342\o207\o230"
| "&#x021D9;" -> "\o342\o207\o231"
| "&#8665;" -> "\o342\o207\o231"
| "&lAarr;" -> "\o342\o207\o232"
| "&#x021DA;" -> "\o342\o207\o232"
| "&#8666;" -> "\o342\o207\o232"
| "&rAarr;" -> "\o342\o207\o233"
| "&#x021DB;" -> "\o342\o207\o233"
| "&#8667;" -> "\o342\o207\o233"
| "&iquest;" -> "\o302\o277"
| "&#x000BF;" -> "\o302\o277"
| "&#191;" -> "\o302\o277"
| "&#x021E4;" -> "\o342\o207\o244"
| "&#8676;" -> "\o342\o207\o244"
| "&#x021E5;" -> "\o342\o207\o245"
| "&#8677;" -> "\o342\o207\o245"
| "&#x021F5;" -> "\o342\o207\o265"
| "&#8693;" -> "\o342\o207\o265"
| "&#x021FD;" -> "\o342\o207\o275"
| "&#8701;" -> "\o342\o207\o275"
| "&#x00024;" -> "$"
| "&#36;" -> "$"
| "&Agrave;" -> "\o303\o200"
| "&#x000C0;" -> "\o303\o200"
| "&#192;" -> "\o303\o200"
| "&#x021FE;" -> "\o342\o207\o276"
| "&#8702;" -> "\o342\o207\o276"
| "&#x021FF;" -> "\o342\o207\o277"
| "&#8703;" -> "\o342\o207\o277"
| "&forall;" -> "\o342\o210\o200"
| "&#x02200;" -> "\o342\o210\o200"
| "&#8704;" -> "\o342\o210\o200"
| "&comp;" -> "\o342\o210\o201"
| "&#x02201;" -> "\o342\o210\o201"
| "&#8705;" -> "\o342\o210\o201"
| "&part;" -> "\o342\o210\o202"
| "&#x02202;" -> "\o342\o210\o202"
| "&#8706;" -> "\o342\o210\o202"
| "&Aacute;" -> "\o303\o201"
| "&#x000C1;" -> "\o303\o201"
| "&#193;" -> "\o303\o201"
| "&exist;" -> "\o342\o210\o203"
| "&#x02203;" -> "\o342\o210\o203"
| "&#8707;" -> "\o342\o210\o203"
| "&nexist;" -> "\o342\o210\o204"
| "&#x02204;" -> "\o342\o210\o204"
| "&#8708;" -> "\o342\o210\o204"
| "&empty;" -> "\o342\o210\o205"
| "&#x02205;" -> "\o342\o210\o205"
| "&#8709;" -> "\o342\o210\o205"
| "&nabla;" -> "\o342\o210\o207"
| "&#x02207;" -> "\o342\o210\o207"
| "&#8711;" -> "\o342\o210\o207"
| "&Acirc;" -> "\o303\o202"
| "&#x000C2;" -> "\o303\o202"
| "&#194;" -> "\o303\o202"
| "&isin;" -> "\o342\o210\o210"
| "&#x02208;" -> "\o342\o210\o210"
| "&#8712;" -> "\o342\o210\o210"
| "&notin;" -> "\o342\o210\o211"
| "&#x02209;" -> "\o342\o210\o211"
| "&#8713;" -> "\o342\o210\o211"
| "&Atilde;" -> "\o303\o203"
| "&#x000C3;" -> "\o303\o203"
| "&#195;" -> "\o303\o203"
| "&ni;" -> "\o342\o210\o213"
| "&#x0220B;" -> "\o342\o210\o213"
| "&#8715;" -> "\o342\o210\o213"
| "&notni;" -> "\o342\o210\o214"
| "&#x0220C;" -> "\o342\o210\o214"
| "&#8716;" -> "\o342\o210\o214"
| "&prod;" -> "\o342\o210\o217"
| "&#x0220F;" -> "\o342\o210\o217"
| "&#8719;" -> "\o342\o210\o217"
| "&coprod;" -> "\o342\o210\o220"
| "&#x02210;" -> "\o342\o210\o220"
| "&#8720;" -> "\o342\o210\o220"
| "&sum;" -> "\o342\o210\o221"
| "&#x02211;" -> "\o342\o210\o221"
| "&#8721;" -> "\o342\o210\o221"
| "&Auml;" -> "\o303\o204"
| "&#x000C4;" -> "\o303\o204"
| "&#196;" -> "\o303\o204"
| "&minus;" -> "\o342\o210\o222"
| "&#x02212;" -> "\o342\o210\o222"
| "&#8722;" -> "\o342\o210\o222"
| "&mnplus;" -> "\o342\o210\o223"
| "&#x02213;" -> "\o342\o210\o223"
| "&#8723;" -> "\o342\o210\o223"
| "&plusdo;" -> "\o342\o210\o224"
| "&#x02214;" -> "\o342\o210\o224"
| "&#8724;" -> "\o342\o210\o224"
| "&setmn;" -> "\o342\o210\o226"
| "&#x02216;" -> "\o342\o210\o226"
| "&#8726;" -> "\o342\o210\o226"
| "&Aring;" -> "\o303\o205"
| "&#x000C5;" -> "\o303\o205"
| "&#197;" -> "\o303\o205"
| "&lowast;" -> "\o342\o210\o227"
| "&#x02217;" -> "\o342\o210\o227"
| "&#8727;" -> "\o342\o210\o227"
| "&compfn;" -> "\o342\o210\o230"
| "&#x02218;" -> "\o342\o210\o230"
| "&#8728;" -> "\o342\o210\o230"
| "&radic;" -> "\o342\o210\o232"
| "&#x0221A;" -> "\o342\o210\o232"
| "&#8730;" -> "\o342\o210\o232"
| "&prop;" -> "\o342\o210\o235"
| "&#x0221D;" -> "\o342\o210\o235"
| "&#8733;" -> "\o342\o210\o235"
| "&AElig;" -> "\o303\o206"
| "&#x000C6;" -> "\o303\o206"
| "&#198;" -> "\o303\o206"
| "&infin;" -> "\o342\o210\o236"
| "&#x0221E;" -> "\o342\o210\o236"
| "&#8734;" -> "\o342\o210\o236"
| "&#x0221F;" -> "\o342\o210\o237"
| "&#8735;" -> "\o342\o210\o237"
| "&ang;" -> "\o342\o210\o240"
| "&#x02220;" -> "\o342\o210\o240"
| "&#8736;" -> "\o342\o210\o240"
| "&angmsd;" -> "\o342\o210\o241"
| "&#x02221;" -> "\o342\o210\o241"
| "&#8737;" -> "\o342\o210\o241"
| "&angsph;" -> "\o342\o210\o242"
| "&#x02222;" -> "\o342\o210\o242"
| "&#8738;" -> "\o342\o210\o242"
| "&Ccedil;" -> "\o303\o207"
| "&#x000C7;" -> "\o303\o207"
| "&#199;" -> "\o303\o207"
| "&mid;" -> "\o342\o210\o243"
| "&#x02223;" -> "\o342\o210\o243"
| "&#8739;" -> "\o342\o210\o243"
| "&nmid;" -> "\o342\o210\o244"
| "&#x02224;" -> "\o342\o210\o244"
| "&#8740;" -> "\o342\o210\o244"
| "&par;" -> "\o342\o210\o245"
| "&#x02225;" -> "\o342\o210\o245"
| "&#8741;" -> "\o342\o210\o245"
| "&Egrave;" -> "\o303\o210"
| "&#x000C8;" -> "\o303\o210"
| "&#200;" -> "\o303\o210"
| "&npar;" -> "\o342\o210\o246"
| "&#x02226;" -> "\o342\o210\o246"
| "&#8742;" -> "\o342\o210\o246"
| "&and;" -> "\o342\o210\o247"
| "&#x02227;" -> "\o342\o210\o247"
| "&#8743;" -> "\o342\o210\o247"
| "&Eacute;" -> "\o303\o211"
| "&#x000C9;" -> "\o303\o211"
| "&#201;" -> "\o303\o211"
| "&or;" -> "\o342\o210\o250"
| "&#x02228;" -> "\o342\o210\o250"
| "&#8744;" -> "\o342\o210\o250"
| "&cap;" -> "\o342\o210\o251"
| "&#x02229;" -> "\o342\o210\o251"
| "&#8745;" -> "\o342\o210\o251"
| "&cup;" -> "\o342\o210\o252"
| "&#x0222A;" -> "\o342\o210\o252"
| "&#8746;" -> "\o342\o210\o252"
| "&int;" -> "\o342\o210\o253"
| "&#x0222B;" -> "\o342\o210\o253"
| "&#8747;" -> "\o342\o210\o253"
| "&#x0222C;" -> "\o342\o210\o254"
| "&#8748;" -> "\o342\o210\o254"
| "&Ecirc;" -> "\o303\o212"
| "&#x000CA;" -> "\o303\o212"
| "&#202;" -> "\o303\o212"
| "&#x0222D;" -> "\o342\o210\o255"
| "&#8749;" -> "\o342\o210\o255"
| "&conint;" -> "\o342\o210\o256"
| "&#x0222E;" -> "\o342\o210\o256"
| "&#8750;" -> "\o342\o210\o256"
| "&#x0222F;" -> "\o342\o210\o257"
| "&#8751;" -> "\o342\o210\o257"
| "&#x02230;" -> "\o342\o210\o260"
| "&#8752;" -> "\o342\o210\o260"
| "&Euml;" -> "\o303\o213"
| "&#x000CB;" -> "\o303\o213"
| "&#203;" -> "\o303\o213"
| "&#x02231;" -> "\o342\o210\o261"
| "&#8753;" -> "\o342\o210\o261"
| "&#x02232;" -> "\o342\o210\o262"
| "&#8754;" -> "\o342\o210\o262"
| "&#x02233;" -> "\o342\o210\o263"
| "&#8755;" -> "\o342\o210\o263"
| "&there4;" -> "\o342\o210\o264"
| "&#x02234;" -> "\o342\o210\o264"
| "&#8756;" -> "\o342\o210\o264"
| "&Igrave;" -> "\o303\o214"
| "&#x000CC;" -> "\o303\o214"
| "&#204;" -> "\o303\o214"
| "&becaus;" -> "\o342\o210\o265"
| "&#x02235;" -> "\o342\o210\o265"
| "&#8757;" -> "\o342\o210\o265"
| "&#x02236;" -> "\o342\o210\o266"
| "&#8758;" -> "\o342\o210\o266"
| "&#x02237;" -> "\o342\o210\o267"
| "&#8759;" -> "\o342\o210\o267"
| "&Iacute;" -> "\o303\o215"
| "&#x000CD;" -> "\o303\o215"
| "&#205;" -> "\o303\o215"
| "&#x02238;" -> "\o342\o210\o270"
| "&#8760;" -> "\o342\o210\o270"
| "&#x0223A;" -> "\o342\o210\o272"
| "&#8762;" -> "\o342\o210\o272"
| "&#x0223B;" -> "\o342\o210\o273"
| "&#8763;" -> "\o342\o210\o273"
| "&sim;" -> "\o342\o210\o274"
| "&#x0223C;" -> "\o342\o210\o274"
| "&#8764;" -> "\o342\o210\o274"
| "&bsim;" -> "\o342\o210\o275"
| "&#x0223D;" -> "\o342\o210\o275"
| "&#8765;" -> "\o342\o210\o275"
| "&Icirc;" -> "\o303\o216"
| "&#x000CE;" -> "\o303\o216"
| "&#206;" -> "\o303\o216"
| "&#x0223E;" -> "\o342\o210\o276"
| "&#8766;" -> "\o342\o210\o276"
| "&#x0223F;" -> "\o342\o210\o277"
| "&#8767;" -> "\o342\o210\o277"
| "&wreath;" -> "\o342\o211\o200"
| "&#x02240;" -> "\o342\o211\o200"
| "&#8768;" -> "\o342\o211\o200"
| "&nsim;" -> "\o342\o211\o201"
| "&#x02241;" -> "\o342\o211\o201"
| "&#8769;" -> "\o342\o211\o201"
| "&Iuml;" -> "\o303\o217"
| "&#x000CF;" -> "\o303\o217"
| "&#207;" -> "\o303\o217"
| "&#x02242;" -> "\o342\o211\o202"
| "&#8770;" -> "\o342\o211\o202"
| "&sime;" -> "\o342\o211\o203"
| "&#x02243;" -> "\o342\o211\o203"
| "&#8771;" -> "\o342\o211\o203"
| "&nsime;" -> "\o342\o211\o204"
| "&#x02244;" -> "\o342\o211\o204"
| "&#8772;" -> "\o342\o211\o204"
| "&ETH;" -> "\o303\o220"
| "&#x000D0;" -> "\o303\o220"
| "&#208;" -> "\o303\o220"
| "&cong;" -> "\o342\o211\o205"
| "&#x02245;" -> "\o342\o211\o205"
| "&#8773;" -> "\o342\o211\o205"
| "&#x02246;" -> "\o342\o211\o206"
| "&#8774;" -> "\o342\o211\o206"
| "&ncong;" -> "\o342\o211\o207"
| "&#x02247;" -> "\o342\o211\o207"
| "&#8775;" -> "\o342\o211\o207"
| "&asymp;" -> "\o342\o211\o215"
| "&#x0224D;" -> "\o342\o211\o215"
| "&#8781;" -> "\o342\o211\o215"
| "&Ntilde;" -> "\o303\o221"
| "&#x000D1;" -> "\o303\o221"
| "&#209;" -> "\o303\o221"
| "&#x02248;" -> "\o342\o211\o210"
| "&#8776;" -> "\o342\o211\o210"
| "&nap;" -> "\o342\o211\o211"
| "&#x02249;" -> "\o342\o211\o211"
| "&#8777;" -> "\o342\o211\o211"
| "&ape;" -> "\o342\o211\o212"
| "&#x0224A;" -> "\o342\o211\o212"
| "&#8778;" -> "\o342\o211\o212"
| "&Ograve;" -> "\o303\o222"
| "&#x000D2;" -> "\o303\o222"
| "&#210;" -> "\o303\o222"
| "&#x0224B;" -> "\o342\o211\o213"
| "&#8779;" -> "\o342\o211\o213"
| "&bcong;" -> "\o342\o211\o214"
| "&#x0224C;" -> "\o342\o211\o214"
| "&#8780;" -> "\o342\o211\o214"
| "&bump;" -> "\o342\o211\o216"
| "&#x0224E;" -> "\o342\o211\o216"
| "&#8782;" -> "\o342\o211\o216"
| "&bumpe;" -> "\o342\o211\o217"
| "&#x0224F;" -> "\o342\o211\o217"
| "&#8783;" -> "\o342\o211\o217"
| "&Oacute;" -> "\o303\o223"
| "&#x000D3;" -> "\o303\o223"
| "&#211;" -> "\o303\o223"
| "&#x02250;" -> "\o342\o211\o220"
| "&#8784;" -> "\o342\o211\o220"
| "&#x02251;" -> "\o342\o211\o221"
| "&#8785;" -> "\o342\o211\o221"
| "&#x00025;" -> "%"
| "&#37;" -> "%"
| "&Ocirc;" -> "\o303\o224"
| "&#x000D4;" -> "\o303\o224"
| "&#212;" -> "\o303\o224"
| "&#x02252;" -> "\o342\o211\o222"
| "&#8786;" -> "\o342\o211\o222"
| "&#x02253;" -> "\o342\o211\o223"
| "&#8787;" -> "\o342\o211\o223"
| "&#x02254;" -> "\o342\o211\o224"
| "&#8788;" -> "\o342\o211\o224"
| "&Otilde;" -> "\o303\o225"
| "&#x000D5;" -> "\o303\o225"
| "&#213;" -> "\o303\o225"
| "&#x02255;" -> "\o342\o211\o225"
| "&#8789;" -> "\o342\o211\o225"
| "&#x02256;" -> "\o342\o211\o226"
| "&#8790;" -> "\o342\o211\o226"
| "&#x02257;" -> "\o342\o211\o227"
| "&#8791;" -> "\o342\o211\o227"
| "&#x02259;" -> "\o342\o211\o231"
| "&#8793;" -> "\o342\o211\o231"
| "&Ouml;" -> "\o303\o226"
| "&#x000D6;" -> "\o303\o226"
| "&#214;" -> "\o303\o226"
| "&#x0225A;" -> "\o342\o211\o232"
| "&#8794;" -> "\o342\o211\o232"
| "&#x0225C;" -> "\o342\o211\o234"
| "&#8796;" -> "\o342\o211\o234"
| "&#x0225F;" -> "\o342\o211\o237"
| "&#8799;" -> "\o342\o211\o237"
| "&ne;" -> "\o342\o211\o240"
| "&#x02260;" -> "\o342\o211\o240"
| "&#8800;" -> "\o342\o211\o240"
| "&equiv;" -> "\o342\o211\o241"
| "&#x02261;" -> "\o342\o211\o241"
| "&#8801;" -> "\o342\o211\o241"
| "&times;" -> "\o303\o227"
| "&#x000D7;" -> "\o303\o227"
| "&#215;" -> "\o303\o227"
| "&#x02262;" -> "\o342\o211\o242"
| "&#8802;" -> "\o342\o211\o242"
| "&le;" -> "\o342\o211\o244"
| "&#x02264;" -> "\o342\o211\o244"
| "&#8804;" -> "\o342\o211\o244"
| "&ge;" -> "\o342\o211\o245"
| "&#x02265;" -> "\o342\o211\o245"
| "&#8805;" -> "\o342\o211\o245"
| "&lE;" -> "\o342\o211\o246"
| "&#x02266;" -> "\o342\o211\o246"
| "&#8806;" -> "\o342\o211\o246"
| "&Oslash;" -> "\o303\o230"
| "&#x000D8;" -> "\o303\o230"
| "&#216;" -> "\o303\o230"
| "&gE;" -> "\o342\o211\o247"
| "&#x02267;" -> "\o342\o211\o247"
| "&#8807;" -> "\o342\o211\o247"
| "&lnE;" -> "\o342\o211\o250"
| "&#x02268;" -> "\o342\o211\o250"
| "&#8808;" -> "\o342\o211\o250"
| "&gnE;" -> "\o342\o211\o251"
| "&#x02269;" -> "\o342\o211\o251"
| "&#8809;" -> "\o342\o211\o251"
| "&Lt;" -> "\o342\o211\o252"
| "&#x0226A;" -> "\o342\o211\o252"
| "&#8810;" -> "\o342\o211\o252"
| "&Ugrave;" -> "\o303\o231"
| "&#x000D9;" -> "\o303\o231"
| "&#217;" -> "\o303\o231"
| "&Gt;" -> "\o342\o211\o253"
| "&#x0226B;" -> "\o342\o211\o253"
| "&#8811;" -> "\o342\o211\o253"
| "&twixt;" -> "\o342\o211\o254"
| "&#x0226C;" -> "\o342\o211\o254"
| "&#8812;" -> "\o342\o211\o254"
| "&#x0226D;" -> "\o342\o211\o255"
| "&#8813;" -> "\o342\o211\o255"
| "&nlt;" -> "\o342\o211\o256"
| "&#x0226E;" -> "\o342\o211\o256"
| "&#8814;" -> "\o342\o211\o256"
| "&Uacute;" -> "\o303\o232"
| "&#x000DA;" -> "\o303\o232"
| "&#218;" -> "\o303\o232"
| "&ngt;" -> "\o342\o211\o257"
| "&#x0226F;" -> "\o342\o211\o257"
| "&#8815;" -> "\o342\o211\o257"
| "&#x02270;" -> "\o342\o211\o260"
| "&#8816;" -> "\o342\o211\o260"
| "&Ucirc;" -> "\o303\o233"
| "&#x000DB;" -> "\o303\o233"
| "&#219;" -> "\o303\o233"
| "&#x02271;" -> "\o342\o211\o261"
| "&#8817;" -> "\o342\o211\o261"
| "&lsim;" -> "\o342\o211\o262"
| "&#x02272;" -> "\o342\o211\o262"
| "&#8818;" -> "\o342\o211\o262"
| "&gsim;" -> "\o342\o211\o263"
| "&#x02273;" -> "\o342\o211\o263"
| "&#8819;" -> "\o342\o211\o263"
| "&Uuml;" -> "\o303\o234"
| "&#x000DC;" -> "\o303\o234"
| "&#220;" -> "\o303\o234"
| "&#x02274;" -> "\o342\o211\o264"
| "&#8820;" -> "\o342\o211\o264"
| "&#x02275;" -> "\o342\o211\o265"
| "&#8821;" -> "\o342\o211\o265"
| "&lg;" -> "\o342\o211\o266"
| "&#x02276;" -> "\o342\o211\o266"
| "&#8822;" -> "\o342\o211\o266"
| "&gl;" -> "\o342\o211\o267"
| "&#x02277;" -> "\o342\o211\o267"
| "&#8823;" -> "\o342\o211\o267"
| "&Yacute;" -> "\o303\o235"
| "&#x000DD;" -> "\o303\o235"
| "&#221;" -> "\o303\o235"
| "&#x02278;" -> "\o342\o211\o270"
| "&#8824;" -> "\o342\o211\o270"
| "&#x02279;" -> "\o342\o211\o271"
| "&#8825;" -> "\o342\o211\o271"
| "&pr;" -> "\o342\o211\o272"
| "&#x0227A;" -> "\o342\o211\o272"
| "&#8826;" -> "\o342\o211\o272"
| "&amp;" -> "&"
| "&#x00026;" -> "&"
| "&#38;" -> "&"
| "&THORN;" -> "\o303\o236"
| "&#x000DE;" -> "\o303\o236"
| "&#222;" -> "\o303\o236"
| "&sc;" -> "\o342\o211\o273"
| "&#x0227B;" -> "\o342\o211\o273"
| "&#8827;" -> "\o342\o211\o273"
| "&#x0227C;" -> "\o342\o211\o274"
| "&#8828;" -> "\o342\o211\o274"
| "&szlig;" -> "\o303\o237"
| "&#x000DF;" -> "\o303\o237"
| "&#223;" -> "\o303\o237"
| "&#x0227D;" -> "\o342\o211\o275"
| "&#8829;" -> "\o342\o211\o275"
| "&prsim;" -> "\o342\o211\o276"
| "&#x0227E;" -> "\o342\o211\o276"
| "&#8830;" -> "\o342\o211\o276"
| "&scsim;" -> "\o342\o211\o277"
| "&#x0227F;" -> "\o342\o211\o277"
| "&#8831;" -> "\o342\o211\o277"
| "&npr;" -> "\o342\o212\o200"
| "&#x02280;" -> "\o342\o212\o200"
| "&#8832;" -> "\o342\o212\o200"
| "&agrave;" -> "\o303\o240"
| "&#x000E0;" -> "\o303\o240"
| "&#224;" -> "\o303\o240"
| "&nsc;" -> "\o342\o212\o201"
| "&#x02281;" -> "\o342\o212\o201"
| "&#8833;" -> "\o342\o212\o201"
| "&sub;" -> "\o342\o212\o202"
| "&#x02282;" -> "\o342\o212\o202"
| "&#8834;" -> "\o342\o212\o202"
| "&sup;" -> "\o342\o212\o203"
| "&#x02283;" -> "\o342\o212\o203"
| "&#8835;" -> "\o342\o212\o203"
| "&nsub;" -> "\o342\o212\o204"
| "&#x02284;" -> "\o342\o212\o204"
| "&#8836;" -> "\o342\o212\o204"
| "&nsup;" -> "\o342\o212\o205"
| "&#x02285;" -> "\o342\o212\o205"
| "&#8837;" -> "\o342\o212\o205"
| "&aacute;" -> "\o303\o241"
| "&#x000E1;" -> "\o303\o241"
| "&#225;" -> "\o303\o241"
| "&sube;" -> "\o342\o212\o206"
| "&#x02286;" -> "\o342\o212\o206"
| "&#8838;" -> "\o342\o212\o206"
| "&supe;" -> "\o342\o212\o207"
| "&#x02287;" -> "\o342\o212\o207"
| "&#8839;" -> "\o342\o212\o207"
| "&#x02288;" -> "\o342\o212\o210"
| "&#8840;" -> "\o342\o212\o210"
| "&acirc;" -> "\o303\o242"
| "&#x000E2;" -> "\o303\o242"
| "&#226;" -> "\o303\o242"
| "&#x02289;" -> "\o342\o212\o211"
| "&#8841;" -> "\o342\o212\o211"
| "&#x0228A;" -> "\o342\o212\o212"
| "&#8842;" -> "\o342\o212\o212"
| "&#x0228B;" -> "\o342\o212\o213"
| "&#8843;" -> "\o342\o212\o213"
| "&atilde;" -> "\o303\o243"
| "&#x000E3;" -> "\o303\o243"
| "&#227;" -> "\o303\o243"
| "&#x0228D;" -> "\o342\o212\o215"
| "&#8845;" -> "\o342\o212\o215"
| "&#x0228E;" -> "\o342\o212\o216"
| "&#8846;" -> "\o342\o212\o216"
| "&#x0228F;" -> "\o342\o212\o217"
| "&#8847;" -> "\o342\o212\o217"
| "&auml;" -> "\o303\o244"
| "&#x000E4;" -> "\o303\o244"
| "&#228;" -> "\o303\o244"
| "&#x02290;" -> "\o342\o212\o220"
| "&#8848;" -> "\o342\o212\o220"
| "&#x02291;" -> "\o342\o212\o221"
| "&#8849;" -> "\o342\o212\o221"
| "&aring;" -> "\o303\o245"
| "&#x000E5;" -> "\o303\o245"
| "&#229;" -> "\o303\o245"
| "&#x02292;" -> "\o342\o212\o222"
| "&#8850;" -> "\o342\o212\o222"
| "&#x02293;" -> "\o342\o212\o223"
| "&#8851;" -> "\o342\o212\o223"
| "&#x02294;" -> "\o342\o212\o224"
| "&#8852;" -> "\o342\o212\o224"
| "&oplus;" -> "\o342\o212\o225"
| "&#x02295;" -> "\o342\o212\o225"
| "&#8853;" -> "\o342\o212\o225"
| "&aelig;" -> "\o303\o246"
| "&#x000E6;" -> "\o303\o246"
| "&#230;" -> "\o303\o246"
| "&#x02296;" -> "\o342\o212\o226"
| "&#8854;" -> "\o342\o212\o226"
| "&otimes;" -> "\o342\o212\o227"
| "&#x02297;" -> "\o342\o212\o227"
| "&#8855;" -> "\o342\o212\o227"
| "&#x02298;" -> "\o342\o212\o230"
| "&#8856;" -> "\o342\o212\o230"
| "&#x02299;" -> "\o342\o212\o231"
| "&#8857;" -> "\o342\o212\o231"
| "&ccedil;" -> "\o303\o247"
| "&#x000E7;" -> "\o303\o247"
| "&#231;" -> "\o303\o247"
| "&#x0229A;" -> "\o342\o212\o232"
| "&#8858;" -> "\o342\o212\o232"
| "&#x0229B;" -> "\o342\o212\o233"
| "&#8859;" -> "\o342\o212\o233"
| "&#x0229D;" -> "\o342\o212\o235"
| "&#8861;" -> "\o342\o212\o235"
| "&egrave;" -> "\o303\o250"
| "&#x000E8;" -> "\o303\o250"
| "&#232;" -> "\o303\o250"
| "&#x0229E;" -> "\o342\o212\o236"
| "&#8862;" -> "\o342\o212\o236"
| "&#x0229F;" -> "\o342\o212\o237"
| "&#8863;" -> "\o342\o212\o237"
| "&#x022A0;" -> "\o342\o212\o240"
| "&#8864;" -> "\o342\o212\o240"
| "&#x022A1;" -> "\o342\o212\o241"
| "&#8865;" -> "\o342\o212\o241"
| "&eacute;" -> "\o303\o251"
| "&#x000E9;" -> "\o303\o251"
| "&#233;" -> "\o303\o251"
| "&#x022A2;" -> "\o342\o212\o242"
| "&#8866;" -> "\o342\o212\o242"
| "&#x022A3;" -> "\o342\o212\o243"
| "&#8867;" -> "\o342\o212\o243"
| "&#x022A4;" -> "\o342\o212\o244"
| "&#8868;" -> "\o342\o212\o244"
| "&ecirc;" -> "\o303\o252"
| "&#x000EA;" -> "\o303\o252"
| "&#234;" -> "\o303\o252"
| "&perp;" -> "\o342\o212\o245"
| "&#x022A5;" -> "\o342\o212\o245"
| "&#8869;" -> "\o342\o212\o245"
| "&#x022A7;" -> "\o342\o212\o247"
| "&#8871;" -> "\o342\o212\o247"
| "&#x022A8;" -> "\o342\o212\o250"
| "&#8872;" -> "\o342\o212\o250"
| "&#x022A9;" -> "\o342\o212\o251"
| "&#8873;" -> "\o342\o212\o251"
| "&euml;" -> "\o303\o253"
| "&#x000EB;" -> "\o303\o253"
| "&#235;" -> "\o303\o253"
| "&#x022AA;" -> "\o342\o212\o252"
| "&#8874;" -> "\o342\o212\o252"
| "&#x022AB;" -> "\o342\o212\o253"
| "&#8875;" -> "\o342\o212\o253"
| "&#x022AC;" -> "\o342\o212\o254"
| "&#8876;" -> "\o342\o212\o254"
| "&#x022AD;" -> "\o342\o212\o255"
| "&#8877;" -> "\o342\o212\o255"
| "&#x022AE;" -> "\o342\o212\o256"
| "&#8878;" -> "\o342\o212\o256"
| "&igrave;" -> "\o303\o254"
| "&#x000EC;" -> "\o303\o254"
| "&#236;" -> "\o303\o254"
| "&#x022AF;" -> "\o342\o212\o257"
| "&#8879;" -> "\o342\o212\o257"
| "&#x022B0;" -> "\o342\o212\o260"
| "&#8880;" -> "\o342\o212\o260"
| "&#x022B2;" -> "\o342\o212\o262"
| "&#8882;" -> "\o342\o212\o262"
| "&iacute;" -> "\o303\o255"
| "&#x000ED;" -> "\o303\o255"
| "&#237;" -> "\o303\o255"
| "&#x022B3;" -> "\o342\o212\o263"
| "&#8883;" -> "\o342\o212\o263"
| "&#x022B4;" -> "\o342\o212\o264"
| "&#8884;" -> "\o342\o212\o264"
| "&#x022B5;" -> "\o342\o212\o265"
| "&#8885;" -> "\o342\o212\o265"
| "&icirc;" -> "\o303\o256"
| "&#x000EE;" -> "\o303\o256"
| "&#238;" -> "\o303\o256"
| "&#x022B6;" -> "\o342\o212\o266"
| "&#8886;" -> "\o342\o212\o266"
| "&#x022B7;" -> "\o342\o212\o267"
| "&#8887;" -> "\o342\o212\o267"
| "&#x022B8;" -> "\o342\o212\o270"
| "&#8888;" -> "\o342\o212\o270"
| "&#x022B9;" -> "\o342\o212\o271"
| "&#8889;" -> "\o342\o212\o271"
| "&iuml;" -> "\o303\o257"
| "&#x000EF;" -> "\o303\o257"
| "&#239;" -> "\o303\o257"
| "&#x022BA;" -> "\o342\o212\o272"
| "&#8890;" -> "\o342\o212\o272"
| "&#x022BB;" -> "\o342\o212\o273"
| "&#8891;" -> "\o342\o212\o273"
| "&#x022BD;" -> "\o342\o212\o275"
| "&#8893;" -> "\o342\o212\o275"
| "&#x022BE;" -> "\o342\o212\o276"
| "&#8894;" -> "\o342\o212\o276"
| "&#x022BF;" -> "\o342\o212\o277"
| "&#8895;" -> "\o342\o212\o277"
| "&eth;" -> "\o303\o260"
| "&#x000F0;" -> "\o303\o260"
| "&#240;" -> "\o303\o260"
| "&#x022C0;" -> "\o342\o213\o200"
| "&#8896;" -> "\o342\o213\o200"
| "&#x022C1;" -> "\o342\o213\o201"
| "&#8897;" -> "\o342\o213\o201"
| "&ntilde;" -> "\o303\o261"
| "&#x000F1;" -> "\o303\o261"
| "&#241;" -> "\o303\o261"
| "&#x022C2;" -> "\o342\o213\o202"
| "&#8898;" -> "\o342\o213\o202"
| "&#x022C3;" -> "\o342\o213\o203"
| "&#8899;" -> "\o342\o213\o203"
| "&#x022C4;" -> "\o342\o213\o204"
| "&#8900;" -> "\o342\o213\o204"
| "&apos;" -> "\'"
| "&#x00027;" -> "\'"
| "&#39;" -> "\'"
| "&ograve;" -> "\o303\o262"
| "&#x000F2;" -> "\o303\o262"
| "&#242;" -> "\o303\o262"
| "&sdot;" -> "\o342\o213\o205"
| "&#x022C5;" -> "\o342\o213\o205"
| "&#8901;" -> "\o342\o213\o205"
| "&#x022C6;" -> "\o342\o213\o206"
| "&#8902;" -> "\o342\o213\o206"
| "&#x022C7;" -> "\o342\o213\o207"
| "&#8903;" -> "\o342\o213\o207"
| "&#x022C8;" -> "\o342\o213\o210"
| "&#8904;" -> "\o342\o213\o210"
| "&oacute;" -> "\o303\o263"
| "&#x000F3;" -> "\o303\o263"
| "&#243;" -> "\o303\o263"
| "&#x022C9;" -> "\o342\o213\o211"
| "&#8905;" -> "\o342\o213\o211"
| "&#x022CA;" -> "\o342\o213\o212"
| "&#8906;" -> "\o342\o213\o212"
| "&#x022CB;" -> "\o342\o213\o213"
| "&#8907;" -> "\o342\o213\o213"
| "&#x022CC;" -> "\o342\o213\o214"
| "&#8908;" -> "\o342\o213\o214"
| "&ocirc;" -> "\o303\o264"
| "&#x000F4;" -> "\o303\o264"
| "&#244;" -> "\o303\o264"
| "&#x022CD;" -> "\o342\o213\o215"
| "&#8909;" -> "\o342\o213\o215"
| "&#x022CE;" -> "\o342\o213\o216"
| "&#8910;" -> "\o342\o213\o216"
| "&#x022CF;" -> "\o342\o213\o217"
| "&#8911;" -> "\o342\o213\o217"
| "&otilde;" -> "\o303\o265"
| "&#x000F5;" -> "\o303\o265"
| "&#245;" -> "\o303\o265"
| "&#x022D0;" -> "\o342\o213\o220"
| "&#8912;" -> "\o342\o213\o220"
| "&#x022D1;" -> "\o342\o213\o221"
| "&#8913;" -> "\o342\o213\o221"
| "&#x022D2;" -> "\o342\o213\o222"
| "&#8914;" -> "\o342\o213\o222"
| "&#x022D3;" -> "\o342\o213\o223"
| "&#8915;" -> "\o342\o213\o223"
| "&ouml;" -> "\o303\o266"
| "&#x000F6;" -> "\o303\o266"
| "&#246;" -> "\o303\o266"
| "&#x022D4;" -> "\o342\o213\o224"
| "&#8916;" -> "\o342\o213\o224"
| "&#x022D5;" -> "\o342\o213\o225"
| "&#8917;" -> "\o342\o213\o225"
| "&#x022D6;" -> "\o342\o213\o226"
| "&#8918;" -> "\o342\o213\o226"
| "&#x022D7;" -> "\o342\o213\o227"
| "&#8919;" -> "\o342\o213\o227"
| "&divide;" -> "\o303\o267"
| "&#x000F7;" -> "\o303\o267"
| "&#247;" -> "\o303\o267"
| "&#x022D8;" -> "\o342\o213\o230"
| "&#8920;" -> "\o342\o213\o230"
| "&#x022D9;" -> "\o342\o213\o231"
| "&#8921;" -> "\o342\o213\o231"
| "&#x022DA;" -> "\o342\o213\o232"
| "&#8922;" -> "\o342\o213\o232"
| "&#x022DB;" -> "\o342\o213\o233"
| "&#8923;" -> "\o342\o213\o233"
| "&#x022DE;" -> "\o342\o213\o236"
| "&#8926;" -> "\o342\o213\o236"
| "&#x022DF;" -> "\o342\o213\o237"
| "&#8927;" -> "\o342\o213\o237"
| "&oslash;" -> "\o303\o270"
| "&#x000F8;" -> "\o303\o270"
| "&#248;" -> "\o303\o270"
| "&#x022E0;" -> "\o342\o213\o240"
| "&#8928;" -> "\o342\o213\o240"
| "&#x022E1;" -> "\o342\o213\o241"
| "&#8929;" -> "\o342\o213\o241"
| "&#x022E2;" -> "\o342\o213\o242"
| "&#8930;" -> "\o342\o213\o242"
| "&#x022E3;" -> "\o342\o213\o243"
| "&#8931;" -> "\o342\o213\o243"
| "&ugrave;" -> "\o303\o271"
| "&#x000F9;" -> "\o303\o271"
| "&#249;" -> "\o303\o271"
| "&#x022E6;" -> "\o342\o213\o246"
| "&#8934;" -> "\o342\o213\o246"
| "&#x022E7;" -> "\o342\o213\o247"
| "&#8935;" -> "\o342\o213\o247"
| "&#x022E8;" -> "\o342\o213\o250"
| "&#8936;" -> "\o342\o213\o250"
| "&#x022E9;" -> "\o342\o213\o251"
| "&#8937;" -> "\o342\o213\o251"
| "&uacute;" -> "\o303\o272"
| "&#x000FA;" -> "\o303\o272"
| "&#250;" -> "\o303\o272"
| "&#x022EA;" -> "\o342\o213\o252"
| "&#8938;" -> "\o342\o213\o252"
| "&#x022EB;" -> "\o342\o213\o253"
| "&#8939;" -> "\o342\o213\o253"
| "&ucirc;" -> "\o303\o273"
| "&#x000FB;" -> "\o303\o273"
| "&#251;" -> "\o303\o273"
| "&#x022EC;" -> "\o342\o213\o254"
| "&#8940;" -> "\o342\o213\o254"
| "&#x022ED;" -> "\o342\o213\o255"
| "&#8941;" -> "\o342\o213\o255"
| "&#x022EE;" -> "\o342\o213\o256"
| "&#8942;" -> "\o342\o213\o256"
| "&#x022EF;" -> "\o342\o213\o257"
| "&#8943;" -> "\o342\o213\o257"
| "&uuml;" -> "\o303\o274"
| "&#x000FC;" -> "\o303\o274"
| "&#252;" -> "\o303\o274"
| "&#x022F0;" -> "\o342\o213\o260"
| "&#8944;" -> "\o342\o213\o260"
| "&#x022F1;" -> "\o342\o213\o261"
| "&#8945;" -> "\o342\o213\o261"
| "&#x022F2;" -> "\o342\o213\o262"
| "&#8946;" -> "\o342\o213\o262"
| "&#x022F3;" -> "\o342\o213\o263"
| "&#8947;" -> "\o342\o213\o263"
| "&#x022F4;" -> "\o342\o213\o264"
| "&#8948;" -> "\o342\o213\o264"
| "&yacute;" -> "\o303\o275"
| "&#x000FD;" -> "\o303\o275"
| "&#253;" -> "\o303\o275"
| "&#x022F5;" -> "\o342\o213\o265"
| "&#8949;" -> "\o342\o213\o265"
| "&#x022F6;" -> "\o342\o213\o266"
| "&#8950;" -> "\o342\o213\o266"
| "&#x022F7;" -> "\o342\o213\o267"
| "&#8951;" -> "\o342\o213\o267"
| "&#x022F9;" -> "\o342\o213\o271"
| "&#8953;" -> "\o342\o213\o271"
| "&#x022FA;" -> "\o342\o213\o272"
| "&#8954;" -> "\o342\o213\o272"
| "&thorn;" -> "\o303\o276"
| "&#x000FE;" -> "\o303\o276"
| "&#254;" -> "\o303\o276"
| "&#x022FB;" -> "\o342\o213\o273"
| "&#8955;" -> "\o342\o213\o273"
| "&#x022FC;" -> "\o342\o213\o274"
| "&#8956;" -> "\o342\o213\o274"
| "&#x022FD;" -> "\o342\o213\o275"
| "&#8957;" -> "\o342\o213\o275"
| "&#x022FE;" -> "\o342\o213\o276"
| "&#8958;" -> "\o342\o213\o276"
| "&yuml;" -> "\o303\o277"
| "&#x000FF;" -> "\o303\o277"
| "&#255;" -> "\o303\o277"
| "&#x02305;" -> "\o342\o214\o205"
| "&#8965;" -> "\o342\o214\o205"
| "&#x02306;" -> "\o342\o214\o206"
| "&#8966;" -> "\o342\o214\o206"
| "&lceil;" -> "\o342\o214\o210"
| "&#x02308;" -> "\o342\o214\o210"
| "&#8968;" -> "\o342\o214\o210"
| "&rceil;" -> "\o342\o214\o211"
| "&#x02309;" -> "\o342\o214\o211"
| "&#8969;" -> "\o342\o214\o211"
| "&lfloor;" -> "\o342\o214\o212"
| "&#x0230A;" -> "\o342\o214\o212"
| "&#8970;" -> "\o342\o214\o212"
| "&rfloor;" -> "\o342\o214\o213"
| "&#x0230B;" -> "\o342\o214\o213"
| "&#8971;" -> "\o342\o214\o213"
| "&#x0230C;" -> "\o342\o214\o214"
| "&#8972;" -> "\o342\o214\o214"
| "&#x0230D;" -> "\o342\o214\o215"
| "&#8973;" -> "\o342\o214\o215"
| "&#x0230E;" -> "\o342\o214\o216"
| "&#8974;" -> "\o342\o214\o216"
| "&#x0230F;" -> "\o342\o214\o217"
| "&#8975;" -> "\o342\o214\o217"
| "&#x00100;" -> "\o304\o200"
| "&#256;" -> "\o304\o200"
| "&#x02310;" -> "\o342\o214\o220"
| "&#8976;" -> "\o342\o214\o220"
| "&#x02312;" -> "\o342\o214\o222"
| "&#8978;" -> "\o342\o214\o222"
| "&#x02313;" -> "\o342\o214\o223"
| "&#8979;" -> "\o342\o214\o223"
| "&#x02315;" -> "\o342\o214\o225"
| "&#8981;" -> "\o342\o214\o225"
| "&#x02316;" -> "\o342\o214\o226"
| "&#8982;" -> "\o342\o214\o226"
| "&#x0231C;" -> "\o342\o214\o234"
| "&#8988;" -> "\o342\o214\o234"
| "&#x0231D;" -> "\o342\o214\o235"
| "&#8989;" -> "\o342\o214\o235"
| "&#x0231E;" -> "\o342\o214\o236"
| "&#8990;" -> "\o342\o214\o236"
| "&#x00101;" -> "\o304\o201"
| "&#257;" -> "\o304\o201"
| "&#x0231F;" -> "\o342\o214\o237"
| "&#8991;" -> "\o342\o214\o237"
| "&#x02322;" -> "\o342\o214\o242"
| "&#8994;" -> "\o342\o214\o242"
| "&#x02323;" -> "\o342\o214\o243"
| "&#8995;" -> "\o342\o214\o243"
| "&#x0232D;" -> "\o342\o214\o255"
| "&#9005;" -> "\o342\o214\o255"
| "&#x0232E;" -> "\o342\o214\o256"
| "&#9006;" -> "\o342\o214\o256"
| "&#x02336;" -> "\o342\o214\o266"
| "&#9014;" -> "\o342\o214\o266"
| "&#x0233D;" -> "\o342\o214\o275"
| "&#9021;" -> "\o342\o214\o275"
| "&#x0233F;" -> "\o342\o214\o277"
| "&#9023;" -> "\o342\o214\o277"
| "&#x0237C;" -> "\o342\o215\o274"
| "&#9084;" -> "\o342\o215\o274"
| "&#x00028;" -> "("
| "&#40;" -> "("
| "&#x00102;" -> "\o304\o202"
| "&#258;" -> "\o304\o202"
| "&#x023B0;" -> "\o342\o216\o260"
| "&#9136;" -> "\o342\o216\o260"
| "&#x023B1;" -> "\o342\o216\o261"
| "&#9137;" -> "\o342\o216\o261"
| "&#x023B4;" -> "\o342\o216\o264"
| "&#9140;" -> "\o342\o216\o264"
| "&#x023B5;" -> "\o342\o216\o265"
| "&#9141;" -> "\o342\o216\o265"
| "&#x023B6;" -> "\o342\o216\o266"
| "&#9142;" -> "\o342\o216\o266"
| "&#x023DC;" -> "\o342\o217\o234"
| "&#9180;" -> "\o342\o217\o234"
| "&#x023DD;" -> "\o342\o217\o235"
| "&#9181;" -> "\o342\o217\o235"
| "&#x023DE;" -> "\o342\o217\o236"
| "&#9182;" -> "\o342\o217\o236"
| "&#x00103;" -> "\o304\o203"
| "&#259;" -> "\o304\o203"
| "&#x023DF;" -> "\o342\o217\o237"
| "&#9183;" -> "\o342\o217\o237"
| "&#x023E2;" -> "\o342\o217\o242"
| "&#9186;" -> "\o342\o217\o242"
| "&#x023E7;" -> "\o342\o217\o247"
| "&#9191;" -> "\o342\o217\o247"
| "&#x02423;" -> "\o342\o220\o243"
| "&#9251;" -> "\o342\o220\o243"
| "&#x024C8;" -> "\o342\o223\o210"
| "&#9416;" -> "\o342\o223\o210"
| "&#x02500;" -> "\o342\o224\o200"
| "&#9472;" -> "\o342\o224\o200"
| "&#x02502;" -> "\o342\o224\o202"
| "&#9474;" -> "\o342\o224\o202"
| "&#x0250C;" -> "\o342\o224\o214"
| "&#9484;" -> "\o342\o224\o214"
| "&#x02510;" -> "\o342\o224\o220"
| "&#9488;" -> "\o342\o224\o220"
| "&#x00104;" -> "\o304\o204"
| "&#260;" -> "\o304\o204"
| "&#x02514;" -> "\o342\o224\o224"
| "&#9492;" -> "\o342\o224\o224"
| "&#x02518;" -> "\o342\o224\o230"
| "&#9496;" -> "\o342\o224\o230"
| "&#x0251C;" -> "\o342\o224\o234"
| "&#9500;" -> "\o342\o224\o234"
| "&#x02524;" -> "\o342\o224\o244"
| "&#9508;" -> "\o342\o224\o244"
| "&#x0252C;" -> "\o342\o224\o254"
| "&#9516;" -> "\o342\o224\o254"
| "&#x02534;" -> "\o342\o224\o264"
| "&#9524;" -> "\o342\o224\o264"
| "&#x0253C;" -> "\o342\o224\o274"
| "&#9532;" -> "\o342\o224\o274"
| "&#x02550;" -> "\o342\o225\o220"
| "&#9552;" -> "\o342\o225\o220"
| "&#x02551;" -> "\o342\o225\o221"
| "&#9553;" -> "\o342\o225\o221"
| "&#x02552;" -> "\o342\o225\o222"
| "&#9554;" -> "\o342\o225\o222"
| "&#x00105;" -> "\o304\o205"
| "&#261;" -> "\o304\o205"
| "&#x02553;" -> "\o342\o225\o223"
| "&#9555;" -> "\o342\o225\o223"
| "&#x02554;" -> "\o342\o225\o224"
| "&#9556;" -> "\o342\o225\o224"
| "&#x02555;" -> "\o342\o225\o225"
| "&#9557;" -> "\o342\o225\o225"
| "&#x02556;" -> "\o342\o225\o226"
| "&#9558;" -> "\o342\o225\o226"
| "&#x02557;" -> "\o342\o225\o227"
| "&#9559;" -> "\o342\o225\o227"
| "&#x02558;" -> "\o342\o225\o230"
| "&#9560;" -> "\o342\o225\o230"
| "&#x02559;" -> "\o342\o225\o231"
| "&#9561;" -> "\o342\o225\o231"
| "&#x0255A;" -> "\o342\o225\o232"
| "&#9562;" -> "\o342\o225\o232"
| "&#x0255B;" -> "\o342\o225\o233"
| "&#9563;" -> "\o342\o225\o233"
| "&#x0255C;" -> "\o342\o225\o234"
| "&#9564;" -> "\o342\o225\o234"
| "&#x00106;" -> "\o304\o206"
| "&#262;" -> "\o304\o206"
| "&#x0255D;" -> "\o342\o225\o235"
| "&#9565;" -> "\o342\o225\o235"
| "&#x0255E;" -> "\o342\o225\o236"
| "&#9566;" -> "\o342\o225\o236"
| "&#x0255F;" -> "\o342\o225\o237"
| "&#9567;" -> "\o342\o225\o237"
| "&#x02560;" -> "\o342\o225\o240"
| "&#9568;" -> "\o342\o225\o240"
| "&#x02561;" -> "\o342\o225\o241"
| "&#9569;" -> "\o342\o225\o241"
| "&#x02562;" -> "\o342\o225\o242"
| "&#9570;" -> "\o342\o225\o242"
| "&#x02563;" -> "\o342\o225\o243"
| "&#9571;" -> "\o342\o225\o243"
| "&#x02564;" -> "\o342\o225\o244"
| "&#9572;" -> "\o342\o225\o244"
| "&#x02565;" -> "\o342\o225\o245"
| "&#9573;" -> "\o342\o225\o245"
| "&#x02566;" -> "\o342\o225\o246"
| "&#9574;" -> "\o342\o225\o246"
| "&#x00009;" -> "\t"
| "&#9;" -> "\t"
| "&#x00107;" -> "\o304\o207"
| "&#263;" -> "\o304\o207"
| "&#x02567;" -> "\o342\o225\o247"
| "&#9575;" -> "\o342\o225\o247"
| "&#x02568;" -> "\o342\o225\o250"
| "&#9576;" -> "\o342\o225\o250"
| "&#x02569;" -> "\o342\o225\o251"
| "&#9577;" -> "\o342\o225\o251"
| "&#x0256A;" -> "\o342\o225\o252"
| "&#9578;" -> "\o342\o225\o252"
| "&#x0256B;" -> "\o342\o225\o253"
| "&#9579;" -> "\o342\o225\o253"
| "&#x0256C;" -> "\o342\o225\o254"
| "&#9580;" -> "\o342\o225\o254"
| "&#x02580;" -> "\o342\o226\o200"
| "&#9600;" -> "\o342\o226\o200"
| "&#x02584;" -> "\o342\o226\o204"
| "&#9604;" -> "\o342\o226\o204"
| "&#x02588;" -> "\o342\o226\o210"
| "&#9608;" -> "\o342\o226\o210"
| "&#x02591;" -> "\o342\o226\o221"
| "&#9617;" -> "\o342\o226\o221"
| "&#x00108;" -> "\o304\o210"
| "&#264;" -> "\o304\o210"
| "&#x02592;" -> "\o342\o226\o222"
| "&#9618;" -> "\o342\o226\o222"
| "&#x02593;" -> "\o342\o226\o223"
| "&#9619;" -> "\o342\o226\o223"
| "&#x025A1;" -> "\o342\o226\o241"
| "&#9633;" -> "\o342\o226\o241"
| "&#x025AA;" -> "\o342\o226\o252"
| "&#9642;" -> "\o342\o226\o252"
| "&#x025AB;" -> "\o342\o226\o253"
| "&#9643;" -> "\o342\o226\o253"
| "&#x025AD;" -> "\o342\o226\o255"
| "&#9645;" -> "\o342\o226\o255"
| "&#x025AE;" -> "\o342\o226\o256"
| "&#9646;" -> "\o342\o226\o256"
| "&#x00109;" -> "\o304\o211"
| "&#265;" -> "\o304\o211"
| "&#x025B1;" -> "\o342\o226\o261"
| "&#9649;" -> "\o342\o226\o261"
| "&#x025B3;" -> "\o342\o226\o263"
| "&#9651;" -> "\o342\o226\o263"
| "&#x025B4;" -> "\o342\o226\o264"
| "&#9652;" -> "\o342\o226\o264"
| "&#x025B5;" -> "\o342\o226\o265"
| "&#9653;" -> "\o342\o226\o265"
| "&#x025B8;" -> "\o342\o226\o270"
| "&#9656;" -> "\o342\o226\o270"
| "&#x025B9;" -> "\o342\o226\o271"
| "&#9657;" -> "\o342\o226\o271"
| "&#x025BD;" -> "\o342\o226\o275"
| "&#9661;" -> "\o342\o226\o275"
| "&#x0010A;" -> "\o304\o212"
| "&#266;" -> "\o304\o212"
| "&#x025BE;" -> "\o342\o226\o276"
| "&#9662;" -> "\o342\o226\o276"
| "&#x025BF;" -> "\o342\o226\o277"
| "&#9663;" -> "\o342\o226\o277"
| "&#x025C2;" -> "\o342\o227\o202"
| "&#9666;" -> "\o342\o227\o202"
| "&#x025C3;" -> "\o342\o227\o203"
| "&#9667;" -> "\o342\o227\o203"
| "&loz;" -> "\o342\o227\o212"
| "&#x025CA;" -> "\o342\o227\o212"
| "&#9674;" -> "\o342\o227\o212"
| "&#x025CB;" -> "\o342\o227\o213"
| "&#9675;" -> "\o342\o227\o213"
| "&#x025EC;" -> "\o342\o227\o254"
| "&#9708;" -> "\o342\o227\o254"
| "&#x025EF;" -> "\o342\o227\o257"
| "&#9711;" -> "\o342\o227\o257"
| "&#x0010B;" -> "\o304\o213"
| "&#267;" -> "\o304\o213"
| "&#x025F8;" -> "\o342\o227\o270"
| "&#9720;" -> "\o342\o227\o270"
| "&#x025F9;" -> "\o342\o227\o271"
| "&#9721;" -> "\o342\o227\o271"
| "&#x025FA;" -> "\o342\o227\o272"
| "&#9722;" -> "\o342\o227\o272"
| "&#x025FB;" -> "\o342\o227\o273"
| "&#9723;" -> "\o342\o227\o273"
| "&#x025FC;" -> "\o342\o227\o274"
| "&#9724;" -> "\o342\o227\o274"
| "&#x02605;" -> "\o342\o230\o205"
| "&#9733;" -> "\o342\o230\o205"
| "&#x02606;" -> "\o342\o230\o206"
| "&#9734;" -> "\o342\o230\o206"
| "&#x0260E;" -> "\o342\o230\o216"
| "&#9742;" -> "\o342\o230\o216"
| "&#x02640;" -> "\o342\o231\o200"
| "&#9792;" -> "\o342\o231\o200"
| "&#x00029;" -> ")"
| "&#41;" -> ")"
| "&#x0010C;" -> "\o304\o214"
| "&#268;" -> "\o304\o214"
| "&#x02642;" -> "\o342\o231\o202"
| "&#9794;" -> "\o342\o231\o202"
| "&spades;" -> "\o342\o231\o240"
| "&#x02660;" -> "\o342\o231\o240"
| "&#9824;" -> "\o342\o231\o240"
| "&clubs;" -> "\o342\o231\o243"
| "&#x02663;" -> "\o342\o231\o243"
| "&#9827;" -> "\o342\o231\o243"
| "&hearts;" -> "\o342\o231\o245"
| "&#x02665;" -> "\o342\o231\o245"
| "&#9829;" -> "\o342\o231\o245"
| "&diams;" -> "\o342\o231\o246"
| "&#x02666;" -> "\o342\o231\o246"
| "&#9830;" -> "\o342\o231\o246"
| "&#x0266A;" -> "\o342\o231\o252"
| "&#9834;" -> "\o342\o231\o252"
| "&#x0266D;" -> "\o342\o231\o255"
| "&#9837;" -> "\o342\o231\o255"
| "&#x0266E;" -> "\o342\o231\o256"
| "&#9838;" -> "\o342\o231\o256"
| "&#x0266F;" -> "\o342\o231\o257"
| "&#9839;" -> "\o342\o231\o257"
| "&#x0010D;" -> "\o304\o215"
| "&#269;" -> "\o304\o215"
| "&#x02713;" -> "\o342\o234\o223"
| "&#10003;" -> "\o342\o234\o223"
| "&#x02717;" -> "\o342\o234\o227"
| "&#10007;" -> "\o342\o234\o227"
| "&#x02720;" -> "\o342\o234\o240"
| "&#10016;" -> "\o342\o234\o240"
| "&#x02736;" -> "\o342\o234\o266"
| "&#10038;" -> "\o342\o234\o266"
| "&#x02758;" -> "\o342\o235\o230"
| "&#10072;" -> "\o342\o235\o230"
| "&#x02772;" -> "\o342\o235\o262"
| "&#10098;" -> "\o342\o235\o262"
| "&#x02773;" -> "\o342\o235\o263"
| "&#10099;" -> "\o342\o235\o263"
| "&#x027E6;" -> "\o342\o237\o246"
| "&#10214;" -> "\o342\o237\o246"
| "&#x027E7;" -> "\o342\o237\o247"
| "&#10215;" -> "\o342\o237\o247"
| "&#x0010E;" -> "\o304\o216"
| "&#270;" -> "\o304\o216"
| "&lang;" -> "\o342\o214\o251"
| "&#x027E8;" -> "\o342\o237\o250"
| "&#10216;" -> "\o342\o237\o250"
| "&rang;" -> "\o342\o214\o252"
| "&#x027E9;" -> "\o342\o237\o251"
| "&#10217;" -> "\o342\o237\o251"
| "&#x027EA;" -> "\o342\o237\o252"
| "&#10218;" -> "\o342\o237\o252"
| "&#x027EB;" -> "\o342\o237\o253"
| "&#10219;" -> "\o342\o237\o253"
| "&#x027EC;" -> "\o342\o237\o254"
| "&#10220;" -> "\o342\o237\o254"
| "&#x027ED;" -> "\o342\o237\o255"
| "&#10221;" -> "\o342\o237\o255"
| "&#x027F5;" -> "\o342\o237\o265"
| "&#10229;" -> "\o342\o237\o265"
| "&#x0010F;" -> "\o304\o217"
| "&#271;" -> "\o304\o217"
| "&#x027F6;" -> "\o342\o237\o266"
| "&#10230;" -> "\o342\o237\o266"
| "&#x027F7;" -> "\o342\o237\o267"
| "&#10231;" -> "\o342\o237\o267"
| "&#x027F8;" -> "\o342\o237\o270"
| "&#10232;" -> "\o342\o237\o270"
| "&#x027F9;" -> "\o342\o237\o271"
| "&#10233;" -> "\o342\o237\o271"
| "&#x027FA;" -> "\o342\o237\o272"
| "&#10234;" -> "\o342\o237\o272"
| "&#x00110;" -> "\o304\o220"
| "&#272;" -> "\o304\o220"
| "&#x027FC;" -> "\o342\o237\o274"
| "&#10236;" -> "\o342\o237\o274"
| "&#x027FF;" -> "\o342\o237\o277"
| "&#10239;" -> "\o342\o237\o277"
| "&#x02902;" -> "\o342\o244\o202"
| "&#10498;" -> "\o342\o244\o202"
| "&#x02903;" -> "\o342\o244\o203"
| "&#10499;" -> "\o342\o244\o203"
| "&#x02904;" -> "\o342\o244\o204"
| "&#10500;" -> "\o342\o244\o204"
| "&#x02905;" -> "\o342\o244\o205"
| "&#10501;" -> "\o342\o244\o205"
| "&#x0290C;" -> "\o342\o244\o214"
| "&#10508;" -> "\o342\o244\o214"
| "&#x0290D;" -> "\o342\o244\o215"
| "&#10509;" -> "\o342\o244\o215"
| "&#x0290E;" -> "\o342\o244\o216"
| "&#10510;" -> "\o342\o244\o216"
| "&#x00111;" -> "\o304\o221"
| "&#273;" -> "\o304\o221"
| "&#x0290F;" -> "\o342\o244\o217"
| "&#10511;" -> "\o342\o244\o217"
| "&#x02910;" -> "\o342\o244\o220"
| "&#10512;" -> "\o342\o244\o220"
| "&#x02911;" -> "\o342\o244\o221"
| "&#10513;" -> "\o342\o244\o221"
| "&#x02912;" -> "\o342\o244\o222"
| "&#10514;" -> "\o342\o244\o222"
| "&#x02913;" -> "\o342\o244\o223"
| "&#10515;" -> "\o342\o244\o223"
| "&#x02916;" -> "\o342\o244\o226"
| "&#10518;" -> "\o342\o244\o226"
| "&#x02919;" -> "\o342\o244\o231"
| "&#10521;" -> "\o342\o244\o231"
| "&#x0291A;" -> "\o342\o244\o232"
| "&#10522;" -> "\o342\o244\o232"
| "&#x0291B;" -> "\o342\o244\o233"
| "&#10523;" -> "\o342\o244\o233"
| "&#x00112;" -> "\o304\o222"
| "&#274;" -> "\o304\o222"
| "&#x0291C;" -> "\o342\o244\o234"
| "&#10524;" -> "\o342\o244\o234"
| "&#x0291D;" -> "\o342\o244\o235"
| "&#10525;" -> "\o342\o244\o235"
| "&#x0291E;" -> "\o342\o244\o236"
| "&#10526;" -> "\o342\o244\o236"
| "&#x0291F;" -> "\o342\o244\o237"
| "&#10527;" -> "\o342\o244\o237"
| "&#x02920;" -> "\o342\o244\o240"
| "&#10528;" -> "\o342\o244\o240"
| "&#x02923;" -> "\o342\o244\o243"
| "&#10531;" -> "\o342\o244\o243"
| "&#x02924;" -> "\o342\o244\o244"
| "&#10532;" -> "\o342\o244\o244"
| "&#x02925;" -> "\o342\o244\o245"
| "&#10533;" -> "\o342\o244\o245"
| "&#x02926;" -> "\o342\o244\o246"
| "&#10534;" -> "\o342\o244\o246"
| "&#x00113;" -> "\o304\o223"
| "&#275;" -> "\o304\o223"
| "&#x02927;" -> "\o342\o244\o247"
| "&#10535;" -> "\o342\o244\o247"
| "&#x02928;" -> "\o342\o244\o250"
| "&#10536;" -> "\o342\o244\o250"
| "&#x02929;" -> "\o342\o244\o251"
| "&#10537;" -> "\o342\o244\o251"
| "&#x0292A;" -> "\o342\o244\o252"
| "&#10538;" -> "\o342\o244\o252"
| "&#x02933;" -> "\o342\o244\o263"
| "&#10547;" -> "\o342\o244\o263"
| "&#x02935;" -> "\o342\o244\o265"
| "&#10549;" -> "\o342\o244\o265"
| "&#x02936;" -> "\o342\o244\o266"
| "&#10550;" -> "\o342\o244\o266"
| "&#x02937;" -> "\o342\o244\o267"
| "&#10551;" -> "\o342\o244\o267"
| "&#x02938;" -> "\o342\o244\o270"
| "&#10552;" -> "\o342\o244\o270"
| "&#x00116;" -> "\o304\o226"
| "&#278;" -> "\o304\o226"
| "&#x02939;" -> "\o342\o244\o271"
| "&#10553;" -> "\o342\o244\o271"
| "&#x0293C;" -> "\o342\o244\o274"
| "&#10556;" -> "\o342\o244\o274"
| "&#x0293D;" -> "\o342\o244\o275"
| "&#10557;" -> "\o342\o244\o275"
| "&#x02945;" -> "\o342\o245\o205"
| "&#10565;" -> "\o342\o245\o205"
| "&#x02948;" -> "\o342\o245\o210"
| "&#10568;" -> "\o342\o245\o210"
| "&#x02949;" -> "\o342\o245\o211"
| "&#10569;" -> "\o342\o245\o211"
| "&#x0294A;" -> "\o342\o245\o212"
| "&#10570;" -> "\o342\o245\o212"
| "&#x0294B;" -> "\o342\o245\o213"
| "&#10571;" -> "\o342\o245\o213"
| "&#x0294E;" -> "\o342\o245\o216"
| "&#10574;" -> "\o342\o245\o216"
| "&#x0294F;" -> "\o342\o245\o217"
| "&#10575;" -> "\o342\o245\o217"
| "&#x00117;" -> "\o304\o227"
| "&#279;" -> "\o304\o227"
| "&#x02950;" -> "\o342\o245\o220"
| "&#10576;" -> "\o342\o245\o220"
| "&#x02951;" -> "\o342\o245\o221"
| "&#10577;" -> "\o342\o245\o221"
| "&#x02952;" -> "\o342\o245\o222"
| "&#10578;" -> "\o342\o245\o222"
| "&#x02953;" -> "\o342\o245\o223"
| "&#10579;" -> "\o342\o245\o223"
| "&#x02954;" -> "\o342\o245\o224"
| "&#10580;" -> "\o342\o245\o224"
| "&#x02955;" -> "\o342\o245\o225"
| "&#10581;" -> "\o342\o245\o225"
| "&#x02956;" -> "\o342\o245\o226"
| "&#10582;" -> "\o342\o245\o226"
| "&#x02957;" -> "\o342\o245\o227"
| "&#10583;" -> "\o342\o245\o227"
| "&#x02958;" -> "\o342\o245\o230"
| "&#10584;" -> "\o342\o245\o230"
| "&#x02959;" -> "\o342\o245\o231"
| "&#10585;" -> "\o342\o245\o231"
| "&#x00118;" -> "\o304\o230"
| "&#280;" -> "\o304\o230"
| "&#x0295A;" -> "\o342\o245\o232"
| "&#10586;" -> "\o342\o245\o232"
| "&#x0295B;" -> "\o342\o245\o233"
| "&#10587;" -> "\o342\o245\o233"
| "&#x0295C;" -> "\o342\o245\o234"
| "&#10588;" -> "\o342\o245\o234"
| "&#x0295D;" -> "\o342\o245\o235"
| "&#10589;" -> "\o342\o245\o235"
| "&#x0295E;" -> "\o342\o245\o236"
| "&#10590;" -> "\o342\o245\o236"
| "&#x0295F;" -> "\o342\o245\o237"
| "&#10591;" -> "\o342\o245\o237"
| "&#x02960;" -> "\o342\o245\o240"
| "&#10592;" -> "\o342\o245\o240"
| "&#x02961;" -> "\o342\o245\o241"
| "&#10593;" -> "\o342\o245\o241"
| "&#x02962;" -> "\o342\o245\o242"
| "&#10594;" -> "\o342\o245\o242"
| "&#x02963;" -> "\o342\o245\o243"
| "&#10595;" -> "\o342\o245\o243"
| "&#x00119;" -> "\o304\o231"
| "&#281;" -> "\o304\o231"
| "&#x02964;" -> "\o342\o245\o244"
| "&#10596;" -> "\o342\o245\o244"
| "&#x02965;" -> "\o342\o245\o245"
| "&#10597;" -> "\o342\o245\o245"
| "&#x02966;" -> "\o342\o245\o246"
| "&#10598;" -> "\o342\o245\o246"
| "&#x02967;" -> "\o342\o245\o247"
| "&#10599;" -> "\o342\o245\o247"
| "&#x02968;" -> "\o342\o245\o250"
| "&#10600;" -> "\o342\o245\o250"
| "&#x02969;" -> "\o342\o245\o251"
| "&#10601;" -> "\o342\o245\o251"
| "&#x0296A;" -> "\o342\o245\o252"
| "&#10602;" -> "\o342\o245\o252"
| "&#x0296B;" -> "\o342\o245\o253"
| "&#10603;" -> "\o342\o245\o253"
| "&#x0296C;" -> "\o342\o245\o254"
| "&#10604;" -> "\o342\o245\o254"
| "&#x0296D;" -> "\o342\o245\o255"
| "&#10605;" -> "\o342\o245\o255"
| "&#x0011A;" -> "\o304\o232"
| "&#282;" -> "\o304\o232"
| "&#x0296E;" -> "\o342\o245\o256"
| "&#10606;" -> "\o342\o245\o256"
| "&#x0296F;" -> "\o342\o245\o257"
| "&#10607;" -> "\o342\o245\o257"
| "&#x02970;" -> "\o342\o245\o260"
| "&#10608;" -> "\o342\o245\o260"
| "&#x02971;" -> "\o342\o245\o261"
| "&#10609;" -> "\o342\o245\o261"
| "&#x02972;" -> "\o342\o245\o262"
| "&#10610;" -> "\o342\o245\o262"
| "&#x02973;" -> "\o342\o245\o263"
| "&#10611;" -> "\o342\o245\o263"
| "&#x02974;" -> "\o342\o245\o264"
| "&#10612;" -> "\o342\o245\o264"
| "&#x02975;" -> "\o342\o245\o265"
| "&#10613;" -> "\o342\o245\o265"
| "&#x02976;" -> "\o342\o245\o266"
| "&#10614;" -> "\o342\o245\o266"
| "&#x0011B;" -> "\o304\o233"
| "&#283;" -> "\o304\o233"
| "&#x02978;" -> "\o342\o245\o270"
| "&#10616;" -> "\o342\o245\o270"
| "&#x02979;" -> "\o342\o245\o271"
| "&#10617;" -> "\o342\o245\o271"
| "&#x0297B;" -> "\o342\o245\o273"
| "&#10619;" -> "\o342\o245\o273"
| "&#x0297C;" -> "\o342\o245\o274"
| "&#10620;" -> "\o342\o245\o274"
| "&#x0297D;" -> "\o342\o245\o275"
| "&#10621;" -> "\o342\o245\o275"
| "&#x0297E;" -> "\o342\o245\o276"
| "&#10622;" -> "\o342\o245\o276"
| "&#x0297F;" -> "\o342\o245\o277"
| "&#10623;" -> "\o342\o245\o277"
| "&#x02985;" -> "\o342\o246\o205"
| "&#10629;" -> "\o342\o246\o205"
| "&#x02986;" -> "\o342\o246\o206"
| "&#10630;" -> "\o342\o246\o206"
| "&#x0298B;" -> "\o342\o246\o213"
| "&#10635;" -> "\o342\o246\o213"
| "&#x0011C;" -> "\o304\o234"
| "&#284;" -> "\o304\o234"
| "&#x0298C;" -> "\o342\o246\o214"
| "&#10636;" -> "\o342\o246\o214"
| "&#x0298D;" -> "\o342\o246\o215"
| "&#10637;" -> "\o342\o246\o215"
| "&#x0298E;" -> "\o342\o246\o216"
| "&#10638;" -> "\o342\o246\o216"
| "&#x0298F;" -> "\o342\o246\o217"
| "&#10639;" -> "\o342\o246\o217"
| "&#x02990;" -> "\o342\o246\o220"
| "&#10640;" -> "\o342\o246\o220"
| "&#x02991;" -> "\o342\o246\o221"
| "&#10641;" -> "\o342\o246\o221"
| "&#x02992;" -> "\o342\o246\o222"
| "&#10642;" -> "\o342\o246\o222"
| "&#x02993;" -> "\o342\o246\o223"
| "&#10643;" -> "\o342\o246\o223"
| "&#x02994;" -> "\o342\o246\o224"
| "&#10644;" -> "\o342\o246\o224"
| "&#x02995;" -> "\o342\o246\o225"
| "&#10645;" -> "\o342\o246\o225"
| "&#x0002A;" -> "*"
| "&#42;" -> "*"
| "&#x0011D;" -> "\o304\o235"
| "&#285;" -> "\o304\o235"
| "&#x02996;" -> "\o342\o246\o226"
| "&#10646;" -> "\o342\o246\o226"
| "&#x0299A;" -> "\o342\o246\o232"
| "&#10650;" -> "\o342\o246\o232"
| "&#x0299C;" -> "\o342\o246\o234"
| "&#10652;" -> "\o342\o246\o234"
| "&#x0299D;" -> "\o342\o246\o235"
| "&#10653;" -> "\o342\o246\o235"
| "&#x029A4;" -> "\o342\o246\o244"
| "&#10660;" -> "\o342\o246\o244"
| "&#x029A5;" -> "\o342\o246\o245"
| "&#10661;" -> "\o342\o246\o245"
| "&#x029A6;" -> "\o342\o246\o246"
| "&#10662;" -> "\o342\o246\o246"
| "&#x029A7;" -> "\o342\o246\o247"
| "&#10663;" -> "\o342\o246\o247"
| "&#x029A8;" -> "\o342\o246\o250"
| "&#10664;" -> "\o342\o246\o250"
| "&#x029A9;" -> "\o342\o246\o251"
| "&#10665;" -> "\o342\o246\o251"
| "&#x0011E;" -> "\o304\o236"
| "&#286;" -> "\o304\o236"
| "&#x029AA;" -> "\o342\o246\o252"
| "&#10666;" -> "\o342\o246\o252"
| "&#x029AB;" -> "\o342\o246\o253"
| "&#10667;" -> "\o342\o246\o253"
| "&#x029AC;" -> "\o342\o246\o254"
| "&#10668;" -> "\o342\o246\o254"
| "&#x029AD;" -> "\o342\o246\o255"
| "&#10669;" -> "\o342\o246\o255"
| "&#x029AE;" -> "\o342\o246\o256"
| "&#10670;" -> "\o342\o246\o256"
| "&#x029AF;" -> "\o342\o246\o257"
| "&#10671;" -> "\o342\o246\o257"
| "&#x029B0;" -> "\o342\o246\o260"
| "&#10672;" -> "\o342\o246\o260"
| "&#x029B1;" -> "\o342\o246\o261"
| "&#10673;" -> "\o342\o246\o261"
| "&#x029B2;" -> "\o342\o246\o262"
| "&#10674;" -> "\o342\o246\o262"
| "&#x029B3;" -> "\o342\o246\o263"
| "&#10675;" -> "\o342\o246\o263"
| "&#x0011F;" -> "\o304\o237"
| "&#287;" -> "\o304\o237"
| "&#x029B4;" -> "\o342\o246\o264"
| "&#10676;" -> "\o342\o246\o264"
| "&#x029B5;" -> "\o342\o246\o265"
| "&#10677;" -> "\o342\o246\o265"
| "&#x029B6;" -> "\o342\o246\o266"
| "&#10678;" -> "\o342\o246\o266"
| "&#x029B7;" -> "\o342\o246\o267"
| "&#10679;" -> "\o342\o246\o267"
| "&#x029B9;" -> "\o342\o246\o271"
| "&#10681;" -> "\o342\o246\o271"
| "&#x029BB;" -> "\o342\o246\o273"
| "&#10683;" -> "\o342\o246\o273"
| "&#x029BC;" -> "\o342\o246\o274"
| "&#10684;" -> "\o342\o246\o274"
| "&#x029BE;" -> "\o342\o246\o276"
| "&#10686;" -> "\o342\o246\o276"
| "&#x029BF;" -> "\o342\o246\o277"
| "&#10687;" -> "\o342\o246\o277"
| "&#x029C0;" -> "\o342\o247\o200"
| "&#10688;" -> "\o342\o247\o200"
| "&#x00120;" -> "\o304\o240"
| "&#288;" -> "\o304\o240"
| "&#x029C1;" -> "\o342\o247\o201"
| "&#10689;" -> "\o342\o247\o201"
| "&#x029C2;" -> "\o342\o247\o202"
| "&#10690;" -> "\o342\o247\o202"
| "&#x029C3;" -> "\o342\o247\o203"
| "&#10691;" -> "\o342\o247\o203"
| "&#x029C4;" -> "\o342\o247\o204"
| "&#10692;" -> "\o342\o247\o204"
| "&#x029C5;" -> "\o342\o247\o205"
| "&#10693;" -> "\o342\o247\o205"
| "&#x029C9;" -> "\o342\o247\o211"
| "&#10697;" -> "\o342\o247\o211"
| "&#x029CD;" -> "\o342\o247\o215"
| "&#10701;" -> "\o342\o247\o215"
| "&#x029CE;" -> "\o342\o247\o216"
| "&#10702;" -> "\o342\o247\o216"
| "&#x029CF;" -> "\o342\o247\o217"
| "&#10703;" -> "\o342\o247\o217"
| "&#x029D0;" -> "\o342\o247\o220"
| "&#10704;" -> "\o342\o247\o220"
| "&#x00121;" -> "\o304\o241"
| "&#289;" -> "\o304\o241"
| "&#x029DA;" -> "\o342\o247\o232"
| "&#10714;" -> "\o342\o247\o232"
| "&#x029DC;" -> "\o342\o247\o234"
| "&#10716;" -> "\o342\o247\o234"
| "&#x029DD;" -> "\o342\o247\o235"
| "&#10717;" -> "\o342\o247\o235"
| "&#x029DE;" -> "\o342\o247\o236"
| "&#10718;" -> "\o342\o247\o236"
| "&#x029E3;" -> "\o342\o247\o243"
| "&#10723;" -> "\o342\o247\o243"
| "&#x029E4;" -> "\o342\o247\o244"
| "&#10724;" -> "\o342\o247\o244"
| "&#x029E5;" -> "\o342\o247\o245"
| "&#10725;" -> "\o342\o247\o245"
| "&#x029EB;" -> "\o342\o247\o253"
| "&#10731;" -> "\o342\o247\o253"
| "&#x029F4;" -> "\o342\o247\o264"
| "&#10740;" -> "\o342\o247\o264"
| "&#x00122;" -> "\o304\o242"
| "&#290;" -> "\o304\o242"
| "&#x029F6;" -> "\o342\o247\o266"
| "&#10742;" -> "\o342\o247\o266"
| "&#x02A00;" -> "\o342\o250\o200"
| "&#10752;" -> "\o342\o250\o200"
| "&#x02A01;" -> "\o342\o250\o201"
| "&#10753;" -> "\o342\o250\o201"
| "&#x02A02;" -> "\o342\o250\o202"
| "&#10754;" -> "\o342\o250\o202"
| "&#x02A04;" -> "\o342\o250\o204"
| "&#10756;" -> "\o342\o250\o204"
| "&#x02A06;" -> "\o342\o250\o206"
| "&#10758;" -> "\o342\o250\o206"
| "&#x02A0C;" -> "\o342\o250\o214"
| "&#10764;" -> "\o342\o250\o214"
| "&#x00124;" -> "\o304\o244"
| "&#292;" -> "\o304\o244"
| "&#x02A0D;" -> "\o342\o250\o215"
| "&#10765;" -> "\o342\o250\o215"
| "&#x02A10;" -> "\o342\o250\o220"
| "&#10768;" -> "\o342\o250\o220"
| "&#x02A11;" -> "\o342\o250\o221"
| "&#10769;" -> "\o342\o250\o221"
| "&#x02A12;" -> "\o342\o250\o222"
| "&#10770;" -> "\o342\o250\o222"
| "&#x02A13;" -> "\o342\o250\o223"
| "&#10771;" -> "\o342\o250\o223"
| "&#x02A14;" -> "\o342\o250\o224"
| "&#10772;" -> "\o342\o250\o224"
| "&#x02A15;" -> "\o342\o250\o225"
| "&#10773;" -> "\o342\o250\o225"
| "&#x02A16;" -> "\o342\o250\o226"
| "&#10774;" -> "\o342\o250\o226"
| "&#x02A17;" -> "\o342\o250\o227"
| "&#10775;" -> "\o342\o250\o227"
| "&#x02A22;" -> "\o342\o250\o242"
| "&#10786;" -> "\o342\o250\o242"
| "&#x00125;" -> "\o304\o245"
| "&#293;" -> "\o304\o245"
| "&#x02A23;" -> "\o342\o250\o243"
| "&#10787;" -> "\o342\o250\o243"
| "&#x02A24;" -> "\o342\o250\o244"
| "&#10788;" -> "\o342\o250\o244"
| "&#x02A25;" -> "\o342\o250\o245"
| "&#10789;" -> "\o342\o250\o245"
| "&#x02A26;" -> "\o342\o250\o246"
| "&#10790;" -> "\o342\o250\o246"
| "&#x02A27;" -> "\o342\o250\o247"
| "&#10791;" -> "\o342\o250\o247"
| "&#x02A29;" -> "\o342\o250\o251"
| "&#10793;" -> "\o342\o250\o251"
| "&#x02A2A;" -> "\o342\o250\o252"
| "&#10794;" -> "\o342\o250\o252"
| "&#x02A2D;" -> "\o342\o250\o255"
| "&#10797;" -> "\o342\o250\o255"
| "&#x02A2E;" -> "\o342\o250\o256"
| "&#10798;" -> "\o342\o250\o256"
| "&#x02A2F;" -> "\o342\o250\o257"
| "&#10799;" -> "\o342\o250\o257"
| "&#x00126;" -> "\o304\o246"
| "&#294;" -> "\o304\o246"
| "&#x02A30;" -> "\o342\o250\o260"
| "&#10800;" -> "\o342\o250\o260"
| "&#x02A31;" -> "\o342\o250\o261"
| "&#10801;" -> "\o342\o250\o261"
| "&#x02A33;" -> "\o342\o250\o263"
| "&#10803;" -> "\o342\o250\o263"
| "&#x02A34;" -> "\o342\o250\o264"
| "&#10804;" -> "\o342\o250\o264"
| "&#x02A35;" -> "\o342\o250\o265"
| "&#10805;" -> "\o342\o250\o265"
| "&#x02A36;" -> "\o342\o250\o266"
| "&#10806;" -> "\o342\o250\o266"
| "&#x02A37;" -> "\o342\o250\o267"
| "&#10807;" -> "\o342\o250\o267"
| "&#x02A38;" -> "\o342\o250\o270"
| "&#10808;" -> "\o342\o250\o270"
| "&#x02A39;" -> "\o342\o250\o271"
| "&#10809;" -> "\o342\o250\o271"
| "&#x02A3A;" -> "\o342\o250\o272"
| "&#10810;" -> "\o342\o250\o272"
| "&#x00127;" -> "\o304\o247"
| "&#295;" -> "\o304\o247"
| "&#x02A3B;" -> "\o342\o250\o273"
| "&#10811;" -> "\o342\o250\o273"
| "&#x02A3C;" -> "\o342\o250\o274"
| "&#10812;" -> "\o342\o250\o274"
| "&#x02A3F;" -> "\o342\o250\o277"
| "&#10815;" -> "\o342\o250\o277"
| "&#x02A40;" -> "\o342\o251\o200"
| "&#10816;" -> "\o342\o251\o200"
| "&#x02A42;" -> "\o342\o251\o202"
| "&#10818;" -> "\o342\o251\o202"
| "&#x02A43;" -> "\o342\o251\o203"
| "&#10819;" -> "\o342\o251\o203"
| "&#x02A44;" -> "\o342\o251\o204"
| "&#10820;" -> "\o342\o251\o204"
| "&#x02A45;" -> "\o342\o251\o205"
| "&#10821;" -> "\o342\o251\o205"
| "&#x02A46;" -> "\o342\o251\o206"
| "&#10822;" -> "\o342\o251\o206"
| "&#x02A47;" -> "\o342\o251\o207"
| "&#10823;" -> "\o342\o251\o207"
| "&#x0002B;" -> "+"
| "&#43;" -> "+"
| "&#x00128;" -> "\o304\o250"
| "&#296;" -> "\o304\o250"
| "&#x02A48;" -> "\o342\o251\o210"
| "&#10824;" -> "\o342\o251\o210"
| "&#x02A49;" -> "\o342\o251\o211"
| "&#10825;" -> "\o342\o251\o211"
| "&#x02A4A;" -> "\o342\o251\o212"
| "&#10826;" -> "\o342\o251\o212"
| "&#x02A4B;" -> "\o342\o251\o213"
| "&#10827;" -> "\o342\o251\o213"
| "&#x02A4C;" -> "\o342\o251\o214"
| "&#10828;" -> "\o342\o251\o214"
| "&#x02A4D;" -> "\o342\o251\o215"
| "&#10829;" -> "\o342\o251\o215"
| "&#x02A50;" -> "\o342\o251\o220"
| "&#10832;" -> "\o342\o251\o220"
| "&#x02A53;" -> "\o342\o251\o223"
| "&#10835;" -> "\o342\o251\o223"
| "&#x02A54;" -> "\o342\o251\o224"
| "&#10836;" -> "\o342\o251\o224"
| "&#x02A55;" -> "\o342\o251\o225"
| "&#10837;" -> "\o342\o251\o225"
| "&#x00129;" -> "\o304\o251"
| "&#297;" -> "\o304\o251"
| "&#x02A56;" -> "\o342\o251\o226"
| "&#10838;" -> "\o342\o251\o226"
| "&#x02A57;" -> "\o342\o251\o227"
| "&#10839;" -> "\o342\o251\o227"
| "&#x02A58;" -> "\o342\o251\o230"
| "&#10840;" -> "\o342\o251\o230"
| "&#x02A5A;" -> "\o342\o251\o232"
| "&#10842;" -> "\o342\o251\o232"
| "&#x02A5B;" -> "\o342\o251\o233"
| "&#10843;" -> "\o342\o251\o233"
| "&#x02A5C;" -> "\o342\o251\o234"
| "&#10844;" -> "\o342\o251\o234"
| "&#x02A5D;" -> "\o342\o251\o235"
| "&#10845;" -> "\o342\o251\o235"
| "&#x02A5F;" -> "\o342\o251\o237"
| "&#10847;" -> "\o342\o251\o237"
| "&#x02A66;" -> "\o342\o251\o246"
| "&#10854;" -> "\o342\o251\o246"
| "&#x02A6A;" -> "\o342\o251\o252"
| "&#10858;" -> "\o342\o251\o252"
| "&#x0012A;" -> "\o304\o252"
| "&#298;" -> "\o304\o252"
| "&#x02A6D;" -> "\o342\o251\o255"
| "&#10861;" -> "\o342\o251\o255"
| "&#x02A6E;" -> "\o342\o251\o256"
| "&#10862;" -> "\o342\o251\o256"
| "&#x02A6F;" -> "\o342\o251\o257"
| "&#10863;" -> "\o342\o251\o257"
| "&#x02A70;" -> "\o342\o251\o260"
| "&#10864;" -> "\o342\o251\o260"
| "&#x02A71;" -> "\o342\o251\o261"
| "&#10865;" -> "\o342\o251\o261"
| "&#x02A72;" -> "\o342\o251\o262"
| "&#10866;" -> "\o342\o251\o262"
| "&#x02A73;" -> "\o342\o251\o263"
| "&#10867;" -> "\o342\o251\o263"
| "&#x02A74;" -> "\o342\o251\o264"
| "&#10868;" -> "\o342\o251\o264"
| "&#x02A75;" -> "\o342\o251\o265"
| "&#10869;" -> "\o342\o251\o265"
| "&#x0012B;" -> "\o304\o253"
| "&#299;" -> "\o304\o253"
| "&#x02A77;" -> "\o342\o251\o267"
| "&#10871;" -> "\o342\o251\o267"
| "&#x02A78;" -> "\o342\o251\o270"
| "&#10872;" -> "\o342\o251\o270"
| "&#x02A79;" -> "\o342\o251\o271"
| "&#10873;" -> "\o342\o251\o271"
| "&#x02A7A;" -> "\o342\o251\o272"
| "&#10874;" -> "\o342\o251\o272"
| "&#x02A7B;" -> "\o342\o251\o273"
| "&#10875;" -> "\o342\o251\o273"
| "&#x02A7C;" -> "\o342\o251\o274"
| "&#10876;" -> "\o342\o251\o274"
| "&#x02A7D;" -> "\o342\o251\o275"
| "&#10877;" -> "\o342\o251\o275"
| "&#x02A7E;" -> "\o342\o251\o276"
| "&#10878;" -> "\o342\o251\o276"
| "&#x0012E;" -> "\o304\o256"
| "&#302;" -> "\o304\o256"
| "&#x02A7F;" -> "\o342\o251\o277"
| "&#10879;" -> "\o342\o251\o277"
| "&#x02A80;" -> "\o342\o252\o200"
| "&#10880;" -> "\o342\o252\o200"
| "&#x02A81;" -> "\o342\o252\o201"
| "&#10881;" -> "\o342\o252\o201"
| "&#x02A82;" -> "\o342\o252\o202"
| "&#10882;" -> "\o342\o252\o202"
| "&#x02A83;" -> "\o342\o252\o203"
| "&#10883;" -> "\o342\o252\o203"
| "&#x02A84;" -> "\o342\o252\o204"
| "&#10884;" -> "\o342\o252\o204"
| "&#x02A85;" -> "\o342\o252\o205"
| "&#10885;" -> "\o342\o252\o205"
| "&#x02A86;" -> "\o342\o252\o206"
| "&#10886;" -> "\o342\o252\o206"
| "&#x02A87;" -> "\o342\o252\o207"
| "&#10887;" -> "\o342\o252\o207"
| "&#x0012F;" -> "\o304\o257"
| "&#303;" -> "\o304\o257"
| "&#x02A88;" -> "\o342\o252\o210"
| "&#10888;" -> "\o342\o252\o210"
| "&#x02A89;" -> "\o342\o252\o211"
| "&#10889;" -> "\o342\o252\o211"
| "&#x02A8A;" -> "\o342\o252\o212"
| "&#10890;" -> "\o342\o252\o212"
| "&#x02A8B;" -> "\o342\o252\o213"
| "&#10891;" -> "\o342\o252\o213"
| "&#x02A8C;" -> "\o342\o252\o214"
| "&#10892;" -> "\o342\o252\o214"
| "&#x02A8D;" -> "\o342\o252\o215"
| "&#10893;" -> "\o342\o252\o215"
| "&#x02A8E;" -> "\o342\o252\o216"
| "&#10894;" -> "\o342\o252\o216"
| "&#x00130;" -> "\o304\o260"
| "&#304;" -> "\o304\o260"
| "&#x02A8F;" -> "\o342\o252\o217"
| "&#10895;" -> "\o342\o252\o217"
| "&#x02A90;" -> "\o342\o252\o220"
| "&#10896;" -> "\o342\o252\o220"
| "&#x02A91;" -> "\o342\o252\o221"
| "&#10897;" -> "\o342\o252\o221"
| "&#x02A92;" -> "\o342\o252\o222"
| "&#10898;" -> "\o342\o252\o222"
| "&#x02A93;" -> "\o342\o252\o223"
| "&#10899;" -> "\o342\o252\o223"
| "&#x02A94;" -> "\o342\o252\o224"
| "&#10900;" -> "\o342\o252\o224"
| "&#x02A95;" -> "\o342\o252\o225"
| "&#10901;" -> "\o342\o252\o225"
| "&#x02A96;" -> "\o342\o252\o226"
| "&#10902;" -> "\o342\o252\o226"
| "&#x02A97;" -> "\o342\o252\o227"
| "&#10903;" -> "\o342\o252\o227"
| "&#x02A98;" -> "\o342\o252\o230"
| "&#10904;" -> "\o342\o252\o230"
| "&#x02A99;" -> "\o342\o252\o231"
| "&#10905;" -> "\o342\o252\o231"
| "&#x02A9A;" -> "\o342\o252\o232"
| "&#10906;" -> "\o342\o252\o232"
| "&#x02A9D;" -> "\o342\o252\o235"
| "&#10909;" -> "\o342\o252\o235"
| "&#x02A9E;" -> "\o342\o252\o236"
| "&#10910;" -> "\o342\o252\o236"
| "&#x00131;" -> "\o304\o261"
| "&#305;" -> "\o304\o261"
| "&#x02A9F;" -> "\o342\o252\o237"
| "&#10911;" -> "\o342\o252\o237"
| "&#x02AA0;" -> "\o342\o252\o240"
| "&#10912;" -> "\o342\o252\o240"
| "&#x02AA1;" -> "\o342\o252\o241"
| "&#10913;" -> "\o342\o252\o241"
| "&#x02AA2;" -> "\o342\o252\o242"
| "&#10914;" -> "\o342\o252\o242"
| "&#x02AA4;" -> "\o342\o252\o244"
| "&#10916;" -> "\o342\o252\o244"
| "&#x02AA5;" -> "\o342\o252\o245"
| "&#10917;" -> "\o342\o252\o245"
| "&#x02AA6;" -> "\o342\o252\o246"
| "&#10918;" -> "\o342\o252\o246"
| "&#x02AA7;" -> "\o342\o252\o247"
| "&#10919;" -> "\o342\o252\o247"
| "&#x02AA8;" -> "\o342\o252\o250"
| "&#10920;" -> "\o342\o252\o250"
| "&#x02AA9;" -> "\o342\o252\o251"
| "&#10921;" -> "\o342\o252\o251"
| "&#x00132;" -> "\o304\o262"
| "&#306;" -> "\o304\o262"
| "&#x02AAA;" -> "\o342\o252\o252"
| "&#10922;" -> "\o342\o252\o252"
| "&#x02AAB;" -> "\o342\o252\o253"
| "&#10923;" -> "\o342\o252\o253"
| "&#x02AAC;" -> "\o342\o252\o254"
| "&#10924;" -> "\o342\o252\o254"
| "&#x02AAD;" -> "\o342\o252\o255"
| "&#10925;" -> "\o342\o252\o255"
| "&#x02AAE;" -> "\o342\o252\o256"
| "&#10926;" -> "\o342\o252\o256"
| "&#x02AAF;" -> "\o342\o252\o257"
| "&#10927;" -> "\o342\o252\o257"
| "&#x02AB0;" -> "\o342\o252\o260"
| "&#10928;" -> "\o342\o252\o260"
| "&#x02AB3;" -> "\o342\o252\o263"
| "&#10931;" -> "\o342\o252\o263"
| "&#x00133;" -> "\o304\o263"
| "&#307;" -> "\o304\o263"
| "&#x02AB4;" -> "\o342\o252\o264"
| "&#10932;" -> "\o342\o252\o264"
| "&#x02AB5;" -> "\o342\o252\o265"
| "&#10933;" -> "\o342\o252\o265"
| "&#x02AB6;" -> "\o342\o252\o266"
| "&#10934;" -> "\o342\o252\o266"
| "&#x02AB7;" -> "\o342\o252\o267"
| "&#10935;" -> "\o342\o252\o267"
| "&#x0002C;" -> ","
| "&#44;" -> ","
| "&#x02AB8;" -> "\o342\o252\o270"
| "&#10936;" -> "\o342\o252\o270"
| "&#x02AB9;" -> "\o342\o252\o271"
| "&#10937;" -> "\o342\o252\o271"
| "&#x02ABA;" -> "\o342\o252\o272"
| "&#10938;" -> "\o342\o252\o272"
| "&#x00134;" -> "\o304\o264"
| "&#308;" -> "\o304\o264"
| "&#x02ABB;" -> "\o342\o252\o273"
| "&#10939;" -> "\o342\o252\o273"
| "&#x02ABC;" -> "\o342\o252\o274"
| "&#10940;" -> "\o342\o252\o274"
| "&#x02ABD;" -> "\o342\o252\o275"
| "&#10941;" -> "\o342\o252\o275"
| "&#x02ABE;" -> "\o342\o252\o276"
| "&#10942;" -> "\o342\o252\o276"
| "&#x02ABF;" -> "\o342\o252\o277"
| "&#10943;" -> "\o342\o252\o277"
| "&#x02AC0;" -> "\o342\o253\o200"
| "&#10944;" -> "\o342\o253\o200"
| "&#x02AC1;" -> "\o342\o253\o201"
| "&#10945;" -> "\o342\o253\o201"
| "&#x02AC2;" -> "\o342\o253\o202"
| "&#10946;" -> "\o342\o253\o202"
| "&#x02AC3;" -> "\o342\o253\o203"
| "&#10947;" -> "\o342\o253\o203"
| "&#x02AC4;" -> "\o342\o253\o204"
| "&#10948;" -> "\o342\o253\o204"
| "&#x00135;" -> "\o304\o265"
| "&#309;" -> "\o304\o265"
| "&#x02AC5;" -> "\o342\o253\o205"
| "&#10949;" -> "\o342\o253\o205"
| "&#x02AC6;" -> "\o342\o253\o206"
| "&#10950;" -> "\o342\o253\o206"
| "&#x02AC7;" -> "\o342\o253\o207"
| "&#10951;" -> "\o342\o253\o207"
| "&#x02AC8;" -> "\o342\o253\o210"
| "&#10952;" -> "\o342\o253\o210"
| "&#x02ACB;" -> "\o342\o253\o213"
| "&#10955;" -> "\o342\o253\o213"
| "&#x02ACC;" -> "\o342\o253\o214"
| "&#10956;" -> "\o342\o253\o214"
| "&#x02ACF;" -> "\o342\o253\o217"
| "&#10959;" -> "\o342\o253\o217"
| "&#x02AD0;" -> "\o342\o253\o220"
| "&#10960;" -> "\o342\o253\o220"
| "&#x00136;" -> "\o304\o266"
| "&#310;" -> "\o304\o266"
| "&#x02AD1;" -> "\o342\o253\o221"
| "&#10961;" -> "\o342\o253\o221"
| "&#x02AD2;" -> "\o342\o253\o222"
| "&#10962;" -> "\o342\o253\o222"
| "&#x02AD3;" -> "\o342\o253\o223"
| "&#10963;" -> "\o342\o253\o223"
| "&#x02AD4;" -> "\o342\o253\o224"
| "&#10964;" -> "\o342\o253\o224"
| "&#x02AD5;" -> "\o342\o253\o225"
| "&#10965;" -> "\o342\o253\o225"
| "&#x02AD6;" -> "\o342\o253\o226"
| "&#10966;" -> "\o342\o253\o226"
| "&#x02AD7;" -> "\o342\o253\o227"
| "&#10967;" -> "\o342\o253\o227"
| "&#x02AD8;" -> "\o342\o253\o230"
| "&#10968;" -> "\o342\o253\o230"
| "&#x02AD9;" -> "\o342\o253\o231"
| "&#10969;" -> "\o342\o253\o231"
| "&#x02ADA;" -> "\o342\o253\o232"
| "&#10970;" -> "\o342\o253\o232"
| "&#x00137;" -> "\o304\o267"
| "&#311;" -> "\o304\o267"
| "&#x02ADB;" -> "\o342\o253\o233"
| "&#10971;" -> "\o342\o253\o233"
| "&#x02AE4;" -> "\o342\o253\o244"
| "&#10980;" -> "\o342\o253\o244"
| "&#x02AE6;" -> "\o342\o253\o246"
| "&#10982;" -> "\o342\o253\o246"
| "&#x02AE7;" -> "\o342\o253\o247"
| "&#10983;" -> "\o342\o253\o247"
| "&#x02AE8;" -> "\o342\o253\o250"
| "&#10984;" -> "\o342\o253\o250"
| "&#x02AE9;" -> "\o342\o253\o251"
| "&#10985;" -> "\o342\o253\o251"
| "&#x02AEB;" -> "\o342\o253\o253"
| "&#10987;" -> "\o342\o253\o253"
| "&#x02AEC;" -> "\o342\o253\o254"
| "&#10988;" -> "\o342\o253\o254"
| "&#x02AED;" -> "\o342\o253\o255"
| "&#10989;" -> "\o342\o253\o255"
| "&#x02AEE;" -> "\o342\o253\o256"
| "&#10990;" -> "\o342\o253\o256"
| "&#x00138;" -> "\o304\o270"
| "&#312;" -> "\o304\o270"
| "&#x02AEF;" -> "\o342\o253\o257"
| "&#10991;" -> "\o342\o253\o257"
| "&#x02AF0;" -> "\o342\o253\o260"
| "&#10992;" -> "\o342\o253\o260"
| "&#x02AF1;" -> "\o342\o253\o261"
| "&#10993;" -> "\o342\o253\o261"
| "&#x02AF2;" -> "\o342\o253\o262"
| "&#10994;" -> "\o342\o253\o262"
| "&#x02AF3;" -> "\o342\o253\o263"
| "&#10995;" -> "\o342\o253\o263"
| "&#x02AFD;" -> "\o342\o253\o275"
| "&#11005;" -> "\o342\o253\o275"
| "&#x0FB00;" -> "\o357\o254\o200"
| "&#64256;" -> "\o357\o254\o200"
| "&#x0FB01;" -> "\o357\o254\o201"
| "&#64257;" -> "\o357\o254\o201"
| "&#x0FB02;" -> "\o357\o254\o202"
| "&#64258;" -> "\o357\o254\o202"
| "&#x0FB03;" -> "\o357\o254\o203"
| "&#64259;" -> "\o357\o254\o203"
| "&#x00139;" -> "\o304\o271"
| "&#313;" -> "\o304\o271"
| "&#x0FB04;" -> "\o357\o254\o204"
| "&#64260;" -> "\o357\o254\o204"
| "&#x1D49C;" -> "\o360\o235\o222\o234"
| "&#119964;" -> "\o360\o235\o222\o234"
| "&#x1D49E;" -> "\o360\o235\o222\o236"
| "&#119966;" -> "\o360\o235\o222\o236"
| "&#x1D49F;" -> "\o360\o235\o222\o237"
| "&#119967;" -> "\o360\o235\o222\o237"
| "&#x1D4A2;" -> "\o360\o235\o222\o242"
| "&#119970;" -> "\o360\o235\o222\o242"
| "&#x1D4A5;" -> "\o360\o235\o222\o245"
| "&#119973;" -> "\o360\o235\o222\o245"
| "&#x1D4A6;" -> "\o360\o235\o222\o246"
| "&#119974;" -> "\o360\o235\o222\o246"
| "&#x1D4A9;" -> "\o360\o235\o222\o251"
| "&#119977;" -> "\o360\o235\o222\o251"
| "&#x1D4AA;" -> "\o360\o235\o222\o252"
| "&#119978;" -> "\o360\o235\o222\o252"
| "&#x1D4AB;" -> "\o360\o235\o222\o253"
| "&#119979;" -> "\o360\o235\o222\o253"
| "&#x0013A;" -> "\o304\o272"
| "&#314;" -> "\o304\o272"
| "&#x1D4AC;" -> "\o360\o235\o222\o254"
| "&#119980;" -> "\o360\o235\o222\o254"
| "&#x1D4AE;" -> "\o360\o235\o222\o256"
| "&#119982;" -> "\o360\o235\o222\o256"
| "&#x1D4AF;" -> "\o360\o235\o222\o257"
| "&#119983;" -> "\o360\o235\o222\o257"
| "&#x1D4B0;" -> "\o360\o235\o222\o260"
| "&#119984;" -> "\o360\o235\o222\o260"
| "&#x1D4B1;" -> "\o360\o235\o222\o261"
| "&#119985;" -> "\o360\o235\o222\o261"
| "&#x1D4B2;" -> "\o360\o235\o222\o262"
| "&#119986;" -> "\o360\o235\o222\o262"
| "&#x1D4B3;" -> "\o360\o235\o222\o263"
| "&#119987;" -> "\o360\o235\o222\o263"
| "&#x1D4B4;" -> "\o360\o235\o222\o264"
| "&#119988;" -> "\o360\o235\o222\o264"
| "&#x1D4B5;" -> "\o360\o235\o222\o265"
| "&#119989;" -> "\o360\o235\o222\o265"
| "&#x1D4B6;" -> "\o360\o235\o222\o266"
| "&#119990;" -> "\o360\o235\o222\o266"
| "&#x0013B;" -> "\o304\o273"
| "&#315;" -> "\o304\o273"
| "&#x1D4B7;" -> "\o360\o235\o222\o267"
| "&#119991;" -> "\o360\o235\o222\o267"
| "&#x1D4B8;" -> "\o360\o235\o222\o270"
| "&#119992;" -> "\o360\o235\o222\o270"
| "&#x1D4B9;" -> "\o360\o235\o222\o271"
| "&#119993;" -> "\o360\o235\o222\o271"
| "&#x1D4BB;" -> "\o360\o235\o222\o273"
| "&#119995;" -> "\o360\o235\o222\o273"
| "&#x1D4BD;" -> "\o360\o235\o222\o275"
| "&#119997;" -> "\o360\o235\o222\o275"
| "&#x1D4BE;" -> "\o360\o235\o222\o276"
| "&#119998;" -> "\o360\o235\o222\o276"
| "&#x1D4BF;" -> "\o360\o235\o222\o277"
| "&#119999;" -> "\o360\o235\o222\o277"
| "&#x1D4C0;" -> "\o360\o235\o223\o200"
| "&#120000;" -> "\o360\o235\o223\o200"
| "&#x1D4C1;" -> "\o360\o235\o223\o201"
| "&#120001;" -> "\o360\o235\o223\o201"
| "&#x1D4C2;" -> "\o360\o235\o223\o202"
| "&#120002;" -> "\o360\o235\o223\o202"
| "&#x0013C;" -> "\o304\o274"
| "&#316;" -> "\o304\o274"
| "&#x1D4C3;" -> "\o360\o235\o223\o203"
| "&#120003;" -> "\o360\o235\o223\o203"
| "&#x1D4C5;" -> "\o360\o235\o223\o205"
| "&#120005;" -> "\o360\o235\o223\o205"
| "&#x1D4C6;" -> "\o360\o235\o223\o206"
| "&#120006;" -> "\o360\o235\o223\o206"
| "&#x1D4C7;" -> "\o360\o235\o223\o207"
| "&#120007;" -> "\o360\o235\o223\o207"
| "&#x1D4C8;" -> "\o360\o235\o223\o210"
| "&#120008;" -> "\o360\o235\o223\o210"
| "&#x1D4C9;" -> "\o360\o235\o223\o211"
| "&#120009;" -> "\o360\o235\o223\o211"
| "&#x1D4CA;" -> "\o360\o235\o223\o212"
| "&#120010;" -> "\o360\o235\o223\o212"
| "&#x1D4CB;" -> "\o360\o235\o223\o213"
| "&#120011;" -> "\o360\o235\o223\o213"
| "&#x1D4CC;" -> "\o360\o235\o223\o214"
| "&#120012;" -> "\o360\o235\o223\o214"
| "&#x1D4CD;" -> "\o360\o235\o223\o215"
| "&#120013;" -> "\o360\o235\o223\o215"
| "&#x0013D;" -> "\o304\o275"
| "&#317;" -> "\o304\o275"
| "&#x1D4CE;" -> "\o360\o235\o223\o216"
| "&#120014;" -> "\o360\o235\o223\o216"
| "&#x1D4CF;" -> "\o360\o235\o223\o217"
| "&#120015;" -> "\o360\o235\o223\o217"
| "&#x1D504;" -> "\o360\o235\o224\o204"
| "&#120068;" -> "\o360\o235\o224\o204"
| "&#x1D505;" -> "\o360\o235\o224\o205"
| "&#120069;" -> "\o360\o235\o224\o205"
| "&#x1D507;" -> "\o360\o235\o224\o207"
| "&#120071;" -> "\o360\o235\o224\o207"
| "&#x0002E;" -> "."
| "&#46;" -> "."
| "&#x1D508;" -> "\o360\o235\o224\o210"
| "&#120072;" -> "\o360\o235\o224\o210"
| "&#x1D509;" -> "\o360\o235\o224\o211"
| "&#120073;" -> "\o360\o235\o224\o211"
| "&#x1D50A;" -> "\o360\o235\o224\o212"
| "&#120074;" -> "\o360\o235\o224\o212"
| "&#x1D50D;" -> "\o360\o235\o224\o215"
| "&#120077;" -> "\o360\o235\o224\o215"
| "&#x1D50E;" -> "\o360\o235\o224\o216"
| "&#120078;" -> "\o360\o235\o224\o216"
| "&#x0013E;" -> "\o304\o276"
| "&#318;" -> "\o304\o276"
| "&#x1D50F;" -> "\o360\o235\o224\o217"
| "&#120079;" -> "\o360\o235\o224\o217"
| "&#x1D510;" -> "\o360\o235\o224\o220"
| "&#120080;" -> "\o360\o235\o224\o220"
| "&#x1D511;" -> "\o360\o235\o224\o221"
| "&#120081;" -> "\o360\o235\o224\o221"
| "&#x1D512;" -> "\o360\o235\o224\o222"
| "&#120082;" -> "\o360\o235\o224\o222"
| "&#x1D513;" -> "\o360\o235\o224\o223"
| "&#120083;" -> "\o360\o235\o224\o223"
| "&#x1D514;" -> "\o360\o235\o224\o224"
| "&#120084;" -> "\o360\o235\o224\o224"
| "&#x1D516;" -> "\o360\o235\o224\o226"
| "&#120086;" -> "\o360\o235\o224\o226"
| "&#x1D517;" -> "\o360\o235\o224\o227"
| "&#120087;" -> "\o360\o235\o224\o227"
| "&#x1D518;" -> "\o360\o235\o224\o230"
| "&#120088;" -> "\o360\o235\o224\o230"
| "&#x1D519;" -> "\o360\o235\o224\o231"
| "&#120089;" -> "\o360\o235\o224\o231"
| "&#x0013F;" -> "\o304\o277"
| "&#319;" -> "\o304\o277"
| "&#x1D51A;" -> "\o360\o235\o224\o232"
| "&#120090;" -> "\o360\o235\o224\o232"
| "&#x1D51B;" -> "\o360\o235\o224\o233"
| "&#120091;" -> "\o360\o235\o224\o233"
| "&#x1D51C;" -> "\o360\o235\o224\o234"
| "&#120092;" -> "\o360\o235\o224\o234"
| "&#x1D51E;" -> "\o360\o235\o224\o236"
| "&#120094;" -> "\o360\o235\o224\o236"
| "&#x1D51F;" -> "\o360\o235\o224\o237"
| "&#120095;" -> "\o360\o235\o224\o237"
| "&#x1D520;" -> "\o360\o235\o224\o240"
| "&#120096;" -> "\o360\o235\o224\o240"
| "&#x1D521;" -> "\o360\o235\o224\o241"
| "&#120097;" -> "\o360\o235\o224\o241"
| "&#x1D522;" -> "\o360\o235\o224\o242"
| "&#120098;" -> "\o360\o235\o224\o242"
| "&#x1D523;" -> "\o360\o235\o224\o243"
| "&#120099;" -> "\o360\o235\o224\o243"
| "&#x1D524;" -> "\o360\o235\o224\o244"
| "&#120100;" -> "\o360\o235\o224\o244"
| "&#x00140;" -> "\o305\o200"
| "&#320;" -> "\o305\o200"
| "&#x1D525;" -> "\o360\o235\o224\o245"
| "&#120101;" -> "\o360\o235\o224\o245"
| "&#x1D526;" -> "\o360\o235\o224\o246"
| "&#120102;" -> "\o360\o235\o224\o246"
| "&#x1D527;" -> "\o360\o235\o224\o247"
| "&#120103;" -> "\o360\o235\o224\o247"
| "&#x1D528;" -> "\o360\o235\o224\o250"
| "&#120104;" -> "\o360\o235\o224\o250"
| "&#x1D529;" -> "\o360\o235\o224\o251"
| "&#120105;" -> "\o360\o235\o224\o251"
| "&#x1D52A;" -> "\o360\o235\o224\o252"
| "&#120106;" -> "\o360\o235\o224\o252"
| "&#x1D52B;" -> "\o360\o235\o224\o253"
| "&#120107;" -> "\o360\o235\o224\o253"
| "&#x1D52C;" -> "\o360\o235\o224\o254"
| "&#120108;" -> "\o360\o235\o224\o254"
| "&#x1D52D;" -> "\o360\o235\o224\o255"
| "&#120109;" -> "\o360\o235\o224\o255"
| "&#x1D52E;" -> "\o360\o235\o224\o256"
| "&#120110;" -> "\o360\o235\o224\o256"
| "&#x00141;" -> "\o305\o201"
| "&#321;" -> "\o305\o201"
| "&#x1D52F;" -> "\o360\o235\o224\o257"
| "&#120111;" -> "\o360\o235\o224\o257"
| "&#x1D530;" -> "\o360\o235\o224\o260"
| "&#120112;" -> "\o360\o235\o224\o260"
| "&#x1D531;" -> "\o360\o235\o224\o261"
| "&#120113;" -> "\o360\o235\o224\o261"
| "&#x1D532;" -> "\o360\o235\o224\o262"
| "&#120114;" -> "\o360\o235\o224\o262"
| "&#x1D533;" -> "\o360\o235\o224\o263"
| "&#120115;" -> "\o360\o235\o224\o263"
| "&#x1D534;" -> "\o360\o235\o224\o264"
| "&#120116;" -> "\o360\o235\o224\o264"
| "&#x1D535;" -> "\o360\o235\o224\o265"
| "&#120117;" -> "\o360\o235\o224\o265"
| "&#x1D536;" -> "\o360\o235\o224\o266"
| "&#120118;" -> "\o360\o235\o224\o266"
| "&#x1D537;" -> "\o360\o235\o224\o267"
| "&#120119;" -> "\o360\o235\o224\o267"
| "&#x1D538;" -> "\o360\o235\o224\o270"
| "&#120120;" -> "\o360\o235\o224\o270"
| "&#x00142;" -> "\o305\o202"
| "&#322;" -> "\o305\o202"
| "&#x1D539;" -> "\o360\o235\o224\o271"
| "&#120121;" -> "\o360\o235\o224\o271"
| "&#x1D53B;" -> "\o360\o235\o224\o273"
| "&#120123;" -> "\o360\o235\o224\o273"
| "&#x1D53C;" -> "\o360\o235\o224\o274"
| "&#120124;" -> "\o360\o235\o224\o274"
| "&#x1D53D;" -> "\o360\o235\o224\o275"
| "&#120125;" -> "\o360\o235\o224\o275"
| "&#x1D53E;" -> "\o360\o235\o224\o276"
| "&#120126;" -> "\o360\o235\o224\o276"
| "&#x1D540;" -> "\o360\o235\o225\o200"
| "&#120128;" -> "\o360\o235\o225\o200"
| "&#x1D541;" -> "\o360\o235\o225\o201"
| "&#120129;" -> "\o360\o235\o225\o201"
| "&#x1D542;" -> "\o360\o235\o225\o202"
| "&#120130;" -> "\o360\o235\o225\o202"
| "&#x1D543;" -> "\o360\o235\o225\o203"
| "&#120131;" -> "\o360\o235\o225\o203"
| "&#x1D544;" -> "\o360\o235\o225\o204"
| "&#120132;" -> "\o360\o235\o225\o204"
| "&#x00143;" -> "\o305\o203"
| "&#323;" -> "\o305\o203"
| "&#x1D546;" -> "\o360\o235\o225\o206"
| "&#120134;" -> "\o360\o235\o225\o206"
| "&#x1D54A;" -> "\o360\o235\o225\o212"
| "&#120138;" -> "\o360\o235\o225\o212"
| "&#x1D54B;" -> "\o360\o235\o225\o213"
| "&#120139;" -> "\o360\o235\o225\o213"
| "&#x1D54C;" -> "\o360\o235\o225\o214"
| "&#120140;" -> "\o360\o235\o225\o214"
| "&#x1D54D;" -> "\o360\o235\o225\o215"
| "&#120141;" -> "\o360\o235\o225\o215"
| "&#x1D54E;" -> "\o360\o235\o225\o216"
| "&#120142;" -> "\o360\o235\o225\o216"
| "&#x1D54F;" -> "\o360\o235\o225\o217"
| "&#120143;" -> "\o360\o235\o225\o217"
| "&#x1D550;" -> "\o360\o235\o225\o220"
| "&#120144;" -> "\o360\o235\o225\o220"
| "&#x1D552;" -> "\o360\o235\o225\o222"
| "&#120146;" -> "\o360\o235\o225\o222"
| "&#x1D553;" -> "\o360\o235\o225\o223"
| "&#120147;" -> "\o360\o235\o225\o223"
| "&#x00144;" -> "\o305\o204"
| "&#324;" -> "\o305\o204"
| "&#x1D554;" -> "\o360\o235\o225\o224"
| "&#120148;" -> "\o360\o235\o225\o224"
| "&#x1D555;" -> "\o360\o235\o225\o225"
| "&#120149;" -> "\o360\o235\o225\o225"
| "&#x1D556;" -> "\o360\o235\o225\o226"
| "&#120150;" -> "\o360\o235\o225\o226"
| "&#x1D557;" -> "\o360\o235\o225\o227"
| "&#120151;" -> "\o360\o235\o225\o227"
| "&#x1D558;" -> "\o360\o235\o225\o230"
| "&#120152;" -> "\o360\o235\o225\o230"
| "&#x1D559;" -> "\o360\o235\o225\o231"
| "&#120153;" -> "\o360\o235\o225\o231"
| "&#x1D55A;" -> "\o360\o235\o225\o232"
| "&#120154;" -> "\o360\o235\o225\o232"
| "&#x1D55B;" -> "\o360\o235\o225\o233"
| "&#120155;" -> "\o360\o235\o225\o233"
| "&#x1D55C;" -> "\o360\o235\o225\o234"
| "&#120156;" -> "\o360\o235\o225\o234"
| "&#x1D55D;" -> "\o360\o235\o225\o235"
| "&#120157;" -> "\o360\o235\o225\o235"
| "&#x00145;" -> "\o305\o205"
| "&#325;" -> "\o305\o205"
| "&#x1D55E;" -> "\o360\o235\o225\o236"
| "&#120158;" -> "\o360\o235\o225\o236"
| "&#x1D55F;" -> "\o360\o235\o225\o237"
| "&#120159;" -> "\o360\o235\o225\o237"
| "&#x1D560;" -> "\o360\o235\o225\o240"
| "&#120160;" -> "\o360\o235\o225\o240"
| "&#x1D561;" -> "\o360\o235\o225\o241"
| "&#120161;" -> "\o360\o235\o225\o241"
| "&#x1D562;" -> "\o360\o235\o225\o242"
| "&#120162;" -> "\o360\o235\o225\o242"
| "&#x1D563;" -> "\o360\o235\o225\o243"
| "&#120163;" -> "\o360\o235\o225\o243"
| "&#x1D564;" -> "\o360\o235\o225\o244"
| "&#120164;" -> "\o360\o235\o225\o244"
| "&#x1D565;" -> "\o360\o235\o225\o245"
| "&#120165;" -> "\o360\o235\o225\o245"
| "&#x1D566;" -> "\o360\o235\o225\o246"
| "&#120166;" -> "\o360\o235\o225\o246"
| "&#x1D567;" -> "\o360\o235\o225\o247"
| "&#120167;" -> "\o360\o235\o225\o247"
| "&#x00146;" -> "\o305\o206"
| "&#326;" -> "\o305\o206"
| "&#x1D568;" -> "\o360\o235\o225\o250"
| "&#120168;" -> "\o360\o235\o225\o250"
| "&#x1D569;" -> "\o360\o235\o225\o251"
| "&#120169;" -> "\o360\o235\o225\o251"
| "&#x1D56A;" -> "\o360\o235\o225\o252"
| "&#120170;" -> "\o360\o235\o225\o252"
| "&#x1D56B;" -> "\o360\o235\o225\o253"
| "&#120171;" -> "\o360\o235\o225\o253"
| "&#x00147;" -> "\o305\o207"
| "&#327;" -> "\o305\o207"
| "&#x0002F;" -> "/"
| "&#47;" -> "/"
| "&#x00148;" -> "\o305\o210"
| "&#328;" -> "\o305\o210"
| "&#x00149;" -> "\o305\o211"
| "&#329;" -> "\o305\o211"
| "&#x0014A;" -> "\o305\o212"
| "&#330;" -> "\o305\o212"
| "&#x0014B;" -> "\o305\o213"
| "&#331;" -> "\o305\o213"
| "&#x0014C;" -> "\o305\o214"
| "&#332;" -> "\o305\o214"
| "&#x0014D;" -> "\o305\o215"
| "&#333;" -> "\o305\o215"
| "&#x00150;" -> "\o305\o220"
| "&#336;" -> "\o305\o220"
| "&#x00151;" -> "\o305\o221"
| "&#337;" -> "\o305\o221"
| "&OElig;" -> "\o305\o222"
| "&#x00152;" -> "\o305\o222"
| "&#338;" -> "\o305\o222"
| "&oelig;" -> "\o305\o223"
| "&#x00153;" -> "\o305\o223"
| "&#339;" -> "\o305\o223"
| "&#x00154;" -> "\o305\o224"
| "&#340;" -> "\o305\o224"
| "&#x0003A;" -> ":"
| "&#58;" -> ":"
| "&#x00155;" -> "\o305\o225"
| "&#341;" -> "\o305\o225"
| "&#x00156;" -> "\o305\o226"
| "&#342;" -> "\o305\o226"
| "&#x00157;" -> "\o305\o227"
| "&#343;" -> "\o305\o227"
| "&#x00158;" -> "\o305\o230"
| "&#344;" -> "\o305\o230"
| "&#x00159;" -> "\o305\o231"
| "&#345;" -> "\o305\o231"
| "&#x0015A;" -> "\o305\o232"
| "&#346;" -> "\o305\o232"
| "&#x0015B;" -> "\o305\o233"
| "&#347;" -> "\o305\o233"
| "&#x0015C;" -> "\o305\o234"
| "&#348;" -> "\o305\o234"
| "&#x0015D;" -> "\o305\o235"
| "&#349;" -> "\o305\o235"
| "&#x0015E;" -> "\o305\o236"
| "&#350;" -> "\o305\o236"
| "&#x0003B;" -> ";"
| "&#59;" -> ";"
| "&#x0015F;" -> "\o305\o237"
| "&#351;" -> "\o305\o237"
| "&Scaron;" -> "\o305\o240"
| "&#x00160;" -> "\o305\o240"
| "&#352;" -> "\o305\o240"
| "&scaron;" -> "\o305\o241"
| "&#x00161;" -> "\o305\o241"
| "&#353;" -> "\o305\o241"
| "&#x00162;" -> "\o305\o242"
| "&#354;" -> "\o305\o242"
| "&#x00163;" -> "\o305\o243"
| "&#355;" -> "\o305\o243"
| "&#x00164;" -> "\o305\o244"
| "&#356;" -> "\o305\o244"
| "&lt;" -> "<"
| "&#x0003C;" -> "<"
| "&#60;" -> "<"
| "&#x00165;" -> "\o305\o245"
| "&#357;" -> "\o305\o245"
| "&#x00166;" -> "\o305\o246"
| "&#358;" -> "\o305\o246"
| "&#x00167;" -> "\o305\o247"
| "&#359;" -> "\o305\o247"
| "&#x00168;" -> "\o305\o250"
| "&#360;" -> "\o305\o250"
| "&#x00169;" -> "\o305\o251"
| "&#361;" -> "\o305\o251"
| "&#x0016A;" -> "\o305\o252"
| "&#362;" -> "\o305\o252"
| "&#x0016B;" -> "\o305\o253"
| "&#363;" -> "\o305\o253"
| "&#x0016C;" -> "\o305\o254"
| "&#364;" -> "\o305\o254"
| "&#x0016D;" -> "\o305\o255"
| "&#365;" -> "\o305\o255"
| "&#x0016E;" -> "\o305\o256"
| "&#366;" -> "\o305\o256"
| "&#x0016F;" -> "\o305\o257"
| "&#367;" -> "\o305\o257"
| "&#x00170;" -> "\o305\o260"
| "&#368;" -> "\o305\o260"
| "&#x00171;" -> "\o305\o261"
| "&#369;" -> "\o305\o261"
| "&#x00172;" -> "\o305\o262"
| "&#370;" -> "\o305\o262"
| "&#x00173;" -> "\o305\o263"
| "&#371;" -> "\o305\o263"
| "&#x0000A;" -> "\n"
| "&#10;" -> "\n"
| "&#x0003D;" -> "="
| "&#61;" -> "="
| "&#x00174;" -> "\o305\o264"
| "&#372;" -> "\o305\o264"
| "&#x00175;" -> "\o305\o265"
| "&#373;" -> "\o305\o265"
| "&#x00176;" -> "\o305\o266"
| "&#374;" -> "\o305\o266"
| "&#x00177;" -> "\o305\o267"
| "&#375;" -> "\o305\o267"
| "&Yuml;" -> "\o305\o270"
| "&#x00178;" -> "\o305\o270"
| "&#376;" -> "\o305\o270"
| "&gt;" -> ">"
| "&#x0003E;" -> ">"
| "&#62;" -> ">"
| "&#x00179;" -> "\o305\o271"
| "&#377;" -> "\o305\o271"
| "&#x0017A;" -> "\o305\o272"
| "&#378;" -> "\o305\o272"
| "&#x0017B;" -> "\o305\o273"
| "&#379;" -> "\o305\o273"
| "&#x0017C;" -> "\o305\o274"
| "&#380;" -> "\o305\o274"
| "&#x0017D;" -> "\o305\o275"
| "&#381;" -> "\o305\o275"
| "&#x0017E;" -> "\o305\o276"
| "&#382;" -> "\o305\o276"
| "&fnof;" -> "\o306\o222"
| "&#x00192;" -> "\o306\o222"
| "&#402;" -> "\o306\o222"
| "&#x001B5;" -> "\o306\o265"
| "&#437;" -> "\o306\o265"
| "&#x001F5;" -> "\o307\o265"
| "&#501;" -> "\o307\o265"
| "&#x00237;" -> "\o310\o267"
| "&#567;" -> "\o310\o267"
| "&circ;" -> "\o313\o206"
| "&#x002C6;" -> "\o313\o206"
| "&#710;" -> "\o313\o206"
| "&#x002C7;" -> "\o313\o207"
| "&#711;" -> "\o313\o207"
| "&#x002D8;" -> "\o313\o230"
| "&#728;" -> "\o313\o230"
| "&#x002D9;" -> "\o313\o231"
| "&#729;" -> "\o313\o231"
| "&#x002DA;" -> "\o313\o232"
| "&#730;" -> "\o313\o232"
| "&#x0003F;" -> "?"
| "&#63;" -> "?"
| "&#x002DB;" -> "\o313\o233"
| "&#731;" -> "\o313\o233"
| "&tilde;" -> "\o313\o234"
| "&#x002DC;" -> "\o313\o234"
| "&#732;" -> "\o313\o234"
| "&#x002DD;" -> "\o313\o235"
| "&#733;" -> "\o313\o235"
| "&#x00311;" -> "\o314\o221"
| "&#785;" -> "\o314\o221"
| "&#x00332;" -> "\o314\o262"
| "&#818;" -> "\o314\o262"
| "&Alpha;" -> "\o316\o221"
| "&#x00391;" -> "\o316\o221"
| "&#913;" -> "\o316\o221"
| "&Beta;" -> "\o316\o222"
| "&#x00392;" -> "\o316\o222"
| "&#914;" -> "\o316\o222"
| "&Gamma;" -> "\o316\o223"
| "&#x00393;" -> "\o316\o223"
| "&#915;" -> "\o316\o223"
| "&Delta;" -> "\o316\o224"
| "&#x00394;" -> "\o316\o224"
| "&#916;" -> "\o316\o224"
| "&Epsilon;" -> "\o316\o225"
| "&#x00395;" -> "\o316\o225"
| "&#917;" -> "\o316\o225"
| "&Zeta;" -> "\o316\o226"
| "&#x00396;" -> "\o316\o226"
| "&#918;" -> "\o316\o226"
| "&Eta;" -> "\o316\o227"
| "&#x00397;" -> "\o316\o227"
| "&#919;" -> "\o316\o227"
| "&Theta;" -> "\o316\o230"
| "&#x00398;" -> "\o316\o230"
| "&#920;" -> "\o316\o230"
| "&Iota;" -> "\o316\o231"
| "&#x00399;" -> "\o316\o231"
| "&#921;" -> "\o316\o231"
| "&#x00040;" -> "@"
| "&#64;" -> "@"
| "&Kappa;" -> "\o316\o232"
| "&#x0039A;" -> "\o316\o232"
| "&#922;" -> "\o316\o232"
| "&Lambda;" -> "\o316\o233"
| "&#x0039B;" -> "\o316\o233"
| "&#923;" -> "\o316\o233"
| "&Mu;" -> "\o316\o234"
| "&#x0039C;" -> "\o316\o234"
| "&#924;" -> "\o316\o234"
| "&Nu;" -> "\o316\o235"
| "&#x0039D;" -> "\o316\o235"
| "&#925;" -> "\o316\o235"
| "&Xi;" -> "\o316\o236"
| "&#x0039E;" -> "\o316\o236"
| "&#926;" -> "\o316\o236"
| "&Omicron;" -> "\o316\o237"
| "&#x0039F;" -> "\o316\o237"
| "&#927;" -> "\o316\o237"
| "&Pi;" -> "\o316\o240"
| "&#x003A0;" -> "\o316\o240"
| "&#928;" -> "\o316\o240"
| "&Rho;" -> "\o316\o241"
| "&#x003A1;" -> "\o316\o241"
| "&#929;" -> "\o316\o241"
| "&Sigma;" -> "\o316\o243"
| "&#x003A3;" -> "\o316\o243"
| "&#931;" -> "\o316\o243"
| "&Tau;" -> "\o316\o244"
| "&#x003A4;" -> "\o316\o244"
| "&#932;" -> "\o316\o244"
| "&Upsilon;" -> "\o316\o245"
| "&#x003A5;" -> "\o316\o245"
| "&#933;" -> "\o316\o245"
| "&Phi;" -> "\o316\o246"
| "&#x003A6;" -> "\o316\o246"
| "&#934;" -> "\o316\o246"
| "&Chi;" -> "\o316\o247"
| "&#x003A7;" -> "\o316\o247"
| "&#935;" -> "\o316\o247"
| "&Psi;" -> "\o316\o250"
| "&#x003A8;" -> "\o316\o250"
| "&#936;" -> "\o316\o250"
| "&Omega;" -> "\o316\o251"
| "&#x003A9;" -> "\o316\o251"
| "&#937;" -> "\o316\o251"
| "&alpha;" -> "\o316\o261"
| "&#x003B1;" -> "\o316\o261"
| "&#945;" -> "\o316\o261"
| "&beta;" -> "\o316\o262"
| "&#x003B2;" -> "\o316\o262"
| "&#946;" -> "\o316\o262"
| "&gamma;" -> "\o316\o263"
| "&#x003B3;" -> "\o316\o263"
| "&#947;" -> "\o316\o263"
| "&delta;" -> "\o316\o264"
| "&#x003B4;" -> "\o316\o264"
| "&#948;" -> "\o316\o264"
| "&epsilon;" -> "\o316\o265"
| "&#x003B5;" -> "\o316\o265"
| "&#949;" -> "\o316\o265"
| "&zeta;" -> "\o316\o266"
| "&#x003B6;" -> "\o316\o266"
| "&#950;" -> "\o316\o266"
| "&eta;" -> "\o316\o267"
| "&#x003B7;" -> "\o316\o267"
| "&#951;" -> "\o316\o267"
| "&theta;" -> "\o316\o270"
| "&#x003B8;" -> "\o316\o270"
| "&#952;" -> "\o316\o270"
| "&iota;" -> "\o316\o271"
| "&#x003B9;" -> "\o316\o271"
| "&#953;" -> "\o316\o271"
| "&kappa;" -> "\o316\o272"
| "&#x003BA;" -> "\o316\o272"
| "&#954;" -> "\o316\o272"
| "&lambda;" -> "\o316\o273"
| "&#x003BB;" -> "\o316\o273"
| "&#955;" -> "\o316\o273"
| "&mu;" -> "\o316\o274"
| "&#x003BC;" -> "\o316\o274"
| "&#956;" -> "\o316\o274"
| "&nu;" -> "\o316\o275"
| "&#x003BD;" -> "\o316\o275"
| "&#957;" -> "\o316\o275"
| "&#x0005B;" -> "["
| "&#91;" -> "["
| "&xi;" -> "\o316\o276"
| "&#x003BE;" -> "\o316\o276"
| "&#958;" -> "\o316\o276"
| "&omicron;" -> "\o316\o277"
| "&#x003BF;" -> "\o316\o277"
| "&#959;" -> "\o316\o277"
| "&pi;" -> "\o317\o200"
| "&#x003C0;" -> "\o317\o200"
| "&#960;" -> "\o317\o200"
| "&rho;" -> "\o317\o201"
| "&#x003C1;" -> "\o317\o201"
| "&#961;" -> "\o317\o201"
| "&sigmaf;" -> "\o317\o202"
| "&#x003C2;" -> "\o317\o202"
| "&#962;" -> "\o317\o202"
| "&sigma;" -> "\o317\o203"
| "&#x003C3;" -> "\o317\o203"
| "&#963;" -> "\o317\o203"
| "&tau;" -> "\o317\o204"
| "&#x003C4;" -> "\o317\o204"
| "&#964;" -> "\o317\o204"
| "&upsilon;" -> "\o317\o205"
| "&#x003C5;" -> "\o317\o205"
| "&#965;" -> "\o317\o205"
| "&phi;" -> "\o317\o206"
| "&#x003C6;" -> "\o317\o206"
| "&#966;" -> "\o317\o206"
| "&chi;" -> "\o317\o207"
| "&#x003C7;" -> "\o317\o207"
| "&#967;" -> "\o317\o207"
| "&psi;" -> "\o317\o210"
| "&#x003C8;" -> "\o317\o210"
| "&#968;" -> "\o317\o210"
| "&omega;" -> "\o317\o211"
| "&#x003C9;" -> "\o317\o211"
| "&#969;" -> "\o317\o211"
| "&thetasym;" -> "\o317\o221"
| "&#x003D1;" -> "\o317\o221"
| "&#977;" -> "\o317\o221"
| "&#x0005C;" -> "\\"
| "&#92;" -> "\\"
| "&upsih;" -> "\o317\o222"
| "&#x003D2;" -> "\o317\o222"
| "&#978;" -> "\o317\o222"
| "&#x003D5;" -> "\o317\o225"
| "&#981;" -> "\o317\o225"
| "&piv;" -> "\o317\o226"
| "&#x003D6;" -> "\o317\o226"
| "&#982;" -> "\o317\o226"
| "&#x003DC;" -> "\o317\o234"
| "&#988;" -> "\o317\o234"
| "&#x003DD;" -> "\o317\o235"
| "&#989;" -> "\o317\o235"
| "&#x003F0;" -> "\o317\o260"
| "&#1008;" -> "\o317\o260"
| "&#x003F1;" -> "\o317\o261"
| "&#1009;" -> "\o317\o261"
| "&#x003F5;" -> "\o317\o265"
| "&#1013;" -> "\o317\o265"
| "&bepsi;" -> "\o342\o210\o215"
| "&#x003F6;" -> "\o317\o266"
| "&#1014;" -> "\o317\o266"
| "&#x00401;" -> "\o320\o201"
| "&#1025;" -> "\o320\o201"
| "&#x00402;" -> "\o320\o202"
| "&#1026;" -> "\o320\o202"
| "&#x00403;" -> "\o320\o203"
| "&#1027;" -> "\o320\o203"
| "&#x0005D;" -> "]"
| "&#93;" -> "]"
| "&#x00404;" -> "\o320\o204"
| "&#1028;" -> "\o320\o204"
| "&#x00405;" -> "\o320\o205"
| "&#1029;" -> "\o320\o205"
| "&#x00406;" -> "\o320\o206"
| "&#1030;" -> "\o320\o206"
| "&#x00407;" -> "\o320\o207"
| "&#1031;" -> "\o320\o207"
| "&#x00408;" -> "\o320\o210"
| "&#1032;" -> "\o320\o210"
| "&#x00409;" -> "\o320\o211"
| "&#1033;" -> "\o320\o211"
| "&#x0040A;" -> "\o320\o212"
| "&#1034;" -> "\o320\o212"
| "&#x0040B;" -> "\o320\o213"
| "&#1035;" -> "\o320\o213"
| "&#x0040C;" -> "\o320\o214"
| "&#1036;" -> "\o320\o214"
| "&#x0040E;" -> "\o320\o216"
| "&#1038;" -> "\o320\o216"
| "&#x0005E;" -> "^"
| "&#94;" -> "^"
| "&#x0040F;" -> "\o320\o217"
| "&#1039;" -> "\o320\o217"
| "&#x00410;" -> "\o320\o220"
| "&#1040;" -> "\o320\o220"
| "&#x00411;" -> "\o320\o221"
| "&#1041;" -> "\o320\o221"
| "&#x00412;" -> "\o320\o222"
| "&#1042;" -> "\o320\o222"
| "&#x00413;" -> "\o320\o223"
| "&#1043;" -> "\o320\o223"
| "&#x00414;" -> "\o320\o224"
| "&#1044;" -> "\o320\o224"
| "&#x00415;" -> "\o320\o225"
| "&#1045;" -> "\o320\o225"
| "&#x00416;" -> "\o320\o226"
| "&#1046;" -> "\o320\o226"
| "&#x00417;" -> "\o320\o227"
| "&#1047;" -> "\o320\o227"
| "&#x00418;" -> "\o320\o230"
| "&#1048;" -> "\o320\o230"
| "&#x0005F;" -> "_"
| "&#95;" -> "_"
| "&#x00419;" -> "\o320\o231"
| "&#1049;" -> "\o320\o231"
| "&#x0041A;" -> "\o320\o232"
| "&#1050;" -> "\o320\o232"
| "&#x0041B;" -> "\o320\o233"
| "&#1051;" -> "\o320\o233"
| "&#x0041C;" -> "\o320\o234"
| "&#1052;" -> "\o320\o234"
| "&#x0041D;" -> "\o320\o235"
| "&#1053;" -> "\o320\o235"
| "&#x0041E;" -> "\o320\o236"
| "&#1054;" -> "\o320\o236"
| "&#x0041F;" -> "\o320\o237"
| "&#1055;" -> "\o320\o237"
| "&#x00420;" -> "\o320\o240"
| "&#1056;" -> "\o320\o240"
| "&#x00421;" -> "\o320\o241"
| "&#1057;" -> "\o320\o241"
| "&#x00422;" -> "\o320\o242"
| "&#1058;" -> "\o320\o242"
| "&#x00021;" -> "!"
| "&#33;" -> "!"
| "&#x00423;" -> "\o320\o243"
| "&#1059;" -> "\o320\o243"
| "&#x00424;" -> "\o320\o244"
| "&#1060;" -> "\o320\o244"
| "&#x00425;" -> "\o320\o245"
| "&#1061;" -> "\o320\o245"
| "&#x00426;" -> "\o320\o246"
| "&#1062;" -> "\o320\o246"
| "&#x00427;" -> "\o320\o247"
| "&#1063;" -> "\o320\o247"
| "&#x00060;" -> "`"
| "&#96;" -> "`"
| "&#x00428;" -> "\o320\o250"
| "&#1064;" -> "\o320\o250"
| "&#x00429;" -> "\o320\o251"
| "&#1065;" -> "\o320\o251"
| "&#x0042A;" -> "\o320\o252"
| "&#1066;" -> "\o320\o252"
| "&#x0042B;" -> "\o320\o253"
| "&#1067;" -> "\o320\o253"
| "&#x0042C;" -> "\o320\o254"
| "&#1068;" -> "\o320\o254"
| "&#x0042D;" -> "\o320\o255"
| "&#1069;" -> "\o320\o255"
| "&#x0042E;" -> "\o320\o256"
| "&#1070;" -> "\o320\o256"
| "&#x0042F;" -> "\o320\o257"
| "&#1071;" -> "\o320\o257"
| "&#x00430;" -> "\o320\o260"
| "&#1072;" -> "\o320\o260"
| "&#x00431;" -> "\o320\o261"
| "&#1073;" -> "\o320\o261"
| "&#x00432;" -> "\o320\o262"
| "&#1074;" -> "\o320\o262"
| "&#x00433;" -> "\o320\o263"
| "&#1075;" -> "\o320\o263"
| "&#x00434;" -> "\o320\o264"
| "&#1076;" -> "\o320\o264"
| "&#x00435;" -> "\o320\o265"
| "&#1077;" -> "\o320\o265"
| "&#x00436;" -> "\o320\o266"
| "&#1078;" -> "\o320\o266"
| "&#x0007B;" -> "{"
| "&#123;" -> "{"
| "&#x00437;" -> "\o320\o267"
| "&#1079;" -> "\o320\o267"
| "&#x00438;" -> "\o320\o270"
| "&#1080;" -> "\o320\o270"
| "&#x00439;" -> "\o320\o271"
| "&#1081;" -> "\o320\o271"
| "&#x0043A;" -> "\o320\o272"
| "&#1082;" -> "\o320\o272"
| "&#x0043B;" -> "\o320\o273"
| "&#1083;" -> "\o320\o273"
| "&#x0043C;" -> "\o320\o274"
| "&#1084;" -> "\o320\o274"
| "&#x0043D;" -> "\o320\o275"
| "&#1085;" -> "\o320\o275"
| "&#x0043E;" -> "\o320\o276"
| "&#1086;" -> "\o320\o276"
| "&#x0043F;" -> "\o320\o277"
| "&#1087;" -> "\o320\o277"
| "&#x00440;" -> "\o321\o200"
| "&#1088;" -> "\o321\o200"
| "&#x00441;" -> "\o321\o201"
| "&#1089;" -> "\o321\o201"
| "&#x00442;" -> "\o321\o202"
| "&#1090;" -> "\o321\o202"
| "&#x00443;" -> "\o321\o203"
| "&#1091;" -> "\o321\o203"
| "&#x00444;" -> "\o321\o204"
| "&#1092;" -> "\o321\o204"
| "&#x00445;" -> "\o321\o205"
| "&#1093;" -> "\o321\o205"
| "&#x00446;" -> "\o321\o206"
| "&#1094;" -> "\o321\o206"
| "&#x00447;" -> "\o321\o207"
| "&#1095;" -> "\o321\o207"
| "&#x00448;" -> "\o321\o210"
| "&#1096;" -> "\o321\o210"
| "&#x00449;" -> "\o321\o211"
| "&#1097;" -> "\o321\o211"
| "&#x0044A;" -> "\o321\o212"
| "&#1098;" -> "\o321\o212"
| "&#x0007C;" -> "|"
| "&#124;" -> "|"
| "&#x0044B;" -> "\o321\o213"
| "&#1099;" -> "\o321\o213"
| "&#x0044C;" -> "\o321\o214"
| "&#1100;" -> "\o321\o214"
| "&#x0044D;" -> "\o321\o215"
| "&#1101;" -> "\o321\o215"
| "&#x0044E;" -> "\o321\o216"
| "&#1102;" -> "\o321\o216"
| "&#x0044F;" -> "\o321\o217"
| "&#1103;" -> "\o321\o217"
| "&#x00451;" -> "\o321\o221"
| "&#1105;" -> "\o321\o221"
| "&#x00452;" -> "\o321\o222"
| "&#1106;" -> "\o321\o222"
| "&#x00453;" -> "\o321\o223"
| "&#1107;" -> "\o321\o223"
| "&#x00454;" -> "\o321\o224"
| "&#1108;" -> "\o321\o224"
| "&#x00455;" -> "\o321\o225"
| "&#1109;" -> "\o321\o225"
| "&quot;" -> "\""
| "&#x00022;" -> "\""
| "&#34;" -> "\""
| "&#x00456;" -> "\o321\o226"
| "&#1110;" -> "\o321\o226"
| "&#x00457;" -> "\o321\o227"
| "&#1111;" -> "\o321\o227"
| "&#x00458;" -> "\o321\o230"
| "&#1112;" -> "\o321\o230"
| "&#x00459;" -> "\o321\o231"
| "&#1113;" -> "\o321\o231"
| "&#x0045A;" -> "\o321\o232"
| "&#1114;" -> "\o321\o232"
| "&#x0007D;" -> "}"
| "&#125;" -> "}"
| "&#x0045B;" -> "\o321\o233"
| "&#1115;" -> "\o321\o233"
| "&#x0045C;" -> "\o321\o234"
| "&#1116;" -> "\o321\o234"
| "&#x0045E;" -> "\o321\o236"
| "&#1118;" -> "\o321\o236"
| "&#x0045F;" -> "\o321\o237"
| "&#1119;" -> "\o321\o237"
| "&ensp;" -> "\o342\o200\o202"
| "&#x02002;" -> "\o342\o200\o202"
| "&#8194;" -> "\o342\o200\o202"
| "&emsp;" -> "\o342\o200\o203"
| "&#x02003;" -> "\o342\o200\o203"
| "&#8195;" -> "\o342\o200\o203"
| "&nbsp;" -> "\o302\o240"
| "&#x000A0;" -> "\o302\o240"
| "&#160;" -> "\o302\o240"
| "&#x02004;" -> "\o342\o200\o204"
| "&#8196;" -> "\o342\o200\o204"
| "&#x02005;" -> "\o342\o200\o205"
| "&#8197;" -> "\o342\o200\o205"
| "&#x02007;" -> "\o342\o200\o207"
| "&#8199;" -> "\o342\o200\o207"
| "&#x02008;" -> "\o342\o200\o210"
| "&#8200;" -> "\o342\o200\o210"
| "&thinsp;" -> "\o342\o200\o211"
| "&#x02009;" -> "\o342\o200\o211"
| "&#8201;" -> "\o342\o200\o211"
| "&#x0200A;" -> "\o342\o200\o212"
| "&#8202;" -> "\o342\o200\o212"
| "&#x0200B;" -> "\o342\o200\o213"
| "&#8203;" -> "\o342\o200\o213"
| "&zwnj;" -> "\o342\o200\o214"
| "&#x0200C;" -> "\o342\o200\o214"
| "&#8204;" -> "\o342\o200\o214"
| "&zwj;" -> "\o342\o200\o215"
| "&#x0200D;" -> "\o342\o200\o215"
| "&#8205;" -> "\o342\o200\o215"
| "&iexcl;" -> "\o302\o241"
| "&#x000A1;" -> "\o302\o241"
| "&#161;" -> "\o302\o241"
| "&lrm;" -> "\o342\o200\o216"
| "&#x0200E;" -> "\o342\o200\o216"
| "&#8206;" -> "\o342\o200\o216"
| "&rlm;" -> "\o342\o200\o217"
| "&#x0200F;" -> "\o342\o200\o217"
| "&#8207;" -> "\o342\o200\o217"
| "&#x02010;" -> "\o342\o200\o220"
| "&#8208;" -> "\o342\o200\o220"
| "&ndash;" -> "\o342\o200\o223"
| "&#x02013;" -> "\o342\o200\o223"
| "&#8211;" -> "\o342\o200\o223"
| "&mdash;" -> "\o342\o200\o224"
| "&#x02014;" -> "\o342\o200\o224"
| "&#8212;" -> "\o342\o200\o224"
| "&#x02015;" -> "\o342\o200\o225"
| "&#8213;" -> "\o342\o200\o225"
| "&cent;" -> "\o302\o242"
| "&#x000A2;" -> "\o302\o242"
| "&#162;" -> "\o302\o242"
| "&#x02016;" -> "\o342\o200\o226"
| "&#8214;" -> "\o342\o200\o226"
| "&lsquo;" -> "\o342\o200\o230"
| "&#x02018;" -> "\o342\o200\o230"
| "&#8216;" -> "\o342\o200\o230"
| "&rsquo;" -> "\o342\o200\o231"
| "&#x02019;" -> "\o342\o200\o231"
| "&#8217;" -> "\o342\o200\o231"
| "&sbquo;" -> "\o342\o200\o232"
| "&#x0201A;" -> "\o342\o200\o232"
| "&#8218;" -> "\o342\o200\o232"
| "&ldquo;" -> "\o342\o200\o234"
| "&#x0201C;" -> "\o342\o200\o234"
| "&#8220;" -> "\o342\o200\o234"
| "&pound;" -> "\o302\o243"
| "&#x000A3;" -> "\o302\o243"
| "&#163;" -> "\o302\o243"
| "&rdquo;" -> "\o342\o200\o235"
| "&#x0201D;" -> "\o342\o200\o235"
| "&#8221;" -> "\o342\o200\o235"
| "&bdquo;" -> "\o342\o200\o236"
| "&#x0201E;" -> "\o342\o200\o236"
| "&#8222;" -> "\o342\o200\o236"
| "&dagger;" -> "\o342\o200\o240"
| "&#x02020;" -> "\o342\o200\o240"
| "&#8224;" -> "\o342\o200\o240"
| "&Dagger;" -> "\o342\o200\o241"
| "&#x02021;" -> "\o342\o200\o241"
| "&#8225;" -> "\o342\o200\o241"
| "&bull;" -> "\o342\o200\o242"
| "&#x02022;" -> "\o342\o200\o242"
| "&#8226;" -> "\o342\o200\o242"
| "&curren;" -> "\o302\o244"
| "&#x000A4;" -> "\o302\o244"
| "&#164;" -> "\o302\o244"
| "&#x02025;" -> "\o342\o200\o245"
| "&#8229;" -> "\o342\o200\o245"
| "&hellip;" -> "\o342\o200\o246"
| "&#x02026;" -> "\o342\o200\o246"
| "&#8230;" -> "\o342\o200\o246"
| "&permil;" -> "\o342\o200\o260"
| "&#x02030;" -> "\o342\o200\o260"
| "&#8240;" -> "\o342\o200\o260"
| "&#x02031;" -> "\o342\o200\o261"
| "&#8241;" -> "\o342\o200\o261"
| "&prime;" -> "\o342\o200\o262"
| "&#x02032;" -> "\o342\o200\o262"
| "&#8242;" -> "\o342\o200\o262"
| "&Prime;" -> "\o342\o200\o263"
| "&#x02033;" -> "\o342\o200\o263"
| "&#8243;" -> "\o342\o200\o263"
| "&yen;" -> "\o302\o245"
| "&#x000A5;" -> "\o302\o245"
| "&#165;" -> "\o302\o245"
| "&#x02034;" -> "\o342\o200\o264"
| "&#8244;" -> "\o342\o200\o264"
| "&#x02035;" -> "\o342\o200\o265"
| "&#8245;" -> "\o342\o200\o265"
| "&lsaquo;" -> "\o342\o200\o271"
| "&#x02039;" -> "\o342\o200\o271"
| "&#8249;" -> "\o342\o200\o271"
| "&rsaquo;" -> "\o342\o200\o272"
| "&#x0203A;" -> "\o342\o200\o272"
| "&#8250;" -> "\o342\o200\o272"
| "&oline;" -> "\o342\o200\o276"
| "&#x0203E;" -> "\o342\o200\o276"
| "&#8254;" -> "\o342\o200\o276"
| "&#x02041;" -> "\o342\o201\o201"
| "&#8257;" -> "\o342\o201\o201"
| "&brvbar;" -> "\o302\o246"
| "&#x000A6;" -> "\o302\o246"
| "&#166;" -> "\o302\o246"
| "&#x02043;" -> "\o342\o201\o203"
| "&#8259;" -> "\o342\o201\o203"
| "&frasl;" -> "\o342\o201\o204"
| "&#x02044;" -> "\o342\o201\o204"
| "&#8260;" -> "\o342\o201\o204"
| "&#x0204F;" -> "\o342\o201\o217"
| "&#8271;" -> "\o342\o201\o217"
| "&#x02057;" -> "\o342\o201\o227"
| "&#8279;" -> "\o342\o201\o227"
| "&#x0205F;" -> "\o342\o201\o237"
| "&#8287;" -> "\o342\o201\o237"
| "&sect;" -> "\o302\o247"
| "&#x000A7;" -> "\o302\o247"
| "&#167;" -> "\o302\o247"
| "&#x02060;" -> "\o342\o201\o240"
| "&#8288;" -> "\o342\o201\o240"
| "&#x02061;" -> "\o342\o201\o241"
| "&#8289;" -> "\o342\o201\o241"
| "&#x02062;" -> "\o342\o201\o242"
| "&#8290;" -> "\o342\o201\o242"
| "&#x02063;" -> "\o342\o201\o243"
| "&#8291;" -> "\o342\o201\o243"
| "&euro;" -> "\o342\o202\o254"
| "&#x020AC;" -> "\o342\o202\o254"
| "&#8364;" -> "\o342\o202\o254"
| "&#x020DB;" -> "\o342\o203\o233"
| "&#8411;" -> "\o342\o203\o233"
| "&#x020DC;" -> "\o342\o203\o234"
| "&#8412;" -> "\o342\o203\o234"
| "&#x02102;" -> "\o342\o204\o202"
| "&#8450;" -> "\o342\o204\o202"
| "&#x02105;" -> "\o342\o204\o205"
| "&#8453;" -> "\o342\o204\o205"
| "&#x0210A;" -> "\o342\o204\o212"
| "&#8458;" -> "\o342\o204\o212"
| "&#x0210B;" -> "\o342\o204\o213"
| "&#8459;" -> "\o342\o204\o213"
| "&#x0210C;" -> "\o342\o204\o214"
| "&#8460;" -> "\o342\o204\o214"
| "&#x0210D;" -> "\o342\o204\o215"
| "&#8461;" -> "\o342\o204\o215"
| "&#x0210E;" -> "\o342\o204\o216"
| "&#8462;" -> "\o342\o204\o216"
| "&#x0210F;" -> "\o342\o204\o217"
| "&#8463;" -> "\o342\o204\o217"
| "&uml;" -> "\o302\o250"
| "&#x000A8;" -> "\o302\o250"
| "&#168;" -> "\o302\o250"
| "&#x02110;" -> "\o342\o204\o220"
| "&#8464;" -> "\o342\o204\o220"
| "&image;" -> "\o342\o204\o221"
| "&#x02111;" -> "\o342\o204\o221"
| "&#8465;" -> "\o342\o204\o221"
| "&copy;" -> "\o302\o251"
| "&#x000A9;" -> "\o302\o251"
| "&#169;" -> "\o302\o251"
| "&#x02112;" -> "\o342\o204\o222"
| "&#8466;" -> "\o342\o204\o222"
| "&#x02113;" -> "\o342\o204\o223"
| "&#8467;" -> "\o342\o204\o223"
| "&#x02115;" -> "\o342\o204\o225"
| "&#8469;" -> "\o342\o204\o225"
| "&#x02116;" -> "\o342\o204\o226"
| "&#8470;" -> "\o342\o204\o226"
| "&#x02117;" -> "\o342\o204\o227"
| "&#8471;" -> "\o342\o204\o227"
| "&weierp;" -> "\o342\o204\o230"
| "&#x02118;" -> "\o342\o204\o230"
| "&#8472;" -> "\o342\o204\o230"
| "&#x02119;" -> "\o342\o204\o231"
| "&#8473;" -> "\o342\o204\o231"
| "&#x0211A;" -> "\o342\o204\o232"
| "&#8474;" -> "\o342\o204\o232"
| "&ordf;" -> "\o302\o252"
| "&#x000AA;" -> "\o302\o252"
| "&#170;" -> "\o302\o252"
| "&#x0211B;" -> "\o342\o204\o233"
| "&#8475;" -> "\o342\o204\o233"
| "&real;" -> "\o342\o204\o234"
| "&#x0211C;" -> "\o342\o204\o234"
| "&#8476;" -> "\o342\o204\o234"
| "&#x0211D;" -> "\o342\o204\o235"
| "&#8477;" -> "\o342\o204\o235"
| "&#x0211E;" -> "\o342\o204\o236"
| "&#8478;" -> "\o342\o204\o236"
| "&laquo;" -> "\o302\o253"
| "&#x000AB;" -> "\o302\o253"
| "&#171;" -> "\o302\o253"
| "&trade;" -> "\o342\o204\o242"
| "&#x02122;" -> "\o342\o204\o242"
| "&#8482;" -> "\o342\o204\o242"
| "&#x02124;" -> "\o342\o204\o244"
| "&#8484;" -> "\o342\o204\o244"
| "&#x02126;" -> "\o342\o204\o246"
| "&#8486;" -> "\o342\o204\o246"
| "&#x02127;" -> "\o342\o204\o247"
| "&#8487;" -> "\o342\o204\o247"
| "&not;" -> "\o302\o254"
| "&#x000AC;" -> "\o302\o254"
| "&#172;" -> "\o302\o254"
| "&#x02128;" -> "\o342\o204\o250"
| "&#8488;" -> "\o342\o204\o250"
| "&#x02129;" -> "\o342\o204\o251"
| "&#8489;" -> "\o342\o204\o251"
| "&#x0212B;" -> "\o342\o204\o253"
| "&#8491;" -> "\o342\o204\o253"
| "&#x0212C;" -> "\o342\o204\o254"
| "&#8492;" -> "\o342\o204\o254"
| "&shy;" -> "\o302\o255"
| "&#x000AD;" -> "\o302\o255"
| "&#173;" -> "\o302\o255"
| "&#x0212D;" -> "\o342\o204\o255"
| "&#8493;" -> "\o342\o204\o255"
| "&#x0212F;" -> "\o342\o204\o257"
| "&#8495;" -> "\o342\o204\o257"
| "&#x02130;" -> "\o342\o204\o260"
| "&#8496;" -> "\o342\o204\o260"
| "&reg;" -> "\o302\o256"
| "&#x000AE;" -> "\o302\o256"
| "&#174;" -> "\o302\o256"
| "&#x02131;" -> "\o342\o204\o261"
| "&#8497;" -> "\o342\o204\o261"
| "&#x02133;" -> "\o342\o204\o263"
| "&#8499;" -> "\o342\o204\o263"
| "&#x02134;" -> "\o342\o204\o264"
| "&#8500;" -> "\o342\o204\o264"
| "&alefsym;" -> "\o342\o204\o265"
| "&#x02135;" -> "\o342\o204\o265"
| "&#8501;" -> "\o342\o204\o265"
| "&#x02136;" -> "\o342\o204\o266"
| "&#8502;" -> "\o342\o204\o266"
| "&#x02137;" -> "\o342\o204\o267"
| "&#8503;" -> "\o342\o204\o267"
| "&#x02138;" -> "\o342\o204\o270"
| "&#8504;" -> "\o342\o204\o270"
| "&#x02145;" -> "\o342\o205\o205"
| "&#8517;" -> "\o342\o205\o205"
| "&#x02146;" -> "\o342\o205\o206"
| "&#8518;" -> "\o342\o205\o206"
| "&#x02147;" -> "\o342\o205\o207"
| "&#8519;" -> "\o342\o205\o207"
| "&#x02148;" -> "\o342\o205\o210"
| "&#8520;" -> "\o342\o205\o210"
| "&macr;" -> "\o302\o257"
| "&#x000AF;" -> "\o302\o257"
| "&#175;" -> "\o302\o257"
| "&#x02153;" -> "\o342\o205\o223"
| "&#8531;" -> "\o342\o205\o223"
| "&#x02154;" -> "\o342\o205\o224"
| "&#8532;" -> "\o342\o205\o224"
| "&#x02155;" -> "\o342\o205\o225"
| "&#8533;" -> "\o342\o205\o225"
| "&#x02156;" -> "\o342\o205\o226"
| "&#8534;" -> "\o342\o205\o226"
| "&#x02157;" -> "\o342\o205\o227"
| "&#8535;" -> "\o342\o205\o227"
| "&#x02158;" -> "\o342\o205\o230"
| "&#8536;" -> "\o342\o205\o230"
| "&#x02159;" -> "\o342\o205\o231"
| "&#8537;" -> "\o342\o205\o231"
| "&#x0215A;" -> "\o342\o205\o232"
| "&#8538;" -> "\o342\o205\o232"
| "&#x0215B;" -> "\o342\o205\o233"
| "&#8539;" -> "\o342\o205\o233"
| "&#x0215C;" -> "\o342\o205\o234"
| "&#8540;" -> "\o342\o205\o234"
| "&#x0215D;" -> "\o342\o205\o235"
| "&#8541;" -> "\o342\o205\o235"
| "&#x0215E;" -> "\o342\o205\o236"
| "&#8542;" -> "\o342\o205\o236"
| "&larr;" -> "\o342\o206\o220"
| "&#x02190;" -> "\o342\o206\o220"
| "&#8592;" -> "\o342\o206\o220"
| "&uarr;" -> "\o342\o206\o221"
| "&#x02191;" -> "\o342\o206\o221"
| "&#8593;" -> "\o342\o206\o221"
| "&deg;" -> "\o302\o260"
| "&#x000B0;" -> "\o302\o260"
| "&#176;" -> "\o302\o260"
| "&rarr;" -> "\o342\o206\o222"
| "&#x02192;" -> "\o342\o206\o222"
| "&#8594;" -> "\o342\o206\o222"
| "&darr;" -> "\o342\o206\o223"
| "&#x02193;" -> "\o342\o206\o223"
| "&#8595;" -> "\o342\o206\o223"
| "&plusmn;" -> "\o302\o261"
| "&#x000B1;" -> "\o302\o261"
| "&#177;" -> "\o302\o261"
| "&harr;" -> "\o342\o206\o224"
| "&#x02194;" -> "\o342\o206\o224"
| "&#8596;" -> "\o342\o206\o224"
| "&#x02195;" -> "\o342\o206\o225"
| "&#8597;" -> "\o342\o206\o225"
| "&#x02196;" -> "\o342\o206\o226"
| "&#8598;" -> "\o342\o206\o226"
| "&#x02197;" -> "\o342\o206\o227"
| "&#8599;" -> "\o342\o206\o227"
| _ -> s

let decode s = Str.global_substitute entity_regex (fun m ->
  decode_entity (Str.matched_string m)
) s

let rec decode_expr (p, e) = (p, decode_expr_ e)

and decode_expr_ e = match e with
  | Ast.String (p, s) -> Ast.String (p, decode s)
  | Ast.Xml (id, attrs, children) ->
      let dec_attrs = List.map (Ast_utils.map_xhp_attr (fun x -> x) decode_expr) attrs in
      let dec_children = List.map decode_expr children in
      Ast.Xml (id, dec_attrs, dec_children)
  | _ -> e
