# -*- coding: utf-8 -*-
#
# diffoscope: in-depth comparison of files, archives, and directories
#
# Copyright © 2016 Emanuel Bronshtein <e3amn2l@gmx.com>
#
# diffoscope is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# diffoscope is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with diffoscope.  If not, see <https://www.gnu.org/licenses/>.

import pytest
import subprocess

from diffoscope.config import Config
from diffoscope.comparators.openssh import PublicKeyFile
from diffoscope.comparators.missing_file import MissingFile

from ..utils.data import load_fixture, get_data
from ..utils.tools import skip_unless_tools_exist, skip_unless_tool_is_at_least


# Generated by: ssh-keygen -t dsa -C "Test1"
opensshpubkey1 = load_fixture('test_openssh_pub_key1.pub')
# Generated by: ssh-keygen -t rsa -b 4096 -C "Test2"
opensshpubkey2 = load_fixture('test_openssh_pub_key2.pub')


def openssh_version():
    out = subprocess.check_output(('ssh', '-V'), stderr=subprocess.STDOUT)
    return out.decode().split()[0].split('_')[1]


def test_identification(opensshpubkey1):
    assert isinstance(opensshpubkey1, PublicKeyFile)

def test_no_differences(opensshpubkey1):
    difference = opensshpubkey1.compare(opensshpubkey1)
    assert difference is None

@pytest.fixture
def differences(opensshpubkey1, opensshpubkey2):
    return opensshpubkey1.compare(opensshpubkey2).details

@skip_unless_tool_is_at_least('ssh-keygen', openssh_version, '6.9')
def test_diff(differences):
    expected_diff = get_data('openssh_pub_key_expected_diff')
    assert differences[0].unified_diff == expected_diff

@skip_unless_tools_exist('ssh-keygen')
def test_compare_non_existing(monkeypatch, opensshpubkey1):
    monkeypatch.setattr(Config(), 'new_file', True)
    difference = opensshpubkey1.compare(MissingFile('/nonexisting', opensshpubkey1))
    assert difference.source2 == '/nonexisting'
    assert len(difference.details) > 0
