# -*- perl -*-
# dgit
# Debian::Dgit: functions common to dgit and its helpers and servers
#
# Copyright (C) 2015-2016  Ian Jackson
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

package Debian::Dgit;

use strict;
use warnings;

use Carp;
use POSIX;
use IO::Handle;
use Config;
use Digest::SHA;
use Data::Dumper;
use IPC::Open2;
use File::Path;
use File::Basename;

BEGIN {
    use Exporter   ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT      = qw(setup_sigwarn forkcheck_setup forkcheck_mainprocess
		      dep14_version_mangle
                      debiantags debiantag_old debiantag_new
		      server_branch server_ref
                      stat_exists link_ltarget
		      hashfile
                      fail failmsg ensuredir must_getcwd executable_on_path
                      waitstatusmsg failedcmd_waitstatus
		      failedcmd_report_cmd failedcmd
                      runcmd shell_cmd cmdoutput cmdoutput_errok
                      git_rev_parse git_cat_file
		      git_get_ref git_get_symref git_for_each_ref
                      git_for_each_tag_referring is_fast_fwd
		      git_check_unmodified
                      $package_re $component_re $deliberately_re
		      $distro_re $versiontag_re $series_filename_re
		      $extra_orig_namepart_re
		      $git_null_obj
                      $branchprefix
		      $ffq_refprefix $gdrlast_refprefix
                      initdebug enabledebug enabledebuglevel
                      printdebug debugcmd
                      $debugprefix *debuglevel *DEBUG
                      shellquote printcmd messagequote
                      $negate_harmful_gitattrs
		      changedir git_slurp_config_src
		      gdr_ffq_prev_branchinfo
		      playtree_setup);
    # implicitly uses $main::us
    %EXPORT_TAGS = ( policyflags => [qw(NOFFCHECK FRESHREPO NOCOMMITCHECK)],
		     playground => [qw(record_maindir $maindir $local_git_cfg
				       $maindir_gitdir $maindir_gitcommon
				       fresh_playground
                                       ensure_a_playground)]);
    @EXPORT_OK   = ( @{ $EXPORT_TAGS{policyflags} },
		     @{ $EXPORT_TAGS{playground} } );
}

our @EXPORT_OK;

our $package_re = '[0-9a-z][-+.0-9a-z]*';
our $component_re = '[0-9a-zA-Z][-+.0-9a-zA-Z]*';
our $deliberately_re = "(?:TEST-)?$package_re";
our $distro_re = $component_re;
our $versiontag_re = qr{[-+.\%_0-9a-zA-Z/]+};
our $branchprefix = 'dgit';
our $series_filename_re = qr{(?:^|\.)series(?!\n)$}s;
our $extra_orig_namepart_re = qr{[-0-9a-z]+};
our $git_null_obj = '0' x 40;
our $ffq_refprefix = 'ffq-prev';
our $gdrlast_refprefix = 'debrebase-last';

# policy hook exit status bits
# see dgit-repos-server head comment for documentation
# 1 is reserved in case something fails with `exit 1' and to spot
# dynamic loader, runtime, etc., failures, which report 127 or 255
sub NOFFCHECK () { return 0x2; }
sub FRESHREPO () { return 0x4; }
sub NOCOMMITCHECK () { return 0x8; }

our $debugprefix;
our $debuglevel = 0;

our $negate_harmful_gitattrs =
    "-text -eol -crlf -ident -filter -working-tree-encoding";
    # ^ when updating this, alter the regexp in dgit:is_gitattrs_setup

our $forkcheck_mainprocess;

sub forkcheck_setup () {
    $forkcheck_mainprocess = $$;
}

sub forkcheck_mainprocess () {
    # You must have called forkcheck_setup or setup_sigwarn already
    getppid != $forkcheck_mainprocess;
}

sub setup_sigwarn () {
    forkcheck_setup();
    $SIG{__WARN__} = sub { 
	confess $_[0] if forkcheck_mainprocess;
    };
}

sub initdebug ($) { 
    ($debugprefix) = @_;
    open DEBUG, ">/dev/null" or die $!;
}

sub enabledebug () {
    open DEBUG, ">&STDERR" or die $!;
    DEBUG->autoflush(1);
    $debuglevel ||= 1;
}
    
sub enabledebuglevel ($) {
    my ($newlevel) = @_; # may be undef (eg from env var)
    die if $debuglevel;
    $newlevel //= 0;
    $newlevel += 0;
    return unless $newlevel;
    $debuglevel = $newlevel;
    enabledebug();
}
    
sub printdebug {
    print DEBUG $debugprefix, @_ or die $! if $debuglevel>0;
}

sub messagequote ($) {
    local ($_) = @_;
    s{\\}{\\\\}g;
    s{\n}{\\n}g;
    s{\x08}{\\b}g;
    s{\t}{\\t}g;
    s{[\000-\037\177]}{ sprintf "\\x%02x", ord $& }ge;
    $_;
}

sub shellquote {
    my @out;
    local $_;
    defined or confess 'internal error' foreach @_;
    foreach my $a (@_) {
	$_ = $a;
	if (!length || m{[^-=_./:0-9a-z]}i) {
	    s{['\\]}{'\\$&'}g;
	    push @out, "'$_'";
	} else {
	    push @out, $_;
	}
    }
    return join ' ', @out;
}

sub printcmd {
    my $fh = shift @_;
    my $intro = shift @_;
    print $fh $intro," " or die $!;
    print $fh shellquote @_ or die $!;
    print $fh "\n" or die $!;
}

sub debugcmd {
    my $extraprefix = shift @_;
    printcmd(\*DEBUG,$debugprefix.$extraprefix,@_) if $debuglevel>0;
}

sub dep14_version_mangle ($) {
    my ($v) = @_;
    # DEP-14 patch proposed 2016-11-09  "Version Mangling"
    $v =~ y/~:/_%/;
    $v =~ s/\.(?=\.|$|lock$)/.#/g;
    return $v;
}

sub debiantag_old ($$) { 
    my ($v,$distro) = @_;
    return "$distro/". dep14_version_mangle $v;
}

sub debiantag_new ($$) { 
    my ($v,$distro) = @_;
    return "archive/$distro/".dep14_version_mangle $v;
}

sub debiantags ($$) {
    my ($version,$distro) = @_;
    map { $_->($version, $distro) } (\&debiantag_new, \&debiantag_old);
}

sub server_branch ($) { return "$branchprefix/$_[0]"; }
sub server_ref ($) { return "refs/".server_branch($_[0]); }

sub stat_exists ($) {
    my ($f) = @_;
    return 1 if stat $f;
    return 0 if $!==&ENOENT;
    die "stat $f: $!";
}

sub _us () {
    $::us // ($0 =~ m#[^/]*$#, $&);
}

sub failmsg {
    my $s = "error: @_\n";
    $s =~ s/\n\n$/\n/;
    my $prefix = _us().": ";
    $s =~ s/^/$prefix/gm;
    return "\n".$s;
}

sub fail {
    die failmsg @_;
}

sub ensuredir ($) {
    my ($dir) = @_; # does not create parents
    return if mkdir $dir;
    return if $! == EEXIST;
    die "mkdir $dir: $!";
}

sub must_getcwd () {
    my $d = getcwd();
    defined $d or fail "getcwd failed: $!";
    return $d;
}

sub executable_on_path ($) {
    my ($program) = @_;
    return 1 if $program =~ m{/};
    my @path = split /:/, ($ENV{PATH} // "/usr/local/bin:/bin:/usr/bin");
    foreach my $pe (@path) {
	my $here = "$pe/$program";
	return $here if stat_exists $here && -x _;
    }
    return undef;
}

our @signames = split / /, $Config{sig_name};

sub waitstatusmsg () {
    if (!$?) {
	return "terminated, reporting successful completion";
    } elsif (!($? & 255)) {
	return "failed with error exit status ".WEXITSTATUS($?);
    } elsif (WIFSIGNALED($?)) {
	my $signum=WTERMSIG($?);
	return "died due to fatal signal ".
	    ($signames[$signum] // "number $signum").
	    ($? & 128 ? " (core dumped)" : ""); # POSIX(3pm) has no WCOREDUMP
    } else {
	return "failed with unknown wait status ".$?;
    }
}

sub failedcmd_report_cmd {
    my $intro = shift @_;
    $intro //= "failed command";
    { local ($!); printcmd \*STDERR, _us().": $intro:", @_ or die $!; };
}

sub failedcmd_waitstatus {
    if ($? < 0) {
	return "failed to fork/exec: $!";
    } elsif ($?) {
	return "subprocess ".waitstatusmsg();
    } else {
	return "subprocess produced invalid output";
    }
}

sub failedcmd {
    # Expects $!,$? as set by close - see below.
    # To use with system(), set $?=-1 first.
    #
    # Actual behaviour of perl operations:
    #   success              $!==0       $?==0       close of piped open
    #   program failed       $!==0       $? >0       close of piped open
    #   syscall failure      $! >0       $?=-1       close of piped open
    #   failure              $! >0       unchanged   close of something else
    #   success              trashed     $?==0       system
    #   program failed       trashed     $? >0       system
    #   syscall failure      $! >0       unchanged   system
    failedcmd_report_cmd undef, @_;
    fail failedcmd_waitstatus();
}

sub runcmd {
    debugcmd "+",@_;
    $!=0; $?=-1;
    failedcmd @_ if system @_;
}

sub shell_cmd {
    my ($first_shell, @cmd) = @_;
    return qw(sh -ec), $first_shell.'; exec "$@"', 'x', @cmd;
}

sub cmdoutput_errok {
    confess Dumper(\@_)." ?" if grep { !defined } @_;
    debugcmd "|",@_;
    open P, "-|", @_ or die "$_[0] $!";
    my $d;
    $!=0; $?=0;
    { local $/ = undef; $d = <P>; }
    die $! if P->error;
    if (!close P) { printdebug "=>!$?\n"; return undef; }
    chomp $d;
    if ($debuglevel > 0) {
	$d =~ m/^.*/;
	my $dd = $&;
	my $more = (length $' ? '...' : ''); #');
	$dd =~ s{[^\n -~]|\\}{ sprintf "\\x%02x", ord $& }ge;
	printdebug "=> \`$dd'",$more,"\n";
    }
    return $d;
}

sub cmdoutput {
    my $d = cmdoutput_errok @_;
    defined $d or failedcmd @_;
    return $d;
}

sub link_ltarget ($$) {
    my ($old,$new) = @_;
    lstat $old or return undef;
    if (-l _) {
	$old = cmdoutput qw(realpath  --), $old;
    }
    my $r = link $old, $new;
    $r = symlink $old, $new if !$r && $!==EXDEV;
    $r or die "(sym)link $old $new: $!";
}

sub hashfile ($) {
    my ($fn) = @_;
    my $h = Digest::SHA->new(256);
    $h->addfile($fn);
    return $h->hexdigest();
}

sub git_rev_parse ($) {
    return cmdoutput qw(git rev-parse), "$_[0]~0";
}

sub git_cat_file ($;$) {
    my ($objname, $etype) = @_;
    # => ($type, $data) or ('missing', undef)
    # in scalar context, just the data
    # if $etype defined, dies unless type is $etype or in @$etype
    our ($gcf_pid, $gcf_i, $gcf_o);
    my $chk = sub {
	my ($gtype, $data) = @_;
	if ($etype) {
	    $etype = [$etype] unless ref $etype;
	    confess "$objname expected @$etype but is $gtype"
		unless grep { $gtype eq $_ } @$etype;
	}
	return ($gtype, $data);
    };
    if (!$gcf_pid) {
	my @cmd = qw(git cat-file --batch);
	debugcmd "GCF|", @cmd;
	$gcf_pid = open2 $gcf_o, $gcf_i, @cmd or die $!;
    }
    printdebug "GCF>| ", $objname, "\n";
    print $gcf_i $objname, "\n" or die $!;
    my $x = <$gcf_o>;
    printdebug "GCF<| ", $x;
    if ($x =~ m/ (missing)$/) { return $chk->($1, undef); }
    my ($type, $size) = $x =~ m/^.* (\w+) (\d+)\n/ or die "$objname ?";
    my $data;
    (read $gcf_o, $data, $size) == $size or die "$objname $!";
    $x = <$gcf_o>;
    $x eq "\n" or die "$objname ($_) $!";
    return $chk->($type, $data);
}

sub git_get_symref (;$) {
    my ($symref) = @_;  $symref //= 'HEAD';
    # => undef if not a symref, otherwise refs/...
    my @cmd = (qw(git symbolic-ref -q HEAD));
    my $branch = cmdoutput_errok @cmd;
    if (!defined $branch) {
	$?==256 or failedcmd @cmd;
    } else {
	chomp $branch;
    }
    return $branch;
}

sub git_for_each_ref ($$;$) {
    my ($pattern,$func,$gitdir) = @_;
    # calls $func->($objid,$objtype,$fullrefname,$reftail);
    # $reftail is RHS of ref after refs/[^/]+/
    # breaks if $pattern matches any ref `refs/blah' where blah has no `/'
    # $pattern may be an array ref to mean multiple patterns
    $pattern = [ $pattern ] unless ref $pattern;
    my @cmd = (qw(git for-each-ref), @$pattern);
    if (defined $gitdir) {
	@cmd = ('sh','-ec','cd "$1"; shift; exec "$@"','x', $gitdir, @cmd);
    }
    open GFER, "-|", @cmd or die $!;
    debugcmd "|", @cmd;
    while (<GFER>) {
	chomp or die "$_ ?";
	printdebug "|> ", $_, "\n";
	m#^(\w+)\s+(\w+)\s+(refs/[^/]+/(\S+))$# or die "$_ ?";
	$func->($1,$2,$3,$4);
    }
    $!=0; $?=0; close GFER or die "$pattern $? $!";
}

sub git_get_ref ($) {
    # => '' if no such ref
    my ($refname) = @_;
    local $_ = $refname;
    s{^refs/}{[r]efs/} or die "$refname $_ ?";
    return cmdoutput qw(git for-each-ref --format=%(objectname)), $_;
}

sub git_for_each_tag_referring ($$) {
    my ($objreferring, $func) = @_;
    # calls $func->($tagobjid,$refobjid,$fullrefname,$tagname);
    printdebug "git_for_each_tag_referring ",
        ($objreferring // 'UNDEF'),"\n";
    git_for_each_ref('refs/tags', sub {
	my ($tagobjid,$objtype,$fullrefname,$tagname) = @_;
	return unless $objtype eq 'tag';
	my $refobjid = git_rev_parse $tagobjid;
	return unless
	    !defined $objreferring # caller wants them all
	    or $tagobjid eq $objreferring
	    or $refobjid eq $objreferring;
	$func->($tagobjid,$refobjid,$fullrefname,$tagname);
    });
}

sub git_check_unmodified () {
    foreach my $cached (qw(0 1)) {
	my @cmd = qw(git diff --quiet);
	push @cmd, qw(--cached) if $cached;
	push @cmd, qw(HEAD);
	debugcmd "+",@cmd;
	$!=0; $?=-1; system @cmd;
	return if !$?;
	if ($?==256) {
	    fail
		$cached
		? "git index contains changes (does not match HEAD)"
		: "working tree is dirty (does not match HEAD)";
	} else {
	    failedcmd @cmd;
	}
    }
}

sub is_fast_fwd ($$) {
    my ($ancestor,$child) = @_;
    my @cmd = (qw(git merge-base), $ancestor, $child);
    my $mb = cmdoutput_errok @cmd;
    if (defined $mb) {
	return git_rev_parse($mb) eq git_rev_parse($ancestor);
    } else {
	$?==256 or failedcmd @cmd;
	return 0;
    }
}

sub changedir ($) {
    my ($newdir) = @_;
    printdebug "CD $newdir\n";
    chdir $newdir or confess "chdir: $newdir: $!";
}

sub git_slurp_config_src ($) {
    my ($src) = @_;
    # returns $r such that $r->{KEY}[] = VALUE
    my @cmd = (qw(git config -z --get-regexp), "--$src", qw(.*));
    debugcmd "|",@cmd;

    local ($debuglevel) = $debuglevel-2;
    local $/="\0";

    my $r = { };
    open GITS, "-|", @cmd or die $!;
    while (<GITS>) {
	chomp or die;
	printdebug "=> ", (messagequote $_), "\n";
	m/\n/ or die "$_ ?";
	push @{ $r->{$`} }, $'; #';
    }
    $!=0; $?=0;
    close GITS
	or ($!==0 && $?==256)
	or failedcmd @cmd;
    return $r;
}

sub gdr_ffq_prev_branchinfo ($) {
    my ($symref) = @_;
    # => ('status', "message", [$symref, $ffq_prev, $gdrlast])
    # 'status' may be
    #    branch         message is undef
    #    weird-symref   } no $symref,
    #    notbranch      }  no $ffq_prev
    return ('detached', 'detached HEAD') unless defined $symref;
    return ('weird-symref', 'HEAD symref is not to refs/')
	unless $symref =~ m{^refs/};
    my $ffq_prev = "refs/$ffq_refprefix/$'";
    my $gdrlast = "refs/$gdrlast_refprefix/$'";
    printdebug "ffq_prev_branchinfo branch current $symref\n";
    return ('branch', undef, $symref, $ffq_prev, $gdrlast);
}

# ========== playground handling ==========

# terminology:
#
#   $maindir      user's git working tree
#   playground    area in .git/ where we can make files, unpack, etc. etc.
#   playtree      git working tree sharing object store with the user's
#                 inside playground, or identical to it
#
# other globals
#
#   $local_git_cfg    hash of arrays of values: git config from $maindir
#
# expected calling pattern
#
#  firstly
#
#    [record_maindir]
#      must be run in directory containing .git
#      assigns to $maindir if not already set
#      also calls git_slurp_config_src to record git config
#        in $local_git_cfg, unless it's already set
#
#    fresh_playground SUBDIR_PATH_COMPONENTS
#      e.g fresh_playground 'dgit/unpack' ('.git/' is implied)
#      default SUBDIR_PATH_COMPONENTS is playground_subdir
#      calls record_maindir
#      sets up a new playground (destroying any old one)
#      returns playground pathname
#      caller may call multiple times with different subdir paths
#       createing different playgrounds
#
#    ensure_a_playground SUBDIR_PATH_COMPONENTS
#      like fresh_playground except:
#      merely ensures the directory exists; does not delete an existing one
#
#  then can use
#
#    changedir playground
#    changedir $maindir
#
#    playtree_setup $local_git_cfg
#            # ^ call in some (perhaps trivial) subdir of playground
#
#    rmtree playground

# ----- maindir -----

# these three all go together
our $maindir;
our $maindir_gitdir;
our $maindir_gitcommon;

our $local_git_cfg;

sub record_maindir () {
    if (!defined $maindir) {
	$maindir = must_getcwd();
	if (!stat "$maindir/.git") {
	    fail "cannot stat $maindir/.git: $!";
	}
	if (-d _) {
	    # we fall back to this in case we have a pre-worktree
	    # git, which may not know git rev-parse --git-common-dir
	    $maindir_gitdir    = "$maindir/.git";
	    $maindir_gitcommon = "$maindir/.git";
	} else {
	    $maindir_gitdir    = cmdoutput qw(git rev-parse --git-dir);
	    $maindir_gitcommon = cmdoutput qw(git rev-parse --git-common-dir);
	}
    }
    $local_git_cfg //= git_slurp_config_src 'local';
}

# ----- playgrounds -----

sub ensure_a_playground_parent ($) {
    my ($spc) = @_;
    record_maindir();
    $spc = "$maindir_gitdir/$spc";
    my $parent = dirname $spc;
    mkdir $parent or $!==EEXIST
	or fail "failed to mkdir playground parent $parent: $!";
    return $spc;
}    

sub ensure_a_playground ($) {
    my ($spc) = @_;
    $spc = ensure_a_playground_parent $spc;
    mkdir $spc or $!==EEXIST or fail "failed to mkdir a playground $spc: $!";
    return $spc;
}    

sub fresh_playground ($) {
    my ($spc) = @_;
    $spc = ensure_a_playground_parent $spc;
    rmtree $spc;
    mkdir $spc or fail "failed to mkdir the playground $spc: $!";
    return $spc;
}

# ----- playtrees -----

sub playtree_setup (;$) {
    my ($t_local_git_cfg) = @_;
    $t_local_git_cfg //= $local_git_cfg;
    # for use in the playtree
    # $maindir must be set, eg by calling record_maindir or fresh_playground
    runcmd qw(git init -q);
    runcmd qw(git config gc.auto 0);
    foreach my $copy (qw(user.email user.name user.useConfigOnly
                         core.sharedRepository
                         core.compression core.looseCompression
                         core.bigFileThreshold core.fsyncObjectFiles)) {
	my $v = $t_local_git_cfg->{$copy};
	next unless $v;
	runcmd qw(git config), $copy, $_ foreach @$v;
    }
    # this is confusing: we have
    #   .                   playtree, not a worktree, has .git/, our cwd
    #   $maindir            might be a worktree so
    #   $maindir_gitdir     contains our main working "dgit", HEAD, etc.
    #   $maindir_gitcommon  the shared stuff, including .objects
    rmtree('.git/objects');
    symlink "$maindir_gitcommon/objects",'.git/objects' or die $!;
    ensuredir '.git/info';
    open GA, "> .git/info/attributes" or die $!;
    print GA "* $negate_harmful_gitattrs\n" or die $!;
    close GA or die $!;
}

1;
