# Copyright 2024 The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""qa workflow."""
from debusine.server.workflows import Workflow
from debusine.server.workflows.base import orchestrate_workflow
from debusine.server.workflows.models import (
    AutopkgtestWorkflowData,
    LintianWorkflowData,
    PiupartsWorkflowData,
    QAWorkflowData,
    WorkRequestWorkflowData,
)
from debusine.tasks.models import (
    BackendType,
    BaseDynamicTaskData,
    LintianFailOnSeverity,
    LookupMultiple,
    LookupSingle,
    TaskTypes,
)


class QAWorkflow(Workflow[QAWorkflowData, BaseDynamicTaskData]):
    """QA workflow."""

    TASK_NAME = "qa"

    def populate(self) -> None:
        """Create work requests."""
        architectures = set(self.get_architectures(self.data.binary_artifacts))

        if (data_archs := self.data.architectures) is not None:
            architectures.intersection_update(data_archs)

        if (
            data_archs_allowlist := self.data.architectures_allowlist
        ) is not None:
            architectures.intersection_update(data_archs_allowlist)

        if (
            data_archs_denylist := self.data.architectures_denylist
        ) is not None:
            architectures.difference_update(data_archs_denylist)

        filtered_binary_artifacts = self.filter_artifact_lookup(
            self.data.binary_artifacts, architectures
        )

        if self.data.enable_autopkgtest:
            self._populate_autopkgtest(
                source_artifact=self.data.source_artifact,
                binary_artifacts=filtered_binary_artifacts,
                vendor=self.data.vendor,
                codename=self.data.codename,
                backend=self.data.autopkgtest_backend,
                architectures=list(architectures),
            )

        if self.data.enable_lintian:
            self._populate_lintian(
                source_artifact=self.data.source_artifact,
                binary_artifacts=filtered_binary_artifacts,
                vendor=self.data.vendor,
                codename=self.data.codename,
                backend=self.data.lintian_backend,
                architectures=list(architectures),
                fail_on_severity=self.data.lintian_fail_on_severity,
            )

        if self.data.enable_piuparts:
            self._populate_piuparts(
                binary_artifacts=filtered_binary_artifacts,
                vendor=self.data.vendor,
                codename=self.data.codename,
                architectures=list(architectures),
                backend=self.data.piuparts_backend,
                arch_all_host_architecture=self.data.arch_all_host_architecture,
            )

    def _populate_autopkgtest(
        self,
        *,
        source_artifact: LookupSingle,
        binary_artifacts: LookupMultiple,
        vendor: str,
        codename: str,
        backend: BackendType,
        architectures: list[str],
    ) -> None:
        """Create work request for autopkgtest workflow."""
        wr = self.work_request_ensure_child(
            task_name="autopkgtest",
            task_type=TaskTypes.WORKFLOW,
            task_data=AutopkgtestWorkflowData(
                source_artifact=source_artifact,
                binary_artifacts=binary_artifacts,
                vendor=vendor,
                codename=codename,
                backend=backend,
                architectures=architectures,
            ).dict(exclude_unset=True),
            workflow_data=WorkRequestWorkflowData(
                display_name="autopkgtest",
                step="autopkgtest",
            ),
        )
        self.requires_artifact(wr, source_artifact)
        self.requires_artifact(wr, binary_artifacts)

        orchestrate_workflow(wr)

    def _populate_lintian(
        self,
        *,
        source_artifact: LookupSingle,
        binary_artifacts: LookupMultiple,
        vendor: str,
        codename: str,
        backend: BackendType,
        architectures: list[str],
        fail_on_severity: LintianFailOnSeverity,
    ) -> None:
        """Create work request for lintian workflow."""
        wr = self.work_request_ensure_child(
            task_name="lintian",
            task_type=TaskTypes.WORKFLOW,
            task_data=LintianWorkflowData(
                source_artifact=source_artifact,
                binary_artifacts=binary_artifacts,
                vendor=vendor,
                codename=codename,
                backend=backend,
                architectures=architectures,
                fail_on_severity=fail_on_severity,
            ).dict(exclude_unset=True),
            workflow_data=WorkRequestWorkflowData(
                display_name="lintian",
                step="lintian",
            ),
        )
        self.requires_artifact(wr, source_artifact)
        self.requires_artifact(wr, binary_artifacts)

        orchestrate_workflow(wr)

    def _populate_piuparts(
        self,
        *,
        binary_artifacts: LookupMultiple,
        vendor: str,
        codename: str,
        architectures: list[str],
        backend: BackendType,
        arch_all_host_architecture: str,
    ) -> None:
        wr = self.work_request_ensure_child(
            task_name="piuparts",
            task_type=TaskTypes.WORKFLOW,
            task_data=PiupartsWorkflowData(
                binary_artifacts=binary_artifacts,
                vendor=vendor,
                codename=codename,
                architectures=architectures,
                backend=backend,
                arch_all_host_architecture=arch_all_host_architecture,
            ).dict(exclude_unset=True),
            workflow_data=WorkRequestWorkflowData(
                display_name="piuparts",
                step="piuparts",
            ),
        )

        self.requires_artifact(wr, binary_artifacts)

        orchestrate_workflow(wr)

    def get_label(self) -> str:
        """Return the task label."""
        return "run QA"
