# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Tests for table columns."""

from django.template import Context

from debusine.db.models import User
from debusine.web.views.table import Ordering, Table
from debusine.web.views.table.columns import ColumnState
from debusine.web.views.table.tests.test_table import TableTestCase


class ColumnTests(TableTestCase):
    """Tests for :py:class:`Column`."""

    def get_table_class(self) -> type[Table[User]]:
        class _Table(Table[User]):
            default_column = "user"

            def init(self) -> None:
                super().init()
                self.add_column("user", "User", Ordering("username"))
                self.add_column("email", "Email", Ordering("email"))

        return _Table

    def test_current(self) -> None:
        """Test the current sorting column."""
        table = self._table(order="user", foo="bar")
        col = table.columns["user"]
        self.assertEqual(col.state, ColumnState.ASCENDING)
        self.assertEqual(col.query_asc, "order=user&foo=bar&asc=1")
        self.assertEqual(col.query_desc, "order=user&foo=bar&asc=0")
        self.assertHTMLValid(col.render(Context()))

    def test_inactive(self) -> None:
        """Test a simple use case."""
        table = self._table()
        col = table.columns["email"]
        self.assertIs(col.table, table)
        self.assertEqual(col.name, "email")
        self.assertEqual(col.title, "Email")
        self.assertEqual(col.state, ColumnState.INACTIVE)
        self.assertEqual(col.query_asc, "order=email&asc=1")
        self.assertEqual(col.query_desc, "order=email&asc=0")
        self.assertHTMLValid(col.render(Context()))
