# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Tabular presentation of QuerySets."""

from functools import cached_property
from typing import Generic, TYPE_CHECKING, TypeVar, cast

from django.core.exceptions import ImproperlyConfigured
from django.db.models import Model, QuerySet
from django.http import HttpRequest

from debusine.web.views.table.columns import Column, ColumnState
from debusine.web.views.table.ordering import Ordering

if TYPE_CHECKING:
    from django.core.paginator import _SupportsPagination

    from debusine.web.views.table.paginator import Paginator

    _SupportsPagination  # fake usage for vulture

M = TypeVar("M", bound=Model)


class Table(Generic[M]):
    """Definition of a tabular presentation of a QuerySet."""

    object_list: "_SupportsPagination[M]"
    columns: dict[str, Column]
    default_column: str
    template_name: str | None = None
    #: The column selected for sorting
    current_column: Column

    # TODO: declarative column definition?

    def __init__(
        self,
        request: HttpRequest,
        object_list: "_SupportsPagination[M]",
        *,
        prefix: str = "",
        default_column: str | None = None,
    ) -> None:
        """
        Build the table definition.

        :param request: the current request object
        :param object_list: the queryset to present
        :param prefix: the query string prefix for paginator arguments (set
                       this if you paginate multiple object lists in the same
                       page)
        :param default_column: name of the default column to use for sorting.
                               Prefix with '-' to make it descending.
        """
        self.request = request
        self.object_list = object_list
        self.prefix = prefix
        self.columns = {}
        self.init()
        # Override definition default with user-provided column name
        if default_column is not None:
            self.default_column = default_column
        self._post_init()

    def init(self) -> None:
        """
        Set up Table attributes.

        This function does nothing by default, and can be overridden by
        subclasses to customize the paginator for their specific use.
        """
        pass

    def _post_init(self) -> None:
        """Finish setting up after the table definition is complete."""
        if getattr(self, "default_column", None) is None:
            raise ImproperlyConfigured(
                "default_column not set in class or constructor arguments,"
                " and not set in init()"
            )

        self._init_current_column()

    def _init_current_column(self) -> None:
        """
        Set self.current_column.

        User selection from query string has precedence. Missing that,
        self.default_column is used.
        """
        # Try setting from query string
        if (
            (column_name := self.request.GET.get(f"{self.prefix}order"))
            and (column := self.columns.get(column_name))
            and (column.ordering is not None)
        ):
            ascending = self.request.GET.get(f"{self.prefix}asc", "1") != "0"
            column.state = (
                ColumnState.ASCENDING if ascending else ColumnState.DESCENDING
            )
            self.current_column = column
            return

        # Fall back to self.default_column
        if self.default_column.startswith("-"):
            state = ColumnState.DESCENDING
            column_name = self.default_column[1:]
        else:
            state = ColumnState.ASCENDING
            column_name = self.default_column
        if (column := self.columns.get(column_name)) is None:
            raise ImproperlyConfigured(
                f"default_column {self.default_column!r} is not"
                " a valid column name"
            )
        column.state = state
        self.current_column = column

    def __getitem__(self, name: str) -> Column:
        """Get a column by name."""
        return self.columns[name]

    def get_paginator(
        self,
        per_page: int,
        *,
        orphans: int = 0,
        allow_empty_first_page: bool = True,
    ) -> "Paginator[M]":
        """Return the Paginator for this table."""
        from debusine.web.views.table.paginator import Paginator

        return Paginator(
            self,
            per_page=per_page,
            orphans=orphans,
            allow_empty_first_page=allow_empty_first_page,
        )

    def add_column(
        self, name: str, title: str, ordering: Ordering | None = None
    ) -> Column:
        """Add a column definition."""
        self.columns[name] = col = Column(
            self,
            name=name,
            title=title,
            ordering=ordering,
        )
        return col

    @cached_property
    def rows(self) -> QuerySet[M, M]:
        """Return the queryset sorted as the user requested."""
        queryset: QuerySet[M, M] = cast(QuerySet[M, M], self.object_list)

        ordering = self.current_column.ordering
        # This is checked in _post_init
        assert ordering is not None
        if self.current_column.state == ColumnState.DESCENDING:
            return queryset.order_by(*ordering.desc)
        else:
            return queryset.order_by(*ordering.asc)
