# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Tests for the table ordering functions."""

from typing import cast

from django.db.models import F

from debusine.test.django import TestCase
from debusine.web.views.table.ordering import OrderBys, Ordering, order_by_tuple


class OrderByTupleTests(TestCase):
    """Tests for :py:func:`order_by_tuple`."""

    def test_single(self) -> None:
        for val in (
            "name",
            F("name"),
            F("name").asc(),
            ["name"],
            [F("name")],
            [F("name").asc()],
        ):
            with self.subTest(val=val):
                self.assertEqual(
                    str(order_by_tuple(val)),
                    "(OrderBy(F(name), descending=False),)",
                )

    def test_multi(self) -> None:
        for val in (
            ["name1", "name2"],
            [F("name1"), F("name2")],
            [F("name1").asc(), F("name2").asc()],
            cast(OrderBys, ["name1", F("name2").asc()]),
            cast(OrderBys, [F("name1"), "name2"]),
        ):
            with self.subTest(val=val):
                self.assertEqual(
                    str(order_by_tuple(val)),
                    "(OrderBy(F(name1), descending=False),"
                    " OrderBy(F(name2), descending=False))",
                )


class OrderingTests(TestCase):
    """Tests for :py:class:`Ordering`."""

    def test_default_desc(self) -> None:
        """Test inferring descending with a single field."""
        for val in ("name", F("name"), F("name").asc()):
            with self.subTest(val=val):
                o = Ordering(val)
                self.assertEqual(
                    str(o.asc), "(OrderBy(F(name), descending=False),)"
                )
                self.assertEqual(
                    str(o.desc), "(OrderBy(F(name), descending=True),)"
                )

    def test_default_desc_multi(self) -> None:
        """Test inferring descending with a single field."""
        for val in (
            ("name1", "name2"),
            (F("name1"), F("name2")),
            (F("name1").asc(), F("name2").asc()),
        ):
            with self.subTest(val=val):
                o = Ordering(val)
                self.assertEqual(
                    str(o.asc),
                    "(OrderBy(F(name1), descending=False),"
                    " OrderBy(F(name2), descending=False))",
                )
                self.assertEqual(
                    str(o.desc),
                    "(OrderBy(F(name1), descending=True),"
                    " OrderBy(F(name2), descending=True))",
                )

    def test_asc_desc(self) -> None:
        """Test inferring descending with a single field."""
        o = Ordering("name1", "name2")
        self.assertEqual(str(o.asc), "(OrderBy(F(name1), descending=False),)")
        self.assertEqual(str(o.desc), "(OrderBy(F(name2), descending=False),)")

    def test_dash(self) -> None:
        """Test inferring descending with a single field."""
        o = Ordering("name", "-name")
        self.assertEqual(str(o.asc), "(OrderBy(F(name), descending=False),)")
        self.assertEqual(str(o.desc), "(OrderBy(F(name), descending=True),)")
