# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Table ordering definitions."""

from collections.abc import Sequence
from typing import TypeAlias, assert_never, cast

from django.db.models import F
from django.db.models.expressions import OrderBy

OrderBys: TypeAlias = str | F | OrderBy | Sequence[str | F | OrderBy]


def to_order_by(val: str | F | OrderBy) -> OrderBy:
    """Convert the argument to an F-expression if needed."""
    match val:
        case str():
            if val.startswith("-"):
                return F(val[1:]).desc()
            else:
                return F(val).asc()
        case F():
            return val.asc()
        case OrderBy():
            return val
        case _ as unreachable:
            assert_never(unreachable)


def order_by_tuple(arg: OrderBys) -> tuple[OrderBy, ...]:
    """Normalize an argument to a tuple of OrderBy expressions."""
    match arg:
        case str() | F() | OrderBy():
            return (to_order_by(arg),)
        case _:
            return tuple(to_order_by(x) for x in arg)


class Ordering:
    """OrderBy expression for ascending and descending ordering."""

    #: QuerySet.order_by arguments for ascending sort
    asc: tuple[OrderBy, ...]
    #: QuerySet.order_by arguments for descending sort
    desc: tuple[OrderBy, ...]

    def __init__(
        self,
        asc: OrderBys,
        desc: OrderBys | None = None,
    ) -> None:
        """Convert arguments to OrderBy expressions."""
        self.asc = order_by_tuple(asc)
        if desc is None:
            self.desc = tuple(
                cast(OrderBy, x.copy().reverse_ordering()) for x in self.asc
            )
        else:
            self.desc = order_by_tuple(desc)
