#!/usr/bin/perl

#%# Copyright (C) 2014 Christoph Biedl <debian.axhn@manchmal.in-ulm.de>
#%# License: GPL-2.0-only

use 5.010;
use strict;
use warnings;

use Test::More;
use Test::Command;
use Test::Differences;

use File::Slurp;
use File::Temp qw(tempdir);

my $temp_dir = tempdir (
    "check-support-status.$$.XXXXX",
    'TMPDIR' => 1,
    'CLEANUP' => 1,
) or BAIL_OUT ('Cannot get a temporary directory');

sub write_tabbed ($$) {
    my ($file_name, $content) = @_;
    write_file (
        $file_name,
        map {
            join ("\t", split (/\s+/, $_)) . "\n"
        } split (/\n/, $content),
    );
}

# work around features missing in Test::Command before 0.11
sub stdout_n_stderr ($) {
    my $run = shift;
    return (
        $run->can ('stdout_value') ? $run->stdout_value :
            scalar read_file ($run->{'result'}{'stdout_file'}),
        $run->can ('stderr_value') ? $run->stderr_value :
            scalar read_file ($run->{'result'}{'stderr_file'}),
    );
}

my $list = "$temp_dir/list";
my $query_result = "$temp_dir/query-result";
my $semaphore_file = "$temp_dir/semaphores";
$ENV{'DPKG_QUERY_FILE'} = $query_result;


diag ('security support ended checks');

my @EXE = (
    './check-support-status',
    '--type', 'ended',
    '--no-heading',
    '--dpkg-query', 't/data/dpkg-query',
    '--list', $list,
    '--semaphore', $semaphore_file,
);


{
    # a single binary package, binary name = source name
    write_file ($list, <<__EOS__);
iceweasel       3.5.16-20       2013-05-01
base-files      6.0squeeze9     2014-05-01  Some spaced  explanation
debconf         1.5.36.0        2014-05-02
__EOS__
    write_tabbed ($query_result, <<__EOS__
base-files      6.0squeeze9
debconf         1.5.36.1
debconf-i18n    1.5.36.1        debconf
__EOS__
    );

    # run a first time
    my $run = Test::Command->new ('cmd' => \@EXE);
    $run->run;
    $run->exit_is_num (0);
    my ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    my $expect_stdout = <<__EOS__;

* Source:base-files, ended on 2014-05-01 at version 6.0squeeze9
  Details: Some spaced  explanation
  Affected binary package:
  - base-files (installed version: 6.0squeeze9)
__EOS__

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );

    if (ok (-f $semaphore_file, 'semaphore file was created')) {
        my $got = read_file ($semaphore_file);
        my $expect = "base-files/6.0squeeze9\n";
        eq_or_diff (
            $got,
            $expect,
            'semaphore file content',
        );
    }

    ## run a second time
    $run->run;
    $run->exit_is_num (0);
    ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    eq_or_diff (
        $stdout,
        '',
        'stdout'
    );

    # semaphore file should be unchanged
    if (ok (-f $semaphore_file, 'semaphore file exists')) {
        my $got = read_file ($semaphore_file);
        my $expect = <<__EOS__;
base-files/6.0squeeze9
__EOS__
        eq_or_diff (
            $got,
            $expect,
            'semaphore file content',
        );
    }

    ## run a third time

    # but create fake records in the semaphore file
    write_file ($semaphore_file, <<__EOS__);
base-files/6.0squeeze7
base-files/6.0squeeze8
__EOS__

    $run->run;
    $run->exit_is_num (0);
    ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );

    # semaphore file should have one line now
    if (ok (-f $semaphore_file, 'semaphore file exists')) {
        my $got = read_file ($semaphore_file);
        my $expect = "base-files/6.0squeeze9\n";
        eq_or_diff (
            $got,
            $expect,
            'semaphore file content',
        );
    }
}

{
    # several binary packages from same source
    write_file ($list, <<__EOS__);
# comments are allowed, too
iceweasel       3.5.16-20       2013-05-01
debconf         1.5.36.1        2014-05-02
__EOS__
    write_tabbed ($query_result, <<__EOS__
base-files      6.0squeeze9
debconf         1.5.36.1
debconf-i18n    1.5.36.1        debconf
__EOS__
    );
    unlink ($semaphore_file);

    # run a first time
    my $run = Test::Command->new ('cmd' => \@EXE);
    $run->run;
    $run->exit_is_num (0);
    my ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    my $expect_stdout = <<__EOS__;

* Source:debconf, ended on 2014-05-02 at version 1.5.36.1
  Affected binary packages:
  - debconf (installed version: 1.5.36.1)
  - debconf-i18n (installed version: 1.5.36.1)
__EOS__

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );

    if (ok (-f $semaphore_file, 'semaphore file was created')) {
        my $got = read_file ($semaphore_file);
        my $expect = <<__EOS__;
debconf/1.5.36.1
debconf-i18n/1.5.36.1
__EOS__
        eq_or_diff (
            $got,
            $expect,
            'semaphore file content',
        );
    }
}

{
    diag ('Bug#748954');

    # Assert any kind of spacing at the beginning the the explanative
    # is handled as expected

    write_tabbed ($query_result, <<__EOS__
openswan      1:2.6.28+dfsg-5+squeeze2
__EOS__
    );
    my %tests = (
        'space1' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31 Not supported in squeeze LTS\n",
        'space2' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31  Not supported in squeeze LTS\n",
        'space2' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31   Not supported in squeeze LTS\n",
        'tabbed' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31\tNot supported in squeeze LTS\n",
        'tabbed2' =>
"openswan 1:2.6.28+dfsg-5+squeeze2 2014-05-31\t\tNot supported in squeeze LTS\n",
    );

    for my $test_name (sort keys %tests) {
        unlink ($semaphore_file);
        write_file ($list, $tests{$test_name});

        my $run = Test::Command->new ('cmd' => \@EXE);
        $run->run;
        $run->exit_is_num (0);
        my ($stdout, $stderr) = stdout_n_stderr ($run);

        $stderr and diag ("stderr:\n" . $stderr);

        my $expect_stdout = <<__EOS__;

* Source:openswan, ended on 2014-05-31 at version 1:2.6.28+dfsg-5+squeeze2
  Details: Not supported in squeeze LTS
  Affected binary package:
  - openswan (installed version: 1:2.6.28+dfsg-5+squeeze2)
__EOS__

        eq_or_diff (
            $stdout,
            $expect_stdout,
            "$test_name, stdout",
        );
    }
}


diag ('limited support checks');

@EXE = (
    './check-support-status',
    '--type', 'limited',
    '--no-heading',
    '--dpkg-query', 't/data/dpkg-query',
    '--list', $list,
    '--semaphore', $semaphore_file,
);

{
    write_file ($list, <<__EOS__);
php5    See README.Debian.security for the PHP security policy
__EOS__
    write_tabbed ($query_result, <<__EOS__
php5            5.3.3-7+squeeze19
__EOS__
    );
    unlink ($semaphore_file);

    # run a first time
    my $run = Test::Command->new ('cmd' => \@EXE);
    $run->run;
    $run->exit_is_num (0);
    my ($stdout, $stderr) = stdout_n_stderr ($run);

    $stderr and diag ("stderr:\n" . $stderr);

    my $expect_stdout = <<__EOS__;

* Source:php5
  Details: See README.Debian.security for the PHP security policy
  Affected binary package:
  - php5 (installed version: 5.3.3-7+squeeze19)
__EOS__

    eq_or_diff (
        $stdout,
        $expect_stdout,
        'stdout'
    );

    if (ok (-f $semaphore_file, 'semaphore file was created')) {
        my $got = read_file ($semaphore_file);
        my $expect = <<__EOS__;
php5/5.3.3-7+squeeze19
__EOS__
        eq_or_diff (
            $got,
            $expect,
            'semaphore file content',
        );
    }
}

{
    diag ('Bug#748954');

    # Assert any kind of spacing at the beginning the the explanative
    # is handled as expected

    write_tabbed ($query_result, <<__EOS__
php5            5.3.3-7+squeeze19
__EOS__
    );
    my %tests = (
        'space1' =>
"php5 See README.Debian.security for the PHP security policy\n",
        'space2' =>
"php5  See README.Debian.security for the PHP security policy\n",
        'space3' =>
"php5  See README.Debian.security for the PHP security policy\n",
        'tabbed' =>
"php5\tSee README.Debian.security for the PHP security policy\n",
        'tabbed2' =>
"php5\t\tSee README.Debian.security for the PHP security policy\n",
    );

    for my $test_name (sort keys %tests) {
        unlink ($semaphore_file);
        write_file ($list, $tests{$test_name});

        my $run = Test::Command->new ('cmd' => \@EXE);
        $run->run;
        $run->exit_is_num (0);
        my ($stdout, $stderr) = stdout_n_stderr ($run);

        $stderr and diag ("stderr:\n" . $stderr);

        my $expect_stdout = <<__EOS__;

* Source:php5
  Details: See README.Debian.security for the PHP security policy
  Affected binary package:
  - php5 (installed version: 5.3.3-7+squeeze19)
__EOS__

        eq_or_diff (
            $stdout,
            $expect_stdout,
            "$test_name, stdout",
        );
    }
}


done_testing;

exit 0;
