/*
 * Copyright (C) 2014-2015 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */


#include <QMessageBox>

#include "dlg_change_pwd.h"
#include "src/io/isds_sessions.h"
#include "src/models/accounts_model.h"


DlgChangePwd::DlgChangePwd(const QString &boxId, const QString &userName,
    QWidget *parent)
    : QDialog(parent),
    m_boxId(boxId),
    m_userName(userName)
{
	setupUi(this);
	initPwdChangeDialog();
}


/* ========================================================================= */
/*
 * Init dialog
 */
void DlgChangePwd::initPwdChangeDialog(void)
/* ========================================================================= */
{
	this->userNameLneEdit->setText(m_userName);
	this->accountLineEdit->setText(m_boxId);
	connect(this->generateButton, SIGNAL(clicked()), this,
	    SLOT(generatePassword()));
	connect(this->showHideButton, SIGNAL(clicked()), this,
	    SLOT(showHidePasswordLine()));

	this->buttonBox->button(QDialogButtonBox::Ok)->setEnabled(false);
	connect(this->newPwdLineEdit, SIGNAL(textChanged(QString)),
	    this, SLOT(checkInputFields()));
	connect(this->currentPwdLineEdit, SIGNAL(textChanged(QString)),
	    this, SLOT(checkInputFields()));
	connect(this->NewPwdLineEdit2, SIGNAL(textChanged(QString)),
	    this, SLOT(checkInputFields()));
	connect(this->secCodeLineEdit, SIGNAL(textChanged(QString)),
	    this, SLOT(checkInputFields()));
	connect(this->buttonBox, SIGNAL(accepted()), this,
	    SLOT(changePassword(void)));

	this->secCodeLineEdit->setEnabled(false);
	this->otpLabel->setEnabled(false);
	this->smsPushButton->setEnabled(false);

	if (AccountModel::globAccounts[m_userName].loginMethod() == LIM_HOTP) {
		this->secCodeLineEdit->setEnabled(true);
		this->otpLabel->setText(tr("Enter security code:"));
		this->otpLabel->setEnabled(true);
	}

	if (AccountModel::globAccounts[m_userName].loginMethod() == LIM_TOTP) {
		this->secCodeLineEdit->setEnabled(true);
		this->smsPushButton->setEnabled(true);
		this->otpLabel->setText(tr("Enter SMS code:"));
		this->otpLabel->setEnabled(true);
		connect(this->smsPushButton, SIGNAL(clicked()), this,
		    SLOT(sendSmsCode()));
	}

	pingTimer = new QTimer(this);
	pingTimer->start(DLG_ISDS_KEEPALIVE_MS);

	connect(pingTimer, SIGNAL(timeout()), this,
	    SLOT(pingIsdsServer()));
}


/* ========================================================================= */
/*
 * Ping isds server, test if connection on isds server is active
 */
void DlgChangePwd::pingIsdsServer(void)
/* ========================================================================= */
{
	if (isdsSessions.isConnectedToIsds(m_userName)) {
		qDebug() << "Connection to ISDS is alive :)";
	} else {
		qDebug() << "Connection to ISDS is dead :(";
	}
}

/* ========================================================================= */
/*
 * Fill the new password in the textlines
 */
void DlgChangePwd::generatePassword(void)
/* ========================================================================= */
{
	QString pwd = generateRandomString();
	this->newPwdLineEdit->setText(pwd);
	this->NewPwdLineEdit2->setText(pwd);
}


/* ========================================================================= */
/*
 * Generate a new password string from set of char
 */
QString DlgChangePwd::generateRandomString(void)
/* ========================================================================= */
{
	QString randomString;

	for(int i=0; i<randomStringLength; ++i) {
		int index = qrand() % possibleCharacters.length();
		QChar nextChar = possibleCharacters.at(index);
		randomString.append(nextChar);
	}
	/* set one digit as last char */
	return randomString + "0";
}


/* ========================================================================= */
/*
 * Check input textline, passwor length and activated OK button
 */
void DlgChangePwd::checkInputFields(void)
/* ========================================================================= */
{
	bool buttonEnabled = !this->currentPwdLineEdit->text().isEmpty() &&
	    !this->newPwdLineEdit->text().isEmpty() &&
	    this->newPwdLineEdit->text().length() >= PWD_MIN_LENGTH &&
	    !this->NewPwdLineEdit2->text().isEmpty() &&
	    this->NewPwdLineEdit2->text().length() >= PWD_MIN_LENGTH &&
	    this->NewPwdLineEdit2->text() == this->newPwdLineEdit->text();

	Q_ASSERT(!m_userName.isEmpty());

	if (AccountModel::globAccounts[m_userName].loginMethod() == LIM_HOTP ||
	    AccountModel::globAccounts[m_userName].loginMethod() == LIM_TOTP) {
		buttonEnabled = buttonEnabled &&
		    !this->secCodeLineEdit->text().isEmpty();
	}

	this->buttonBox->button(QDialogButtonBox::Ok)->setEnabled(
	    buttonEnabled);
}

/* ========================================================================= */
/*
 * Show/hide text represantion of password in the textlines
 */
void DlgChangePwd::showHidePasswordLine(void)
/* ========================================================================= */
{
	if (this->currentPwdLineEdit->echoMode() == QLineEdit::Password) {
		this->currentPwdLineEdit->setEchoMode(QLineEdit::Normal);
		this->newPwdLineEdit->setEchoMode(QLineEdit::Normal);
		this->NewPwdLineEdit2->setEchoMode(QLineEdit::Normal);
		this->showHideButton->setText(tr("Hide"));
	} else {
		this->currentPwdLineEdit->setEchoMode(QLineEdit::Password);
		this->newPwdLineEdit->setEchoMode(QLineEdit::Password);
		this->NewPwdLineEdit2->setEchoMode(QLineEdit::Password);
		this->showHideButton->setText(tr("Show"));
	}
}

/* ========================================================================= */
/*
 * Send SMS code request into ISDS
 */
void DlgChangePwd::sendSmsCode(void)
/* ========================================================================= */
{
	Q_ASSERT(!m_userName.isEmpty());

	/* show Premium SMS request dialog */
	QMessageBox::StandardButton reply = QMessageBox::question(this,
	    tr("SMS code for account ") +
	    AccountModel::globAccounts[m_userName].accountName(),
	    tr("Account \"%1\" requires authentication via security code "
	    "for connection to databox.")
	        .arg(AccountModel::globAccounts[m_userName].accountName())
	    + "<br/>" +
	    tr("Security code will be sent you via Premium SMS.") +
	    "<br/><br/>" +
	    tr("Do you want to send Premium SMS with "
	    "security code into your mobile phone?"),
	    QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes);

	if (reply == QMessageBox::No) {
		return;
	}

	struct isds_ctx *session = isdsSessions.session(m_userName);
	if (NULL == session) {
		Q_ASSERT(0);
		return;
	}

	isds_error status;
	char * refnumber = NULL;
	struct isds_otp *otp = NULL;
	otp = (struct isds_otp *) malloc(sizeof(struct isds_otp));
	memset(otp, 0, sizeof(struct isds_otp));
	otp->method = OTP_TIME;
	otp->otp_code = NULL;

	status = isds_change_password(session,
	    this->currentPwdLineEdit->text().toUtf8().constData(),
	    this->newPwdLineEdit->text().toUtf8().constData(),
	    otp, &refnumber);

	free(refnumber);
	free(otp->otp_code);
	free(otp);

	if (IE_PARTIAL_SUCCESS == status) {
		QMessageBox::information(this, tr("Enter SMS security code"),
		    tr("SMS security code for account \"%1\"<br/>"
		    "has been sent on your mobile phone...")
		    .arg(AccountModel::globAccounts[m_userName].accountName())
		     + "<br/><br/>" +
		    tr("Enter SMS security code for account")
		    + "<br/><b>"
		    + AccountModel::globAccounts[m_userName].accountName()
		    + " </b>(" + m_userName + ").",
		    QMessageBox::Ok);
		this->otpLabel->setText(tr("Enter SMS code:"));
		this->smsPushButton->setEnabled(false);
	} else {
		QMessageBox::critical(this, tr("Login error"),
		    tr("An error occurred while preparing "
		    "request for SMS with OTP security code.") +
		    "<br/><br/>" +
		    tr("Please try again later or you have to use the "
		    "official web interface of Datové schránky for "
		    "access to your data box."),
		    QMessageBox::Ok);
	}
}


/* ========================================================================= */
/*
 * Sent new password request into ISDS
 */
void DlgChangePwd::changePassword(void)
/* ========================================================================= */
{
	isds_error status;
	char * refnumber = NULL;

	struct isds_ctx *session = isdsSessions.session(m_userName);
	if (NULL == session) {
		Q_ASSERT(0);
		return;
	}

	if (AccountModel::globAccounts[m_userName].loginMethod() == LIM_HOTP ||
	    AccountModel::globAccounts[m_userName].loginMethod() == LIM_TOTP) {
		struct isds_otp *otp = NULL;
		otp = (struct isds_otp *) malloc(sizeof(struct isds_otp));
		memset(otp, 0, sizeof(struct isds_otp));

		if (AccountModel::globAccounts[m_userName].loginMethod() ==
		    LIM_HOTP) {
			otp->method = OTP_HMAC;
		} else {
			otp->method = OTP_TIME;
		}

		otp->otp_code = !this->secCodeLineEdit->text().isEmpty() ?
		    strdup(this->secCodeLineEdit->text().toUtf8().constData())
		    : NULL;

		status = isds_change_password(session,
		    this->currentPwdLineEdit->text().toUtf8().constData(),
		    this->newPwdLineEdit->text().toUtf8().constData(),
		    otp, &refnumber);

		free(otp->otp_code);
		free(otp);
	} else {
		status = isds_change_password(session,
		    this->currentPwdLineEdit->text().toUtf8().constData(),
		    this->newPwdLineEdit->text().toUtf8().constData(),
		    NULL, &refnumber);
	}

	free(refnumber);

	if (status == IE_SUCCESS) {
		QMessageBox::information(this, tr("Password has been changed"),
		    tr("Password has been changed "
		        "successfully on the server ISDS.")
		    + "\n\n" +
		    tr("Restart the application. Also don't forget to remember "
		        "the new password so you will still be able to log "
		        "into your data box via the web interface."),
		    QMessageBox::Ok);

		AccountModel::globAccounts[m_userName].setPassword(
		    this->newPwdLineEdit->text());

		/* TODO - delete and create new
		 * isds context with new settings
		 */
	} else {
		Q_ASSERT(!m_userName.isEmpty());
		QString error = tr("Error: ") + isds_strerror(status);
		QString isdslog = isds_long_message(session);
		if (!isdslog.isEmpty()) {
			error = tr("ISDS returns: ") + isdslog;
		}

		QMessageBox::warning(this, tr("Password error"),
		    tr("An error occurred while password was changed.")
		    + "\n\n" + error + "\n\n" +
		    tr("You have to fix the problem and try to again."),
		    QMessageBox::Ok);
	}
}


/* ========================================================================= */
/*
 * Set of possible chars for generation of new password
 */
const QString DlgChangePwd::possibleCharacters(
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    "abcdefghijklmnopqrstuvwxyz"
    "0123456789"
    "!#$%&()*+,-.:=?@[]_{|}~");
const int DlgChangePwd::randomStringLength = PWD_MIN_LENGTH;
/* ========================================================================= */
