/*
    This file is part of darktable,
    copyright (c) 2013 johannes hanika.

    darktable is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    darktable is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with darktable.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "common/image.h"
#include <string.h>

typedef struct dt_noiseprofile_t
{
  const char *name;
  const char *maker;
  const char *model;
  int iso;
  float a[3]; // poissonian part
  float b[3]; // gaussian part
}
dt_noiseprofile_t;

// these should be sorted by same maker/model and increasing iso
const dt_noiseprofile_t dt_noiseprofiles[] =
{
  // clang-format off

// preset name                   maker    model            iso    a a a                                 b b b
  {N_("generic poissonian"),       "",      "",              0,    {0.0001f, 0.0001f, 0.0001},           {0.0f, 0.0f, 0.0f}},

// -------------------------------------------------------------------
// Canon.
// -------------------------------------------------------------------

// canon eos 1d mark ii by cedric
  {"Canon EOS-1D Mark II iso 50",   "Canon", "Canon EOS-1D Mark II", 50,   {5.43759458841523e-06, 2.24638626295538e-06, 2.71899730492479e-06}, {-1.40561392601687e-07, -1.1669530186266e-07, -5.51165182845818e-08}},
// {"Canon EOS-1D Mark II iso 50",   "Canon", "Canon EOS-1D Mark II", 50,   {5.13356792619893e-06, 2.21119301823563e-06, 2.6179117472501e-06}, {-1.10151821053457e-07, -1.16624854597179e-07, -5.15937050248643e-08}},
  {"Canon EOS-1D Mark II iso 100",  "Canon", "Canon EOS-1D Mark II", 100,  {6.12685534716694e-06, 2.59784855091013e-06, 3.16382070553262e-06}, {-1.38924459341199e-07, -1.34147138206385e-07, -7.10172359862074e-08}},
// {"Canon EOS-1D Mark II iso 100",  "Canon", "Canon EOS-1D Mark II", 100,  {6.20111466218951e-06, 2.6095676278732e-06, 3.05753183060972e-06}, {-1.44833169656777e-07, -1.39197320858836e-07, -5.87732357985472e-08}},
  {"Canon EOS-1D Mark II iso 125",  "Canon", "Canon EOS-1D Mark II", 125,  {7.12229281080873e-06, 2.96412692431319e-06, 3.51064854856123e-06}, {-1.52811307935643e-07, -1.40815056032341e-07, -4.51171570082741e-08}},
// {"Canon EOS-1D Mark II iso 125",  "Canon", "Canon EOS-1D Mark II", 125,  {6.95988330619185e-06, 2.91084458068914e-06, 3.4623717337244e-06}, {-1.40714465376015e-07, -1.31810533520507e-07, -4.48371917096529e-08}},
  {"Canon EOS-1D Mark II iso 160",  "Canon", "Canon EOS-1D Mark II", 160,  {8.25769459892563e-06, 3.43721555598261e-06, 4.15574707952495e-06}, {-1.56966271793147e-07, -1.38460977579807e-07, -3.39889619346344e-08}},
// {"Canon EOS-1D Mark II iso 160",  "Canon", "Canon EOS-1D Mark II", 160,  {8.09360158677059e-06, 3.40687374716927e-06, 4.01671760397673e-06}, {-1.46925419200659e-07, -1.43721581855241e-07, -2.70409970211167e-08}},
  {"Canon EOS-1D Mark II iso 200",  "Canon", "Canon EOS-1D Mark II", 200,  {9.65554408183143e-06, 4.0359190965776e-06, 5.20192827470806e-06}, {-1.78892006773322e-07, -1.41188812904924e-07, -7.38655454985893e-08}},
// {"Canon EOS-1D Mark II iso 200",  "Canon", "Canon EOS-1D Mark II", 200,  {9.5972898238925e-06, 4.08760066003704e-06, 4.98696430640313e-06}, {-1.67861775784962e-07, -1.65727064844721e-07, -5.49646527601929e-08}},
  {"Canon EOS-1D Mark II iso 250",  "Canon", "Canon EOS-1D Mark II", 250,  {1.12476268583339e-05, 4.81109708338133e-06, 5.95413583221349e-06}, {-1.77603058698993e-07, -1.70505928788222e-07, -3.98627726233775e-08}},
// {"Canon EOS-1D Mark II iso 250",  "Canon", "Canon EOS-1D Mark II", 250,  {1.13073679866827e-05, 4.71426309533806e-06, 5.96266524002632e-06}, {-1.80452888235536e-07, -1.53686395811583e-07, -4.49435153050384e-08}},
  {"Canon EOS-1D Mark II iso 320",  "Canon", "Canon EOS-1D Mark II", 320,  {1.31882483769008e-05, 5.50184344135338e-06, 7.0930756434241e-06}, {-1.77153503693489e-07, -1.48643476126428e-07, -2.62449903266945e-08}},
// {"Canon EOS-1D Mark II iso 320",  "Canon", "Canon EOS-1D Mark II", 320,  {1.33791968026931e-05, 5.58119624218431e-06, 7.01146598381994e-06}, {-1.86806364999686e-07, -1.62695505172472e-07, -1.67407608561494e-08}},
  {"Canon EOS-1D Mark II iso 400",  "Canon", "Canon EOS-1D Mark II", 400,  {1.60150629163158e-05, 6.76433906236383e-06, 8.93491260047119e-06}, {-2.01556208474724e-07, -1.56649712082471e-07, -5.55581236280623e-08}},
// {"Canon EOS-1D Mark II iso 400",  "Canon", "Canon EOS-1D Mark II", 400,  {1.63389757557517e-05, 6.87143944659454e-06, 8.87354251539462e-06}, {-2.28389870464101e-07, -1.99831413492604e-07, -6.08366023590217e-08}},
  {"Canon EOS-1D Mark II iso 500",  "Canon", "Canon EOS-1D Mark II", 500,  {1.92500886659244e-05, 8.15918070462954e-06, 1.067104170575e-05}, {-2.12180139178311e-07, -1.77908879362265e-07, -2.70222807425058e-08}},
// {"Canon EOS-1D Mark II iso 500",  "Canon", "Canon EOS-1D Mark II", 500,  {1.95448502989152e-05, 8.31548072817153e-06, 1.05796309069549e-05}, {-2.11056094762947e-07, -2.16785830687682e-07, -2.58311770702918e-08}},
  {"Canon EOS-1D Mark II iso 640",  "Canon", "Canon EOS-1D Mark II", 640,  {2.32539072493961e-05, 9.78607422326848e-06, 1.23201955168683e-05}, {-2.35937344122421e-07, -1.99144689307414e-07, 6.8779842636897e-08}},
// {"Canon EOS-1D Mark II iso 640",  "Canon", "Canon EOS-1D Mark II", 640,  {2.31050590420975e-05, 9.93220524424988e-06, 1.25530843218975e-05}, {-2.49542455014965e-07, -2.2538598981614e-07, 5.39883301212723e-08}},
// {"Canon EOS-1D Mark II iso 640",  "Canon", "Canon EOS-1D Mark II", 640,  {2.33892980464104e-05, 9.85765301127244e-06, 1.24125817263289e-05}, {-2.59715167555753e-07, -2.29825080027701e-07, 3.25018068279896e-08}},
  {"Canon EOS-1D Mark II iso 800",  "Canon", "Canon EOS-1D Mark II", 800,  {2.85441231675095e-05, 1.20526601587326e-05, 1.58124714186487e-05}, {-2.51731714697906e-07, -2.22755719094829e-07, 4.65129003041861e-08}},
// {"Canon EOS-1D Mark II iso 800",  "Canon", "Canon EOS-1D Mark II", 800,  {2.84676908251911e-05, 1.19249442585498e-05, 1.53868595619681e-05}, {-1.9411601704993e-07, -1.73285325826641e-07, 9.47918909645027e-08}},
  {"Canon EOS-1D Mark II iso 1000", "Canon", "Canon EOS-1D Mark II", 1000, {3.45418997843976e-05, 1.44970518151978e-05, 1.91926426580989e-05}, {-1.77810688691049e-07, -1.12772829813669e-07, 1.90740759315603e-07}},
// {"Canon EOS-1D Mark II iso 1000", "Canon", "Canon EOS-1D Mark II", 1000, {3.4531717936251e-05, 1.45602805023053e-05, 1.90573103870564e-05}, {-1.29777063378645e-07, -1.08503537021703e-07, 1.77307171447416e-07}},
  {"Canon EOS-1D Mark II iso 1250", "Canon", "Canon EOS-1D Mark II", 1250, {4.14731391124046e-05, 1.72303428844692e-05, 2.28479811897812e-05}, {-9.77472503377245e-08, -7.8979546194721e-09, 3.75082172752967e-07}},
// {"Canon EOS-1D Mark II iso 1250", "Canon", "Canon EOS-1D Mark II", 1250, {4.13643631007913e-05, 1.74722917144243e-05, 2.26592066815217e-05}, {-4.26481181165303e-08, -5.38944409561333e-08, 3.80682548162181e-07}},
  {"Canon EOS-1D Mark II iso 1600", "Canon", "Canon EOS-1D Mark II", 1600, {5.14768610998129e-05, 2.16651294792692e-05, 2.86146675946199e-05}, {1.99892873564407e-07, 2.05394631529031e-07, 7.21531611278429e-07}},
// {"Canon EOS-1D Mark II iso 1600", "Canon", "Canon EOS-1D Mark II", 1600, {5.22635114088786e-05, 2.17733087267772e-05, 2.86791655761423e-05}, {1.22198041176339e-07, 1.60041604724974e-07, 6.56185427156001e-07}},
  {"Canon EOS-1D Mark II iso 3200", "Canon", "Canon EOS-1D Mark II", 3200, {0.000100840229405782, 4.25172448937779e-05, 5.77544620216772e-05}, {1.15854936341615e-06, 1.05390603833945e-06, 2.32709636212999e-06}},
// {"Canon EOS-1D Mark II iso 3200", "Canon", "Canon EOS-1D Mark II", 3200, {9.91570744088403e-05, 4.10147161617878e-05, 5.58635093420947e-05}, {1.44727844071082e-06, 1.3827135933762e-06, 2.42251215454214e-06}},
// {"Canon EOS-1D Mark II iso 3200", "Canon", "Canon EOS-1D Mark II", 3200, {0.000100559810968374, 4.17523224087309e-05, 5.50964562550622e-05}, {1.46595454279235e-06, 1.25093633941195e-06, 2.4950253666976e-06}},

//
  {"Canon EOS-1Ds Mark II iso 100",  "Canon", "Canon EOS-1Ds Mark II", 100,  {6.51470928483648e-06, 3.18468931448863e-06, 3.87773570974099e-06}, {-1.63702473902993e-07, -2.56512420302175e-07, -9.99855168396345e-08}},
  {"Canon EOS-1Ds Mark II iso 125",  "Canon", "Canon EOS-1Ds Mark II", 125,  {7.83731076230367e-06, 3.61202159302289e-06, 4.32722769527927e-06}, {-1.69377487183087e-07, -2.40449801731156e-07, -7.46569238202499e-08}},
  {"Canon EOS-1Ds Mark II iso 160",  "Canon", "Canon EOS-1Ds Mark II", 160,  {9.0274397715917e-06, 4.04031403384626e-06, 5.28605911498418e-06}, {-2.02238365695335e-07, -2.36930425931926e-07, -9.32942290775983e-08}},
  {"Canon EOS-1Ds Mark II iso 200",  "Canon", "Canon EOS-1Ds Mark II", 200,  {1.04177466593278e-05, 4.82101872735394e-06, 5.98730416790264e-06}, {-2.14598863495593e-07, -3.0993798081488e-07, -9.08571015787385e-08}},
  {"Canon EOS-1Ds Mark II iso 250",  "Canon", "Canon EOS-1Ds Mark II", 250,  {1.18563345892614e-05, 5.33105850581603e-06, 7.01587702761345e-06}, {-1.84553062554683e-07, -2.57571353447476e-07, -5.14672701488431e-08}},
  {"Canon EOS-1Ds Mark II iso 320",  "Canon", "Canon EOS-1Ds Mark II", 320,  {1.44975046895807e-05, 6.51777697821914e-06, 8.33491727780834e-06}, {-2.25671622569728e-07, -3.29825636508596e-07, -6.94982333598271e-08}},
  {"Canon EOS-1Ds Mark II iso 400",  "Canon", "Canon EOS-1Ds Mark II", 400,  {1.76439797226241e-05, 7.5260945462981e-06, 1.06826608131939e-05}, {-2.32664687314674e-07, -2.386558078267e-07, -8.57880840937389e-08}},
  {"Canon EOS-1Ds Mark II iso 500",  "Canon", "Canon EOS-1Ds Mark II", 500,  {2.02987780642822e-05, 8.91772612010865e-06, 1.17982566201638e-05}, {-2.39299010312377e-07, -3.35724827849638e-07, 4.58164144951259e-08}},
  {"Canon EOS-1Ds Mark II iso 640",  "Canon", "Canon EOS-1Ds Mark II", 640,  {2.52092725807517e-05, 1.08667031681006e-05, 1.47273224971139e-05}, {-2.77288437486969e-07, -3.36045717158296e-07, 1.4673130687958e-08}},
  {"Canon EOS-1Ds Mark II iso 800",  "Canon", "Canon EOS-1Ds Mark II", 800,  {3.02272437820821e-05, 1.28529931354699e-05, 1.85633115185683e-05}, {-2.61842020361295e-07, -2.69868656348702e-07, 1.0794570688356e-08}},
  {"Canon EOS-1Ds Mark II iso 1000", "Canon", "Canon EOS-1Ds Mark II", 1000, {3.69066892244919e-05, 1.58809216680767e-05, 2.20560471635755e-05}, {-2.47190029386922e-07, -3.62454972036591e-07, 9.13367521522064e-08}},
  {"Canon EOS-1Ds Mark II iso 1250", "Canon", "Canon EOS-1Ds Mark II", 1250, {4.48723532582542e-05, 1.94891786186444e-05, 2.72993940249479e-05}, {-1.64959339068835e-07, -3.14816260309544e-07, 2.54051672570559e-07}},
  {"Canon EOS-1Ds Mark II iso 1600", "Canon", "Canon EOS-1Ds Mark II", 1600, {5.27996019320874e-05, 2.21636686621325e-05, 3.31009973772538e-05}, {-8.77191966132312e-08, 1.23776555481282e-08, 4.7190086760678e-07}},
  {"Canon EOS-1Ds Mark II iso 3200", "Canon", "Canon EOS-1Ds Mark II", 3200, {0.000104338261298007, 4.43621740473327e-05, 6.38167343052592e-05}, {4.95130329445823e-07, 2.19593645490202e-07, 1.47886344355236e-06}},

// canon eos 1d mark iii contributed by bo zhang
  {"Canon EOS-1D Mark III iso 100",  "Canon", "Canon EOS-1D Mark III", 100,  {4.58296131724664e-06, 2.10162212763496e-06, 3.66892364711553e-06}, {-1.03318617310606e-07, -1.11429531551922e-07, -1.07045348865387e-07}},
  {"Canon EOS-1D Mark III iso 200",  "Canon", "Canon EOS-1D Mark III", 200,  {7.26533580606485e-06, 3.26837260515196e-06, 5.80493811123667e-06}, {-1.15453766061359e-07, -1.27963979820127e-07, -1.17921759597658e-07}},
  {"Canon EOS-1D Mark III iso 400",  "Canon", "Canon EOS-1D Mark III", 400,  {1.25860789598798e-05, 5.59949273233731e-06, 1.01882880824537e-05}, {-1.44150412318621e-07, -1.47647159325e-07, -1.45680788825842e-07}},
  {"Canon EOS-1D Mark III iso 800",  "Canon", "Canon EOS-1D Mark III", 800,  {2.24051298789793e-05, 9.85711206579791e-06, 1.82231255239551e-05}, {-1.22143408066726e-07, -1.21475747481105e-07, -1.29738725360031e-07}},
  {"Canon EOS-1D Mark III iso 1600", "Canon", "Canon EOS-1D Mark III", 1600, {4.35048049639462e-05, 1.87445848900945e-05, 3.42857558176292e-05}, {-1.78875859263032e-07, -1.09334632386373e-07, -3.42278020895721e-08}},
  {"Canon EOS-1D Mark III iso 3200", "Canon", "Canon EOS-1D Mark III", 3200, {8.32312895938775e-05, 3.53203829294442e-05, 6.41247851225428e-05}, {6.09768965365006e-08, 2.00009961002376e-07, 3.86165971019487e-07}},
  {"Canon EOS-1D Mark III iso 6400", "Canon", "Canon EOS-1D Mark III", 6400, {0.000161100090897265, 6.940076958814e-05, 0.00013187343958378}, {9.52789706179762e-07, 1.08934863777305e-06, 1.71240568021316e-06}},

// canon eos 1d mark iv contributed by rene weiss
  {"Canon EOS-1D Mark IV iso 100",   "Canon", "Canon EOS-1D Mark IV", 100,   {7.45109568644446e-06, 3.20591663834476e-06, 5.08843083080938e-06}, {-1.04851726266237e-07, -1.07554215503887e-07, -3.70348202855373e-08}},
// {"Canon EOS-1D Mark IV iso 100",   "Canon", "Canon EOS-1D Mark IV", 100,   {7.38693260475098e-06, 3.11122028930089e-06, 4.88420529630975e-06}, {-1.0538422469935e-07, -9.38298584589576e-08, -1.3925430038121e-08}},
// {"Canon EOS-1D Mark IV iso 100",   "Canon", "Canon EOS-1D Mark IV", 100,   {7.4750921750857e-06, 3.17084585592443e-06, 4.97245688819829e-06}, {-1.1388728158341e-07, -1.05966139050281e-07, -2.93869637718587e-08}},
  {"Canon EOS-1D Mark IV iso 200",   "Canon", "Canon EOS-1D Mark IV", 200,   {1.4946701653055e-05, 6.33521104541457e-06, 9.90911149608666e-06}, {-3.22769449661988e-07, -3.06623882896144e-07, -1.5706778792938e-07}},
// {"Canon EOS-1D Mark IV iso 200",   "Canon", "Canon EOS-1D Mark IV", 200,   {1.47641092851842e-05, 6.25166079490655e-06, 9.86305032021224e-06}, {-2.83696860762671e-07, -2.6691589887551e-07, -1.39873911179286e-07}},
// {"Canon EOS-1D Mark IV iso 200",   "Canon", "Canon EOS-1D Mark IV", 200,   {1.45333883108176e-05, 6.15302552274516e-06, 9.90976555594503e-06}, {-2.62788680808174e-07, -2.46892180052442e-07, -1.45311554549588e-07}},
  {"Canon EOS-1D Mark IV iso 400",   "Canon", "Canon EOS-1D Mark IV", 400,   {2.97055650450978e-05, 1.24329327624361e-05, 1.98156298991241e-05}, {-8.17855128181519e-07, -7.22730051032484e-07, -5.00595031072756e-07}},
// {"Canon EOS-1D Mark IV iso 400",   "Canon", "Canon EOS-1D Mark IV", 400,   {3.05796737531127e-05, 1.25031799969253e-05, 1.97704202339593e-05}, {-9.29249520008643e-07, -7.51088766404873e-07, -4.99178780344566e-07}},
// {"Canon EOS-1D Mark IV iso 400",   "Canon", "Canon EOS-1D Mark IV", 400,   {2.9977914717365e-05, 1.23973398544219e-05, 2.00469897344627e-05}, {-8.64147934035783e-07, -7.32029705254182e-07, -5.37614638913992e-07}},
  {"Canon EOS-1D Mark IV iso 800",   "Canon", "Canon EOS-1D Mark IV", 800,   {4.72422965279481e-05, 1.97580260905711e-05, 3.21289001222551e-05}, {-9.53622024472837e-07, -7.76801747431642e-07, -4.26625441160059e-07}},
// {"Canon EOS-1D Mark IV iso 800",   "Canon", "Canon EOS-1D Mark IV", 800,   {4.70360076843665e-05, 1.99506080436871e-05, 3.18943321041694e-05}, {-9.33977307034658e-07, -8.28369051499717e-07, -4.41681939950077e-07}},
// {"Canon EOS-1D Mark IV iso 800",   "Canon", "Canon EOS-1D Mark IV", 800,   {4.71787014602712e-05, 1.99524528919399e-05, 3.21026243866393e-05}, {-9.6698349140286e-07, -8.31240804078893e-07, -4.49387242355577e-07}},
  {"Canon EOS-1D Mark IV iso 1600",  "Canon", "Canon EOS-1D Mark IV", 1600,  {8.41501287392266e-05, 3.54984017391027e-05, 5.85232299809813e-05}, {-1.31887235955975e-06, -1.07732220601431e-06, -5.61776508490559e-07}},
// {"Canon EOS-1D Mark IV iso 1600",  "Canon", "Canon EOS-1D Mark IV", 1600,  {8.50686280733962e-05, 3.63679097184033e-05, 5.85329259245716e-05}, {-1.37077590341673e-06, -1.25956524835715e-06, -4.87518613593813e-07}},
// {"Canon EOS-1D Mark IV iso 1600",  "Canon", "Canon EOS-1D Mark IV", 1600,  {8.42537031975942e-05, 3.53331628860368e-05, 5.84842298411963e-05}, {-1.31471361618914e-06, -1.02402473117831e-06, -5.06509778180392e-07}},
  {"Canon EOS-1D Mark IV iso 3200",  "Canon", "Canon EOS-1D Mark IV", 3200,  {0.000151233814353015, 6.43083270295469e-05, 0.000105968894234959}, {-1.6329659551799e-06, -1.3686338118191e-06, -2.23502903112962e-07}},
// {"Canon EOS-1D Mark IV iso 3200",  "Canon", "Canon EOS-1D Mark IV", 3200,  {0.000148607207651594, 6.41631202572518e-05, 0.000104579098472442}, {-1.23747603714832e-06, -1.19008550539033e-06, 6.04087063916941e-08}},
// {"Canon EOS-1D Mark IV iso 3200",  "Canon", "Canon EOS-1D Mark IV", 3200,  {0.000148318838347008, 6.19514193060157e-05, 0.000103183765839855}, {-1.31943686167617e-06, -7.2412505435596e-07, 2.52187301625476e-07}},
  {"Canon EOS-1D Mark IV iso 6400",  "Canon", "Canon EOS-1D Mark IV", 6400,  {0.000274479921008939, 0.000115280769264271, 0.000193221356469422}, {-1.11355583694802e-06, -2.91819311517776e-07, 1.61680520052483e-06}},
// {"Canon EOS-1D Mark IV iso 6400",  "Canon", "Canon EOS-1D Mark IV", 6400,  {0.000271187914015963, 0.00011618420731714, 0.000190643434391458}, {-8.0385267921451e-07, -5.27874150920695e-07, 2.04005133591042e-06}},
// {"Canon EOS-1D Mark IV iso 6400",  "Canon", "Canon EOS-1D Mark IV", 6400,  {0.000271270054375198, 0.00011526334375617, 0.000192307894923944}, {-8.5683956283426e-07, -3.14510788774634e-07, 1.81884632861664e-06}},
  {"Canon EOS-1D Mark IV iso 12800", "Canon", "Canon EOS-1D Mark IV", 12800, {0.000512920806158297, 0.000215549000111297, 0.000369737119750107}, {7.9502192228526e-07, 2.20537589324103e-06, 5.14296586379743e-06}},
// {"Canon EOS-1D Mark IV iso 12800", "Canon", "Canon EOS-1D Mark IV", 12800, {0.000514133075312802, 0.000218859268952812, 0.000362571025472635}, {3.14616169304635e-07, 1.15525010352116e-06, 5.54915288349669e-06}},
// {"Canon EOS-1D Mark IV iso 12800", "Canon", "Canon EOS-1D Mark IV", 12800, {0.000517455632869977, 0.000220032867677586, 0.000366402208811136}, {-2.11691818292466e-07, 5.57617439328265e-07, 4.69821712171949e-06}},

  // canon eos 1dx contributed by Jack "FF" Zyberk
  {"Canon EOS-1D X iso 100",       "Canon",      "Canon EOS-1D X",              100,         {1.10719553247465e-06, 9.79108465602694e-07, 2.80622389823524e-06}, {1.88993940029056e-08, 1.45565809835822e-08, 2.23355943476966e-08}},
  {"Canon EOS-1D X iso 400",       "Canon",      "Canon EOS-1D X",              400,         {4.29780520762241e-06, 3.5002464805713e-06, 9.90209633894192e-06}, {2.17023400564796e-08, 1.57387067279191e-08, 3.66686588085695e-08}},
  {"Canon EOS-1D X iso 800",       "Canon",      "Canon EOS-1D X",              800,         {8.47343147503807e-06, 6.76271202981485e-06, 1.85780276794397e-05}, {2.61517231925115e-08, 2.14606059006801e-08, 6.07178845517838e-08}},
  {"Canon EOS-1D X iso 1600",       "Canon",      "Canon EOS-1D X",              1600,         {1.61432148441978e-05, 1.27216468899923e-05, 3.50881969328389e-05}, {4.78368021031139e-08, 3.76901387932446e-08, 1.08430483423159e-07}},
  {"Canon EOS-1D X iso 3200",       "Canon",      "Canon EOS-1D X",              3200,         {3.23874679853318e-05, 2.5412085437246e-05, 7.00200066869962e-05}, {9.55416526973525e-08, 7.87270794396542e-08, 2.12970832542746e-07}},
  {"Canon EOS-1D X iso 3200",       "Canon",      "Canon EOS-1D X",              3200,         {3.23960229094327e-05, 2.54359779678498e-05, 6.33613650578256e-05}, {9.46885806695973e-08, 7.16356016383103e-08, 2.32964172026421e-07}},
  {"Canon EOS-1D X iso 6400",       "Canon",      "Canon EOS-1D X",              6400,         {6.29540521890096e-05, 4.98209711364181e-05, 0.00013786291322687}, {2.18319931902851e-07, 1.79536101918382e-07, 4.58361273116836e-07}},
  {"Canon EOS-1D X iso 6400",       "Canon",      "Canon EOS-1D X",              6400,         {6.23625413696182e-05, 4.92706402324525e-05, 0.000123223426846689}, {2.22349263163289e-07, 1.81422561259337e-07, 5.37267447953398e-07}},
  {"Canon EOS-1D X iso 12800",       "Canon",      "Canon EOS-1D X",              12800,         {0.000128492833591225, 0.000103342258750159, 0.000278672223466148}, {5.76472351172718e-07, 4.67602312410342e-07, 1.02801726595474e-06}},
  {"Canon EOS-1D X iso 12800",       "Canon",      "Canon EOS-1D X",              12800,         {0.000128688616109552, 0.000102170424010651, 0.000245753072736946}, {5.92061699317042e-07, 4.90330426798581e-07, 1.24232845916652e-06}},
  {"Canon EOS-1D X iso 25600",       "Canon",      "Canon EOS-1D X",              25600,         {0.000276989337710469, 0.00023041566045222, 0.000540011820643258}, {1.5446743665946e-06, 1.40515714061852e-06, 2.55509012682351e-06}},
  {"Canon EOS-1D X iso 25600",       "Canon",      "Canon EOS-1D X",              25600,         {0.000272807306934276, 0.000225286645526352, 0.000498068874039496}, {1.66565295724854e-06, 1.54041832548758e-06, 3.16997393474142e-06}},
  {"Canon EOS-1D X iso 51200",       "Canon",      "Canon EOS-1D X",              51200,         {0.000564904680041895, 0.00053182109033213, 0.000982284609336086}, {3.90156668150741e-06, 3.29348596994299e-06, 5.92713141415749e-06}},
  {"Canon EOS-1D X iso 51200",       "Canon",      "Canon EOS-1D X",              51200,         {0.000545930344025055, 0.000504158767910717, 0.000948255336996169}, {4.64734484599178e-06, 3.94389927555252e-06, 7.25995318157872e-06}},

// canon 5d contributed by dimitri gathy
  {"Canon EOS 5D iso 100",  "Canon", "Canon EOS 5D", 100,  {6.24234645225204e-06, 2.55845822874628e-06, 3.59655610483783e-06}, {-1.5325737443352e-07, -1.32378793573221e-07, -6.75021586602017e-08}},
  {"Canon EOS 5D iso 125",  "Canon", "Canon EOS 5D", 125,  {7.53906223921016e-06, 2.94272879830261e-06, 4.0828757390852e-06}, {-1.8680824092464e-07, -1.27673145877061e-07, -4.49504645730567e-08}},
  {"Canon EOS 5D iso 160",  "Canon", "Canon EOS 5D", 160,  {8.94863721936524e-06, 3.41934770840688e-06, 4.6918053098172e-06}, {-1.96339862445497e-07, -1.14585498300565e-07, -8.38190361364528e-09}},
  {"Canon EOS 5D iso 200",  "Canon", "Canon EOS 5D", 200,  {9.95700275675423e-06, 4.02595954618857e-06, 5.79742313317198e-06}, {-2.07627661677296e-07, -1.49513445031451e-07, -5.54867104457192e-08}},
  {"Canon EOS 5D iso 250",  "Canon", "Canon EOS 5D", 250,  {1.22909344826235e-05, 4.77286433382598e-06, 6.81273531463893e-06}, {-2.55006892949659e-07, -1.46920231455729e-07, -2.57157690574663e-08}},
  {"Canon EOS 5D iso 320",  "Canon", "Canon EOS 5D", 320,  {1.38687051392793e-05, 5.65484887082275e-06, 8.11290604030725e-06}, {-1.903068239085e-07, -1.32536357457758e-07, 3.49871720419949e-08}},
  {"Canon EOS 5D iso 400",  "Canon", "Canon EOS 5D", 400,  {1.72583150371525e-05, 6.76955469224534e-06, 1.00346874186507e-05}, {-2.80377584331863e-07, -1.62934930840068e-07, -4.92182718817914e-08}},
  {"Canon EOS 5D iso 500",  "Canon", "Canon EOS 5D", 500,  {2.08266435188932e-05, 8.20438327424064e-06, 1.20730794006522e-05}, {-3.15016068642882e-07, -1.72392629198408e-07, 6.97288493670656e-09}},
  {"Canon EOS 5D iso 640",  "Canon", "Canon EOS 5D", 640,  {2.44687602515385e-05, 1.01815795742491e-05, 1.52812029271331e-05}, {-2.45534327038093e-07, -1.94540024720349e-07, 3.1367347686399e-08}},
  {"Canon EOS 5D iso 800",  "Canon", "Canon EOS 5D", 800,  {3.02287781752651e-05, 1.21672792504954e-05, 1.83582932536327e-05}, {-3.10240501432566e-07, -1.76049036297875e-07, 3.82626137437165e-08}},
  {"Canon EOS 5D iso 1000", "Canon", "Canon EOS 5D", 1000, {3.81839071494328e-05, 1.52966076574799e-05, 2.30267781167696e-05}, {-3.85779137949504e-07, -2.11059245564553e-07, 6.15596893503432e-08}},
  {"Canon EOS 5D iso 1250", "Canon", "Canon EOS 5D", 1250, {4.65263083216319e-05, 1.87021490824755e-05, 2.84637780275189e-05}, {-4.05872814782544e-07, -1.47159852907206e-07, 1.93492320934524e-07}},
  {"Canon EOS 5D iso 1600", "Canon", "Canon EOS 5D", 1600, {5.85789143173781e-05, 2.33615162346404e-05, 3.49648457895592e-05}, {-5.8374947724596e-07, -2.46052041574057e-07, 2.39793214861529e-07}},
  {"Canon EOS 5D iso 3200", "Canon", "Canon EOS 5D", 3200, {0.00010354748880159, 4.33973972391077e-05, 6.92302029760012e-05}, {-2.65786436510373e-07, 1.41768971577289e-07, 9.01647805667886e-07}},

// jo's new one 3rd stop iso 5dm2 profile (red channel useless)
  {"Canon EOS 5D Mark II iso 50",    "Canon", "Canon EOS 5D Mark II", 50,    {5.73140815948624e-06, 2.79198435542968e-06, 3.59631490707634e-06}, {8.37558079329795e-08, -1.89998352130387e-07, -9.92781903456611e-08}},
  {"Canon EOS 5D Mark II iso 100",   "Canon", "Canon EOS 5D Mark II", 100,   {4.65396160771323e-06, 2.54268633057631e-06, 3.3686061042396e-06}, {2.01266624304179e-07, -1.2767166916e-07, -1.03366386987475e-07}},
  {"Canon EOS 5D Mark II iso 125",   "Canon", "Canon EOS 5D Mark II", 125,   {5.93753905928801e-06, 3.10558995355718e-06, 3.83247349057912e-06}, {2.20584174052093e-07, -1.61726338269355e-07, -5.87898428438765e-08}},
  {"Canon EOS 5D Mark II iso 160",   "Canon", "Canon EOS 5D Mark II", 160,   {7.81426491543047e-06, 4.32743448833956e-06, 5.48754334677837e-06}, {3.5234507375559e-07, -2.26334401144962e-07, -8.56570287479908e-08}},
  {"Canon EOS 5D Mark II iso 200",   "Canon", "Canon EOS 5D Mark II", 200,   {7.63003484846413e-06, 4.24544423641621e-06, 5.35998557834612e-06}, {3.84000297738811e-07, -2.03384292076851e-07, -7.84301826664221e-08}},
  {"Canon EOS 5D Mark II iso 250",   "Canon", "Canon EOS 5D Mark II", 250,   {8.86575413652491e-06, 4.94871701189896e-06, 6.27431503805867e-06}, {4.75399054403638e-07, -2.06294530900694e-07, -5.47501369985216e-08}},
  {"Canon EOS 5D Mark II iso 320",   "Canon", "Canon EOS 5D Mark II", 320,   {1.29257976479177e-05, 7.40125208259984e-06, 9.95250366934991e-06}, {7.28430553117513e-07, -2.90152662915473e-07, -1.18674500119361e-07}},
  {"Canon EOS 5D Mark II iso 400",   "Canon", "Canon EOS 5D Mark II", 400,   {1.31783476785429e-05, 7.44712481727351e-06, 9.68767753191495e-06}, {7.41672052475118e-07, -2.955896862455e-07, -9.181768099233e-08}},
  {"Canon EOS 5D Mark II iso 500",   "Canon", "Canon EOS 5D Mark II", 500,   {1.59948885802758e-05, 8.71003610946561e-06, 1.15635603437108e-05}, {8.69227422973739e-07, -2.75873796483352e-07, -7.80113640737588e-08}},
  {"Canon EOS 5D Mark II iso 640",   "Canon", "Canon EOS 5D Mark II", 640,   {2.28437253981891e-05, 1.32476058121959e-05, 1.79724166782725e-05}, {1.4247028666317e-06, -4.12109225293271e-07, -9.89891695685032e-08}},
  {"Canon EOS 5D Mark II iso 800",   "Canon", "Canon EOS 5D Mark II", 800,   {2.31459922454041e-05, 1.31628457332571e-05, 1.79661966641095e-05}, {1.39635429101038e-06, -3.93740986248811e-07, -1.1314729847805e-07}},
  {"Canon EOS 5D Mark II iso 1000",  "Canon", "Canon EOS 5D Mark II", 1000,  {3.00088298053439e-05, 1.57972515131918e-05, 2.27616969126849e-05}, {1.43641056707739e-06, -4.01630709011093e-07, -1.82626036118674e-07}},
  {"Canon EOS 5D Mark II iso 1250",  "Canon", "Canon EOS 5D Mark II", 1250,  {4.99660478038754e-05, 2.42616805598858e-05, 3.55320078195639e-05}, {1.68906378525877e-06, -6.31242634742706e-07, -3.25160749593304e-07}},
  {"Canon EOS 5D Mark II iso 1600",  "Canon", "Canon EOS 5D Mark II", 1600,  {4.94861738919414e-05, 2.41142155841303e-05, 3.58493532307298e-05}, {1.69173774814829e-06, -5.98926395478041e-07, -3.60804221590132e-07}},
  {"Canon EOS 5D Mark II iso 2000",  "Canon", "Canon EOS 5D Mark II", 2000,  {4.6828082508478e-05, 2.74418745719276e-05, 3.93140539075569e-05}, {3.14453370251941e-06, -6.30693081882824e-07, -1.25768821260655e-07}},
  {"Canon EOS 5D Mark II iso 2500",  "Canon", "Canon EOS 5D Mark II", 2500,  {5.89757695690232e-05, 3.27111511496431e-05, 4.84424456540464e-05}, {3.67885122164099e-06, -4.62226239940358e-07, -1.07292936690233e-07}},
  {"Canon EOS 5D Mark II iso 3200",  "Canon", "Canon EOS 5D Mark II", 3200,  {7.98930334230088e-05, 3.95371780409199e-05, 5.88671308596389e-05}, {3.87890530191021e-06, -2.97958880268045e-07, 1.02740702308392e-07}},
  {"Canon EOS 5D Mark II iso 4000",  "Canon", "Canon EOS 5D Mark II", 4000,  {0.000100609703110426, 4.86730052148627e-05, 7.26227010274504e-05}, {4.58162026699074e-06, -2.91944870177546e-07, 2.21554637176777e-07}},
  {"Canon EOS 5D Mark II iso 5000",  "Canon", "Canon EOS 5D Mark II", 5000,  {0.000103052625157473, 5.9798856833857e-05, 9.01799090278334e-05}, {8.07981634271859e-06, -1.32325878772108e-07, 4.52471173504848e-07}},
  {"Canon EOS 5D Mark II iso 6400",  "Canon", "Canon EOS 5D Mark II", 6400,  {0.000126853391237565, 7.34335232069023e-05, 0.00011247668092831}, {1.05228978541747e-05, 3.47619065786586e-07, 8.90637463286515e-07}},
  {"Canon EOS 5D Mark II iso 12800", "Canon", "Canon EOS 5D Mark II", 12800, {0.000307962334201965, 0.000147032463221761, 0.00021868161056636}, {1.546099749578e-05, 9.53592101336839e-07, 3.29130268291292e-06}},
  {"Canon EOS 5D Mark II iso 25600", "Canon", "Canon EOS 5D Mark II", 25600, {0.000583399751494278, 0.000281229180172485, 0.000430416244892868}, {3.56291638092497e-05, 8.60303261565331e-06, 1.26042671608886e-05}},
// jo's old 5dm2 presets, commented out for variance comparisons:
//{"Canon EOS 5D Mark II iso 100",   "Canon", "EOS 5D Mark II", 100,   {2.83158285671061e-06, 2.83158285671061e-06, 2.83158285671061e-06}, {-2.03690658213514e-07, -2.03690658213514e-07, -2.03690658213514e-07}},
//{"Canon EOS 5D Mark II iso 200",   "Canon", "EOS 5D Mark II", 200,   {4.13958998800824e-06, 4.13958998800824e-06, 4.13958998800824e-06}, {-2.21304398662743e-07, -2.21304398662743e-07, -2.21304398662743e-07}},
//{"Canon EOS 5D Mark II iso 400",   "Canon", "EOS 5D Mark II", 400,   {7.24049540870435e-06, 7.24049540870435e-06, 7.24049540870435e-06}, {-3.02238583363423e-07, -3.02238583363423e-07, -3.02238583363423e-07}},
//{"Canon EOS 5D Mark II iso 800",   "Canon", "EOS 5D Mark II", 800,   {1.31341686272862e-05, 1.31341686272862e-05, 1.31341686272862e-05}, {-4.6914485275861e-07, -4.6914485275861e-07, -4.6914485275861e-07}},
//{"Canon EOS 5D Mark II iso 1600",  "Canon", "EOS 5D Mark II", 1600,  {2.50463590614947e-05, 2.50463590614947e-05, 2.50463590614947e-05}, {-7.3061992822639e-07, -7.3061992822639e-07, -7.3061992822639e-07}},
//{"Canon EOS 5D Mark II iso 3200",  "Canon", "EOS 5D Mark II", 3200,  {4.33825185453575e-05, 4.33825185453575e-05, 4.33825185453575e-05}, {-5.10134456111927e-07, -5.10134456111927e-07, -5.10134456111927e-07}},
//{"Canon EOS 5D Mark II iso 6400",  "Canon", "EOS 5D Mark II", 6400,  {7.97916401445384e-05, 7.97916401445384e-05, 7.97916401445384e-05}, {4.19094215400523e-07, 4.19094215400523e-07, 4.19094215400523e-07}},
//{"Canon EOS 5D Mark II iso 12800", "Canon", "EOS 5D Mark II", 12800, {0.000162588350196486, 0.000162588350196486, 0.000162588350196486}, {4.536854078923e-07, 4.536854078923e-07, 4.536854078923e-07}},
//{"Canon EOS 5D Mark II iso 25600", "Canon", "EOS 5D Mark II", 25600, {0.000315319994328854, 0.000315319994328854, 0.000315319994328854}, {9.01403346714491e-06, 9.01403346714491e-06, 9.01403346714491e-06}},

// Canon 5D Mark III contributed by Jesper Pedersen
  {"Canon EOS 5D Mark III iso 50",     "Canon", "Canon EOS 5D Mark III", 50,     {2.64054695517265e-06, 1.57567510343875e-06, -1.81937570658966e-07}, {-3.73651904195122e-09, -3.31203242227072e-08, 4.79009001935846e-07}},
  {"Canon EOS 5D Mark III iso 100",    "Canon", "Canon EOS 5D Mark III", 100,    {2.35525873489583e-06, 1.55413089907708e-06, 8.60663421379715e-07}, {1.61191836573806e-08, -3.4355760673268e-08, 4.46630475780481e-07}},
  {"Canon EOS 5D Mark III iso 125",    "Canon", "Canon EOS 5D Mark III", 125,    {2.86826043362566e-06, 1.89596025994823e-06, 9.25643039484462e-07}, {3.01826505306002e-08, -3.57949767553021e-08, 5.73189783174826e-07}},
  {"Canon EOS 5D Mark III iso 160",    "Canon", "Canon EOS 5D Mark III", 160,    {3.49317098204992e-06, 9.87895642982923e-07, -1.09797654526914e-06}, {1.30726158771101e-08, 3.0209773187539e-07, 7.59483103204285e-07}},
  {"Canon EOS 5D Mark III iso 200",    "Canon", "Canon EOS 5D Mark III", 200,    {4.26693189339406e-06, 2.67707426632491e-06, 1.39536150643145e-06}, {2.40981922409499e-08, -4.10926301783204e-08, 8.05645795577146e-07}},
  {"Canon EOS 5D Mark III iso 250",    "Canon", "Canon EOS 5D Mark III", 250,    {5.14963365490139e-06, 3.31854621076302e-06, 1.66404165287465e-06}, {5.25490820903688e-08, -5.15937653230729e-08, 1.00789021591999e-06}},
  {"Canon EOS 5D Mark III iso 320",    "Canon", "Canon EOS 5D Mark III", 320,    {6.56032912186801e-06, 1.56913702527632e-06, -2.10456645016528e-06}, {3.8797752729713e-08, 6.30470888441145e-07, 1.39691665889852e-06}},
  {"Canon EOS 5D Mark III iso 400",    "Canon", "Canon EOS 5D Mark III", 400,    {7.97404666971406e-06, 4.85532987239621e-06, 2.63814513393442e-06}, {6.70430403472064e-08, -3.01316455303083e-08, 1.46753391519147e-06}},
  {"Canon EOS 5D Mark III iso 500",    "Canon", "Canon EOS 5D Mark III", 500,    {9.89944977477999e-06, 6.06986980774951e-06, 3.2468022250825e-06}, {9.30326007342633e-08, -4.45138009410806e-08, 1.82943604637796e-06}},
  {"Canon EOS 5D Mark III iso 640",    "Canon", "Canon EOS 5D Mark III", 640,    {1.22155623294684e-05, 2.59834767157403e-06, -3.80727094052324e-06}, {1.11554950833544e-07, 1.28314734852969e-06, 2.60625929142179e-06}},
  {"Canon EOS 5D Mark III iso 800",    "Canon", "Canon EOS 5D Mark III", 800,    {1.51869597015169e-05, 9.19875515505219e-06, 3.04396352225128e-06}, {1.37970608838347e-07, -5.65660706961435e-08, 2.98826451678776e-06}},
  {"Canon EOS 5D Mark III iso 1000",   "Canon", "Canon EOS 5D Mark III", 1000,   {1.90805385579024e-05, 1.14206618505543e-05, 3.40750981843532e-06}, {1.80159271234994e-07, -3.18387958490488e-08, 3.78234328711264e-06}},
  {"Canon EOS 5D Mark III iso 1250",   "Canon", "Canon EOS 5D Mark III", 1250,   {2.42004836737217e-05, 4.75371463640183e-06, -7.5578419245181e-06}, {2.18263347664671e-07, 2.58986608889593e-06, 5.15948151424169e-06}},
  {"Canon EOS 5D Mark III iso 1600",   "Canon", "Canon EOS 5D Mark III", 1600,   {3.00589287337036e-05, 1.76407773010225e-05, 8.47523648921099e-06}, {2.90719065941547e-07, 2.31809156468877e-08, 5.49671726842532e-06}},
  {"Canon EOS 5D Mark III iso 2000",   "Canon", "Canon EOS 5D Mark III", 2000,   {3.80302476654417e-05, 2.24153331526808e-05, 4.06745192416337e-06}, {3.55293809484702e-07, -2.0511199149037e-08, 7.69205940968721e-06}},
  {"Canon EOS 5D Mark III iso 2500",   "Canon", "Canon EOS 5D Mark III", 2500,   {4.66175541737836e-05, 1.00436410813393e-05, -1.44412058065656e-05}, {6.00331632789034e-07, 4.9388276425486e-06, 1.01395578887271e-05}},
  {"Canon EOS 5D Mark III iso 3200",   "Canon", "Canon EOS 5D Mark III", 3200,   {5.91326745192966e-05, 3.49134658907306e-05, 5.74996630014487e-05}, {6.87156100626095e-07, 5.18597698364703e-08, 5.38966013043384e-06}},
  {"Canon EOS 5D Mark III iso 4000",   "Canon", "Canon EOS 5D Mark III", 4000,   {7.50185043193998e-05, 4.38315509065056e-05, 4.22118745036671e-05}, {8.1458013089113e-07, 1.72489512737824e-07, 1.05613208972616e-05}},
  {"Canon EOS 5D Mark III iso 5000",   "Canon", "Canon EOS 5D Mark III", 5000,   {9.58672060630966e-05, 1.85820443202576e-05, -3.06515800562342e-05}, {1.23331719366903e-06, 1.05540517164007e-05, 2.06621868087601e-05}},
  {"Canon EOS 5D Mark III iso 6400",   "Canon", "Canon EOS 5D Mark III", 6400,   {0.000121685874927878, 6.84961363739768e-05, 6.5457532223609e-05}, {1.45786098195459e-06, 8.89311063445895e-07, 1.60516117123092e-05}},
  {"Canon EOS 5D Mark III iso 12800",  "Canon", "Canon EOS 5D Mark III", 12800,  {0.000233538156093256, 0.000125257064541362, 2.60741986045709e-05}, {4.05258017424011e-06, 4.74983038443139e-06, 4.67537021435587e-05}},
  {"Canon EOS 5D Mark III iso 25600",  "Canon", "Canon EOS 5D Mark III", 25600,  {0.000372736481428556, 0.000215851179747733, 0.000267553232593844}, {1.42757563915053e-05, 1.51485673828923e-05, 5.29822764873008e-05}},
  {"Canon EOS 5D Mark III iso 51200",  "Canon", "Canon EOS 5D Mark III", 51200,  {0.000610725651264109, 0.000372501752903588, 0.000618122910576193}, {3.71967792268875e-05, 3.57776341970749e-05, 6.82475091042931e-05}},
  {"Canon EOS 5D Mark III iso 102400", "Canon", "Canon EOS 5D Mark III", 102400, {0.000859965585710808, 0.000505543774583139, 0.00053546110862992}, {7.43995315666412e-05, 7.42346112346366e-05, 0.000166621464100798}},

// Canon eos 5ds r contributed by Jack 'FF'
  {"Canon EOS 5DS R iso 100",  "Canon", "Canon EOS 5DS R", 100,  {4.11815392079092e-06, 3.46503849117778e-06, 5.02661517291753e-06}, {2.12289814171936e-08, 1.65008574001287e-08, 2.11413511739703e-08}},
  {"Canon EOS 5DS R iso 200",  "Canon", "Canon EOS 5DS R", 200,  {7.37092441643394e-06, 6.03402113810628e-06, 1.02379751159391e-05}, {2.17724126768704e-08, 1.77007709881493e-08, 2.16780005370838e-08}},
  {"Canon EOS 5DS R iso 400",  "Canon", "Canon EOS 5DS R", 400,  {1.34853885966277e-05, 1.07785810512403e-05, 1.65347808809915e-05}, {3.33328365673261e-08, 2.58968538553752e-08, 3.47119515815336e-08}},
  {"Canon EOS 5DS R iso 800",  "Canon", "Canon EOS 5DS R", 800,  {2.56712595620091e-05, 1.97922689832358e-05, 3.03394497720616e-05}, {5.41675997509208e-08, 4.45663042154251e-08, 5.85388938152329e-08}},
  {"Canon EOS 5DS R iso 1600", "Canon", "Canon EOS 5DS R", 1600, {4.84228270525326e-05, 3.60080574613549e-05, 4.69431808846379e-05}, {1.4407678102083e-07, 1.14971116626832e-07, 1.55730732349072e-07}},
  {"Canon EOS 5DS R iso 3200", "Canon", "Canon EOS 5DS R", 3200, {9.70216728944392e-05, 6.9224794514711e-05, 0.000100301540293112}, {2.36515947495961e-07, 1.76337736102171e-07, 2.63465444194838e-07}},
  {"Canon EOS 5DS R iso 6400", "Canon", "Canon EOS 5DS R", 6400, {0.000182870387839005, 0.000129171546154818, 0.00019351051065462}, {4.89216995732489e-07, 3.52248199525925e-07, 5.94428337284862e-07}},

// canon eos 6d, contributed by markus kanet.
  {"Canon EOS 6D iso 50",     "Canon", "Canon EOS 6D",  50,      {3.48933489323533e-06, 1.53689528756e-06, 2.44436772256532e-06}, {-5.03789176390158e-08, -5.07275714555471e-08, -3.48108842841341e-08}},
  {"Canon EOS 6D iso 100",    "Canon", "Canon EOS 6D",  100,     {3.56119064886049e-06, 1.5987072547862e-06, 2.46158788123732e-06}, {-5.10831182675752e-08, -6.12763334497633e-08, -4.11263477931922e-08}},
  {"Canon EOS 6D iso 125",    "Canon", "Canon EOS 6D",  125,     {4.24782197299846e-06, 1.93189496295134e-06, 2.94374339122666e-06}, {-5.12458547107755e-08, -7.07800557199711e-08, -4.47975895289208e-08}},
  {"Canon EOS 6D iso 160",    "Canon", "Canon EOS 6D",  160,     {5.30499740919098e-06, 2.21028107065167e-06, 3.62778170207987e-06}, {-8.61751881178675e-08, -5.50279884266667e-08, -6.93495712157447e-08}},
  {"Canon EOS 6D iso 200",    "Canon", "Canon EOS 6D",  200,     {6.33769994416361e-06, 2.77747065894902e-06, 4.36705883379554e-06}, {-8.75103392915963e-08, -8.88409357787755e-08, -7.22296157340536e-08}},
  {"Canon EOS 6D iso 250",    "Canon", "Canon EOS 6D",  250,     {7.68250963698401e-06, 3.37844251571366e-06, 5.21424780381357e-06}, {-1.04356686472227e-07, -1.0951328014439e-07, -7.70415668478448e-08}},
  {"Canon EOS 6D iso 320",    "Canon", "Canon EOS 6D",  320,     {9.62693703246292e-06, 3.90334848888101e-06, 6.65515733274367e-06}, {-1.2780031086969e-07, -4.51882005317834e-08, -1.1046677027244e-07}},
  {"Canon EOS 6D iso 400",    "Canon", "Canon EOS 6D",  400,     {1.19483641019807e-05, 5.20897924163349e-06, 8.09318561220902e-06}, {-1.76542434834142e-07, -1.6765174229657e-07, -1.25650124906979e-07}},
  {"Canon EOS 6D iso 500",    "Canon", "Canon EOS 6D",  500,     {1.4132486347748e-05, 6.22504672380681e-06, 9.62239097636473e-06}, {-1.70423761799236e-07, -1.74874884543563e-07, -1.17336095084976e-07}},
  {"Canon EOS 6D iso 640",    "Canon", "Canon EOS 6D",  640,     {1.70411434273523e-05, 7.88827640184618e-06, 1.18692737342217e-05}, {-1.74391552616794e-07, -2.78619647875394e-07, -1.50397585873748e-07}},
  {"Canon EOS 6D iso 800",    "Canon", "Canon EOS 6D",  800,     {2.11227491679735e-05, 9.24829119262408e-06, 1.43900598038654e-05}, {-2.26004972319872e-07, -2.18261169995688e-07, -1.42588507258176e-07}},
  {"Canon EOS 6D iso 1000",   "Canon", "Canon EOS 6D",  1000,    {2.5597351903982e-05, 1.13300047319769e-05, 1.74908824484401e-05}, {-2.40742205838392e-07, -2.62482747562608e-07, -1.51125618732945e-07}},
  {"Canon EOS 6D iso 1250",   "Canon", "Canon EOS 6D",  1250,    {3.07578557796775e-05, 1.31668466368835e-05, 2.12629033123987e-05}, {-2.01977601072517e-07, -1.09477463882234e-07, -1.30125609557305e-07}},
  {"Canon EOS 6D iso 1600",   "Canon", "Canon EOS 6D",  1600,    {3.75100628255528e-05, 1.67316257523913e-05, 2.57776364100145e-05}, {-1.93862239754759e-07, -2.7201786112254e-07, -8.23247448881098e-08}},
  {"Canon EOS 6D iso 2000",   "Canon", "Canon EOS 6D",  2000,    {4.65575531654142e-05, 2.05638020099461e-05, 3.19241309399665e-05}, {-2.07841417161501e-07, -2.52164945057148e-07, -6.52255495611706e-08}},
  {"Canon EOS 6D iso 2500",   "Canon", "Canon EOS 6D",  2500,    {5.59600347241286e-05, 2.2384577873195e-05, 3.82123385518105e-05}, {-1.10470699370511e-07, 4.30517709643735e-07, 5.88219050925261e-08}},
  {"Canon EOS 6D iso 3200",   "Canon", "Canon EOS 6D",  3200,    {6.98974831006011e-05, 3.08370277788928e-05, 4.7644633059019e-05}, {-1.41528382234233e-07, -1.89937717366239e-07, 1.42122675642048e-07}},
  {"Canon EOS 6D iso 4000",   "Canon", "Canon EOS 6D",  4000,    {8.74124963004756e-05, 3.82060998438052e-05, 5.95530354290044e-05}, {-1.66116975231058e-07, -1.75390713455677e-07, 1.76487718907234e-07}},
  {"Canon EOS 6D iso 5000",   "Canon", "Canon EOS 6D",  5000,    {0.00010845220386593, 4.19759705950653e-05, 7.53449570666526e-05}, {2.60039568738544e-07, 1.56980257110369e-06, 3.64710335985559e-07}},
  {"Canon EOS 6D iso 6400",   "Canon", "Canon EOS 6D",  6400,    {0.000138682113808644, 6.12134246170608e-05, 9.51232327957498e-05}, {1.13661262726304e-07, -7.89766269075377e-08, 5.06359258015854e-07}},
  {"Canon EOS 6D iso 8000",   "Canon", "Canon EOS 6D",  8000,    {0.000173808548522945, 7.57310384411755e-05, 0.000118995708399251}, {1.70975649987605e-07, 2.24338382865702e-07, 7.87106694888938e-07}},
  {"Canon EOS 6D iso 10000",  "Canon", "Canon EOS 6D",  10000,   {0.000206983154041228, 8.33782452369167e-05, 0.000148191399140809}, {1.77059989183455e-06, 3.29861412099654e-06, 1.06563345983027e-06}},
  {"Canon EOS 6D iso 12800",  "Canon", "Canon EOS 6D",  12800,   {0.000272476598173815, 0.000118614216101543, 0.000182196530728433}, {8.20687385613918e-07, 6.45729111249633e-07, 1.82328362252706e-06}},
  {"Canon EOS 6D iso 16000",  "Canon", "Canon EOS 6D",  16000,   {0.000332739729678078, 0.000146199699454705, 0.000220718844248388}, {1.8943071742547e-06, 1.85763855836677e-06, 3.64085625578323e-06}},
  {"Canon EOS 6D iso 20000",  "Canon", "Canon EOS 6D",  20000,   {0.000357656455516511, 0.000154222815426572, 0.000257934884760857}, {8.65866188837763e-06, 9.39259119451833e-06, 6.55754567071249e-06}},
  {"Canon EOS 6D iso 25600",  "Canon", "Canon EOS 6D",  25600,   {0.000443861723915905, 0.000205972156089486, 0.000289658718713024}, {8.70657042791824e-06, 7.02472342727687e-06, 9.70414934161018e-06}},
  {"Canon EOS 6D iso 51200",  "Canon", "Canon EOS 6D",  51200,   {0.000795003543013078, 0.000366952884594071, 0.000508649478109384}, {2.01461488217616e-05, 2.02574239590522e-05, 2.28524010951692e-05}},
  {"Canon EOS 6D iso 102400", "Canon", "Canon EOS 6D",  102400,  {0.00112441610246235, 0.000501469493307691, 0.000726797109799539}, {4.68408530124778e-05, 5.52708300752646e-05, 5.19022553117051e-05}},

// canon eos 7d, pictures by PolarFox, green channel fit xrange reduced to [0:0.8]
  {"Canon EOS 7D iso 100",   "Canon",  "Canon EOS 7D",  100,   {1.45385477914914e-05, 5.08062324662208e-06, 7.94750305622489e-06}, {3.55977874304643e-08, 9.6111705500635e-08, 1.31240322036089e-07}},
  {"Canon EOS 7D iso 125",   "Canon",  "Canon EOS 7D",  125,   {1.61055705230647e-05, 5.97737737516147e-06, 1.00416055361428e-05}, {3.68156941476166e-07, 3.17376713835814e-07, 2.38551576228785e-07}},
  {"Canon EOS 7D iso 160",   "Canon",  "Canon EOS 7D",  160,   {1.65410735444078e-05, 8.42739317260249e-06, 1.34869878328005e-05}, {9.01323139156234e-07, 1.53684477846932e-07, 1.30166743786218e-07}},
  {"Canon EOS 7D iso 200",   "Canon",  "Canon EOS 7D",  200,   {3.32925045357627e-05, 1.23373627457696e-05, 1.93412224352399e-05}, {-1.38309189991401e-07, -1.35175982125089e-07, -1.50297112121663e-07}},
  {"Canon EOS 7D iso 250",   "Canon",  "Canon EOS 7D",  250,   {4.73263381532581e-05, 1.75613931534952e-05, 2.56375205767366e-05}, {-7.65188790034087e-07, -6.84947443970748e-07, -4.21569713720997e-07}},
  {"Canon EOS 7D iso 320",   "Canon",  "Canon EOS 7D",  320,   {4.22100281978624e-05, 2.36862375727912e-05, 3.43047061223223e-05}, {1.11732515027279e-06, -1.18672547309989e-06, -7.77002159719945e-07}},
  {"Canon EOS 7D iso 400",   "Canon",  "Canon EOS 7D",  400,   {8.25814353155754e-05, 2.9812514770923e-05, 4.37369066932059e-05}, {-2.19168418023209e-06, -1.76717826521587e-06, -1.33910685210985e-06}},
  {"Canon EOS 7D iso 500",   "Canon",  "Canon EOS 7D",  500,   {0.00010030420433304, 3.56657904268527e-05, 5.24699739532811e-05}, {-2.72754079602427e-06, -2.04760302136336e-06, -1.48603682606911e-06}},
  {"Canon EOS 7D iso 640",   "Canon",  "Canon EOS 7D",  640,   {7.70346415948609e-05, 4.37587397763144e-05, 6.58373410393954e-05}, {1.45418182466659e-06, -2.68962035035972e-06, -2.18197969031289e-06}},
  {"Canon EOS 7D iso 800",   "Canon",  "Canon EOS 7D",  800,   {0.000142607688763495, 5.11350623771918e-05, 7.85209647962388e-05}, {-3.99497891097666e-06, -3.02419470191422e-06, -2.60575811959014e-06}},
  {"Canon EOS 7D iso 1000",  "Canon",  "Canon EOS 7D",  1000,  {0.000167277957117275, 6.13338341373814e-05, 9.18057711442985e-05}, {-4.55651936345402e-06, -3.72232104914368e-06, -2.74629524965152e-06}},
  {"Canon EOS 7D iso 1250",  "Canon",  "Canon EOS 7D",  1250,  {0.000119761133790789, 6.85359039696903e-05, 0.000108034753435368}, {2.92882533188716e-06, -3.60858870820808e-06, -3.1997991535438e-06}},
  {"Canon EOS 7D iso 1600",  "Canon",  "Canon EOS 7D",  1600,  {0.000217794651908724, 7.92500952149917e-05, 0.000125605055274147}, {-5.22356230101447e-06, -3.99409235985563e-06, -3.48131819602048e-06}},
  {"Canon EOS 7D iso 2000",  "Canon",  "Canon EOS 7D",  2000,  {0.000262388458115222, 9.56517631683031e-05, 0.000152194039644565}, {-6.13286493432092e-06, -4.71832625072412e-06, -4.09312622240076e-06}},
  {"Canon EOS 7D iso 2500",  "Canon",  "Canon EOS 7D",  2500,  {0.000163886186357097, 8.84654890249178e-05, 0.000145284791994033}, {5.27937508561386e-06, -2.13955053527868e-06, -1.84212082632327e-06}},
  {"Canon EOS 7D iso 3200",  "Canon",  "Canon EOS 7D",  3200,  {0.000280206169358584, 0.000104835089912973, 0.000174422111550717}, {-2.73131126040004e-06, -1.98560991954522e-06, -1.8560298943013e-06}},
  {"Canon EOS 7D iso 4000",  "Canon",  "Canon EOS 7D",  4000,  {0.000347401689240694, 0.000131619688555538, 0.000213948270337137}, {-3.40813128693954e-06, -2.93174589524485e-06, -1.76827978296263e-06}},
  {"Canon EOS 7D iso 5000",  "Canon",  "Canon EOS 7D",  5000,  {0.000412917816972777, 0.000157067554651868, 0.000254328347600614}, {-2.52586973439055e-06, -2.15502142920993e-06, -2.32610482932543e-07}},
  {"Canon EOS 7D iso 6400",  "Canon",  "Canon EOS 7D",  6400,  {0.00050677300610677, 0.000189291959662312, 0.000319428290314647}, {-1.89818748378485e-06, -5.60804084835121e-07, -6.66080687201159e-08}},
  {"Canon EOS 7D iso 12800", "Canon",  "Canon EOS 7D",  12800, {0.00208527972666432, 0.000822134386051009, 0.00144687938513341}, {3.21268951639325e-05, 3.81346887787642e-05, 3.70734367961145e-05}},

// canon eos 7d mark II contributed by Mukund Sivaraman
  {"Canon EOS 7D Mark II iso 100",   "Canon",  "Canon EOS 7D Mark II",  100,   {6.15389760905073e-06, 2.84388787644327e-06, 5.74375405208284e-06}, {3.04323162236553e-08, 1.77677340399602e-08, 2.66383841535416e-08}},
  {"Canon EOS 7D Mark II iso 125",   "Canon",  "Canon EOS 7D Mark II",  125,   {7.59162733384928e-06, 3.53303442065187e-06, 7.08484086229841e-06}, {4.75796237362741e-08, 2.7635944707417e-08, 4.22365975862411e-08}},
  {"Canon EOS 7D Mark II iso 160",   "Canon",  "Canon EOS 7D Mark II",  160,   {9.3879975344023e-06, 4.35676925303503e-06, 8.72583655235731e-06}, {2.93647314026155e-08, 1.25169327242484e-08, 2.44052773155781e-08}},
  {"Canon EOS 7D Mark II iso 200",   "Canon",  "Canon EOS 7D Mark II",  200,   {1.16101046263724e-05, 5.3826901904648e-06, 1.08426945352691e-05}, {4.40588403096687e-08, 2.20308440648695e-08, 3.6443839320266e-08}},
  {"Canon EOS 7D Mark II iso 250",   "Canon",  "Canon EOS 7D Mark II",  250,   {1.44549361219217e-05, 6.69530078393603e-06, 1.34668602798363e-05}, {6.37498358171992e-08, 3.77458820289314e-08, 5.69190008384041e-08}},
  {"Canon EOS 7D Mark II iso 320",   "Canon",  "Canon EOS 7D Mark II",  320,   {1.81784105117155e-05, 8.44282878736993e-06, 1.69566740363775e-05}, {5.04608324422317e-08, 2.22092411391331e-08, 4.17634862665898e-08}},
  {"Canon EOS 7D Mark II iso 400",   "Canon",  "Canon EOS 7D Mark II",  400,   {1.97330345882828e-05, 9.14832113246983e-06, 1.85243166433392e-05}, {6.02455825988485e-08, 2.81105572723225e-08, 4.959618558867e-08}},
  {"Canon EOS 7D Mark II iso 500",   "Canon",  "Canon EOS 7D Mark II",  500,   {2.45901544202011e-05, 1.13958612273182e-05, 2.3004979392545e-05}, {8.55428670002603e-08, 4.51310779077098e-08, 7.29398535708539e-08}},
  {"Canon EOS 7D Mark II iso 640",   "Canon",  "Canon EOS 7D Mark II",  640,   {3.09229084905091e-05, 1.42962270495915e-05, 2.79253313570943e-05}, {8.40086267671223e-08, 3.73590604389426e-08, 7.75707715542558e-08}},
  {"Canon EOS 7D Mark II iso 800",   "Canon",  "Canon EOS 7D Mark II",  800,   {3.84885351377677e-05, 1.78229703745007e-05, 3.6082833941544e-05}, {1.14216760719623e-07, 5.34920277729241e-08, 9.40685705556626e-08}},
  {"Canon EOS 7D Mark II iso 1000",  "Canon",  "Canon EOS 7D Mark II",  1000,  {4.84204745877259e-05, 2.24037167937016e-05, 4.52591436580035e-05}, {1.59821714607197e-07, 7.91416805160528e-08, 1.35927117126112e-07}},
  {"Canon EOS 7D Mark II iso 1250",  "Canon",  "Canon EOS 7D Mark II",  1250,  {6.0545981127279e-05, 2.79222159489582e-05, 5.50310243797642e-05}, {1.78314481303872e-07, 8.72291354888823e-08, 1.57783818333402e-07}},
  {"Canon EOS 7D Mark II iso 1600",  "Canon",  "Canon EOS 7D Mark II",  1600,  {7.55552716102232e-05, 3.49962210158134e-05, 7.1013616573494e-05}, {2.39638309356792e-07, 1.25638194934939e-07, 1.88054136181048e-07}},
  {"Canon EOS 7D Mark II iso 2000",  "Canon",  "Canon EOS 7D Mark II",  2000,  {9.42742214978594e-05, 4.39825294878873e-05, 8.97673644307658e-05}, {3.33278861035654e-07, 1.9102840906313e-07, 2.86409975726094e-07}},
  {"Canon EOS 7D Mark II iso 2500",  "Canon",  "Canon EOS 7D Mark II",  2500,  {0.00012056689722321, 5.4774261067391e-05, 0.000107454632540903}, {3.89162169689827e-07, 2.2790331866145e-07, 3.79618046165792e-07}},
  {"Canon EOS 7D Mark II iso 3200",  "Canon",  "Canon EOS 7D Mark II",  3200,  {0.000150429620645509, 6.92989880040282e-05, 0.000139998652567944}, {5.10782935969087e-07, 3.24245631846328e-07, 4.7810515793373e-07}},
  {"Canon EOS 7D Mark II iso 4000",  "Canon",  "Canon EOS 7D Mark II",  4000,  {0.000188293426768234, 8.64860853831897e-05, 0.000173342185582239}, {7.61742620864557e-07, 4.86544257027301e-07, 6.6425288967439e-07}},
  {"Canon EOS 7D Mark II iso 5000",  "Canon",  "Canon EOS 7D Mark II",  5000,  {0.000234039759215084, 0.000106833547444562, 0.000208144382413738}, {8.62435545085871e-07, 6.46613674788602e-07, 9.9782251980152e-07}},
  {"Canon EOS 7D Mark II iso 6400",  "Canon",  "Canon EOS 7D Mark II",  6400,  {0.000287859297140918, 0.000134295211198586, 0.00026628393421641}, {1.25939848034095e-06, 8.80266371487095e-07, 1.18844190795365e-06}},
  {"Canon EOS 7D Mark II iso 8000",  "Canon",  "Canon EOS 7D Mark II",  8000,  {0.000285259879684499, 0.000132960643282581, 0.000264149081743729}, {1.81069959598775e-06, 1.31068028568777e-06, 1.74871446008455e-06}},
  {"Canon EOS 7D Mark II iso 10000", "Canon",  "Canon EOS 7D Mark II",  10000, {0.000346347280903315, 0.00016440053237588, 0.000321044138971251}, {2.48575086800874e-06, 1.72528349207106e-06, 2.21044857022988e-06}},
  {"Canon EOS 7D Mark II iso 12800", "Canon",  "Canon EOS 7D Mark II",  12800, {0.000530773661277798, 0.000259746836117293, 0.000495440582274337}, {3.46809847559111e-06, 2.60610305316299e-06, 3.54349550582954e-06}},
  {"Canon EOS 7D Mark II iso 16000", "Canon",  "Canon EOS 7D Mark II",  16000, {0.000647298429652717, 0.000319724386335623, 0.000604278835969922}, {4.63157499941126e-06, 3.84846479772772e-06, 5.23970902445733e-06}},
  {"Canon EOS 7D Mark II iso 25600", "Canon",  "Canon EOS 7D Mark II",  25600, {0.000956234550743688, 0.00049949117437502, 0.00093175219430423}, {8.34840843490752e-06, 7.03032187492473e-06, 8.54658328178595e-06}},
  {"Canon EOS 7D Mark II iso 51200", "Canon",  "Canon EOS 7D Mark II",  51200, {0.00172552333669734, 0.000919286752750476, 0.00159812360335718}, {1.95415314077534e-05, 1.9533844854049e-05, 2.10050528497937e-05}},

// Canon EOS 20D, contributed by Jean-Sébastien Pédron <jean-sebastien.pedron@dumbbell.fr>.
  {"Canon EOS 20D iso 100",  "Canon", "Canon EOS 20D", 100,  {3.49945097782558e-06, 3.49945097782558e-06, 3.49945097782558e-06}, {-1.83310906811176e-07, -1.83310906811176e-07, -1.83310906811176e-07}},
  {"Canon EOS 20D iso 200",  "Canon", "Canon EOS 20D", 200,  {5.92880023553901e-06, 5.92880023553901e-06, 5.92880023553901e-06}, {-3.28519463926131e-07, -3.28519463926131e-07, -3.28519463926131e-07}},
  {"Canon EOS 20D iso 400",  "Canon", "Canon EOS 20D", 400,  {9.65526122541132e-06, 9.65526122541132e-06, 9.65526122541132e-06}, {-3.17639857708167e-07, -3.17639857708167e-07, -3.17639857708167e-07}},
  {"Canon EOS 20D iso 800",  "Canon", "Canon EOS 20D", 800,  {1.72686069580337e-05, 1.72686069580337e-05, 1.72686069580337e-05}, {-3.01878906022867e-07, -3.01878906022867e-07, -3.01878906022867e-07}},
  {"Canon EOS 20D iso 1600", "Canon", "Canon EOS 20D", 1600, {3.27651360244258e-05, 3.27651360244258e-05, 3.27651360244258e-05}, {-4.29583314312522e-07, -4.29583314312522e-07, -4.29583314312522e-07}},
  {"Canon EOS 20D iso 3200", "Canon", "Canon EOS 20D", 3200, {6.24125498832736e-05, 6.24125498832736e-05, 6.24125498832736e-05}, {-1.36259041568833e-07, -1.36259041568833e-07, -1.36259041568833e-07}},

// canon eos 30d contributed by jan
  {"Canon EOS 30D iso 100",  "Canon", "Canon EOS 30D", 100,  {6.92556259964358e-06, 3.59522489081961e-06, 6.26601906888801e-06}, {-1.8693514361687e-07, -2.23033732234139e-07, -1.23015217553507e-07}},
  {"Canon EOS 30D iso 125",  "Canon", "Canon EOS 30D", 125,  {8.34384485442626e-06, 4.25989115056595e-06, 7.40680527887986e-06}, {-1.95802962732187e-07, -2.33207833252852e-07, -1.02629498528343e-07}},
  {"Canon EOS 30D iso 160",  "Canon", "Canon EOS 30D", 160,  {9.57131397814295e-06, 4.81411654251688e-06, 6.99660443050845e-06}, {-2.72958804671955e-07, -2.79019617564348e-07, 8.78922963459732e-08}},
  {"Canon EOS 30D iso 200",  "Canon", "Canon EOS 30D", 200,  {1.13581205610112e-05, 5.63602633520446e-06, 9.97600871516035e-06}, {-2.60926258554313e-07, -2.56531153425035e-07, -9.01692847509403e-08}},
  {"Canon EOS 30D iso 250",  "Canon", "Canon EOS 30D", 250,  {1.41270225082293e-05, 6.88715307885318e-06, 1.22615698666483e-05}, {-2.85786589883639e-07, -2.58238579210888e-07, -6.00549037486534e-08}},
  {"Canon EOS 30D iso 320",  "Canon", "Canon EOS 30D", 320,  {1.63245543604545e-05, 7.64311182941073e-06, 1.12251585939902e-05}, {-3.66474852702173e-07, -2.4723425681417e-07, 3.64824742230515e-07}},
  {"Canon EOS 30D iso 400",  "Canon", "Canon EOS 30D", 400,  {1.99190764275868e-05, 9.38839997634002e-06, 1.68436392977825e-05}, {-4.05177393701998e-07, -2.68062653559699e-07, 6.08579807110867e-09}},
  {"Canon EOS 30D iso 500",  "Canon", "Canon EOS 30D", 500,  {2.48703329978e-05, 1.1762680540089e-05, 2.1559137050949e-05}, {-4.21031576508175e-07, -2.6960720244979e-07, -2.68362097569541e-08}},
  {"Canon EOS 30D iso 640",  "Canon", "Canon EOS 30D", 640,  {2.95617130433054e-05, 1.36240863378755e-05, 2.00500366369905e-05}, {-5.32636518433996e-07, -2.51828457209973e-07, 8.59664476513531e-07}},
  {"Canon EOS 30D iso 800",  "Canon", "Canon EOS 30D", 800,  {3.75828608875569e-05, 1.72337263423682e-05, 3.17006422639576e-05}, {-6.69913325140232e-07, -2.86060418641343e-07, 1.28205871095802e-07}},
  {"Canon EOS 30D iso 1000", "Canon", "Canon EOS 30D", 1000, {4.7494045695989e-05, 2.14354309588551e-05, 3.91657115156405e-05}, {-7.11158723219616e-07, -1.60875462849595e-07, 4.23471591288255e-07}},
  {"Canon EOS 30D iso 1250", "Canon", "Canon EOS 30D", 1250, {5.5448452409391e-05, 2.56394458809485e-05, 3.70990066302991e-05}, {-7.88137313159257e-07, -2.60754749748091e-07, 2.02219122984187e-06}},
  {"Canon EOS 30D iso 1600", "Canon", "Canon EOS 30D", 1600, {7.01305535595335e-05, 3.2393186544808e-05, 5.88768889479757e-05}, {-8.6896087292477e-07, -2.90226362137948e-07, 6.36553272939441e-07}},
  {"Canon EOS 30D iso 3200", "Canon", "Canon EOS 30D", 3200, {0.000135260258791614, 6.22816730055036e-05, 0.000108115559426433}, {-1.26799283570817e-06, 1.1679348700365e-07, 3.03951540166954e-06}},

// canon eos 40d, contributed by houz
  {"Canon EOS 40D iso 100",  "Canon", "Canon EOS 40D", 100,  {7.39900423781777e-06, 2.9713313813436e-06, 4.21511325063104e-06}, {-6.99012510541225e-08, -1.13757516438559e-07, -4.74585576913066e-08}},
  {"Canon EOS 40D iso 125",  "Canon", "Canon EOS 40D", 125,  {9.02520177304784e-06, 3.73356648475064e-06, 5.43129834968573e-06}, {-7.57515744406694e-08, -1.47305265001261e-07, -7.86026933820827e-08}},
  {"Canon EOS 40D iso 160",  "Canon", "Canon EOS 40D", 160,  {1.00695554103393e-05, 4.02260779664769e-06, 6.43019802119816e-06}, {-9.32648100232828e-08, -1.05388947605862e-07, -1.04979783445669e-07}},
  {"Canon EOS 40D iso 200",  "Canon", "Canon EOS 40D", 200,  {1.22130386957006e-05, 5.0619783360062e-06, 7.78527682034452e-06}, {-5.47059132570496e-08, -1.39674063010831e-07, -9.43694501196755e-08}},
  {"Canon EOS 40D iso 250",  "Canon", "Canon EOS 40D", 250,  {1.52311905987912e-05, 6.18623920147629e-06, 1.00824298208424e-05}, {-5.39766189393016e-08, -1.17378821476111e-07, -9.94781704427921e-08}},
  {"Canon EOS 40D iso 320",  "Canon", "Canon EOS 40D", 320,  {1.78189984559572e-05, 6.88772516010262e-06, 1.18044807355514e-05}, {-6.73495977169258e-08, -5.41485434483161e-08, -1.13316667331978e-07}},
  {"Canon EOS 40D iso 400",  "Canon", "Canon EOS 40D", 400,  {2.25207336278158e-05, 9.16637028101817e-06, 1.51022686185286e-05}, {-6.67982810941066e-08, -1.72680707638979e-07, -1.65183704301437e-07}},
  {"Canon EOS 40D iso 500",  "Canon", "Canon EOS 40D", 500,  {2.87016816891022e-05, 1.13751303189441e-05, 1.88805885491619e-05}, {-1.66266540843082e-08, -9.50441379899329e-08, -1.18859216761113e-07}},
  {"Canon EOS 40D iso 640",  "Canon", "Canon EOS 40D", 640,  {3.33532785153493e-05, 1.2822157848651e-05, 2.22882748703524e-05}, {-9.16548547608322e-09, 2.70703820871384e-09, -1.76217684475927e-07}},
  {"Canon EOS 40D iso 800",  "Canon", "Canon EOS 40D", 800,  {4.3427335466968e-05, 1.73525437031211e-05, 2.69563542347771e-05}, {-2.31121972917224e-08, -2.37170559393232e-07, -9.23581613434914e-08}},
  {"Canon EOS 40D iso 1000", "Canon", "Canon EOS 40D", 1000, {5.46164679422114e-05, 2.170836468898e-05, 3.51997957800681e-05}, {7.64814308622481e-08, -1.19021497494906e-07, -1.28745064222917e-07}},
  {"Canon EOS 40D iso 1250", "Canon", "Canon EOS 40D", 1250, {5.92037887340325e-05, 2.30204381503481e-05, 4.09829292024051e-05}, {6.15951348896358e-07, 5.08319828653446e-07, 1.65226199327568e-08}},
  {"Canon EOS 40D iso 1600", "Canon", "Canon EOS 40D", 1600, {8.17119781522567e-05, 3.21480155099706e-05, 5.42859931775532e-05}, {4.18607021461971e-07, 9.53720721738403e-08, -7.27540162444034e-08}},
  {"Canon EOS 40D iso 3200", "Canon", "Canon EOS 40D", 3200, {0.000162632060167329, 6.31089962556068e-05, 0.000105224710396166}, {1.47894831033359e-06, 6.43348607486187e-07, 3.34708983739415e-07}},

// canon eos 50d contributed by Gary Thompson
  {"Canon EOS 50D iso 100",   "Canon", "Canon EOS 50D", 100,   {3.65119887150514e-06, 1.41804430582757e-06, 4.20056690693373e-06}, {2.49015480499094e-07, 3.59634018097147e-07, 1.6256200418467e-08}},
  {"Canon EOS 50D iso 125",   "Canon", "Canon EOS 50D", 125,   {4.12518876416891e-06, 1.70360948749046e-06, 5.08506271155967e-06}, {3.50050617207495e-07, 4.45511479706759e-07, 2.23891234341573e-08}},
  {"Canon EOS 50D iso 160",   "Canon", "Canon EOS 50D", 160,   {3.15302129746124e-06, 1.28234892739865e-06, 6.32891526211199e-06}, {6.03386008786961e-07, 7.42812879349585e-07, 2.01759659285421e-08}},
  {"Canon EOS 50D iso 200",   "Canon", "Canon EOS 50D", 200,   {1.40808342234254e-05, 5.57449962480568e-06, 7.4185890657514e-06}, {-3.26692168344733e-07, -1.9642936990796e-07, 6.75517004491447e-08}},
  {"Canon EOS 50D iso 250",   "Canon", "Canon EOS 50D", 250,   {1.68179784175387e-05, 6.52433456665819e-06, 8.99735271060846e-06}, {-3.44336902320927e-07, -1.63842632486968e-07, 9.47277627448483e-08}},
  {"Canon EOS 50D iso 320",   "Canon", "Canon EOS 50D", 320,   {5.18981814213732e-06, 2.27226438633271e-06, 1.13665496496253e-05}, {1.1916982968176e-06, 1.40302704829048e-06, 1.10308739911126e-07}},
  {"Canon EOS 50D iso 400",   "Canon", "Canon EOS 50D", 400,   {2.51186051044003e-05, 9.97422484222412e-06, 1.372753254728e-05}, {-4.45594393276475e-07, -2.29110902248345e-07, 1.80338439338472e-07}},
  {"Canon EOS 50D iso 500",   "Canon", "Canon EOS 50D", 500,   {3.06881790563018e-05, 1.20969233335181e-05, 1.68428085164239e-05}, {-4.99871099010147e-07, -2.28404116319417e-07, 2.27817347714685e-07}},
  {"Canon EOS 50D iso 640",   "Canon", "Canon EOS 50D", 640,   {9.27013403383006e-06, 4.02885065731462e-06, 2.10569627093623e-05}, {2.38811359124316e-06, 2.78966098649624e-06, 3.30618282923898e-07}},
  {"Canon EOS 50D iso 800",   "Canon", "Canon EOS 50D", 800,   {4.64998718174892e-05, 1.87242732707568e-05, 2.59927861624033e-05}, {-6.04373643538651e-07, -2.73574610217238e-07, 4.53240525623921e-07}},
  {"Canon EOS 50D iso 1000",  "Canon", "Canon EOS 50D", 1000,  {5.69601431728101e-05, 2.29192878127525e-05, 3.24220844230367e-05}, {-6.53855423172667e-07, -2.64656925332736e-07, 5.77859053180841e-07}},
  {"Canon EOS 50D iso 1250",  "Canon", "Canon EOS 50D", 1250,  {1.80826836374187e-05, 7.36340351664906e-06, 4.1278116536478e-05}, {4.68192122193593e-06, 5.57828044702822e-06, 7.03570958595317e-07}},
  {"Canon EOS 50D iso 1600",  "Canon", "Canon EOS 50D", 1600,  {8.59550845110915e-05, 3.60049534547188e-05, 5.10758257998641e-05}, {-6.66050770579754e-07, -3.96872962372533e-07, 8.57169086457167e-07}},
  {"Canon EOS 50D iso 2000",  "Canon", "Canon EOS 50D", 2000,  {0.000107940781566033, 4.48292116219906e-05, 6.41620240441875e-05}, {-8.79064858967683e-07, -4.12157576759275e-07, 1.13754381676725e-06}},
  {"Canon EOS 50D iso 2500",  "Canon", "Canon EOS 50D", 2500,  {0.00013293015580757, 5.5318474055388e-05, 7.7877025564722e-05}, {-1.01078151161403e-06, -3.49605461455842e-07, 1.81756688034203e-06}},
  {"Canon EOS 50D iso 3200",  "Canon", "Canon EOS 50D", 3200,  {0.000166089054590224, 6.95264048397821e-05, 0.00010203464940724}, {-1.0406972514094e-06, -2.77059605946228e-07, 1.7659576283223e-06}},
  {"Canon EOS 50D iso 6400",  "Canon", "Canon EOS 50D", 6400,  {0.000309716106036363, 0.000133233771008051, 0.000205206246108705}, {-3.26266707310558e-07, 1.13839116446073e-06, 4.87592784368407e-06}},
  {"Canon EOS 50D iso 12800", "Canon", "Canon EOS 50D", 12800, {0.000591454854520338, 0.000275195972130737, 0.000445695275935004}, {1.51379068534984e-06, 1.61912116109517e-06, 7.02429683081879e-06}},

// canon eos 60d, contributed by Florian Manschwetus
  {"Canon EOS 60D iso 100",   "Canon", "Canon EOS 60D",    100,   {1.31121714888708e-05, 5.26999331130292e-06, 7.99991130442632e-06}, {-2.28726793571268e-07, -2.19944694883375e-07, -9.95669585573294e-08}},
  {"Canon EOS 60D iso 125",   "Canon", "Canon EOS 60D",    125,   {1.58625324968983e-05, 6.29506057916066e-06, 9.73299874841644e-06}, {-2.76832054881688e-07, -2.43626105640731e-07, -1.2726085929006e-07}},
  {"Canon EOS 60D iso 160",   "Canon", "Canon EOS 60D",    160,   {1.96183724831609e-05, 6.14570272061983e-06, 1.20063188889166e-05}, {-3.8667061851549e-07, 1.38408362436976e-07, -2.12315437461367e-07}},
  {"Canon EOS 60D iso 200",   "Canon", "Canon EOS 60D",    200,   {2.36520689836195e-05, 9.48534452696131e-06, 1.41386584695684e-05}, {-4.69672515536032e-07, -4.35086181043988e-07, -1.81994935273287e-07}},
  {"Canon EOS 60D iso 250",   "Canon", "Canon EOS 60D",    250,   {2.76627103239648e-05, 1.13124665261589e-05, 1.71326529914965e-05}, {-4.18531919907717e-07, -4.45418302085005e-07, -2.11317863768734e-07}},
  {"Canon EOS 60D iso 320",   "Canon", "Canon EOS 60D",    320,   {3.49400533521631e-05, 1.10098018543143e-05, 2.15003700071679e-05}, {-6.20253006446156e-07, 3.26277634193263e-07, -3.05445084573876e-07}},
  {"Canon EOS 60D iso 400",   "Canon", "Canon EOS 60D",    400,   {4.10411318111962e-05, 1.66187823593164e-05, 2.49702917027962e-05}, {-6.20512736777035e-07, -5.89620740659187e-07, -1.87022885964358e-07}},
  {"Canon EOS 60D iso 500",   "Canon", "Canon EOS 60D",    500,   {4.9678144477107e-05, 2.03016672827812e-05, 3.03787708295823e-05}, {-6.79897632556612e-07, -7.1935742485329e-07, -2.10566462362852e-07}},
  {"Canon EOS 60D iso 640",   "Canon", "Canon EOS 60D",    640,   {6.0888741784475e-05, 1.87539791535157e-05, 3.74607513247103e-05}, {-8.18910680213853e-07, 9.14653189034919e-07, -2.59448420581904e-07}},
  {"Canon EOS 60D iso 800",   "Canon", "Canon EOS 60D",    800,   {7.38861537550156e-05, 2.96700973438898e-05, 4.44828952361299e-05}, {-9.90437067511467e-07, -8.72930719879446e-07, -1.39833458039997e-07}},
  {"Canon EOS 60D iso 1000",  "Canon", "Canon EOS 60D",    1000,  {8.69420628782132e-05, 3.56165444294411e-05, 5.46118466971289e-05}, {-7.30751871618242e-07, -7.6474930321168e-07, -9.12801179907822e-08}},
  {"Canon EOS 60D iso 1250",  "Canon", "Canon EOS 60D",    1250,  {0.000108223770577205, 3.33581532704339e-05, 6.72225145807548e-05}, {-8.57353671771589e-07, 2.33333821649609e-06, 7.03112091115938e-08}},
  {"Canon EOS 60D iso 1600",  "Canon", "Canon EOS 60D",    1600,  {0.000134735792090419, 5.40852298120399e-05, 8.10044536681572e-05}, {-1.26417458752746e-06, -1.02842255194929e-06, 3.0522451011112e-07}},
  {"Canon EOS 60D iso 2000",  "Canon", "Canon EOS 60D",    2000,  {0.000161871950649551, 6.60152658073274e-05, 9.87157377576594e-05}, {-8.65613910847706e-07, -8.75443096692096e-07, 9.30717262571471e-07}},
  {"Canon EOS 60D iso 2500",  "Canon", "Canon EOS 60D",    2500,  {0.000192676217884015, 5.80147457818684e-05, 0.000120963623827105}, {-1.89855657996003e-07, 5.75136192214431e-06, 1.32980071209689e-06}},
  {"Canon EOS 60D iso 3200",  "Canon", "Canon EOS 60D",    3200,  {0.000243660781741377, 9.68586113336412e-05, 0.000148510183328506}, {-8.25091606634677e-07, -2.10052509733366e-07, 1.80918912813799e-06}},
  {"Canon EOS 60D iso 4000",  "Canon", "Canon EOS 60D",    4000,  {0.000303783701473757, 0.000120107677878482, 0.000186257746287851}, {-7.08950276781649e-07, 2.21355895637694e-07, 2.40387814989085e-06}},
  {"Canon EOS 60D iso 5000",  "Canon", "Canon EOS 60D",    5000,  {0.000382191687512258, 0.000152516160409536, 0.000229405500998639}, {-6.19366663707892e-07, 1.38280808008605e-08, 4.3163730483531e-06}},
  {"Canon EOS 60D iso 6400",  "Canon", "Canon EOS 60D",    6400,  {0.000476010804396947, 0.000185507540045765, 0.0002841771622425}, {-6.1596785438463e-07, 1.38398786132011e-06, 6.18475120096443e-06}},
  {"Canon EOS 60D iso 12800", "Canon", "Canon EOS 60D",    12800, {0.000936462824660247, 0.000374183160108462, 0.000564024246890363}, {3.92781392029706e-06, 5.82587516152094e-06, 1.65614338469453e-05}},

// canon eos 70d contributed by John Pham
{"Canon EOS 70D iso 100",   "Canon", "Canon EOS 70D", 100,   {1.02366555927774e-05, 4.71336587334147e-06, 9.41518307027218e-06}, {4.77874430647099e-08, 3.61362309038995e-08, 4.64834613672136e-08}},
{"Canon EOS 70D iso 125",   "Canon", "Canon EOS 70D", 125,   {1.33832616130131e-05, 6.23755910456249e-06, 1.20304603159491e-05}, {6.67487946811199e-08, 4.94407338794371e-08, 6.53632275766298e-08}},
{"Canon EOS 70D iso 160",   "Canon", "Canon EOS 70D", 160,   {1.51063071075179e-05, 6.73989417113525e-06, 1.36188704143178e-05}, {4.06364326089709e-08, 3.07581510240634e-08, 3.96731783427566e-08}},
{"Canon EOS 70D iso 200",   "Canon", "Canon EOS 70D", 200,   {1.68161436589604e-05, 7.55208088124731e-06, 1.52033526800398e-05}, {4.52562065207674e-08, 3.32483323354023e-08, 4.38818901241461e-08}},
{"Canon EOS 70D iso 250",   "Canon", "Canon EOS 70D", 250,   {2.12678611339611e-05, 9.67439730524358e-06, 1.90295655284948e-05}, {6.5632655618193e-08, 4.74789348092554e-08, 6.36500126820641e-08}},
{"Canon EOS 70D iso 320",   "Canon", "Canon EOS 70D", 320,   {2.57675795536312e-05, 1.15026738465454e-05, 2.33241041030745e-05}, {4.61964316728602e-08, 3.3233285134337e-08, 4.39306813012744e-08}},
{"Canon EOS 70D iso 400",   "Canon", "Canon EOS 70D", 400,   {3.19860140672929e-05, 1.41997639397925e-05, 2.82026446667505e-05}, {7.01062749925707e-08, 5.12082449144834e-08, 6.79742673721593e-08}},
{"Canon EOS 70D iso 500",   "Canon", "Canon EOS 70D", 500,   {4.05331704276288e-05, 1.81685558081423e-05, 3.57776457235e-05}, {1.02622034746615e-07, 7.37461209580217e-08, 9.89643964210319e-08}},
{"Canon EOS 70D iso 640",   "Canon", "Canon EOS 70D", 640,   {4.92893947052259e-05, 2.17710960704705e-05, 4.31764769945242e-05}, {9.92894525730111e-08, 7.29162340248079e-08, 9.54021931361206e-08}},
{"Canon EOS 70D iso 800",   "Canon", "Canon EOS 70D", 800,   {6.18850947556639e-05, 2.73573565427144e-05, 5.40340172987442e-05}, {1.46744050728875e-07, 1.08373314583172e-07, 1.42034580002987e-07}},
{"Canon EOS 70D iso 1000",  "Canon", "Canon EOS 70D", 1000,  {7.72897976987699e-05, 3.44985983087504e-05, 6.78497213405127e-05}, {2.17069700570195e-07, 1.59688802687493e-07, 2.09529518121322e-07}},
{"Canon EOS 70D iso 1250",  "Canon", "Canon EOS 70D", 1250,  {9.60952939874667e-05, 4.2859323387644e-05, 8.44501715411551e-05}, {2.43027179732144e-07, 1.7688112737584e-07, 2.34147066578732e-07}},
{"Canon EOS 70D iso 1600",  "Canon", "Canon EOS 70D", 1600,  {0.000120796174208245, 5.3833641671168e-05, 0.000104918932744754}, {3.719308415964e-07, 2.60478161645679e-07, 3.32556402255019e-07}},
{"Canon EOS 70D iso 2000",  "Canon", "Canon EOS 70D", 2000,  {0.00015288788621907, 6.83805561900404e-05, 0.000133745502744302}, {5.25506624692092e-07, 3.8200431887703e-07, 4.91866367850298e-07}},
{"Canon EOS 70D iso 2500",  "Canon", "Canon EOS 70D", 2500,  {0.000186308530927853, 8.42319317394176e-05, 0.000163175522279704}, {5.93544151101002e-07, 4.2495244352555e-07, 5.53339989304596e-07}},
{"Canon EOS 70D iso 3200",  "Canon", "Canon EOS 70D", 3200,  {0.000227012129969685, 0.000103669315469858, 0.000199463861360761}, {9.60818382673305e-07, 6.7987869703717e-07, 8.44345303544965e-07}},
{"Canon EOS 70D iso 4000",  "Canon", "Canon EOS 70D", 4000,  {0.000279449720368174, 0.0001283604525642, 0.000243986711220877}, {1.35775837759487e-06, 1.03900812709858e-06, 1.31058600706174e-06}},
{"Canon EOS 70D iso 5000",  "Canon", "Canon EOS 70D", 5000,  {0.000344293617535991, 0.000157985112298176, 0.000293782899154973}, {1.97912841178628e-06, 1.48859120832054e-06, 1.86206632506448e-06}},
{"Canon EOS 70D iso 6400",  "Canon", "Canon EOS 70D", 6400,  {0.000388250503327972, 0.000179909995508149, 0.000328054455845511}, {2.76853113690083e-06, 2.0507969067918e-06, 2.53428501781705e-06}},
{"Canon EOS 70D iso 8000",  "Canon", "Canon EOS 70D", 8000,  {0.000381541327911864, 0.000178650927855174, 0.00032495474759514}, {3.59792674035572e-06, 2.71028955701499e-06, 3.43030299035555e-06}},
{"Canon EOS 70D iso 10000", "Canon", "Canon EOS 70D", 10000, {0.000574491301888283, 0.00027243680928375, 0.000477774821602298}, {6.37832181067294e-06, 4.91821464995193e-06, 6.12603913052511e-06}},
{"Canon EOS 70D iso 12800", "Canon", "Canon EOS 70D", 12800, {0.000677603755810159, 0.000329351163446979, 0.000566356079888976}, {9.74424011144617e-06, 7.57149145113493e-06, 9.18922582147459e-06}},

// canon eos 700d
{"Canon EOS 700D iso 100",       "Canon",      "Canon EOS 700D",              100,         {9.33443513895962e-06, 5.77718873224879e-06, 1.04872883444818e-05}, {1.61966247275776e-08, 9.74788495260838e-09, 1.54267238985533e-08}},
{"Canon EOS 700D iso 200",       "Canon",      "Canon EOS 700D",              200,         {9.24249878639859e-06, 6.56383760353218e-06, 1.72544810774946e-05}, {4.19843180509055e-08, 2.87002750277452e-08, 5.81643537297737e-08}},
{"Canon EOS 700D iso 400",       "Canon",      "Canon EOS 700D",              400,         {1.92683127853234e-05, 1.26400842024086e-05, 3.35234344056789e-05}, {2.48168009948757e-08, 1.72788226826623e-08, 2.55811068120994e-08}},
{"Canon EOS 700D iso 800",       "Canon",      "Canon EOS 700D",              800,         {3.38828671695122e-05, 2.25535970819288e-05, 6.32202255121967e-05}, {6.94958754788793e-08, 5.14329362684368e-08, 6.75518080488632e-08}},
{"Canon EOS 700D iso 1600",       "Canon",      "Canon EOS 700D",              1600,         {6.72655435949405e-05, 4.50050396600133e-05, 0.000120571892877895}, {2.07228245509927e-07, 1.56707564977191e-07, 2.16473914349088e-07}},
{"Canon EOS 700D iso 3200",       "Canon",      "Canon EOS 700D",              3200,         {0.000131843860280291, 9.01607663183049e-05, 0.000232541896977304}, {5.50876449916586e-07, 4.20470564375695e-07, 5.48682913517518e-07}},
{"Canon EOS 700D iso 6400",       "Canon",      "Canon EOS 700D",              6400,         {0.00021820581455627, 0.000152527572104248, 0.000367738940590004}, {1.77555090193565e-06, 1.48908974673607e-06, 1.71872197901033e-06}},
{"Canon EOS 700D iso 12800",       "Canon",      "Canon EOS 700D",              12800,         {0.000426515436770138, 0.000309979087450361, 0.000651386769201565}, {7.90967667501258e-06, 6.73343361368924e-06, 8.01117615824857e-06}},
{"Canon EOS 700D iso 25600",       "Canon",      "Canon EOS 700D",              25600,         {0.000794962023810538, 0.000620234733849788, 0.000984955007670991}, {2.87889996479252e-05, 2.55562008372421e-05, 3.27414275789509e-05}},


// canon eos 350d, contributed by Raphaël Monnard
  {"Canon EOS 350D DIGITAL iso 100",  "Canon", "Canon EOS 350D DIGITAL", 100,  {2.9939971518732e-06, 2.9939971518732e-06, 2.9939971518732e-06}, {-1.31152382549579e-07, -1.31152382549579e-07, -1.31152382549579e-07}},
  {"Canon EOS 350D DIGITAL iso 200",  "Canon", "Canon EOS 350D DIGITAL", 200,  {5.05400213358357e-06, 5.05400213358357e-06, 5.05400213358357e-06}, {-1.25251292215947e-07, -1.25251292215947e-07, -1.25251292215947e-07}},
  {"Canon EOS 350D DIGITAL iso 400",  "Canon", "Canon EOS 350D DIGITAL", 400,  {9.49544720779503e-06, 9.49544720779503e-06, 9.49544720779503e-06}, {-2.13227612511398e-07, -2.13227612511398e-07, -2.13227612511398e-07}},
  {"Canon EOS 350D DIGITAL iso 800",  "Canon", "Canon EOS 350D DIGITAL", 800,  {1.79849679087035e-05, 1.79849679087035e-05, 1.79849679087035e-05}, {-1.99538378104687e-07, -1.99538378104687e-07, -1.99538378104687e-07}},
  {"Canon EOS 350D DIGITAL iso 1600", "Canon", "Canon EOS 350D DIGITAL", 1600, {3.48239731607602e-05, 3.48239731607602e-05, 3.48239731607602e-05}, {-1.66787472157111e-07, -1.66787472157111e-07, -1.66787472157111e-07}},
  {"Canon EOS 350D iso 100",  "Canon", "Canon EOS 350D", 100,  {2.9939971518732e-06, 2.9939971518732e-06, 2.9939971518732e-06}, {-1.31152382549579e-07, -1.31152382549579e-07, -1.31152382549579e-07}},
  {"Canon EOS 350D iso 200",  "Canon", "Canon EOS 350D", 200,  {5.05400213358357e-06, 5.05400213358357e-06, 5.05400213358357e-06}, {-1.25251292215947e-07, -1.25251292215947e-07, -1.25251292215947e-07}},
  {"Canon EOS 350D iso 400",  "Canon", "Canon EOS 350D", 400,  {9.49544720779503e-06, 9.49544720779503e-06, 9.49544720779503e-06}, {-2.13227612511398e-07, -2.13227612511398e-07, -2.13227612511398e-07}},
  {"Canon EOS 350D iso 800",  "Canon", "Canon EOS 350D", 800,  {1.79849679087035e-05, 1.79849679087035e-05, 1.79849679087035e-05}, {-1.99538378104687e-07, -1.99538378104687e-07, -1.99538378104687e-07}},
  {"Canon EOS 350D iso 1600", "Canon", "Canon EOS 350D", 1600, {3.48239731607602e-05, 3.48239731607602e-05, 3.48239731607602e-05}, {-1.66787472157111e-07, -1.66787472157111e-07, -1.66787472157111e-07}},
  {"Canon EOS DIGITAL REBEL XT iso 100",  "Canon", "Canon EOS DIGITAL REBEL XT", 100,  {2.9939971518732e-06, 2.9939971518732e-06, 2.9939971518732e-06}, {-1.31152382549579e-07, -1.31152382549579e-07, -1.31152382549579e-07}},
  {"Canon EOS DIGITAL REBEL XT iso 200",  "Canon", "Canon EOS DIGITAL REBEL XT", 200,  {5.05400213358357e-06, 5.05400213358357e-06, 5.05400213358357e-06}, {-1.25251292215947e-07, -1.25251292215947e-07, -1.25251292215947e-07}},
  {"Canon EOS DIGITAL REBEL XT iso 400",  "Canon", "Canon EOS DIGITAL REBEL XT", 400,  {9.49544720779503e-06, 9.49544720779503e-06, 9.49544720779503e-06}, {-2.13227612511398e-07, -2.13227612511398e-07, -2.13227612511398e-07}},
  {"Canon EOS DIGITAL REBEL XT iso 800",  "Canon", "Canon EOS DIGITAL REBEL XT", 800,  {1.79849679087035e-05, 1.79849679087035e-05, 1.79849679087035e-05}, {-1.99538378104687e-07, -1.99538378104687e-07, -1.99538378104687e-07}},
  {"Canon EOS DIGITAL REBEL XT iso 1600", "Canon", "Canon EOS DIGITAL REBEL XT", 1600, {3.48239731607602e-05, 3.48239731607602e-05, 3.48239731607602e-05}, {-1.66787472157111e-07, -1.66787472157111e-07, -1.66787472157111e-07}},
  {"Canon EOS Kiss Digital N iso 100",  "Canon", "Canon EOS Kiss Digital N", 100,  {2.9939971518732e-06, 2.9939971518732e-06, 2.9939971518732e-06}, {-1.31152382549579e-07, -1.31152382549579e-07, -1.31152382549579e-07}},
  {"Canon EOS Kiss Digital N iso 200",  "Canon", "Canon EOS Kiss Digital N", 200,  {5.05400213358357e-06, 5.05400213358357e-06, 5.05400213358357e-06}, {-1.25251292215947e-07, -1.25251292215947e-07, -1.25251292215947e-07}},
  {"Canon EOS Kiss Digital N iso 400",  "Canon", "Canon EOS Kiss Digital N", 400,  {9.49544720779503e-06, 9.49544720779503e-06, 9.49544720779503e-06}, {-2.13227612511398e-07, -2.13227612511398e-07, -2.13227612511398e-07}},
  {"Canon EOS Kiss Digital N iso 800",  "Canon", "Canon EOS Kiss Digital N", 800,  {1.79849679087035e-05, 1.79849679087035e-05, 1.79849679087035e-05}, {-1.99538378104687e-07, -1.99538378104687e-07, -1.99538378104687e-07}},
  {"Canon EOS Kiss Digital N iso 1600", "Canon", "Canon EOS Kiss Digital N", 1600, {3.48239731607602e-05, 3.48239731607602e-05, 3.48239731607602e-05}, {-1.66787472157111e-07, -1.66787472157111e-07, -1.66787472157111e-07}},

// canon eos 400d, contributed by Ave
  {"Canon EOS 400D iso 100",  "Canon", "Canon EOS 400D DIGITAL", 100,  {3.23924209360781e-06, 3.23924209360781e-06, 3.23924209360781e-06}, {-1.51280801902123e-07, -1.51280801902123e-07, -1.51280801902123e-07}},
  {"Canon EOS 400D iso 200",  "Canon", "Canon EOS 400D DIGITAL", 200,  {5.23460800933504e-06, 5.23460800933504e-06, 5.23460800933504e-06}, {-7.83000359350187e-08, -7.83000359350187e-08, -7.83000359350187e-08}},
  {"Canon EOS 400D iso 400",  "Canon", "Canon EOS 400D DIGITAL", 400,  {9.42922200608766e-06, 9.42922200608766e-06, 9.42922200608766e-06}, {6.12980943706827e-09, 6.12980943706827e-09, 6.12980943706827e-09}},
  {"Canon EOS 400D iso 800",  "Canon", "Canon EOS 400D DIGITAL", 800,  {1.82868816068149e-05, 1.82868816068149e-05, 1.82868816068149e-05}, {9.67111182617347e-08, 9.67111182617347e-08, 9.67111182617347e-08}},
  {"Canon EOS 400D iso 1600", "Canon", "Canon EOS 400D DIGITAL", 1600, {3.60711841360623e-05, 3.60711841360623e-05, 3.60711841360623e-05}, {2.31253334344516e-07, 2.31253334344516e-07, 2.31253334344516e-07}},
  {"Canon EOS DIGITAL REBEL XTi iso 100",  "Canon", "Canon EOS DIGITAL REBEL XTi", 100,  {3.23924209360781e-06, 3.23924209360781e-06, 3.23924209360781e-06}, {-1.51280801902123e-07, -1.51280801902123e-07, -1.51280801902123e-07}},
  {"Canon EOS DIGITAL REBEL XTi iso 200",  "Canon", "Canon EOS DIGITAL REBEL XTi", 200,  {5.23460800933504e-06, 5.23460800933504e-06, 5.23460800933504e-06}, {-7.83000359350187e-08, -7.83000359350187e-08, -7.83000359350187e-08}},
  {"Canon EOS DIGITAL REBEL XTi iso 400",  "Canon", "Canon EOS DIGITAL REBEL XTi", 400,  {9.42922200608766e-06, 9.42922200608766e-06, 9.42922200608766e-06}, {6.12980943706827e-09, 6.12980943706827e-09, 6.12980943706827e-09}},
  {"Canon EOS DIGITAL REBEL XTi iso 800",  "Canon", "Canon EOS DIGITAL REBEL XTi", 800,  {1.82868816068149e-05, 1.82868816068149e-05, 1.82868816068149e-05}, {9.67111182617347e-08, 9.67111182617347e-08, 9.67111182617347e-08}},
  {"Canon EOS DIGITAL REBEL XTi iso 1600", "Canon", "Canon EOS DIGITAL REBEL XTi", 1600, {3.60711841360623e-05, 3.60711841360623e-05, 3.60711841360623e-05}, {2.31253334344516e-07, 2.31253334344516e-07, 2.31253334344516e-07}},
  {"Canon EOS Kiss Digital X iso 100",  "Canon", "Canon EOS Kiss Digital X", 100,  {3.23924209360781e-06, 3.23924209360781e-06, 3.23924209360781e-06}, {-1.51280801902123e-07, -1.51280801902123e-07, -1.51280801902123e-07}},
  {"Canon EOS Kiss Digital X iso 200",  "Canon", "Canon EOS Kiss Digital X", 200,  {5.23460800933504e-06, 5.23460800933504e-06, 5.23460800933504e-06}, {-7.83000359350187e-08, -7.83000359350187e-08, -7.83000359350187e-08}},
  {"Canon EOS Kiss Digital X iso 400",  "Canon", "Canon EOS Kiss Digital X", 400,  {9.42922200608766e-06, 9.42922200608766e-06, 9.42922200608766e-06}, {6.12980943706827e-09, 6.12980943706827e-09, 6.12980943706827e-09}},
  {"Canon EOS Kiss Digital X iso 800",  "Canon", "Canon EOS Kiss Digital X", 800,  {1.82868816068149e-05, 1.82868816068149e-05, 1.82868816068149e-05}, {9.67111182617347e-08, 9.67111182617347e-08, 9.67111182617347e-08}},
  {"Canon EOS Kiss Digital X iso 1600", "Canon", "Canon EOS Kiss Digital X", 1600, {3.60711841360623e-05, 3.60711841360623e-05, 3.60711841360623e-05}, {2.31253334344516e-07, 2.31253334344516e-07, 2.31253334344516e-07}},

// more 400d data, contributed by pascal de bruijn, blue off
//{"Canon EOS 400D iso 100",  "Canon", "Canon EOS 400D DIGITAL", 100,  {3.74938685069499e-06, 3.43152295700962e-06, 0.0}, {-1.30491328338649e-08, -1.47260834256225e-07, 0.0}},
//{"Canon EOS 400D iso 200",  "Canon", "Canon EOS 400D DIGITAL", 200,  {6.45847930614648e-06, 5.33095216151134e-06, 0.0}, {-1.55154823493019e-08, -1.0065155340124e-07,  0.0}},
//{"Canon EOS 400D iso 400",  "Canon", "Canon EOS 400D DIGITAL", 400,  {1.24348933930323e-05, 9.6396352308293e-06,  0.0}, {-7.31050415745556e-09, -7.18216690951134e-08, 0.0}},
//{"Canon EOS 400D iso 800",  "Canon", "Canon EOS 400D DIGITAL", 800,  {2.5402867976164e-05, 1.76450056263476e-05,  0.0}, {-8.83109357635946e-11, 1.73955006722456e-07,  0.0}},
//{"Canon EOS 400D iso 1600", "Canon", "Canon EOS 400D DIGITAL", 1600, {4.88766848048645e-05, 3.56747172756397e-05, 0.0}, {3.87581291217768e-07, 4.64133591818911e-07,   0.0}},

// canon eos 450D, contributed by Florian Manschwetus
  {"Canon EOS 450D iso 100",  "Canon", "Canon EOS 450D", 100,  {6.68310885002143e-06, 3.25158005573989e-06, 5.98503523787481e-06}, {-2.93078317725893e-08, -1.02721932832005e-07, -1.01147557813498e-07}},
  {"Canon EOS 450D iso 200",  "Canon", "Canon EOS 450D", 200,  {1.19059746130446e-05, 5.80490821668169e-06, 1.05938873020356e-05}, {-1.06536799300496e-08, -1.41428136621121e-07, -1.21160878489446e-07}},
  {"Canon EOS 450D iso 400",  "Canon", "Canon EOS 450D", 400,  {2.30327112771758e-05, 1.11099218732025e-05, 2.05911281160199e-05}, {-3.53311433234569e-08, -2.45698935192051e-07, -2.63656161455854e-07}},
  {"Canon EOS 450D iso 800",  "Canon", "Canon EOS 450D", 800,  {4.44957328105123e-05, 2.11059397838198e-05, 3.93436408865041e-05}, {3.99007770218833e-08, -3.30632463604462e-07, -4.28198711692841e-07}},
  {"Canon EOS 450D iso 1600", "Canon", "Canon EOS 450D", 1600, {8.61612271952716e-05, 4.04104614056175e-05, 7.67722875389305e-05}, {5.16799155915551e-07, -1.69313846547132e-07, -4.87301460892088e-07}},
  {"Canon EOS DIGITAL REBEL XSi iso 100",  "Canon", "Canon EOS DIGITAL REBEL XSi", 100,  {6.68310885002143e-06, 3.25158005573989e-06, 5.98503523787481e-06}, {-2.93078317725893e-08, -1.02721932832005e-07, -1.01147557813498e-07}},
  {"Canon EOS DIGITAL REBEL XSi iso 200",  "Canon", "Canon EOS DIGITAL REBEL XSi", 200,  {1.19059746130446e-05, 5.80490821668169e-06, 1.05938873020356e-05}, {-1.06536799300496e-08, -1.41428136621121e-07, -1.21160878489446e-07}},
  {"Canon EOS DIGITAL REBEL XSi iso 400",  "Canon", "Canon EOS DIGITAL REBEL XSi", 400,  {2.30327112771758e-05, 1.11099218732025e-05, 2.05911281160199e-05}, {-3.53311433234569e-08, -2.45698935192051e-07, -2.63656161455854e-07}},
  {"Canon EOS DIGITAL REBEL XSi iso 800",  "Canon", "Canon EOS DIGITAL REBEL XSi", 800,  {4.44957328105123e-05, 2.11059397838198e-05, 3.93436408865041e-05}, {3.99007770218833e-08, -3.30632463604462e-07, -4.28198711692841e-07}},
  {"Canon EOS DIGITAL REBEL XSi iso 1600", "Canon", "Canon EOS DIGITAL REBEL XSi", 1600, {8.61612271952716e-05, 4.04104614056175e-05, 7.67722875389305e-05}, {5.16799155915551e-07, -1.69313846547132e-07, -4.87301460892088e-07}},
  {"Canon EOS Kiss X2 iso 100",  "Canon", "Canon EOS Kiss X2", 100,  {6.68310885002143e-06, 3.25158005573989e-06, 5.98503523787481e-06}, {-2.93078317725893e-08, -1.02721932832005e-07, -1.01147557813498e-07}},
  {"Canon EOS Kiss X2 iso 200",  "Canon", "Canon EOS Kiss X2", 200,  {1.19059746130446e-05, 5.80490821668169e-06, 1.05938873020356e-05}, {-1.06536799300496e-08, -1.41428136621121e-07, -1.21160878489446e-07}},
  {"Canon EOS Kiss X2 iso 400",  "Canon", "Canon EOS Kiss X2", 400,  {2.30327112771758e-05, 1.11099218732025e-05, 2.05911281160199e-05}, {-3.53311433234569e-08, -2.45698935192051e-07, -2.63656161455854e-07}},
  {"Canon EOS Kiss X2 iso 800",  "Canon", "Canon EOS Kiss X2", 800,  {4.44957328105123e-05, 2.11059397838198e-05, 3.93436408865041e-05}, {3.99007770218833e-08, -3.30632463604462e-07, -4.28198711692841e-07}},
  {"Canon EOS Kiss X2 iso 1600", "Canon", "Canon EOS Kiss X2", 1600, {8.61612271952716e-05, 4.04104614056175e-05, 7.67722875389305e-05}, {5.16799155915551e-07, -1.69313846547132e-07, -4.87301460892088e-07}},

// canon eos 500d and silly marketing name, blue broken, contributed by Chris Lindsay
  {"Canon EOS REBEL T1i iso 100",  "Canon", "Canon EOS REBEL T1i", 100,  {2.62795214550122e-06, 3.74867763427961e-06, 0.0}, {-2.52485499910036e-08, -5.98362404338554e-08, 0.0}},
  {"Canon EOS REBEL T1i iso 200",  "Canon", "Canon EOS REBEL T1i", 200,  {3.91301145901479e-06, 6.66936067502093e-06, 0.0}, {3.91167116535526e-08, -1.23689950506302e-07,  0.0}},
  {"Canon EOS REBEL T1i iso 400",  "Canon", "Canon EOS REBEL T1i", 400,  {7.2290500777017e-06, 1.11652410715795e-05,  0.0}, {1.07354830741825e-07, 1.9355276764562e-08,    0.0}},
// {"Canon EOS REBEL T1i iso 400",  "Canon", "Canon EOS REBEL T1i", 400,  {7.25057407723151e-06, 1.09389498765465e-05, 0.0}, {6.31205553524109e-08, 4.54604435722641e-08,   0.0}},
  {"Canon EOS REBEL T1i iso 800",  "Canon", "Canon EOS REBEL T1i", 800,  {1.42209526106018e-05, 2.01724093926433e-05, 0.0}, {2.45496830880531e-07, 1.66836915141575e-07,   0.0}},
  {"Canon EOS REBEL T1i iso 1600", "Canon", "Canon EOS REBEL T1i", 1600, {2.70567388007939e-05, 4.01925318743725e-05, 0.0}, {5.12112464571712e-07, 1.82671416225439e-07,   0.0}},
  {"Canon EOS REBEL T1i iso 3200", "Canon", "Canon EOS REBEL T1i", 3200, {5.45819191522443e-05, 7.81380186745157e-05, 0.0}, {1.12897288419996e-06, 4.81540194093711e-07,   0.0}},
// {"Canon EOS REBEL T1i iso 3200", "Canon", "Canon EOS REBEL T1i", 3200, {5.34542493482535e-05, 7.32437391071676e-05, -2.96603753057407e-05}, {1.69131373197382e-06, 1.52039972227823e-06, 3.21379355151765e-05}},
  {"Canon EOS 500D iso 100",  "Canon", "Canon EOS 500D", 100,  {2.62795214550122e-06, 3.74867763427961e-06, 0.0}, {-2.52485499910036e-08, -5.98362404338554e-08, 0.0}},
  {"Canon EOS 500D iso 200",  "Canon", "Canon EOS 500D", 200,  {3.91301145901479e-06, 6.66936067502093e-06, 0.0}, {3.91167116535526e-08, -1.23689950506302e-07,  0.0}},
  {"Canon EOS 500D iso 400",  "Canon", "Canon EOS 500D", 400,  {7.2290500777017e-06, 1.11652410715795e-05,  0.0}, {1.07354830741825e-07, 1.9355276764562e-08,    0.0}},
  {"Canon EOS 500D iso 800",  "Canon", "Canon EOS 500D", 800,  {1.42209526106018e-05, 2.01724093926433e-05, 0.0}, {2.45496830880531e-07, 1.66836915141575e-07,   0.0}},
  {"Canon EOS 500D iso 1600", "Canon", "Canon EOS 500D", 1600, {2.70567388007939e-05, 4.01925318743725e-05, 0.0}, {5.12112464571712e-07, 1.82671416225439e-07,   0.0}},
  {"Canon EOS 500D iso 3200", "Canon", "Canon EOS 500D", 3200, {5.45819191522443e-05, 7.81380186745157e-05, 0.0}, {1.12897288419996e-06, 4.81540194093711e-07,   0.0}},
  {"Canon EOS Kiss X3 iso 100",  "Canon", "Canon EOS Kiss X3", 100,  {2.62795214550122e-06, 3.74867763427961e-06, 0.0}, {-2.52485499910036e-08, -5.98362404338554e-08, 0.0}},
  {"Canon EOS Kiss X3 iso 200",  "Canon", "Canon EOS Kiss X3", 200,  {3.91301145901479e-06, 6.66936067502093e-06, 0.0}, {3.91167116535526e-08, -1.23689950506302e-07,  0.0}},
  {"Canon EOS Kiss X3 iso 400",  "Canon", "Canon EOS Kiss X3", 400,  {7.2290500777017e-06, 1.11652410715795e-05,  0.0}, {1.07354830741825e-07, 1.9355276764562e-08,    0.0}},
  {"Canon EOS Kiss X3 iso 800",  "Canon", "Canon EOS Kiss X3", 800,  {1.42209526106018e-05, 2.01724093926433e-05, 0.0}, {2.45496830880531e-07, 1.66836915141575e-07,   0.0}},
  {"Canon EOS Kiss X3 iso 1600", "Canon", "Canon EOS Kiss X3", 1600, {2.70567388007939e-05, 4.01925318743725e-05, 0.0}, {5.12112464571712e-07, 1.82671416225439e-07,   0.0}},
  {"Canon EOS Kiss X3 iso 3200", "Canon", "Canon EOS Kiss X3", 3200, {5.45819191522443e-05, 7.81380186745157e-05, 0.0}, {1.12897288419996e-06, 4.81540194093711e-07,   0.0}},

// canon eos 550d, christian iuga and emre meydan
  {"Canon EOS 550D iso 100",   "Canon", "Canon EOS 550D", 100,   {3.42038078891161e-06, 3.42038078891161e-06, 3.42038078891161e-06}, {1.69357216945505e-07, 1.69357216945505e-07, 1.69357216945505e-07}},
  {"Canon EOS 550D iso 200",   "Canon", "Canon EOS 550D", 200,   {8.16755837136271e-06, 8.16755837136271e-06, 8.16755837136271e-06}, {-3.02248208072624e-07, -3.02248208072624e-07, -3.02248208072624e-07}},
  {"Canon EOS 550D iso 400",   "Canon", "Canon EOS 550D", 400,   {1.43903816225803e-05, 1.43903816225803e-05, 1.43903816225803e-05}, {-4.29590582559664e-07, -4.29590582559664e-07, -4.29590582559664e-07}},
  {"Canon EOS 550D iso 800",   "Canon", "Canon EOS 550D", 800,   {2.59261141982164e-05, 2.59261141982164e-05, 2.59261141982164e-05}, {-6.27255134585092e-07, -6.27255134585092e-07, -6.27255134585092e-07}},
  {"Canon EOS 550D iso 1600",  "Canon", "Canon EOS 550D", 1600,  {4.74084403123113e-05, 4.74084403123113e-05, 4.74084403123113e-05}, {-7.11190221721385e-07, -7.11190221721385e-07, -7.11190221721385e-07}},
  {"Canon EOS 550D iso 3200",  "Canon", "Canon EOS 550D", 3200,  {8.77139608339509e-05, 8.77139608339509e-05, 8.77139608339509e-05}, {-5.01423917714178e-07, -5.01423917714178e-07, -5.01423917714178e-07}},
  {"Canon EOS 550D iso 6400",  "Canon", "Canon EOS 550D", 6400,  {0.000160273378428909, 0.000160273378428909, 0.000160273378428909}, {1.20254196898319e-06, 1.20254196898319e-06, 1.20254196898319e-06}},
  {"Canon EOS 550D iso 12800", "Canon", "Canon EOS 550D", 12800, {0.000478078146589793, 0.000329086963169747, -2.77604100359821e-05}, {1.04421579128176e-05, 5.09698006807269e-06, 0.000120444473297075}}, // just that one by eisnerd, too
  {"Canon EOS REBEL T2i iso 100",   "Canon", "Canon EOS REBEL T2i", 100,   {3.42038078891161e-06, 3.42038078891161e-06, 3.42038078891161e-06}, {1.69357216945505e-07, 1.69357216945505e-07, 1.69357216945505e-07}},
  {"Canon EOS REBEL T2i iso 200",   "Canon", "Canon EOS REBEL T2i", 200,   {8.16755837136271e-06, 8.16755837136271e-06, 8.16755837136271e-06}, {-3.02248208072624e-07, -3.02248208072624e-07, -3.02248208072624e-07}},
  {"Canon EOS REBEL T2i iso 400",   "Canon", "Canon EOS REBEL T2i", 400,   {1.43903816225803e-05, 1.43903816225803e-05, 1.43903816225803e-05}, {-4.29590582559664e-07, -4.29590582559664e-07, -4.29590582559664e-07}},
  {"Canon EOS REBEL T2i iso 800",   "Canon", "Canon EOS REBEL T2i", 800,   {2.59261141982164e-05, 2.59261141982164e-05, 2.59261141982164e-05}, {-6.27255134585092e-07, -6.27255134585092e-07, -6.27255134585092e-07}},
  {"Canon EOS REBEL T2i iso 1600",  "Canon", "Canon EOS REBEL T2i", 1600,  {4.74084403123113e-05, 4.74084403123113e-05, 4.74084403123113e-05}, {-7.11190221721385e-07, -7.11190221721385e-07, -7.11190221721385e-07}},
  {"Canon EOS REBEL T2i iso 3200",  "Canon", "Canon EOS REBEL T2i", 3200,  {8.77139608339509e-05, 8.77139608339509e-05, 8.77139608339509e-05}, {-5.01423917714178e-07, -5.01423917714178e-07, -5.01423917714178e-07}},
  {"Canon EOS REBEL T2i iso 6400",  "Canon", "Canon EOS REBEL T2i", 6400,  {0.000160273378428909, 0.000160273378428909, 0.000160273378428909}, {1.20254196898319e-06, 1.20254196898319e-06, 1.20254196898319e-06}},
  {"Canon EOS REBEL T2i iso 12800", "Canon", "Canon EOS REBEL T2i", 12800, {0.000478078146589793, 0.000329086963169747, -2.77604100359821e-05}, {1.04421579128176e-05, 5.09698006807269e-06, 0.000120444473297075}},
  {"Canon EOS Kiss X4 iso 100",   "Canon", "Canon EOS Kiss X4", 100,   {3.42038078891161e-06, 3.42038078891161e-06, 3.42038078891161e-06}, {1.69357216945505e-07, 1.69357216945505e-07, 1.69357216945505e-07}},
  {"Canon EOS Kiss X4 iso 200",   "Canon", "Canon EOS Kiss X4", 200,   {8.16755837136271e-06, 8.16755837136271e-06, 8.16755837136271e-06}, {-3.02248208072624e-07, -3.02248208072624e-07, -3.02248208072624e-07}},
  {"Canon EOS Kiss X4 iso 400",   "Canon", "Canon EOS Kiss X4", 400,   {1.43903816225803e-05, 1.43903816225803e-05, 1.43903816225803e-05}, {-4.29590582559664e-07, -4.29590582559664e-07, -4.29590582559664e-07}},
  {"Canon EOS Kiss X4 iso 800",   "Canon", "Canon EOS Kiss X4", 800,   {2.59261141982164e-05, 2.59261141982164e-05, 2.59261141982164e-05}, {-6.27255134585092e-07, -6.27255134585092e-07, -6.27255134585092e-07}},
  {"Canon EOS Kiss X4 iso 1600",  "Canon", "Canon EOS Kiss X4", 1600,  {4.74084403123113e-05, 4.74084403123113e-05, 4.74084403123113e-05}, {-7.11190221721385e-07, -7.11190221721385e-07, -7.11190221721385e-07}},
  {"Canon EOS Kiss X4 iso 3200",  "Canon", "Canon EOS Kiss X4", 3200,  {8.77139608339509e-05, 8.77139608339509e-05, 8.77139608339509e-05}, {-5.01423917714178e-07, -5.01423917714178e-07, -5.01423917714178e-07}},
  {"Canon EOS Kiss X4 iso 6400",  "Canon", "Canon EOS Kiss X4", 6400,  {0.000160273378428909, 0.000160273378428909, 0.000160273378428909}, {1.20254196898319e-06, 1.20254196898319e-06, 1.20254196898319e-06}},
  {"Canon EOS Kiss X4 iso 12800", "Canon", "Canon EOS Kiss X4", 12800, {0.000478078146589793, 0.000329086963169747, -2.77604100359821e-05}, {1.04421579128176e-05, 5.09698006807269e-06, 0.000120444473297075}},
// backup presets, same camera (for comparison/later averaging) contributed by eisnerd
// {"Canon EOS 550D iso 100",  "Canon", "Canon EOS 550D", 100,  {7.28641719672708e-06, 3.10998718805443e-06, -1.65777441382599e-06}, {-9.07222924461287e-08, 4.15284304120892e-07, 1.71603575577396e-06}},
// {"Canon EOS 550D iso 200",  "Canon", "Canon EOS 550D", 200,  {1.28190961128002e-05, 9.30129672280659e-06, -4.22323606267509e-08}, {-1.83392932947613e-07, -3.70902159949275e-07, 3.02053855373243e-06}},
// {"Canon EOS 550D iso 400",  "Canon", "Canon EOS 550D", 400,  {2.21807031134581e-05, 1.68544302418158e-05, -4.55683440260447e-07}, {-2.48967025993626e-07, -7.02531999960958e-07, 5.39220988655476e-06}},
// {"Canon EOS 550D iso 800",  "Canon", "Canon EOS 550D", 800,  {3.88965276508691e-05, 2.94140626721987e-05, -8.63105129933908e-07}, {-2.75604216266995e-07, -1.10515326521988e-06, 9.50126593148967e-06}},
// {"Canon EOS 550D iso 1600", "Canon", "Canon EOS 550D", 1600, {6.88556069308421e-05, 5.19255165703588e-05, -1.64338880841229e-06}, {2.76479946475538e-08, -1.45790225370961e-06, 1.72611616855464e-05}},
// {"Canon EOS 550D iso 3200", "Canon", "Canon EOS 550D", 3200, {0.000128827155984775, 9.22135121524709e-05, -3.79496169478196e-06}, {4.96917353601182e-07, -1.37043683761658e-06, 3.1892304781311e-05}},
// {"Canon EOS 550D iso 6400", "Canon", "Canon EOS 550D", 6400, {0.000244596474004226, 0.00017020643478596, -1.55641062563628e-05}, {2.92750763415957e-06, 3.99086390607721e-07, 6.23048555111102e-05}},

// canon eos 600d contributed by richard levitte
  {"Canon EOS 600D iso 100",   "Canon", "Canon EOS 600D", 100,   {1.56964927317659e-05, 5.64203987431857e-06, 8.54509144546119e-06}, {-2.78292860134465e-07, -2.62508393627864e-07, -1.90471550107442e-07}},
  {"Canon EOS 600D iso 200",   "Canon", "Canon EOS 600D", 200,   {2.52527882903771e-05, 8.95050061845734e-06, 1.34843616136932e-05}, {-5.02221818740559e-07, -4.36387724800272e-07, -3.22667112032989e-07}},
  {"Canon EOS 600D iso 400",   "Canon", "Canon EOS 600D", 400,   {4.44048313771188e-05, 1.58267298380414e-05, 2.38068790866284e-05}, {-7.94134953266295e-07, -6.81086851472936e-07, -4.84976824637679e-07}},
  {"Canon EOS 600D iso 800",   "Canon", "Canon EOS 600D", 800,   {7.74806339502997e-05, 2.78600316089851e-05, 4.23089950020385e-05}, {-1.05973485640312e-06, -9.25815397474957e-07, -6.25594478987156e-07}},
  {"Canon EOS 600D iso 1600",  "Canon", "Canon EOS 600D", 1600,  {0.000138092325126929, 4.98770571764875e-05, 7.6408940925059e-05}, {-1.20216152929454e-06, -1.06229175524964e-06, -5.46230880272032e-07}},
  {"Canon EOS 600D iso 3200",  "Canon", "Canon EOS 600D", 3200,  {0.0002520680164703, 9.1215331102101e-05, 0.000140221278484595}, {-9.49921854006117e-07, -8.71331914763491e-07, -1.42011607904406e-07}},
  {"Canon EOS 600D iso 6400",  "Canon", "Canon EOS 600D", 6400,  {0.000472648193622493, 0.000175879568992008, 0.000274872960874646}, {1.42044885190645e-06, 3.45352931149838e-08, 1.04663399988129e-06}},
  {"Canon EOS 600D iso 12800", "Canon", "Canon EOS 600D", 12800, {0.000879033570586821, 0.000344745711358442, 0.000539039854971459}, {1.25023825152327e-05, 4.05404708427205e-06, 6.45543205845978e-06}},
  {"Canon EOS Kiss X5 iso 100",   "Canon", "Canon EOS Kiss X5", 100,   {1.56964927317659e-05, 5.64203987431857e-06, 8.54509144546119e-06}, {-2.78292860134465e-07, -2.62508393627864e-07, -1.90471550107442e-07}},
  {"Canon EOS Kiss X5 iso 200",   "Canon", "Canon EOS Kiss X5", 200,   {2.52527882903771e-05, 8.95050061845734e-06, 1.34843616136932e-05}, {-5.02221818740559e-07, -4.36387724800272e-07, -3.22667112032989e-07}},
  {"Canon EOS Kiss X5 iso 400",   "Canon", "Canon EOS Kiss X5", 400,   {4.44048313771188e-05, 1.58267298380414e-05, 2.38068790866284e-05}, {-7.94134953266295e-07, -6.81086851472936e-07, -4.84976824637679e-07}},
  {"Canon EOS Kiss X5 iso 800",   "Canon", "Canon EOS Kiss X5", 800,   {7.74806339502997e-05, 2.78600316089851e-05, 4.23089950020385e-05}, {-1.05973485640312e-06, -9.25815397474957e-07, -6.25594478987156e-07}},
  {"Canon EOS Kiss X5 iso 1600",  "Canon", "Canon EOS Kiss X5", 1600,  {0.000138092325126929, 4.98770571764875e-05, 7.6408940925059e-05}, {-1.20216152929454e-06, -1.06229175524964e-06, -5.46230880272032e-07}},
  {"Canon EOS Kiss X5 iso 3200",  "Canon", "Canon EOS Kiss X5", 3200,  {0.0002520680164703, 9.1215331102101e-05, 0.000140221278484595}, {-9.49921854006117e-07, -8.71331914763491e-07, -1.42011607904406e-07}},
  {"Canon EOS Kiss X5 iso 6400",  "Canon", "Canon EOS Kiss X5", 6400,  {0.000472648193622493, 0.000175879568992008, 0.000274872960874646}, {1.42044885190645e-06, 3.45352931149838e-08, 1.04663399988129e-06}},
  {"Canon EOS Kiss X5 iso 12800", "Canon", "Canon EOS Kiss X5", 12800, {0.000879033570586821, 0.000344745711358442, 0.000539039854971459}, {1.25023825152327e-05, 4.05404708427205e-06, 6.45543205845978e-06}},
  {"Canon EOS REBEL T3i iso 100",   "Canon", "Canon EOS REBEL T3i", 100,   {1.56964927317659e-05, 5.64203987431857e-06, 8.54509144546119e-06}, {-2.78292860134465e-07, -2.62508393627864e-07, -1.90471550107442e-07}},
  {"Canon EOS REBEL T3i iso 200",   "Canon", "Canon EOS REBEL T3i", 200,   {2.52527882903771e-05, 8.95050061845734e-06, 1.34843616136932e-05}, {-5.02221818740559e-07, -4.36387724800272e-07, -3.22667112032989e-07}},
  {"Canon EOS REBEL T3i iso 400",   "Canon", "Canon EOS REBEL T3i", 400,   {4.44048313771188e-05, 1.58267298380414e-05, 2.38068790866284e-05}, {-7.94134953266295e-07, -6.81086851472936e-07, -4.84976824637679e-07}},
  {"Canon EOS REBEL T3i iso 800",   "Canon", "Canon EOS REBEL T3i", 800,   {7.74806339502997e-05, 2.78600316089851e-05, 4.23089950020385e-05}, {-1.05973485640312e-06, -9.25815397474957e-07, -6.25594478987156e-07}},
  {"Canon EOS REBEL T3i iso 1600",  "Canon", "Canon EOS REBEL T3i", 1600,  {0.000138092325126929, 4.98770571764875e-05, 7.6408940925059e-05}, {-1.20216152929454e-06, -1.06229175524964e-06, -5.46230880272032e-07}},
  {"Canon EOS REBEL T3i iso 3200",  "Canon", "Canon EOS REBEL T3i", 3200,  {0.0002520680164703, 9.1215331102101e-05, 0.000140221278484595}, {-9.49921854006117e-07, -8.71331914763491e-07, -1.42011607904406e-07}},
  {"Canon EOS REBEL T3i iso 6400",  "Canon", "Canon EOS REBEL T3i", 6400,  {0.000472648193622493, 0.000175879568992008, 0.000274872960874646}, {1.42044885190645e-06, 3.45352931149838e-08, 1.04663399988129e-06}},
  {"Canon EOS REBEL T3i iso 12800", "Canon", "Canon EOS REBEL T3i", 12800, {0.000879033570586821, 0.000344745711358442, 0.000539039854971459}, {1.25023825152327e-05, 4.05404708427205e-06, 6.45543205845978e-06}},

// canon eos 650d contributed by anton katsarov
  {"Canon EOS 650D iso 100",   "Canon", "Canon EOS 650D", 100,   {1.09222980654353e-05, 4.99615555350955e-06, 8.28395119656996e-06}, {-1.42599057683976e-07, -1.112552836445e-07, -4.43752494740867e-08}},
  {"Canon EOS 650D iso 200",   "Canon", "Canon EOS 650D", 200,   {1.84061293994185e-05, 8.44943939928403e-06, 1.39498610055119e-05}, {-2.98860135470659e-07, -2.44546651829809e-07, -1.60882920252563e-07}},
  {"Canon EOS 650D iso 400",   "Canon", "Canon EOS 650D", 400,   {3.5246059095079e-05, 1.56064729790192e-05, 2.54811146408561e-05}, {-6.68894219003798e-07, -4.53283917748811e-07, -2.35347468161495e-07}},
  {"Canon EOS 650D iso 800",   "Canon", "Canon EOS 650D", 800,   {6.19926281745852e-05, 2.80757273854255e-05, 4.61052995730811e-05}, {-8.69452090936115e-07, -6.53294485290291e-07, -2.71493045588635e-07}},
  {"Canon EOS 650D iso 1600",  "Canon", "Canon EOS 650D", 1600,  {0.000116105492644914, 5.18757939781702e-05, 8.66342500485647e-05}, {-1.05818793928176e-06, -5.37115578819023e-07, 1.49311865341044e-07}},
  {"Canon EOS 650D iso 3200",  "Canon", "Canon EOS 650D", 3200,  {0.000200408412611951, 8.88557877323598e-05, 0.000140380200379715}, {1.30344292891158e-07, 1.17195296514638e-06, 3.41243799778574e-06}},
  {"Canon EOS 650D iso 6400",  "Canon", "Canon EOS 650D", 6400,  {0.000344339214778842, 0.000155839461100911, 0.000238197939667864}, {3.51438843256531e-06, 5.08193323598392e-06, 9.30241162608003e-06}},
  {"Canon EOS 650D iso 12800", "Canon", "Canon EOS 650D", 12800, {0.000716806603861633, 0.00031792383247977, 0.000490672045092177}, {7.83487414496372e-06, 1.67704953732377e-05, 2.17506580050997e-05}},
  {"Canon EOS 650D iso 25600", "Canon", "Canon EOS 650D", 25600, {0.00130371244430682, 0.000585809344484062, 0.000896904948344877}, {3.38973850953549e-05, 5.51123712105227e-05, 5.9273541039609e-05}},
  {"Canon EOS Kiss X6i iso 100",   "Canon", "Canon EOS Kiss X6i", 100,   {1.09222980654353e-05, 4.99615555350955e-06, 8.28395119656996e-06}, {-1.42599057683976e-07, -1.112552836445e-07, -4.43752494740867e-08}},
  {"Canon EOS Kiss X6i iso 200",   "Canon", "Canon EOS Kiss X6i", 200,   {1.84061293994185e-05, 8.44943939928403e-06, 1.39498610055119e-05}, {-2.98860135470659e-07, -2.44546651829809e-07, -1.60882920252563e-07}},
  {"Canon EOS Kiss X6i iso 400",   "Canon", "Canon EOS Kiss X6i", 400,   {3.5246059095079e-05, 1.56064729790192e-05, 2.54811146408561e-05}, {-6.68894219003798e-07, -4.53283917748811e-07, -2.35347468161495e-07}},
  {"Canon EOS Kiss X6i iso 800",   "Canon", "Canon EOS Kiss X6i", 800,   {6.19926281745852e-05, 2.80757273854255e-05, 4.61052995730811e-05}, {-8.69452090936115e-07, -6.53294485290291e-07, -2.71493045588635e-07}},
  {"Canon EOS Kiss X6i iso 1600",  "Canon", "Canon EOS Kiss X6i", 1600,  {0.000116105492644914, 5.18757939781702e-05, 8.66342500485647e-05}, {-1.05818793928176e-06, -5.37115578819023e-07, 1.49311865341044e-07}},
  {"Canon EOS Kiss X6i iso 3200",  "Canon", "Canon EOS Kiss X6i", 3200,  {0.000200408412611951, 8.88557877323598e-05, 0.000140380200379715}, {1.30344292891158e-07, 1.17195296514638e-06, 3.41243799778574e-06}},
  {"Canon EOS Kiss X6i iso 6400",  "Canon", "Canon EOS Kiss X6i", 6400,  {0.000344339214778842, 0.000155839461100911, 0.000238197939667864}, {3.51438843256531e-06, 5.08193323598392e-06, 9.30241162608003e-06}},
  {"Canon EOS Kiss X6i iso 12800", "Canon", "Canon EOS Kiss X6i", 12800, {0.000716806603861633, 0.00031792383247977, 0.000490672045092177}, {7.83487414496372e-06, 1.67704953732377e-05, 2.17506580050997e-05}},
  {"Canon EOS Kiss X6i iso 25600", "Canon", "Canon EOS Kiss X6i", 25600, {0.00130371244430682, 0.000585809344484062, 0.000896904948344877}, {3.38973850953549e-05, 5.51123712105227e-05, 5.9273541039609e-05}},
  {"Canon EOS REBEL T4i iso 100",   "Canon", "Canon EOS REBEL T4i", 100,   {1.09222980654353e-05, 4.99615555350955e-06, 8.28395119656996e-06}, {-1.42599057683976e-07, -1.112552836445e-07, -4.43752494740867e-08}},
  {"Canon EOS REBEL T4i iso 200",   "Canon", "Canon EOS REBEL T4i", 200,   {1.84061293994185e-05, 8.44943939928403e-06, 1.39498610055119e-05}, {-2.98860135470659e-07, -2.44546651829809e-07, -1.60882920252563e-07}},
  {"Canon EOS REBEL T4i iso 400",   "Canon", "Canon EOS REBEL T4i", 400,   {3.5246059095079e-05, 1.56064729790192e-05, 2.54811146408561e-05}, {-6.68894219003798e-07, -4.53283917748811e-07, -2.35347468161495e-07}},
  {"Canon EOS REBEL T4i iso 800",   "Canon", "Canon EOS REBEL T4i", 800,   {6.19926281745852e-05, 2.80757273854255e-05, 4.61052995730811e-05}, {-8.69452090936115e-07, -6.53294485290291e-07, -2.71493045588635e-07}},
  {"Canon EOS REBEL T4i iso 1600",  "Canon", "Canon EOS REBEL T4i", 1600,  {0.000116105492644914, 5.18757939781702e-05, 8.66342500485647e-05}, {-1.05818793928176e-06, -5.37115578819023e-07, 1.49311865341044e-07}},
  {"Canon EOS REBEL T4i iso 3200",  "Canon", "Canon EOS REBEL T4i", 3200,  {0.000200408412611951, 8.88557877323598e-05, 0.000140380200379715}, {1.30344292891158e-07, 1.17195296514638e-06, 3.41243799778574e-06}},
  {"Canon EOS REBEL T4i iso 6400",  "Canon", "Canon EOS REBEL T4i", 6400,  {0.000344339214778842, 0.000155839461100911, 0.000238197939667864}, {3.51438843256531e-06, 5.08193323598392e-06, 9.30241162608003e-06}},
  {"Canon EOS REBEL T4i iso 12800", "Canon", "Canon EOS REBEL T4i", 12800, {0.000716806603861633, 0.00031792383247977, 0.000490672045092177}, {7.83487414496372e-06, 1.67704953732377e-05, 2.17506580050997e-05}},
  {"Canon EOS REBEL T4i iso 25600", "Canon", "Canon EOS REBEL T4i", 25600, {0.00130371244430682, 0.000585809344484062, 0.000896904948344877}, {3.38973850953549e-05, 5.51123712105227e-05, 5.9273541039609e-05}},

// canon eos whatever, also goes by marketing name of rebel sl1. contributed by Rich Renomeron
{"Canon EOS REBEL SL1 iso 100",       "Canon",      "Canon EOS REBEL SL1",              100,         {8.40290759997169e-06, 3.42391883273338e-06, 5.54593037662222e-06}, {2.45733679880351e-08, 1.73076992795037e-08, 2.94599944606634e-08}},
{"Canon EOS REBEL SL1 iso 200",       "Canon",      "Canon EOS REBEL SL1",              200,         {1.51597436077441e-05, 6.20864883759421e-06, 1.00515756313269e-05}, {2.94352285495318e-08, 1.95811284846343e-08, 3.9147549779967e-08}},
{"Canon EOS REBEL SL1 iso 400",       "Canon",      "Canon EOS REBEL SL1",              400,         {2.79038974465957e-05, 1.14006503003557e-05, 1.85318392850722e-05}, {5.68401446819346e-08, 3.60437017476378e-08, 6.77948800652848e-08}},
{"Canon EOS REBEL SL1 iso 800",       "Canon",      "Canon EOS REBEL SL1",              800,         {5.17436685457657e-05, 2.09830715189085e-05, 3.37396889822256e-05}, {1.22432624409086e-07, 8.8353509456234e-08, 1.51301336631871e-07}},
{"Canon EOS REBEL SL1 iso 1600",       "Canon",      "Canon EOS REBEL SL1",              1600,         {9.77387589116612e-05, 3.94226308615317e-05, 6.54710543720829e-05}, {3.21501129165801e-07, 2.45287317884165e-07, 3.54018468451164e-07}},
{"Canon EOS REBEL SL1 iso 3200",       "Canon",      "Canon EOS REBEL SL1",              3200,         {0.000188291003106508, 7.59549569895622e-05, 0.000119668174088052}, {8.21996339051808e-07, 6.93571623725504e-07, 9.31329162959799e-07}},
{"Canon EOS REBEL SL1 iso 6400",       "Canon",      "Canon EOS REBEL SL1",              6400,         {0.000343350208678828, 0.000140691236731272, 0.000224591531379189}, {2.26227332579486e-06, 2.09770695969866e-06, 2.48823795196786e-06}},
{"Canon EOS REBEL SL1 iso 12800",       "Canon",      "Canon EOS REBEL SL1",              12800,         {0.000706986384971469, 0.000302576502107097, 0.00045608855832935}, {5.74137588345948e-06, 5.77921173181026e-06, 6.57841206802419e-06}},
{"Canon EOS REBEL SL1 iso 25600",       "Canon",      "Canon EOS REBEL SL1",              25600,         {0.00137321964382331, 0.000608913386825599, 0.000882351746829485}, {1.45103905105436e-05, 1.7475786412304e-05, 1.77179366287965e-05}},

// canon eos 1000d contributed by stefano fornari
{"Canon EOS 1000D iso 100",  "Canon", "Canon EOS 1000D", 100,  {4.71791230780098e-06, 2.5515140530151e-06, 5.5858147939005e-06}, {3.86212416322e-08, 2.94326864701975e-08, 3.88396346374234e-08}},
{"Canon EOS 1000D iso 200",  "Canon", "Canon EOS 1000D", 200,  {7.42242236324998e-06, 3.96286152884993e-06, 8.70015351739806e-06}, {4.32811265603039e-08, 3.08989328806359e-08, 4.50757394157793e-08}},
{"Canon EOS 1000D iso 400",  "Canon", "Canon EOS 1000D", 400,  {1.4117394149149e-05, 7.5719882306795e-06, 1.6640183865513e-05}, {7.19060515072525e-08, 5.14660549350632e-08, 7.95100688120219e-08}},
{"Canon EOS 1000D iso 800",  "Canon", "Canon EOS 1000D", 800,  {2.81560714796828e-05, 1.48649134393785e-05, 3.35015803996356e-05}, {1.37383712843412e-07, 9.7682024352611e-08, 1.41092404495695e-07}},
{"Canon EOS 1000D iso 1600", "Canon", "Canon EOS 1000D", 1600, {5.77366446659682e-05, 3.01911227563509e-05, 6.72930761927429e-05}, {3.29180275796318e-07, 2.29872185793448e-07, 3.29173463595823e-07}},

// Canon EOS 1100D, contributed by levente novák
// EOS 1100D == REBEL T3. Note: The camera name for T3 are guessed based on the T3i strings.
  {"Canon EOS 1100D iso 100",       "Canon",      "Canon EOS 1100D",              100,         {6.18589472171319e-06, 2.66429865728526e-06, 4.09058109030574e-06}, {-3.78123404507743e-08, -5.23556786817321e-08, -1.50271401265386e-08}},
  {"Canon EOS 1100D iso 200",       "Canon",      "Canon EOS 1100D",              200,         {1.04215498490955e-05, 4.3334160734584e-06, 7.07965226648673e-06}, {7.26737074799896e-09, 1.2646251114546e-08, 2.19779267553642e-08}},
  {"Canon EOS 1100D iso 400",       "Canon",      "Canon EOS 1100D",              400,         {1.94661204458126e-05, 8.07658820643526e-06, 1.3379400744701e-05}, {8.34003276431836e-08, 8.80976634839685e-08, 8.07686813739757e-08}},
  {"Canon EOS 1100D iso 800",       "Canon",      "Canon EOS 1100D",              800,         {3.73547174279682e-05, 1.57416767786194e-05, 2.55920866097905e-05}, {2.1392237462742e-07, 1.63459263655984e-07, 2.72007036293641e-07}},
  {"Canon EOS 1100D iso 1600",       "Canon",      "Canon EOS 1100D",              1600,         {7.19436235982651e-05, 2.9630641031117e-05, 5.01943980797386e-05}, {7.16114879731369e-07, 7.44198832366465e-07, 6.34976586025192e-07}},
  {"Canon EOS 1100D iso 3200",       "Canon",      "Canon EOS 1100D",              3200,         {0.000146256218105018, 6.0053651777805e-05, 9.89786155946811e-05}, {1.30341930331394e-06, 1.37112765817257e-06, 1.71313582318351e-06}},
  {"Canon EOS 1100D iso 6400",       "Canon",      "Canon EOS 1100D",              6400,         {0.000304902209423889, 0.000127824045974259, 0.000205600497846824}, {2.09948526590663e-06, 1.73581748166543e-06, 3.17196719392282e-06}},
  {"Canon EOS REBEL T3 iso 100",       "Canon",      "Canon EOS REBEL T3",              100,         {6.18589472171319e-06, 2.66429865728526e-06, 4.09058109030574e-06}, {-3.78123404507743e-08, -5.23556786817321e-08, -1.50271401265386e-08}},
  {"Canon EOS REBEL T3 iso 200",       "Canon",      "Canon EOS REBEL T3",              200,         {1.04215498490955e-05, 4.3334160734584e-06, 7.07965226648673e-06}, {7.26737074799896e-09, 1.2646251114546e-08, 2.19779267553642e-08}},
  {"Canon EOS REBEL T3 iso 400",       "Canon",      "Canon EOS REBEL T3",              400,         {1.94661204458126e-05, 8.07658820643526e-06, 1.3379400744701e-05}, {8.34003276431836e-08, 8.80976634839685e-08, 8.07686813739757e-08}},
  {"Canon EOS REBEL T3 iso 800",       "Canon",      "Canon EOS REBEL T3",              800,         {3.73547174279682e-05, 1.57416767786194e-05, 2.55920866097905e-05}, {2.1392237462742e-07, 1.63459263655984e-07, 2.72007036293641e-07}},
  {"Canon EOS REBEL T3 iso 1600",       "Canon",      "Canon EOS REBEL T3",              1600,         {7.19436235982651e-05, 2.9630641031117e-05, 5.01943980797386e-05}, {7.16114879731369e-07, 7.44198832366465e-07, 6.34976586025192e-07}},
  {"Canon EOS REBEL T3 iso 3200",       "Canon",      "Canon EOS REBEL T3",              3200,         {0.000146256218105018, 6.0053651777805e-05, 9.89786155946811e-05}, {1.30341930331394e-06, 1.37112765817257e-06, 1.71313582318351e-06}},
  {"Canon EOS REBEL T3 iso 6400",       "Canon",      "Canon EOS REBEL T3",              6400,         {0.000304902209423889, 0.000127824045974259, 0.000205600497846824}, {2.09948526590663e-06, 1.73581748166543e-06, 3.17196719392282e-06}},

// powershot g9 contributed by Piter Dias
  {"Canon PowerShot G9 iso 80",   "Canon", "Canon PowerShot G9", 80,   {4.16221268153887e-05, 1.59430548098424e-05, 2.94563535158547e-05}, {8.63963096063442e-08, 8.50946111585325e-08, 1.61731732878457e-07}},
  {"Canon PowerShot G9 iso 100",  "Canon", "Canon PowerShot G9", 100,  {5.26548456566547e-05, 2.08037473939027e-05, 3.71047242837785e-05}, {1.88813946683806e-07, 1.40153820276122e-07, 2.13432236511846e-07}},
  {"Canon PowerShot G9 iso 200",  "Canon", "Canon PowerShot G9", 200,  {9.4655905000008e-05, 3.79340853308318e-05, 6.71919111220071e-05}, {6.21618420789902e-07, 5.05106460266207e-07, 7.09001508988777e-07}},
  {"Canon PowerShot G9 iso 400",  "Canon", "Canon PowerShot G9", 400,  {0.000185970887841986, 7.38236701115088e-05, 0.00013251483136741}, {2.1129320614085e-06, 1.84820891835712e-06, 2.29281074141372e-06}},
  {"Canon PowerShot G9 iso 800",  "Canon", "Canon PowerShot G9", 800,  {0.000358531929344148, 0.000143157393510934, 0.000260641680076965}, {6.36412283062147e-06, 6.55796592445082e-06, 7.67534375039396e-06}},
  {"Canon PowerShot G9 iso 1600", "Canon", "Canon PowerShot G9", 1600, {0.000764082734850105, 0.000305521247097447, 0.000554240160467605}, {2.36072210017424e-05, 2.42464650485087e-05, 2.67344789728929e-05}},

// powershot g10 contributed by alexander wagner
// duplicate measurements kept for statistical validation (maybe we should average them at some point)
  {"Canon PowerShot G10 iso 80",   "Canon", "Canon PowerShot G10", 80,   {2.73428807534154e-05, 1.50153987715131e-05, 2.92697895659219e-05}, {-2.06406594118226e-07, -6.52864840815829e-08, 5.24177674593002e-08}},
// {"Canon PowerShot G10 iso 80",   "Canon", "Canon PowerShot G10", 80,   {2.61801992656869e-05, 1.73207581957186e-05, 3.33430306959041e-05}, {-1.3237374886028e-07, -5.59362623497808e-07, -2.64400058082588e-07}},
// {"Canon PowerShot G10 iso 80",   "Canon", "Canon PowerShot G10", 80,   {2.60685366199586e-05, 1.49253014323529e-05, 2.90649116385541e-05}, {-5.60978707569432e-08, -7.14126777691565e-08, 1.44224744668023e-08}},
// {"Canon PowerShot G10 iso 80",   "Canon", "Canon PowerShot G10", 80,   {2.66718508479091e-05, 1.74179849934818e-05, 3.3488996882963e-05}, {-1.55828577635967e-07, -5.52471745221835e-07, -2.95124161138527e-07}},
// {"Canon PowerShot G10 iso 80",   "Canon", "Canon PowerShot G10", 80,   {2.61187595770946e-05, 1.61244194177805e-05, 3.05776828364577e-05}, {-1.58437888262107e-07, -1.83601324717079e-07, 1.82056906778928e-07}},
// {"Canon PowerShot G10 iso 80",   "Canon", "Canon PowerShot G10", 80,   {2.60540348307044e-05, 1.62097885093456e-05, 3.08833404089956e-05}, {-1.31622435002306e-07, -1.53795611493465e-07, 2.06238043445184e-07}},
  {"Canon PowerShot G10 iso 100",  "Canon", "Canon PowerShot G10", 100,  {3.54662866782445e-05, 2.18535302848262e-05, 4.26899143771442e-05}, {-2.8000324417082e-07, -5.25063510090621e-07, -3.07573236670584e-07}},
// {"Canon PowerShot G10 iso 100",  "Canon", "Canon PowerShot G10", 100,  {3.63422556041079e-05, 2.01265641058569e-05, 3.91618166559495e-05}, {-2.62249117517737e-07, 2.35722617899258e-09, 2.41128405548682e-07}},
// {"Canon PowerShot G10 iso 100",  "Canon", "Canon PowerShot G10", 100,  {3.65158679861917e-05, 2.03531942758461e-05, 3.93830305623315e-05}, {-2.51987661741019e-07, -2.27056123214839e-08, 2.35485352794015e-07}},
// {"Canon PowerShot G10 iso 100",  "Canon", "Canon PowerShot G10", 100,  {3.31234074777658e-05, 2.1353554899437e-05, 4.17781988497469e-05}, {-4.53630255964539e-08, -3.71149239801952e-07, -2.95145534967924e-08}},
  {"Canon PowerShot G10 iso 200",  "Canon", "Canon PowerShot G10", 200,  {6.47760899425798e-05, 3.72147039446405e-05, 7.33779866081324e-05}, {4.61510203438374e-07, 2.20930973989115e-07, 3.3480093074705e-07}},
// {"Canon PowerShot G10 iso 200",  "Canon", "Canon PowerShot G10", 200,  {6.7023877189443e-05, 3.79178923131506e-05, 7.32084901438931e-05}, {9.74143880663644e-08, 3.05116787495513e-07, 9.34841728926791e-07}},
// {"Canon PowerShot G10 iso 200",  "Canon", "Canon PowerShot G10", 200,  {6.71463329693801e-05, 3.86823399727686e-05, 7.47920181177353e-05}, {1.42700975634576e-07, 1.28430181995446e-07, 5.92574728843914e-07}},
// {"Canon PowerShot G10 iso 200",  "Canon", "Canon PowerShot G10", 200,  {6.3724485632264e-05, 3.87816358753833e-05, 7.67755192214324e-05}, {2.66696495479551e-07, 1.38711420650905e-07, 7.29447591992661e-07}},
  {"Canon PowerShot G10 iso 400",  "Canon", "Canon PowerShot G10", 400,  {0.000128587982317636, 7.10930280637188e-05, 0.000140169445190555}, {1.92601654723743e-06, 1.90451119420506e-06, 2.57195984143356e-06}},
// {"Canon PowerShot G10 iso 400",  "Canon", "Canon PowerShot G10", 400,  {0.000129939198150769, 7.21350452638313e-05, 0.000139412951314105}, {1.63053360662814e-06, 2.10867861282579e-06, 3.61805705889095e-06}},
// {"Canon PowerShot G10 iso 400",  "Canon", "Canon PowerShot G10", 400,  {0.000129188520181067, 7.16235139912162e-05, 0.000141281570792141}, {1.9037703928351e-06, 2.13115766705184e-06, 2.94001810328851e-06}},
// {"Canon PowerShot G10 iso 400",  "Canon", "Canon PowerShot G10", 400,  {0.000125763835932034, 7.34457374702767e-05, 0.000146866420711598}, {1.508640872265e-06, 1.53915240774817e-06, 2.45871017971559e-06}},
// {"Canon PowerShot G10 iso 800",  "Canon", "Canon PowerShot G10", 800,  {0.000252560171095727, 0.000136285375468082, 0.000267969499211961}, {7.35484011484535e-06, 8.51286191390169e-06, 1.12649644950225e-05}},
  {"Canon PowerShot G10 iso 800",  "Canon", "Canon PowerShot G10", 800,  {0.000255270026999922, 0.00013963736660932, 0.000271370542494757}, {8.99068229387513e-06, 8.81444687451428e-06, 1.09261899193573e-05}},
// {"Canon PowerShot G10 iso 800",  "Canon", "Canon PowerShot G10", 800,  {0.000257936869959536, 0.000142193868903855, 0.000271383060821238}, {7.67525780424473e-06, 7.9180753253684e-06, 1.21655553311143e-05}},
// {"Canon PowerShot G10 iso 800",  "Canon", "Canon PowerShot G10", 800,  {0.000239580633534094, 0.000137007720638992, 0.000272281251486359}, {1.25608600371875e-05, 1.17692043766598e-05, 1.37590252078801e-05}},
  {"Canon PowerShot G10 iso 1600", "Canon", "Canon PowerShot G10", 1600, {0.000555576893603772, 0.000276894921722596, 0.000539398544031398}, {2.25239151759371e-05, 3.0920740143651e-05, 3.96055442438055e-05}},
// {"Canon PowerShot G10 iso 1600", "Canon", "Canon PowerShot G10", 1600, {0.000553056315762539, 0.000277037440149057, 0.000543664933312324}, {2.92116061283419e-05, 3.16509847484227e-05, 3.49356221023829e-05}},
// {"Canon PowerShot G10 iso 1600", "Canon", "Canon PowerShot G10", 1600, {0.000555866781980383, 0.000278679334116548, 0.000528708642730697}, {2.31509445213517e-05, 3.07980957213539e-05, 4.3072739648452e-05}},
// {"Canon PowerShot G10 iso 1600", "Canon", "Canon PowerShot G10", 1600, {0.000539490734924118, 0.000278997045002209, 0.000537297857066124}, {2.71199689637453e-05, 3.38255680220687e-05, 4.58400831175742e-05}},

// power shot g11, contributed by leptom
{"Canon PowerShot G11 iso 80",   "Canon", "Canon PowerShot G11", 80,   {3.00053968968027e-05, 1.90601808386304e-05, 4.58737145986478e-05}, {9.87333294782291e-09, -9.70610341399634e-10, 1.69441991699024e-07}},
{"Canon PowerShot G11 iso 100",  "Canon", "Canon PowerShot G11", 100,  {3.36681459000053e-05, 2.12731719014255e-05, 5.24346419402035e-05}, {3.17721668929989e-08, 2.1396183646546e-08, 2.08208364892332e-07}},
{"Canon PowerShot G11 iso 200",  "Canon", "Canon PowerShot G11", 200,  {5.35735853993432e-05, 3.36663778749854e-05, 8.2415867272334e-05}, {1.77478176838506e-07, 1.25290493565633e-07, 4.24148653991182e-07}},
{"Canon PowerShot G11 iso 400",  "Canon", "Canon PowerShot G11", 400,  {8.69557244239545e-05, 5.52739705928068e-05, 0.000136522629884841}, {7.6713587563016e-07, 5.1544973027403e-07, 8.4545052132138e-07}},
{"Canon PowerShot G11 iso 800",  "Canon", "Canon PowerShot G11", 800,  {0.000148128842301236, 9.61671369858834e-05, 0.000238716651714023}, {1.73371031612017e-06, 1.67248910192915e-06, 2.64615376884405e-06}},
{"Canon PowerShot G11 iso 1600", "Canon", "Canon PowerShot G11", 1600, {0.000288139519992185, 0.000189105852990317, 0.000483957815754294}, {6.87372036541187e-06, 6.37012678916104e-06, 8.68624344671011e-06}},
{"Canon PowerShot G11 iso 3200", "Canon", "Canon PowerShot G11", 3200, {0.000690314289325266, 0.000408731830157916, 0.00104531033702465}, {1.78523612921471e-05, 1.7433933315988e-05, 2.08232061533943e-05}},

// power shot g12, contributed by jb
  {"Canon PowerShot G12 iso 80",   "Canon", "Canon PowerShot G12", 80,   {1.66373453340236e-05, 7.98565716624025e-06, 1.72101601478046e-05}, {-7.2494754215245e-08, -1.30431874794963e-07, -2.12027626355374e-07}},
  {"Canon PowerShot G12 iso 100",  "Canon", "Canon PowerShot G12", 100,  {2.19874157585194e-05, 1.05374764670276e-05, 2.25689918931081e-05}, {-1.16227295483149e-07, -1.54927155275663e-07, -2.49212357383336e-07}},
  {"Canon PowerShot G12 iso 125",  "Canon", "Canon PowerShot G12", 125,  {2.65987614849283e-05, 1.30316901570455e-05, 2.76200632464885e-05}, {-7.71411655291923e-08, -2.09928755764173e-07, -2.48027844734929e-07}},
  {"Canon PowerShot G12 iso 160",  "Canon", "Canon PowerShot G12", 160,  {3.38709154321323e-05, 1.63974046141945e-05, 3.51403792784587e-05}, {6.22856352171109e-09, -1.40501750242174e-07, -2.30544548091938e-07}},
  {"Canon PowerShot G12 iso 200",  "Canon", "Canon PowerShot G12", 200,  {4.18902151394745e-05, 2.00889952239158e-05, 4.24428023692456e-05}, {4.65290719464294e-09, -1.17254110313227e-07, -1.5655446648428e-07}},
  {"Canon PowerShot G12 iso 250",  "Canon", "Canon PowerShot G12", 250,  {5.26570932890087e-05, 2.54994776500875e-05, 5.40242959508379e-05}, {-8.92920294059189e-09, -2.21244122275202e-07, -2.89942199630968e-07}},
  {"Canon PowerShot G12 iso 320",  "Canon", "Canon PowerShot G12", 320,  {6.60494574099879e-05, 3.19206644443907e-05, 6.65348643565219e-05}, {2.44600840037982e-07, 3.47256952727975e-08, 1.40950063189939e-07}},
  {"Canon PowerShot G12 iso 400",  "Canon", "Canon PowerShot G12", 400,  {8.12013347179818e-05, 4.01750299113591e-05, 8.20835660973662e-05}, {5.54863348875197e-07, 1.35913574877597e-07, 5.18615270360328e-07}},
  {"Canon PowerShot G12 iso 500",  "Canon", "Canon PowerShot G12", 500,  {0.000105221371179263, 4.93285821993281e-05, 0.000103335587367201}, {4.1567137381167e-07, 4.03288925248406e-07, 6.17381688000843e-07}},
  {"Canon PowerShot G12 iso 640",  "Canon", "Canon PowerShot G12", 640,  {0.000132575795080954, 6.46228262688137e-05, 0.000129205135304144}, {1.00085293629813e-06, 5.40162135628532e-07, 1.36928007462698e-06}},
  {"Canon PowerShot G12 iso 800",  "Canon", "Canon PowerShot G12", 800,  {0.000161947266479745, 7.96412389872235e-05, 0.000160828824754609}, {1.66686931886837e-06, 7.22662463201939e-07, 1.67372212922919e-06}},
  {"Canon PowerShot G12 iso 1000", "Canon", "Canon PowerShot G12", 1000, {0.000204081176047946, 9.48607367100099e-05, 0.000196214291634401}, {2.35058163499971e-06, 2.36077060645318e-06, 2.86945276689557e-06}},
  {"Canon PowerShot G12 iso 1250", "Canon", "Canon PowerShot G12", 1250, {0.000251825167208881, 0.000122741985639404, 0.00024224232223238}, {2.53252875730771e-06, 1.61621463635483e-06, 3.9232439915753e-06}},
  {"Canon PowerShot G12 iso 1600", "Canon", "Canon PowerShot G12", 1600, {0.000330683493571144, 0.00015400885547248, 0.000310129560046854}, {3.42111323825123e-06, 3.93064652470089e-06, 6.71544325981027e-06}},
  {"Canon PowerShot G12 iso 2000", "Canon", "Canon PowerShot G12", 2000, {0.000419851272157461, 0.000190318189467426, 0.000378587862605566}, {4.04832548564553e-06, 5.44607790597673e-06, 9.06189824927038e-06}},
  {"Canon PowerShot G12 iso 2500", "Canon", "Canon PowerShot G12", 2500, {0.000520554750840581, 0.000241295337072734, 0.000463630729610204}, {1.01061903576085e-05, 9.54741263148793e-06, 1.65988103855306e-05}},
  {"Canon PowerShot G12 iso 3200", "Canon", "Canon PowerShot G12", 3200, {0.000666240521841474, 0.000297147045027247, 0.000595456552762322}, {1.20197957437732e-05, 1.55616812090161e-05, 1.94157576902133e-05}},

// power shot s90, contributed by bert blockx
  {"Canon PowerShot S90 iso 80",   "Canon", "Canon PowerShot S90", 80,   {2.23402830078831e-05, 1.13293072129965e-05, 2.26518062943997e-05}, {-1.68817328268497e-07, -1.38476611162249e-07, 5.76322409693217e-08}},
  {"Canon PowerShot S90 iso 100",  "Canon", "Canon PowerShot S90", 100,  {2.9097047305495e-05, 1.38242540389492e-05, 2.80474545595763e-05}, {-1.25693835071815e-07, 1.50790773068346e-08, 2.44150121694932e-07}},
  {"Canon PowerShot S90 iso 125",  "Canon", "Canon PowerShot S90", 125,  {3.63136013028816e-05, 1.6617761934626e-05, 3.31161324513594e-05}, {-2.2536366197286e-07, 2.29784269639736e-07, 6.32389423583499e-07}},
  {"Canon PowerShot S90 iso 160",  "Canon", "Canon PowerShot S90", 160,  {4.55460521343609e-05, 2.16011834460345e-05, 4.41996235808223e-05}, {-1.98621208425165e-07, 1.3597289208908e-07, 4.5617322399893e-07}},
  {"Canon PowerShot S90 iso 200",  "Canon", "Canon PowerShot S90", 200,  {5.59406976316606e-05, 2.52381040840054e-05, 4.96786285063045e-05}, {-1.63965094980624e-07, 5.02664937105297e-07, 1.24566095289377e-06}},
  {"Canon PowerShot S90 iso 250",  "Canon", "Canon PowerShot S90", 250,  {6.97293973865742e-05, 3.18204237598074e-05, 6.42376424027029e-05}, {-3.18170533222725e-07, 5.62944776721851e-07, 1.37905514766758e-06}},
  {"Canon PowerShot S90 iso 320",  "Canon", "Canon PowerShot S90", 320,  {8.99362282725663e-05, 3.91090745806738e-05, 8.10640441901214e-05}, {-1.92417494874308e-07, 1.355942364896e-06, 2.13458459485407e-06}},
  {"Canon PowerShot S90 iso 400",  "Canon", "Canon PowerShot S90", 400,  {0.000110709041432025, 4.75972712088692e-05, 9.77382312839684e-05}, {-3.44351967594611e-07, 1.79392684530718e-06, 3.00993175516642e-06}},
  {"Canon PowerShot S90 iso 500",  "Canon", "Canon PowerShot S90", 500,  {0.000136022284774141, 6.24171799219195e-05, 0.00012466286306786}, {2.73998853197579e-07, 1.92067098716532e-06, 3.97499987541901e-06}},
  {"Canon PowerShot S90 iso 640",  "Canon", "Canon PowerShot S90", 640,  {0.000176897163563571, 8.22533337068896e-05, 0.000152391288360623}, {5.10036649953148e-07, 2.21271977654846e-06, 7.10712782375118e-06}},
  {"Canon PowerShot S90 iso 800",  "Canon", "Canon PowerShot S90", 800,  {0.000222883880628866, 9.30697939055267e-05, 0.000181716836729403}, {9.63750323900614e-07, 5.70117038826935e-06, 9.99056944016582e-06}},
  {"Canon PowerShot S90 iso 1000", "Canon", "Canon PowerShot S90", 1000, {0.000271635363293903, 0.000127645776283878, 0.000254421412201616}, {2.5025807726172e-06, 4.36055821838095e-06, 8.4307075787327e-06}},
  {"Canon PowerShot S90 iso 1250", "Canon", "Canon PowerShot S90", 1250, {0.000347124677024373, 0.000141869737277038, 0.000288319187187432}, {5.10295070940821e-06, 1.09344676865853e-05, 1.44031491700305e-05}},
  {"Canon PowerShot S90 iso 1600", "Canon", "Canon PowerShot S90", 1600, {0.000445427638735476, 0.000187728480667588, 0.000385055387425163}, {6.18517443826724e-06, 1.27387225077043e-05, 1.65535583423842e-05}},
  {"Canon PowerShot S90 iso 2000", "Canon", "Canon PowerShot S90", 2000, {0.000554808572241562, 0.000241036003419097, 0.00048352514273103}, {8.01118957470208e-06, 1.64768952289338e-05, 2.39157319265181e-05}},
  {"Canon PowerShot S90 iso 2500", "Canon", "Canon PowerShot S90", 2500, {0.000685666313166337, 0.000310981411998743, 0.000590364894967055}, {1.96460121109787e-05, 2.40176212315589e-05, 3.76310643509494e-05}},
  {"Canon PowerShot S90 iso 3200", "Canon", "Canon PowerShot S90", 3200, {0.000921127601815889, 0.000425044676330445, 0.000821372270846736}, {2.33853361916465e-05, 3.5452722869014e-05, 5.58117794273884e-05}},

// power shot s95 by pawel zogal. variance stabilizes only up to 2.5x near dark values.
{"Canon PowerShot S95 iso 80",   "Canon", "Canon PowerShot S95", 80,   {1.18439226261264e-05, 1.00551652271641e-05, 2.80457869107516e-05}, {5.10356405355434e-08, 3.59809681163793e-08, 1.68497594765686e-07}},
{"Canon PowerShot S95 iso 100",  "Canon", "Canon PowerShot S95", 100,  {1.55347695945461e-05, 1.30629017922171e-05, 3.54436491442598e-05}, {8.46746696502538e-08, 6.31099659602214e-08, 2.53437132265783e-07}},
{"Canon PowerShot S95 iso 125",  "Canon", "Canon PowerShot S95", 125,  {1.92909360764039e-05, 1.61145681640705e-05, 4.47415970715207e-05}, {1.24942203083822e-07, 9.97216305112686e-08, 3.25904590376027e-07}},
{"Canon PowerShot S95 iso 160",  "Canon", "Canon PowerShot S95", 160,  {2.45635146194055e-05, 2.0263552776392e-05, 5.52236324321745e-05}, {1.68109011233807e-07, 1.45447605581441e-07, 4.47363006801099e-07}},
{"Canon PowerShot S95 iso 200",  "Canon", "Canon PowerShot S95", 200,  {3.05995990681018e-05, 2.52723762415818e-05, 6.90752814385808e-05}, {2.32987692414163e-07, 2.03511440075134e-07, 5.90952461283409e-07}},
{"Canon PowerShot S95 iso 250",  "Canon", "Canon PowerShot S95", 250,  {3.83506267868882e-05, 3.16903615491479e-05, 8.70739893060689e-05}, {3.6983096222261e-07, 2.99376973521879e-07, 7.63398481816495e-07}},
{"Canon PowerShot S95 iso 320",  "Canon", "Canon PowerShot S95", 320,  {4.87627533013296e-05, 4.01129412945676e-05, 0.000108339162881219}, {5.78455111838268e-07, 4.71688916148114e-07, 1.09420761904955e-06}},
{"Canon PowerShot S95 iso 400",  "Canon", "Canon PowerShot S95", 400,  {6.15606336973604e-05, 4.99553098412221e-05, 0.000135734389090747}, {7.77961150780909e-07, 6.78561101628476e-07, 1.47789885458591e-06}},
{"Canon PowerShot S95 iso 500",  "Canon", "Canon PowerShot S95", 500,  {7.57956079449427e-05, 6.18026150689876e-05, 0.000168291609051499}, {1.24369355274324e-06, 1.01616010614118e-06, 1.95852009041006e-06}},
{"Canon PowerShot S95 iso 640",  "Canon", "Canon PowerShot S95", 640,  {9.70548096569801e-05, 7.85444109510484e-05, 0.000214214830080711}, {1.89564508968999e-06, 1.58763666018465e-06, 2.87262409656247e-06}},
{"Canon PowerShot S95 iso 800",  "Canon", "Canon PowerShot S95", 800,  {0.000121225359381451, 9.79121616656253e-05, 0.000269576203306117}, {2.85266272924952e-06, 2.42595564737837e-06, 4.03550957362364e-06}},
{"Canon PowerShot S95 iso 1000", "Canon", "Canon PowerShot S95", 1000, {0.000150975107530704, 0.000121529165141403, 0.000332274364007286}, {3.98282613105306e-06, 3.45183665090733e-06, 5.52275490018129e-06}},
{"Canon PowerShot S95 iso 1250", "Canon", "Canon PowerShot S95", 1250, {0.000192532999436271, 0.000153592813171431, 0.000431963091946209}, {6.02741262051944e-06, 5.21698729699677e-06, 7.59508133347304e-06}},
{"Canon PowerShot S95 iso 1600", "Canon", "Canon PowerShot S95", 1600, {0.000249112250867348, 0.00019732468155438, 0.000556372386907657}, {9.01791856315701e-06, 7.95232395300927e-06, 1.08756384572218e-05}},
{"Canon PowerShot S95 iso 2000", "Canon", "Canon PowerShot S95", 2000, {0.000316403217955974, 0.000248677761518421, 0.000649663115106273}, {1.33869590163099e-05, 1.15411530495953e-05, 1.51471535555296e-05}},
{"Canon PowerShot S95 iso 2500", "Canon", "Canon PowerShot S95", 2500, {0.000408716078952974, 0.000318911640555892, 0.000825590779297192}, {1.94548597636044e-05, 1.6939730283834e-05, 2.11157498219989e-05}},
{"Canon PowerShot S95 iso 3200", "Canon", "Canon PowerShot S95", 3200, {0.000576241267243901, 0.000447720834863916, 0.00117840999068917}, {2.7304212571096e-05, 2.43662258991297e-05, 2.92620812038599e-05}},

// canon eos m, contributed by christoph
{"Canon EOS M iso 100",   "Canon", "Canon EOS M", 100,   {1.11641672192638e-05, 4.71521553440566e-06, 7.49808070807092e-06}, {-8.18724331824796e-08, -8.52093423929878e-08, -5.79352962234394e-08}},
// {"Canon EOS M iso 100",   "Canon", "Canon EOS M", 100,   {1.12628336204934e-05, 4.75397167484912e-06, 7.83085569652525e-06}, {-9.51975062278489e-08, -8.70487529272313e-08, -1.20448612873438e-07}},
// {"Canon EOS M iso 100",   "Canon", "Canon EOS M", 100,   {1.1234663107314e-05, 4.77667214239042e-06, 7.5823719396532e-06}, {-1.03430562647904e-07, -1.11989998035029e-07, -8.31006471653699e-08}},
// {"Canon EOS M iso 100",   "Canon", "Canon EOS M", 100,   {1.14337621123174e-05, 4.96998208236973e-06, 7.77432256117918e-06}, {-1.11641860221595e-07, -1.52367488549563e-07, -1.20000918440909e-07}},
// {"Canon EOS M iso 100",   "Canon", "Canon EOS M", 100,   {1.13541607399499e-05, 4.82251976601931e-06, 7.76071317034635e-06}, {-9.86810917617511e-08, -1.09980632548755e-07, -1.13743219739311e-07}},
{"Canon EOS M iso 200",   "Canon", "Canon EOS M", 200,   {1.87769236554544e-05, 7.90853485458251e-06, 1.28295735599131e-05}, {-2.47775856692074e-07, -2.24672580837098e-07, -2.42446959515113e-07}},
// {"Canon EOS M iso 200",   "Canon", "Canon EOS M", 200,   {1.86090576919473e-05, 7.93461586612739e-06, 1.23230116837233e-05}, {-2.14970596097142e-07, -2.44321159728957e-07, -1.71086311659733e-07}},
{"Canon EOS M iso 400",   "Canon", "Canon EOS M", 400,   {3.61295496093936e-05, 1.54446300964106e-05, 2.41765335308023e-05}, {-4.96506445801197e-07, -4.98578329020665e-07, -4.08459912502321e-07}},
// {"Canon EOS M iso 400",   "Canon", "Canon EOS M", 400,   {3.58948049880829e-05, 1.49066170736645e-05, 2.4096820259044e-05}, {-4.83102825751139e-07, -4.14053580503136e-07, -4.25338415754892e-07}},
{"Canon EOS M iso 800",   "Canon", "Canon EOS M", 800,   {6.3758034824456e-05, 2.77132275986375e-05, 4.44987095906148e-05}, {-6.03170650889967e-07, -7.66045018708318e-07, -7.20222900965185e-07}},
// {"Canon EOS M iso 800",   "Canon", "Canon EOS M", 800,   {6.39574701700126e-05, 2.71097909874132e-05, 4.25463100377908e-05}, {-7.29072109947381e-07, -6.9079471031801e-07, -4.56554072362173e-07}},
{"Canon EOS M iso 1600",  "Canon", "Canon EOS M", 1600,  {0.000115161448952614, 4.89293979937037e-05, 7.88646155262307e-05}, {-3.52080293127189e-07, -4.86581438145507e-07, -2.62418338512384e-07}},
// {"Canon EOS M iso 1600",  "Canon", "Canon EOS M", 1600,  {0.000119923009725727, 5.04698037184693e-05, 8.04818212139435e-05}, {-7.85018154838832e-07, -8.34807450043266e-07, -3.44756813419367e-07}},
// {"Canon EOS M iso 1600",  "Canon", "Canon EOS M", 1600,  {0.000121641161755257, 5.10031101536108e-05, 8.13092730461869e-05}, {-8.01577618497169e-07, -6.59814853976083e-07, -3.73784519824583e-07}},
{"Canon EOS M iso 3200",  "Canon", "Canon EOS M", 3200,  {0.000205167837222831, 8.97274115668859e-05, 0.000133955947056762}, {5.73191825622512e-07, -1.06710975316243e-07, 1.62229907024654e-06}},
// {"Canon EOS M iso 3200",  "Canon", "Canon EOS M", 3200,  {0.000204007656719233, 8.79982348151288e-05, 0.000135090613380184}, {9.80911564694389e-07, 5.10090177685395e-07, 1.77413854237045e-06}},
// {"Canon EOS M iso 3200",  "Canon", "Canon EOS M", 3200,  {0.000206705731752169, 9.10030475060457e-05, 0.000137283782784568}, {6.76113698079504e-07, -3.52676458849526e-07, 1.36267635021421e-06}},
{"Canon EOS M iso 6400",  "Canon", "Canon EOS M", 6400,  {0.000338801073171192, 0.000145583136535893, 0.000225475854759297}, {5.730648440317e-06, 5.25500443958457e-06, 6.31841978699661e-06}},
// {"Canon EOS M iso 6400",  "Canon", "Canon EOS M", 6400,  {0.000358909015783253, 0.000157736333992688, 0.000235112972904256}, {3.958574973773e-06, 2.76147561454301e-06, 4.90426044943236e-06}},
// {"Canon EOS M iso 6400",  "Canon", "Canon EOS M", 6400,  {0.000348550115084609, 0.000153727384266431, 0.000227642663753909}, {5.30886220924474e-06, 3.95447764612416e-06, 6.29047090258156e-06}},
{"Canon EOS M iso 12800", "Canon", "Canon EOS M", 12800, {0.000683222833186207, 0.000310553921274102, 0.000441956254217659}, {1.28930243056148e-05, 1.17864898555037e-05, 1.55511774455849e-05}},
// {"Canon EOS M iso 12800", "Canon", "Canon EOS M", 12800, {0.000689107114778809, 0.000309727583240152, 0.000448833860662997}, {1.34517219797395e-05, 1.2169524156222e-05, 1.48252349600781e-05}},
// {"Canon EOS M iso 12800", "Canon", "Canon EOS M", 12800, {0.000691914347219495, 0.000308354831630015, 0.000449390978839201}, {1.32820011894124e-05, 1.31393573853364e-05, 1.55286619988088e-05}},
{"Canon EOS M iso 25600", "Canon", "Canon EOS M", 25600, {0.00129515377258183, 0.000579827205148805, 0.00084223803305125}, {3.36818376755313e-05, 3.83934745670336e-05, 4.12974707703e-05}},
// {"Canon EOS M iso 25600", "Canon", "Canon EOS M", 25600, {0.00129443048527525, 0.00057441154093349, 0.000820279083672851}, {2.97186425202213e-05, 3.93368705726618e-05, 4.55450906844359e-05}},
// {"Canon EOS M iso 25600", "Canon", "Canon EOS M", 25600, {0.00129582412384482, 0.000591946851891503, 0.000834153634574368}, {3.01303857139438e-05, 3.49998178948505e-05, 3.97931355153481e-05}},

// -------------------------------------------------------------------
// Konica Minolta.
// -------------------------------------------------------------------

// konica minolta dynax 5d contributed by wolfgang kuehnel
  {"Konica Minolta Dynax 5D iso 80",   "MINOLTA", "DYNAX 5D", 80,   {4.99558922425934e-06, 3.4857105751347e-06, 5.52500669860534e-06}, {-4.45186331133066e-08, -1.66278707150093e-07, -2.54880057777153e-08}},
  {"Konica Minolta Dynax 5D iso 100",  "MINOLTA", "DYNAX 5D", 100,  {3.16291282862422e-06, 3.17298743398786e-06, -2.93528708837128e-06}, {-1.36982827450839e-08, -4.52927179882681e-08, 1.45190456679439e-06}},
  {"Konica Minolta Dynax 5D iso 200",  "MINOLTA", "DYNAX 5D", 200,  {7.44753978728847e-06, 5.04027911311091e-06, 7.42272114900283e-06}, {2.26185277905545e-08, -1.30793331527215e-07, 1.57755103477397e-07}},
//{"Konica Minolta Dynax 5D iso 200 HIGH",  "MINOLTA", "DYNAX 5D", 200,  {4.80185946195336e-06, 3.42207499074864e-06, 2.32284088957187e-06}, {-3.49226794267578e-08, -1.50017601925446e-07, 5.41602022639718e-07}},
  {"Konica Minolta Dynax 5D iso 400",  "MINOLTA", "DYNAX 5D", 400,  {1.34082174557926e-05, 9.13624999414434e-06, 1.32083536994715e-05}, {2.35640832022502e-07, -9.39167482874429e-08, 4.88377926125095e-07}},
  {"Konica Minolta Dynax 5D iso 800",  "MINOLTA", "DYNAX 5D", 800,  {2.47417413821591e-05, 1.47741919809619e-05, 2.2101423382754e-05}, {6.66200361889303e-07, 5.03164922220295e-07, 1.41766641258918e-06}},
  {"Konica Minolta Dynax 5D iso 1600", "MINOLTA", "DYNAX 5D", 1600, {4.47706028030664e-05, 2.5896677736367e-05, 3.70961411570825e-05}, {2.17227195285334e-06, 1.79295314040607e-06, 3.91326288953691e-06}},
  {"Konica Minolta Dynax 5D iso 3200", "MINOLTA", "DYNAX 5D", 3200, {8.56380293640629e-05, 4.96205912412778e-05, 5.01298354357276e-05}, {8.89456284241461e-06, 7.69970432518879e-06, 1.63832916841139e-05}},

// -------------------------------------------------------------------
// Nikon.
// -------------------------------------------------------------------

// nikon 1 V2 by lemmi
{"NIKON 1 V2 iso 160",  "NIKON CORPORATION", "NIKON 1 V2", 160,  {3.1129733316286e-05, 1.38188915177967e-05, 2.66113795791533e-05}, {2.13092759702146e-07, 1.17042036505299e-07, 1.54253439732846e-07}},
{"NIKON 1 V2 iso 200",  "NIKON CORPORATION", "NIKON 1 V2", 200,  {3.88782371257324e-05, 1.7201428754777e-05, 3.32749520540871e-05}, {3.30672272383518e-07, 1.90760668414919e-07, 2.48865074269825e-07}},
{"NIKON 1 V2 iso 400",  "NIKON CORPORATION", "NIKON 1 V2", 400,  {7.58057555233669e-05, 3.32932467223333e-05, 6.44107957999141e-05}, {4.34566005714514e-07, 2.18864576675269e-07, 2.49497266413039e-07}},
{"NIKON 1 V2 iso 800",  "NIKON CORPORATION", "NIKON 1 V2", 800,  {0.00014779353456173, 6.53249596045684e-05, 0.00012532349703287}, {8.80953021660024e-07, 4.45380923908348e-07, 5.88344440903766e-07}},
{"NIKON 1 V2 iso 1600", "NIKON CORPORATION", "NIKON 1 V2", 1600, {0.000294714580230648, 0.000129136678440125, 0.000248577353430174}, {1.96686829473346e-06, 1.18162706295073e-06, 1.47818815880857e-06}},
{"NIKON 1 V2 iso 3200", "NIKON CORPORATION", "NIKON 1 V2", 3200, {0.000590884834964985, 0.00025685568432619, 0.000491711538325884}, {4.96053366573419e-06, 3.55488211985422e-06, 4.35721631601616e-06}},
{"NIKON 1 V2 iso 6400", "NIKON CORPORATION", "NIKON 1 V2", 6400, {0.00122104852594239, 0.000537554212900666, 0.00102193794645602}, {1.1444510627609e-05, 1.11674608962633e-05, 1.25256504328047e-05}},

// copied from Nikon 1 V2
{"NIKON 1 J3 iso 160",  "NIKON CORPORATION", "NIKON 1 V2", 160,  {3.1129733316286e-05, 1.38188915177967e-05, 2.66113795791533e-05}, {2.13092759702146e-07, 1.17042036505299e-07, 1.54253439732846e-07}},
{"NIKON 1 J3 iso 200",  "NIKON CORPORATION", "NIKON 1 V2", 200,  {3.88782371257324e-05, 1.7201428754777e-05, 3.32749520540871e-05}, {3.30672272383518e-07, 1.90760668414919e-07, 2.48865074269825e-07}},
{"NIKON 1 J3 iso 400",  "NIKON CORPORATION", "NIKON 1 V2", 400,  {7.58057555233669e-05, 3.32932467223333e-05, 6.44107957999141e-05}, {4.34566005714514e-07, 2.18864576675269e-07, 2.49497266413039e-07}},
{"NIKON 1 J3 iso 800",  "NIKON CORPORATION", "NIKON 1 V2", 800,  {0.00014779353456173, 6.53249596045684e-05, 0.00012532349703287}, {8.80953021660024e-07, 4.45380923908348e-07, 5.88344440903766e-07}},
{"NIKON 1 J3 iso 1600", "NIKON CORPORATION", "NIKON 1 V2", 1600, {0.000294714580230648, 0.000129136678440125, 0.000248577353430174}, {1.96686829473346e-06, 1.18162706295073e-06, 1.47818815880857e-06}},
{"NIKON 1 J3 iso 3200", "NIKON CORPORATION", "NIKON 1 V2", 3200, {0.000590884834964985, 0.00025685568432619, 0.000491711538325884}, {4.96053366573419e-06, 3.55488211985422e-06, 4.35721631601616e-06}},
{"NIKON 1 J3 iso 6400", "NIKON CORPORATION", "NIKON 1 V2", 6400, {0.00122104852594239, 0.000537554212900666, 0.00102193794645602}, {1.1444510627609e-05, 1.11674608962633e-05, 1.25256504328047e-05}},

// copied from Nikon 1 V2
{"NIKON 1 AW1 iso 160",  "NIKON CORPORATION", "NIKON 1 V2", 160,  {3.1129733316286e-05, 1.38188915177967e-05, 2.66113795791533e-05}, {2.13092759702146e-07, 1.17042036505299e-07, 1.54253439732846e-07}},
{"NIKON 1 AW1 iso 200",  "NIKON CORPORATION", "NIKON 1 V2", 200,  {3.88782371257324e-05, 1.7201428754777e-05, 3.32749520540871e-05}, {3.30672272383518e-07, 1.90760668414919e-07, 2.48865074269825e-07}},
{"NIKON 1 AW1 iso 400",  "NIKON CORPORATION", "NIKON 1 V2", 400,  {7.58057555233669e-05, 3.32932467223333e-05, 6.44107957999141e-05}, {4.34566005714514e-07, 2.18864576675269e-07, 2.49497266413039e-07}},
{"NIKON 1 AW1 iso 800",  "NIKON CORPORATION", "NIKON 1 V2", 800,  {0.00014779353456173, 6.53249596045684e-05, 0.00012532349703287}, {8.80953021660024e-07, 4.45380923908348e-07, 5.88344440903766e-07}},
{"NIKON 1 AW1 iso 1600", "NIKON CORPORATION", "NIKON 1 V2", 1600, {0.000294714580230648, 0.000129136678440125, 0.000248577353430174}, {1.96686829473346e-06, 1.18162706295073e-06, 1.47818815880857e-06}},
{"NIKON 1 AW1 iso 3200", "NIKON CORPORATION", "NIKON 1 V2", 3200, {0.000590884834964985, 0.00025685568432619, 0.000491711538325884}, {4.96053366573419e-06, 3.55488211985422e-06, 4.35721631601616e-06}},
{"NIKON 1 AW1 iso 6400", "NIKON CORPORATION", "NIKON 1 V2", 6400, {0.00122104852594239, 0.000537554212900666, 0.00102193794645602}, {1.1444510627609e-05, 1.11674608962633e-05, 1.25256504328047e-05}},

// nikon 1 J1 by Berthold Crysmann
{"NIKON 1 J1 iso 100",  "NIKON CORPORATION", "NIKON 1 J1", 100,  {4.90999882328769e-06, 3.72770728920672e-06, 9.38533902345952e-06}, {4.04683461257664e-08, 2.72316251014706e-08, 4.93487694880424e-08}},
{"NIKON 1 J1 iso 200",  "NIKON CORPORATION", "NIKON 1 J1", 200,  {9.38381065173745e-06, 7.25311935367292e-06, 1.82527813232121e-05}, {6.16441485614302e-08, 3.77392483631955e-08, 7.33578135290988e-08}},
{"NIKON 1 J1 iso 400",  "NIKON CORPORATION", "NIKON 1 J1", 400,  {1.80886257541815e-05, 1.3156008409002e-05, 3.54787388698654e-05}, {7.44101132506294e-08, 5.45295419519314e-08, 7.95174124547573e-08}},
{"NIKON 1 J1 iso 800",  "NIKON CORPORATION", "NIKON 1 J1", 800,  {3.38725721836299e-05, 2.40676991644325e-05, 6.41796214848887e-05}, {9.06765159143054e-08, 6.36082175908108e-08, 1.05077775708997e-07}},
{"NIKON 1 J1 iso 1600", "NIKON CORPORATION", "NIKON 1 J1", 1600, {6.59162033030226e-05, 4.71030028443966e-05, 0.000119673724193934}, {1.4156543089364e-07, 7.27293259680273e-08, 2.08091321173889e-07}},
{"NIKON 1 J1 iso 3200", "NIKON CORPORATION", "NIKON 1 J1", 3200, {0.000130167064357819, 9.22134055705514e-05, 0.00023531414173755}, {1.59232497350005e-07, 1.09177437990126e-07, 2.79601047975249e-07}},

// copied from Nikon 1 J1
{"NIKON 1 V1 iso 100",  "NIKON CORPORATION", "NIKON 1 V1", 100,  {4.90999882328769e-06, 3.72770728920672e-06, 9.38533902345952e-06}, {4.04683461257664e-08, 2.72316251014706e-08, 4.93487694880424e-08}},
{"NIKON 1 V1 iso 200",  "NIKON CORPORATION", "NIKON 1 V1", 200,  {9.38381065173745e-06, 7.25311935367292e-06, 1.82527813232121e-05}, {6.16441485614302e-08, 3.77392483631955e-08, 7.33578135290988e-08}},
{"NIKON 1 V1 iso 400",  "NIKON CORPORATION", "NIKON 1 V1", 400,  {1.80886257541815e-05, 1.3156008409002e-05, 3.54787388698654e-05}, {7.44101132506294e-08, 5.45295419519314e-08, 7.95174124547573e-08}},
{"NIKON 1 V1 iso 800",  "NIKON CORPORATION", "NIKON 1 V1", 800,  {3.38725721836299e-05, 2.40676991644325e-05, 6.41796214848887e-05}, {9.06765159143054e-08, 6.36082175908108e-08, 1.05077775708997e-07}},
{"NIKON 1 V1 iso 1600", "NIKON CORPORATION", "NIKON 1 V1", 1600, {6.59162033030226e-05, 4.71030028443966e-05, 0.000119673724193934}, {1.4156543089364e-07, 7.27293259680273e-08, 2.08091321173889e-07}},
{"NIKON 1 V1 iso 3200", "NIKON CORPORATION", "NIKON 1 V1", 3200, {0.000130167064357819, 9.22134055705514e-05, 0.00023531414173755}, {1.59232497350005e-07, 1.09177437990126e-07, 2.79601047975249e-07}},

// copied from Nikon 1 J1
{"NIKON 1 J2 iso 100",  "NIKON CORPORATION", "NIKON 1 J2", 100,  {4.90999882328769e-06, 3.72770728920672e-06, 9.38533902345952e-06}, {4.04683461257664e-08, 2.72316251014706e-08, 4.93487694880424e-08}},
{"NIKON 1 J2 iso 200",  "NIKON CORPORATION", "NIKON 1 J2", 200,  {9.38381065173745e-06, 7.25311935367292e-06, 1.82527813232121e-05}, {6.16441485614302e-08, 3.77392483631955e-08, 7.33578135290988e-08}},
{"NIKON 1 J2 iso 400",  "NIKON CORPORATION", "NIKON 1 J2", 400,  {1.80886257541815e-05, 1.3156008409002e-05, 3.54787388698654e-05}, {7.44101132506294e-08, 5.45295419519314e-08, 7.95174124547573e-08}},
{"NIKON 1 J2 iso 800",  "NIKON CORPORATION", "NIKON 1 J2", 800,  {3.38725721836299e-05, 2.40676991644325e-05, 6.41796214848887e-05}, {9.06765159143054e-08, 6.36082175908108e-08, 1.05077775708997e-07}},
{"NIKON 1 J2 iso 1600", "NIKON CORPORATION", "NIKON 1 J2", 1600, {6.59162033030226e-05, 4.71030028443966e-05, 0.000119673724193934}, {1.4156543089364e-07, 7.27293259680273e-08, 2.08091321173889e-07}},
{"NIKON 1 J2 iso 3200", "NIKON CORPORATION", "NIKON 1 J2", 3200, {0.000130167064357819, 9.22134055705514e-05, 0.00023531414173755}, {1.59232497350005e-07, 1.09177437990126e-07, 2.79601047975249e-07}},

// copied from Nikon 1 J1
{"NIKON 1 S1 iso 100",  "NIKON CORPORATION", "NIKON 1 S1", 100,  {4.90999882328769e-06, 3.72770728920672e-06, 9.38533902345952e-06}, {4.04683461257664e-08, 2.72316251014706e-08, 4.93487694880424e-08}},
{"NIKON 1 S1 iso 200",  "NIKON CORPORATION", "NIKON 1 S1", 200,  {9.38381065173745e-06, 7.25311935367292e-06, 1.82527813232121e-05}, {6.16441485614302e-08, 3.77392483631955e-08, 7.33578135290988e-08}},
{"NIKON 1 S1 iso 400",  "NIKON CORPORATION", "NIKON 1 S1", 400,  {1.80886257541815e-05, 1.3156008409002e-05, 3.54787388698654e-05}, {7.44101132506294e-08, 5.45295419519314e-08, 7.95174124547573e-08}},
{"NIKON 1 S1 iso 800",  "NIKON CORPORATION", "NIKON 1 S1", 800,  {3.38725721836299e-05, 2.40676991644325e-05, 6.41796214848887e-05}, {9.06765159143054e-08, 6.36082175908108e-08, 1.05077775708997e-07}},
{"NIKON 1 S1 iso 1600", "NIKON CORPORATION", "NIKON 1 S1", 1600, {6.59162033030226e-05, 4.71030028443966e-05, 0.000119673724193934}, {1.4156543089364e-07, 7.27293259680273e-08, 2.08091321173889e-07}},
{"NIKON 1 S1 iso 3200", "NIKON CORPORATION", "NIKON 1 S1", 3200, {0.000130167064357819, 9.22134055705514e-05, 0.00023531414173755}, {1.59232497350005e-07, 1.09177437990126e-07, 2.79601047975249e-07}},

// nikon coolpix 330 by joe
{"Nikon COOLPIX P330 iso 80",   "NIKON", "COOLPIX P330", 80,   {2.09599146078921e-05, 9.35530619743515e-06, 1.41162113056803e-05}, {-1.57142094116978e-07, -2.76714470675303e-07, -2.11552283374124e-07}},
{"Nikon COOLPIX P330 iso 100",  "NIKON", "COOLPIX P330", 100,  {2.49866553069371e-05, 1.12973389832284e-05, 1.75292969518313e-05}, {-1.18418348525342e-07, -2.84738817357828e-07, -2.64471873719813e-07}},
{"Nikon COOLPIX P330 iso 200",  "NIKON", "COOLPIX P330", 200,  {4.68534636255408e-05, 2.05919353904991e-05, 3.2390478212075e-05}, {-1.21545068438233e-07, -3.1580137611657e-07, -3.03273804674723e-07}},
{"Nikon COOLPIX P330 iso 400",  "NIKON", "COOLPIX P330", 400,  {8.88813034595805e-05, 3.86403674407693e-05, 6.17064268785228e-05}, {-2.04688978850393e-08, -2.87670017801527e-07, -3.67784779233916e-07}},
{"Nikon COOLPIX P330 iso 800",  "NIKON", "COOLPIX P330", 800,  {0.000170307392541083, 7.31825956821824e-05, 0.000119041791622882}, {4.46014041152979e-07, 1.05782586049092e-07, -2.25982243468161e-07}},
{"Nikon COOLPIX P330 iso 1600", "NIKON", "COOLPIX P330", 1600, {0.000348627112573196, 0.00014538504494338, 0.00022871627420166}, {3.36930544968361e-07, 8.75258732192005e-07, 1.43756091978256e-06}},
{"Nikon COOLPIX P330 iso 3200", "NIKON", "COOLPIX P330", 3200, {0.000686781261783857, 0.00028580486286834, 0.00042517993381144}, {1.02690064064929e-06, 4.18114934644053e-06, 1.31063185934972e-05}},

// nikon p7100 by Toast
{"Nikon COOLPIX P7100 iso 100",  "NIKON", "COOLPIX P7100", 100,  {2.00430999068242e-05, 1.02682361871173e-05, 2.93340235630089e-05}, {5.83850941427869e-08, 3.83454840524156e-08, 6.87161722123222e-08}},
{"Nikon COOLPIX P7100 iso 200",  "NIKON", "COOLPIX P7100", 200,  {3.84164763033608e-05, 2.04444427421168e-05, 3.98652288139578e-05}, {1.88859101994145e-07, 1.06723891957197e-07, 3.1588800086944e-07}},
// {"Nikon COOLPIX P7100 iso 200",  "NIKON", "COOLPIX P7100", 200,  {4.00186220808067e-05, 2.044271742534e-05, 6.07658878504006e-05}, {1.39966548547432e-07, 1.00550341964961e-07, 1.57806484317047e-07}},
// {"Nikon COOLPIX P7100 iso 200",  "NIKON", "COOLPIX P7100", 200,  {4.05205569332422e-05, 1.99192729113391e-05, 6.19168633879578e-05}, {1.52601672892903e-07, 1.10967686402919e-07, 1.37978277465694e-07}},
{"Nikon COOLPIX P7100 iso 400",  "NIKON", "COOLPIX P7100", 400,  {7.43607875693378e-05, 3.83930658396094e-05, 0.000102799426630286}, {4.33593880570865e-07, 3.28529315214962e-07, 4.36031518790925e-07}},
// {"Nikon COOLPIX P7100 iso 400",  "NIKON", "COOLPIX P7100", 400,  {7.73793320838011e-05, 3.79126636055276e-05, 0.000119090825062442}, {4.04625590503288e-07, 3.35670656831003e-07, 2.49121136369506e-07}},
// {"Nikon COOLPIX P7100 iso 400",  "NIKON", "COOLPIX P7100", 400,  {7.63157626567337e-05, 3.87908109519356e-05, 0.000103776813738949}, {3.61009914150081e-07, 2.93488535128322e-07, 3.88945921503575e-07}},
{"Nikon COOLPIX P7100 iso 800",  "NIKON", "COOLPIX P7100", 800,  {0.000143775260153701, 7.63794863862664e-05, 0.00020024748432059}, {1.5969785909559e-06, 1.21860257495093e-06, 1.61895490887286e-06}},
// {"Nikon COOLPIX P7100 iso 800",  "NIKON", "COOLPIX P7100", 800,  {0.000152557332767169, 7.61487933060391e-05, 0.000234324558671302}, {1.54340822681127e-06, 1.24120756645147e-06, 1.30213301420061e-06}},
// {"Nikon COOLPIX P7100 iso 800",  "NIKON", "COOLPIX P7100", 800,  {0.000148927812975738, 7.67931118078119e-05, 0.000155448293744662}, {1.68789436571118e-06, 1.3291271063871e-06, 2.41835732829503e-06}},
{"Nikon COOLPIX P7100 iso 1600", "NIKON", "COOLPIX P7100", 1600, {0.000277243068047381, 0.000148812573652948, 0.000311036318400619}, {5.8751656486578e-06, 4.50479046034848e-06, 6.86419329685787e-06}},
// {"Nikon COOLPIX P7100 iso 1600", "NIKON", "COOLPIX P7100", 1600, {0.000281198144548966, 0.000148120244201172, 0.000293229619290656}, {5.56306965046814e-06, 4.37554371176729e-06, 6.99410081583617e-06}},
// {"Nikon COOLPIX P7100 iso 1600", "NIKON", "COOLPIX P7100", 1600, {0.000314853616724073, 0.000153739023843188, 0.000463860700797479}, {4.49576100464013e-06, 3.89999679953989e-06, 4.21211632928674e-06}},
{"Nikon COOLPIX P7100 iso 3200", "NIKON", "COOLPIX P7100", 3200, {0.000584584467508276, 0.000314474201251524, 0.00088393479717497}, {1.15701392584414e-05, 8.78301467927268e-06, 9.82334014813557e-06}},
// {"Nikon COOLPIX P7100 iso 3200", "NIKON", "COOLPIX P7100", 3200, {0.000636152017990261, 0.000344448017249525, 0.000973174385946965}, {7.31611685086501e-06, 6.01812474479556e-06, 6.45460003865861e-06}},
// {"Nikon COOLPIX P7100 iso 3200", "NIKON", "COOLPIX P7100", 3200, {0.000636929330873888, 0.000342847893731477, 0.000990822414871622}, {7.41324115434082e-06, 6.11910678897133e-06, 6.65388675969039e-06}},
// {"Nikon COOLPIX P7100 iso 3200", "NIKON", "COOLPIX P7100", 3200, {0.000627314223092868, 0.000313554919189146, 0.000842479270031033}, {1.09957699974096e-05, 8.95319551940802e-06, 1.09345914369341e-05}},
{"Nikon COOLPIX P7100 iso 6400", "NIKON", "COOLPIX P7100", 6400, {0.00056515311340767, 0.000278488480984131, 0.0008214829956804}, {1.67352114472894e-06, 9.36568974874431e-07, 1.54384333316831e-06}},

// nikon d40 by Björn Sonnenschein, fit for iso 3200 is a little off
{"NIKON D40 iso 200",       "NIKON CORPORATION",      "NIKON D40",              200,         {9.06184639231022e-06, 3.32876464255606e-06, 4.43038938103244e-06}, {3.79748456848388e-08, 2.17209633247412e-08, 3.07898485638319e-08}},
{"NIKON D40 iso 400",       "NIKON CORPORATION",      "NIKON D40",              400,         {1.81203565425609e-05, 6.49280004156721e-06, 8.20585226166676e-06}, {7.67029864881738e-08, 6.12847455529282e-08, 8.45136364779602e-08}},
{"NIKON D40 iso 800",       "NIKON CORPORATION",      "NIKON D40",              800,         {3.27157216757972e-05, 1.19998665219278e-05, 1.66319065031043e-05}, {1.86834859605992e-07, 1.50922985348677e-07, 2.0421295212633e-07}},
{"NIKON D40 iso 1600",       "NIKON CORPORATION",      "NIKON D40",              1600,         {6.63415718145666e-05, 2.65328871952215e-05, 3.78845551924937e-05}, {4.4515387559075e-07, 3.28793581633797e-07, 4.79732314042812e-07}},
{"NIKON D40 iso 3200",       "NIKON CORPORATION",      "NIKON D40",              3200,         {0.000190508430660164, 9.02437217524401e-05, 0.000160690229752482}, {2.63848648343068e-06, 1.96995498392379e-06, 2.74279885703267e-06}},
// additional d40 data, fits looks much the same as above
// {"NIKON D40 iso 200",       "NIKON CORPORATION",      "NIKON D40",              200,         {7.76749376547193e-06, 3.04541509480509e-06, 5.45160897607319e-06}, {2.45845600699799e-08, 1.9765160790749e-08, 2.54741069675667e-08}},
// {"NIKON D40 iso 400",       "NIKON CORPORATION",      "NIKON D40",              400,         {1.50549568901977e-05, 6.19684175395593e-06, 1.02323642436365e-05}, {4.63787114231445e-08, 3.58294267828274e-08, 4.90308077987114e-08}},
// {"NIKON D40 iso 800",       "NIKON CORPORATION",      "NIKON D40",              800,         {2.80988749464391e-05, 1.14455328784908e-05, 1.92258449090517e-05}, {1.12910689936351e-07, 7.74750961500393e-08, 1.17919330180329e-07}},
// {"NIKON D40 iso 1600",       "NIKON CORPORATION",      "NIKON D40",              1600,         {5.86165757609237e-05, 2.4006475392281e-05, 3.98229264708542e-05}, {2.89627885101928e-07, 1.71764471258341e-07, 2.97652058505259e-07}},
// {"NIKON D40 iso 3200",       "NIKON CORPORATION",      "NIKON D40",              3200,         {0.00016979493439124, 7.10164786655147e-05, 0.000111596149252809}, {2.18582065052034e-06, 1.38518409189016e-06, 2.3400945736856e-06}},

// nikon d50 by Togan Muftuoglu. noisy estimates for brighter regions (fits look decent though)
{"NIKON D50 iso 200",  "NIKON CORPORATION", "NIKON D50", 200,  {8.73771682849909e-06, 3.06025626665327e-06, 5.10914354828642e-06}, {3.12270679446127e-08, 9.21703932306338e-09, 3.89933169798386e-08}},
{"NIKON D50 iso 400",  "NIKON CORPORATION", "NIKON D50", 400,  {1.56498348286341e-05, 5.88904853403261e-06, 8.99848194511375e-06}, {9.60825221628058e-08, 4.85305368650233e-08, 1.10362608041602e-07}},
{"NIKON D50 iso 800",  "NIKON CORPORATION", "NIKON D50", 800,  {2.86846873139778e-05, 1.08528820075906e-05, 1.6686494279922e-05}, {2.97305103525993e-07, 2.04832853248993e-07, 3.34312737618865e-07}},
// {"NIKON D50 iso 1600", "NIKON CORPORATION", "NIKON D50", 1600, {5.90400416574637e-05, 2.18214148148958e-05, 3.4445688956951e-05}, {9.6539664890477e-07, 7.68684562881256e-07, 1.04601907339127e-06}},
{"NIKON D50 iso 1600", "NIKON CORPORATION", "NIKON D50", 1600, {5.7137294180781e-05, 2.11246538747481e-05, 3.40341099252318e-05}, {1.07373016803497e-06, 8.17301143604719e-07, 1.14850767051908e-06}},


// nikon d80, contributed by João Trindade
  {"Nikon D80 iso 100",  "NIKON CORPORATION", "NIKON D80", 100,  {5.47287145667241e-06, 2.83444490162939e-06, 5.30106436494686e-06}, {4.33084290382343e-08, -7.58149355625877e-09, 1.58131008158757e-08}},
  {"Nikon D80 iso 125",  "NIKON CORPORATION", "NIKON D80", 125,  {7.01088815420339e-06, 3.68865135220965e-06, 6.71571599832051e-06}, {4.86604409389741e-08, -4.68820074532959e-09, 2.50019937969203e-08}},
  {"Nikon D80 iso 160",  "NIKON CORPORATION", "NIKON D80", 160,  {8.74923336485094e-06, 4.75877954766136e-06, 8.39696071391484e-06}, {9.63376422767792e-08, 5.58960576640849e-09, 6.62861526466917e-08}},
  {"Nikon D80 iso 200",  "NIKON CORPORATION", "NIKON D80", 200,  {1.07756115674025e-05, 6.0445852380152e-06, 1.07184966781425e-05}, {1.27638522867984e-07, 2.49587757345139e-08, 4.41723668878374e-08}},
  {"Nikon D80 iso 250",  "NIKON CORPORATION", "NIKON D80", 250,  {1.27046348196911e-05, 7.10177555633891e-06, 1.25592723413574e-05}, {2.62101812214431e-07, 9.13147450349533e-08, 1.65069464329149e-07}},
  {"Nikon D80 iso 320",  "NIKON CORPORATION", "NIKON D80", 320,  {1.57919515142251e-05, 8.25960962135072e-06, 1.58962292734888e-05}, {3.970770887842e-07, 1.80410500650466e-07, 2.6884194652807e-07}},
  {"Nikon D80 iso 400",  "NIKON CORPORATION", "NIKON D80", 400,  {1.99206958841119e-05, 1.00445577591182e-05, 1.9798709791322e-05}, {5.882535504691e-07, 4.57023916145861e-07, 4.6497281044717e-07}},
  {"Nikon D80 iso 500",  "NIKON CORPORATION", "NIKON D80", 500,  {2.41837704630347e-05, 1.36596266980592e-05, 2.38689235614882e-05}, {9.62543867082136e-07, 4.98421819821685e-07, 8.19635235455349e-07}},
  {"Nikon D80 iso 640",  "NIKON CORPORATION", "NIKON D80", 640,  {3.08650411937983e-05, 1.62597852906041e-05, 2.97653694777343e-05}, {1.43513232881477e-06, 8.04146084172786e-07, 1.42978139215542e-06}},
  {"Nikon D80 iso 800",  "NIKON CORPORATION", "NIKON D80", 800,  {3.94712960910059e-05, 2.04934710846754e-05, 3.80074137069815e-05}, {1.55597687274684e-06, 1.12680688274482e-06, 1.49502062623437e-06}},
  {"Nikon D80 iso 1000", "NIKON CORPORATION", "NIKON D80", 1000, {4.86656300157051e-05, 2.63339481281419e-05, 4.5905684544555e-05}, {2.48617777554714e-06, 1.52046612831401e-06, 2.53313294333105e-06}},
  {"Nikon D80 iso 1250", "NIKON CORPORATION", "NIKON D80", 1250, {6.19893572508705e-05, 3.26625767673155e-05, 5.90513980616375e-05}, {3.60781419807241e-06, 2.62349821485562e-06, 3.69241382896107e-06}},
  {"Nikon D80 iso 1600", "NIKON CORPORATION", "NIKON D80", 1600, {8.04025337161777e-05, 4.10090610758394e-05, 7.35563798058682e-05}, {5.47587749588899e-06, 4.01111320590916e-06, 6.0708563001881e-06}},

// nikon d90, contributed by sismo
  {"Nikon D90 iso 100",  "NIKON CORPORATION", "NIKON D90", 100,  {6.86850451306969e-06, 3.54862491129016e-06, 4.97295417716348e-06}, {-1.25804723755625e-07, -1.75638387202353e-07, -9.08330740926672e-08}},
  {"Nikon D90 iso 126",  "NIKON CORPORATION", "NIKON D90", 126,  {6.23660143149099e-06, 2.95599567624574e-06, 4.46612980507398e-06}, {-8.70667504560387e-09, -1.81592425025098e-08, 4.90934343310815e-09}},
  {"Nikon D90 iso 159",  "NIKON CORPORATION", "NIKON D90", 159,  {6.04406470882879e-06, 2.88365223200481e-06, 4.48717743787467e-06}, {2.62334576654337e-09, 6.46405648306378e-09, -1.11562137738907e-08}},
  {"Nikon D90 iso 200",  "NIKON CORPORATION", "NIKON D90", 200,  {6.72698045219701e-06, 3.36472820970503e-06, 5.01907891547534e-06}, {3.19229680351793e-08, 2.92750858474177e-09, 1.50417413294233e-08}},
  {"Nikon D90 iso 250",  "NIKON CORPORATION", "NIKON D90", 250,  {8.99623074762613e-06, 3.45567573360262e-06, 6.8016666668232e-06}, {-4.98652015893091e-08, 1.5448530733265e-07, -7.91796071035873e-08}},
  {"Nikon D90 iso 320",  "NIKON CORPORATION", "NIKON D90", 320,  {1.05731537892965e-05, 3.79093658926564e-06, 8.09642020287693e-06}, {-2.89005263408187e-08, 2.56177804203168e-07, -7.39199844944572e-08}},
  {"Nikon D90 iso 400",  "NIKON CORPORATION", "NIKON D90", 400,  {1.19677865306815e-05, 6.33734955443999e-06, 8.95358450255129e-06}, {1.41853008067773e-07, -5.86205393501792e-08, 1.28427523368453e-07}},
  {"Nikon D90 iso 500",  "NIKON CORPORATION", "NIKON D90", 500,  {1.51618538825421e-05, 8.0437591618316e-06, 1.11794430657724e-05}, {1.77261655454406e-07, -6.25455896139984e-08, 1.91005558059814e-07}},
  {"Nikon D90 iso 640",  "NIKON CORPORATION", "NIKON D90", 640,  {2.04987455020811e-05, 8.98506221378615e-06, 1.49088425440335e-05}, {1.17697635440016e-08, 1.43597096047144e-07, 4.46440726004809e-08}},
  {"Nikon D90 iso 800",  "NIKON CORPORATION", "NIKON D90", 800,  {2.6111100964348e-05, 1.13331876417705e-05, 1.9003045701872e-05}, {1.21521557647828e-07, 3.64283922826794e-07, 1.76556991750179e-07}},
  {"Nikon D90 iso 1000", "NIKON CORPORATION", "NIKON D90", 1000, {3.29817783044431e-05, 1.52719595535266e-05, 2.3816018119032e-05}, {1.04544853319899e-07, 1.34609134492078e-07, 2.00994390771397e-07}},
  {"Nikon D90 iso 1250", "NIKON CORPORATION", "NIKON D90", 1250, {4.3965442534295e-05, 1.97558159326258e-05, 3.20200511871965e-05}, {4.07782168703457e-08, 1.72409645816128e-07, 1.60717633664245e-07}},
  {"Nikon D90 iso 1600", "NIKON CORPORATION", "NIKON D90", 1600, {5.64611549178139e-05, 2.38410350834287e-05, 4.05349555382123e-05}, {-1.91586931378523e-07, 5.72801979984449e-07, 2.27498102201229e-07}},
  {"Nikon D90 iso 2000", "NIKON CORPORATION", "NIKON D90", 2000, {6.71800881359098e-05, 3.2102358594961e-05, 5.01915639738066e-05}, {5.35800000737985e-07, 2.48673591677887e-07, 6.45912152243162e-07}},
  {"Nikon D90 iso 2500", "NIKON CORPORATION", "NIKON D90", 2500, {8.41272549964895e-05, 3.63200455768812e-05, 5.9421359904324e-05}, {5.96471932211773e-07, 1.39278888523735e-06, 1.35170478550807e-06}},
  {"Nikon D90 iso 3200", "NIKON CORPORATION", "NIKON D90", 3200, {9.48099102265449e-05, 4.59877036362871e-05, 7.22956153862606e-05}, {1.97666387072524e-06, 1.23035030463043e-06, 2.13693479581873e-06}},
  {"Nikon D90 iso 4032", "NIKON CORPORATION", "NIKON D90", 4032, {0.000124097416564837, 5.54124139423422e-05, 9.11944698437289e-05}, {2.4443599690721e-06, 2.64013513553918e-06, 3.03607819469662e-06}},
  {"Nikon D90 iso 5080", "NIKON CORPORATION", "NIKON D90", 5080, {0.000142469848208086, 6.0920961430653e-05, 0.000103326719116896}, {4.17181926169198e-06, 4.91229459605856e-06, 5.4870190949478e-06}},
  {"Nikon D90 iso 6400", "NIKON CORPORATION", "NIKON D90", 6400, {0.000187041275653718, 7.65384488442822e-05, 0.000130795773598612}, {5.4329665813471e-06, 7.16171942228706e-06, 7.37306938180301e-06}},

// Nikon D200, contributed by Dennis Gnad / some reds (@ISO 125,160,250,320,500,800) slightly underexposed, but fits look good
  {"NIKON D200 iso 100",       "NIKON CORPORATION",      "NIKON D200",              100,         {9.64606097024692e-06, 4.41030170372344e-06, 7.18483869393439e-06}, {-1.12236103755523e-07, -1.61612826717619e-07, -1.55949521166153e-07}},
  {"NIKON D200 iso 125",       "NIKON CORPORATION",      "NIKON D200",              125,         {1.17055039554268e-05, 5.28640752478803e-06, 8.60483250062752e-06}, {-1.3000639031293e-07, -1.81739271326044e-07, -1.57620290720214e-07}},
  {"NIKON D200 iso 160",       "NIKON CORPORATION",      "NIKON D200",              160,         {1.36932259136332e-05, 6.10702014358409e-06, 9.83481817284682e-06}, {-1.12240761300918e-07, -1.72179161168115e-07, -1.06937039900856e-07}},
  {"NIKON D200 iso 200",       "NIKON CORPORATION",      "NIKON D200",              200,         {1.62820108652777e-05, 7.13667679819941e-06, 1.19574321932631e-05}, {-4.72602316121525e-08, -8.79795159601918e-08, -7.51021873889331e-08}},
  {"NIKON D200 iso 250",       "NIKON CORPORATION",      "NIKON D200",              250,         {1.94145539332657e-05, 8.75939482496633e-06, 1.37539837424372e-05}, {-6.30041946004009e-09, -1.41872714218907e-07, -6.97020051091796e-09}},
  {"NIKON D200 iso 320",       "NIKON CORPORATION",      "NIKON D200",              320,         {2.23034261948727e-05, 9.69364446141606e-06, 1.69252908288564e-05}, {1.11767378161326e-07, 1.63047341056719e-08, 9.8267597421482e-08}},
  {"NIKON D200 iso 400",       "NIKON CORPORATION",      "NIKON D200",              400,         {2.64701904748047e-05, 1.17495538751695e-05, 1.92683743616239e-05}, {3.0833835558205e-07, 1.48809528770272e-07, 2.23185204295689e-07}},
  {"NIKON D200 iso 500",       "NIKON CORPORATION",      "NIKON D200",              500,         {3.34460158171325e-05, 1.43349669370385e-05, 2.51275382868979e-05}, {4.11027052750156e-07, 2.72771934728253e-07, 3.70214732910864e-07}},
  {"NIKON D200 iso 640",       "NIKON CORPORATION",      "NIKON D200",              640,         {4.35249575308842e-05, 1.83929323634288e-05, 3.11150607775962e-05}, {6.02975450956811e-07, 4.53044511059966e-07, 5.12554481097188e-07}},
  {"NIKON D200 iso 800",       "NIKON CORPORATION",      "NIKON D200",              800,         {5.35511752057119e-05, 2.24725825486571e-05, 3.88618345391768e-05}, {9.60184537107269e-07, 8.20260353981162e-07, 1.22016750427822e-06}},
  {"NIKON D200 iso 1000",       "NIKON CORPORATION",      "NIKON D200",              1000,         {6.80065202218504e-05, 2.85779598670398e-05, 4.90142043331179e-05}, {1.73640169701887e-06, 1.36272760090195e-06, 1.98604785049497e-06}},
  {"NIKON D200 iso 1250",       "NIKON CORPORATION",      "NIKON D200",              1250,         {8.54077021535537e-05, 3.60443689846473e-05, 6.17714209027257e-05}, {2.91557707595194e-06, 2.33583240199166e-06, 2.89777311868911e-06}},
  {"NIKON D200 iso 1600",       "NIKON CORPORATION",      "NIKON D200",              1600,         {0.000106518966865773, 4.51492877119725e-05, 8.02129975003714e-05}, {4.31170665697393e-06, 3.5157626372026e-06, 4.18939816133056e-06}},
  {"NIKON D200 iso 2000",       "NIKON CORPORATION",      "NIKON D200",              2000,         {0.000134095572528399, 5.61264193712477e-05, 9.65933289438272e-05}, {7.17504033582549e-06, 6.06240957515235e-06, 7.75171476404277e-06}},
  {"NIKON D200 iso 2500",       "NIKON CORPORATION",      "NIKON D200",              2500,         {0.000172512198615463, 7.05914448847493e-05, 0.000124752653466413}, {1.12034320819103e-05, 9.98215947348613e-06, 1.16377454545884e-05}},
  {"NIKON D200 iso 3200",       "NIKON CORPORATION",      "NIKON D200",              3200,         {0.000216937164931719, 8.72372029474752e-05, 0.000152463774412467}, {1.6469796835056e-05, 1.51053654309683e-05, 1.76267246882735e-05}},

// Nikon D300, contributed by Pascal Obry
  {"Nikon D300 iso 200",  "NIKON", "NIKON D300", 200,  {2.76441410887216e-06, 3.93137259241921e-06, 3.60173317503285e-06}, {1.2551198377131e-07, -1.72259917511157e-07, 5.83559181619043e-08}},
  {"Nikon D300 iso 250",  "NIKON", "NIKON D300", 250,  {3.40212442956997e-06, 4.66259487247611e-06, 4.53955260336663e-06}, {1.66297017276e-07, -1.76580500732339e-07, 6.33085378996414e-08}},
  {"Nikon D300 iso 320",  "NIKON", "NIKON D300", 320,  {4.27978387320824e-06, 5.48913453264727e-06, 5.5655422076672e-06}, {2.01164812748216e-07, -1.37659281388884e-07, 9.20376401357687e-08}},
  {"Nikon D300 iso 400",  "NIKON", "NIKON D300", 400,  {5.07608157962842e-06, 6.33315638914097e-06, 6.87211986957974e-06}, {2.77892619785069e-07, -3.76970872907096e-08, 1.34574708101329e-07}},
  {"Nikon D300 iso 500",  "NIKON", "NIKON D300", 500,  {6.20369083293106e-06, 8.22110271468612e-06, 8.457946167755e-06}, {3.69162830652185e-07, -1.45727708693641e-07, 2.00787663358835e-07}},
  {"Nikon D300 iso 640",  "NIKON", "NIKON D300", 640,  {8.03991964992429e-06, 9.73919699498294e-06, 1.0519932188017e-05}, {4.52707768808315e-07, -5.17086883329023e-08, 2.69000549021842e-07}},
  {"Nikon D300 iso 800",  "NIKON", "NIKON D300", 800,  {1.09605154407783e-05, 1.14905671062208e-05, 1.33445139099252e-05}, {5.39472948800832e-07, 1.53957473787375e-07, 3.82848145034845e-07}},
  {"Nikon D300 iso 1000", "NIKON", "NIKON D300", 1000, {1.25002550888452e-05, 1.45920457849254e-05, 1.69700912789896e-05}, {7.68895142464593e-07, 1.51676994569483e-07, 4.12113981576668e-07}},
  {"Nikon D300 iso 1250", "NIKON", "NIKON D300", 1250, {1.4025306903885e-05, 1.77472609232017e-05, 2.01385750829861e-05}, {1.04544689558732e-06, 3.31203189725297e-07, 6.40955295478197e-07}},
  {"Nikon D300 iso 1600", "NIKON", "NIKON D300", 1600, {2.14195846532532e-05, 2.66287145035161e-05, 3.01995959776475e-05}, {1.66084735548565e-06, 6.21869879321773e-07, 1.08480241618365e-06}},
  {"Nikon D300 iso 2000", "NIKON", "NIKON D300", 2000, {3.08612950955274e-05, 3.28260395002601e-05, 3.89157915202392e-05}, {1.99280488657823e-06, 1.05241242045483e-06, 1.44376535172786e-06}},
  {"Nikon D300 iso 2500", "NIKON", "NIKON D300", 2500, {3.61238472366171e-05, 4.11554499287525e-05, 4.87647477121592e-05}, {2.81525258264222e-06, 1.40259790106482e-06, 1.90931740970812e-06}},
  {"Nikon D300 iso 3200", "NIKON", "NIKON D300", 3200, {4.82612470670375e-05, 5.04851700591095e-05, 6.36655749857791e-05}, {3.73428045194131e-06, 2.4508953626996e-06, 2.42966735951493e-06}},
  {"Nikon D300 iso 4032", "NIKON", "NIKON D300", 4032, {5.96743630930957e-05, 6.64143339138741e-05, 7.94455794022399e-05}, {5.21147237379628e-06, 2.64586079410712e-06, 3.55688677857871e-06}},
  {"Nikon D300 iso 5080", "NIKON", "NIKON D300", 5080, {7.26377685276826e-05, 8.2138257011462e-05, 0.000102704944680837}, {7.32174238500767e-06, 4.25876022676086e-06, 4.66108449204189e-06}},
  {"Nikon D300 iso 6400", "NIKON", "NIKON D300", 6400, {9.58630420717688e-05, 0.000100838638503078, 0.00013075132310209}, {9.59807626007555e-06, 6.85835273265595e-06, 6.62705295295392e-06}},

// nikon d300s contributed by christte
  {"NIKON D300S iso 100",  "NIKON CORPORATION", "NIKON D300S", 100,  {7.25253488374287e-06, 3.15500838764334e-06, 6.46492055433208e-06}, {5.59709047789483e-08, 2.2721038353183e-07, -1.97652268959473e-09}},
  {"NIKON D300S iso 126",  "NIKON CORPORATION", "NIKON D300S", 126,  {7.5626046854387e-06, 2.93649841394747e-06, 2.78452396909868e-06}, {-1.39438482536918e-08, 1.43319995662138e-08, 4.46724713010864e-08}},
  {"NIKON D300S iso 159",  "NIKON CORPORATION", "NIKON D300S", 159,  {8.18463393460055e-06, 3.17839446530294e-06, 2.80026339618263e-06}, {-8.86796461022548e-08, -6.28699811418974e-08, 3.75805019473137e-08}},
  {"NIKON D300S iso 200",  "NIKON CORPORATION", "NIKON D300S", 200,  {9.94755952111756e-06, 3.86101532417617e-06, 3.45427833006783e-06}, {-8.64461520510319e-08, -6.09019595459925e-08, 6.45658214561295e-08}},
  {"NIKON D300S iso 250",  "NIKON CORPORATION", "NIKON D300S", 250,  {1.26210242199034e-05, 4.81077736114503e-06, 4.23726783738364e-06}, {-1.57748578105728e-07, -1.02693902964221e-07, 6.99090009185867e-08}},
  {"NIKON D300S iso 320",  "NIKON CORPORATION", "NIKON D300S", 320,  {1.56396643587233e-05, 5.98500653429102e-06, 5.72836660649057e-06}, {-1.25640545795541e-07, -8.38564601895707e-08, 5.61287324828653e-08}},
  {"NIKON D300S iso 400",  "NIKON CORPORATION", "NIKON D300S", 400,  {1.97093278389697e-05, 7.53247987508177e-06, 6.85042829963387e-06}, {-1.88632154844987e-07, -1.49486842148962e-07, 1.24629057541441e-07}},
  {"NIKON D300S iso 500",  "NIKON CORPORATION", "NIKON D300S", 500,  {2.44001118833589e-05, 9.27912589533184e-06, 8.98872228613772e-06}, {-1.87292598462205e-07, -1.4105600128626e-07, 9.5661084094368e-08}},
  {"NIKON D300S iso 640",  "NIKON CORPORATION", "NIKON D300S", 640,  {3.04299560568822e-05, 1.16411962650876e-05, 1.05043117800886e-05}, {-2.46158423167969e-07, -1.81997057902151e-07, 3.12100160937671e-07}},
  {"NIKON D300S iso 800",  "NIKON CORPORATION", "NIKON D300S", 800,  {3.86311356485491e-05, 1.48707969173882e-05, 1.45999725565982e-05}, {-2.7113521020062e-07, -2.55126883138336e-07, 2.99169496611621e-07}},
  {"NIKON D300S iso 1000", "NIKON CORPORATION", "NIKON D300S", 1000, {4.80173213201687e-05, 1.84430440449785e-05, 1.8221690697258e-05}, {-2.11736315941544e-07, -2.37171974837961e-07, 3.62273603830787e-07}},
  {"NIKON D300S iso 1250", "NIKON CORPORATION", "NIKON D300S", 1250, {6.09050649518795e-05, 2.3554746370741e-05, 2.29878861448381e-05}, {-2.22842460924418e-07, -3.10772336678599e-07, 4.44432914121173e-07}},
  {"NIKON D300S iso 1600", "NIKON CORPORATION", "NIKON D300S", 1600, {7.63090006824688e-05, 2.9436568530722e-05, 2.9749676234489e-05}, {-2.63048468495522e-07, -4.2786324631427e-07, 3.43019987734839e-07}},
  {"NIKON D300S iso 2000", "NIKON CORPORATION", "NIKON D300S", 2000, {9.65238757454544e-05, 3.68492410599395e-05, 3.72529546595316e-05}, {-1.2431988753585e-07, -1.30280565054366e-07, 9.51843123196086e-07}},
  {"NIKON D300S iso 2500", "NIKON CORPORATION", "NIKON D300S", 2500, {0.000121376729394008, 4.62613162265646e-05, 5.05100044606591e-05}, {4.31508667363549e-07, 3.71666774934291e-07, 7.96874282185789e-07}},
  {"NIKON D300S iso 3200", "NIKON CORPORATION", "NIKON D300S", 3200, {0.000154332846901273, 5.84708028581151e-05, 6.46683818244214e-05}, {1.24699026396974e-06, 8.80653649284952e-07, 1.50384614654356e-06}},
  {"NIKON D300S iso 4000", "NIKON CORPORATION", "NIKON D300S", 4000, {0.000195736495534235, 7.50604090116156e-05, 8.55807241518871e-05}, {1.88344386640941e-06, 1.07369388913016e-06, 1.68752410772312e-06}},
  {"NIKON D300S iso 5000", "NIKON CORPORATION", "NIKON D300S", 5000, {0.000248934424924582, 9.50454276310419e-05, 0.000112842915400042}, {3.02501952498374e-06, 2.03157317994836e-06, 2.17673854734017e-06}},
  {"NIKON D300S iso 6400", "NIKON CORPORATION", "NIKON D300S", 6400, {0.000317752644088187, 0.000118874151319726, 0.000144736659953121}, {6.10056333173075e-06, 4.46382293269047e-06, 4.65107969383875e-06}},

// nikon d3200 by Aurelien Deharbe
{"NIKON D3200 iso 100",   "NIKON CORPORATION", "NIKON D3200", 100,   {4.14741804910369e-06, 3.06801429595815e-06, 7.16685842156854e-06}, {9.88600870785514e-09, 4.71724497457251e-09, 1.0371664616345e-08}},
{"NIKON D3200 iso 200",   "NIKON CORPORATION", "NIKON D3200", 200,   {8.06327754540964e-06, 5.68641787236454e-06, 1.32423173817937e-05}, {2.05144050932473e-08, 1.24633564116069e-08, 2.35790086974405e-08}},
{"NIKON D3200 iso 400",   "NIKON CORPORATION", "NIKON D3200", 400,   {1.59005040006756e-05, 1.10346481799759e-05, 2.51060426673701e-05}, {4.07775540757222e-08, 1.99318139860741e-08, 5.3552238014189e-08}},
{"NIKON D3200 iso 800",   "NIKON CORPORATION", "NIKON D3200", 800,   {3.21999412194699e-05, 2.26248753609092e-05, 4.99540798331043e-05}, {1.03981414507899e-07, 6.20310042960127e-08, 1.30722221782634e-07}},
{"NIKON D3200 iso 1600",  "NIKON CORPORATION", "NIKON D3200", 1600,  {6.18812668390223e-05, 4.46755102461591e-05, 9.84245644086234e-05}, {2.0324268489491e-07, 1.45325313012159e-07, 2.97479867002672e-07}},
{"NIKON D3200 iso 3200",  "NIKON CORPORATION", "NIKON D3200", 3200,  {0.000127486321360696, 9.12320197251667e-05, 0.000197303840997251}, {4.49112580494137e-07, 3.45334581567514e-07, 6.17931072355593e-07}},
{"NIKON D3200 iso 6400",  "NIKON CORPORATION", "NIKON D3200", 6400,  {0.000255456555879409, 0.000177903768916944, 0.000391868567368787}, {1.04220966968983e-06, 8.86080850920979e-07, 1.36426424236031e-06}},
{"NIKON D3200 iso 12800", "NIKON CORPORATION", "NIKON D3200", 12800, {0.000532224566543877, 0.000362268290537384, 0.00077676838626364}, {2.76089267207156e-06, 2.41447909171385e-06, 3.01575101117523e-06}},

// nikon d60, fails to stabilize dark areas (like 5x variance in this image)
{"NIKON D60 iso 100",  "NIKON CORPORATION", "NIKON D60", 100,  {6.73289940594222e-06, 2.84476308147378e-06, 4.41701553915816e-06}, {2.80287571081868e-08, 1.90558710840497e-08, 3.48123049230028e-08}},
{"NIKON D60 iso 200",  "NIKON CORPORATION", "NIKON D60", 200,  {1.31827079803715e-05, 5.93337953644391e-06, 8.06876940771874e-06}, {7.26859957382335e-08, 4.75192254743017e-08, 8.93281635830903e-08}},
{"NIKON D60 iso 400",  "NIKON CORPORATION", "NIKON D60", 400,  {2.37687706576215e-05, 1.02590836392907e-05, 1.52903986562517e-05}, {2.1743566837447e-07, 1.53254958549535e-07, 2.51725164114832e-07}},
{"NIKON D60 iso 800",  "NIKON CORPORATION", "NIKON D60", 800,  {4.38406747143593e-05, 1.96495334583617e-05, 3.08665257161435e-05}, {7.5870794950345e-07, 5.6086626064588e-07, 8.54575125268495e-07}},
{"NIKON D60 iso 1600", "NIKON CORPORATION", "NIKON D60", 1600, {7.98272265961742e-05, 4.01894476436638e-05, 6.59937753295275e-05}, {3.00932656588442e-06, 2.18358230990622e-06, 3.30540144452806e-06}},
{"NIKON D60 iso 3200", "NIKON CORPORATION", "NIKON D60", 3200, {0.000151156761987638, 8.06641021294837e-05, 0.000134994915743362}, {1.19568420000066e-05, 8.56534222930722e-06, 1.29499170414084e-05}},

// nikon d600, contributed by Guilherme Brondani Torri
  {"Nikon D600 iso 100",  "NIKON CORPORATION", "NIKON D600", 100,  {2.79336625696654e-06, 1.31639742448043e-06, 1.83261245114861e-06}, {-1.49344157238591e-08, -2.84209221970076e-08, -1.57191856352517e-08}},
  {"Nikon D600 iso 125",  "NIKON CORPORATION", "NIKON D600", 125,  {3.45038716525766e-06, 1.60566242985672e-06, 2.29894944750702e-06}, {-1.5719631284966e-08, -2.64525251698482e-08, -2.25097857182153e-08}},
  {"Nikon D600 iso 160",  "NIKON CORPORATION", "NIKON D600", 160,  {4.25037511018984e-06, 1.97452460670131e-06, 2.76951331027489e-06}, {-2.37567763586969e-08, -3.90868257831298e-08, -1.61659037280836e-08}},
  {"Nikon D600 iso 200",  "NIKON CORPORATION", "NIKON D600", 200,  {5.14953851235845e-06, 2.38634431622847e-06, 3.39244394439858e-06}, {-2.33974943797398e-08, -4.251944783787e-08, -2.01428602722498e-08}},
  {"Nikon D600 iso 250",  "NIKON CORPORATION", "NIKON D600", 250,  {6.43983750710829e-06, 2.96995674711851e-06, 4.17990088421724e-06}, {-3.77263090333848e-08, -5.3531333251307e-08, -1.95037839686578e-08}},
  {"Nikon D600 iso 320",  "NIKON CORPORATION", "NIKON D600", 320,  {8.02275844771855e-06, 3.67968185223716e-06, 5.26155641846798e-06}, {-4.9393586843406e-08, -6.40654940233095e-08, -4.10501907101697e-08}},
  {"Nikon D600 iso 400",  "NIKON CORPORATION", "NIKON D600", 400,  {9.63661144167742e-06, 4.45661018382983e-06, 6.55413503704402e-06}, {-1.85760683313466e-08, -5.12127967455804e-08, -4.21268321091107e-08}},
  {"Nikon D600 iso 500",  "NIKON CORPORATION", "NIKON D600", 500,  {1.20569503793044e-05, 5.69631176823036e-06, 8.23947063039034e-06}, {-3.99058446865467e-08, -1.02047395334666e-07, -6.38929291212311e-08}},
  {"Nikon D600 iso 640",  "NIKON CORPORATION", "NIKON D600", 640,  {1.4962928045518e-05, 7.05377010338088e-06, 1.00653048117476e-05}, {-1.89863985317671e-08, -1.00022993015848e-07, -2.40632228047592e-08}},
  {"Nikon D600 iso 800",  "NIKON CORPORATION", "NIKON D600", 800,  {1.86902466788352e-05, 8.5270931500201e-06, 1.27357253938634e-05}, {-5.94700841934197e-09, -3.38914673246306e-08, -5.22391365457154e-08}},
  {"Nikon D600 iso 1000", "NIKON CORPORATION", "NIKON D600", 1000, {2.36400624206873e-05, 1.083082704221e-05, 1.5834346917419e-05}, {-2.92002443810454e-08, -8.61537852308941e-08, -3.62439942559703e-08}},
  {"Nikon D600 iso 1250", "NIKON CORPORATION", "NIKON D600", 1250, {2.97358638007557e-05, 1.31022991406577e-05, 2.00588474570871e-05}, {-4.16561840065312e-08, 4.09428700990611e-08, -5.16842881132528e-08}},
  {"Nikon D600 iso 1600", "NIKON CORPORATION", "NIKON D600", 1600, {3.75615567993373e-05, 1.69755478081177e-05, 2.50227280858447e-05}, {-1.16075043664017e-07, -9.19779675274509e-08, -2.90777433165336e-08}},
  {"Nikon D600 iso 2000", "NIKON CORPORATION", "NIKON D600", 2000, {4.63095919034324e-05, 2.12330297677989e-05, 3.14080348394477e-05}, {-6.13014914502891e-09, -9.45427608231511e-08, -2.02867673129836e-09}},
  {"Nikon D600 iso 2500", "NIKON CORPORATION", "NIKON D600", 2500, {5.85184570910876e-05, 2.63780185752517e-05, 3.90596232176281e-05}, {4.67590279213405e-08, -8.47431827285922e-09, 1.20782420758395e-07}},
  {"Nikon D600 iso 3200", "NIKON CORPORATION", "NIKON D600", 3200, {7.29097443878201e-05, 3.34761765893987e-05, 5.03091224686683e-05}, {5.76238642002023e-08, -7.96184910594476e-08, 1.69912172691656e-08}},
  {"Nikon D600 iso 4000", "NIKON CORPORATION", "NIKON D600", 4000, {8.98212683543976e-05, 4.11920956998072e-05, 6.24415058554266e-05}, {3.18323094577922e-07, 1.97388628813987e-07, 2.31548959983686e-07}},
  {"Nikon D600 iso 5000", "NIKON CORPORATION", "NIKON D600", 5000, {0.000118504116918991, 5.281907108616e-05, 7.90331414426069e-05}, {-1.10266602335644e-07, 2.38706643615142e-08, 2.49307957228523e-07}},
  {"Nikon D600 iso 6400", "NIKON CORPORATION", "NIKON D600", 6400, {0.000142198001010633, 6.53177941612927e-05, 9.62762271871283e-05}, {5.67416417473035e-07, 4.05567310073306e-07, 1.02231320984015e-06}},

// nikon d610 by christian touzé
{"NIKON D610 iso 50",       "NIKON CORPORATION",      "NIKON D610",              50,         {1.33389168540608e-06, 1.04333331492103e-06, 3.85947724461414e-06}, {1.5327428007772e-09, 1.16489693474116e-09, 2.0817351735666e-09}},
{"NIKON D610 iso 63",       "NIKON CORPORATION",      "NIKON D610",              63,         {1.33405547973297e-06, 1.03864673760115e-06, 3.78681652242855e-06}, {1.53288010935909e-09, 1.16461036744499e-09, 2.08262115636805e-09}},
{"NIKON D610 iso 79",       "NIKON CORPORATION",      "NIKON D610",              79,         {1.3326996733273e-06, 1.04354689096308e-06, 3.88922965228804e-06}, {1.53074144699955e-09, 1.16412304405292e-09, 2.08058296318643e-09}},
{"NIKON D610 iso 100",       "NIKON CORPORATION",      "NIKON D610",              100,         {1.33725950179034e-06, 1.04306175536825e-06, 3.84849760408937e-06}, {1.5320145544613e-09, 1.16294638880731e-09, 2.0821871705275e-09}},
{"NIKON D610 iso 125",       "NIKON CORPORATION",      "NIKON D610",              125,         {1.62513369241023e-06, 1.28783796287226e-06, 5.0807004265799e-06}, {2.02753280010291e-09, 1.59863178596195e-09, 2.91709881378151e-09}},
{"NIKON D610 iso 160",       "NIKON CORPORATION",      "NIKON D610",              160,         {2.01863947145265e-06, 1.60079262427713e-06, 6.59293512995217e-06}, {2.67502090198355e-09, 2.18828451923769e-09, 4.29096274993689e-09}},
{"NIKON D610 iso 200",       "NIKON CORPORATION",      "NIKON D610",              200,         {2.51546974134829e-06, 1.98041235887865e-06, 8.35501352469883e-06}, {3.79026836889725e-09, 3.0198659533454e-09, 6.04763593102512e-09}},
{"NIKON D610 iso 250",       "NIKON CORPORATION",      "NIKON D610",              250,         {3.17606440384869e-06, 2.47827959336577e-06, 9.40378145494001e-06}, {5.46051098057063e-09, 4.18414179073923e-09, 9.24255131847612e-09}},
{"NIKON D610 iso 320",       "NIKON CORPORATION",      "NIKON D610",              320,         {3.95463227100463e-06, 3.09889477882296e-06, 1.10515517716102e-05}, {7.52589014556431e-09, 5.79685143507049e-09, 1.37687688589644e-08}},
{"NIKON D610 iso 400",       "NIKON CORPORATION",      "NIKON D610",              400,         {4.95357758132818e-06, 3.8848139401247e-06, 1.40989263021441e-05}, {1.06099805118906e-08, 7.97417074187409e-09, 1.94146338992497e-08}},
{"NIKON D610 iso 500",       "NIKON CORPORATION",      "NIKON D610",              500,         {6.21432574550086e-06, 4.86050293559566e-06, 1.89551135883285e-05}, {1.49038605900991e-08, 1.11974089982916e-08, 2.76590328896263e-08}},
{"NIKON D610 iso 640",       "NIKON CORPORATION",      "NIKON D610",              640,         {7.77679335891089e-06, 6.09143712381729e-06, 2.45043192919288e-05}, {2.13377045765387e-08, 1.57850488678375e-08, 3.82829533183658e-08}},
{"NIKON D610 iso 800",       "NIKON CORPORATION",      "NIKON D610",              800,         {9.79061965280775e-06, 7.72060262697918e-06, 3.18614882373519e-05}, {2.85053120347545e-08, 1.96016880957395e-08, 5.32620203302336e-08}},
{"NIKON D610 iso 1000",       "NIKON CORPORATION",      "NIKON D610",              1000,         {1.23721986189696e-05, 9.82784725859885e-06, 4.07457724773996e-05}, {3.67246411197041e-08, 2.43166120852202e-08, 7.34285117648202e-08}},
{"NIKON D610 iso 1250",       "NIKON CORPORATION",      "NIKON D610",              1250,         {1.55206911871078e-05, 1.23045511575253e-05, 5.05375259404722e-05}, {4.93620704956647e-08, 3.38977194650751e-08, 1.01660730752351e-07}},
{"NIKON D610 iso 1600",       "NIKON CORPORATION",      "NIKON D610",              1600,         {1.94730104170903e-05, 1.54501501122892e-05, 6.21735204968732e-05}, {7.08564393102011e-08, 4.83088481571774e-08, 1.45661439604224e-07}},
{"NIKON D610 iso 2000",       "NIKON CORPORATION",      "NIKON D610",              2000,         {2.43847348772002e-05, 1.94693533377173e-05, 7.467638016864e-05}, {1.03368170602888e-07, 6.74525114188528e-08, 2.18113047566083e-07}},
{"NIKON D610 iso 2500",       "NIKON CORPORATION",      "NIKON D610",              2500,         {3.07479915288704e-05, 2.46484915704274e-05, 9.22941246351179e-05}, {1.52826199474935e-07, 9.44858700825828e-08, 3.12784296632007e-07}},
{"NIKON D610 iso 3200",       "NIKON CORPORATION",      "NIKON D610",              3200,         {3.83692966094946e-05, 3.09256330364608e-05, 0.000110063737578098}, {2.41470249484009e-07, 1.45598634413679e-07, 4.65486778941555e-07}},
{"NIKON D610 iso 4000",       "NIKON CORPORATION",      "NIKON D610",              4000,         {4.8477110389949e-05, 3.90872966590449e-05, 0.000131857741250155}, {3.60162996577071e-07, 2.11325968112254e-07, 7.52234913885437e-07}},
{"NIKON D610 iso 5000",       "NIKON CORPORATION",      "NIKON D610",              5000,         {6.11885709920365e-05, 4.92047017754462e-05, 0.000143999360811014}, {5.52727447441824e-07, 3.27743963964585e-07, 1.37690533301942e-06}},
{"NIKON D610 iso 6400",       "NIKON CORPORATION",      "NIKON D610",              6400,         {7.67067307026447e-05, 6.19715357670278e-05, 0.000134280109555084}, {8.665616805765e-07, 4.87070761369996e-07, 2.81699357757922e-06}},
{"NIKON D610 iso 8063",       "NIKON CORPORATION",      "NIKON D610",              8063,         {9.29866972437395e-05, 7.76821272367299e-05, 0.000126474173813004}, {1.08066559382489e-06, 6.10910025147476e-07, 4.798983030075e-06}},
{"NIKON D610 iso 101597",       "NIKON CORPORATION",      "NIKON D610",              101597,         {0.000116578118348919, 9.78997603582644e-05, 0.000141547145935222}, {1.71396185020676e-06, 9.38512097463954e-07, 7.9416389291467e-06}},
{"NIKON D610 iso 128000",       "NIKON CORPORATION",      "NIKON D610",              128000,         {0.000146127525597308, 0.000123971736866239, 0.000176435404810147}, {2.74043043983759e-06, 1.40848851813724e-06, 1.2086829208607e-05}},
{"NIKON D610 iso 256000",       "NIKON CORPORATION",      "NIKON D610",              256000,         {0.000309232414769709, 0.00024745403857761, 0.00019881036044426}, {8.45216731291557e-06, 5.18555821840487e-06, 4.64053814898822e-05}},

// nikon d700 by jb
  {"NIKON D700 iso 100",   "NIKON CORPORATION", "NIKON D700", 100,   {3.32506770452272e-06, 1.56929671746885e-06, 2.14313757059123e-06}, {-2.35156124424237e-08, -4.17724836680532e-08, -1.96492072177666e-08}},
  {"NIKON D700 iso 126",   "NIKON CORPORATION", "NIKON D700", 126,   {3.24460666010012e-06, 1.57542476875828e-06, 2.19148212723839e-06}, {-1.71775900458685e-08, -4.3175741528223e-08, -2.82217702206006e-08}},
  {"NIKON D700 iso 159",   "NIKON CORPORATION", "NIKON D700", 159,   {3.30101765738054e-06, 1.62004901689507e-06, 2.19524785788015e-06}, {-2.49190994755581e-08, -5.24832416128134e-08, -2.98340836757536e-08}},
  {"NIKON D700 iso 200",   "NIKON CORPORATION", "NIKON D700", 200,   {3.32953981128277e-06, 1.62092602107333e-06, 2.21125335648305e-06}, {-3.23706367858176e-08, -5.80348814583771e-08, -3.50533987238336e-08}},
  {"NIKON D700 iso 250",   "NIKON CORPORATION", "NIKON D700", 250,   {4.1346701497262e-06, 1.93029177242611e-06, 2.66503857735182e-06}, {-3.78221252586881e-08, -5.09215844891317e-08, -2.82956575834101e-08}},
  {"NIKON D700 iso 320",   "NIKON CORPORATION", "NIKON D700", 320,   {4.95531108282815e-06, 2.34012453538351e-06, 3.2125330207537e-06}, {-2.38722508156915e-08, -5.0755166850524e-08, -1.75144406193141e-08}},
  {"NIKON D700 iso 400",   "NIKON CORPORATION", "NIKON D700", 400,   {6.20442152966299e-06, 2.89463857533303e-06, 4.038365180942e-06}, {-4.34732393954794e-08, -6.19515587639236e-08, -3.34608382822773e-08}},
  {"NIKON D700 iso 500",   "NIKON CORPORATION", "NIKON D700", 500,   {7.62860840042155e-06, 3.50207377828827e-06, 4.97151699823815e-06}, {-3.67744795077658e-08, -5.50992830995503e-08, -3.25777727388071e-08}},
  {"NIKON D700 iso 640",   "NIKON CORPORATION", "NIKON D700", 640,   {9.20437825834857e-06, 4.27253587039924e-06, 6.18518503150267e-06}, {-3.93483672053365e-09, -3.72841074958055e-08, -2.53499797303117e-08}},
  {"NIKON D700 iso 800",   "NIKON CORPORATION", "NIKON D700", 800,   {1.18905134957785e-05, 5.49015823439183e-06, 7.71367209155366e-06}, {-5.74226636480158e-08, -8.96984929268894e-08, -4.45285593126819e-08}},
  {"NIKON D700 iso 1000",  "NIKON CORPORATION", "NIKON D700", 1000,  {1.45605465967974e-05, 6.59690646468279e-06, 9.46762832746613e-06}, {-4.44051544395908e-08, -4.68331128438844e-08, -2.43015255755695e-08}},
  {"NIKON D700 iso 1250",  "NIKON CORPORATION", "NIKON D700", 1250,  {1.78763901555226e-05, 8.303363577961e-06, 1.18122612059374e-05}, {1.02697039155813e-08, -6.81156303545365e-08, -9.66435446518481e-09}},
  {"NIKON D700 iso 1600",  "NIKON CORPORATION", "NIKON D700", 1600,  {2.31953076409403e-05, 1.06641765105635e-05, 1.50888732111918e-05}, {-6.22167333930692e-08, -1.04510301320831e-07, -2.52495941891977e-08}},
  {"NIKON D700 iso 2000",  "NIKON CORPORATION", "NIKON D700", 2000,  {2.77738971200008e-05, 1.29565743111474e-05, 1.84288646555725e-05}, {1.19903664370923e-07, -1.3328981719625e-08, 6.42019221993382e-08}},
  {"NIKON D700 iso 2500",  "NIKON CORPORATION", "NIKON D700", 2500,  {3.54122561973795e-05, 1.60351691603136e-05, 2.35491728813667e-05}, {1.21782618282388e-07, 8.35596180323573e-08, 7.6155031313438e-08}},
  {"NIKON D700 iso 3200",  "NIKON CORPORATION", "NIKON D700", 3200,  {4.40434497371827e-05, 2.02069081960572e-05, 2.89719501107012e-05}, {2.38253154477188e-07, 1.45744530705963e-07, 2.78179372986552e-07}},
  {"NIKON D700 iso 4000",  "NIKON CORPORATION", "NIKON D700", 4000,  {5.54456306759794e-05, 2.55774792869846e-05, 3.68596447163748e-05}, {3.5334475433873e-07, 1.71160957404903e-07, 3.19555748058209e-07}},
  {"NIKON D700 iso 5000",  "NIKON CORPORATION", "NIKON D700", 5000,  {6.94444651292649e-05, 3.22133734582461e-05, 4.63563167532013e-05}, {6.06134715579859e-07, 3.66350903008516e-07, 5.59280725606529e-07}},
  {"NIKON D700 iso 6400",  "NIKON CORPORATION", "NIKON D700", 6400,  {8.95784915384139e-05, 4.10424461544512e-05, 5.952542511193e-05}, {8.00016773224296e-07, 5.22042264360731e-07, 7.51783225328407e-07}},
  {"NIKON D700 iso 8000",  "NIKON CORPORATION", "NIKON D700", 8000,  {0.000112487501375074, 5.22499932743965e-05, 7.42108258777708e-05}, {1.13959103294069e-06, 6.48842416382553e-07, 1.04335378883468e-06}},
  {"NIKON D700 iso 10000", "NIKON CORPORATION", "NIKON D700", 10000, {0.000142413112346117, 6.45727986850154e-05, 9.1834890384527e-05}, {1.73648650923461e-06, 1.26446438906432e-06, 1.91751781446375e-06}},
  {"NIKON D700 iso 12800", "NIKON CORPORATION", "NIKON D700", 12800, {0.000177046794063112, 8.12184712838708e-05, 0.000118086287582622}, {3.1081812842324e-06, 2.29562795533186e-06, 2.8028886738805e-06}},

// Nikon D750, contributed by Erik Gustavsson
  {"NIKON D750 iso 50",    "NIKON CORPORATION", "NIKON D750", 50,    {1.90047163973098e-06, 9.40575234963588e-07, 1.31500280944697e-06}, {9.67290581168983e-10, -6.90555417909306e-11, 2.33749022811006e-09}},
  {"NIKON D750 iso 100",   "NIKON CORPORATION", "NIKON D750", 100,   {2.42360911595177e-06, 1.20056172825695e-06, 1.69233174386527e-06}, {4.46157504287823e-09, 5.76086698503822e-10, 3.51529572836847e-09}},
  {"NIKON D750 iso 125",   "NIKON CORPORATION", "NIKON D750", 125,   {2.92240892710775e-06, 1.4486797422406e-06, 2.053779682192e-06}, {5.28540028637499e-09, 6.23260576243414e-10, 4.85281951564326e-09}},
  {"NIKON D750 iso 160",   "NIKON CORPORATION", "NIKON D750", 160,   {3.65600795270168e-06, 1.78714135660411e-06, 2.5675079566574e-06}, {7.00060691221245e-09, 1.31321180359226e-09, 5.87599308774233e-09}},
  {"NIKON D750 iso 200",   "NIKON CORPORATION", "NIKON D750", 200,   {4.55792415461259e-06, 2.21606685384487e-06, 3.18148989656681e-06}, {8.97051974762457e-09, 1.8580020703188e-09, 7.34536035212033e-09}},
  {"NIKON D750 iso 250",   "NIKON CORPORATION", "NIKON D750", 250,   {5.69782918516125e-06, 2.74829029064758e-06, 3.98200792073778e-06}, {1.11744012235515e-08, 2.40466789830564e-09, 1.03100168289073e-08}},
  {"NIKON D750 iso 320",   "NIKON CORPORATION", "NIKON D750", 320,   {7.0907541803724e-06, 3.43319300214284e-06, 4.99231422185626e-06}, {1.50482198095191e-08, 2.88932584887584e-09, 1.11241610378853e-08}},
  {"NIKON D750 iso 400",   "NIKON CORPORATION", "NIKON D750", 400,   {8.85950925046396e-06, 4.26987585465475e-06, 6.22791044863531e-06}, {1.94931213830761e-08, 4.47699103037782e-09, 1.45906980800919e-08}},
  {"NIKON D750 iso 500",   "NIKON CORPORATION", "NIKON D750", 500,   {1.10272883481939e-05, 5.32141221487288e-06, 7.78989423752655e-06}, {2.46258245090512e-08, 6.13296682921759e-09, 2.00062634021575e-08}},
  {"NIKON D750 iso 640",   "NIKON CORPORATION", "NIKON D750", 640,   {1.38612222414604e-05, 6.66196474737319e-06, 9.81886715028293e-06}, {3.2338193238146e-08, 8.88337941399994e-09, 2.52830977885239e-08}},
  {"NIKON D750 iso 800",   "NIKON CORPORATION", "NIKON D750", 800,   {1.72921997254288e-05, 8.31258398051797e-06, 1.22471875496814e-05}, {4.31780105222373e-08, 1.23343894501685e-08, 3.01678446375303e-08}},
  {"NIKON D750 iso 1000",  "NIKON CORPORATION", "NIKON D750", 1000,  {2.17492354304431e-05, 1.04151415775375e-05, 1.53752489735734e-05}, {5.58915572070955e-08, 1.74279085906604e-08, 3.9240910680417e-08}},
  {"NIKON D750 iso 1250",  "NIKON CORPORATION", "NIKON D750", 1250,  {2.74658978947965e-05, 1.30619273558425e-05, 1.94468832116735e-05}, {7.24463283354872e-08, 2.53213181443154e-08, 4.9090003054237e-08}},
  {"NIKON D750 iso 1600",  "NIKON CORPORATION", "NIKON D750", 1600,  {3.40464814472533e-05, 1.62729959244864e-05, 2.42208519850058e-05}, {9.85264481455116e-08, 4.017819297963e-08, 6.68744822592319e-08}},
  {"NIKON D750 iso 2000",  "NIKON CORPORATION", "NIKON D750", 2000,  {4.28739178489306e-05, 2.05282443957106e-05, 3.05647179328696e-05}, {1.27146276769508e-07, 5.66056197415319e-08, 9.23645883403929e-08}},
  {"NIKON D750 iso 2500",  "NIKON CORPORATION", "NIKON D750", 2500,  {5.43171343506759e-05, 2.58981162088024e-05, 3.84918434341061e-05}, {1.63986445598184e-07, 8.15511956569334e-08, 1.31681759555961e-07}},
  {"NIKON D750 iso 3200",  "NIKON CORPORATION", "NIKON D750", 3200,  {6.80388842960932e-05, 3.23987099799284e-05, 4.82467446730746e-05}, {1.88619966844748e-07, 9.24790304582035e-08, 1.40135444032243e-07}},
  {"NIKON D750 iso 4000",  "NIKON CORPORATION", "NIKON D750", 4000,  {8.60900894854977e-05, 4.07105470022081e-05, 6.12990024341613e-05}, {2.30338581961023e-07, 1.31344425165354e-07, 1.90150311588618e-07}},
  {"NIKON D750 iso 5000",  "NIKON CORPORATION", "NIKON D750", 5000,  {0.000107887573418563, 5.09510832833098e-05, 7.64263384749014e-05}, {2.82305533594827e-07, 1.8577104754557e-07, 2.59253012856509e-07}},
  {"NIKON D750 iso 6400",  "NIKON CORPORATION", "NIKON D750", 6400,  {0.000135362227157091, 6.37997230999324e-05, 9.55846277378415e-05}, {3.45483577876654e-07, 2.53621297393937e-07, 3.41349638899364e-07}},
  {"NIKON D750 iso 8000",  "NIKON CORPORATION", "NIKON D750", 8000,  {0.000170018689526746, 8.02858357223386e-05, 0.000120194215202164}, {4.25065139387745e-07, 3.57673342168673e-07, 4.62765846332884e-07}},
  {"NIKON D750 iso 10000", "NIKON CORPORATION", "NIKON D750", 10000, {0.000215194098570213, 0.000101029934726629, 0.000152414628038704}, {5.43968443854035e-07, 5.0145913842819e-07, 6.06354948559691e-07}},
  {"NIKON D750 iso 12800", "NIKON CORPORATION", "NIKON D750", 12800, {0.000270412013428891, 0.000126980165620768, 0.000191797458415874}, {6.99423489887055e-07, 7.15377442556186e-07, 8.41257027583835e-07}},
  {"NIKON D750 iso 25600", "NIKON CORPORATION", "NIKON D750", 25600, {0.000545400955258936, 0.000256385233383518, 0.000388930816711704}, {1.52887689473581e-06, 1.97340112364026e-06, 2.17005469182635e-06}},
  {"NIKON D750 iso 51200", "NIKON CORPORATION", "NIKON D750", 51200, {0.000799577067830329, 0.000382040200129061, 0.000572207545635177}, {3.8870201614581e-06, 5.20219155105498e-06, 5.42725379556055e-06}},

// Nikon D800, contributed by Pascal Obry
  {"Nikon D800 iso 100",   "NIKON", "NIKON D800", 100,   {2.89359953712432e-06, 2.89359953712432e-06, 2.89359953712432e-06}, {-1.87189278790825e-07, -1.87189278790825e-07, -1.87189278790825e-07}},
  {"Nikon D800 iso 125",   "NIKON", "NIKON D800", 125,   {3.31368501080653e-06, 3.31368501080653e-06, 3.31368501080653e-06}, {-2.03020280693191e-07, -2.03020280693191e-07, -2.03020280693191e-07}},
  {"Nikon D800 iso 160",   "NIKON", "NIKON D800", 160,   {3.79066515417781e-06, 3.79066515417781e-06, 3.79066515417781e-06}, {-2.10888452221342e-07, -2.10888452221342e-07, -2.10888452221342e-07}},
  {"Nikon D800 iso 200",   "NIKON", "NIKON D800", 200,   {4.3235841999295e-06, 4.3235841999295e-06, 4.3235841999295e-06}, {-2.04976096610362e-07, -2.04976096610362e-07, -2.04976096610362e-07}},
  {"Nikon D800 iso 250",   "NIKON", "NIKON D800", 250,   {5.12677247107878e-06, 5.12677247107878e-06, 5.12677247107878e-06}, {-2.17826073240097e-07, -2.17826073240097e-07, -2.17826073240097e-07}},
  {"Nikon D800 iso 320",   "NIKON", "NIKON D800", 320,   {6.02071337078309e-06, 6.02071337078309e-06, 6.02071337078309e-06}, {-2.16754478106988e-07, -2.16754478106988e-07, -2.16754478106988e-07}},
  {"Nikon D800 iso 400",   "NIKON", "NIKON D800", 400,   {7.27658977046e-06, 7.27658977046e-06, 7.27658977046e-06}, {-2.27850739880476e-07, -2.27850739880476e-07, -2.27850739880476e-07}},
  {"Nikon D800 iso 500",   "NIKON", "NIKON D800", 500,   {8.73725308221732e-06, 8.73725308221732e-06, 8.73725308221732e-06}, {-2.23732912828083e-07, -2.23732912828083e-07, -2.23732912828083e-07}},
  {"Nikon D800 iso 640",   "NIKON", "NIKON D800", 640,   {1.06264476615276e-05, 1.06264476615276e-05, 1.06264476615276e-05}, {-2.24459193074953e-07, -2.24459193074953e-07, -2.24459193074953e-07}},
  {"Nikon D800 iso 800",   "NIKON", "NIKON D800", 800,   {1.28623401124144e-05, 1.28623401124144e-05, 1.28623401124144e-05}, {-1.88342190808096e-07, -1.88342190808096e-07, -1.88342190808096e-07}},
  {"Nikon D800 iso 1000",  "NIKON", "NIKON D800", 1000,  {1.61068636122712e-05, 1.61068636122712e-05, 1.61068636122712e-05}, {-2.5518014212713e-07, -2.5518014212713e-07, -2.5518014212713e-07}},
  {"Nikon D800 iso 1250",  "NIKON", "NIKON D800", 1250,  {1.95898660076489e-05, 1.95898660076489e-05, 1.95898660076489e-05}, {-4.25663565785904e-08, -4.25663565785904e-08, -4.25663565785904e-08}},
  {"Nikon D800 iso 1600",  "NIKON", "NIKON D800", 1600,  {2.41873948992182e-05, 2.41873948992182e-05, 2.41873948992182e-05}, {2.81937455362971e-08, 2.81937455362971e-08, 2.81937455362971e-08}},
  {"Nikon D800 iso 2000",  "NIKON", "NIKON D800", 2000,  {3.07947953837184e-05, 3.07947953837184e-05, 3.07947953837184e-05}, {-5.10854946293296e-08, -5.10854946293296e-08, -5.10854946293296e-08}},
  {"Nikon D800 iso 2500",  "NIKON", "NIKON D800", 2500,  {3.76878433958526e-05, 3.76878433958526e-05, 3.76878433958526e-05}, {1.86204759688554e-07, 1.86204759688554e-07, 1.86204759688554e-07}},
  {"Nikon D800 iso 3200",  "NIKON", "NIKON D800", 3200,  {4.72287381353268e-05, 4.72287381353268e-05, 4.72287381353268e-05}, {2.86646566040084e-07, 2.86646566040084e-07, 2.86646566040084e-07}},
  {"Nikon D800 iso 4000",  "NIKON", "NIKON D800", 4000,  {5.92687085008541e-05, 5.92687085008541e-05, 5.92687085008541e-05}, {3.72949030703814e-07, 3.72949030703814e-07, 3.72949030703814e-07}},
  {"Nikon D800 iso 5000",  "NIKON", "NIKON D800", 5000,  {7.50122091568944e-05, 7.50122091568944e-05, 7.50122091568944e-05}, {4.01750654303191e-07, 4.01750654303191e-07, 4.01750654303191e-07}},
  {"Nikon D800 iso 6400",  "NIKON", "NIKON D800", 6400,  {9.36523059856223e-05, 9.36523059856223e-05, 9.36523059856223e-05}, {8.57128037097117e-07, 8.57128037097117e-07, 8.57128037097117e-07}},
  {"Nikon D800 iso 8063",  "NIKON", "NIKON D800", 8063,  {0.000115936707308515, 0.000115936707308515, 0.000115936707308515}, {4.02557677955228e-07, 4.02557677955228e-07, 4.02557677955228e-07}},
  {"Nikon D800 iso 10159", "NIKON", "NIKON D800", 10159, {0.000145129335230523, 0.000145129335230523, 0.000145129335230523}, {8.27792795709805e-07, 8.27792795709805e-07, 8.27792795709805e-07}},
  {"Nikon D800 iso 12800", "NIKON", "NIKON D800", 12800, {0.00018493145619636, 0.00018493145619636, 0.00018493145619636}, {7.71909224546891e-07, 7.71909224546891e-07, 7.71909224546891e-07}},
  {"Nikon D800 iso 25600", "NIKON", "NIKON D800", 25600, {0.000359528021714155, 0.000359528021714155, 0.000359528021714155}, {6.42176785593581e-06, 6.42176785593581e-06, 6.42176785593581e-06}},

// nikon d800e by wolfgang goetz, blue underexposed
  {"Nikon D800E iso 50",    "NIKON CORPORATION", "NIKON D800E", 50,    {2.47950042812616e-06, 1.89007913739829e-06, 0.0}, {-2.54764522868205e-08, -4.65866010174205e-08, 0.0}},
  {"Nikon D800E iso 63",    "NIKON CORPORATION", "NIKON D800E", 63,    {2.52617605471794e-06, 1.9155957625239e-06,  0.0}, {-3.14632529718683e-08, -5.27982510676594e-08, 0.0}},
  {"Nikon D800E iso 79",    "NIKON CORPORATION", "NIKON D800E", 79,    {2.48019787217052e-06, 1.74724483086114e-06, 0.0}, {-2.46215332902138e-08, -1.92040692739085e-08, 0.0}},
  {"Nikon D800E iso 100",   "NIKON CORPORATION", "NIKON D800E", 100,   {2.33950662988413e-06, 1.80481136484837e-06, 0.0}, {-1.11781130599113e-08, -3.93379525783742e-08, 0.0}},
  {"Nikon D800E iso 125",   "NIKON CORPORATION", "NIKON D800E", 125,   {2.92050181503123e-06, 2.16648878154414e-06, 0.0}, {-1.37481195172961e-08, -4.31324669745174e-08, 0.0}},
  {"Nikon D800E iso 160",   "NIKON CORPORATION", "NIKON D800E", 160,   {3.49916608427732e-06, 2.59199142512512e-06, 0.0}, {-1.49947984295679e-09, -2.91579683903105e-08, 0.0}},
  {"Nikon D800E iso 200",   "NIKON CORPORATION", "NIKON D800E", 200,   {4.64241122822783e-06, 3.24838185405598e-06, 0.0}, {-3.07994649461231e-08, -5.71373308339205e-08, 0.0}},
  {"Nikon D800E iso 250",   "NIKON CORPORATION", "NIKON D800E", 250,   {5.56597025227697e-06, 3.95459126956011e-06, 0.0}, {-1.03609603479457e-09, -4.32374912265535e-08, 0.0}},
  {"Nikon D800E iso 320",   "NIKON CORPORATION", "NIKON D800E", 320,   {6.8091394880186e-06, 4.86023511621434e-06,  0.0}, {1.19546695555644e-09, -3.81127020874309e-08,  0.0}},
  {"Nikon D800E iso 400",   "NIKON CORPORATION", "NIKON D800E", 400,   {8.60302405059914e-06, 6.18340742933378e-06, 0.0}, {-7.86680552970148e-10, -5.56623410283349e-08, 0.0}},
  {"Nikon D800E iso 500",   "NIKON CORPORATION", "NIKON D800E", 500,   {1.04832015265563e-05, 7.66280906891528e-06, 0.0}, {1.31014781356844e-08, -8.83055860723616e-08,  0.0}},
  {"Nikon D800E iso 640",   "NIKON CORPORATION", "NIKON D800E", 640,   {1.35586012679998e-05, 9.60732247821001e-06, 0.0}, {-8.68901435402503e-09, -8.90176717819707e-08, 0.0}},
  {"Nikon D800E iso 800",   "NIKON CORPORATION", "NIKON D800E", 800,   {1.67457496748671e-05, 1.19192176897617e-05, 0.0}, {5.59281632519398e-09, -1.29487739805406e-07,  0.0}},
  {"Nikon D800E iso 1000",  "NIKON CORPORATION", "NIKON D800E", 1000,  {2.10260702117938e-05, 1.48621856710804e-05, 0.0}, {5.71825525523424e-08, -6.25159281831868e-08,  0.0}},
  {"Nikon D800E iso 1250",  "NIKON CORPORATION", "NIKON D800E", 1250,  {2.60757021362497e-05, 1.82470179088582e-05, 0.0}, {9.36630412442482e-08, -4.60744622987197e-08,  0.0}},
  {"Nikon D800E iso 1600",  "NIKON CORPORATION", "NIKON D800E", 1600,  {3.41323546872789e-05, 2.33428377680625e-05, 0.0}, {2.36404014179177e-08, -1.51001611664244e-08,  0.0}},
  {"Nikon D800E iso 2000",  "NIKON CORPORATION", "NIKON D800E", 2000,  {4.19229991249149e-05, 2.95851825924165e-05, 0.0}, {1.13894360680683e-07, 3.19409754814293e-08,   0.0}},
  {"Nikon D800E iso 2500",  "NIKON CORPORATION", "NIKON D800E", 2500,  {5.43060301441471e-05, 3.76980387278306e-05, 0.0}, {5.09355266546247e-08, -9.80382380204792e-08,  0.0}},
  {"Nikon D800E iso 3200",  "NIKON CORPORATION", "NIKON D800E", 3200,  {6.77957487614127e-05, 4.67789559899518e-05, 0.0}, {8.21669384446783e-08, 6.17698986247756e-08,   0.0}},
  {"Nikon D800E iso 4000",  "NIKON CORPORATION", "NIKON D800E", 4000,  {8.85772591321022e-05, 5.87482496651853e-05, 0.0}, {6.27533106152045e-09, 4.69100490772446e-07,   0.0}},
  {"Nikon D800E iso 5000",  "NIKON CORPORATION", "NIKON D800E", 5000,  {0.000107254225983481, 7.57978599603096e-05, 0.0}, {2.5882113212772e-07, 8.91570054116542e-09,    0.0}},
  {"Nikon D800E iso 6400",  "NIKON CORPORATION", "NIKON D800E", 6400,  {0.000138909402379952, 9.39800732084756e-05, 0.0}, {1.89262809921205e-07, 4.68341268476712e-07,   0.0}},
  {"Nikon D800E iso 8063",  "NIKON CORPORATION", "NIKON D800E", 8063,  {0.000165678934174764, 0.000117178318465476, 0.0}, {-1.33327895685009e-07, 6.66495883517521e-07,  0.0}},
  {"Nikon D800E iso 10159", "NIKON CORPORATION", "NIKON D800E", 10159, {0.000203692601185298, 0.000151585658459017, 0.0}, {5.05618471814225e-07, -1.59171325515882e-07,  0.0}},
  {"Nikon D800E iso 12800", "NIKON CORPORATION", "NIKON D800E", 12800, {0.000250548092879238, 0.00019114882117511,  0.0}, {1.88793608574928e-06, 4.84091766485229e-07,   0.0}},
  {"Nikon D800E iso 25600", "NIKON CORPORATION", "NIKON D800E", 25600, {0.000505949016676869, 0.000363178791825,    0.0}, {6.92718931739042e-06, 7.28461519946033e-06,   0.0}},

// nikon d810 by colin adams
{"NIKON D810 iso 31",    "NIKON CORPORATION", "NIKON D810", 31,    {1.91419112283932e-06, 1.04305102457956e-06, 1.65266693193826e-06}, {-1.83694278670334e-09, -1.38796909373288e-09, 1.30122836268126e-09}},
{"NIKON D810 iso 40",    "NIKON CORPORATION", "NIKON D810", 40,    {2.00119592079649e-06, 1.08875089563077e-06, 1.68798107890454e-06}, {-2.84104863013969e-09, -2.2210764223408e-09, 9.26192862673863e-10}},
{"NIKON D810 iso 50",    "NIKON CORPORATION", "NIKON D810", 50,    {1.68002070197917e-06, 9.16695995265967e-07, 1.59231515922765e-06}, {-8.46071562937279e-10, -3.13396601750323e-10, 1.94458906748291e-09}},
{"NIKON D810 iso 64",    "NIKON CORPORATION", "NIKON D810", 64,    {1.74504674668459e-06, 1.04403515956444e-06, 1.83225793307412e-06}, {-1.08423336721341e-09, -1.83402261708667e-09, -2.49366927570069e-10}},
{"NIKON D810 iso 80",    "NIKON CORPORATION", "NIKON D810", 80,    {2.2062049510463e-06, 1.24610135141228e-06, 2.11022925984925e-06}, {-2.52701924142232e-09, -2.33498150666012e-09, 4.25072284741949e-10}},
{"NIKON D810 iso 100",   "NIKON CORPORATION", "NIKON D810", 100,   {2.79803700224995e-06, 1.49561965616568e-06, 2.46585245014457e-06}, {-1.73100090131379e-09, -1.29041820132242e-09, 2.07430226376118e-09}},
{"NIKON D810 iso 125",   "NIKON CORPORATION", "NIKON D810", 125,   {3.39400143012551e-06, 1.83935408557221e-06, 3.08340828386201e-06}, {-5.68750740707152e-10, -6.9204954715178e-10, 3.0529258533718e-09}},
{"NIKON D810 iso 160",   "NIKON CORPORATION", "NIKON D810", 160,   {4.32749440859355e-06, 2.22068739624284e-06, 3.60392052778779e-06}, {1.69511631550695e-10, 3.35415743062643e-10, 4.84059981606481e-09}},
{"NIKON D810 iso 200",   "NIKON CORPORATION", "NIKON D810", 200,   {5.18674499968611e-06, 2.74261092520763e-06, 4.56494811357296e-06}, {6.54479643155963e-11, 2.34255977385017e-10, 6.58128100756457e-09}},
{"NIKON D810 iso 250",   "NIKON CORPORATION", "NIKON D810", 250,   {6.29945019215722e-06, 3.35253906910115e-06, 5.67841025622135e-06}, {2.47006158198073e-10, 6.7474252894849e-10, 8.80764227100166e-09}},
{"NIKON D810 iso 250",   "NIKON CORPORATION", "NIKON D810", 250,   {6.62322269263338e-06, 3.27659093567707e-06, 5.25827012701519e-06}, {2.4044091399411e-09, 2.73924763777806e-09, 9.83660104631602e-09}},
{"NIKON D810 iso 320",   "NIKON CORPORATION", "NIKON D810", 320,   {8.13938983272752e-06, 4.07049384455514e-06, 6.65925152380243e-06}, {3.22242878232516e-09, 4.00332956603889e-09, 1.2800745747971e-08}},
{"NIKON D810 iso 400",   "NIKON CORPORATION", "NIKON D810", 400,   {1.00832646451302e-05, 5.07950867958741e-06, 8.43484504490868e-06}, {4.54265246228849e-09, 5.49125370856363e-09, 1.67160750400244e-08}},
{"NIKON D810 iso 500",   "NIKON CORPORATION", "NIKON D810", 500,   {1.24254111614644e-05, 6.32805255743212e-06, 1.06218469011926e-05}, {5.57633478759444e-09, 8.00787104783911e-09, 2.33789764048166e-08}},
{"NIKON D810 iso 640",   "NIKON CORPORATION", "NIKON D810", 640,   {1.51705380261007e-05, 7.89319075429263e-06, 1.34618905095217e-05}, {8.70629370037511e-09, 1.09382912763603e-08, 3.32952273429319e-08}},
{"NIKON D810 iso 800",   "NIKON CORPORATION", "NIKON D810", 800,   {1.86900486212558e-05, 9.85022784248319e-06, 1.71096957681159e-05}, {1.44503561743673e-08, 1.58725252115992e-08, 4.41087575884465e-08}},
{"NIKON D810 iso 1000",  "NIKON CORPORATION", "NIKON D810", 1000,  {2.27000208891841e-05, 1.22697916532243e-05, 2.19454386613152e-05}, {3.21531378028945e-08, 2.68919417225337e-08, 5.80847311369579e-08}},
{"NIKON D810 iso 1250",  "NIKON CORPORATION", "NIKON D810", 1250,  {2.77783055995711e-05, 1.53679660664965e-05, 2.84087463282166e-05}, {5.72122256924508e-08, 4.34858185097745e-08, 7.65260181120137e-08}},
{"NIKON D810 iso 1600",  "NIKON CORPORATION", "NIKON D810", 1600,  {5.1686919804762e-05, 2.53574848519842e-05, 4.2363736452446e-05}, {7.11790261909932e-08, 7.23763682075376e-08, 1.47102973238214e-07}},
{"NIKON D810 iso 2000",  "NIKON CORPORATION", "NIKON D810", 2000,  {6.36403311678281e-05, 3.20489770230972e-05, 5.48695665267616e-05}, {1.03096091663283e-07, 1.08681781903776e-07, 2.27203260193637e-07}},
{"NIKON D810 iso 2500",  "NIKON CORPORATION", "NIKON D810", 2500,  {7.88069413115363e-05, 4.01547489279009e-05, 6.84523633858917e-05}, {1.6586633571678e-07, 1.58355550594556e-07, 3.27201043503106e-07}},
{"NIKON D810 iso 3200",  "NIKON CORPORATION", "NIKON D810", 3200,  {9.75656607530386e-05, 5.06169282128343e-05, 8.76107374541035e-05}, {1.67346378555029e-07, 1.56663002420698e-07, 3.34179217113923e-07}},
{"NIKON D810 iso 4000",  "NIKON CORPORATION", "NIKON D810", 4000,  {0.000118215487313651, 6.32760287010773e-05, 0.000113757555408082}, {2.80501986886937e-07, 2.26868354873694e-07, 4.36095669456792e-07}},
{"NIKON D810 iso 5000",  "NIKON CORPORATION", "NIKON D810", 5000,  {0.000144360937330271, 7.96673708811409e-05, 0.000147059950132417}, {4.61140666249439e-07, 3.39833930514578e-07, 6.17475659021703e-07}},
{"NIKON D810 iso 6400",  "NIKON CORPORATION", "NIKON D810", 6400,  {0.00019294730361134, 0.000100879340600729, 0.000176307690321636}, {5.4689728057824e-07, 4.73148054147386e-07, 8.79230010545117e-07}},
{"NIKON D810 iso 8000",  "NIKON CORPORATION", "NIKON D810", 8000,  {0.000241125102221986, 0.000126923196932402, 0.000222798184929392}, {7.49641658958946e-07, 6.52989503927242e-07, 1.221039031362e-06}},
{"NIKON D810 iso 10000", "NIKON CORPORATION", "NIKON D810", 10000, {0.000298075723974364, 0.000160783107478364, 0.000290237609836687}, {1.18418425777302e-06, 9.45025813931316e-07, 1.64350497553327e-06}},
{"NIKON D810 iso 12800", "NIKON CORPORATION", "NIKON D810", 12800, {0.0004146831266304, 0.000206120839340983, 0.000344668558679042}, {1.57228920547563e-06, 1.49643020622513e-06, 2.17284310991586e-06}},
{"NIKON D810 iso 16127", "NIKON CORPORATION", "NIKON D810", 16127, {0.000368791382655599, 0.000191864934480633, 0.000316970664844804}, {1.8721820258199e-06, 1.83151123599882e-06, 2.55947743401031e-06}},
{"NIKON D810 iso 20319", "NIKON CORPORATION", "NIKON D810", 20319, {0.000460818381647844, 0.000243973923124936, 0.000406678959031947}, {2.80464268747246e-06, 2.77223567754002e-06, 3.71530388230312e-06}},
{"NIKON D810 iso 25600", "NIKON CORPORATION", "NIKON D810", 25600, {0.000582848968735719, 0.000310601938139762, 0.000531760512990336}, {4.20526888423026e-06, 4.28880232558605e-06, 5.47868899749292e-06}},
{"NIKON D810 iso 51200", "NIKON CORPORATION", "NIKON D810", 51200, {0.00116472825910333, 0.000666076115924157, 0.00119632913627083}, {1.48301212598733e-05, 1.46641717989999e-05, 1.66360192641117e-05}},

// nikon d3100, contributed by sismo
  {"Nikon D3100 iso 100",   "NIKON CORPORATION", "NIKON D3100", 100,   {8.24450984493645e-06, 4.57339084914489e-06, 6.72909677335241e-06}, {6.34439375134399e-08, -1.68257151487887e-07, -1.42662968173028e-07}},
  {"Nikon D3100 iso 200",   "NIKON CORPORATION", "NIKON D3100", 200,   {1.69522646284645e-05, 7.73797088383233e-06, 9.96806279498521e-06}, {-2.10460064704677e-07, -2.50467421709781e-07, 1.62253095955509e-07}},
  {"Nikon D3100 iso 400",   "NIKON CORPORATION", "NIKON D3100", 400,   {3.28269056158574e-05, 1.37196541079687e-05, 1.89047793092216e-05}, {-6.04635558253237e-07, -3.3799658474822e-07, 3.30897934283473e-07}},
  {"Nikon D3100 iso 800",   "NIKON CORPORATION", "NIKON D3100", 800,   {6.10624323084075e-05, 2.50722554821587e-05, 3.33456397759853e-05}, {-1.12657599360067e-06, -4.17062549997286e-07, 1.09145387719525e-06}},
  {"Nikon D3100 iso 1600",  "NIKON CORPORATION", "NIKON D3100", 1600,  {0.000113174489750871, 4.59766163604518e-05, 6.47271024884694e-05}, {-1.53609455546408e-06, -2.15313893571546e-07, 1.9593932189429e-06}},
  {"Nikon D3100 iso 3200",  "NIKON CORPORATION", "NIKON D3100", 3200,  {0.000190933415993536, 7.78001027383676e-05, 0.000121563433860682}, {8.49796399236868e-07, 1.7132575979933e-06, 4.37107144211369e-06}},
  {"Nikon D3100 iso 6400",  "NIKON CORPORATION", "NIKON D3100", 6400,  {0.000374188842405974, 0.000146349922208483, 0.000236688249144134}, {4.66356455064968e-06, 6.87114739363747e-06, 1.14245267603242e-05}},
  {"Nikon D3100 iso 12800", "NIKON CORPORATION", "NIKON D3100", 12800, {0.000725411287438688, 0.000303681149852151, 0.0003572397035864}, {9.37043975482592e-06, 1.19625585591539e-05, 2.82039620769088e-05}},

// nikon d5000, contributed by Raphaël Monnard
  {"Nikon D5000 iso 100",  "NIKON", "NIKON D5000", 100,  {3.55434034592321e-06, 3.55434034592321e-06, 3.55434034592321e-06}, {-1.03227712861061e-07, -1.03227712861061e-07, -1.03227712861061e-07}},
  {"Nikon D5000 iso 125",  "NIKON", "NIKON D5000", 125,  {3.72492240851358e-06, 3.72492240851358e-06, 3.72492240851358e-06}, {-1.35521958369885e-07, -1.35521958369885e-07, -1.35521958369885e-07}},
  {"Nikon D5000 iso 160",  "NIKON", "NIKON D5000", 160,  {3.29225795293138e-06, 3.29225795293138e-06, 3.29225795293138e-06}, {8.48279202960177e-09, 8.48279202960177e-09, 8.48279202960177e-09}},
  {"Nikon D5000 iso 200",  "NIKON", "NIKON D5000", 200,  {3.59345364056741e-06, 3.59345364056741e-06, 3.59345364056741e-06}, {6.12980277963006e-08, 6.12980277963006e-08, 6.12980277963006e-08}},
  {"Nikon D5000 iso 250",  "NIKON", "NIKON D5000", 250,  {4.15439500131005e-06, 4.15439500131005e-06, 4.15439500131005e-06}, {1.21916079668766e-07, 1.21916079668766e-07, 1.21916079668766e-07}},
  {"Nikon D5000 iso 320",  "NIKON", "NIKON D5000", 320,  {5.21102878857756e-06, 5.21102878857756e-06, 5.21102878857756e-06}, {8.84574252141181e-08, 8.84574252141181e-08, 8.84574252141181e-08}},
  {"Nikon D5000 iso 400",  "NIKON", "NIKON D5000", 400,  {6.68065899855839e-06, 6.68065899855839e-06, 6.68065899855839e-06}, {1.50637387918195e-07, 1.50637387918195e-07, 1.50637387918195e-07}},
  {"Nikon D5000 iso 500",  "NIKON", "NIKON D5000", 500,  {8.43738532308592e-06, 8.43738532308592e-06, 8.43738532308592e-06}, {1.9887353081052e-07, 1.9887353081052e-07, 1.9887353081052e-07}},
  {"Nikon D5000 iso 640",  "NIKON", "NIKON D5000", 640,  {1.01204931421007e-05, 1.01204931421007e-05, 1.01204931421007e-05}, {3.25653266866488e-07, 3.25653266866488e-07, 3.25653266866488e-07}},
  {"Nikon D5000 iso 800",  "NIKON", "NIKON D5000", 800,  {1.3503191058199e-05, 1.3503191058199e-05, 1.3503191058199e-05}, {4.18411101301127e-07, 4.18411101301127e-07, 4.18411101301127e-07}},
  {"Nikon D5000 iso 1000", "NIKON", "NIKON D5000", 1000, {1.69052928194055e-05, 1.69052928194055e-05, 1.69052928194055e-05}, {5.44260862263629e-07, 5.44260862263629e-07, 5.44260862263629e-07}},
  {"Nikon D5000 iso 1250", "NIKON", "NIKON D5000", 1250, {2.22782069380325e-05, 2.22782069380325e-05, 2.22782069380325e-05}, {6.95430630653238e-07, 6.95430630653238e-07, 6.95430630653238e-07}},
  {"Nikon D5000 iso 1600", "NIKON", "NIKON D5000", 1600, {2.87907901583626e-05, 2.87907901583626e-05, 2.87907901583626e-05}, {9.36169085563434e-07, 9.36169085563434e-07, 9.36169085563434e-07}},
  {"Nikon D5000 iso 2000", "NIKON", "NIKON D5000", 2000, {3.39490808473876e-05, 3.39490808473876e-05, 3.39490808473876e-05}, {1.56088276996701e-06, 1.56088276996701e-06, 1.56088276996701e-06}},
  {"Nikon D5000 iso 2500", "NIKON", "NIKON D5000", 2500, {4.13170410396535e-05, 4.13170410396535e-05, 4.13170410396535e-05}, {2.45270765023671e-06, 2.45270765023671e-06, 2.45270765023671e-06}},
  {"Nikon D5000 iso 3200", "NIKON", "NIKON D5000", 3200, {4.13170410396535e-05, 4.13170410396535e-05, 4.13170410396535e-05}, {2.45270765023671e-06, 2.45270765023671e-06, 2.45270765023671e-06}},
  {"Nikon D5000 iso 4000", "NIKON", "NIKON D5000", 4000, {5.4708082446611e-05, 5.4708082446611e-05, 5.4708082446611e-05}, {6.31708282891435e-06, 6.31708282891435e-06, 6.31708282891435e-06}},
  {"Nikon D5000 iso 5000", "NIKON", "NIKON D5000", 5000, {6.49088846760222e-05, 6.49088846760222e-05, 6.49088846760222e-05}, {9.10172565593788e-06, 9.10172565593788e-06, 9.10172565593788e-06}},
  {"Nikon D5000 iso 6400", "NIKON", "NIKON D5000", 6400, {8.49174688072452e-05, 8.49174688072452e-05, 8.49174688072452e-05}, {1.20040650662599e-05, 1.20040650662599e-05, 1.20040650662599e-05}},

// additional data for the D5000 by jcsogo (red channel underexposed)
// {"Nikon D5000 iso 100",  "NIKON CORPORATION", "NIKON D5000", 100,  {6.1512535329363e-06, 3.09223054118538e-06, 3.23336716285593e-06}, {-3.08401292798532e-07, -7.08364594006153e-08, 6.0112348358447e-08}},
// {"Nikon D5000 iso 126",  "NIKON CORPORATION", "NIKON D5000", 126,  {6.98690431132611e-06, 2.85106020156105e-06, 3.20069024852301e-06}, {-2.82567207318614e-07, 3.84498022863215e-08, 8.44501663456521e-08}},
// {"Nikon D5000 iso 159",  "NIKON CORPORATION", "NIKON D5000", 159,  {5.23269700198639e-06, 3.0324970731496e-06, 2.95051264910342e-06}, {-1.33495182894816e-07, -9.89885722162507e-08, 1.27271420505176e-07}},
// {"Nikon D5000 iso 200",  "NIKON CORPORATION", "NIKON D5000", 200,  {6.9281380992745e-06, 3.71520353945057e-06, 3.58947672242302e-06}, {-7.48470070020605e-08, -1.55137304929891e-07, 7.59162233673396e-08}},
// {"Nikon D5000 iso 250",  "NIKON CORPORATION", "NIKON D5000", 250,  {9.92065793904596e-06, 4.00461593792745e-06, 4.85359325694816e-06}, {-2.55303214806824e-07, 1.05204404787667e-08, 1.05842851325454e-07}},
// {"Nikon D5000 iso 320",  "NIKON CORPORATION", "NIKON D5000", 320,  {1.08729517701411e-05, 4.93550595184729e-06, 5.10888262284422e-06}, {-2.83908437139554e-07, -5.07257964124655e-08, 1.92792132368003e-07}},
// {"Nikon D5000 iso 400",  "NIKON CORPORATION", "NIKON D5000", 400,  {1.15597335468978e-05, 7.19638940089929e-06, 6.42841695283211e-06}, {-2.14027493354953e-07, -2.81675075423156e-07, 4.08891973712063e-07}},
// {"Nikon D5000 iso 500",  "NIKON CORPORATION", "NIKON D5000", 500,  {5.53952057476577e-06, 6.09503618524699e-06, 8.54597919937944e-06}, {7.42535163560547e-07, 4.03487546897121e-07, 6.83312094194462e-07}},
// {"Nikon D5000 iso 640",  "NIKON CORPORATION", "NIKON D5000", 640,  {8.27974860810463e-06, 8.67436792244045e-06, 1.02421524399765e-05}, {9.53713880367811e-07, 1.24437143388825e-07, 2.84065234376274e-07}},
// {"Nikon D5000 iso 800",  "NIKON CORPORATION", "NIKON D5000", 800,  {3.24303605190424e-05, 1.36658131907775e-05, 1.7282505359094e-05}, {-5.14214788952064e-07, -6.87462347429879e-08, 9.94341519417255e-09}},
// {"Nikon D5000 iso 1000", "NIKON CORPORATION", "NIKON D5000", 1000, {2.2669789958316e-05, 1.51367040668737e-05, 1.75086025984389e-05}, {5.28535870496816e-07, 1.80536852563688e-07, 5.93036767608416e-07}},
// {"Nikon D5000 iso 1250", "NIKON CORPORATION", "NIKON D5000", 1250, {2.06458352959243e-05, 1.81383927464604e-05, 2.25257286527819e-05}, {1.82049080433878e-06, 6.92231703207224e-07, 8.50713042102117e-07}},
// {"Nikon D5000 iso 1600", "NIKON CORPORATION", "NIKON D5000", 1600, {3.18304261904274e-05, 2.33956581807984e-05, 2.77759440064371e-05}, {2.48258936949875e-06, 4.9024139797646e-07, 9.74934428853884e-07}},
// {"Nikon D5000 iso 2000", "NIKON CORPORATION", "NIKON D5000", 2000, {7.74692773159107e-05, 2.7763262625528e-05, 4.08374383117065e-05}, {-8.99013341611321e-07, 1.67280809157491e-06, 1.17633800485435e-06}},
// {"Nikon D5000 iso 2500", "NIKON CORPORATION", "NIKON D5000", 2500, {8.68514135903604e-05, 3.21860378577855e-05, 4.44441442582268e-05}, {-7.63189048131094e-07, 2.27758865772349e-06, 2.02119890233028e-06}},
// {"Nikon D5000 iso 3200", "NIKON CORPORATION", "NIKON D5000", 3200, {9.19910444251046e-05, 3.76821856221102e-05, 5.82859864578552e-05}, {-1.27057189502843e-07, 2.98584002724639e-06, 3.01012976301927e-06}},
// {"Nikon D5000 iso 4032", "NIKON CORPORATION", "NIKON D5000", 4032, {0.000104658240368716, 3.43044855187938e-05, 5.49504689948044e-05}, {1.99091307697753e-06, 6.02868851285234e-06, 4.25856681090112e-06}},
// {"Nikon D5000 iso 5080", "NIKON CORPORATION", "NIKON D5000", 5080, {0.000147362086154336, 5.47312954349898e-05, 8.20611812533196e-05}, {-1.23786457701566e-06, 6.13096173322137e-06, 6.93251242839155e-06}},
// {"Nikon D5000 iso 6400", "NIKON CORPORATION", "NIKON D5000", 6400, {0.000180811594351548, 5.90727913246872e-05, 9.58697782461738e-05}, {-7.38426237942118e-07, 9.96550317695348e-06, 1.06308806636217e-05}},

// nikon d5100, contributed by Gapato
  {"Nikon D5100 iso 100",       "NIKON CORPORATION",      "NIKON D5100",              100,         {6.00347254725191e-06, 2.4536568911529e-06, 3.49826722703996e-06}, {-6.32246108634169e-08, -7.96146817723714e-08, -2.41060716610728e-08}},
  {"Nikon D5100 iso 125",       "NIKON CORPORATION",      "NIKON D5100",              125,         {7.36643105042435e-06, 3.00178273889704e-06, 4.34106849796113e-06}, {-6.3162575340077e-08, -7.71967802061511e-08, -3.07282105834241e-08}},
  {"Nikon D5100 iso 160",       "NIKON CORPORATION",      "NIKON D5100",              160,         {9.05413704224353e-06, 3.6422221266489e-06, 5.30492615072131e-06}, {-7.30638786190598e-08, -7.51764066761565e-08, -2.18433390052059e-08}},
  {"Nikon D5100 iso 200",       "NIKON CORPORATION",      "NIKON D5100",              200,         {1.0930735997453e-05, 4.47481822117062e-06, 6.63326517949302e-06}, {-5.13885259693646e-08, -7.63438112898295e-08, -3.3211546529321e-08}},
  {"Nikon D5100 iso 250",       "NIKON CORPORATION",      "NIKON D5100",              250,         {1.35680202862822e-05, 5.53737343930049e-06, 7.96738017944367e-06}, {-5.91014463257327e-08, -7.7465599786616e-08, 1.3357299731531e-09}},
  {"Nikon D5100 iso 320",       "NIKON CORPORATION",      "NIKON D5100",              320,         {1.68381577281314e-05, 6.81952110085582e-06, 9.80732274759506e-06}, {-6.94506779010643e-08, -8.49839009231905e-08, 1.26377832992282e-08}},
  {"Nikon D5100 iso 400",       "NIKON CORPORATION",      "NIKON D5100",              400,         {2.09197878529468e-05, 8.3266975075968e-06, 1.23585319731244e-05}, {-9.68192893599822e-08, -6.86753100926439e-08, -1.54443641427669e-08}},
  {"Nikon D5100 iso 500",       "NIKON CORPORATION",      "NIKON D5100",              500,         {2.5178284360545e-05, 1.03254083051314e-05, 1.5325230307586e-05}, {-3.34552253659556e-08, -5.9665508774562e-08, -3.52657689741153e-09}},
  {"Nikon D5100 iso 640",       "NIKON CORPORATION",      "NIKON D5100",              640,         {3.17767032161939e-05, 1.27145428482526e-05, 1.89222807838262e-05}, {-4.67574685007116e-08, -3.5494878447234e-09, 6.0627751063434e-08}},
  {"Nikon D5100 iso 800",       "NIKON CORPORATION",      "NIKON D5100",              800,         {3.90396038303384e-05, 1.57991482997026e-05, 2.36093551885064e-05}, {3.75015987373701e-09, -6.615166354274e-09, 7.38135141312298e-08}},
  {"Nikon D5100 iso 1000",      "NIKON CORPORATION",      "NIKON D5100",              1000,        {5.02731533940301e-05, 1.99946162493057e-05, 3.08061539277321e-05}, {-1.47195579810825e-07, -3.4525371001442e-09, -1.031013031254e-07}},
  {"Nikon D5100 iso 1250",      "NIKON CORPORATION",      "NIKON D5100",              1250,        {6.12041180785305e-05, 2.44303505790641e-05, 3.63519009482432e-05}, {-2.6952342992734e-08, 8.8488029784803e-08, 2.13149491415536e-07}},
  {"Nikon D5100 iso 1600",      "NIKON CORPORATION",      "NIKON D5100",              1600,        {7.66673155266919e-05, 3.14310844048772e-05, 4.53727254979959e-05}, {-4.09901794415542e-08, -8.25328905814313e-08, 2.89853300997283e-07}},
  {"Nikon D5100 iso 2000",      "NIKON CORPORATION",      "NIKON D5100",              2000,        {9.52784814149401e-05, 3.78195492941784e-05, 5.80470901966462e-05}, {6.84069221891299e-08, 3.0903088398848e-07, 2.80925179314904e-07}},
  {"Nikon D5100 iso 2500",      "NIKON CORPORATION",      "NIKON D5100",              2500,        {0.000118670318606295, 4.79670696213899e-05, 7.31344807165462e-05}, {2.11418989175334e-07, 3.03053296671945e-07, 3.04196887088606e-07}},
  {"Nikon D5100 iso 3200",      "NIKON CORPORATION",      "NIKON D5100",              3200,        {0.000149990882548448, 6.00703775194213e-05, 9.24136436321706e-05}, {2.43950893637305e-07, 5.94824622937027e-07, 3.71862548910824e-07}},
  {"Nikon D5100 iso 4000",      "NIKON CORPORATION",      "NIKON D5100",              4000,        {0.000188483408511233, 7.74674898253284e-05, 0.000117044507636061}, {5.19121123736326e-07, 4.89151244141477e-07, 5.42145468467772e-07}},
  {"Nikon D5100 iso 5000",      "NIKON CORPORATION",      "NIKON D5100",              5000,        {0.00024005162086748, 9.6031313415756e-05, 0.000150439627219754}, {6.43896123406464e-08, 8.88225268551071e-07, 1.33036267904149e-07}},
  {"Nikon D5100 iso 6400",      "NIKON CORPORATION",      "NIKON D5100",              6400,        {0.000302048290391193, 0.000122740137804183, 0.000187013710437976}, {2.60677879647947e-07, 9.29702679155386e-07, 8.39685582557919e-07}},
  {"Nikon D5100 iso 8063",      "NIKON CORPORATION",      "NIKON D5100",              8063,        {0.000384018504555059, 0.000157345948295093, 0.000234629582383797}, {-7.59930834877574e-07, 3.28546337575991e-07, 1.22271990735987e-06}},
  {"Nikon D5100 iso 10159",     "NIKON CORPORATION",      "NIKON D5100",              10159,       {0.000462149445662638, 0.000188808449130524, 0.000293856873315601}, {2.10362414110719e-06, 3.44833982496902e-06, 2.39038347904731e-06}},
  {"Nikon D5100 iso 12800",     "NIKON CORPORATION",      "NIKON D5100",              12800,       {0.00058323068069753, 0.000245244428798187, 0.000358427886130159}, {2.50195743452962e-06, 2.85042196619763e-06, 5.39387926216124e-06}},
  {"Nikon D5100 iso 25600",     "NIKON CORPORATION",      "NIKON D5100",              25600,       {0.00115776578669188, 0.00046597663902027, 0.000675863951110872}, {1.35772498100564e-05, 2.10582055080733e-05, 2.93893336328782e-05}},
// same, slightly underexposed, numbers should be fine (left in here for variance reference)
//{"Nikon D5100 iso 100",       "NIKON CORPORATION",      "NIKON D5100",              100,         {5.86295492443593e-06, 2.53891819099038e-06, 3.41738609289793e-06}, {-6.93218664996464e-08, -9.75783718318464e-08, -4.80406093176159e-08}},
//{"Nikon D5100 iso 125",       "NIKON CORPORATION",      "NIKON D5100",              125,         {7.16483388109376e-06, 3.05402039036977e-06, 4.16499083381021e-06}, {-6.69916084366908e-08, -9.09600773097571e-08, -3.61918938834322e-08}},
//{"Nikon D5100 iso 160",       "NIKON CORPORATION",      "NIKON D5100",              160,         {8.77536538843916e-06, 3.72012248953686e-06, 5.19970500790524e-06}, {-7.08099241089059e-08, -9.4238717806462e-08, -4.58022718442645e-08}},
//{"Nikon D5100 iso 200",       "NIKON CORPORATION",      "NIKON D5100",              200,         {1.08139057356883e-05, 4.54226244236994e-06, 6.38474197730906e-06}, {-7.07927678963872e-08, -9.31196035040007e-08, -4.13922691621209e-08}},
//{"Nikon D5100 iso 250",       "NIKON CORPORATION",      "NIKON D5100",              250,         {1.32782572503635e-05, 5.58576873554951e-06, 7.94760574293392e-06}, {-6.45471697633433e-08, -9.36776892406314e-08, -4.67311708332366e-08}},
//{"Nikon D5100 iso 320",       "NIKON CORPORATION",      "NIKON D5100",              320,         {1.62800344916086e-05, 6.78044259498082e-06, 9.75401818169469e-06}, {-6.60705548994956e-08, -9.04832247143341e-08, -4.09714129736642e-08}},
//{"Nikon D5100 iso 400",       "NIKON CORPORATION",      "NIKON D5100",              400,         {2.03042377121186e-05, 8.36719071991299e-06, 1.2162559992601e-05}, {-6.48368260358449e-08, -7.66554705378132e-08, -2.45599451705476e-08}},
//{"Nikon D5100 iso 500",       "NIKON CORPORATION",      "NIKON D5100",              500,         {2.51508414371024e-05, 1.03158019774824e-05, 1.5036863393712e-05}, {-5.99689118765558e-08, -5.54824258477672e-08, 2.23171448378657e-09}},
//{"Nikon D5100 iso 640",       "NIKON CORPORATION",      "NIKON D5100",              640,         {3.13607007541968e-05, 1.28795155153313e-05, 1.87098396695899e-05}, {-7.23375753806584e-08, -7.72753472113923e-08, 5.03663227103341e-09}},
//{"Nikon D5100 iso 800",       "NIKON CORPORATION",      "NIKON D5100",              800,         {3.90242702359908e-05, 1.59560804544554e-05, 2.34309108035274e-05}, {-6.29163101726844e-08, -6.78132025291848e-08, 7.15435872072615e-09}},
//{"Nikon D5100 iso 1000",      "NIKON CORPORATION",      "NIKON D5100",              1000,        {4.87511322693238e-05, 1.99409519577222e-05, 2.93514423391217e-05}, {-4.6578694307761e-08, -8.07584552879584e-08, 1.90944218416291e-08}},
//{"Nikon D5100 iso 1250",      "NIKON CORPORATION",      "NIKON D5100",              1250,        {6.1107048834549e-05, 2.49895765286552e-05, 3.65690773939604e-05}, {-3.02296592597814e-08, -7.00528313786714e-08, 1.03890883297531e-07}},
//{"Nikon D5100 iso 1600",      "NIKON CORPORATION",      "NIKON D5100",              1600,        {7.62320681027145e-05, 3.11643310684653e-05, 4.57153256110289e-05}, {1.14514782437638e-08, -5.10609958519991e-08, 1.46614173401195e-07}},
//{"Nikon D5100 iso 2000",      "NIKON CORPORATION",      "NIKON D5100",              2000,        {9.54186351615094e-05, 3.87138926596809e-05, 5.73202339043625e-05}, {1.90777052221808e-08, 1.08352911230272e-07, 2.67719013975043e-07}},
//{"Nikon D5100 iso 2500",      "NIKON CORPORATION",      "NIKON D5100",              2500,        {0.000120254109990478, 4.86933803160992e-05, 7.19147550662044e-05}, {5.66024120329264e-08, 1.47207699286631e-07, 4.42713379143771e-07}},
//{"Nikon D5100 iso 3200",      "NIKON CORPORATION",      "NIKON D5100",              3200,        {0.000149253492963744, 6.01120702740969e-05, 8.98214490296902e-05}, {3.78677929451747e-07, 5.02823649697139e-07, 6.15800093570754e-07}},
//{"Nikon D5100 iso 4000",      "NIKON CORPORATION",      "NIKON D5100",              4000,        {0.000190578551709597, 7.61955849891878e-05, 0.000115859262167701}, {2.4395434704013e-07, 7.38364323826861e-07, 5.10647676617472e-07}},
//{"Nikon D5100 iso 5000",      "NIKON CORPORATION",      "NIKON D5100",              5000,        {0.000243307890055477, 9.70114504620688e-05, 0.000147012198494978}, {-1.02825957830929e-07, 5.88488342143576e-07, 6.40773170957563e-07}},
//{"Nikon D5100 iso 6400",      "NIKON CORPORATION",      "NIKON D5100",              6400,        {0.000310592605881356, 0.000123649232694049, 0.000185708679391227}, {-3.59182878295293e-07, 5.80694133998079e-07, 6.50101602524919e-07}},
//{"Nikon D5100 iso 8063",      "NIKON CORPORATION",      "NIKON D5100",              8063,        {0.000386572458428817, 0.000159829918779002, 0.000234274899410233}, {6.46522586252859e-07, -9.26193560577207e-10, 1.0751078494275e-06}},
//{"Nikon D5100 iso 10159",     "NIKON CORPORATION",      "NIKON D5100",              10159,       {0.000513123633542711, 0.000200046488277459, 0.000294028621269134}, {-1.88924509639636e-06, 1.08483373881359e-06, 2.35203681516458e-06}},
//{"Nikon D5100 iso 12800",     "NIKON CORPORATION",      "NIKON D5100",              12800,       {0.000654885845997662, 0.000251222495273516, 0.000365515022519041}, {-3.19398741752082e-06, 1.5849793096837e-06, 4.54761027090074e-06}},
//{"Nikon D5100 iso 25600",     "NIKON CORPORATION",      "NIKON D5100",              25600,       {0.0593756187282211, -0.0139090767488816, -0.00793148024059399}, {0.00162591459339716, 0.0118276165921827, 0.0122968020462748}},

// nikon d5200 by timur davletshin
{"Nikon D5200 iso 100",   "NIKON CORPORATION", "NIKON D5200", 100,   {8.44357279975327e-06, 2.87700239498207e-06, 4.71471435241959e-06}, {-1.59851390906603e-07, -9.45056675364089e-08, -2.59836013503805e-08}},
// {"Nikon D5200 iso 100",   "NIKON CORPORATION", "NIKON D5200", 100,   {8.44284597322115e-06, 2.88355825192422e-06, 4.70775411918953e-06}, {-1.52816724559017e-07, -9.48224045041411e-08, -2.66070728441539e-08}},
{"Nikon D5200 iso 125",   "NIKON CORPORATION", "NIKON D5200", 125,   {1.00323452391988e-05, 3.46787255715821e-06, 5.62832925371119e-06}, {-1.55926360262396e-07, -1.06216898381974e-07, -1.81779226997151e-08}},
// {"Nikon D5200 iso 125",   "NIKON CORPORATION", "NIKON D5200", 125,   {1.00341033991099e-05, 3.48316263555564e-06, 5.64563888648967e-06}, {-1.58235349561254e-07, -1.0758864758329e-07, -1.99377348679036e-08}},
{"Nikon D5200 iso 160",   "NIKON CORPORATION", "NIKON D5200", 160,   {1.19100294653547e-05, 4.10697591702703e-06, 6.76408915222684e-06}, {-1.48249873424569e-07, -9.72132672258075e-08, 4.72049690531132e-10}},
// {"Nikon D5200 iso 160",   "NIKON CORPORATION", "NIKON D5200", 160,   {1.19088656164533e-05, 4.14540941684071e-06, 6.75949093063126e-06}, {-1.40551096093225e-07, -1.07846903665761e-07, 2.27332694441256e-09}},
{"Nikon D5200 iso 200",   "NIKON CORPORATION", "NIKON D5200", 200,   {1.48781753038719e-05, 5.03443134652623e-06, 8.4287137088007e-06}, {-1.8922394232853e-07, -9.77828207681845e-08, 8.24061253435231e-09}},
// {"Nikon D5200 iso 200",   "NIKON CORPORATION", "NIKON D5200", 200,   {1.48740629249513e-05, 5.02264815621037e-06, 8.36259669664122e-06}, {-1.88656417526869e-07, -9.50151214053652e-08, 2.00892496803737e-08}},
{"Nikon D5200 iso 250",   "NIKON CORPORATION", "NIKON D5200", 250,   {1.79375698731279e-05, 6.10988380853492e-06, 1.01170756434247e-05}, {-1.75175086287957e-07, -9.61821375218515e-08, 5.29101553686193e-08}},
// {"Nikon D5200 iso 250",   "NIKON CORPORATION", "NIKON D5200", 250,   {1.81150479733404e-05, 6.21815858579106e-06, 1.02614710812768e-05}, {-1.88761772693072e-07, -1.13426455063273e-07, 3.98337661042186e-08}},
{"Nikon D5200 iso 320",   "NIKON CORPORATION", "NIKON D5200", 320,   {2.21811738741183e-05, 7.55378607484753e-06, 1.25228388143851e-05}, {-2.03441107215999e-07, -1.18420405042707e-07, 7.18733824960796e-08}},
// {"Nikon D5200 iso 320",   "NIKON CORPORATION", "NIKON D5200", 320,   {2.20356371494873e-05, 7.56972065348146e-06, 1.2587113930007e-05}, {-1.75928425695014e-07, -1.05029871351121e-07, 6.88909810739923e-08}},
{"Nikon D5200 iso 400",   "NIKON CORPORATION", "NIKON D5200", 400,   {2.76205858927432e-05, 9.43370246160183e-06, 1.58035228612042e-05}, {-2.41953635010021e-07, -1.32367428624117e-07, 8.95891588707371e-08}},
// {"Nikon D5200 iso 400",   "NIKON CORPORATION", "NIKON D5200", 400,   {2.74855379011818e-05, 9.34567472573437e-06, 1.5697421601795e-05}, {-2.23524736486788e-07, -1.07906711870505e-07, 1.0556765253095e-07}},
{"Nikon D5200 iso 500",   "NIKON CORPORATION", "NIKON D5200", 500,   {3.40723097354357e-05, 1.15747194693605e-05, 1.9453802452444e-05}, {-2.61254392386765e-07, -1.17134601022111e-07, 1.49655671656927e-07}},
// {"Nikon D5200 iso 500",   "NIKON CORPORATION", "NIKON D5200", 500,   {3.42609629443002e-05, 1.15786529948837e-05, 1.93178338191777e-05}, {-3.03268968088941e-07, -1.28382890899541e-07, 1.64665141231621e-07}},
{"Nikon D5200 iso 640",   "NIKON CORPORATION", "NIKON D5200", 640,   {4.22542663586451e-05, 1.42581818257156e-05, 2.4126511705351e-05}, {-3.3197513163697e-07, -1.01643556012438e-07, 2.37337080284404e-07}},
// {"Nikon D5200 iso 640",   "NIKON CORPORATION", "NIKON D5200", 640,   {4.23414478506376e-05, 1.4351844918031e-05, 2.43855537412057e-05}, {-3.3004612858488e-07, -1.37373597917818e-07, 2.17352127612163e-07}},
{"Nikon D5200 iso 800",   "NIKON CORPORATION", "NIKON D5200", 800,   {5.25182732551798e-05, 1.76983300020347e-05, 2.9817580688967e-05}, {-3.41656414002365e-07, -1.15759187535868e-07, 3.42691556234674e-07}},
// {"Nikon D5200 iso 800",   "NIKON CORPORATION", "NIKON D5200", 800,   {5.2691997791211e-05, 1.77169371038659e-05, 2.98907643822226e-05}, {-3.48165934223154e-07, -8.83689743263425e-08, 3.66524361397577e-07}},
{"Nikon D5200 iso 1000",  "NIKON CORPORATION", "NIKON D5200", 1000,  {6.43980957562216e-05, 2.22975174571773e-05, 3.80602680061281e-05}, {-3.03657087533898e-07, -1.64139700713266e-07, 4.0133286601746e-07}},
// {"Nikon D5200 iso 1000",  "NIKON CORPORATION", "NIKON D5200", 1000,  {6.43474434416799e-05, 2.20829536979448e-05, 3.70861012531511e-05}, {-3.0017402682929e-07, -9.3547827725742e-08, 5.92143805917115e-07}},
{"Nikon D5200 iso 1250",  "NIKON CORPORATION", "NIKON D5200", 1250,  {6.17813539291427e-05, 2.75949874805772e-05, 4.89158460286423e-05}, {1.63859880767661e-06, -9.81177640149995e-08, 4.37251413368824e-07}},
// {"Nikon D5200 iso 1250",  "NIKON CORPORATION", "NIKON D5200", 1250,  {6.20113446511656e-05, 2.7563243347592e-05, 4.79556447000728e-05}, {1.59537939358682e-06, -8.86728494101214e-08, 5.51488599815646e-07}},
{"Nikon D5200 iso 1600",  "NIKON CORPORATION", "NIKON D5200", 1600,  {7.96936557590889e-05, 3.45081648064049e-05, 6.10462478145966e-05}, {1.83573727045477e-06, -3.85003209023099e-08, 6.72268807939948e-07}},
// {"Nikon D5200 iso 1600",  "NIKON CORPORATION", "NIKON D5200", 1600,  {7.91316642896006e-05, 3.42707809836481e-05, 6.02820337249671e-05}, {1.84014499459596e-06, -6.65992067365421e-08, 6.87508078691507e-07}},
{"Nikon D5200 iso 2000",  "NIKON CORPORATION", "NIKON D5200", 2000,  {0.000103717030151601, 4.33802672890609e-05, 7.80491919266391e-05}, {1.76653276119872e-06, -1.21230127242505e-07, 6.52960117676014e-07}},
// {"Nikon D5200 iso 2000",  "NIKON CORPORATION", "NIKON D5200", 2000,  {0.000101828692590772, 4.29941349795629e-05, 7.80067079821854e-05}, {1.96357671170375e-06, -4.09236598750099e-08, 6.50160388559181e-07}},
{"Nikon D5200 iso 2500",  "NIKON CORPORATION", "NIKON D5200", 2500,  {0.000118355635485965, 5.38571492536469e-05, 9.78572166150933e-05}, {3.43800433711892e-06, -9.17347322943289e-09, 8.18632713776396e-07}},
// {"Nikon D5200 iso 2500",  "NIKON CORPORATION", "NIKON D5200", 2500,  {0.000119227298058161, 5.38400988726301e-05, 9.63145679359502e-05}, {3.27370499075898e-06, -8.90464863799022e-09, 1.03895974448882e-06}},
{"Nikon D5200 iso 3200",  "NIKON CORPORATION", "NIKON D5200", 3200,  {0.000156420037474102, 6.76301103764255e-05, 0.000127684101268656}, {4.00635208353513e-06, 2.77368094242082e-07, 7.0098448416014e-07}},
// {"Nikon D5200 iso 3200",  "NIKON CORPORATION", "NIKON D5200", 3200,  {0.000151787329346013, 6.78938439311905e-05, 0.00012574281790783}, {4.24947084557357e-06, 1.20517348685832e-07, 9.79159397214376e-07}},
{"Nikon D5200 iso 4000",  "NIKON CORPORATION", "NIKON D5200", 4000,  {0.00017604963052036, 8.54527430897547e-05, 0.000158388835448727}, {7.02217155515639e-06, 3.10362058704026e-07, 1.21504970735432e-06}},
// {"Nikon D5200 iso 4000",  "NIKON CORPORATION", "NIKON D5200", 4000,  {0.000180495476756873, 8.56236135712338e-05, 0.000156376682005501}, {6.7566883882983e-06, 2.22616911754533e-07, 1.66577412114205e-06}},
{"Nikon D5200 iso 5000",  "NIKON CORPORATION", "NIKON D5200", 5000,  {0.000234424919806788, 0.000106323677038399, 0.00019978549451366}, {7.68085843003221e-06, 7.5227018049085e-07, 1.79900694774288e-06}},
// {"Nikon D5200 iso 5000",  "NIKON CORPORATION", "NIKON D5200", 5000,  {0.000231210831434891, 0.000108133259374108, 0.000201312904394123}, {8.11095651775595e-06, 3.31982785627177e-07, 1.6232511862367e-06}},
{"Nikon D5200 iso 6400",  "NIKON CORPORATION", "NIKON D5200", 6400,  {0.000258278652091857, 0.000134170731124883, 0.000249049784492806}, {1.37798558650508e-05, 1.08931532598026e-06, 2.87702211520804e-06}},
// {"Nikon D5200 iso 6400",  "NIKON CORPORATION", "NIKON D5200", 6400,  {0.000267271263303772, 0.000136225590870488, 0.000247567491546161}, {1.30238581388155e-05, 5.40342050145035e-07, 3.20545582868999e-06}},
{"Nikon D5200 iso 8063",  "NIKON CORPORATION", "NIKON D5200", 8063,  {0.000253063310997502, 0.000139439587420027, 0.00026318709575747}, {1.78870549716572e-05, 2.95448888676036e-06, 3.84704237188935e-06}},
// {"Nikon D5200 iso 8063",  "NIKON CORPORATION", "NIKON D5200", 8063,  {0.000255794672071238, 0.000141442988516811, 0.000257849843842585}, {1.74357629724333e-05, 2.46819637335835e-06, 4.68133376783062e-06}},
{"Nikon D5200 iso 10159", "NIKON CORPORATION", "NIKON D5200", 10159, {0.000305665490806281, 0.00017920761915633, 0.000326728095381751}, {2.42448707221021e-05, 3.12888802211521e-06, 5.94280410158255e-06}},
// {"Nikon D5200 iso 10159", "NIKON CORPORATION", "NIKON D5200", 10159, {0.000299058450942801, 0.000182380467420953, 0.000330006676570054}, {2.48303070661371e-05, 2.37622950049843e-06, 5.45548069854394e-06}},
{"Nikon D5200 iso 12800", "NIKON CORPORATION", "NIKON D5200", 12800, {0.000385238004612291, 0.000228054338409934, 0.00039684034612927}, {3.08999832494445e-05, 3.5320541173557e-06, 9.3871331166903e-06}},
// {"Nikon D5200 iso 12800", "NIKON CORPORATION", "NIKON D5200", 12800, {0.000360555597758708, 0.000226311176452228, 0.000413054479356792}, {3.28277235054288e-05, 3.83057941911004e-06, 7.22700171709458e-06}},
{"Nikon D5200 iso 25600", "NIKON CORPORATION", "NIKON D5200", 25600, {0.000722736750302018, 0.000473192739679852, 0.000805354557194288}, {6.54169043669521e-05, 4.83323067162694e-06, 1.78395986418234e-05}},
// {"Nikon D5200 iso 25600", "NIKON CORPORATION", "NIKON D5200", 25600, {0.000732235227410513, 0.000470626308902042, 0.000805502784811692}, {6.5796190457872e-05, 4.9644626259132e-06, 1.80518838964151e-05}},

// nikon d5300 by Aurélien Pierre
{"NIKON D5300 iso 100",   "NIKON CORPORATION", "NIKON D5300", 100,   {2.79272882505438e-06, 1.94227932173818e-06, 6.06479796111042e-06}, {4.72465074785891e-09, 3.80009929338461e-09, 5.45295137862688e-09}},
{"NIKON D5300 iso 125",   "NIKON CORPORATION", "NIKON D5300", 125,   {3.29784356369169e-06, 2.44468461030138e-06, 6.74494786115919e-06}, {7.02974370794074e-09, 6.21774784249355e-09, 8.33839368858756e-09}},
{"NIKON D5300 iso 160",   "NIKON CORPORATION", "NIKON D5300", 160,   {3.90674216592394e-06, 3.02620732979847e-06, 7.90805499420815e-06}, {9.18734307119905e-09, 8.05508488598808e-09, 1.18532614593221e-08}},
{"NIKON D5300 iso 200",   "NIKON CORPORATION", "NIKON D5300", 200,   {4.57800537291512e-06, 3.70365483338714e-06, 1.17672926287425e-05}, {1.10056256291291e-08, 9.04232556818195e-09, 1.17262087913665e-08}},
{"NIKON D5300 iso 250",   "NIKON CORPORATION", "NIKON D5300", 250,   {5.70671167575504e-06, 4.63586785880374e-06, 1.04572758073623e-05}, {1.76792421593545e-08, 1.49751353198122e-08, 2.03747054127859e-08}},
{"NIKON D5300 iso 320",   "NIKON CORPORATION", "NIKON D5300", 320,   {1.02589557517719e-05, 7.71899599088561e-06, 2.49221451559896e-05}, {2.18980196853377e-08, 1.84200252048583e-08, 2.14628726204244e-08}},
{"NIKON D5300 iso 400",   "NIKON CORPORATION", "NIKON D5300", 400,   {1.25613384589757e-05, 9.60499160471028e-06, 2.88498252887724e-05}, {3.06290177142976e-08, 2.41852562667344e-08, 3.49861549513487e-08}},
{"NIKON D5300 iso 500",   "NIKON CORPORATION", "NIKON D5300", 500,   {1.57926448847148e-05, 1.19583539557234e-05, 3.08735034379645e-05}, {4.24258314691395e-08, 3.36356858446083e-08, 6.27714647116505e-08}},
{"NIKON D5300 iso 640",   "NIKON CORPORATION", "NIKON D5300", 640,   {2.11356332641914e-05, 1.50730774854151e-05, 2.24558364102849e-05}, {6.77222641670075e-08, 5.08719158826738e-08, 1.11580751511746e-07}},
{"NIKON D5300 iso 800",   "NIKON CORPORATION", "NIKON D5300", 800,   {2.53036844422274e-05, 1.88057042896507e-05, 5.20578315007887e-05}, {7.07415776309795e-08, 5.60274034175615e-08, 8.63306403713223e-08}},
{"NIKON D5300 iso 1000",  "NIKON CORPORATION", "NIKON D5300", 1000,  {3.18789757901179e-05, 2.34004296037071e-05, 6.47448908755866e-05}, {9.35921298169384e-08, 7.6072152747768e-08, 1.06486046127703e-07}},
{"NIKON D5300 iso 1250",  "NIKON CORPORATION", "NIKON D5300", 1250,  {4.11421176750838e-05, 2.95855736055648e-05, 7.13344225285495e-05}, {1.35022398270056e-07, 1.06887037376673e-07, 1.74895930411063e-07}},
{"NIKON D5300 iso 1600",  "NIKON CORPORATION", "NIKON D5300", 1600,  {5.16140674524484e-05, 3.72788710636901e-05, 9.79402182077933e-05}, {1.65871698220009e-07, 1.34563281422791e-07, 1.90518861503683e-07}},
{"NIKON D5300 iso 2000",  "NIKON CORPORATION", "NIKON D5300", 2000,  {6.55893576384423e-05, 4.65523968897294e-05, 0.000112961261775805}, {2.19163499567385e-07, 1.75896590199518e-07, 2.73368680185328e-07}},
{"NIKON D5300 iso 2500",  "NIKON CORPORATION", "NIKON D5300", 2500,  {8.26018071358041e-05, 5.88337584485401e-05, 0.0001656195483434}, {2.66525112253415e-07, 2.1199374212844e-07, 2.99095160422962e-07}},
{"NIKON D5300 iso 3200",  "NIKON CORPORATION", "NIKON D5300", 3200,  {0.00010320982128952, 7.40200531044544e-05, 0.000198716717951387}, {3.50748916309811e-07, 2.74022751471041e-07, 4.19609800865086e-07}},
{"NIKON D5300 iso 4000",  "NIKON CORPORATION", "NIKON D5300", 4000,  {0.000132550618767384, 9.24638210030386e-05, 0.000249849488146551}, {4.48429705886821e-07, 3.47735408833539e-07, 5.31004629876699e-07}},
{"NIKON D5300 iso 5000",  "NIKON CORPORATION", "NIKON D5300", 5000,  {0.000172841160753175, 0.000116297515671142, 0.000279242438169136}, {5.86086677452963e-07, 4.58980277283023e-07, 8.01999866151198e-07}},
{"NIKON D5300 iso 6400",  "NIKON CORPORATION", "NIKON D5300", 6400,  {0.000221445747362323, 0.000146525701739334, 0.000401453923721259}, {7.38113301507383e-07, 5.70674213211663e-07, 8.22011686714298e-07}},
{"NIKON D5300 iso 8000",  "NIKON CORPORATION", "NIKON D5300", 8000,  {0.000285835843142439, 0.000183442234408625, 0.000520841197036439}, {9.66068153245805e-07, 7.48578284217903e-07, 1.01957653528131e-06}},
{"NIKON D5300 iso 10000", "NIKON CORPORATION", "NIKON D5300", 10000, {0.000380342683611403, 0.000235303316030835, 0.000764405293205038}, {1.20770148645545e-06, 9.66347223267161e-07, 1.02449985455979e-06}},
{"NIKON D5300 iso 12800", "NIKON CORPORATION", "NIKON D5300", 12800, {0.000483455963237284, 0.000295469503820786, 0.000927217529277774}, {1.7200003369641e-06, 1.37050099240763e-06, 1.31571892830579e-06}},
{"NIKON D5300 iso 16127", "NIKON CORPORATION", "NIKON D5300", 16127, {0.000605772829044858, 0.00037892436315739, 0.00121054207386054}, {2.15683398979963e-06, 1.89447322973983e-06, 1.34437440730742e-06}},
{"NIKON D5300 iso 20319", "NIKON CORPORATION", "NIKON D5300", 20319, {0.00075231618825486, 0.000477021338616527, 0.00147748343332832}, {3.06263533549201e-06, 2.77429135970268e-06, 1.58333942522939e-06}},
{"NIKON D5300 iso 25600", "NIKON CORPORATION", "NIKON D5300", 25600, {0.000948108849147448, 0.000598058764818648, 0.00188627437185137}, {4.37614670090513e-06, 4.14772991546052e-06, 2.07590037231389e-06}},

// nikon d7000 by GomGom v2, all channels good
  {"NIKON D7000 iso 100",   "NIKON CORPORATION", "NIKON D7000", 100,   {5.70361058119957e-06, 2.40643718480073e-06, 3.44819151541497e-06}, {-7.26371535967068e-08, -4.86153510987303e-08, -4.03155675958822e-08}},
  {"NIKON D7000 iso 125",   "NIKON CORPORATION", "NIKON D7000", 125,   {7.0192773316183e-06, 2.98737822647183e-06, 4.24634871680488e-06}, {-9.16315896369596e-08, -6.10174192194244e-08, -4.02434507248189e-08}},
  {"NIKON D7000 iso 160",   "NIKON CORPORATION", "NIKON D7000", 160,   {8.60362293843687e-06, 3.63072322295503e-06, 5.29337751642272e-06}, {-1.16687636867991e-07, -6.28872850326511e-08, -6.03035512915354e-08}},
  {"NIKON D7000 iso 200",   "NIKON CORPORATION", "NIKON D7000", 200,   {9.91685972984502e-06, 4.38567627618459e-06, 6.67015499792575e-06}, {-6.7927811213465e-08, -5.13851992705295e-08, -3.69774845811691e-08}},
  {"NIKON D7000 iso 250",   "NIKON CORPORATION", "NIKON D7000", 250,   {1.22562462936052e-05, 5.4922638946455e-06, 8.33685555054764e-06}, {-7.92196957205539e-08, -8.07243503893391e-08, -5.17927154518577e-08}},
  {"NIKON D7000 iso 320",   "NIKON CORPORATION", "NIKON D7000", 320,   {1.54918407605515e-05, 6.85555182882116e-06, 1.0481922544074e-05}, {-9.62736991342674e-08, -7.70370344331459e-08, -7.07766558863733e-08}},
  {"NIKON D7000 iso 400",   "NIKON CORPORATION", "NIKON D7000", 400,   {1.90236296662045e-05, 8.36120197596719e-06, 1.30303016677143e-05}, {-8.91937293216182e-08, -5.29971306358586e-08, -7.26500328230203e-08}},
  {"NIKON D7000 iso 500",   "NIKON CORPORATION", "NIKON D7000", 500,   {2.39501961103012e-05, 1.03823151963248e-05, 1.6256109967366e-05}, {-1.35708498547995e-07, -6.41078378661287e-08, -9.12581739611544e-08}},
  {"NIKON D7000 iso 640",   "NIKON CORPORATION", "NIKON D7000", 640,   {2.88904470029672e-05, 1.26298404706172e-05, 1.98826735750109e-05}, {-7.38373858966737e-08, -1.25327442632388e-08, -4.13357977267619e-08}},
  {"NIKON D7000 iso 800",   "NIKON CORPORATION", "NIKON D7000", 800,   {3.67570484188119e-05, 1.59157950358734e-05, 2.50371118015167e-05}, {-1.42521171411666e-07, -1.6163827626363e-08, -5.46088775316727e-08}},
  {"NIKON D7000 iso 1000",  "NIKON CORPORATION", "NIKON D7000", 1000,  {4.6731927590618e-05, 2.00413363790222e-05, 3.12195229892156e-05}, {-2.04494630251378e-07, -3.57506760894719e-08, -5.22771512778234e-08}},
  {"NIKON D7000 iso 1250",  "NIKON CORPORATION", "NIKON D7000", 1250,  {5.77073741405525e-05, 2.52282748086166e-05, 3.90256746192118e-05}, {-1.811734368672e-07, -7.9307479464471e-08, -2.84398402342365e-08}},
  {"NIKON D7000 iso 1600",  "NIKON CORPORATION", "NIKON D7000", 1600,  {7.24199393500435e-05, 3.13933096685103e-05, 4.8707445289747e-05}, {-1.88678254188173e-07, -9.95683421643879e-09, 1.39083795491489e-08}},
  {"NIKON D7000 iso 2000",  "NIKON CORPORATION", "NIKON D7000", 2000,  {8.99923285039505e-05, 3.95554857393559e-05, 6.1594489822124e-05}, {-1.51154535367529e-07, -1.75505920081565e-09, 1.89951868476325e-08}},
  {"NIKON D7000 iso 2500",  "NIKON CORPORATION", "NIKON D7000", 2500,  {0.000113440430763707, 4.90408056125597e-05, 7.64007228830492e-05}, {-1.86986676107945e-07, 2.05536435473377e-07, 2.41980517647869e-07}},
  {"NIKON D7000 iso 3200",  "NIKON CORPORATION", "NIKON D7000", 3200,  {0.000142121183114241, 6.12515693143564e-05, 9.51548616527045e-05}, {3.26533198738207e-09, 4.70576253510708e-07, 5.48939513975651e-07}},
  {"NIKON D7000 iso 4000",  "NIKON CORPORATION", "NIKON D7000", 4000,  {0.000182654063236049, 7.84343199823704e-05, 0.000120683806047773}, {-2.16557773860082e-07, 4.1641717302481e-07, 6.38784229778294e-07}},
  {"NIKON D7000 iso 5000",  "NIKON CORPORATION", "NIKON D7000", 5000,  {0.000227201963710214, 9.7214828556275e-05, 0.000151617505561498}, {-2.23709139857807e-07, 7.81771428112485e-07, 7.8992710631253e-07}},
  {"NIKON D7000 iso 6400",  "NIKON CORPORATION", "NIKON D7000", 6400,  {0.000281185172903767, 0.000122844612170073, 0.000192720788276939}, {7.39206855044454e-07, 1.54076565894208e-06, 1.27185405659268e-06}},
  {"NIKON D7000 iso 8000",  "NIKON CORPORATION", "NIKON D7000", 8000,  {0.000357384506110654, 0.000154095826552526, 0.000241288430006912}, {9.35577657045401e-07, 2.00159234994379e-06, 1.9194184055553e-06}},
  {"NIKON D7000 iso 10000", "NIKON CORPORATION", "NIKON D7000", 10000, {0.000450207075233977, 0.0001942081194222, 0.000301555761309877}, {1.49928653215792e-06, 3.41880824771837e-06, 3.31959273875454e-06}},
  {"NIKON D7000 iso 12800", "NIKON CORPORATION", "NIKON D7000", 12800, {0.000557414687958056, 0.000247149173694856, 0.000375048810625821}, {4.22461226771128e-06, 4.07046551375823e-06, 5.96183696606146e-06}},
  {"NIKON D7000 iso 25600", "NIKON CORPORATION", "NIKON D7000", 25600, {0.00107380387275319, 0.000449260690182118, 0.000715703498266989}, {2.12794503625274e-05, 3.15918965578568e-05, 3.24101887378875e-05}},

// nikon d7000 by GomGom, blue is broken.
//{"NIKON D7000 iso 100",   "NIKON CORPORATION", "NIKON D7000", 100,   {3.27399688488054e-06, 2.38179789462537e-06, 0.0}, {-1.55729925920341e-08, -5.05299005887526e-08, 0.0}},
//{"NIKON D7000 iso 125",   "NIKON CORPORATION", "NIKON D7000", 125,   {4.05856652874075e-06, 2.95853016970173e-06, 0.0}, {-1.87492651711358e-08, -6.28706778392051e-08, 0.0}},
//{"NIKON D7000 iso 160",   "NIKON CORPORATION", "NIKON D7000", 160,   {5.07113275733823e-06, 3.59910893636234e-06, 0.0}, {-2.52536004587184e-08, -6.11016409349953e-08, 0.0}},
//{"NIKON D7000 iso 200",   "NIKON CORPORATION", "NIKON D7000", 200,   {6.2494851617538e-06, 4.43805295998874e-06,  0.0}, {-1.99118938792045e-08, -5.96190429166684e-08, 0.0}},
//{"NIKON D7000 iso 250",   "NIKON CORPORATION", "NIKON D7000", 250,   {7.8308321444199e-06, 5.52306879096044e-06,  0.0}, {-1.44801158395422e-08, -7.61326159691674e-08, 0.0}},
//{"NIKON D7000 iso 320",   "NIKON CORPORATION", "NIKON D7000", 320,   {9.69562391400911e-06, 6.71698983191291e-06, 0.0}, {-6.90776042329424e-09, -5.90505925162934e-08, 0.0}},
//{"NIKON D7000 iso 400",   "NIKON CORPORATION", "NIKON D7000", 400,   {1.22835282634258e-05, 8.38057921983336e-06, 0.0}, {-3.50860498731999e-08, -5.98484763151164e-08, 0.0}},
//{"NIKON D7000 iso 500",   "NIKON CORPORATION", "NIKON D7000", 500,   {1.55157535845539e-05, 1.03482601518511e-05, 0.0}, {-3.00552546554713e-08, -2.34762889558318e-08, 0.0}},
//{"NIKON D7000 iso 640",   "NIKON CORPORATION", "NIKON D7000", 640,   {1.89258361190014e-05, 1.29487901450686e-05, 0.0}, {1.26312452123837e-08, -3.89382157037495e-08,  0.0}},
//{"NIKON D7000 iso 800",   "NIKON CORPORATION", "NIKON D7000", 800,   {2.36968559377842e-05, 1.60817501024492e-05, 0.0}, {-9.45132685501379e-09, -4.5923505710025e-08,  0.0}},
//{"NIKON D7000 iso 1000",  "NIKON CORPORATION", "NIKON D7000", 1000,  {2.99229212221913e-05, 2.0208015539154e-05,  0.0}, {1.47696540411429e-08, -2.43714853702977e-08,  0.0}},
//{"NIKON D7000 iso 1250",  "NIKON CORPORATION", "NIKON D7000", 1250,  {3.7608911182586e-05, 2.53282382509448e-05,  0.0}, {1.14171373799865e-08, -7.71612007930266e-08,  0.0}},
//{"NIKON D7000 iso 1600",  "NIKON CORPORATION", "NIKON D7000", 1600,  {4.68233659791617e-05, 3.22697003351715e-05, 0.0}, {1.44459397036525e-07, -1.87377968093503e-07,  0.0}},
//{"NIKON D7000 iso 2000",  "NIKON CORPORATION", "NIKON D7000", 2000,  {5.87439470354981e-05, 3.99697236698932e-05, 0.0}, {1.40867259347372e-07, -2.98228206528265e-08,  0.0}},
//{"NIKON D7000 iso 2500",  "NIKON CORPORATION", "NIKON D7000", 2500,  {7.54195084354873e-05, 5.00224553895957e-05, 0.0}, {1.02923787934263e-07, 1.09423275696709e-07,   0.0}},
//{"NIKON D7000 iso 3200",  "NIKON CORPORATION", "NIKON D7000", 3200,  {9.48414271753929e-05, 6.22133719876228e-05, 0.0}, {2.75172093356657e-07, 2.92197109515196e-07,   0.0}},
//{"NIKON D7000 iso 4000",  "NIKON CORPORATION", "NIKON D7000", 4000,  {0.000116843069607486, 7.8307990480686e-05,  0.0}, {6.54646148872752e-07, 4.32960840919969e-07,   0.0}},
//{"NIKON D7000 iso 5000",  "NIKON CORPORATION", "NIKON D7000", 5000,  {0.000146430319060965, 9.79789818643772e-05, 0.0}, {1.01446723036264e-06, 6.84097225203974e-07,   0.0}},
//{"NIKON D7000 iso 6400",  "NIKON CORPORATION", "NIKON D7000", 6400,  {0.000187276628109155, 0.000125300903006307, 0.0}, {1.50610834504272e-06, 8.36990357642936e-07,   0.0}},
//{"NIKON D7000 iso 8000",  "NIKON CORPORATION", "NIKON D7000", 8000,  {0.000233293188975703, 0.000156179901535164, 0.0}, {2.30402365566127e-06, 1.62926768883885e-06,   0.0}},
//{"NIKON D7000 iso 10000", "NIKON CORPORATION", "NIKON D7000", 10000, {0.000283972529768703, 0.000197254163889745, 0.0}, {4.70660674330084e-06, 2.61309123663252e-06,   0.0}},
//{"NIKON D7000 iso 12800", "NIKON CORPORATION", "NIKON D7000", 12800, {0.000356626651268876, 0.00024560717429587,  0.0}, {7.56561457510978e-06, 4.57890128995118e-06,   0.0}},
//{"NIKON D7000 iso 25600", "NIKON CORPORATION", "NIKON D7000", 25600, {0.000826677107257534, 0.000462756983791003, 0.0}, {1.53175885129437e-05, 2.79831470022703e-05,   0.0}},

// nikon d7100 by swimmer
{"NIKON D7100 iso 100",   "NIKON CORPORATION", "NIKON D7100", 100,   {7.13395429263508e-06, 2.89387639736237e-06, 4.83792461750559e-06}, {-9.65063880099703e-08, -7.90223348785576e-08, -5.7480319939753e-08}},
{"NIKON D7100 iso 140",   "NIKON CORPORATION", "NIKON D7100", 140,   {9.58357491634568e-06, 3.82401244835942e-06, 6.47943988475936e-06}, {-1.24278251869937e-07, -8.55915561072445e-08, -5.70537261571787e-08}},
{"NIKON D7100 iso 200",   "NIKON CORPORATION", "NIKON D7100", 200,   {1.31100307506723e-05, 5.26877722452604e-06, 8.96795302279959e-06}, {-1.37996937971487e-07, -1.02036923005422e-07, -6.80611166274668e-08}},
{"NIKON D7100 iso 280",   "NIKON CORPORATION", "NIKON D7100", 280,   {1.80035962422163e-05, 7.16035135299421e-06, 1.22149096469976e-05}, {-1.92772068802928e-07, -1.06253972732985e-07, -5.32170938690644e-08}},
{"NIKON D7100 iso 400",   "NIKON CORPORATION", "NIKON D7100", 400,   {2.44997942155661e-05, 9.89175229876533e-06, 1.68275109393317e-05}, {-2.42108157535442e-07, -1.42322857265457e-07, -3.02110394691017e-08}},
{"NIKON D7100 iso 560",   "NIKON CORPORATION", "NIKON D7100", 560,   {3.40014360761117e-05, 1.35348908134126e-05, 2.34287612233754e-05}, {-3.04858652942389e-07, -1.23671379307107e-07, 4.89232991610322e-09}},
{"NIKON D7100 iso 800",   "NIKON CORPORATION", "NIKON D7100", 800,   {4.78484837360733e-05, 1.88433378379243e-05, 3.24963087324664e-05}, {-3.80844427404062e-07, -1.53751558406285e-07, -5.8965341088939e-09}},
{"NIKON D7100 iso 1100",  "NIKON CORPORATION", "NIKON D7100", 1100,  {6.66285353615179e-05, 2.62776893067389e-05, 4.55745807859241e-05}, {-4.21873716573962e-07, -1.26862656305826e-07, 7.69377902739988e-08}},
{"NIKON D7100 iso 1600",  "NIKON CORPORATION", "NIKON D7100", 1600,  {9.28228443384124e-05, 3.64087712911462e-05, 6.32849984949855e-05}, {-3.86211023379425e-07, -4.4496520607553e-08, 1.93177455723763e-07}},
{"NIKON D7100 iso 2200",  "NIKON CORPORATION", "NIKON D7100", 2200,  {0.00013004818087756, 5.12107569198274e-05, 8.88393705415331e-05}, {-5.27302030937505e-07, -6.354989241576e-08, 3.51368788457312e-07}},
{"NIKON D7100 iso 3200",  "NIKON CORPORATION", "NIKON D7100", 3200,  {0.000181539071942405, 7.15567198379109e-05, 0.000125154329098665}, {-4.28485769172521e-07, 2.40017558611474e-07, 8.1024987102878e-07}},
{"NIKON D7100 iso 4500",  "NIKON CORPORATION", "NIKON D7100", 4500,  {0.00025712765653404, 0.00010084045114295, 0.000176749559097901}, {-6.76364525538991e-07, 4.24876995101768e-07, 1.14380835135156e-06}},
{"NIKON D7100 iso 6400",  "NIKON CORPORATION", "NIKON D7100", 6400,  {0.000362335485388524, 0.000142046920572635, 0.000249056943205344}, {-4.83311820352629e-07, 9.06609376755095e-07, 2.22724685429262e-06}},
{"NIKON D7100 iso 9051",  "NIKON CORPORATION", "NIKON D7100", 9051,  {0.00045409548951379, 0.000175557452039411, 0.000308122585842107}, {-1.45191249510755e-06, 9.48939383403495e-07, 2.19416689709179e-06}},
{"NIKON D7100 iso 12800", "NIKON CORPORATION", "NIKON D7100", 12800, {0.000647267737776887, 0.00024829868366536, 0.000437833006394249}, {-2.11129556379606e-06, 1.57627623510199e-06, 3.38042998421905e-06}},
{"NIKON D7100 iso 25600", "NIKON CORPORATION", "NIKON D7100", 25600, {0.00123396350223046, 0.000503851910110282, 0.00086149414210644}, {-9.08873144694737e-06, 4.35004897816792e-07, 8.99110677802244e-06}},

// Nikon D7200 contributed by Ralf Brown
{"NIKON D7200 iso 100",   "NIKON CORPORATION", "NIKON D7200", 100,   {5.6646483151042e-06, 2.51102646279178e-06, 4.31476499425856e-06}, {6.5939425710672e-09, 4.00532842324563e-09, 6.08001761808847e-09}},
{"NIKON D7200 iso 125",   "NIKON CORPORATION", "NIKON D7200", 125,   {6.97225500065967e-06, 3.04604813628374e-06, 5.28910596244935e-06}, {7.99074711043659e-09, 4.85739908607359e-09, 6.52797112307063e-09}},
{"NIKON D7200 iso 140",   "NIKON CORPORATION", "NIKON D7200", 140,   {7.71413664031495e-06, 3.394072836406e-06, 5.84466955314476e-06}, {1.11389030168378e-08, 6.45282372212101e-09, 9.80271738528965e-09}},
{"NIKON D7200 iso 160",   "NIKON CORPORATION", "NIKON D7200", 160,   {8.59794550259047e-06, 3.75862994318132e-06, 6.53603046672671e-06}, {9.97720828316238e-09, 6.73054529972986e-09, 8.7754975148993e-09}},
{"NIKON D7200 iso 200",   "NIKON CORPORATION", "NIKON D7200", 200,   {1.05489100972666e-05, 4.61825030151422e-06, 8.03549347792393e-06}, {1.30639310260466e-08, 8.65988533534198e-09, 1.15709733319455e-08}},
{"NIKON D7200 iso 250",   "NIKON CORPORATION", "NIKON D7200", 250,   {1.30470631799806e-05, 5.69667393566003e-06, 9.94240607392553e-06}, {1.70097276134317e-08, 1.13642175657676e-08, 1.44458686976798e-08}},
{"NIKON D7200 iso 280",   "NIKON CORPORATION", "NIKON D7200", 280,   {1.46127226068181e-05, 6.37041647461426e-06, 1.10809032420441e-05}, {2.30101655830714e-08, 1.31283820955433e-08, 2.07866003954794e-08}},
{"NIKON D7200 iso 320",   "NIKON CORPORATION", "NIKON D7200", 320,   {1.63911637405806e-05, 7.11835061871788e-06, 1.24470392799785e-05}, {2.21326346849525e-08, 1.4903100632569e-08, 1.90376944650687e-08}},
{"NIKON D7200 iso 400",   "NIKON CORPORATION", "NIKON D7200", 400,   {2.04788565628154e-05, 8.91228275277293e-06, 1.55923602306233e-05}, {2.75463997646025e-08, 1.82639818284617e-08, 2.38503230315871e-08}},
{"NIKON D7200 iso 500",   "NIKON CORPORATION", "NIKON D7200", 500,   {2.56207724555847e-05, 1.11348279426262e-05, 1.94927718323902e-05}, {3.54480292646255e-08, 2.34271028305556e-08, 3.08932463742597e-08}},
{"NIKON D7200 iso 560",   "NIKON CORPORATION", "NIKON D7200", 560,   {2.89108465404976e-05, 1.24930594871801e-05, 2.17990132479931e-05}, {4.54485406107963e-08, 2.9028526508779e-08, 4.1312584608509e-08}},
{"NIKON D7200 iso 640",   "NIKON CORPORATION", "NIKON D7200", 640,   {3.22487133295837e-05, 1.3996404930997e-05, 2.45420839614575e-05}, {4.60670047321866e-08, 3.26420180766461e-08, 4.16908701081947e-08}},
{"NIKON D7200 iso 800",   "NIKON CORPORATION", "NIKON D7200", 800,   {4.05060532338518e-05, 1.75667381254947e-05, 3.06900843122546e-05}, {6.40463827313768e-08, 4.06388012830265e-08, 5.70415860452285e-08}},
{"NIKON D7200 iso 1000",  "NIKON CORPORATION", "NIKON D7200", 1000,  {5.07425283462278e-05, 2.20376225273459e-05, 3.86185075071114e-05}, {8.05700877633304e-08, 5.17238907736027e-08, 7.21768925364178e-08}},
{"NIKON D7200 iso 1100",  "NIKON CORPORATION", "NIKON D7200", 1100,  {5.72504616820956e-05, 2.4680984031421e-05, 4.31332094272095e-05}, {1.06452891764239e-07, 5.84340587996482e-08, 9.65355285096736e-08}},
{"NIKON D7200 iso 1250",  "NIKON CORPORATION", "NIKON D7200", 1250,  {6.34870727042341e-05, 2.75706721770455e-05, 4.83767920473091e-05}, {1.08449355655413e-07, 7.05105889663218e-08, 9.86998238242453e-08}},
{"NIKON D7200 iso 1600",  "NIKON CORPORATION", "NIKON D7200", 1600,  {8.01461992875406e-05, 3.47316970507234e-05, 6.0988850567616e-05}, {1.41080681259233e-07, 8.97341993319214e-08, 1.2737628890635e-07}},
{"NIKON D7200 iso 2000",  "NIKON CORPORATION", "NIKON D7200", 2000,  {0.000101217679757221, 4.35634539292122e-05, 7.67081207505347e-05}, {1.82688563765993e-07, 1.17265655095138e-07, 1.66332178470284e-07}},
{"NIKON D7200 iso 2200",  "NIKON CORPORATION", "NIKON D7200", 2200,  {0.000113927350772504, 4.8790904836417e-05, 8.57630338101561e-05}, {2.25858446959669e-07, 1.33257889778556e-07, 2.0087553547794e-07}},
{"NIKON D7200 iso 2500",  "NIKON CORPORATION", "NIKON D7200", 2500,  {0.000127806930711957, 5.48337835943925e-05, 9.70733459913882e-05}, {2.3356178142328e-07, 1.52292492099413e-07, 2.12733293041216e-07}},
{"NIKON D7200 iso 3200",  "NIKON CORPORATION", "NIKON D7200", 3200,  {0.000163369341805368, 7.00405760522611e-05, 0.000124380353698624}, {2.97412634565586e-07, 1.97483157754125e-07, 2.73401686726352e-07}},
{"NIKON D7200 iso 4000",  "NIKON CORPORATION", "NIKON D7200", 4000,  {0.000206182523908575, 8.8426619981112e-05, 0.000156863631370874}, {3.8292539649772e-07, 2.56906750114845e-07, 3.49475808901658e-07}},
{"NIKON D7200 iso 4500",  "NIKON CORPORATION", "NIKON D7200", 4500,  {0.000232377385160306, 9.94336494562366e-05, 0.000175626950256088}, {4.56828690099421e-07, 3.06953294486722e-07, 4.16170111423484e-07}},
{"NIKON D7200 iso 5000",  "NIKON CORPORATION", "NIKON D7200", 5000,  {0.000260960086192795, 0.000112258988826879, 0.000198507127294337}, {4.70176956707553e-07, 3.26248835320776e-07, 4.39039114138706e-07}},
{"NIKON D7200 iso 6400",  "NIKON CORPORATION", "NIKON D7200", 6400,  {0.000330205264187921, 0.000142208081184854, 0.000251544659373157}, {5.87311966871809e-07, 4.09460447857676e-07, 5.38441160421192e-07}},
{"NIKON D7200 iso 8000",  "NIKON CORPORATION", "NIKON D7200", 8000,  {0.000416178934362434, 0.000179829881142111, 0.000317422699834152}, {7.55626179777203e-07, 5.38981723091553e-07, 6.98141629856049e-07}},
{"NIKON D7200 iso 9000",  "NIKON CORPORATION", "NIKON D7200", 9000,  {0.000468903812123651, 0.000202248531598419, 0.00035510694391786}, {8.35619748387782e-07, 6.02519771524517e-07, 7.78228117351085e-07}},
{"NIKON D7200 iso 10000", "NIKON CORPORATION", "NIKON D7200", 10000, {0.000522905661974421, 0.000227108783078534, 0.000400213846514861}, {9.30339221648025e-07, 6.82280083460506e-07, 8.71406610504702e-07}},
{"NIKON D7200 iso 12800", "NIKON CORPORATION", "NIKON D7200", 12800, {0.000649739694289362, 0.000285243374869071, 0.000498410118933436}, {1.11505455580722e-06, 8.72185508819499e-07, 1.08171401009187e-06}},
{"NIKON D7200 iso 16000", "NIKON CORPORATION", "NIKON D7200", 16000, {0.000814577161409891, 0.000359352847838177, 0.00062543636107965}, {1.44625981810466e-06, 1.1576977824447e-06, 1.40887798740116e-06}},
{"NIKON D7200 iso 18000", "NIKON CORPORATION", "NIKON D7200", 18000, {0.00091477018314438, 0.000402805490035979, 0.000696829180124965}, {1.74826126556286e-06, 1.28552513772963e-06, 1.62611238836111e-06}},
{"NIKON D7200 iso 20000", "NIKON CORPORATION", "NIKON D7200", 20000, {0.00102591057269777, 0.000453390498262005, 0.000788713718103477}, {1.8069173746092e-06, 1.55588132080522e-06, 1.86526175421085e-06}},
{"NIKON D7200 iso 25600", "NIKON CORPORATION", "NIKON D7200", 25600, {0.0012861621111958, 0.000571072970485041, 0.000991021891657565}, {2.33935414314412e-06, 2.17575750979589e-06, 2.49837775424018e-06}},

// -------------------------------------------------------------------
// Olympus.
// -------------------------------------------------------------------

// olympus e-1 contributed by Damjan Vrenčur
{"Olympus E-1 iso 100",       "OLYMPUS CORPORATION",      "E-1",              100,         {8.74217127024554e-06, 5.2878702810528e-06, 5.26501875218597e-06}, {-2.20532019462792e-07, -2.01005398728175e-07, 1.09193447076722e-09}},
{"Olympus E-1 iso 200",       "OLYMPUS CORPORATION",      "E-1",              200,         {1.47387742014962e-05, 8.69224181798825e-06, 9.04462700373314e-06}, {-3.01405447532379e-07, -2.38876988606118e-07, 1.24233871582961e-07}},
{"Olympus E-1 iso 400",       "OLYMPUS CORPORATION",      "E-1",              400,         {2.36903015338395e-05, 1.43769633146132e-05, 1.5013478109016e-05}, {3.97047685913641e-09, 1.85545911670722e-08, 8.45472247580033e-07}},
{"Olympus E-1 iso 800",       "OLYMPUS CORPORATION",      "E-1",              800,         {4.32037785664211e-05, 2.47080431346722e-05, 2.88391247965338e-05}, {1.10430101502086e-06, 1.07780574613107e-06, 2.0757129002039e-06}},

// olympus e-p2 contributed by Damjan Vrenčur
{"Olympus E-P2 iso 100", "OLYMPUS IMAGING CORP.", "E-P2", 100,  {9.77938622752664e-06, 5.50631328403374e-06, 7.93222949932615e-06}, {-9.84226935907546e-08, 8.81250381399964e-09, 1.20170803975941e-07}},
{"Olympus E-P2 iso 200", "OLYMPUS IMAGING CORP.", "E-P2", 200,  {9.96687630163281e-06, 5.54964450649148e-06, 8.25379144519012e-06}, {-4.26343622260664e-08, -2.70430059162575e-08, 3.13041519470428e-08}},
{"Olympus E-P2 iso 400", "OLYMPUS IMAGING CORP.", "E-P2", 400,  {1.78676694231019e-05, 9.37782111270381e-06, 1.41124841952903e-05}, {1.90133132211059e-08, 1.4860011508442e-07, 2.9166085183925e-07}},
{"Olympus E-P2 iso 800", "OLYMPUS IMAGING CORP.", "E-P2", 800,  {3.32414951341578e-05, 1.74381147818894e-05, 2.72952685162943e-05}, {4.2653528547826e-07, 5.87051568572042e-07, 7.38032043111424e-07}},
{"Olympus E-P2 iso 1600","OLYMPUS IMAGING CORP.", "E-P2", 1600, {6.29672283506924e-05, 3.12004382551687e-05, 4.92714689561855e-05}, {2.68133434050926e-06, 3.17217124979968e-06, 3.88552559539031e-06}},
{"Olympus E-P2 iso 3200","OLYMPUS IMAGING CORP.", "E-P2", 3200, {0.000127477964247651, 6.21727330399859e-05, 0.000101041381234184}, {1.12422239925038e-05, 1.24409126387774e-05, 1.4407158890129e-05}},
{"Olympus E-P2 iso 6400","OLYMPUS IMAGING CORP.", "E-P2", 6400, {0.000298721601371108, 0.000133634030218994, 0.000220677683810357}, {4.09683079298609e-05, 4.53566871489171e-05, 5.36590292460859e-05}},

// olympus e-p3 contributed by ant, iso 12800 broken (preprocessed)
{"Olympus E-P3 iso 200",       "OLYMPUS IMAGING CORP.",      "E-P3",              200,         {6.15619918647734e-06, 4.16020947791911e-06, 1.21801608203362e-05}, {7.25094207967001e-08, 5.69489016626155e-08, 7.45895121802068e-08}},
{"Olympus E-P3 iso 250",       "OLYMPUS IMAGING CORP.",      "E-P3",              250,         {7.81898768604687e-06, 5.28878613776943e-06, 1.54604526263279e-05}, {1.10357123189981e-07, 8.69266150035675e-08, 1.13950989485014e-07}},
{"Olympus E-P3 iso 320",       "OLYMPUS IMAGING CORP.",      "E-P3",              320,         {1.00886647020904e-05, 6.8243148116244e-06, 1.96622611945243e-05}, {1.67234521776807e-07, 1.30112672716952e-07, 1.73289655863266e-07}},
{"Olympus E-P3 iso 400",       "OLYMPUS IMAGING CORP.",      "E-P3",              400,         {1.24101561717391e-05, 8.38350506350029e-06, 2.42647881113207e-05}, {1.34530361017604e-07, 1.02553327050412e-07, 1.41457608242855e-07}},
{"Olympus E-P3 iso 500",       "OLYMPUS IMAGING CORP.",      "E-P3",              500,         {1.59201199847706e-05, 1.07262243112729e-05, 3.10816730123436e-05}, {2.0135977070011e-07, 1.52253676705687e-07, 2.09986144397725e-07}},
{"Olympus E-P3 iso 640",       "OLYMPUS IMAGING CORP.",      "E-P3",              640,         {2.05306581744876e-05, 1.38279753461943e-05, 4.00179793810831e-05}, {3.087453026887e-07, 2.33199279044475e-07, 3.226826125255e-07}},
{"Olympus E-P3 iso 800",       "OLYMPUS IMAGING CORP.",      "E-P3",              800,         {2.57099104036087e-05, 1.72574605095918e-05, 5.00741191950761e-05}, {3.07519548381373e-07, 2.24485399286868e-07, 3.22142852527639e-07}},
{"Olympus E-P3 iso 1000",       "OLYMPUS IMAGING CORP.",      "E-P3",              1000,         {3.33635403960771e-05, 2.24853604993404e-05, 6.51359690731164e-05}, {4.65014325838843e-07, 3.38375965862234e-07, 4.82666963273052e-07}},
{"Olympus E-P3 iso 1250",       "OLYMPUS IMAGING CORP.",      "E-P3",              1250,         {4.34896899521553e-05, 2.93565396110127e-05, 8.53535145821489e-05}, {7.09694464892971e-07, 5.13438393537708e-07, 7.21673128077861e-07}},
{"Olympus E-P3 iso 1600",       "OLYMPUS IMAGING CORP.",      "E-P3",              1600,         {5.71555013842648e-05, 3.87463835144703e-05, 0.000112651260318146}, {1.03075607821111e-06, 7.34037118210004e-07, 1.04287535897745e-06}},
{"Olympus E-P3 iso 2000",       "OLYMPUS IMAGING CORP.",      "E-P3",              2000,         {7.58561920977265e-05, 5.12543770990082e-05, 0.000148126209208589}, {1.58392740408294e-06, 1.14092885289175e-06, 1.58422881976866e-06}},
{"Olympus E-P3 iso 2500",       "OLYMPUS IMAGING CORP.",      "E-P3",              2500,         {0.000103020890723356, 6.93761610317161e-05, 0.000200759322626083}, {2.42231430600711e-06, 1.77705395663356e-06, 2.42222785795237e-06}},
{"Olympus E-P3 iso 3200",       "OLYMPUS IMAGING CORP.",      "E-P3",              3200,         {0.000139022447497922, 9.51452670802385e-05, 0.000281853048076324}, {3.2861051113083e-06, 2.33650456481839e-06, 3.21644290978257e-06}},
{"Olympus E-P3 iso 4000",       "OLYMPUS IMAGING CORP.",      "E-P3",              4000,         {0.000194444320275143, 0.000131723176400371, 0.000377253618720688}, {4.93510772555196e-06, 3.55872382802495e-06, 4.85435280169394e-06}},
{"Olympus E-P3 iso 5000",       "OLYMPUS IMAGING CORP.",      "E-P3",              5000,         {0.00027421073430683, 0.000185869916330069, 0.000533961137110812}, {7.42542963742729e-06, 5.40034884621189e-06, 7.31688737222079e-06}},
{"Olympus E-P3 iso 6400",       "OLYMPUS IMAGING CORP.",      "E-P3",              6400,         {0.000416245723012963, 0.000285488220218586, 0.00081714186637563}, {1.02153704384536e-05, 7.15132496746668e-06, 9.85486201466949e-06}},
{"Olympus E-P3 iso 8000",       "OLYMPUS IMAGING CORP.",      "E-P3",              8000,         {0.000550918659775735, 0.000372167266873588, 0.00104533251833112}, {1.57486117564577e-05, 1.11949429779954e-05, 1.48399125020925e-05}},
{"Olympus E-P3 iso 10000",       "OLYMPUS IMAGING CORP.",      "E-P3",              10000,         {0.000813898850690959, 0.000561809727452035, 0.00150245251136926}, {2.3138173944301e-05, 1.64745381974565e-05, 2.11134550786376e-05}},
{"Olympus E-P3 iso 12800",       "OLYMPUS IMAGING CORP.",      "E-P3",              12800,         {0.00124589714428727, 0.000886354074331235, 0.00219189965905086}, {3.09863460565562e-05, 2.10625721213607e-05, 2.74646287687619e-05}},

// olympus e-pm1, iso > 1600 is preprocessed and works suboptimally. contributed by Rodney Heyd
{"Olympus E-PM1 iso 200",   "OLYMPUS IMAGING CORP.", "E-PM1", 200,   {8.66652261559245e-06, 4.35087917998192e-06, 7.69389371981886e-06}, {8.34855168661318e-08, 6.95534914139979e-08, 9.7717261649068e-08}},
{"Olympus E-PM1 iso 250",   "OLYMPUS IMAGING CORP.", "E-PM1", 250,   {1.07851798254364e-05, 5.40239731359464e-06, 9.53269082266039e-06}, {1.27544880488124e-07, 1.05646088912587e-07, 1.46003664623085e-07}},
{"Olympus E-PM1 iso 320",   "OLYMPUS IMAGING CORP.", "E-PM1", 320,   {1.35215383719135e-05, 6.97337209100638e-06, 1.19251535927031e-05}, {2.01204609434355e-07, 1.63770013175513e-07, 2.25958058395147e-07}},
{"Olympus E-PM1 iso 400",   "OLYMPUS IMAGING CORP.", "E-PM1", 400,   {1.67160191317563e-05, 8.21088649866628e-06, 1.45506894816976e-05}, {1.63101940047005e-07, 1.3769898362322e-07, 1.99528945110646e-07}},
{"Olympus E-PM1 iso 500",   "OLYMPUS IMAGING CORP.", "E-PM1", 500,   {2.09648681507859e-05, 1.03157654413783e-05, 1.83575856708931e-05}, {2.53271127944652e-07, 2.10757223864688e-07, 2.96829711247901e-07}},
{"Olympus E-PM1 iso 640",   "OLYMPUS IMAGING CORP.", "E-PM1", 640,   {2.62525965568347e-05, 1.27055247070234e-05, 2.25336547959266e-05}, {3.94498039548274e-07, 3.28722040201453e-07, 4.47089124475814e-07}},
{"Olympus E-PM1 iso 800",   "OLYMPUS IMAGING CORP.", "E-PM1", 800,   {3.38620752429216e-05, 1.64241990922245e-05, 2.91888110396548e-05}, {4.35864587523108e-07, 3.59668739974078e-07, 5.07835974370017e-07}},
{"Olympus E-PM1 iso 1000",  "OLYMPUS IMAGING CORP.", "E-PM1", 1000,  {4.26673909490675e-05, 2.06801406251036e-05, 3.71663646941301e-05}, {6.72127806461744e-07, 5.64816219862707e-07, 7.42705794590336e-07}},
{"Olympus E-PM1 iso 1250",  "OLYMPUS IMAGING CORP.", "E-PM1", 1250,  {5.43496075628852e-05, 2.58589768174911e-05, 4.60976752036865e-05}, {1.05551975026376e-06, 8.7575288619518e-07, 1.13548376520419e-06}},
{"Olympus E-PM1 iso 1600",  "OLYMPUS IMAGING CORP.", "E-PM1", 1600,  {6.52074691615036e-05, 3.16064110161702e-05, 5.66526583972657e-05}, {1.50598759853095e-06, 1.2610387708201e-06, 1.55824947057983e-06}},
{"Olympus E-PM1 iso 2000",  "OLYMPUS IMAGING CORP.", "E-PM1", 2000,  {8.6100393518731e-05, 4.11674468316151e-05, 7.43486587308852e-05}, {2.20517892622213e-06, 1.87974673151666e-06, 2.22588245616624e-06}},
{"Olympus E-PM1 iso 2500",  "OLYMPUS IMAGING CORP.", "E-PM1", 2500,  {0.000112435193863964, 5.36164166992351e-05, 9.76027097028646e-05}, {3.22712528983639e-06, 2.76426841503141e-06, 3.22427947447836e-06}},
{"Olympus E-PM1 iso 3200",  "OLYMPUS IMAGING CORP.", "E-PM1", 3200,  {0.000147198255068724, 6.99869395357261e-05, 0.000129175217047646}, {4.30408591639975e-06, 3.72126270409099e-06, 4.15859451736954e-06}},
{"Olympus E-PM1 iso 4000",  "OLYMPUS IMAGING CORP.", "E-PM1", 4000,  {0.000197800769253332, 9.34255973793394e-05, 0.000172210126188477}, {6.29970632912779e-06, 5.46817614866052e-06, 6.09879881278941e-06}},
{"Olympus E-PM1 iso 5000",  "OLYMPUS IMAGING CORP.", "E-PM1", 5000,  {0.00027023221991854, 0.000126939257045243, 0.000235092329657128}, {9.10067213148147e-06, 7.88807441560908e-06, 8.77810437980563e-06}},
{"Olympus E-PM1 iso 6400",  "OLYMPUS IMAGING CORP.", "E-PM1", 6400,  {0.000384139531187656, 0.000182471033736721, 0.000332754493761912}, {1.16098939281632e-05, 9.81981297422856e-06, 1.11763398026721e-05}},
{"Olympus E-PM1 iso 8000",  "OLYMPUS IMAGING CORP.", "E-PM1", 8000,  {0.000552257273320099, 0.000260976593322883, 0.000474047354143049}, {1.66575901170245e-05, 1.38762689895813e-05, 1.62264374256839e-05}},
{"Olympus E-PM1 iso 10000", "OLYMPUS IMAGING CORP.", "E-PM1", 10000, {0.000790426440028207, 0.000375511081823265, 0.000682258358364149}, {2.4170841395407e-05, 1.99499114447758e-05, 2.39146058594996e-05}},
{"Olympus E-PM1 iso 12800", "OLYMPUS IMAGING CORP.", "E-PM1", 12800, {0.00126050109907959, 0.000600416436043165, 0.00110791304196838}, {3.49432960826599e-05, 2.75877356596721e-05, 3.35377516637321e-05}},

// olympus e-3 contributed by Damjan Vrenčur, blue channel dodgy
{"Olympus E-3 iso 100",  "OLYMPUS IMAGING CORP.", "E-3", 100,  {6.84566571893634e-06, 4.98951315081828e-06, 0.0f}, {-9.5642661191206e-08, -1.27485239580422e-07,  0.0f}},
{"Olympus E-3 iso 200",  "OLYMPUS IMAGING CORP.", "E-3", 200,  {1.09233103474304e-05, 7.81695305969381e-06, 0.0f}, {-3.89659523097461e-08, -4.19137325550867e-09, 0.0f}},
{"Olympus E-3 iso 400",  "OLYMPUS IMAGING CORP.", "E-3", 400,  {1.84207158812108e-05, 1.27225044172833e-05, 0.0f}, {-1.28462949023062e-07, -2.78487627640167e-08, 0.0f}},
{"Olympus E-3 iso 800",  "OLYMPUS IMAGING CORP.", "E-3", 800,  {3.51214594948481e-05, 2.33755276497792e-05, 0.0f}, {2.16906711036928e-07, 4.43871944237754e-07,   0.0f}},
{"Olympus E-3 iso 1600", "OLYMPUS IMAGING CORP.", "E-3", 1600, {6.87751542847639e-05, 4.3800291983203e-05,  0.0f}, {1.98604376431088e-06, 2.40050901999681e-06,   0.0f}},
{"Olympus E-3 iso 3200", "OLYMPUS IMAGING CORP.", "E-3", 3200, {0.000125593540839062, 8.45783704597783e-05, 0.0f}, {1.05512769663653e-05, 9.30657365426296e-06,   0.0f}},

// olympus e-30 by Jean-Luc Le Corre
  {"Olympus E-30 iso 100",  "OLYMPUS IMAGING CORP.", "E-30", 100,  {9.7071376303464e-06, 7.11119859126147e-06, 1.43234477170861e-05}, {-9.06560192729126e-08, -2.41125952783539e-07, -3.19385326622867e-07}},
  {"Olympus E-30 iso 125",  "OLYMPUS IMAGING CORP.", "E-30", 125,  {1.20456303941201e-05, 8.5916937354069e-06, 1.63507564674459e-05}, {-7.39375265246587e-08, -2.69554361744923e-07, -2.48410739378445e-07}},
  {"Olympus E-30 iso 160",  "OLYMPUS IMAGING CORP.", "E-30", 160,  {1.56529989102257e-05, 1.02934362650981e-05, 1.77166879017183e-05}, {-4.20365734464088e-08, -2.02996957670653e-07, 5.29589874186308e-09}},
  {"Olympus E-30 iso 200",  "OLYMPUS IMAGING CORP.", "E-30", 200,  {1.0095886562545e-05, 6.93104653792698e-06, 1.16843434000443e-05}, {-1.22201572980062e-07, -2.27477805535749e-07, 3.23770042930603e-08}},
  {"Olympus E-30 iso 250",  "OLYMPUS IMAGING CORP.", "E-30", 250,  {1.29475199645689e-05, 8.46056520489638e-06, 1.35672020706393e-05}, {-1.56424286258763e-07, -2.17640666926632e-07, 1.39913184326539e-07}},
  {"Olympus E-30 iso 320",  "OLYMPUS IMAGING CORP.", "E-30", 320,  {1.54937257192008e-05, 9.96565953037391e-06, 1.67345924078165e-05}, {-8.68535574704753e-08, -1.56806153982179e-07, 2.43131977515659e-07}},
  {"Olympus E-30 iso 400",  "OLYMPUS IMAGING CORP.", "E-30", 400,  {1.9113446058363e-05, 1.07745915338773e-05, 1.75354364481865e-05}, {-1.89298531918018e-07, 1.12954758094748e-07, 6.39061318416547e-07}},
  {"Olympus E-30 iso 500",  "OLYMPUS IMAGING CORP.", "E-30", 500,  {2.11949380361048e-05, 1.35577131775584e-05, 2.31360696860893e-05}, {7.23334285808473e-08, 1.22967405892309e-07, 7.02832916225672e-07}},
  {"Olympus E-30 iso 640",  "OLYMPUS IMAGING CORP.", "E-30", 640,  {2.73199109169886e-05, 1.67468277151008e-05, 2.95174774898525e-05}, {1.99717400841382e-07, 2.75231709016361e-07, 8.64971660648951e-07}},
  {"Olympus E-30 iso 800",  "OLYMPUS IMAGING CORP.", "E-30", 800,  {3.37153575569615e-05, 2.06893507139736e-05, 3.88227466208396e-05}, {1.6088342752527e-07, 2.65337245132019e-07, 6.41368781875726e-07}},
  {"Olympus E-30 iso 1000", "OLYMPUS IMAGING CORP.", "E-30", 1000, {4.35040657457758e-05, 2.74901261504848e-05, 4.77520441360138e-05}, {3.50283692595926e-07, 1.52239280251802e-07, 1.20426607195639e-06}},
  {"Olympus E-30 iso 1250", "OLYMPUS IMAGING CORP.", "E-30", 1250, {5.26077166899586e-05, 3.41240085546217e-05, 5.88240938452458e-05}, {8.44266815099888e-07, 5.27504382892626e-07, 1.93639030690002e-06}},
  {"Olympus E-30 iso 1600", "OLYMPUS IMAGING CORP.", "E-30", 1600, {6.57353551949384e-05, 3.65886597333215e-05, 6.54938510019982e-05}, {1.49787788804922e-06, 2.30740528605681e-06, 3.78507040181792e-06}},
  {"Olympus E-30 iso 2000", "OLYMPUS IMAGING CORP.", "E-30", 2000, {8.62141802726288e-05, 5.03284121108448e-05, 7.89194460796899e-05}, {2.44624695228374e-06, 2.29047427336703e-06, 5.80698306285845e-06}},
  {"Olympus E-30 iso 2500", "OLYMPUS IMAGING CORP.", "E-30", 2500, {0.000110342048549196, 6.48384215703911e-05, 0.000101826544077179}, {4.20346704162012e-06, 3.90138066601788e-06, 7.79212482176447e-06}},
  {"Olympus E-30 iso 3200", "OLYMPUS IMAGING CORP.", "E-30", 3200, {0.000139880356662346, 8.13578184026246e-05, 0.000152070112576575}, {5.78577431993228e-06, 6.62245038671539e-06, 9.69460041876962e-06}},

// olympus e-400 by Jean-Luc Le Corre
  {"Olympus E-400 iso 100",  "OLYMPUS IMAGING CORP.", "E-400", 100,  {2.30684791960491e-05, 1.20696033495597e-05, 1.00848865658141e-05}, {-8.54103331013092e-07, -1.42527349623183e-06, -2.87841720924075e-07}},
  {"Olympus E-400 iso 125",  "OLYMPUS IMAGING CORP.", "E-400", 125,  {2.83955222632967e-05, 1.40985072850271e-05, 1.21391729015729e-05}, {-1.08953999985679e-06, -1.62029565464596e-06, -3.81919881022713e-07}},
  {"Olympus E-400 iso 160",  "OLYMPUS IMAGING CORP.", "E-400", 160,  {2.70515138285308e-05, 1.33172520598202e-05, 1.37785726019925e-05}, {-6.98482844527664e-07, -1.19479663725119e-06, -3.22837508114414e-07}},
  {"Olympus E-400 iso 200",  "OLYMPUS IMAGING CORP.", "E-400", 200,  {3.09344532234244e-05, 1.47371604111484e-05, 1.61463658491032e-05}, {-6.89720465048891e-07, -1.12230112966622e-06, -3.19988535123893e-07}},
  {"Olympus E-400 iso 250",  "OLYMPUS IMAGING CORP.", "E-400", 250,  {3.67794423685217e-05, 1.66150087079536e-05, 1.84788621177898e-05}, {-6.1875709374297e-07, -9.63912483154833e-07, -1.15978794348621e-07}},
  {"Olympus E-400 iso 320",  "OLYMPUS IMAGING CORP.", "E-400", 320,  {4.11920649501078e-05, 1.95000849297568e-05, 2.19723626568078e-05}, {-4.24516904011125e-07, -1.06526805602891e-06, -6.60788628775535e-08}},
  {"Olympus E-400 iso 400",  "OLYMPUS IMAGING CORP.", "E-400", 400,  {5.40046297608869e-05, 2.4119695912163e-05, 2.54657796327534e-05}, {-6.05112132690245e-07, -1.2216537951603e-06, 2.9245418452642e-07}},
  {"Olympus E-400 iso 500",  "OLYMPUS IMAGING CORP.", "E-400", 500,  {5.5961733798744e-05, 2.40121213529456e-05, 3.02249723260322e-05}, {2.63601349364295e-07, -1.07275487055428e-07, 9.05872534385637e-07}},
  {"Olympus E-400 iso 640",  "OLYMPUS IMAGING CORP.", "E-400", 640,  {7.43157598744775e-05, 3.10455889474033e-05, 3.68331487149578e-05}, {3.69025880450784e-07, -2.35530989431568e-07, 1.32258141728663e-06}},
  {"Olympus E-400 iso 800",  "OLYMPUS IMAGING CORP.", "E-400", 800,  {9.12475252161197e-05, 3.63676129281178e-05, 4.64287060710312e-05}, {1.30621046453119e-06, 7.84832322568365e-07, 2.27114315111596e-06}},
  {"Olympus E-400 iso 1250", "OLYMPUS IMAGING CORP.", "E-400", 1250, {7.94019058951963e-05, 3.36648275372961e-05, 4.73971380480946e-05}, {2.53433339286257e-06, 1.48458720969285e-06, 1.80705024956534e-06}},
  {"Olympus E-400 iso 1000", "OLYMPUS IMAGING CORP.", "E-400", 1000, {9.27541922228684e-05, 3.71047452003509e-05, 4.63187947193072e-05}, {1.54490295533143e-06, 8.63513656152105e-07, 2.16845292944425e-06}},
  {"Olympus E-400 iso 1600", "OLYMPUS IMAGING CORP.", "E-400", 1600, {8.18339699191006e-05, 3.48044865596607e-05, 4.70967171662682e-05}, {2.21093414150949e-06, 1.17066257582798e-06, 2.26165073492259e-06}},

// olympus e-420, contributed by marcel
  {"Olympus E-420 iso 100",  "OLYMPUS IMAGING CORP.", "E-420", 100,  {1.11327038070984e-05, 5.04649607859026e-06, 6.79807682396903e-06}, {-5.04175907769004e-08, -9.55775785390468e-08, -4.91154444033116e-08}},
  {"Olympus E-420 iso 200",  "OLYMPUS IMAGING CORP.", "E-420", 200,  {1.62740093902056e-05, 7.08181594644404e-06, 1.03997806932252e-05}, {6.02642623744443e-08, 3.79984846167837e-08, 7.3091753845666e-08}},
  {"Olympus E-420 iso 400",  "OLYMPUS IMAGING CORP.", "E-420", 400,  {3.08819743786241e-05, 1.32150588665836e-05, 1.82101244554622e-05}, {1.90493285627675e-07, 1.7011691963151e-07, 2.03211058319469e-07}},
  {"Olympus E-420 iso 800",  "OLYMPUS IMAGING CORP.", "E-420", 800,  {5.87041699537705e-05, 2.62647034859489e-05, 3.4558979210781e-05}, {1.04002463544227e-06, 5.53781235750961e-07, 9.91505184059283e-07}},
  {"Olympus E-420 iso 1600", "OLYMPUS IMAGING CORP.", "E-420", 1600, {0.000113713905893592, 4.96771878652259e-05, 6.79786468674286e-05}, {4.88544869503018e-06, 3.82951314330521e-06, 4.81990675996362e-06}},

// olympus e-520 by daniel schury
{"Olympus E-520 iso 100",  "OLYMPUS IMAGING CORP.",  "E-520", 100,  {2.45930250418796e-05, 1.02099264252535e-05, 1.03094391065689e-05}, {-8.7398616535417e-07, -7.21658765122154e-07, -1.60676226973807e-07}},
{"Olympus E-520 iso 200",  "OLYMPUS IMAGING CORP.",  "E-520", 200,  {3.46489855516779e-05, 1.45903107693433e-05, 1.5220629132344e-05}, {-1.1666398111728e-06, -9.94818909397097e-07, -2.6759738242418e-07}},
{"Olympus E-520 iso 400",  "OLYMPUS IMAGING CORP.",  "E-520", 400,  {5.45836721196911e-05, 2.43797935845142e-05, 2.66962175131977e-05}, {-1.45942557378248e-06, -1.52180153754017e-06, -4.2630023922702e-07}},
{"Olympus E-520 iso 800",  "OLYMPUS IMAGING CORP.",  "E-520", 800,  {8.81114470869528e-05, 3.7733253425414e-05, 4.4909524017681e-05}, {-7.16051325129026e-07, -7.65555332821054e-07, 5.88023339950319e-07}},
{"Olympus E-520 iso 1600", "OLYMPUS IMAGING CORP.",  "E-520", 1600, {0.000150903402037362, 6.14496726488485e-05, 7.8148183706356e-05}, {3.56181751137984e-06, 3.52884421017037e-06, 5.72834587473515e-06}},

  // clone of e-30 for e-600 (apparently has the same sensor)
  {"Olympus E-600 iso 100",  "OLYMPUS IMAGING CORP.", "E-600", 100,  {9.7071376303464e-06, 7.11119859126147e-06, 1.43234477170861e-05}, {-9.06560192729126e-08, -2.41125952783539e-07, -3.19385326622867e-07}},
  {"Olympus E-600 iso 125",  "OLYMPUS IMAGING CORP.", "E-600", 125,  {1.20456303941201e-05, 8.5916937354069e-06, 1.63507564674459e-05}, {-7.39375265246587e-08, -2.69554361744923e-07, -2.48410739378445e-07}},
  {"Olympus E-600 iso 160",  "OLYMPUS IMAGING CORP.", "E-600", 160,  {1.56529989102257e-05, 1.02934362650981e-05, 1.77166879017183e-05}, {-4.20365734464088e-08, -2.02996957670653e-07, 5.29589874186308e-09}},
  {"Olympus E-600 iso 200",  "OLYMPUS IMAGING CORP.", "E-600", 200,  {1.0095886562545e-05, 6.93104653792698e-06, 1.16843434000443e-05}, {-1.22201572980062e-07, -2.27477805535749e-07, 3.23770042930603e-08}},
  {"Olympus E-600 iso 250",  "OLYMPUS IMAGING CORP.", "E-600", 250,  {1.29475199645689e-05, 8.46056520489638e-06, 1.35672020706393e-05}, {-1.56424286258763e-07, -2.17640666926632e-07, 1.39913184326539e-07}},
  {"Olympus E-600 iso 320",  "OLYMPUS IMAGING CORP.", "E-600", 320,  {1.54937257192008e-05, 9.96565953037391e-06, 1.67345924078165e-05}, {-8.68535574704753e-08, -1.56806153982179e-07, 2.43131977515659e-07}},
  {"Olympus E-600 iso 400",  "OLYMPUS IMAGING CORP.", "E-600", 400,  {1.9113446058363e-05, 1.07745915338773e-05, 1.75354364481865e-05}, {-1.89298531918018e-07, 1.12954758094748e-07, 6.39061318416547e-07}},
  {"Olympus E-600 iso 500",  "OLYMPUS IMAGING CORP.", "E-600", 500,  {2.11949380361048e-05, 1.35577131775584e-05, 2.31360696860893e-05}, {7.23334285808473e-08, 1.22967405892309e-07, 7.02832916225672e-07}},
  {"Olympus E-600 iso 640",  "OLYMPUS IMAGING CORP.", "E-600", 640,  {2.73199109169886e-05, 1.67468277151008e-05, 2.95174774898525e-05}, {1.99717400841382e-07, 2.75231709016361e-07, 8.64971660648951e-07}},
  {"Olympus E-600 iso 800",  "OLYMPUS IMAGING CORP.", "E-600", 800,  {3.37153575569615e-05, 2.06893507139736e-05, 3.88227466208396e-05}, {1.6088342752527e-07, 2.65337245132019e-07, 6.41368781875726e-07}},
  {"Olympus E-600 iso 1000", "OLYMPUS IMAGING CORP.", "E-600", 1000, {4.35040657457758e-05, 2.74901261504848e-05, 4.77520441360138e-05}, {3.50283692595926e-07, 1.52239280251802e-07, 1.20426607195639e-06}},
  {"Olympus E-600 iso 1250", "OLYMPUS IMAGING CORP.", "E-600", 1250, {5.26077166899586e-05, 3.41240085546217e-05, 5.88240938452458e-05}, {8.44266815099888e-07, 5.27504382892626e-07, 1.93639030690002e-06}},
  {"Olympus E-600 iso 1600", "OLYMPUS IMAGING CORP.", "E-600", 1600, {6.57353551949384e-05, 3.65886597333215e-05, 6.54938510019982e-05}, {1.49787788804922e-06, 2.30740528605681e-06, 3.78507040181792e-06}},
  {"Olympus E-600 iso 2000", "OLYMPUS IMAGING CORP.", "E-600", 2000, {8.62141802726288e-05, 5.03284121108448e-05, 7.89194460796899e-05}, {2.44624695228374e-06, 2.29047427336703e-06, 5.80698306285845e-06}},
  {"Olympus E-600 iso 2500", "OLYMPUS IMAGING CORP.", "E-600", 2500, {0.000110342048549196, 6.48384215703911e-05, 0.000101826544077179}, {4.20346704162012e-06, 3.90138066601788e-06, 7.79212482176447e-06}},
  {"Olympus E-600 iso 3200", "OLYMPUS IMAGING CORP.", "E-600", 3200, {0.000139880356662346, 8.13578184026246e-05, 0.000152070112576575}, {5.78577431993228e-06, 6.62245038671539e-06, 9.69460041876962e-06}},

  // clone of e-30 for e-620 (apparently has the same sensor)
  {"Olympus E-620 iso 100",  "OLYMPUS IMAGING CORP.", "E-620", 100,  {9.7071376303464e-06, 7.11119859126147e-06, 1.43234477170861e-05}, {-9.06560192729126e-08, -2.41125952783539e-07, -3.19385326622867e-07}},
  {"Olympus E-620 iso 125",  "OLYMPUS IMAGING CORP.", "E-620", 125,  {1.20456303941201e-05, 8.5916937354069e-06, 1.63507564674459e-05}, {-7.39375265246587e-08, -2.69554361744923e-07, -2.48410739378445e-07}},
  {"Olympus E-620 iso 160",  "OLYMPUS IMAGING CORP.", "E-620", 160,  {1.56529989102257e-05, 1.02934362650981e-05, 1.77166879017183e-05}, {-4.20365734464088e-08, -2.02996957670653e-07, 5.29589874186308e-09}},
  {"Olympus E-620 iso 200",  "OLYMPUS IMAGING CORP.", "E-620", 200,  {1.0095886562545e-05, 6.93104653792698e-06, 1.16843434000443e-05}, {-1.22201572980062e-07, -2.27477805535749e-07, 3.23770042930603e-08}},
  {"Olympus E-620 iso 250",  "OLYMPUS IMAGING CORP.", "E-620", 250,  {1.29475199645689e-05, 8.46056520489638e-06, 1.35672020706393e-05}, {-1.56424286258763e-07, -2.17640666926632e-07, 1.39913184326539e-07}},
  {"Olympus E-620 iso 320",  "OLYMPUS IMAGING CORP.", "E-620", 320,  {1.54937257192008e-05, 9.96565953037391e-06, 1.67345924078165e-05}, {-8.68535574704753e-08, -1.56806153982179e-07, 2.43131977515659e-07}},
  {"Olympus E-620 iso 400",  "OLYMPUS IMAGING CORP.", "E-620", 400,  {1.9113446058363e-05, 1.07745915338773e-05, 1.75354364481865e-05}, {-1.89298531918018e-07, 1.12954758094748e-07, 6.39061318416547e-07}},
  {"Olympus E-620 iso 500",  "OLYMPUS IMAGING CORP.", "E-620", 500,  {2.11949380361048e-05, 1.35577131775584e-05, 2.31360696860893e-05}, {7.23334285808473e-08, 1.22967405892309e-07, 7.02832916225672e-07}},
  {"Olympus E-620 iso 640",  "OLYMPUS IMAGING CORP.", "E-620", 640,  {2.73199109169886e-05, 1.67468277151008e-05, 2.95174774898525e-05}, {1.99717400841382e-07, 2.75231709016361e-07, 8.64971660648951e-07}},
  {"Olympus E-620 iso 800",  "OLYMPUS IMAGING CORP.", "E-620", 800,  {3.37153575569615e-05, 2.06893507139736e-05, 3.88227466208396e-05}, {1.6088342752527e-07, 2.65337245132019e-07, 6.41368781875726e-07}},
  {"Olympus E-620 iso 1000", "OLYMPUS IMAGING CORP.", "E-620", 1000, {4.35040657457758e-05, 2.74901261504848e-05, 4.77520441360138e-05}, {3.50283692595926e-07, 1.52239280251802e-07, 1.20426607195639e-06}},
  {"Olympus E-620 iso 1250", "OLYMPUS IMAGING CORP.", "E-620", 1250, {5.26077166899586e-05, 3.41240085546217e-05, 5.88240938452458e-05}, {8.44266815099888e-07, 5.27504382892626e-07, 1.93639030690002e-06}},
  {"Olympus E-620 iso 1600", "OLYMPUS IMAGING CORP.", "E-620", 1600, {6.57353551949384e-05, 3.65886597333215e-05, 6.54938510019982e-05}, {1.49787788804922e-06, 2.30740528605681e-06, 3.78507040181792e-06}},
  {"Olympus E-620 iso 2000", "OLYMPUS IMAGING CORP.", "E-620", 2000, {8.62141802726288e-05, 5.03284121108448e-05, 7.89194460796899e-05}, {2.44624695228374e-06, 2.29047427336703e-06, 5.80698306285845e-06}},
  {"Olympus E-620 iso 2500", "OLYMPUS IMAGING CORP.", "E-620", 2500, {0.000110342048549196, 6.48384215703911e-05, 0.000101826544077179}, {4.20346704162012e-06, 3.90138066601788e-06, 7.79212482176447e-06}},
  {"Olympus E-620 iso 3200", "OLYMPUS IMAGING CORP.", "E-620", 3200, {0.000139880356662346, 8.13578184026246e-05, 0.000152070112576575}, {5.78577431993228e-06, 6.62245038671539e-06, 9.69460041876962e-06}},

// olympus e-m1 by steve adler
{"Olympus E-M1 iso 100",   "OLYMPUS IMAGING CORP.", "E-M1", 100,   {7.74190217788802e-06, 4.99514479968385e-06, 1.24707994540308e-05}, {9.00673204757928e-09, 6.7615625234093e-09, 3.21290513504933e-08}},
{"Olympus E-M1 iso 200",   "OLYMPUS IMAGING CORP.", "E-M1", 200,   {6.77238178962881e-06, 4.40947998595696e-06, 1.20897181686562e-05}, {1.19538393887721e-08, 9.34819487803036e-09, 1.76732068647542e-08}},
{"Olympus E-M1 iso 250",   "OLYMPUS IMAGING CORP.", "E-M1", 250,   {8.66267895478374e-06, 5.60798923757757e-06, 1.51791432704428e-05}, {1.52102952885493e-08, 1.15495033410535e-08, 2.07605594271582e-08}},
{"Olympus E-M1 iso 320",   "OLYMPUS IMAGING CORP.", "E-M1", 320,   {1.0713858700149e-05, 6.99706479831536e-06, 1.91592961573295e-05}, {1.8259943648198e-08, 1.39565930264875e-08, 2.6718571066234e-08}},
{"Olympus E-M1 iso 400",   "OLYMPUS IMAGING CORP.", "E-M1", 400,   {1.36383521557291e-05, 8.77283475595945e-06, 2.37394702882755e-05}, {2.44279164128332e-08, 1.9247486697176e-08, 3.30069772798631e-08}},
{"Olympus E-M1 iso 500",   "OLYMPUS IMAGING CORP.", "E-M1", 500,   {1.69949600260163e-05, 1.09593490350519e-05, 2.96274859745454e-05}, {3.07375991590068e-08, 2.48470813223883e-08, 4.23656097822847e-08}},
{"Olympus E-M1 iso 640",   "OLYMPUS IMAGING CORP.", "E-M1", 640,   {2.14069798926058e-05, 1.38504469072936e-05, 3.78369789899263e-05}, {3.86991510394671e-08, 3.31278810225392e-08, 5.3260647792296e-08}},
{"Olympus E-M1 iso 800",   "OLYMPUS IMAGING CORP.", "E-M1", 800,   {2.66852185305474e-05, 1.73226524383083e-05, 4.73030454880297e-05}, {5.27120834585534e-08, 4.29284998660963e-08, 7.09907402619808e-08}},
{"Olympus E-M1 iso 1000",  "OLYMPUS IMAGING CORP.", "E-M1", 1000,  {3.40225110143011e-05, 2.20731050564408e-05, 5.9213687825035e-05}, {7.11844413191646e-08, 5.68934899896266e-08, 9.18760638122351e-08}},
{"Olympus E-M1 iso 1250",  "OLYMPUS IMAGING CORP.", "E-M1", 1250,  {4.26969019934966e-05, 2.76694411620427e-05, 7.45693673999991e-05}, {8.95038834277528e-08, 7.37322491076928e-08, 1.13644936577435e-07}},
{"Olympus E-M1 iso 1600",  "OLYMPUS IMAGING CORP.", "E-M1", 1600,  {5.27923446436706e-05, 3.46016784719241e-05, 9.38528910529107e-05}, {1.18542118886949e-07, 9.97292075228335e-08, 1.48053258500048e-07}},
{"Olympus E-M1 iso 2000",  "OLYMPUS IMAGING CORP.", "E-M1", 2000,  {6.71959306123621e-05, 4.3734478378563e-05, 0.000105955198172694}, {1.56873649495158e-07, 1.36950513574431e-07, 2.67596543041763e-07}},
{"Olympus E-M1 iso 2500",  "OLYMPUS IMAGING CORP.", "E-M1", 2500,  {8.33688670758327e-05, 5.49275672348253e-05, 0.000135092029580169}, {2.10772264478323e-07, 1.84646967751446e-07, 3.64677180634245e-07}},
{"Olympus E-M1 iso 3200",  "OLYMPUS IMAGING CORP.", "E-M1", 3200,  {0.000105956351244575, 6.93957545571624e-05, 0.000166926459574278}, {3.0686492191593e-07, 2.57471009594172e-07, 5.65562286292852e-07}},
{"Olympus E-M1 iso 4000",  "OLYMPUS IMAGING CORP.", "E-M1", 4000,  {0.000132404033935657, 8.78464493477256e-05, 0.000212072057636582}, {3.79117806753487e-07, 3.54060601151263e-07, 7.56314102202783e-07}},
{"Olympus E-M1 iso 5000",  "OLYMPUS IMAGING CORP.", "E-M1", 5000,  {0.000165780183117309, 0.000109362367275444, 0.000263268223754436}, {5.61815060315291e-07, 5.0807598818653e-07, 1.05686986025883e-06}},
{"Olympus E-M1 iso 6400",  "OLYMPUS IMAGING CORP.", "E-M1", 6400,  {0.000207474595606609, 0.000138374417152766, 0.000337469523149218}, {7.72518654731912e-07, 7.57471267471607e-07, 1.45444333509722e-06}},
{"Olympus E-M1 iso 8000",  "OLYMPUS IMAGING CORP.", "E-M1", 8000,  {0.000262385609010228, 0.00017475179693112, 0.000419250122440518}, {1.25603178904549e-06, 1.13751715809676e-06, 2.24578441552896e-06}},
{"Olympus E-M1 iso 10000", "OLYMPUS IMAGING CORP.", "E-M1", 10000, {0.000328341354577138, 0.000220180402050329, 0.000510836414893689}, {1.7293264895302e-06, 1.76156760941109e-06, 3.67552325438007e-06}},
{"Olympus E-M1 iso 12800", "OLYMPUS IMAGING CORP.", "E-M1", 12800, {0.000418953490513845, 0.000280795389477785, 0.000606125421488139}, {2.72223703196694e-06, 2.59448468874635e-06, 6.86097987458425e-06}},
{"Olympus E-M1 iso 16000", "OLYMPUS IMAGING CORP.", "E-M1", 16000, {0.00052510487155853, 0.000349790365434269, 0.000644006946224191}, {4.40262943393105e-06, 4.12131468947183e-06, 1.46215956052916e-05}},
{"Olympus E-M1 iso 20000", "OLYMPUS IMAGING CORP.", "E-M1", 20000, {0.000656367280208501, 0.000441841924552818, 0.000736327793054226}, {6.02497589163941e-06, 6.31995888916609e-06, 2.26325193141484e-05}},
{"Olympus E-M1 iso 25600", "OLYMPUS IMAGING CORP.", "E-M1", 25600, {0.000835695986506021, 0.000560610641298958, 0.00077001770237218}, {1.02770271316684e-05, 9.80974699811208e-06, 4.17848241272335e-05}},

// olympus e-m10 contributed by Matthias Grocholl
{"Olympus E-M10 iso 100",   "OLYMPUS IMAGING CORP.", "E-M10", 100, {5.52970064374362e-06, 4.21304571433177e-06, 1.3324373439563e-05}, {-1.0018336612807e-08, 3.94025652121711e-09, 5.41767953489006e-08}},
{"Olympus E-M10 iso 200",   "OLYMPUS IMAGING CORP.", "E-M10", 200, {5.27885096621069e-06, 4.0311073728833e-06, 1.42791100255731e-05}, {6.18695598001519e-09, 1.09251397994329e-08, 3.56149326146519e-08}},
{"Olympus E-M10 iso 250",   "OLYMPUS IMAGING CORP.", "E-M10", 250, {6.57290921518075e-06, 5.07547535664151e-06, 1.78979830259905e-05}, {8.31234437180656e-09, 1.46571193014622e-08, 4.86555935791215e-08}},
{"Olympus E-M10 iso 320",   "OLYMPUS IMAGING CORP.", "E-M10", 320, {8.23268475282371e-06, 6.2530560960975e-06, 2.20075078774826e-05}, {1.75119623061569e-08, 2.15114605348119e-08, 6.27005903251517e-08}},
{"Olympus E-M10 iso 400",   "OLYMPUS IMAGING CORP.", "E-M10", 400, {1.02740831329348e-05, 7.76891031908947e-06, 2.68189177808402e-05}, {1.22779439826579e-08, 1.95798802723303e-08, 6.30419053993293e-08}},
{"Olympus E-M10 iso 500",   "OLYMPUS IMAGING CORP.", "E-M10", 500, {1.27487664407172e-05, 9.76629901277535e-06, 3.3985537553202e-05}, {1.5728707654148e-08, 2.4892043827518e-08, 8.42717425085516e-08}},
{"Olympus E-M10 iso 640",   "OLYMPUS IMAGING CORP.", "E-M10", 640, {1.54625801809514e-05, 1.18262634819645e-05, 4.1076604362959e-05}, {3.01529318840337e-08, 3.91150274835145e-08, 1.07246457149077e-07}},
{"Olympus E-M10 iso 800",   "OLYMPUS IMAGING CORP.", "E-M10", 800, {1.97274205750951e-05, 1.49300120842881e-05, 5.16290589584989e-05}, {4.37640260057681e-08, 5.41752067511797e-08, 1.35248326892837e-07}},
{"Olympus E-M10 iso 1000",  "OLYMPUS IMAGING CORP.", "E-M10", 1000, {2.48940025499384e-05, 1.87371396115374e-05, 6.49906719408842e-05}, {7.19304633073414e-08, 7.78212582658316e-08, 1.83956219830277e-07}},
{"Olympus E-M10 iso 1250",  "OLYMPUS IMAGING CORP.", "E-M10", 1250, {3.11531205482021e-05, 2.34524754619465e-05, 8.06221647125521e-05}, {9.8508623771691e-08, 1.10455401926812e-07, 2.45487042736294e-07}},
{"Olympus E-M10 iso 1600",  "OLYMPUS IMAGING CORP.", "E-M10", 1600, {3.95054702261325e-05, 2.98932019899239e-05, 0.000102609631032762}, {1.46316418785828e-07, 1.53754825291345e-07, 3.24060815486051e-07}},
{"Olympus E-M10 iso 2000",  "OLYMPUS IMAGING CORP.", "E-M10", 2000, {5.04587681958752e-05, 3.81244119678433e-05, 0.000127187706430444}, {2.17831131568732e-07, 2.08357901803733e-07, 3.96892396874616e-07}},
{"Olympus E-M10 iso 2500",  "OLYMPUS IMAGING CORP.", "E-M10", 2500, {6.33700113749826e-05, 4.7917032118296e-05, 0.000162546352417275}, {3.124312513936e-07, 2.95890584688271e-07, 5.54323877042126e-07}},
{"Olympus E-M10 iso 3200",  "OLYMPUS IMAGING CORP.", "E-M10", 3200, {8.04101037092208e-05, 6.13047283878647e-05, 0.000208299853424575}, {4.40283551285417e-07, 4.30360045581258e-07, 7.66033227736199e-07}},
{"Olympus E-M10 iso 4000",  "OLYMPUS IMAGING CORP.", "E-M10", 4000, {0.000103501977642578, 7.87206458974012e-05, 0.000263518813373455}, {6.64706723985038e-07, 6.18853804432904e-07, 1.01980789494233e-06}},
{"Olympus E-M10 iso 5000",  "OLYMPUS IMAGING CORP.", "E-M10", 5000, {0.000135298740644269, 0.00010327680825576, 0.000343875628667601}, {9.38764705489347e-07, 8.50656073268359e-07, 1.29256621272383e-06}},
{"Olympus E-M10 iso 6400",  "OLYMPUS IMAGING CORP.", "E-M10", 6400, {0.00017190771276232, 0.000132265407945499, 0.00043138635174981}, {1.37896879988534e-06, 1.27102449268754e-06, 1.8298086084211e-06}},
{"Olympus E-M10 iso 8000",  "OLYMPUS IMAGING CORP.", "E-M10", 8000, {0.000229154996282747, 0.000178545440094271, 0.000560575016755586}, {1.87947872573366e-06, 1.73392014464826e-06, 2.41457431900873e-06}},
{"Olympus E-M10 iso 10000", "OLYMPUS IMAGING CORP.", "E-M10", 10000, {0.000303399551410581, 0.000239417970052446, 0.000710253470589204}, {2.6600888039988e-06, 2.40037443577335e-06, 3.17436600456475e-06}},
{"Olympus E-M10 iso 12800", "OLYMPUS IMAGING CORP.", "E-M10", 12800, {0.000379720435342194, 0.000296923127994401, 0.000859041761073275}, {4.1499308254859e-06, 3.89183522978768e-06, 5.00589858910225e-06}},
{"Olympus E-M10 iso 16000", "OLYMPUS IMAGING CORP.", "E-M10", 16000, {0.000512909615826058, 0.000434366865123056, 0.00112314725009763}, {5.17462665366535e-06, 4.59518057515397e-06, 5.82748593792202e-06}},
{"Olympus E-M10 iso 20000", "OLYMPUS IMAGING CORP.", "E-M10", 20000, {0.000676685859629288, 0.000592958370763385, 0.00141518451822841}, {7.05261752822402e-06, 6.04913098617647e-06, 7.68860701559556e-06}},
{"Olympus E-M10 iso 25600", "OLYMPUS IMAGING CORP.", "E-M10", 25600, {0.000869807091934271, 0.000821977485910616, 0.00175285520277303}, {9.31076597964603e-06, 7.30041023706674e-06, 9.61829465997919e-06}},
  // additional data by mikko ruohola
// {"E-M10 iso 100",       "OLYMPUS IMAGING CORP.",      "E-M10",              100,         {7.98098524793963e-06, 3.58321114711828e-06, 6.2275048229058e-06}, {1.0711134062885e-08, 1.60991127527557e-09, 9.68046558068677e-09}},
// {"E-M10 iso 200",       "OLYMPUS IMAGING CORP.",      "E-M10",              200,         {7.75038814963806e-06, 3.43925205954935e-06, 6.05321038477629e-06}, {1.30885172875239e-08, 6.91413956191518e-09, 1.29833801841222e-08}},
// {"E-M10 iso 250",       "OLYMPUS IMAGING CORP.",      "E-M10",              250,         {9.70922384576952e-06, 4.35499077224943e-06, 7.59421397262083e-06}, {1.7816135610543e-08, 8.50915785936426e-09, 1.42795112605054e-08}},
// {"E-M10 iso 320",       "OLYMPUS IMAGING CORP.",      "E-M10",              320,         {1.19734619688467e-05, 5.37491488545513e-06, 9.39479277307395e-06}, {2.37168191267104e-08, 1.27921259692802e-08, 2.34972328183344e-08}},
// {"E-M10 iso 400",       "OLYMPUS IMAGING CORP.",      "E-M10",              400,         {1.50672514636552e-05, 6.78358225121592e-06, 1.17974333325514e-05}, {2.17937129087455e-08, 1.10426643420075e-08, 2.01440187625277e-08}},
// {"E-M10 iso 500",       "OLYMPUS IMAGING CORP.",      "E-M10",              500,         {2.45425465681807e-05, 1.09636621504231e-05, 1.92046595248163e-05}, {4.71017781800194e-08, 2.47097187192792e-08, 4.64156246954803e-08}},
// {"E-M10 iso 640",       "OLYMPUS IMAGING CORP.",      "E-M10",              640,         {2.37427949461541e-05, 1.06354613026899e-05, 1.86745022545958e-05}, {3.93274285886345e-08, 2.13320463423718e-08, 3.7772261684897e-08}},
// {"E-M10 iso 800",       "OLYMPUS IMAGING CORP.",      "E-M10",              800,         {2.98695119705811e-05, 1.3334995487512e-05, 2.35142869968519e-05}, {5.53808280343389e-08, 2.91245442424457e-08, 5.3082591862793e-08}},
// {"E-M10 iso 1000",       "OLYMPUS IMAGING CORP.",      "E-M10",              1000,         {3.70736498845211e-05, 1.66868468315352e-05, 2.91783058878668e-05}, {7.71592671559755e-08, 4.12900214644293e-08, 7.41130612547522e-08}},
// {"E-M10 iso 1250",       "OLYMPUS IMAGING CORP.",      "E-M10",              1250,         {4.68810477669831e-05, 2.09477803134772e-05, 3.67976942090768e-05}, {1.01320746923039e-07, 6.36979465877723e-08, 1.02286039099308e-07}},
// {"E-M10 iso 1600",       "OLYMPUS IMAGING CORP.",      "E-M10",              1600,         {5.85805786774318e-05, 2.61114547803644e-05, 4.59914046668682e-05}, {1.47007433013208e-07, 8.75061736444635e-08, 1.44470119722315e-07}},
// {"E-M10 iso 2000",       "OLYMPUS IMAGING CORP.",      "E-M10",              2000,         {9.63919931436932e-05, 4.31939750600662e-05, 7.58917648215175e-05}, {2.85881559454424e-07, 2.05791597500946e-07, 3.15305292416101e-07}},
// {"E-M10 iso 2500",       "OLYMPUS IMAGING CORP.",      "E-M10",              2500,         {0.00012151540572364, 5.45468962480455e-05, 9.59982727198973e-05}, {4.1192266652444e-07, 2.97659203438895e-07, 4.38135866030707e-07}},
// {"E-M10 iso 3200",       "OLYMPUS IMAGING CORP.",      "E-M10",              3200,         {0.000131445347568587, 5.92963461824783e-05, 0.000104164507188931}, {4.43835490401728e-07, 3.15304230377292e-07, 4.57729289743145e-07}},
// {"E-M10 iso 4000",       "OLYMPUS IMAGING CORP.",      "E-M10",              4000,         {0.000147858416564552, 6.64980260474526e-05, 0.000117818823899349}, {5.05263998467631e-07, 3.87625046218177e-07, 5.26313555699351e-07}},
// {"E-M10 iso 5000",       "OLYMPUS IMAGING CORP.",      "E-M10",              5000,         {0.000187150495322929, 8.42575642766885e-05, 0.000149236362222044}, {7.25382591346563e-07, 5.5507508906693e-07, 7.351837224222e-07}},
// {"E-M10 iso 6400",       "OLYMPUS IMAGING CORP.",      "E-M10",              6400,         {0.000239140617273916, 0.000106937051983151, 0.000189717098192844}, {1.03066526060702e-06, 8.03253533072218e-07, 1.06026405307289e-06}},
// {"E-M10 iso 8000",       "OLYMPUS IMAGING CORP.",      "E-M10",              8000,         {0.000307047067380873, 0.000140061883142901, 0.000246044958130282}, {1.3984351220959e-06, 1.1295408660182e-06, 1.44648138492605e-06}},
// {"E-M10 iso 10000",       "OLYMPUS IMAGING CORP.",      "E-M10",              10000,         {0.000395738804196794, 0.000182084194975288, 0.000322821221931731}, {1.95879058604654e-06, 1.59043253916097e-06, 2.02785040861988e-06}},
// {"E-M10 iso 12800",       "OLYMPUS IMAGING CORP.",      "E-M10",              12800,         {0.000511102953641101, 0.000239894807973159, 0.000417705896112809}, {2.74898560580191e-06, 2.19689935241208e-06, 2.85791031067042e-06}},
// {"E-M10 iso 16000",       "OLYMPUS IMAGING CORP.",      "E-M10",              16000,         {0.00066167522624143, 0.000325184630933552, 0.000556799880314662}, {3.72192070337355e-06, 2.98815890324798e-06, 3.85996731877265e-06}},
// {"E-M10 iso 20000",       "OLYMPUS IMAGING CORP.",      "E-M10",              20000,         {0.000842601940758084, 0.000428148044468382, 0.000722441342166263}, {5.15164072818708e-06, 4.24102065540232e-06, 5.32966026522801e-06}},
// {"E-M10 iso 25600",       "OLYMPUS IMAGING CORP.",      "E-M10",              25600,         {0.00105032860475424, 0.000566752962419451, 0.000920807652403111}, {6.7300655062615e-06, 5.55906354084577e-06, 6.97320979108275e-06}},

// olympus omd-e5 contributed by matt feifarek
  {"Olympus E-M5 iso 200",   "OLYMPUS IMAGING CORP.", "E-M5", 200,   {3.99985740365396e-06, 3.99985740365396e-06, 3.99985740365396e-06}, {-1.30374418360161e-07, -1.30374418360161e-07, -1.30374418360161e-07}},
  {"Olympus E-M5 iso 400",   "OLYMPUS IMAGING CORP.", "E-M5", 400,   {7.17225384650838e-06, 7.17225384650838e-06, 7.17225384650838e-06}, {-1.15696430190777e-07, -1.15696430190777e-07, -1.15696430190777e-07}},
  {"Olympus E-M5 iso 800",   "OLYMPUS IMAGING CORP.", "E-M5", 800,   {1.39777063155836e-05, 1.39777063155836e-05, 1.39777063155836e-05}, {-2.42337675961104e-07, -2.42337675961104e-07, -2.42337675961104e-07}},
  {"Olympus E-M5 iso 1600",  "OLYMPUS IMAGING CORP.", "E-M5", 1600,  {2.64695429520035e-05, 2.64695429520035e-05, 2.64695429520035e-05}, {-1.48010933567677e-07, -1.48010933567677e-07, -1.48010933567677e-07}},
  {"Olympus E-M5 iso 3200",  "OLYMPUS IMAGING CORP.", "E-M5", 3200,  {5.05148160339341e-05, 5.05148160339341e-05, 5.05148160339341e-05}, {1.86795334545125e-07, 1.86795334545125e-07, 1.86795334545125e-07}},
  {"Olympus E-M5 iso 6400",  "OLYMPUS IMAGING CORP.", "E-M5", 6400,  {9.86709410704645e-05, 9.86709410704645e-05, 9.86709410704645e-05}, {1.04469073600282e-06, 1.04469073600282e-06, 1.04469073600282e-06}},
  {"Olympus E-M5 iso 12800", "OLYMPUS IMAGING CORP.", "E-M5", 12800, {0.000197400445449016, 0.000197400445449016, 0.000197400445449016}, {3.8799676106741e-06, 3.8799676106741e-06, 3.8799676106741e-06}},
  {"Olympus E-M5 iso 25600", "OLYMPUS IMAGING CORP.", "E-M5", 25600, {0.000398880680356815, 0.000398880680356815, 0.000398880680356815}, {1.52202902674769e-05, 1.52202902674769e-05, 1.52202902674769e-05}},

// olympus e-m10 by polarfox
{"E-M10 iso 100",   "OLYMPUS IMAGING CORP.", "E-M10", 100,   {7.98098524793963e-06, 3.58321114711828e-06, 6.2275048229058e-06}, {1.0711134062885e-08, 1.60991127527557e-09, 9.68046558068677e-09}},
{"E-M10 iso 200",   "OLYMPUS IMAGING CORP.", "E-M10", 200,   {7.75038814963806e-06, 3.43925205954935e-06, 6.05321038477629e-06}, {1.30885172875239e-08, 6.91413956191518e-09, 1.29833801841222e-08}},
{"E-M10 iso 250",   "OLYMPUS IMAGING CORP.", "E-M10", 250,   {9.70922384576952e-06, 4.35499077224943e-06, 7.59421397262083e-06}, {1.7816135610543e-08, 8.50915785936426e-09, 1.42795112605054e-08}},
{"E-M10 iso 320",   "OLYMPUS IMAGING CORP.", "E-M10", 320,   {1.19734619688467e-05, 5.37491488545513e-06, 9.39479277307395e-06}, {2.37168191267104e-08, 1.27921259692802e-08, 2.34972328183344e-08}},
{"E-M10 iso 400",   "OLYMPUS IMAGING CORP.", "E-M10", 400,   {1.50672514636552e-05, 6.78358225121592e-06, 1.17974333325514e-05}, {2.17937129087455e-08, 1.10426643420075e-08, 2.01440187625277e-08}},
{"E-M10 iso 500",   "OLYMPUS IMAGING CORP.", "E-M10", 500,   {2.45425465681807e-05, 1.09636621504231e-05, 1.92046595248163e-05}, {4.71017781800194e-08, 2.47097187192792e-08, 4.64156246954803e-08}},
{"E-M10 iso 640",   "OLYMPUS IMAGING CORP.", "E-M10", 640,   {2.37427949461541e-05, 1.06354613026899e-05, 1.86745022545958e-05}, {3.93274285886345e-08, 2.13320463423718e-08, 3.7772261684897e-08}},
{"E-M10 iso 800",   "OLYMPUS IMAGING CORP.", "E-M10", 800,   {2.98695119705811e-05, 1.3334995487512e-05, 2.35142869968519e-05}, {5.53808280343389e-08, 2.91245442424457e-08, 5.3082591862793e-08}},
{"E-M10 iso 1000",  "OLYMPUS IMAGING CORP.", "E-M10", 1000,  {3.70736498845211e-05, 1.66868468315352e-05, 2.91783058878668e-05}, {7.71592671559755e-08, 4.12900214644293e-08, 7.41130612547522e-08}},
{"E-M10 iso 1250",  "OLYMPUS IMAGING CORP.", "E-M10", 1250,  {4.68810477669831e-05, 2.09477803134772e-05, 3.67976942090768e-05}, {1.01320746923039e-07, 6.36979465877723e-08, 1.02286039099308e-07}},
{"E-M10 iso 1600",  "OLYMPUS IMAGING CORP.", "E-M10", 1600,  {5.85805786774318e-05, 2.61114547803644e-05, 4.59914046668682e-05}, {1.47007433013208e-07, 8.75061736444635e-08, 1.44470119722315e-07}},
{"E-M10 iso 2000",  "OLYMPUS IMAGING CORP.", "E-M10", 2000,  {9.63919931436932e-05, 4.31939750600662e-05, 7.58917648215175e-05}, {2.85881559454424e-07, 2.05791597500946e-07, 3.15305292416101e-07}},
{"E-M10 iso 2500",  "OLYMPUS IMAGING CORP.", "E-M10", 2500,  {0.00012151540572364, 5.45468962480455e-05, 9.59982727198973e-05}, {4.1192266652444e-07, 2.97659203438895e-07, 4.38135866030707e-07}},
{"E-M10 iso 3200",  "OLYMPUS IMAGING CORP.", "E-M10", 3200,  {0.000131445347568587, 5.92963461824783e-05, 0.000104164507188931}, {4.43835490401728e-07, 3.15304230377292e-07, 4.57729289743145e-07}},
{"E-M10 iso 4000",  "OLYMPUS IMAGING CORP.", "E-M10", 4000,  {0.000147858416564552, 6.64980260474526e-05, 0.000117818823899349}, {5.05263998467631e-07, 3.87625046218177e-07, 5.26313555699351e-07}},
{"E-M10 iso 5000",  "OLYMPUS IMAGING CORP.", "E-M10", 5000,  {0.000187150495322929, 8.42575642766885e-05, 0.000149236362222044}, {7.25382591346563e-07, 5.5507508906693e-07, 7.351837224222e-07}},
{"E-M10 iso 6400",  "OLYMPUS IMAGING CORP.", "E-M10", 6400,  {0.000239140617273916, 0.000106937051983151, 0.000189717098192844}, {1.03066526060702e-06, 8.03253533072218e-07, 1.06026405307289e-06}},
{"E-M10 iso 8000",  "OLYMPUS IMAGING CORP.", "E-M10", 8000,  {0.000307047067380873, 0.000140061883142901, 0.000246044958130282}, {1.3984351220959e-06, 1.1295408660182e-06, 1.44648138492605e-06}},
{"E-M10 iso 10000", "OLYMPUS IMAGING CORP.", "E-M10", 10000, {0.000395738804196794, 0.000182084194975288, 0.000322821221931731}, {1.95879058604654e-06, 1.59043253916097e-06, 2.02785040861988e-06}},
{"E-M10 iso 12800", "OLYMPUS IMAGING CORP.", "E-M10", 12800, {0.000511102953641101, 0.000239894807973159, 0.000417705896112809}, {2.74898560580191e-06, 2.19689935241208e-06, 2.85791031067042e-06}},
{"E-M10 iso 16000", "OLYMPUS IMAGING CORP.", "E-M10", 16000, {0.00066167522624143, 0.000325184630933552, 0.000556799880314662}, {3.72192070337355e-06, 2.98815890324798e-06, 3.85996731877265e-06}},
{"E-M10 iso 20000", "OLYMPUS IMAGING CORP.", "E-M10", 20000, {0.000842601940758084, 0.000428148044468382, 0.000722441342166263}, {5.15164072818708e-06, 4.24102065540232e-06, 5.32966026522801e-06}},
{"E-M10 iso 25600", "OLYMPUS IMAGING CORP.", "E-M10", 25600, {0.00105032860475424, 0.000566752962419451, 0.000920807652403111}, {6.7300655062615e-06, 5.55906354084577e-06, 6.97320979108275e-06}},

// olympus e-pl1 contributed by José Carlos R. Casimiro
  {"Olympus E-PL1 iso 100",  "OLYMPUS IMAGING CORP.", "E-PL1", 100,  {6.1332646664107e-06, 6.1332646664107e-06, 6.1332646664107e-06}, {-1.70427018703982e-07, -1.70427018703982e-07, -1.70427018703982e-07}},
  {"Olympus E-PL1 iso 200",  "OLYMPUS IMAGING CORP.", "E-PL1", 200,  {6.17771627349177e-06, 6.17771627349177e-06, 6.17771627349177e-06}, {-1.94243887646926e-07, -1.94243887646926e-07, -1.94243887646926e-07}},
  {"Olympus E-PL1 iso 400",  "OLYMPUS IMAGING CORP.", "E-PL1", 400,  {1.00980942726634e-05, 1.00980942726634e-05, 1.00980942726634e-05}, {-8.10099319236512e-08, -8.10099319236512e-08, -8.10099319236512e-08}},
  {"Olympus E-PL1 iso 800",  "OLYMPUS IMAGING CORP.", "E-PL1", 800,  {1.78191571180193e-05, 1.78191571180193e-05, 1.78191571180193e-05}, {3.41992783976417e-07, 3.41992783976417e-07, 3.41992783976417e-07}},
  {"Olympus E-PL1 iso 1600", "OLYMPUS IMAGING CORP.", "E-PL1", 1600, {1.78191571180193e-05, 1.78191571180193e-05, 1.78191571180193e-05}, {3.41992783976417e-07, 3.41992783976417e-07, 3.41992783976417e-07}},
  {"Olympus E-PL1 iso 3200", "OLYMPUS IMAGING CORP.", "E-PL1", 3200, {6.38579880360713e-05, 6.38579880360713e-05, 6.38579880360713e-05}, {6.71632186716273e-06, 6.71632186716273e-06, 6.71632186716273e-06}},

// olympus e-pl2 contributed by Urs Schutz
{"E-PL2 iso 200",       "OLYMPUS IMAGING CORP.",      "E-PL2",              200,         {7.17859172533767e-06, 5.93229441990318e-06, 9.11380252816966e-06}, {7.1028372829633e-08, 4.96051534630654e-08, 6.24619122436368e-08}},
{"E-PL2 iso 400",       "OLYMPUS IMAGING CORP.",      "E-PL2",              400,         {1.40499152872468e-05, 9.85356529568553e-06, 1.56202315056903e-05}, {1.38053773647176e-07, 1.05705954559186e-07, 1.30458775902424e-07}},
{"E-PL2 iso 800",       "OLYMPUS IMAGING CORP.",      "E-PL2",              800,         {2.95547678982877e-05, 1.81224982085093e-05, 2.88031054723477e-05}, {3.16852143058334e-07, 2.56433741339548e-07, 3.19222314494574e-07}},
{"E-PL2 iso 1600",       "OLYMPUS IMAGING CORP.",      "E-PL2",              1600,         {6.85619743579416e-05, 3.40100249948929e-05, 5.45722211445395e-05}, {8.86555121241657e-07, 7.76911509545224e-07, 9.41808653334525e-07}},
{"E-PL2 iso 3200",       "OLYMPUS IMAGING CORP.",      "E-PL2",              3200,         {0.000183240622375314, 7.26925410940567e-05, 0.000116929049093393}, {2.40069611217306e-06, 2.06977539048519e-06, 2.62820526272223e-06}},
{"E-PL2 iso 6400",       "OLYMPUS IMAGING CORP.",      "E-PL2",              6400,         {0.000423209946241346, 0.000167041599646668, 0.000379781926017424}, {8.00812348699607e-06, 7.10536338988602e-06, 8.14090508892091e-06}},

// olympus e-pl5 contributed by Rico Wendrock
  {"Olympus E-PL5 iso 200",   "OLYMPUS IMAGING CORP.", "E-PL5", 200,   {7.79970060482639e-06, 3.63384384214904e-06, 7.1645113170689e-06}, {-5.02938982974519e-08, -7.56468563617011e-08, -9.14079087378575e-08}},
  {"Olympus E-PL5 iso 400",   "OLYMPUS IMAGING CORP.", "E-PL5", 400,   {1.4999803518458e-05, 6.8276939391256e-06, 1.33829035510533e-05}, {-8.83032855413638e-08, -9.10894719157746e-08, -1.01323814916069e-07}},
  {"Olympus E-PL5 iso 800",   "OLYMPUS IMAGING CORP.", "E-PL5", 800,   {2.74928553733738e-05, 1.28641267251855e-05, 2.56303896886474e-05}, {-6.9562752523189e-09, -1.38235507501577e-07, -2.0860473342068e-07}},
  {"Olympus E-PL5 iso 1600",  "OLYMPUS IMAGING CORP.", "E-PL5", 1600,  {5.47203320704907e-05, 2.51280509396047e-05, 4.82747612414445e-05}, {-3.79091689460292e-08, -1.52131363393638e-07, -3.00603221025983e-09}},
  {"Olympus E-PL5 iso 3200",  "OLYMPUS IMAGING CORP.", "E-PL5", 3200,  {0.000106643631884172, 4.97588454533551e-05, 9.59703445329939e-05}, {6.49911297621869e-07, 1.6286308250972e-08, 2.90434423042749e-07}},
  {"Olympus E-PL5 iso 6400",  "OLYMPUS IMAGING CORP.", "E-PL5", 6400,  {0.000215450848195064, 9.7930351516024e-05, 0.000184661069922989}, {1.71005371680095e-06, 1.09231454296802e-06, 2.34135031395419e-06}},
  {"Olympus E-PL5 iso 12800", "OLYMPUS IMAGING CORP.", "E-PL5", 12800, {0.000441433341496105, 0.0001966656707969, 0.00034298841525635}, {3.92232457779717e-06, 3.29338684721452e-06, 1.07887638216722e-05}},
  {"Olympus E-PL5 iso 25600", "OLYMPUS IMAGING CORP.", "E-PL5", 25600, {0.000846529735950616, 0.000405001875141275, 0.000664127243688802}, {1.64814458973157e-05, 8.79583800572565e-06, 3.54639433956789e-05}},

// olympus xz-1 contributed by Jon Leighton
  {"Olympus XZ-1 iso 100",       "OLYMPUS IMAGING CORP.",      "XZ-1",              100,         {1.83766236289818e-05, 1.63573022995415e-05, 0.0}, {-3.89959841285073e-08, -2.52312052596704e-07, 0.0}},
  {"Olympus XZ-1 iso 125",       "OLYMPUS IMAGING CORP.",      "XZ-1",              125,         {2.38239110998373e-05, 1.9568277789284e-05, 0.0}, {-9.82215085156375e-08, -1.71318323182518e-07, 0.0}},
  {"Olympus XZ-1 iso 160",       "OLYMPUS IMAGING CORP.",      "XZ-1",              160,         {2.80248218781261e-05, 2.51225807160481e-05, 0.0}, {1.42336086415929e-07, -3.60139008411238e-07, 0.0}},
  {"Olympus XZ-1 iso 200",       "OLYMPUS IMAGING CORP.",      "XZ-1",              200,         {3.4753642963341e-05, 2.99900287159675e-05, 0.0}, {1.80261462541105e-07, -1.98492009208341e-07, 0.0}},
  {"Olympus XZ-1 iso 250",       "OLYMPUS IMAGING CORP.",      "XZ-1",              250,         {4.40485608896222e-05, 3.75025448175385e-05, 0.0}, {3.42798476385059e-07, -1.40107657305746e-07, 0.0}},
  {"Olympus XZ-1 iso 320",       "OLYMPUS IMAGING CORP.",      "XZ-1",              320,         {5.42572515624067e-05, 4.55785617218092e-05, 0.0}, {7.76802872798857e-07, 4.56796576836697e-07, 0.0}},
  {"Olympus XZ-1 iso 400",       "OLYMPUS IMAGING CORP.",      "XZ-1",              400,         {6.78973031532625e-05, 5.57161952410561e-05, 0.0}, {1.09875757081419e-06, 6.56044862084029e-07, 0.0}},
  {"Olympus XZ-1 iso 500",       "OLYMPUS IMAGING CORP.",      "XZ-1",              500,         {8.67865196816675e-05, 7.35915267390703e-05, 0.0}, {1.7780571783433e-06, 6.94009735906417e-07, 0.0}},
  {"Olympus XZ-1 iso 640",       "OLYMPUS IMAGING CORP.",      "XZ-1",              640,         {0.000107948055053395, 8.78480964976015e-05, 0.0}, {3.07334462036648e-06, 2.32173769971444e-06, 0.0}},
  {"Olympus XZ-1 iso 640",       "OLYMPUS IMAGING CORP.",      "XZ-1",              640,         {0.000115666451167255, 8.79910775745639e-05, 0.0}, {2.32816475327147e-06, 2.62260860917912e-06, 0.0}},
  {"Olympus XZ-1 iso 800",       "OLYMPUS IMAGING CORP.",      "XZ-1",              800,         {0.000138606101856196, 0.000112818034387658, 0.0}, {3.88836666572533e-06, 2.73534714120273e-06, 0.0}},
  {"Olympus XZ-1 iso 1000",       "OLYMPUS IMAGING CORP.",      "XZ-1",              1000,         {0.000168392509317166, 0.000137061820325419, 0.0}, {6.71360787892347e-06, 5.75932403580755e-06, 0.0}},
  {"Olympus XZ-1 iso 1250",       "OLYMPUS IMAGING CORP.",      "XZ-1",              1250,         {0.000213318677263629, 0.000174914454458554, 0.0}, {1.08415657147076e-05, 8.77265438670263e-06, 0.0}},
  {"Olympus XZ-1 iso 1600",       "OLYMPUS IMAGING CORP.",      "XZ-1",              1600,         {0.000268805862801649, 0.000223199504420489, 0.0}, {1.59651556700822e-05, 1.20083622898958e-05, 0.0}},
  {"Olympus XZ-1 iso 2000",       "OLYMPUS IMAGING CORP.",      "XZ-1",              2000,         {0.000369344229105508, 0.000286243613749221, 0.0}, {2.27543190197377e-05, 1.9272330027282e-05, 0.0}},
  {"Olympus XZ-1 iso 2500",       "OLYMPUS IMAGING CORP.",      "XZ-1",              2500,         {0.000483275150692309, 0.000339846864457883, 0.0}, {3.40270058080847e-05, 3.79127786774278e-05, 0.0}},
  {"Olympus XZ-1 iso 3200",       "OLYMPUS IMAGING CORP.",      "XZ-1",              3200,         {0.000628830722946185, 0.000462429807371774, 0.0}, {5.13296563995064e-05, 5.32941975865594e-05, 0.0}},
  {"Olympus XZ-1 iso 4000",       "OLYMPUS IMAGING CORP.",      "XZ-1",              4000,         {0.000893948316112433, 0.000558795420317466, 0.0}, {6.89765685293738e-05, 9.09575924622249e-05, 0.0}},
  {"Olympus XZ-1 iso 4000",       "OLYMPUS IMAGING CORP.",      "XZ-1",              4000,         {0.000910248198849545, 0.000571593442914448, 0.0}, {6.61337165881163e-05, 8.77694457678316e-05, 0.0}},
  {"Olympus XZ-1 iso 5000",       "OLYMPUS IMAGING CORP.",      "XZ-1",              5000,         {0.00118009634220889, 0.000756956088920663, 0.0}, {9.78648450190991e-05, 0.000128988641601859, 0.0}},
  {"Olympus XZ-1 iso 6400",       "OLYMPUS IMAGING CORP.",      "XZ-1",              6400,         {0.00162783578394172, 0.000972999125734296, 0.0}, {0.000134937899527169, 0.000207420763341138, 0.0}},

// olympus xz-2 contributed by paul vasak
  {"Olympus XZ-2 iso 100",   "OLYMPUS IMAGING CORP.", "XZ-2", 100,   {2.40572717646829e-05, 1.0761423277282e-05, 1.97524686213407e-05}, {-2.17117304108971e-07, -2.71502472738537e-07, -3.10569744739568e-07}},
  {"Olympus XZ-2 iso 125",   "OLYMPUS IMAGING CORP.", "XZ-2", 125,   {2.8055377050673e-05, 1.25232691759621e-05, 2.32452573617515e-05}, {-2.1669622915719e-07, -2.77993682832804e-07, -3.40749313948231e-07}},
  {"Olympus XZ-2 iso 160",   "OLYMPUS IMAGING CORP.", "XZ-2", 160,   {3.34820832620415e-05, 1.47864941695464e-05, 2.74253403355693e-05}, {-2.003726840712e-07, -2.49351115951388e-07, -3.27673587204076e-07}},
  {"Olympus XZ-2 iso 200",   "OLYMPUS IMAGING CORP.", "XZ-2", 200,   {4.00171596741048e-05, 1.7791515828903e-05, 3.29743920328994e-05}, {-2.03721754295421e-07, -2.78457119942867e-07, -3.15684478424174e-07}},
  {"Olympus XZ-2 iso 250",   "OLYMPUS IMAGING CORP.", "XZ-2", 250,   {4.9105343077648e-05, 2.15550279725676e-05, 3.94790545871038e-05}, {-2.31244510292908e-07, -2.49667732595434e-07, -2.48857328895779e-07}},
  {"Olympus XZ-2 iso 320",   "OLYMPUS IMAGING CORP.", "XZ-2", 320,   {6.27967164915158e-05, 2.79051303456383e-05, 5.09870970240145e-05}, {-3.73990489795218e-07, -4.44767926637302e-07, -4.53157116930979e-07}},
  {"Olympus XZ-2 iso 400",   "OLYMPUS IMAGING CORP.", "XZ-2", 400,   {7.52451535029224e-05, 3.34848079302428e-05, 6.13592542855367e-05}, {-2.45690363017741e-07, -3.81518790354131e-07, -3.70813240258755e-07}},
  {"Olympus XZ-2 iso 500",   "OLYMPUS IMAGING CORP.", "XZ-2", 500,   {9.44170507840627e-05, 4.07619978878895e-05, 7.4918629738926e-05}, {-3.25007764575769e-07, -2.2208493938833e-07, -2.18462269168774e-07}},
  {"Olympus XZ-2 iso 640",   "OLYMPUS IMAGING CORP.", "XZ-2", 640,   {0.000117273569302217, 5.13388230620496e-05, 9.47504111613659e-05}, {-1.19007929781288e-07, -2.06879341002184e-07, -1.32117677760119e-07}},
  {"Olympus XZ-2 iso 800",   "OLYMPUS IMAGING CORP.", "XZ-2", 800,   {0.000144950226065977, 6.2922109719896e-05, 0.000117197841020932}, {4.75248346871406e-08, 6.13937095516588e-08, -8.68434797465609e-09}},
  {"Olympus XZ-2 iso 1000",  "OLYMPUS IMAGING CORP.", "XZ-2", 1000,  {0.000177732081840999, 7.73594848255931e-05, 0.000143300635318154}, {2.72164508001976e-07, 2.02918542791621e-07, 1.82406448606511e-07}},
  {"Olympus XZ-2 iso 1250",  "OLYMPUS IMAGING CORP.", "XZ-2", 1250,  {0.000232393831901221, 0.00010110220671143, 0.000182896609930416}, {-1.61747724232926e-07, 8.88899253902048e-08, 7.61068378324026e-07}},
  {"Olympus XZ-2 iso 1600",  "OLYMPUS IMAGING CORP.", "XZ-2", 1600,  {0.00029278329377039, 0.000124926921593667, 0.000226419083253069}, {-5.32370023033149e-07, 6.30434038881838e-07, 1.69180092183163e-06}},
  {"Olympus XZ-2 iso 2000",  "OLYMPUS IMAGING CORP.", "XZ-2", 2000,  {0.000361628298291688, 0.000150610870952321, 0.000275145464770079}, {-8.83309380299229e-07, 1.71672339500525e-06, 2.47246313860634e-06}},
  {"Olympus XZ-2 iso 2500",  "OLYMPUS IMAGING CORP.", "XZ-2", 2500,  {0.000444200994182697, 0.000194987596284559, 0.000354446815588185}, {3.54129018036269e-07, 1.91752057746292e-06, 4.58900738772169e-06}},
  {"Olympus XZ-2 iso 3200",  "OLYMPUS IMAGING CORP.", "XZ-2", 3200,  {0.000552578834360541, 0.000242257926239312, 0.000446527636885845}, {1.47565912840644e-06, 3.82547853332068e-06, 6.46243051363692e-06}},
  {"Olympus XZ-2 iso 4000",  "OLYMPUS IMAGING CORP.", "XZ-2", 4000,  {0.00069751196434821, 0.000306076689590648, 0.000541733068328942}, {2.03367894661059e-06, 4.67266793137715e-06, 1.01691519707471e-05}},
  {"Olympus XZ-2 iso 5000",  "OLYMPUS IMAGING CORP.", "XZ-2", 5000,  {0.000861476902752481, 0.000387460327590718, 0.000692624344323799}, {3.52955845225811e-06, 5.76954089317204e-06, 1.31323031574614e-05}},
  {"Olympus XZ-2 iso 6400",  "OLYMPUS IMAGING CORP.", "XZ-2", 6400,  {0.00104697201934961, 0.000483575658390397, 0.000887961919122172}, {1.34729893779625e-05, 1.43280158845053e-05, 2.0911183510849e-05}},
  {"Olympus XZ-2 iso 8000",  "OLYMPUS IMAGING CORP.", "XZ-2", 8000,  {0.001454063576213, 0.000619820872005883, 0.00114367329489707}, {3.26562360613721e-06, 1.63489534009698e-05, 2.03593167856629e-05}},
  {"Olympus XZ-2 iso 10000", "OLYMPUS IMAGING CORP.", "XZ-2", 10000, {0.00160146918737965, 0.000714184399514146, 0.00132825603104515}, {1.7898559048173e-05, 2.83372439080334e-05, 3.21756580400381e-05}},
  {"Olympus XZ-2 iso 12800", "OLYMPUS IMAGING CORP.", "XZ-2", 12800, {0.00197274962013922, 0.000887984223518664, 0.00159090003502228}, {3.04161428529926e-05, 4.55876275053981e-05, 5.99806785358221e-05}},

// -------------------------------------------------------------------
// Panasonic.
// -------------------------------------------------------------------

// dmc-gm1 by thokster
{"DMC-GM1 iso 200",       "Panasonic",      "DMC-GM1",              200,         {1.6363272441348e-05, 5.96460721599018e-06, 1.00380274628482e-05}, {1.39389913671773e-08, 1.19746261031131e-08, 3.3618099565601e-08}},
{"DMC-GM1 iso 400",       "Panasonic",      "DMC-GM1",              400,         {3.21518671859676e-05, 1.180506396837e-05, 1.97516637094405e-05}, {3.3333637109306e-08, 2.81927908629262e-08, 7.21876520457764e-08}},
{"DMC-GM1 iso 800",       "Panasonic",      "DMC-GM1",              800,         {6.30213257385461e-05, 2.31634453873582e-05, 3.8878359031688e-05}, {8.50607141532477e-08, 6.58362725813762e-08, 1.49875088379925e-07}},
{"DMC-GM1 iso 1600",       "Panasonic",      "DMC-GM1",              1600,         {0.000125338880598504, 4.65662219766677e-05, 7.70962207986323e-05}, {1.68374151322925e-07, 1.30913447717618e-07, 2.97244381537983e-07}},
{"DMC-GM1 iso 3200",       "Panasonic",      "DMC-GM1",              3200,         {0.000245622043463519, 9.20237655059064e-05, 0.000151991555060464}, {4.63772511569562e-07, 3.80060674430949e-07, 7.1730687258219e-07}},
{"DMC-GM1 iso 6400",       "Panasonic",      "DMC-GM1",              6400,         {0.000488643992819912, 0.0001836150866073, 0.000305410802380559}, {1.13921526502568e-06, 1.03656814260882e-06, 1.53116020449575e-06}},
{"DMC-GM1 iso 12800",       "Panasonic",      "DMC-GM1",              12800,         {0.000973787756382622, 0.000373067575956284, 0.000621315437814162}, {2.4885517845698e-06, 2.66995285897356e-06, 3.35538439020585e-06}},
{"DMC-GM1 iso 25600",       "Panasonic",      "DMC-GM1",              25600,         {0.00185903163702921, 0.000759385279460093, 0.00124088840004957}, {4.63004962492069e-06, 6.18492554738651e-06, 6.81248796869435e-06}},

// dmc-lx5 by Romano
{"DMC-LX5 iso 80",       "Panasonic",      "DMC-LX5",              80,         {1.59110484862836e-05, 9.26869976815034e-06, 1.9471443836629e-05}, {5.03364078558839e-08, 3.72522879654584e-08, 6.01725162365395e-08}},
{"DMC-LX5 iso 100",       "Panasonic",      "DMC-LX5",              100,         {1.92703420832992e-05, 1.12537235911161e-05, 2.3523360588336e-05}, {6.45833580034223e-08, 4.85718270973939e-08, 7.64046192735273e-08}},
{"DMC-LX5 iso 200",       "Panasonic",      "DMC-LX5",              200,         {3.66176313311188e-05, 2.13849619718541e-05, 4.42400109188138e-05}, {1.5793184103613e-07, 1.24694724210479e-07, 1.8183945867649e-07}},
{"DMC-LX5 iso 400",       "Panasonic",      "DMC-LX5",              400,         {7.19354871114195e-05, 4.23443948441202e-05, 8.58177995740906e-05}, {3.99406841844551e-07, 3.22067009907329e-07, 4.31667572132589e-07}},
{"DMC-LX5 iso 800",       "Panasonic",      "DMC-LX5",              800,         {0.000160326929185133, 9.7499058489097e-05, 0.000191707950274305}, {1.07850968236824e-06, 8.93830303532209e-07, 1.12052645720807e-06}},
{"DMC-LX5 iso 1600",       "Panasonic",      "DMC-LX5",              1600,         {0.000326973781621303, 0.000192401877578863, 0.000396997597766266}, {1.01859507179224e-06, 7.6121401858243e-07, 1.14768878870081e-06}},
{"DMC-LX5 iso 3200",       "Panasonic",      "DMC-LX5",              3200,         {0.000753385115315346, 0.000499216658014752, 0.000934713494792712}, {3.07623783208053e-06, 2.192189314122e-06, 3.07499154736357e-06}},

// dmc-lx7 by Deryk Lister. fits are in general very good, goes up to 2x variance below 0.1 or so.
{"Panasonic DMC-LX7 iso 80",   "Panasonic", "DMC-LX7", 80,   {1.21927337863881e-05, 9.10066962350128e-06, 2.99704552994125e-05}, {2.11544322927929e-08, 1.72250747457525e-08, 1.06977229157525e-07}},
{"Panasonic DMC-LX7 iso 100",  "Panasonic", "DMC-LX7", 100,  {1.5284629867112e-05, 1.14390335854103e-05, 4.44964298536897e-05}, {2.60559471469043e-08, 1.6503542446782e-08, 1.09445248661759e-07}},
{"Panasonic DMC-LX7 iso 125",  "Panasonic", "DMC-LX7", 125,  {1.92927100450229e-05, 1.44131681368303e-05, 5.77720429255622e-05}, {3.22643141735452e-08, 1.93810346085923e-08, 9.84581736469078e-08}},
{"Panasonic DMC-LX7 iso 160",  "Panasonic", "DMC-LX7", 160,  {2.29787893636122e-05, 1.70358718417122e-05, 2.59986104607399e-05}, {6.46977623634805e-08, 5.00628618434537e-08, 2.46073061875402e-07}},
{"Panasonic DMC-LX7 iso 200",  "Panasonic", "DMC-LX7", 200,  {2.8649068457691e-05, 2.10590366006481e-05, 7.80791019943791e-05}, {9.42667945329647e-08, 7.81295904563742e-08, 2.37165529658449e-07}},
{"Panasonic DMC-LX7 iso 250",  "Panasonic", "DMC-LX7", 250,  {3.66895648804606e-05, 2.71860217976583e-05, 0.000108164137398154}, {1.2116735248411e-07, 9.25759864581005e-08, 2.69121323272301e-07}},
{"Panasonic DMC-LX7 iso 320",  "Panasonic", "DMC-LX7", 320,  {4.57709466388591e-05, 3.3896437768491e-05, 0.00013369733960354}, {1.64107297296824e-07, 1.3232904888129e-07, 3.20496060407108e-07}},
{"Panasonic DMC-LX7 iso 400",  "Panasonic", "DMC-LX7", 400,  {5.73044642538001e-05, 4.22827088668122e-05, 0.00016801107662137}, {2.20448482134481e-07, 1.62527664527192e-07, 4.11729710142906e-07}},
{"Panasonic DMC-LX7 iso 500",  "Panasonic", "DMC-LX7", 500,  {7.15048448056626e-05, 5.31314090758286e-05, 0.000211125529587164}, {3.16827624663753e-07, 2.49093187524134e-07, 5.64729395800579e-07}},
{"Panasonic DMC-LX7 iso 640",  "Panasonic", "DMC-LX7", 640,  {9.06579098383378e-05, 6.66116842974091e-05, 0.000266847866538387}, {4.6180633815394e-07, 3.61928920472811e-07, 7.16210441190569e-07}},
{"Panasonic DMC-LX7 iso 800",  "Panasonic", "DMC-LX7", 800,  {0.000112445733075578, 8.32206303480511e-05, 0.000327700642939444}, {6.49056110641167e-07, 5.00345975636821e-07, 1.10104850501599e-06}},
{"Panasonic DMC-LX7 iso 1000", "Panasonic", "DMC-LX7", 1000, {0.000140610458538795, 0.000103757091376245, 0.000413136953192513}, {9.22883211416327e-07, 6.73855099963928e-07, 1.42060689593717e-06}},
{"Panasonic DMC-LX7 iso 1250", "Panasonic", "DMC-LX7", 1250, {0.000181610825098946, 0.000134917597508616, 0.00054049377668518}, {-5.04830656720749e-07, -6.71997376797618e-07, 7.47320052424847e-08}},
{"Panasonic DMC-LX7 iso 1600", "Panasonic", "DMC-LX7", 1600, {0.000226806183420878, 0.00016514167473341, 0.000638542605196157}, {-1.04425631249494e-06, -1.11794949025072e-06, -1.32726300759437e-07}},
{"Panasonic DMC-LX7 iso 2000", "Panasonic", "DMC-LX7", 2000, {0.000282262068906761, 0.000208286340162449, 0.000805634983376735}, {-9.95790339686706e-07, -1.21896131708759e-06, 1.67029649675797e-07}},
{"Panasonic DMC-LX7 iso 2500", "Panasonic", "DMC-LX7", 2500, {0.00035785040095679, 0.000263702583678982, 0.00102595174113114}, {-2.40089762647301e-06, -2.57367914754276e-06, -3.94127373098259e-07}},
{"Panasonic DMC-LX7 iso 3200", "Panasonic", "DMC-LX7", 3200, {0.000439830730614632, 0.000323607640334832, 0.0012630301431057}, {-2.02014389033586e-06, -2.18626864980807e-06, 1.84571234986503e-07}},
{"Panasonic DMC-LX7 iso 4000", "Panasonic", "DMC-LX7", 4000, {0.000531444558025446, 0.000386578350310615, 0.00145398494733343}, {-4.79836562978599e-06, -3.96688583120201e-06, -9.69860164182844e-07}},
{"Panasonic DMC-LX7 iso 5000", "Panasonic", "DMC-LX7", 5000, {0.00069818697422707, 0.000511093858042714, 0.00201196118576844}, {-5.09391361580374e-06, -5.24353114889176e-06, 2.26830737523784e-07}},
{"Panasonic DMC-LX7 iso 6400", "Panasonic", "DMC-LX7", 6400, {0.000946655321903868, 0.00067145897047522, 0.00259479408952385}, {-3.63521160933705e-06, -3.85784586364296e-06, 4.57833601060677e-06}},

// panasonic dmc-fz18, contributed by sergey
{"Panasonic DMC-FZ18 iso 100",  "Panasonic", "DMC-FZ18", 100,  {4.16401970099069e-05, 1.89735592385362e-05, 3.51108971077795e-05}, {-6.3677312768354e-07, -3.58896425704885e-07, -2.52801382905561e-07}},
{"Panasonic DMC-FZ18 iso 200",  "Panasonic", "DMC-FZ18", 200,  {8.10284471823843e-05, 3.7242927889186e-05, 6.75239123655193e-05}, {-6.8829230409332e-07, -3.48873400487409e-07, -3.63886211817628e-08}},
{"Panasonic DMC-FZ18 iso 400",  "Panasonic", "DMC-FZ18", 400,  {0.000160970891899617, 7.22572686584776e-05, 0.000130701462066712}, {3.90655870390923e-07, 5.37873026235376e-07, 8.52013291268056e-07}},
{"Panasonic DMC-FZ18 iso 800",  "Panasonic", "DMC-FZ18", 800,  {0.000299305762151068, 0.000133443968280794, 0.000245740650460262}, {2.49407502921694e-06, 4.04119896569051e-06, 4.78691683526493e-06}},
{"Panasonic DMC-FZ18 iso 1250", "Panasonic", "DMC-FZ18", 1250, {0.000491151924364374, 0.000220471001481997, 0.000392896404988607}, {6.30900067962441e-06, 7.237351988114e-06, 1.0121887077351e-05}},
{"Panasonic DMC-FZ18 iso 1600", "Panasonic", "DMC-FZ18", 1600, {0.000595790740211985, 0.000269916437624247, 0.00049314652119945}, {1.51963936138399e-05, 1.57143565063181e-05, 1.89266116935137e-05}},

// panasonic dmc-g2, contributed by bruce albert, blue is off
  {"Panasonic DMC-G2 iso 100",  "Panasonic", "DMC-G2", 100,  {1.28863117302705e-05, 6.4897657152274e-06,  0.0}, {2.23866645315887e-07, -5.61294188710384e-08, 0.0}},
  {"Panasonic DMC-G2 iso 200",  "Panasonic", "DMC-G2", 200,  {2.50036296001071e-05, 1.14591090521316e-05, 0.0}, {4.72196206767348e-07, 1.34590860475144e-07,  0.0}},
  {"Panasonic DMC-G2 iso 400",  "Panasonic", "DMC-G2", 400,  {4.68359129158422e-05, 2.16168188111367e-05, 0.0}, {1.38633675858918e-06, 6.06834803346104e-07,  0.0}},
  {"Panasonic DMC-G2 iso 800",  "Panasonic", "DMC-G2", 800,  {9.64682273607663e-05, 4.45998272483029e-05, 0.0}, {3.9090839571423e-06, 2.04845212972453e-06,   0.0}},
  {"Panasonic DMC-G2 iso 1600", "Panasonic", "DMC-G2", 1600, {0.000198119439259803, 8.82431328537856e-05, 0.0}, {1.12267764502829e-05, 7.95762462612043e-06,  0.0}},
  {"Panasonic DMC-G2 iso 3200", "Panasonic", "DMC-G2", 3200, {0.000414452544656728, 0.000181909604273877, 0.0}, {3.748984824167e-05, 2.89976355269634e-05,    0.0}},
  {"Panasonic DMC-G2 iso 6400", "Panasonic", "DMC-G2", 6400, {0.000981911150517475, 0.000406395513355329, 0.0}, {0.000109720888451889, 9.95986185790299e-05,  0.0}},

// pana dmc g6 by Bernhard Kappler
{"Panasonic DMC-G6 iso 160",   "Panasonic", "DMC-G6", 160,   {1.5780915021662e-05, 7.01399292273453e-06, 1.43466313906541e-05}, {2.54138527680568e-08, 2.57559637129589e-08, 5.11460792038577e-08}},
{"Panasonic DMC-G6 iso 200",   "Panasonic", "DMC-G6", 200,   {1.92518611805957e-05, 8.56153731056309e-06, 1.73556219887831e-05}, {4.08240710566109e-08, 3.98591492889263e-08, 7.13669736529514e-08}},
{"Panasonic DMC-G6 iso 400",   "Panasonic", "DMC-G6", 400,   {3.56208282151563e-05, 1.61151466614279e-05, 3.25553110358377e-05}, {9.96001197914379e-08, 8.82898512800543e-08, 1.4941317196432e-07}},
{"Panasonic DMC-G6 iso 800",   "Panasonic", "DMC-G6", 800,   {6.77962109251139e-05, 3.09823524010034e-05, 6.26623411901002e-05}, {2.63264005848084e-07, 2.13373297010619e-07, 3.32061859281897e-07}},
{"Panasonic DMC-G6 iso 1600",  "Panasonic", "DMC-G6", 1600,  {0.000134043892901807, 6.16940170822923e-05, 0.000127082551476612}, {6.49001726475163e-07, 5.54762345272061e-07, 7.86490481805828e-07}},
{"Panasonic DMC-G6 iso 3200",  "Panasonic", "DMC-G6", 3200,  {0.000275609481248173, 0.0001274826185927, 0.000263650444992015}, {1.56920413696794e-06, 1.50604002289925e-06, 1.83623798715862e-06}},
{"Panasonic DMC-G6 iso 6400",  "Panasonic", "DMC-G6", 6400,  {0.000599035976280823, 0.000294622004241837, 0.000618861540386321}, {3.84546943020711e-06, 4.35027867756884e-06, 4.7608577337749e-06}},
{"Panasonic DMC-G6 iso 12800", "Panasonic", "DMC-G6", 12800, {0.00129388820073503, 0.000713954935442586, 0.0014658462258173}, {8.85082998557482e-06, 1.12352282775213e-05, 1.14587301942128e-05}},
{"Panasonic DMC-G6 iso 25600", "Panasonic", "DMC-G6", 25600, {0.0026309817767123, 0.00192487521752666, 0.00350465203783809}, {2.77535707799781e-05, 3.03814976958121e-05, 3.11503049891723e-05}},

// pana g10 by sergej nikolaev. very noisy variance estimates, fits surprisingly good, variance stabilization goes up to ~3 near zero.
{"Panasonic DMC-G10 iso 100",  "Panasonic", "DMC-G10", 100,  {1.12261009023862e-05, 1.00126636086952e-05, 9.90687518977869e-06}, {-5.21524667357915e-07, -1.24786553891377e-06, -1.68545446451732e-07}},
{"Panasonic DMC-G10 iso 125",  "Panasonic", "DMC-G10", 125,  {1.29168788282981e-05, 1.14548028988677e-05, 9.34886167123922e-06}, {-5.12054191184941e-07, -1.29619341680852e-06, 2.90344443116951e-07}},
{"Panasonic DMC-G10 iso 160",  "Panasonic", "DMC-G10", 160,  {1.78083892298608e-05, 1.171737397529e-05, 1.47464563783729e-05}, {-7.85354860515029e-07, -9.53217208373343e-07, -7.5917604527014e-08}},
{"Panasonic DMC-G10 iso 200",  "Panasonic", "DMC-G10", 200,  {1.79237662520455e-05, 1.34896367447827e-05, 1.41722113840186e-05}, {-5.81367726189874e-07, -1.0444512331827e-06, 4.14856001677019e-07}},
{"Panasonic DMC-G10 iso 250",  "Panasonic", "DMC-G10", 250,  {2.41661563673123e-05, 1.55958832608481e-05, 1.48972519138968e-05}, {-8.72597911455858e-07, -9.07157885252161e-07, 9.65664908942579e-07}},
{"Panasonic DMC-G10 iso 320",  "Panasonic", "DMC-G10", 320,  {3.3213771014683e-05, 1.946684536698e-05, 2.23535500781124e-05}, {-1.32824741814194e-06, -9.7350596421945e-07, 7.942005599417e-07}},
{"Panasonic DMC-G10 iso 400",  "Panasonic", "DMC-G10", 400,  {2.63665371352973e-05, 2.24373512828462e-05, 2.44324780844092e-05}, {4.67920887888322e-09, -9.24555737204257e-07, 1.37813046565223e-06}},
{"Panasonic DMC-G10 iso 500",  "Panasonic", "DMC-G10", 500,  {3.77343626224594e-05, 2.96546963086404e-05, 3.49497277813669e-05}, {-3.19192402333096e-07, -1.46574654005367e-06, 1.03789384273711e-06}},
{"Panasonic DMC-G10 iso 640",  "Panasonic", "DMC-G10", 640,  {4.54183065919518e-05, 3.00433173807882e-05, 3.918067621488e-05}, {4.18753369693168e-08, 7.72346285999333e-08, 2.87554593303457e-06}},
{"Panasonic DMC-G10 iso 800",  "Panasonic", "DMC-G10", 800,  {5.56573866915174e-05, 4.18308700169092e-05, 5.99225322506473e-05}, {9.8610014973254e-07, 2.64840291992665e-07, 2.74841122210632e-06}},
{"Panasonic DMC-G10 iso 1000", "Panasonic", "DMC-G10", 1000, {6.56999958367293e-05, 5.49901069963697e-05, 5.77208060090439e-05}, {2.32894420026412e-06, -9.35303338234587e-08, 6.79383892157983e-06}},
{"Panasonic DMC-G10 iso 1250", "Panasonic", "DMC-G10", 1250, {8.1633939989716e-05, 6.04485660868771e-05, 8.54611806517871e-05}, {3.950648331982e-06, 2.68111377686731e-06, 7.70794130725149e-06}},
{"Panasonic DMC-G10 iso 1600", "Panasonic", "DMC-G10", 1600, {0.000107838745181099, 7.74209736450752e-05, 0.000107619636216348}, {5.89833035745227e-06, 4.28248301499812e-06, 1.12239519525354e-05}},
{"Panasonic DMC-G10 iso 2000", "Panasonic", "DMC-G10", 2000, {0.000135467058502925, 0.000103398424314797, 0.000131349158809152}, {9.7329979129253e-06, 6.31888151474442e-06, 1.81864730990362e-05}},
{"Panasonic DMC-G10 iso 2500", "Panasonic", "DMC-G10", 2500, {0.000174603550218772, 0.000117224638051475, 0.000151672263753349}, {1.55454197117085e-05, 1.30421698209037e-05, 2.88881069888145e-05}},
{"Panasonic DMC-G10 iso 3200", "Panasonic", "DMC-G10", 3200, {0.000237707713783773, 0.000168018053957259, 0.000233771794645976}, {2.37225066916069e-05, 1.78293689523681e-05, 3.59569985371811e-05}},
{"Panasonic DMC-G10 iso 4000", "Panasonic", "DMC-G10", 4000, {0.00031586164289557, 0.000189502307583628, 0.0002635713500754}, {3.61096857008159e-05, 3.49320394852212e-05, 6.08884091301562e-05}},
{"Panasonic DMC-G10 iso 5000", "Panasonic", "DMC-G10", 5000, {0.000398421666451168, 0.000264296294972876, 0.000384405873321081}, {5.85440835974543e-05, 5.36573120979847e-05, 9.30040538088022e-05}},
{"Panasonic DMC-G10 iso 6400", "Panasonic", "DMC-G10", 6400, {0.000590704898866065, 0.000306048163868126, 0.000444512770400911}, {8.3249027104092e-05, 9.07193754361519e-05, 0.000140610561505732}},

// panasonic gh2 by moritz moeller
  {"Panasonic DMC-GH2 iso 160",       "Panasonic",      "DMC-GH2",              160,         {1.89270205445016e-05, 7.99834828651038e-06, 1.34198040510158e-05}, {3.49658480177552e-07, -2.30234231616608e-07, -2.68232880304034e-07}},
// {"Panasonic DMC-GH2 iso 160",       "Panasonic",      "DMC-GH2",              160,         {1.8226797057964e-05, 7.87846452988407e-06, 1.31285277363057e-05}, {3.54062609189498e-07, -2.17188521497174e-07, -2.21227854606662e-07}},
// {"Panasonic DMC-GH2 iso 160",       "Panasonic",      "DMC-GH2",              160,         {1.82266738120649e-05, 7.96490109190985e-06, 1.35465509096455e-05}, {3.9628402627846e-07, -2.09466532199586e-07, -2.69204283703184e-07}},
  {"Panasonic DMC-GH2 iso 200",       "Panasonic",      "DMC-GH2",              200,         {2.3133162810518e-05, 1.00941325539983e-05, 1.6896695444511e-05}, {3.83079498719147e-07, -3.35268465582744e-07, -4.01611430664602e-07}},
// {"Panasonic DMC-GH2 iso 200",       "Panasonic",      "DMC-GH2",              200,         {2.30380650892986e-05, 1.00191241423453e-05, 1.62151078348025e-05}, {3.89051137668343e-07, -3.16756968141555e-07, -2.82466349429554e-07}},
// {"Panasonic DMC-GH2 iso 200",       "Panasonic",      "DMC-GH2",              200,         {2.25269336842682e-05, 1.00571116951725e-05, 1.62222542172345e-05}, {4.41677614680555e-07, -3.25977516243255e-07, -2.85206786097157e-07}},
  {"Panasonic DMC-GH2 iso 250",       "Panasonic",      "DMC-GH2",              250,         {2.7808276515294e-05, 1.20606559645744e-05, 1.95792160449325e-05}, {5.67116892835644e-07, -2.89314709481293e-07, -2.41880686730256e-07}},
// {"Panasonic DMC-GH2 iso 250",       "Panasonic",      "DMC-GH2",              250,         {2.77206834748769e-05, 1.23038730320824e-05, 2.01074054689656e-05}, {5.29884033399611e-07, -3.54538358061179e-07, -3.25818009626065e-07}},
// {"Panasonic DMC-GH2 iso 250",       "Panasonic",      "DMC-GH2",              250,         {2.69617196549686e-05, 1.21099358292924e-05, 2.08958609560743e-05}, {6.31083277354996e-07, -3.46890551634024e-07, -4.71298689350039e-07}},
  {"Panasonic DMC-GH2 iso 320",       "Panasonic",      "DMC-GH2",              320,         {3.28210859174139e-05, 1.44323843045695e-05, 2.45907448753219e-05}, {8.44661662559387e-07, -2.44027655437745e-07, -3.89431927440291e-07}},
// {"Panasonic DMC-GH2 iso 320",       "Panasonic",      "DMC-GH2",              320,         {3.26309695004751e-05, 1.44643846638843e-05, 2.48571299734751e-05}, {8.30115629863324e-07, -3.17171907047533e-07, -4.74404102640725e-07}},
// {"Panasonic DMC-GH2 iso 320",       "Panasonic",      "DMC-GH2",              320,         {3.32089063052988e-05, 1.43582354785348e-05, 2.39372554109205e-05}, {7.65089906000245e-07, -2.84085668592382e-07, -3.20064084518731e-07}},
  {"Panasonic DMC-GH2 iso 400",       "Panasonic",      "DMC-GH2",              400,         {4.3512514702447e-05, 1.84270542539292e-05, 2.84200427806186e-05}, {6.46388898762575e-07, -5.14316896592473e-07, -1.7253515210863e-07}},
// {"Panasonic DMC-GH2 iso 400",       "Panasonic",      "DMC-GH2",              400,         {4.2974751612323e-05, 1.78413242195645e-05, 2.97120433617585e-05}, {6.9512267538433e-07, -3.0906420649805e-07, -3.71209661632257e-07}},
// {"Panasonic DMC-GH2 iso 400",       "Panasonic",      "DMC-GH2",              400,         {4.25827084139978e-05, 1.86552768969244e-05, 3.07936488930159e-05}, {8.37832659620483e-07, -4.80055360429456e-07, -4.59578004711314e-07}},
  {"Panasonic DMC-GH2 iso 500",       "Panasonic",      "DMC-GH2",              500,         {5.270529542615e-05, 2.1390947025784e-05, 3.53470112894029e-05}, {8.82687413624964e-07, -2.80886013633108e-07, -2.17234530960418e-07}},
// {"Panasonic DMC-GH2 iso 500",       "Panasonic",      "DMC-GH2",              500,         {5.16470502808731e-05, 2.0955342410845e-05, 3.63209143612118e-05}, {1.06264545555843e-06, -6.19008854996885e-08, -2.98023755866507e-07}},
// {"Panasonic DMC-GH2 iso 500",       "Panasonic",      "DMC-GH2",              500,         {5.18501276643689e-05, 2.24241178649677e-05, 3.61044842423266e-05}, {1.05299352665384e-06, -4.88869917771271e-07, -3.23129411575441e-07}},
  {"Panasonic DMC-GH2 iso 640",       "Panasonic",      "DMC-GH2",              640,         {6.29265009678382e-05, 2.61653620986977e-05, 4.44826462803236e-05}, {1.23356063250856e-06, -2.44050602964552e-07, -3.65422795247278e-07}},
// {"Panasonic DMC-GH2 iso 640",       "Panasonic",      "DMC-GH2",              640,         {6.28240190867108e-05, 2.59132851683965e-05, 4.37449497498776e-05}, {1.25166210321563e-06, -2.37593897234142e-07, -2.78262550051989e-07}},
// {"Panasonic DMC-GH2 iso 640",       "Panasonic",      "DMC-GH2",              640,         {6.23939951398563e-05, 2.6677466097104e-05, 4.67706511530863e-05}, {1.3549331241038e-06, -4.18976306062024e-07, -8.68747549861465e-07}},
  {"Panasonic DMC-GH2 iso 800",       "Panasonic",      "DMC-GH2",              800,         {7.54246868008188e-05, 3.29043028894462e-05, 5.31788696382216e-05}, {1.93116835702027e-06, -3.98225886069213e-07, -1.23303621479774e-07}},
// {"Panasonic DMC-GH2 iso 800",       "Panasonic",      "DMC-GH2",              800,         {7.67177082449538e-05, 3.37448704105288e-05, 5.38225147359715e-05}, {1.71567993448516e-06, -5.69296799066601e-07, -2.07205193926967e-07}},
// {"Panasonic DMC-GH2 iso 800",       "Panasonic",      "DMC-GH2",              800,         {7.53980211684284e-05, 3.21695426568513e-05, 5.33511870033629e-05}, {1.86500139929243e-06, -2.5709214157854e-07, -2.06229903158953e-07}},
  {"Panasonic DMC-GH2 iso 1000",       "Panasonic",      "DMC-GH2",              1000,         {0.000112285084457911, 3.89799459906781e-05, 6.90324537912573e-05}, {4.14943899109909e-07, 1.89318948127753e-07, -3.98822865032301e-07}},
// {"Panasonic DMC-GH2 iso 1000",       "Panasonic",      "DMC-GH2",              1000,         {0.000109493001751749, 3.9562150824219e-05, 7.02703067315143e-05}, {7.03349240607835e-07, 6.62556827558668e-08, -5.55408888176799e-07}},
// {"Panasonic DMC-GH2 iso 1000",       "Panasonic",      "DMC-GH2",              1000,         {0.000107988427720931, 4.09256782394555e-05, 6.58249988697796e-05}, {8.83810318642862e-07, -2.59404441010724e-07, 1.31129686530138e-07}},
  {"Panasonic DMC-GH2 iso 1250",       "Panasonic",      "DMC-GH2",              1250,         {0.000132341741753442, 4.91344974809805e-05, 8.08601618553551e-05}, {1.19832599502866e-06, 7.46832233744021e-08, 5.14345820928433e-07}},
// {"Panasonic DMC-GH2 iso 1250",       "Panasonic",      "DMC-GH2",              1250,         {0.000131631034823382, 4.83173777725425e-05, 8.17536641042016e-05}, {1.4154643681659e-06, 5.1972589098323e-07, 2.65668735538528e-07}},
// {"Panasonic DMC-GH2 iso 1250",       "Panasonic",      "DMC-GH2",              1250,         {0.000127567123643416, 4.75087471886525e-05, 7.70353048159838e-05}, {1.67469294730141e-06, 4.5427385068003e-07, 9.26418445010468e-07}},
  {"Panasonic DMC-GH2 iso 1600",       "Panasonic",      "DMC-GH2",              1600,         {0.000181372787583862, 6.58610594967533e-05, 0.000111144146730905}, {6.5987107847001e-07, -3.93040616129467e-07, -2.53872623124923e-07}},
// {"Panasonic DMC-GH2 iso 1600",       "Panasonic",      "DMC-GH2",              1600,         {0.000187131967023986, 6.70774390339241e-05, 0.000111929926910505}, {3.44344525534545e-07, -8.10683177884788e-07, -5.97336575309096e-07}},
// {"Panasonic DMC-GH2 iso 1600",       "Panasonic",      "DMC-GH2",              1600,         {0.000181740666138449, 6.3869832299098e-05, 0.000109516223189374}, {9.54754425770609e-07, 3.31914910236727e-07, 5.63532688804107e-08}},
// {"Panasonic DMC-GH2 iso 1600",       "Panasonic",      "DMC-GH2",              1600,         {0.000179269938148442, 6.45040691860879e-05, 0.000111681293314905}, {1.35148674854555e-06, 2.36824221829705e-07, -7.47145936277327e-08}},
// {"Panasonic DMC-GH2 iso 1600",       "Panasonic",      "DMC-GH2",              1600,         {0.000187383844689693, 6.9644185623975e-05, 0.000116324030702913}, {7.53738330250399e-07, -1.06069485539908e-06, -9.6871855652257e-07}},
// {"Panasonic DMC-GH2 iso 1600",       "Panasonic",      "DMC-GH2",              1600,         {0.000182289438722624, 6.84715718123998e-05, 0.000111284005621727}, {8.06098751827258e-07, -7.75261218735965e-07, -3.53578782915966e-07}},
  {"Panasonic DMC-GH2 iso 2000",       "Panasonic",      "DMC-GH2",              2000,         {0.00022337947589753, 8.25126640649617e-05, 0.000133027975046207}, {1.86294266132456e-06, 2.3788849849459e-07, 1.26245970864563e-06}},
// {"Panasonic DMC-GH2 iso 2000",       "Panasonic",      "DMC-GH2",              2000,         {0.000230737020983874, 8.27100115031416e-05, 0.000139847734709886}, {1.20852891272474e-06, 6.96457598013981e-08, 1.44374421877815e-07}},
// {"Panasonic DMC-GH2 iso 2000",       "Panasonic",      "DMC-GH2",              2000,         {0.000229551759211857, 8.41357582455954e-05, 0.00013662680668575}, {1.23031918508895e-06, -2.78970136416056e-07, 6.10454818625792e-07}},
  {"Panasonic DMC-GH2 iso 2500",       "Panasonic",      "DMC-GH2",              2500,         {0.000277097541429557, 0.000101410671803775, 0.00017035653889861}, {2.67370963661405e-06, 5.98348452760539e-07, 4.87116639585418e-07}},
// {"Panasonic DMC-GH2 iso 2500",       "Panasonic",      "DMC-GH2",              2500,         {0.000275600006312595, 0.00010155177978948, 0.000166417300017639}, {2.86475741415684e-06, 3.18478499144945e-07, 1.26167332009352e-06}},
// {"Panasonic DMC-GH2 iso 2500",       "Panasonic",      "DMC-GH2",              2500,         {0.000272500109627457, 0.000102928573872776, 0.000169436312840442}, {3.17279177745365e-06, -1.28473412157919e-07, 7.5598614088203e-07}},
  {"Panasonic DMC-GH2 iso 3200",       "Panasonic",      "DMC-GH2",              3200,         {0.000313949047712229, 0.000122998749201784, 0.000213219468325574}, {7.01879562205314e-06, 1.96962144791641e-06, 1.33343936893674e-06}},
// {"Panasonic DMC-GH2 iso 3200",       "Panasonic",      "DMC-GH2",              3200,         {0.000317409052439849, 0.000124214072834635, 0.00021047224062765}, {6.73274397396351e-06, 1.87475468772257e-06, 2.05411555169486e-06}},
// {"Panasonic DMC-GH2 iso 3200",       "Panasonic",      "DMC-GH2",              3200,         {0.000322012454042082, 0.00012101983640727, 0.000214484063474836}, {6.51157097659171e-06, 2.70811867775577e-06, 1.51832830347601e-06}},
  {"Panasonic DMC-GH2 iso 4000",       "Panasonic",      "DMC-GH2",              4000,         {0.000417818071582836, 0.000155662733003424, 0.000251992389712746}, {9.22061847300145e-06, 5.17341939199248e-06, 7.56202992942988e-06}},
// {"Panasonic DMC-GH2 iso 4000",       "Panasonic",      "DMC-GH2",              4000,         {0.000403287556460611, 0.000147907213743486, 0.000250439381769453}, {1.10222817137257e-05, 6.360405174804e-06, 7.58144273644377e-06}},
// {"Panasonic DMC-GH2 iso 4000",       "Panasonic",      "DMC-GH2",              4000,         {0.000416120173100964, 0.000151067707674894, 0.000255844021707377}, {9.93261648090873e-06, 6.55528190929531e-06, 7.91369532149949e-06}},
  {"Panasonic DMC-GH2 iso 5000",       "Panasonic",      "DMC-GH2",              5000,         {0.000477940758312781, 0.0001852360729931, 0.000313901505840974}, {1.4537226153747e-05, 7.74919879446124e-06, 9.02370123511893e-06}},
// {"Panasonic DMC-GH2 iso 5000",       "Panasonic",      "DMC-GH2",              5000,         {0.000450330364694479, 0.000184243884787783, 0.000320021070129249}, {1.7243950404875e-05, 7.08076673218931e-06, 7.10300585841591e-06}},
// {"Panasonic DMC-GH2 iso 5000",       "Panasonic",      "DMC-GH2",              5000,         {0.000488192047969363, 0.000196316219510303, 0.000326720139056993}, {1.41204878883754e-05, 4.80125453612719e-06, 6.13236449402836e-06}},
  {"Panasonic DMC-GH2 iso 6400",       "Panasonic",      "DMC-GH2",              6400,         {0.000604860238583645, 0.000241831484398793, 0.000402693327919412}, {2.14961593421955e-05, 9.6634262902759e-06, 1.25596198876955e-05}},
// {"Panasonic DMC-GH2 iso 6400",       "Panasonic",      "DMC-GH2",              6400,         {0.000613538136764406, 0.000242772967037239, 0.000393120791716206}, {1.99569134718832e-05, 9.73992730543723e-06, 1.36604840444571e-05}},
// {"Panasonic DMC-GH2 iso 6400",       "Panasonic",      "DMC-GH2",              6400,         {0.000644528045450721, 0.000244208988469399, 0.000406419584487094}, {1.76014071781936e-05, 9.82547911846456e-06, 1.34776619576967e-05}},
  {"Panasonic DMC-GH2 iso 8000",       "Panasonic",      "DMC-GH2",              8000,         {0.00071498014774894, 0.000285937584567672, 0.00048567761399953}, {3.37830035967657e-05, 2.06145034426403e-05, 2.40566151027654e-05}},
// {"Panasonic DMC-GH2 iso 8000",       "Panasonic",      "DMC-GH2",              8000,         {0.000702709535451303, 0.000292963477907501, 0.00049258253248269}, {3.64830100318679e-05, 1.94798627913564e-05, 2.38926746938975e-05}},
// {"Panasonic DMC-GH2 iso 8000",       "Panasonic",      "DMC-GH2",              8000,         {0.00070052110367074, 0.000307847443687519, 0.000494573511494508}, {3.65851310651613e-05, 1.55252595913832e-05, 2.42012664508781e-05}},
  {"Panasonic DMC-GH2 iso 10000",       "Panasonic",      "DMC-GH2",              10000,         {0.000863483096275913, 0.000350710018787375, 0.000647279814178636}, {4.66853393309581e-05, 3.33053382236365e-05, 3.20498686142792e-05}},
// {"Panasonic DMC-GH2 iso 10000",       "Panasonic",      "DMC-GH2",              10000,         {0.000843851560271897, 0.000367269619864526, 0.000666249218551523}, {5.04169882354229e-05, 2.95780124110386e-05, 2.54849558786756e-05}},
// {"Panasonic DMC-GH2 iso 10000",       "Panasonic",      "DMC-GH2",              10000,         {0.000881146194707806, 0.000364141708915348, 0.000630282682268507}, {4.61398859544311e-05, 2.83513370230735e-05, 3.27004363600218e-05}},
  {"Panasonic DMC-GH2 iso 12800",       "Panasonic",      "DMC-GH2",              12800,         {0.00102872432426638, 0.000484138376438294, 0.000876401769766701}, {6.99342192533772e-05, 4.22665815865089e-05, 3.57927612934234e-05}},
// {"Panasonic DMC-GH2 iso 12800",       "Panasonic",      "DMC-GH2",              12800,         {0.00106473103401707, 0.000490500203541779, 0.000877631550841204}, {6.81477240658733e-05, 3.98495481932477e-05, 3.5925380077772e-05}},
// {"Panasonic DMC-GH2 iso 12800",       "Panasonic",      "DMC-GH2",              12800,         {0.000998129336639947, 0.000508024501052357, 0.000889953878471562}, {7.42720992200759e-05, 3.55149426703435e-05, 3.5585991004626e-05}},

// panasonic gh-3 by thouks
// {"Panasonic DMC-GH3 iso 125",   "Panasonic", "DMC-GH3", 125,   {1.01416886699687e-05, 3.62631628681706e-06, 7.20278349222426e-06}, {-2.27268568671298e-08, -4.75354151417703e-08, 6.2889643499808e-08}},
// {"Panasonic DMC-GH3 iso 200",   "Panasonic", "DMC-GH3", 200,   {1.53773292997987e-05, 5.43125266908477e-06, 1.09301284367148e-05}, {-2.31738821817311e-08, -4.93686692936241e-08, 9.98224537347725e-08}},
// {"Panasonic DMC-GH3 iso 400",   "Panasonic", "DMC-GH3", 400,   {2.87159086065901e-05, 1.02781221192509e-05, 2.05744461327117e-05}, {2.87855315825181e-08, -5.34192335924304e-08, 2.47534182053436e-07}},
// {"Panasonic DMC-GH3 iso 800",   "Panasonic", "DMC-GH3", 800,   {5.33491613799709e-05, 1.97099289844296e-05, 3.95787767032685e-05}, {2.33545664158858e-07, -5.36788095395187e-08, 5.52928055970549e-07}},
// {"Panasonic DMC-GH3 iso 1600",  "Panasonic", "DMC-GH3", 1600,  {0.00010314956279316, 3.79048095197015e-05, 7.76520926372414e-05}, {4.76802690436105e-07, 1.11616565563265e-07, 1.09578086055682e-06}},
// {"Panasonic DMC-GH3 iso 3200",  "Panasonic", "DMC-GH3", 3200,  {0.000201897562570908, 7.52246416149234e-05, 0.000151446467990392}, {1.20430024819731e-06, 3.63418673532779e-07, 2.60995801564557e-06}},
// {"Panasonic DMC-GH3 iso 6400",  "Panasonic", "DMC-GH3", 6400,  {0.000397209237405763, 0.000149498617650985, 0.000300573767056221}, {4.01655602002391e-06, 2.30974169894196e-06, 7.21663294455228e-06}},
// {"Panasonic DMC-GH3 iso 12800", "Panasonic", "DMC-GH3", 12800, {0.000567873945995667, 0.000211138468045572, 0.000419241776549446}, {6.60993844131908e-06, 5.63424983571341e-06, 1.23923447566781e-05}},
// {"Panasonic DMC-GH3 iso 25600", "Panasonic", "DMC-GH3", 25600, {0.000944286473800959, 0.000360357607627817, 0.000683617687375674}, {1.87250722357695e-05, 1.44660145297859e-05, 3.22308039971408e-05}},
// more detailed iso stops for dmc-gh3 by thokster
{"DMC-GH3 iso 125",       "Panasonic",      "DMC-GH3",              125,         {9.85187704485256e-06, 3.75647787457979e-06, 7.20140154867181e-06}, {1.83944497372149e-08, 1.42005121159747e-08, 1.87361580738926e-08}},
{"DMC-GH3 iso 160",       "Panasonic",      "DMC-GH3",              160,         {1.22673795306915e-05, 4.69909828231389e-06, 8.95123196310763e-06}, {2.39347881228185e-08, 1.87873816180159e-08, 2.43358011798371e-08}},
{"DMC-GH3 iso 200",       "Panasonic",      "DMC-GH3",              200,         {1.51651846898214e-05, 5.80863518249195e-06, 1.10688201258311e-05}, {2.32992901240764e-08, 1.77798944364534e-08, 2.37032917933967e-08}},
{"DMC-GH3 iso 250",       "Panasonic",      "DMC-GH3",              250,         {1.89401687029122e-05, 7.19927518623237e-06, 1.36322432878929e-05}, {3.05008060185872e-08, 2.38716402039969e-08, 3.13147906421069e-08}},
{"DMC-GH3 iso 320",       "Panasonic",      "DMC-GH3",              320,         {2.33490698347419e-05, 8.95601375908163e-06, 1.70282156375675e-05}, {3.9464368900885e-08, 3.08588940442119e-08, 4.04173165703425e-08}},
{"DMC-GH3 iso 400",       "Panasonic",      "DMC-GH3",              400,         {2.90044280144097e-05, 1.10426749295404e-05, 2.10217884133538e-05}, {5.17326322469351e-08, 4.07481340759248e-08, 5.2801449968815e-08}},
{"DMC-GH3 iso 500",       "Panasonic",      "DMC-GH3",              500,         {3.61625516785174e-05, 1.38385739859384e-05, 2.64433446949056e-05}, {6.87785289456135e-08, 5.39821076943888e-08, 7.04713458852305e-08}},
{"DMC-GH3 iso 640",       "Panasonic",      "DMC-GH3",              640,         {4.48475799774877e-05, 1.71745974243146e-05, 3.28240743863861e-05}, {9.25560693039162e-08, 7.32921437198041e-08, 9.50918913627361e-08}},
{"DMC-GH3 iso 800",       "Panasonic",      "DMC-GH3",              800,         {5.54664397677488e-05, 2.14438910422999e-05, 4.08933209162082e-05}, {1.22520594151136e-07, 9.66037000738486e-08, 1.2526643697903e-07}},
{"DMC-GH3 iso 1000",       "Panasonic",      "DMC-GH3",              1000,         {7.02215885599914e-05, 2.72856490251717e-05, 5.14547015730763e-05}, {1.60578077029787e-07, 1.27513922716749e-07, 1.65346772308902e-07}},
{"DMC-GH3 iso 1250",       "Panasonic",      "DMC-GH3",              1250,         {8.72795626748159e-05, 3.37445928910752e-05, 6.37469714476212e-05}, {2.11210353994209e-07, 1.6799519692113e-07, 2.16674568632086e-07}},
{"DMC-GH3 iso 1600",       "Panasonic",      "DMC-GH3",              1600,         {0.000108141830391863, 4.26940611530677e-05, 8.00516475222328e-05}, {2.74001337809303e-07, 2.18384487067903e-07, 2.79795077164385e-07}},
{"DMC-GH3 iso 2000",       "Panasonic",      "DMC-GH3",              2000,         {0.000134869064318381, 5.36579486664949e-05, 9.95349454985739e-05}, {3.33579447122244e-07, 2.74363188795231e-07, 3.44945374333255e-07}},
{"DMC-GH3 iso 2500",       "Panasonic",      "DMC-GH3",              2500,         {0.000170494079075526, 6.84240336619346e-05, 0.000126980598517814}, {4.21532916274368e-07, 3.6185793387178e-07, 4.50556140956338e-07}},
{"DMC-GH3 iso 3200",       "Panasonic",      "DMC-GH3",              3200,         {0.000212202348929981, 8.71899558442478e-05, 0.000159533068945056}, {5.01715278061301e-07, 4.69961854905455e-07, 5.69669582490859e-07}},
{"DMC-GH3 iso 4000",       "Panasonic",      "DMC-GH3",              4000,         {0.000267245758595397, 0.000111575454674412, 0.000201437111566725}, {5.66087493808847e-07, 6.01040380793801e-07, 6.95753593953223e-07}},
{"DMC-GH3 iso 5000",       "Panasonic",      "DMC-GH3",              5000,         {0.000336485940434357, 0.000142582039016312, 0.000255571057533607}, {6.83222952067224e-07, 8.24292097838732e-07, 9.12204524542864e-07}},
{"DMC-GH3 iso 6400",       "Panasonic",      "DMC-GH3",              6400,         {0.000426614441233833, 0.000185586443339752, 0.000325854567377813}, {8.24155494224214e-07, 1.11306983468071e-06, 1.19525941111562e-06}},
{"DMC-GH3 iso 8000",       "Panasonic",      "DMC-GH3",              8000,         {0.000278474742261123, 0.000111052180412153, 0.000206754888376751}, {4.03370940844251e-07, 3.85219184582672e-07, 5.16685257984978e-07}},
{"DMC-GH3 iso 12800",       "Panasonic",      "DMC-GH3",              12800,         {0.000443533758477395, 0.000181397817037139, 0.000331329019028572}, {4.89623558307518e-07, 7.62320858113395e-07, 8.67408707101319e-07}},

// panasonic gh4 by martin schitter
{"DMC-GH4 iso 100",       "Panasonic",      "DMC-GH4",              100,         {3.57707749121624e-06, 2.12201509273029e-06, 7.04777000933672e-06}, {3.01786606774281e-09, 3.87897593526041e-09, 2.31937941275879e-08}},
{"DMC-GH4 iso 125",       "Panasonic",      "DMC-GH4",              125,         {4.44648902126317e-06, 2.63466843867568e-06, 8.76914290990608e-06}, {4.16108369877281e-09, 5.45061261130752e-09, 2.93530146510308e-08}},
{"DMC-GH4 iso 160",       "Panasonic",      "DMC-GH4",              160,         {5.54042770144228e-06, 3.27462124144126e-06, 1.08492205854013e-05}, {6.36361876131921e-09, 7.74627637413618e-09, 3.78435674927871e-08}},
{"DMC-GH4 iso 200",       "Panasonic",      "DMC-GH4",              200,         {6.93250939170529e-06, 4.10729137920921e-06, 1.36963715148297e-05}, {3.85002617866236e-09, 6.71105505430219e-09, 4.3200344331366e-08}},
{"DMC-GH4 iso 250",       "Panasonic",      "DMC-GH4",              250,         {8.62709116436961e-06, 5.08546132490219e-06, 1.68456237772895e-05}, {6.00973679700637e-09, 9.46857032404381e-09, 5.52266176037561e-08}},
{"DMC-GH4 iso 320",       "Panasonic",      "DMC-GH4",              320,         {1.07877365778046e-05, 6.35640929910184e-06, 2.11344841430651e-05}, {1.00102441106097e-08, 1.36341600893405e-08, 7.10411268023101e-08}},
{"DMC-GH4 iso 400",       "Panasonic",      "DMC-GH4",              400,         {1.34582752127752e-05, 7.96666598455718e-06, 2.6282529725979e-05}, {1.08026063065315e-08, 1.78160844444053e-08, 8.54342841921366e-08}},
{"DMC-GH4 iso 500",       "Panasonic",      "DMC-GH4",              500,         {1.67737419217109e-05, 9.93270132193865e-06, 3.27660811199064e-05}, {2.11781725635346e-08, 2.3916920664738e-08, 1.15139708727277e-07}},
{"DMC-GH4 iso 640",       "Panasonic",      "DMC-GH4",              640,         {2.10675557291628e-05, 1.24530164078382e-05, 4.12026817574144e-05}, {2.87462328369865e-08, 3.71766348750053e-08, 1.44556433170918e-07}},
{"DMC-GH4 iso 800",       "Panasonic",      "DMC-GH4",              800,         {2.64108727307494e-05, 1.55961456382241e-05, 5.13228136636857e-05}, {4.33791604086453e-08, 5.21953988366273e-08, 1.88838629687179e-07}},
{"DMC-GH4 iso 1000",       "Panasonic",      "DMC-GH4",              1000,         {3.3097144749982e-05, 1.95130962124486e-05, 6.4322539472493e-05}, {6.37890227255676e-08, 7.3746358326126e-08, 2.46821577327854e-07}},
{"DMC-GH4 iso 1250",       "Panasonic",      "DMC-GH4",              1250,         {4.14067547879603e-05, 2.45467429992145e-05, 8.08631096393149e-05}, {9.48056066483363e-08, 1.03016372563731e-07, 3.24658087091723e-07}},
{"DMC-GH4 iso 1600",       "Panasonic",      "DMC-GH4",              1600,         {5.21800229044833e-05, 3.08091007994307e-05, 0.000101618740454017}, {1.45763323743732e-07, 1.49037938015639e-07, 4.31574915557068e-07}},
{"DMC-GH4 iso 2000",       "Panasonic",      "DMC-GH4",              2000,         {6.57254662189985e-05, 3.88602024101656e-05, 0.000128125775756838}, {2.24863443228675e-07, 2.1708549714592e-07, 5.8253266568458e-07}},
{"DMC-GH4 iso 2500",       "Panasonic",      "DMC-GH4",              2500,         {8.28202305097857e-05, 4.9030051216045e-05, 0.000160833634196213}, {3.1968657739885e-07, 3.096518552241e-07, 7.5204870312405e-07}},
{"DMC-GH4 iso 3200",       "Panasonic",      "DMC-GH4",              3200,         {0.000103169381305759, 6.09167842923992e-05, 0.000201731828963695}, {5.13020423939375e-07, 4.78682770005182e-07, 1.07141372839345e-06}},
{"DMC-GH4 iso 4000",       "Panasonic",      "DMC-GH4",              4000,         {0.000129896124732458, 7.71056821105417e-05, 0.000254434496120469}, {7.67667696833172e-07, 7.02360071724437e-07, 1.42242937115439e-06}},
{"DMC-GH4 iso 5000",       "Panasonic",      "DMC-GH4",              5000,         {0.000163941127887316, 9.77289356597993e-05, 0.000321587452103538}, {1.15192595809846e-06, 1.05572691243744e-06, 2.01576577685986e-06}},
{"DMC-GH4 iso 6400",       "Panasonic",      "DMC-GH4",              6400,         {0.00020525511977133, 0.00012218698162368, 0.000402506408924346}, {1.83820009211279e-06, 1.62395409874353e-06, 2.78409498796949e-06}},
{"DMC-GH4 iso 8000",       "Panasonic",      "DMC-GH4",              8000,         {0.000259850556695459, 0.000155058752827724, 0.000503918727407543}, {2.68634142168977e-06, 2.40543359942463e-06, 3.71393577749521e-06}},
{"DMC-GH4 iso 10000",       "Panasonic",      "DMC-GH4",              10000,         {0.000329285829787183, 0.000197769538480885, 0.00064009745134942}, {4.04069571950847e-06, 3.68652624717883e-06, 5.49757324127808e-06}},
{"DMC-GH4 iso 12800",       "Panasonic",      "DMC-GH4",              12800,         {0.00042140545942068, 0.000254409388067434, 0.000810776898795268}, {5.9548884178483e-06, 5.34921527021815e-06, 7.40099814834264e-06}},
{"DMC-GH4 iso 16000",       "Panasonic",      "DMC-GH4",              16000,         {0.000268531127443933, 0.000163864641932368, 0.000517390881528932}, {2.3410450579859e-06, 2.00721775796183e-06, 2.8729251850247e-06}},
{"DMC-GH4 iso 20000",       "Panasonic",      "DMC-GH4",              20000,         {0.000349230909018073, 0.000216416307843024, 0.000668480357195119}, {3.34964933542287e-06, 2.92181598705319e-06, 3.96465031328341e-06}},
{"DMC-GH4 iso 25600",       "Panasonic",      "DMC-GH4",              25600,         {0.000457851034900599, 0.000294940622708204, 0.000849524101595582}, {4.47616636168629e-06, 3.81115999546647e-06, 5.08694264490566e-06}},

// panasonic gf1 by nicolas dade
  {"Panasonic DMC-GF1 iso 100",  "Panasonic", "DMC-GF1", 100,  {1.04535552878391e-05, 5.45502496125567e-06, 1.07671264778553e-05}, {7.58430227413935e-08, 8.86517305345379e-08, 1.04792248085416e-07}},
  {"Panasonic DMC-GF1 iso 125",  "Panasonic", "DMC-GF1", 125,  {1.33660974945872e-05, 7.43883491244048e-06, 1.41955834296851e-05}, {1.02720919579337e-07, -7.21874380860034e-09, 6.71210829756692e-08}},
  {"Panasonic DMC-GF1 iso 160",  "Panasonic", "DMC-GF1", 160,  {1.74145881201241e-05, 9.41928662795895e-06, 1.82042031627369e-05}, {8.88810318020993e-08, 6.96077756208576e-09, 9.81876530240034e-08}},
  {"Panasonic DMC-GF1 iso 200",  "Panasonic", "DMC-GF1", 200,  {1.97311801234147e-05, 1.03300818158606e-05, 2.03761898239435e-05}, {2.06878280289234e-07, 1.98222728988268e-07, 2.29428716199233e-07}},
  {"Panasonic DMC-GF1 iso 250",  "Panasonic", "DMC-GF1", 250,  {2.54496751221709e-05, 1.31956906919831e-05, 2.6118224824588e-05}, {2.16870617764627e-07, 2.04506487500374e-07, 2.53208588554574e-07}},
  {"Panasonic DMC-GF1 iso 320",  "Panasonic", "DMC-GF1", 320,  {3.20514144136813e-05, 1.67024861309939e-05, 3.29933136679525e-05}, {3.47563109674808e-07, 3.57730621126532e-07, 4.77842470428686e-07}},
  {"Panasonic DMC-GF1 iso 400",  "Panasonic", "DMC-GF1", 400,  {5.15947042393052e-05, 2.12991160119356e-05, 2.81727213865324e-05}, {5.41617295850964e-07, 3.96071156252656e-07, 6.01052759326755e-07}},
  {"Panasonic DMC-GF1 iso 500",  "Panasonic", "DMC-GF1", 500,  {6.29447239666803e-05, 2.67055882766265e-05, 3.58784545544703e-05}, {1.05684487186229e-06, 6.61518637141707e-07, 7.32572957408507e-07}},
  {"Panasonic DMC-GF1 iso 640",  "Panasonic", "DMC-GF1", 640,  {7.71256683840778e-05, 3.33200936565624e-05, 4.57790695089905e-05}, {1.79592304221405e-06, 1.09678784347683e-06, 1.06564986116275e-06}},
  {"Panasonic DMC-GF1 iso 800",  "Panasonic", "DMC-GF1", 800,  {0.000100921487755147, 4.51435026810075e-05, 6.0109237810212e-05}, {2.61452929378619e-06, 1.40832279354645e-06, 1.79553574612207e-06}},
  {"Panasonic DMC-GF1 iso 1000", "Panasonic", "DMC-GF1", 1000, {0.000136412761454267, 5.63694706153799e-05, 7.31438579488359e-05}, {3.24969700964785e-06, 2.31940240020084e-06, 3.26032669822112e-06}},
  {"Panasonic DMC-GF1 iso 1250", "Panasonic", "DMC-GF1", 1250, {0.00016525459872769, 7.08134943673697e-05, 9.47210128360164e-05}, {5.71739603396868e-06, 4.02191357158404e-06, 5.02841512874366e-06}},
  {"Panasonic DMC-GF1 iso 1600", "Panasonic", "DMC-GF1", 1600, {0.000212242436366194, 8.78138515170234e-05, 0.000121981542076272}, {8.60819610760796e-06, 6.73642784685995e-06, 7.36798440146139e-06}},
  {"Panasonic DMC-GF1 iso 2000", "Panasonic", "DMC-GF1", 2000, {0.000279968929974892, 0.000112178542367875, 0.000148489328185495}, {1.20184023109914e-05, 1.00672262333492e-05, 1.26152441721689e-05}},
  {"Panasonic DMC-GF1 iso 2500", "Panasonic", "DMC-GF1", 2500, {0.000342545378996356, 0.000139708669207157, 0.000188959354218091}, {1.93105681190276e-05, 1.66479338346392e-05, 1.96441195845495e-05}},
  {"Panasonic DMC-GF1 iso 3200", "Panasonic", "DMC-GF1", 3200, {0.000456941650563597, 0.000175563573427286, 0.000244623340362466}, {2.78449350381569e-05, 2.7250348714179e-05, 3.00689556113699e-05}},

// panasonic dmc gf3 by richard isted, iso 3200 and 6400 are pre-processed in camera and results are suboptimal there. rest is fine.
{"Panasonic DMC-GF3 iso 160",       "Panasonic",      "DMC-GF3",              160,         {2.02350626237294e-05, 8.65976784270689e-06, 1.34218258146732e-05}, {9.51454227507865e-08, 7.32022307137317e-08, 9.53154888380488e-08}},
{"Panasonic DMC-GF3 iso 200",       "Panasonic",      "DMC-GF3",              200,         {2.41648195440764e-05, 1.07312437413445e-05, 1.60582742985123e-05}, {9.74146050355725e-08, 7.8434776175889e-08, 9.70457615429697e-08}},
{"Panasonic DMC-GF3 iso 400",       "Panasonic",      "DMC-GF3",              400,         {4.52387882934033e-05, 1.98511037697891e-05, 2.99799553466259e-05}, {2.14679901847439e-07, 1.60523661664973e-07, 2.14794692825462e-07}},
{"Panasonic DMC-GF3 iso 800",       "Panasonic",      "DMC-GF3",              800,         {9.38257088021957e-05, 4.1623647930152e-05, 6.43000694373224e-05}, {6.34491520117409e-07, 4.82178892207265e-07, 6.35731046236163e-07}},
{"Panasonic DMC-GF3 iso 800",       "Panasonic",      "DMC-GF3",              800,         {9.36040424136415e-05, 4.14476287957258e-05, 6.374263337856e-05}, {6.13383691659087e-07, 4.52173305765121e-07, 6.15614674152588e-07}},
{"Panasonic DMC-GF3 iso 1600",       "Panasonic",      "DMC-GF3",              1600,         {0.000210363139417169, 9.19276975500467e-05, 0.00014819446226543}, {1.90724176686859e-06, 1.43695817501097e-06, 1.93171690573701e-06}},
{"Panasonic DMC-GF3 iso 3200",       "Panasonic",      "DMC-GF3",              3200,         {0.000529516578912191, 0.000238051867665224, 0.000408511059682062}, {5.60100338551534e-06, 4.20384956256743e-06, 5.65252949327065e-06}},
{"Panasonic DMC-GF3 iso 6400",       "Panasonic",      "DMC-GF3",              6400,         {0.001413516477831, 0.000775993642726022, 0.00128681546589859}, {1.57217158415014e-05, 1.28106429144672e-05, 1.51128183789506e-05}},

// panasonic gx1 contributed by richard wonka, red and blue weird. (kept here for an additional green channel sample for data mining)
//{"Panasonic DMC-GX1 iso 160",   "Panasonic", "DMC-GX1", 160,   {0.0, 7.72223529615256e-06, 0.0}, {0.0, -6.888766555649e-08,   0.0}},
//{"Panasonic DMC-GX1 iso 200",   "Panasonic", "DMC-GX1", 200,   {0.0, 9.64595616531457e-06, 0.0}, {0.0, -6.6569329741824e-08,  0.0}},
//{"Panasonic DMC-GX1 iso 250",   "Panasonic", "DMC-GX1", 250,   {0.0, 1.23507222355249e-05, 0.0}, {0.0, -4.11045186798127e-07, 0.0}},
//{"Panasonic DMC-GX1 iso 320",   "Panasonic", "DMC-GX1", 320,   {0.0, 1.51512657609805e-05, 0.0}, {0.0, -2.4749062934159e-07,  0.0}},
//{"Panasonic DMC-GX1 iso 400",   "Panasonic", "DMC-GX1", 400,   {0.0, 1.88723846858177e-05, 0.0}, {0.0, -4.46367750684392e-07, 0.0}},
//{"Panasonic DMC-GX1 iso 1600",  "Panasonic", "DMC-GX1", 1600,  {0.0, 5.99088263699075e-05, 0.0}, {0.0, 6.73267594796343e-07,  0.0}},
//{"Panasonic DMC-GX1 iso 2000",  "Panasonic", "DMC-GX1", 2000,  {0.0, 7.9722392572226e-05,  0.0}, {0.0, -5.28451446018427e-08, 0.0}},
//{"Panasonic DMC-GX1 iso 2500",  "Panasonic", "DMC-GX1", 2500,  {0.0, 9.94259118650839e-05, 0.0}, {0.0, 1.79072192077369e-07,  0.0}},
//{"Panasonic DMC-GX1 iso 3200",  "Panasonic", "DMC-GX1", 3200,  {0.0, 0.00012793752435239,  0.0}, {0.0, -2.20153349009731e-08, 0.0}},
//{"Panasonic DMC-GX1 iso 4000",  "Panasonic", "DMC-GX1", 4000,  {0.0, 0.000154709292531849, 0.0}, {0.0, 1.25260035328767e-06,  0.0}},
//{"Panasonic DMC-GX1 iso 5000",  "Panasonic", "DMC-GX1", 5000,  {0.0, 0.000195804178807441, 0.0}, {0.0, 4.69312211229742e-06,  0.0}},
//{"Panasonic DMC-GX1 iso 6400",  "Panasonic", "DMC-GX1", 6400,  {0.0, 0.000246609262698591, 0.0}, {0.0, 7.26908367124262e-06,  0.0}},
//{"Panasonic DMC-GX1 iso 8000",  "Panasonic", "DMC-GX1", 8000,  {0.0, 0.000317912383391281, 0.0}, {0.0, 1.22055037100857e-05,  0.0}},
//{"Panasonic DMC-GX1 iso 10000", "Panasonic", "DMC-GX1", 10000, {0.0, 0.000387863432430305, 0.0}, {0.0, 1.79704768639272e-05,  0.0}},
//{"Panasonic DMC-GX1 iso 12800", "Panasonic", "DMC-GX1", 12800, {0.0, 0.000490152181962666, 0.0}, {0.0, 2.59598697699368e-05,  0.0}},
// same with better matches for red and blue, and more steps in between:
  {"Panasonic DMC-GX1 iso 160",   "Panasonic", "DMC-GX1", 160,   {2.25610005942022e-05, 8.43102881725226e-06, 1.60520675069957e-05}, {-3.81560811538944e-07, -2.77360344328293e-07, 8.86697855024986e-08}},
  {"Panasonic DMC-GX1 iso 200",   "Panasonic", "DMC-GX1", 200,   {2.64124247426029e-05, 9.80349098201341e-06, 1.90490427779351e-05}, {-3.45821367697589e-07, -2.17627214382141e-07, 1.72211312138351e-07}},
  {"Panasonic DMC-GX1 iso 250",   "Panasonic", "DMC-GX1", 250,   {3.04927021440971e-05, 1.05331775232163e-05, 2.17866327137394e-05}, {-4.10742894091177e-07, -3.72527229796411e-08, 2.33617119577972e-07}},
  {"Panasonic DMC-GX1 iso 320",   "Panasonic", "DMC-GX1", 320,   {3.56189332573906e-05, 1.37113783284234e-05, 2.58485400989334e-05}, {-1.81928225710476e-07, -1.46131883236718e-07, 5.49658067981408e-07}},
  {"Panasonic DMC-GX1 iso 400",   "Panasonic", "DMC-GX1", 400,   {4.62444063194779e-05, 1.75382432080569e-05, 3.36961198983445e-05}, {-4.1704863421711e-07, -3.29120382807837e-07, 4.36779965811009e-07}},
// {"Panasonic DMC-GX1 iso 400",   "Panasonic", "DMC-GX1", 400,   {4.53512495035128e-05, 1.74709593097413e-05, 3.33606280048935e-05}, {-2.84202446139982e-07, -2.7327170729911e-07, 5.10172528577565e-07}},
  {"Panasonic DMC-GX1 iso 500",   "Panasonic", "DMC-GX1", 500,   {7.2570142083284e-05, 2.21989457357698e-05, 4.96628117423767e-05}, {-2.37989217590185e-06, -7.37581908416101e-07, -9.30612379303307e-07}},
  {"Panasonic DMC-GX1 iso 640",   "Panasonic", "DMC-GX1", 640,   {6.55440270361526e-05, 2.36422407747861e-05, 4.63067047150749e-05}, {-2.68014165295071e-07, 1.8294274402725e-07, 1.18690901328372e-06}},
  {"Panasonic DMC-GX1 iso 800",   "Panasonic", "DMC-GX1", 800,   {8.01519595070828e-05, 3.09384262328816e-05, 5.9098398003935e-05}, {-9.34688854443932e-08, -3.58382483395874e-08, 1.47268763808856e-06}},
  {"Panasonic DMC-GX1 iso 1000",  "Panasonic", "DMC-GX1", 1000,  {0.000123247564807667, 3.84566201864243e-05, 8.5051106151817e-05}, {-2.76274062436701e-06, -1.71948589094298e-07, -1.13275352526187e-07}},
  {"Panasonic DMC-GX1 iso 1250",  "Panasonic", "DMC-GX1", 1250,  {0.000126867722735507, 4.55060245187499e-05, 8.95192884835856e-05}, {-2.68295978720705e-07, 7.75180092914863e-07, 2.90126074003082e-06}},
  {"Panasonic DMC-GX1 iso 1600",  "Panasonic", "DMC-GX1", 1600,  {0.000152146399617487, 5.8565367817483e-05, 0.000112439969511265}, {5.95274434857656e-07, 6.48690196658333e-07, 3.58946664205093e-06}},
  {"Panasonic DMC-GX1 iso 2000",  "Panasonic", "DMC-GX1", 2000,  {0.000191954190776922, 7.6005589742941e-05, 0.000148594078412686}, {9.68533727234613e-07, 3.62096380831812e-07, 3.85198033722599e-06}},
  {"Panasonic DMC-GX1 iso 2500",  "Panasonic", "DMC-GX1", 2500,  {0.000236488194628525, 9.4602543046884e-05, 0.000182171711406015}, {1.88761417585411e-06, 9.14321382905128e-07, 5.7916931882301e-06}},
  {"Panasonic DMC-GX1 iso 3200",  "Panasonic", "DMC-GX1", 3200,  {0.000292292347694649, 0.000117110006860544, 0.000224665266940306}, {3.23408410626095e-06, 1.81856269561625e-06, 8.15529948346614e-06}},
  {"Panasonic DMC-GX1 iso 4000",  "Panasonic", "DMC-GX1", 4000,  {0.000364567828794302, 0.000150430874414097, 0.000278748564916136}, {5.16532360250294e-06, 2.41504542938176e-06, 1.14511026120252e-05}},
  {"Panasonic DMC-GX1 iso 5000",  "Panasonic", "DMC-GX1", 5000,  {0.000460872077087472, 0.000190389435926031, 0.000356577212561424}, {7.20108230213754e-06, 3.48474300718122e-06, 1.42981816344779e-05}},
  {"Panasonic DMC-GX1 iso 6400",  "Panasonic", "DMC-GX1", 6400,  {0.000567967306824856, 0.00023594015155701, 0.00044355050164276}, {1.28882597396064e-05, 6.68500438964432e-06, 2.08488399270575e-05}},
  {"Panasonic DMC-GX1 iso 8000",  "Panasonic", "DMC-GX1", 8000,  {0.000705109958420111, 0.000296468647931019, 0.000548088005072015}, {1.81668564557104e-05, 1.06561376035258e-05, 3.1007109089614e-05}},
  {"Panasonic DMC-GX1 iso 10000", "Panasonic", "DMC-GX1", 10000, {0.000816958070153822, 0.000380761554845624, 0.000691782505603086}, {3.32161214117429e-05, 1.46423072401747e-05, 4.21583308111487e-05}},
  {"Panasonic DMC-GX1 iso 12800", "Panasonic", "DMC-GX1", 12800, {0.00100449707889175, 0.000476975622368013, 0.000816127799702047}, {4.94242263731821e-05, 2.41924946503053e-05, 6.55448957102485e-05}},

// gx7 contributed by richard isted
// {"Panasonic DMC-GX7 iso 200",       "Panasonic",      "DMC-GX7",              200,         {1.54993685443704e-05, 6.05878465811322e-06, 1.10666762504456e-05}, {2.1878022193467e-08, 1.55868536176415e-08, 2.12394980802588e-08}},
// {"Panasonic DMC-GX7 iso 400",       "Panasonic",      "DMC-GX7",              400,         {3.27286454518961e-05, 1.23050620288822e-05, 2.24382697384105e-05}, {3.6316212737807e-08, 2.82056917011574e-08, 3.73701253156086e-08}},
// {"Panasonic DMC-GX7 iso 800",       "Panasonic",      "DMC-GX7",              800,         {6.41338234744364e-05, 2.44992700720629e-05, 4.50313154906175e-05}, {7.81564751559069e-08, 5.74983944626736e-08, 7.74886140724314e-08}},
// {"Panasonic DMC-GX7 iso 1600",       "Panasonic",      "DMC-GX7",              1600,         {0.000129603016923408, 4.98050170020576e-05, 9.10231124124235e-05}, {1.69119863080161e-07, 1.18466595565976e-07, 1.669142556179e-07}},
// {"Panasonic DMC-GX7 iso 3200",       "Panasonic",      "DMC-GX7",              3200,         {0.000252438843453206, 9.950285164147e-05, 0.00018404612022497}, {5.51221946370664e-07, 3.7082814826385e-07, 4.95427308271235e-07}},
// {"Panasonic DMC-GX7 iso 6400",       "Panasonic",      "DMC-GX7",              6400,         {0.000498510514474999, 0.000198556643047007, 0.000369330183005606}, {1.75280698753396e-06, 1.23149606790733e-06, 1.47537065230039e-06}},
// {"Panasonic DMC-GX7 iso 12800",       "Panasonic",      "DMC-GX7",              12800,         {0.000991920755920762, 0.000396709274022879, 0.000739251435885601}, {5.61729905561357e-06, 4.27583466481009e-06, 4.68537201885592e-06}},
// {"Panasonic DMC-GX7 iso 25600",       "Panasonic",      "DMC-GX7",              25600,         {0.00196703832775278, 0.000805912336992413, 0.00144098471686257}, {1.9984426907528e-05, 1.55489556456422e-05, 1.71051983928139e-05}},
// additional dmc-gx7 by thokster, slightly less aggressive:
{"DMC-GX7 iso 125",       "Panasonic",      "DMC-GX7",              125,         {1.02469553101861e-05, 3.86598354436075e-06, 6.62506511383392e-06}, {1.25894559914195e-08, 9.60576515276903e-09, 1.27805155908475e-08}},
{"DMC-GX7 iso 200",       "Panasonic",      "DMC-GX7",              200,         {1.71192468795171e-05, 6.28239883382605e-06, 1.05653687375902e-05}, {1.92717160849479e-08, 1.50577630791781e-08, 2.01668286955249e-08}},
{"DMC-GX7 iso 400",       "Panasonic",      "DMC-GX7",              400,         {3.32797877447743e-05, 1.21936872600359e-05, 2.06303600331317e-05}, {3.80008811340371e-08, 2.96586802371498e-08, 4.00360875572228e-08}},
{"DMC-GX7 iso 800",       "Panasonic",      "DMC-GX7",              800,         {6.4997122107329e-05, 2.39018586858697e-05, 4.00710925149001e-05}, {7.46972236753608e-08, 5.86600001018794e-08, 7.86614773542351e-08}},
{"DMC-GX7 iso 3200",       "Panasonic",      "DMC-GX7",              3200,         {0.000247362015434284, 9.23123084113172e-05, 0.000153104325402181}, {2.98744536833586e-07, 2.19187654751199e-07, 3.08254989654335e-07}},
{"DMC-GX7 iso 6400",       "Panasonic",      "DMC-GX7",              6400,         {0.000489123396382914, 0.000185753553079761, 0.000302319980408346}, {6.28973988933001e-07, 4.80381835172596e-07, 6.69281517432149e-07}},
{"DMC-GX7 iso 12800",       "Panasonic",      "DMC-GX7",              12800,         {0.00096454198599105, 0.000376697309273113, 0.00061339226802943}, {1.06889994305435e-06, 1.07990518200627e-06, 1.39608635033523e-06}},
{"DMC-GX7 iso 25600",       "Panasonic",      "DMC-GX7",              25600,         {0.00186787652101452, 0.000780445760065833, 0.00123670433194128}, {1.67498888633184e-06, 2.78109874037158e-06, 3.12080550951644e-06}},

// panasonic g3 contributed by thouks, red broken
  {"Panasonic DMC-G3 iso 160",  "Panasonic", "DMC-G3", 160,  {0.0, 7.23504210061448e-06, 1.08791003702389e-05}, {0.0, -3.85334833993451e-08, -1.31005693812467e-08}},
  {"Panasonic DMC-G3 iso 200",  "Panasonic", "DMC-G3", 200,  {0.0, 9.98324327029253e-06, 1.39948187221934e-05}, {0.0, -2.09817988614912e-07, -5.11638063765022e-08}},
  {"Panasonic DMC-G3 iso 250",  "Panasonic", "DMC-G3", 250,  {0.0, 3.18527336446393e-05, 1.71309119102642e-05}, {0.0, -5.15829140391381e-06, -1.56451957698504e-07}},
  {"Panasonic DMC-G3 iso 320",  "Panasonic", "DMC-G3", 320,  {0.0, 1.43066410925168e-05, 1.94722158409763e-05}, {0.0, -2.26632312525303e-07, 1.83957953662665e-07}},
  {"Panasonic DMC-G3 iso 400",  "Panasonic", "DMC-G3", 400,  {0.0, 1.64222707438293e-05, 2.48882583868023e-05}, {0.0, 8.87203705181001e-08, 1.12011794027408e-07}},
  {"Panasonic DMC-G3 iso 500",  "Panasonic", "DMC-G3", 500,  {0.0, 6.11129895087523e-05, 3.24963474561692e-05}, {0.0, -9.76023439686201e-06, -2.21766373976411e-07}},
  {"Panasonic DMC-G3 iso 640",  "Panasonic", "DMC-G3", 640,  {0.0, 2.8286218601342e-05, 3.67609406351971e-05},  {0.0, -6.57126573316964e-07, 4.0613804649811e-07}},
  {"Panasonic DMC-G3 iso 800",  "Panasonic", "DMC-G3", 800,  {0.0, 3.09387375886329e-05, 4.7198474561392e-05},  {0.0, 4.2487608072946e-07, 4.48582379518968e-07}},
  {"Panasonic DMC-G3 iso 1000", "Panasonic", "DMC-G3", 1000, {0.0, 8.1633700649718e-05, 5.88687614163853e-05},  {0.0, -1.00818432764847e-05, 3.14067308154704e-07}},
  {"Panasonic DMC-G3 iso 1250", "Panasonic", "DMC-G3", 1250, {0.0, 5.06732281115145e-05, 7.07780460295258e-05}, {0.0, -3.73578296665881e-08, 1.24541990983687e-06}},
  {"Panasonic DMC-G3 iso 1600", "Panasonic", "DMC-G3", 1600, {0.0, 5.88862862609876e-05, 9.07380073069054e-05}, {0.0, 1.4728234997545e-06, 1.41225918651852e-06}},
  {"Panasonic DMC-G3 iso 2000", "Panasonic", "DMC-G3", 2000, {0.0, 7.83313110144213e-05, 0.000116875878944293}, {0.0, 7.0882523738507e-07, 1.40265049571464e-06}},
  {"Panasonic DMC-G3 iso 2500", "Panasonic", "DMC-G3", 2500, {0.0, 9.99059484321908e-05, 0.000147860605487246}, {0.0, 9.11629739157505e-07, 2.08691888646595e-06}},
  {"Panasonic DMC-G3 iso 3200", "Panasonic", "DMC-G3", 3200, {0.0, 0.000125372496668539, 0.000185288208457497}, {0.0, 1.5566341458627e-06, 2.79283517039385e-06}},
  {"Panasonic DMC-G3 iso 4000", "Panasonic", "DMC-G3", 4000, {0.0, 0.000156707631513177, 0.000231048309067183}, {0.0, 2.25546383041305e-06, 4.4362542828725e-06}},
  {"Panasonic DMC-G3 iso 5000", "Panasonic", "DMC-G3", 5000, {0.0, 0.00019839248009273, 0.000292096855214219},  {0.0, 4.08832836539948e-06, 6.78485344147642e-06}},
  {"Panasonic DMC-G3 iso 6400", "Panasonic", "DMC-G3", 6400, {0.0, 0.000249740158932934, 0.000374226427397067}, {0.0, 6.63094833968962e-06, 1.00160735215934e-05}},

// panasonic g5 contributed by thouks, blue broken
  {"Panasonic DMC-G5 iso 160",   "Panasonic", "DMC-G5", 160,   {7.80364397000074e-06, 7.7976972435071e-06,  0.0}, {-2.14867324860313e-08, -1.75539665037712e-07, 0.0}},
  {"Panasonic DMC-G5 iso 200",   "Panasonic", "DMC-G5", 200,   {9.44452319121292e-06, 9.4444592358414e-06,  0.0}, {6.78616795247314e-09, -1.96410822967284e-07,  0.0}},
  {"Panasonic DMC-G5 iso 400",   "Panasonic", "DMC-G5", 400,   {1.74094996090056e-05, 1.70748329797573e-05, 0.0}, {1.10569313707048e-07, -2.30923685678883e-07,  0.0}},
  {"Panasonic DMC-G5 iso 800",   "Panasonic", "DMC-G5", 800,   {3.34043671261072e-05, 3.04614338355274e-05, 0.0}, {3.89487130287642e-07, 2.06393818486134e-07,   0.0}},
  {"Panasonic DMC-G5 iso 1600",  "Panasonic", "DMC-G5", 1600,  {6.69038937292441e-05, 6.12120126047493e-05, 0.0}, {1.11664075775327e-06, 4.66321953950977e-07,   0.0}},
  {"Panasonic DMC-G5 iso 3200",  "Panasonic", "DMC-G5", 3200,  {0.000135240849791888, 0.00011707202139752,  0.0}, {3.40449630528109e-06, 3.46287388082615e-06,   0.0}},
  {"Panasonic DMC-G5 iso 6400",  "Panasonic", "DMC-G5", 6400,  {0.000285643797378891, 0.00022998234345618,  0.0}, {9.97026670397084e-06, 1.3180761692217e-05,    0.0}},
  {"Panasonic DMC-G5 iso 12800", "Panasonic", "DMC-G5", 12800, {0.00061744310156768, 0.000475796782281499,  0.0}, {3.21295060558385e-05, 4.42969155936244e-05,   0.0}},

// -------------------------------------------------------------------
// PENTAX.
// -------------------------------------------------------------------

// pentax *ist DL by simon harhues
{"PENTAX *ist DL iso 200",       "PENTAX Corporation",      "PENTAX *ist DL",              200,         {2.99163379217187e-06, 3.83163191739904e-06, 7.52885680139871e-06}, {4.28245746973165e-08, 1.90428204192517e-08, 4.23572635614955e-08}},
// {"PENTAX *ist DL iso 200",       "PENTAX Corporation",      "PENTAX *ist DL",              200,         {3.75642334433331e-06, 3.52036458465918e-06, 8.62867250980668e-06}, {4.00801358138718e-08, 3.43148265705634e-08, 3.64448197110518e-08}},
{"PENTAX *ist DL iso 400",       "PENTAX Corporation",      "PENTAX *ist DL",              400,         {7.73829814472718e-06, 4.58320726362355e-06, 6.63863989505961e-06}, {7.03041975818957e-08, 4.29403005233088e-08, 7.33245569013743e-08}},
// {"PENTAX *ist DL iso 400",       "PENTAX Corporation",      "PENTAX *ist DL",              400,         {6.14687287568041e-06, 5.14472172974309e-06, 8.86632630549476e-06}, {5.84274598683322e-08, 4.69318991679077e-08, 7.88079165408211e-08}},
// {"PENTAX *ist DL iso 400",       "PENTAX Corporation",      "PENTAX *ist DL",              400,         {4.66911547604429e-06, 4.69927275413539e-06, 1.08170077581298e-05}, {6.77661825632395e-08, 4.52281117689964e-08, 6.92812807563583e-08}},
// {"PENTAX *ist DL iso 400",       "PENTAX Corporation",      "PENTAX *ist DL",              400,         {4.14608569460242e-06, 4.92733911677844e-06, 1.20556537221338e-05}, {7.23846371157344e-08, 4.43900140725381e-08, 6.60496371370306e-08}},
{"PENTAX *ist DL iso 800",       "PENTAX Corporation",      "PENTAX *ist DL",              800,         {1.56235116891899e-05, 8.78729801778483e-06, 1.30000050161256e-05}, {2.50664653626102e-07, 1.79085901856254e-07, 2.6855239841425e-07}},
// {"PENTAX *ist DL iso 800",       "PENTAX Corporation",      "PENTAX *ist DL",              800,         {1.62929158949367e-05, 9.40189214528963e-06, 1.34829848752077e-05}, {2.57967143564267e-07, 1.66326686905415e-07, 2.62306848644228e-07}},
{"PENTAX *ist DL iso 1600",       "PENTAX Corporation",      "PENTAX *ist DL",              1600,         {3.35175535377488e-05, 1.7967468239465e-05, 2.54935456686573e-05}, {8.81630812108316e-07, 6.17153159387754e-07, 9.32238812149751e-07}},
// {"PENTAX *ist DL iso 1600",       "PENTAX Corporation",      "PENTAX *ist DL",              1600,         {3.29024075857362e-05, 1.73981705030019e-05, 2.41320859670995e-05}, {9.15350589659632e-07, 6.39769816590013e-07, 9.91040555900127e-07}},
// {"PENTAX *ist DL iso 1600",       "PENTAX Corporation",      "PENTAX *ist DL",              1600,         {3.44102905162781e-05, 2.04498734204619e-05, 3.38381496237146e-05}, {1.31428744257887e-06, 9.41268012297171e-07, 1.39638254986501e-06}},
// {"PENTAX *ist DL iso 1600",       "PENTAX Corporation",      "PENTAX *ist DL",              1600,         {3.07868922778192e-05, 1.75166336967868e-05, 2.85234543793148e-05}, {9.37692677364703e-07, 6.53008807385743e-07, 9.48132497065059e-07}},
{"PENTAX *ist DL iso 3200",       "PENTAX Corporation",      "PENTAX *ist DL",              3200,         {7.32931797676231e-05, 3.51194390052015e-05, 5.7064239608782e-05}, {3.24259468845176e-06, 2.3827631264612e-06, 3.47623151456767e-06}},
// {"PENTAX *ist DL iso 3200",       "PENTAX Corporation",      "PENTAX *ist DL",              3200,         {7.27242956520745e-05, 3.57922123946732e-05, 4.83626767612554e-05}, {3.20991248459299e-06, 2.34323953286627e-06, 3.45544230554926e-06}},
// {"PENTAX *ist DL iso 3200",       "PENTAX Corporation",      "PENTAX *ist DL",              3200,         {6.47643757904682e-05, 3.47044098167856e-05, 5.28528714396698e-05}, {3.60842436898298e-06, 2.55861098565413e-06, 3.7839180628943e-06}},
// {"PENTAX *ist DL iso 3200",       "PENTAX Corporation",      "PENTAX *ist DL",              3200,         {6.26361802020194e-05, 3.4289670525043e-05, 5.12948231066463e-05}, {4.0212359199225e-06, 2.76432312537909e-06, 4.31794471437447e-06}},

// pentax k20d contributed by Florian Franzmann
{"PENTAX K20D iso 100",       "PENTAX",      "PENTAX K20D",              100,         {4.03512079837188e-06, 2.31310823952034e-06, 3.72654785194799e-06}, {4.48434387114298e-08, 3.59171630863747e-08, 4.56183722901822e-08}},
{"PENTAX K20D iso 125",       "PENTAX",      "PENTAX K20D",              125,         {5.201751914679e-06, 2.93447897646261e-06, 4.70203086903034e-06}, {5.51586048674761e-08, 4.1168171112395e-08, 5.56619859856805e-08}},
{"PENTAX K20D iso 160",       "PENTAX",      "PENTAX K20D",              160,         {6.29295714473944e-06, 3.49924893556453e-06, 5.62152351266612e-06}, {6.76420341983647e-08, 4.90669740481894e-08, 6.76521653090315e-08}},
{"PENTAX K20D iso 200",       "PENTAX",      "PENTAX K20D",              200,         {8.36863306228191e-06, 4.64006701611217e-06, 7.51534296744729e-06}, {9.67845541105706e-08, 6.67936355023338e-08, 9.45095552388911e-08}},
{"PENTAX K20D iso 250",       "PENTAX",      "PENTAX K20D",              250,         {1.02598391710906e-05, 5.71243845496259e-06, 9.31978483791779e-06}, {7.28223102970396e-08, 4.89716302975205e-08, 7.03090994530299e-08}},
{"PENTAX K20D iso 320",       "PENTAX",      "PENTAX K20D",              320,         {1.2460779459358e-05, 6.84553688565498e-06, 1.11664130967319e-05}, {9.3108993451478e-08, 6.46385683503627e-08, 8.92159897359798e-08}},
{"PENTAX K20D iso 400",       "PENTAX",      "PENTAX K20D",              400,         {1.6503395079422e-05, 9.06424071661894e-06, 1.46971500377345e-05}, {1.40440897697093e-07, 9.19813497432329e-08, 1.33674878147282e-07}},
{"PENTAX K20D iso 500",       "PENTAX",      "PENTAX K20D",              500,         {2.1473774622676e-05, 1.16597569465889e-05, 1.9069840986989e-05}, {1.28465107055042e-07, 7.85905654562589e-08, 1.17590444732399e-07}},
{"PENTAX K20D iso 640",       "PENTAX",      "PENTAX K20D",              640,         {2.62894636006405e-05, 1.42208704362837e-05, 2.3343784651238e-05}, {1.64360113581119e-07, 9.73924222706565e-08, 1.49348924884942e-07}},
{"PENTAX K20D iso 800",       "PENTAX",      "PENTAX K20D",              800,         {3.51393604617489e-05, 1.91534616164118e-05, 3.12169746424556e-05}, {2.74407043557864e-07, 1.69204012209104e-07, 2.5635412197271e-07}},
{"PENTAX K20D iso 1000",       "PENTAX",      "PENTAX K20D",              1000,         {4.21682824368782e-05, 2.29656811046294e-05, 3.81753379451198e-05}, {2.31363501898136e-07, 1.28343770414844e-07, 2.12761519015389e-07}},
{"PENTAX K20D iso 1250",       "PENTAX",      "PENTAX K20D",              1250,         {5.05940823804796e-05, 2.69710427393341e-05, 4.50017313659246e-05}, {2.78248427949641e-07, 1.49382238760523e-07, 2.55724328099266e-07}},
{"PENTAX K20D iso 1600",       "PENTAX",      "PENTAX K20D",              1600,         {7.10078058397053e-05, 3.68060356885546e-05, 6.13617243424204e-05}, {4.11721192260525e-07, 2.24790407121514e-07, 3.90255156513248e-07}},
{"PENTAX K20D iso 2000",       "PENTAX",      "PENTAX K20D",              2000,         {9.34542169290652e-05, 4.79600623094616e-05, 8.07012798898809e-05}, {5.37711955551401e-07, 3.04395806294451e-07, 5.08726304543126e-07}},
{"PENTAX K20D iso 2500",       "PENTAX",      "PENTAX K20D",              2500,         {0.000118177441558288, 6.07837422585803e-05, 0.000101526854695375}, {7.42774488753386e-07, 4.50910063926644e-07, 7.30996010101479e-07}},
{"PENTAX K20D iso 3200",       "PENTAX",      "PENTAX K20D",              3200,         {0.000172615575693433, 8.49494396299149e-05, 0.000142834531768243}, {1.27382575291531e-06, 8.50931252241583e-07, 1.27200334560935e-06}},
{"PENTAX K20D iso 4000",       "PENTAX",      "PENTAX K20D",              4000,         {0.000244723130012044, 0.000117893415466082, 0.000197312055848286}, {1.84780006766473e-06, 1.24766326597173e-06, 1.90654535565937e-06}},
{"PENTAX K20D iso 5000",       "PENTAX",      "PENTAX K20D",              5000,         {0.0003141990739661, 0.000150771884387404, 0.000250331988164976}, {2.42421710247928e-06, 1.67669133957508e-06, 2.52662754383186e-06}},
{"PENTAX K20D iso 6400",       "PENTAX",      "PENTAX K20D",              6400,         {0.000468226779602687, 0.000215868244182429, 0.00036025931577558}, {3.80264024175038e-06, 2.76883350732546e-06, 4.11544722737646e-06}},

// pentax k200d contributed by Réza Meralli
  {"PENTAX K200D iso 100",  "PENTAX", "PENTAX K200D", 100,  {5.00045502819751e-06, 3.85316445963773e-06, 1.00379864507032e-05}, {-5.56992903182943e-08, -1.03834331573044e-07, -1.3697793717953e-07}},
  {"PENTAX K200D iso 200",  "PENTAX", "PENTAX K200D", 200,  {9.56294041351012e-06, 6.50524026205696e-06, 1.38608013732202e-05}, {-5.39961950121469e-08, 5.63133308618009e-08, 5.34612219608483e-07}},
  {"PENTAX K200D iso 400",  "PENTAX", "PENTAX K200D", 400,  {1.71610506310794e-05, 1.2700452200367e-05, 1.88448309727875e-05}, {1.3531487540378e-07, 3.47167879047707e-08, 2.21981756143856e-06}},
  {"PENTAX K200D iso 800",  "PENTAX", "PENTAX K200D", 800,  {2.6532373184506e-05, 1.97873258203881e-05, 3.23516896929341e-05}, {-1.9629366578538e-07, -2.0131442359146e-07, 2.98480770921618e-06}},
  {"PENTAX K200D iso 1600", "PENTAX", "PENTAX K200D", 1600, {4.81712141005253e-05, 3.97699126522855e-05, 4.32329378328961e-05}, {4.29605900325977e-07, -7.62108836847819e-07, 8.37957014442456e-06}},

// pentax k-7 contributed by elliott, blue channel broken
  {"PENTAX K-7 iso 100",  "PENTAX", "PENTAX K-7", 100,  {4.27094150065464e-06, 3.87740034111406e-06, 0.0}, {1.81430298382862e-08, -5.85689284091062e-08,  0.0}},
  {"PENTAX K-7 iso 160",  "PENTAX", "PENTAX K-7", 160,  {6.32605601552533e-06, 5.79050413229674e-06, 0.0}, {8.24331972830702e-08, -3.49510920058497e-08,  0.0}},
  {"PENTAX K-7 iso 200",  "PENTAX", "PENTAX K-7", 200,  {8.49171202242796e-06, 7.52341331928086e-06, 0.0}, {-3.23012279203106e-09, -1.16683528731805e-07, 0.0}},
  {"PENTAX K-7 iso 250",  "PENTAX", "PENTAX K-7", 250,  {1.05063521430323e-05, 9.57691175446278e-06, 0.0}, {6.22191768561298e-08, -1.47688359618983e-07,  0.0}},
  {"PENTAX K-7 iso 320",  "PENTAX", "PENTAX K-7", 320,  {1.20655330510873e-05, 1.11613997720898e-05, 0.0}, {1.48274329267279e-07, -1.28492436185813e-07,  0.0}},
  {"PENTAX K-7 iso 400",  "PENTAX", "PENTAX K-7", 400,  {1.54954238146394e-05, 1.39426126559638e-05, 0.0}, {1.38366294702248e-07, -1.08827510681911e-07,  0.0}},
  {"PENTAX K-7 iso 500",  "PENTAX", "PENTAX K-7", 500,  {1.99366691911348e-05, 1.80802072918949e-05, 0.0}, {2.6275403306085e-07, -2.21628487995168e-07,   0.0}},
  {"PENTAX K-7 iso 640",  "PENTAX", "PENTAX K-7", 640,  {2.34939275220807e-05, 2.08665236959291e-05, 0.0}, {4.18959919484903e-07, -4.68463679638996e-08,  0.0}},
  {"PENTAX K-7 iso 800",  "PENTAX", "PENTAX K-7", 800,  {3.24443713875635e-05, 2.76060239730679e-05, 0.0}, {5.68711912830384e-07, 1.61887533152749e-07,   0.0}},
  {"PENTAX K-7 iso 1000", "PENTAX", "PENTAX K-7", 1000, {4.03680750079107e-05, 3.41587204728668e-05, 0.0}, {1.13396003398658e-06, 7.5479393850128e-07,    0.0}},
  {"PENTAX K-7 iso 1250", "PENTAX", "PENTAX K-7", 1250, {4.82170133991815e-05, 4.09899451883279e-05, 0.0}, {1.61527442872177e-06, 9.05859196618094e-07,   0.0}},
  {"PENTAX K-7 iso 1600", "PENTAX", "PENTAX K-7", 1600, {6.38729120940155e-05, 5.21408914527437e-05, 0.0}, {3.02846222316358e-06, 2.70325939608082e-06,   0.0}},
  {"PENTAX K-7 iso 2000", "PENTAX", "PENTAX K-7", 2000, {4.71124731002744e-05, 3.86406604194635e-05, 0.0}, {1.83299986782294e-06, 1.73645246806797e-06,   0.0}},
  {"PENTAX K-7 iso 2500", "PENTAX", "PENTAX K-7", 2500, {5.6679128407337e-05, 4.66825013820314e-05,  0.0}, {2.80805710418134e-06, 2.52114016037366e-06,   0.0}},
  {"PENTAX K-7 iso 3200", "PENTAX", "PENTAX K-7", 3200, {7.8819453907085e-05, 6.07965689471899e-05,  0.0}, {4.73766848134806e-06, 4.85417847929952e-06,   0.0}},
  {"PENTAX K-7 iso 4000", "PENTAX", "PENTAX K-7", 4000, {0.000108200391654255, 8.12095071632062e-05, 0.0}, {8.68465780012167e-06, 9.05483764740589e-06,   0.0}},
  {"PENTAX K-7 iso 5000", "PENTAX", "PENTAX K-7", 5000, {0.000135356844756659, 0.000102503035430278, 0.0}, {1.26684856109744e-05, 1.27199120268548e-05,   0.0}},
  {"PENTAX K-7 iso 6400", "PENTAX", "PENTAX K-7", 6400, {0.00019878664913378, 0.000134950719089934,  0.0}, {2.00316908449782e-05, 2.36038806789022e-05,   0.0}},

// pentax k-5 contributed by elliott
  {"PENTAX K-5 iso 80",    "PENTAX", "PENTAX K-5", 80,    {2.03116436785254e-06, 2.03116436785254e-06, 2.03116436785254e-06}, {-3.02041906259105e-08, -3.02041906259105e-08, -3.02041906259105e-08}},
  {"PENTAX K-5 iso 100",   "PENTAX", "PENTAX K-5", 100,   {2.24972405550644e-06, 2.24972405550644e-06, 2.24972405550644e-06}, {-1.73907843875546e-08, -1.73907843875546e-08, -1.73907843875546e-08}},
  {"PENTAX K-5 iso 200",   "PENTAX", "PENTAX K-5", 200,   {4.35081569929625e-06, 4.35081569929625e-06, 4.35081569929625e-06}, {-3.53505222478958e-08, -3.53505222478958e-08, -3.53505222478958e-08}},
  {"PENTAX K-5 iso 400",   "PENTAX", "PENTAX K-5", 400,   {8.61579206202407e-06, 8.61579206202407e-06, 8.61579206202407e-06}, {-5.77072212691143e-08, -5.77072212691143e-08, -5.77072212691143e-08}},
  {"PENTAX K-5 iso 800",   "PENTAX", "PENTAX K-5", 800,   {1.71175144229784e-05, 1.71175144229784e-05, 1.71175144229784e-05}, {-1.46590586063609e-08, -1.46590586063609e-08, -1.46590586063609e-08}},
  {"PENTAX K-5 iso 1600",  "PENTAX", "PENTAX K-5", 1600,  {3.2697044230099e-05, 3.2697044230099e-05, 3.2697044230099e-05}, {3.31216462704584e-08, 3.31216462704584e-08, 3.31216462704584e-08}},
  {"PENTAX K-5 iso 3200",  "PENTAX", "PENTAX K-5", 3200,  {4.43416642030573e-05, 4.43416642030573e-05, 4.43416642030573e-05}, {-4.24937607243593e-07, -4.24937607243593e-07, -4.24937607243593e-07}},
  {"PENTAX K-5 iso 6400",  "PENTAX", "PENTAX K-5", 6400,  {8.71489939508059e-05, 8.71489939508059e-05, 8.71489939508059e-05}, {-1.07266764686662e-06, -1.07266764686662e-06, -1.07266764686662e-06}},
  {"PENTAX K-5 iso 12800", "PENTAX", "PENTAX K-5", 12800, {0.000164985441637024, 0.000164985441637024, 0.000164985441637024}, {-9.70017976316732e-09, -9.70017976316732e-09, -9.70017976316732e-09}},
  {"PENTAX K-5 iso 25600", "PENTAX", "PENTAX K-5", 25600, {0.000306721894329492, 0.000306721894329492, 0.000306721894329492}, {7.01957276944239e-06, 7.01957276944239e-06, 7.01957276944239e-06}},
  {"PENTAX K-5 iso 51200", "PENTAX", "PENTAX K-5", 51200, {0.000576230159234024, 0.000576230159234024, 0.000576230159234024}, {4.78351150568581e-05, 4.78351150568581e-05, 4.78351150568581e-05}},

// pentax k-5 ii contributed by michael born
  {"PENTAX K-5 II s iso 80",    "PENTAX", "PENTAX K-5 II s", 80,    {6.06823738661909e-06, 2.5060084130828e-06, 3.99271331379763e-06}, {-9.60380898281594e-08, -1.13645431623165e-07, -1.19456447655131e-07}},
  {"PENTAX K-5 II s iso 100",   "PENTAX", "PENTAX K-5 II s", 100,   {7.55220183397548e-06, 2.94109201892952e-06, 4.86172198833939e-06}, {-1.31982085096648e-07, -1.07569608401725e-07, -1.07416629535595e-07}},
  {"PENTAX K-5 II s iso 200",   "PENTAX", "PENTAX K-5 II s", 200,   {1.3610031439386e-05, 5.25468929066328e-06, 8.78767467569257e-06}, {-1.85428671703909e-07, -1.30436894688551e-07, -1.41318377532799e-07}},
  {"PENTAX K-5 II s iso 400",   "PENTAX", "PENTAX K-5 II s", 400,   {2.54148453721959e-05, 9.89549495343958e-06, 1.65951515741497e-05}, {-2.42444175989697e-07, -1.70088479301098e-07, -1.66323449829236e-07}},
  {"PENTAX K-5 II s iso 800",   "PENTAX", "PENTAX K-5 II s", 800,   {4.9996186717934e-05, 1.96609588829684e-05, 3.37952397526469e-05}, {-3.42161534135391e-07, -3.20880860241239e-07, -3.80437799089494e-07}},
  {"PENTAX K-5 II s iso 1600",  "PENTAX", "PENTAX K-5 II s", 1600,  {9.18189441746795e-05, 3.50893899720135e-05, 6.36495490949706e-05}, {-4.20099504160284e-07, -6.45838643242076e-08, -5.58448717451745e-07}},
  {"PENTAX K-5 II s iso 3200",  "PENTAX", "PENTAX K-5 II s", 3200,  {0.000111333803948486, 4.14219566203499e-05, 7.73662855972626e-05}, {-2.78412228726629e-07, 2.91474040656078e-07, -2.66248386068056e-07}},
  {"PENTAX K-5 II s iso 6400",  "PENTAX", "PENTAX K-5 II s", 6400,  {0.000215158106784479, 8.51631591137548e-05, 0.000152943712831637}, {8.39249447851446e-08, -1.18406848790585e-07, -2.21008260889656e-07}},
  {"PENTAX K-5 II s iso 12800", "PENTAX", "PENTAX K-5 II s", 12800, {0.000427149884973211, 0.000173502221307025, 0.000317471812122481}, {-1.68498549545253e-07, -2.64257551832908e-07, -1.260501653136e-06}},
  {"PENTAX K-5 II s iso 25600", "PENTAX", "PENTAX K-5 II s", 25600, {0.000786806104254908, 0.000318444769127179, 0.000580623362967988}, {1.07605618497251e-05, 8.85524358979821e-06, 1.05973086686003e-05}},
  {"PENTAX K-5 II s iso 51200", "PENTAX", "PENTAX K-5 II s", 51200, {0.00178921977431309, 0.000654317011303329, 0.00118282192145807}, {1.94167051650271e-05, 4.30029801799139e-05, 4.38118211917756e-05}},

// additional k-5 ii by christopher engelhard
  {"PENTAX K-5 II iso 80",    "PENTAX", "PENTAX K-5 II", 80,    {4.78294546465367e-06, 2.08486530704051e-06, 3.51401778098885e-06}, {-4.34589971058721e-09, -4.95180937705658e-08, -8.47645776979705e-08}},
// {"PENTAX K-5 II iso 80",    "PENTAX", "PENTAX K-5 II", 80,    {4.86432599236703e-06, 2.08542387701303e-06, 3.44005986761032e-06}, {-1.3940907154804e-08, -5.03951690624912e-08, -7.61915695520051e-08}},
// {"PENTAX K-5 II iso 80",    "PENTAX", "PENTAX K-5 II", 80,    {4.93069850107309e-06, 2.13845272739553e-06, 3.48656532654273e-06}, {-1.60182093922847e-08, -6.1646078510094e-08, -8.34047952122581e-08}},
// {"PENTAX K-5 II iso 80",    "PENTAX", "PENTAX K-5 II", 80,    {4.79096746298686e-06, 2.10593353035851e-06, 3.50088909947083e-06}, {-2.83931545664069e-09, -5.36525449234966e-08, -8.25419967224215e-08}},
  {"PENTAX K-5 II iso 100",   "PENTAX", "PENTAX K-5 II", 100,   {5.86500053280523e-06, 2.47313309788663e-06, 4.18229697055565e-06}, {-1.10541687126322e-08, -4.94908408144217e-08, -9.02378850234444e-08}},
// {"PENTAX K-5 II iso 100",   "PENTAX", "PENTAX K-5 II", 100,   {5.85589965754688e-06, 2.50671397687113e-06, 4.22747459053795e-06}, {-8.17109101418034e-09, -5.47374289227434e-08, -9.32579191580843e-08}},
  {"PENTAX K-5 II iso 125",   "PENTAX", "PENTAX K-5 II", 125,   {7.47426266003914e-06, 3.15345881237758e-06, 5.11897178617216e-06}, {-1.3815286261535e-08, -6.3535163955169e-08, -8.37642710467731e-08}},
// {"PENTAX K-5 II iso 125",   "PENTAX", "PENTAX K-5 II", 125,   {7.46423965971055e-06, 3.14637638708558e-06, 5.25722774834426e-06}, {-1.18055159209674e-08, -6.27516046267984e-08, -1.01239380795091e-07}},
  {"PENTAX K-5 II iso 160",   "PENTAX", "PENTAX K-5 II", 160,   {8.65810171418877e-06, 3.65959435677312e-06, 6.11666468658166e-06}, {-4.37594149263634e-09, -6.66894854477789e-08, -1.09739926612696e-07}},
// {"PENTAX K-5 II iso 160",   "PENTAX", "PENTAX K-5 II", 160,   {8.48833369485954e-06, 3.57543343851018e-06, 5.99700089805203e-06}, {1.58569713241111e-08, -4.05904500928927e-08, -8.4747158288292e-08}},
  {"PENTAX K-5 II iso 200",   "PENTAX", "PENTAX K-5 II", 200,   {1.09944912032922e-05, 4.60232039330261e-06, 7.61128001072729e-06}, {9.95691678222724e-09, -6.35351158272883e-08, -1.12105466258389e-07}},
// {"PENTAX K-5 II iso 200",   "PENTAX", "PENTAX K-5 II", 200,   {1.11676657328593e-05, 4.62062924291593e-06, 7.54658035799819e-06}, {-8.67714476618877e-09, -7.14904440544012e-08, -1.04166155725135e-07}},
  {"PENTAX K-5 II iso 250",   "PENTAX", "PENTAX K-5 II", 250,   {1.37669526591998e-05, 5.92272357249048e-06, 9.80642841283829e-06}, {5.42954733193063e-08, -8.85072422741786e-08, -1.48795229037757e-07}},
// {"PENTAX K-5 II iso 250",   "PENTAX", "PENTAX K-5 II", 250,   {1.36454474977586e-05, 5.73074427093349e-06, 9.51305218763014e-06}, {6.51928293953681e-08, -4.25621191618167e-08, -1.13278272073509e-07}},
  {"PENTAX K-5 II iso 320",   "PENTAX", "PENTAX K-5 II", 320,   {1.62588692495727e-05, 6.71760221002206e-06, 1.12632662110618e-05}, {6.42175417021587e-08, -4.64052574526122e-08, -1.3339641369302e-07}},
// {"PENTAX K-5 II iso 320",   "PENTAX", "PENTAX K-5 II", 320,   {1.58823701897494e-05, 6.66868275644033e-06, 1.13756606778284e-05}, {9.68400658504809e-08, -3.06880071402537e-08, -1.40468811841592e-07}},
  {"PENTAX K-5 II iso 400",   "PENTAX", "PENTAX K-5 II", 400,   {2.13239388389127e-05, 8.80064492009721e-06, 1.44489410765095e-05}, {6.24280994167059e-08, -8.39951381633678e-08, -1.49605315485404e-07}},
// {"PENTAX K-5 II iso 400",   "PENTAX", "PENTAX K-5 II", 400,   {2.11518749242789e-05, 8.62024939144104e-06, 1.44645432517382e-05}, {8.05667373014051e-08, -3.11565325099622e-08, -1.4705960101443e-07}},
  {"PENTAX K-5 II iso 500",   "PENTAX", "PENTAX K-5 II", 500,   {2.65391992688552e-05, 1.1024824824261e-05, 1.79282902147238e-05}, {1.49675448698581e-07, -4.51061205763106e-08, -7.93536199066525e-08}},
// {"PENTAX K-5 II iso 500",   "PENTAX", "PENTAX K-5 II", 500,   {2.71204001746073e-05, 1.10781655736404e-05, 1.84452101259221e-05}, {9.80889505694825e-08, -4.7703930700544e-08, -1.47528319560386e-07}},
  {"PENTAX K-5 II iso 640",   "PENTAX", "PENTAX K-5 II", 640,   {3.25030573517831e-05, 1.3155436737329e-05, 2.20567781621566e-05}, {5.10424634315293e-08, -7.70874750148514e-08, -1.90557070340964e-07}},
// {"PENTAX K-5 II iso 640",   "PENTAX", "PENTAX K-5 II", 640,   {3.15685980272177e-05, 1.28390152060865e-05, 2.13491638381594e-05}, {1.58115798060782e-07, -6.62898699874222e-09, -7.89430655005123e-08}},
  {"PENTAX K-5 II iso 800",   "PENTAX", "PENTAX K-5 II", 800,   {4.21844729431497e-05, 1.72293692424274e-05, 2.83618961124141e-05}, {8.97357255084967e-08, -1.16836813153619e-07, -1.85493730090582e-07}},
// {"PENTAX K-5 II iso 800",   "PENTAX", "PENTAX K-5 II", 800,   {4.05730536156761e-05, 1.67607576443425e-05, 2.8069359381357e-05}, {2.51491362533344e-07, 6.66667255333999e-09, -1.38911938054971e-07}},
  {"PENTAX K-5 II iso 1000",  "PENTAX", "PENTAX K-5 II", 1000,  {5.25063550655669e-05, 2.18768614147221e-05, 3.67755668943162e-05}, {3.0473680691554e-07, -4.64879102869052e-08, -2.42443988479712e-07}},
// {"PENTAX K-5 II iso 1000",  "PENTAX", "PENTAX K-5 II", 1000,  {5.30905679698663e-05, 2.18614966529888e-05, 3.65703269980508e-05}, {2.57433293946385e-07, -1.89014371283348e-08, -2.1798714686412e-07}},
  {"PENTAX K-5 II iso 1250",  "PENTAX", "PENTAX K-5 II", 1250,  {6.31327193341209e-05, 2.58901392809804e-05, 4.34390617126956e-05}, {2.10978041050702e-07, -8.4518824425154e-08, -2.63858540756829e-07}},
// {"PENTAX K-5 II iso 1250",  "PENTAX", "PENTAX K-5 II", 1250,  {6.40197713142907e-05, 2.62379109682787e-05, 4.33603775856626e-05}, {1.35542501124817e-07, -1.60016486896746e-07, -2.46462151382923e-07}},
  {"PENTAX K-5 II iso 1600",  "PENTAX", "PENTAX K-5 II", 1600,  {7.84315032505991e-05, 3.27092198591803e-05, 5.59442585301799e-05}, {2.90075176421845e-07, -1.68874798444088e-07, -5.0996861267781e-07}},
// {"PENTAX K-5 II iso 1600",  "PENTAX", "PENTAX K-5 II", 1600,  {7.84419634166111e-05, 3.23685473650792e-05, 5.48959830930567e-05}, {3.22687439822532e-07, -8.94186016330655e-08, -3.54992084522751e-07}},
  {"PENTAX K-5 II iso 2000",  "PENTAX", "PENTAX K-5 II", 2000,  {6.34317273279809e-05, 2.61214653222634e-05, 4.35979279346638e-05}, {1.46693066653346e-07, -1.97883137204267e-07, -3.44017096249006e-07}},
// {"PENTAX K-5 II iso 2000",  "PENTAX", "PENTAX K-5 II", 2000,  {6.55430179003604e-05, 2.65835445018243e-05, 4.5166139739522e-05}, {-1.09794358984383e-07, -3.48346439733689e-07, -6.08008550774746e-07}},
  {"PENTAX K-5 II iso 2500",  "PENTAX", "PENTAX K-5 II", 2500,  {7.49553948126126e-05, 3.0191184413285e-05, 4.98043553881967e-05}, {1.80160596465679e-07, -8.59901968490153e-08, -1.48514986122341e-07}},
// {"PENTAX K-5 II iso 2500",  "PENTAX", "PENTAX K-5 II", 2500,  {7.58999048975137e-05, 3.03214949495517e-05, 5.1879740013621e-05}, {6.74495485889244e-08, -9.69288594883671e-08, -4.19983447114398e-07}},
  {"PENTAX K-5 II iso 3200",  "PENTAX", "PENTAX K-5 II", 3200,  {9.53127556424175e-05, 3.89290755899931e-05, 6.62224784128816e-05}, {3.81821743702491e-07, -1.01727966287706e-07, -4.55702907856479e-07}},
// {"PENTAX K-5 II iso 3200",  "PENTAX", "PENTAX K-5 II", 3200,  {9.58056441645148e-05, 3.90121561539072e-05, 6.72216176917948e-05}, {3.13693393845006e-07, -1.15281237496667e-07, -5.84715118313398e-07}},
  {"PENTAX K-5 II iso 4000",  "PENTAX", "PENTAX K-5 II", 4000,  {0.00012403612466623, 5.16819885609624e-05, 8.84851123021319e-05}, {3.92159277742705e-07, -3.83647499072888e-07, -9.37204327189527e-07}},
// {"PENTAX K-5 II iso 4000",  "PENTAX", "PENTAX K-5 II", 4000,  {0.000121516443208268, 4.99194977146033e-05, 8.57621297605323e-05}, {5.97919457414944e-07, 3.19941589713569e-08, -5.46448744769967e-07}},
  {"PENTAX K-5 II iso 5000",  "PENTAX", "PENTAX K-5 II", 5000,  {0.000144498246551272, 5.9428825406911e-05, 0.000102267792778527}, {8.74651561068246e-07, -5.95458764573145e-08, -8.72238082049525e-07}},
// {"PENTAX K-5 II iso 5000",  "PENTAX", "PENTAX K-5 II", 5000,  {0.000144218352962032, 5.9743980069528e-05, 0.000102225276727013}, {8.1120903352457e-07, -8.78629987667917e-08, -8.08099844005415e-07}},
  {"PENTAX K-5 II iso 6400",  "PENTAX", "PENTAX K-5 II", 6400,  {0.000181197593431634, 7.52325989860919e-05, 0.000128522660012413}, {1.76747661172162e-06, 2.95750845397837e-07, -5.50397850785252e-07}},
// {"PENTAX K-5 II iso 6400",  "PENTAX", "PENTAX K-5 II", 6400,  {0.00018747859191775, 7.80358991484882e-05, 0.000131056794123799}, {9.91788857783028e-07, -3.49209215412362e-07, -9.18915782604748e-07}},
// {"PENTAX K-5 II iso 6400",  "PENTAX", "PENTAX K-5 II", 6400,  {0.000184222476068916, 7.53961839756192e-05, 0.000126637450452869}, {1.34304112189516e-06, 2.70453326764633e-07, -2.11807977454923e-07}},
  {"PENTAX K-5 II iso 8000",  "PENTAX", "PENTAX K-5 II", 8000,  {0.000235288272543989, 9.858436766129e-05, 0.000167883120247556}, {1.88837249347484e-06, 2.18065698968218e-07, -5.50388922813286e-07}},
// {"PENTAX K-5 II iso 8000",  "PENTAX", "PENTAX K-5 II", 8000,  {0.000237287714812751, 0.000101412814423024, 0.000167862920409919}, {1.67358014361036e-06, -5.42130798255096e-07, -6.05727009764256e-07}},
  {"PENTAX K-5 II iso 10000", "PENTAX", "PENTAX K-5 II", 10000, {0.000260205257239538, 0.000120811202669683, 0.000236898690822115}, {3.24400804911215e-06, 3.4784320183724e-07, -8.00157899691999e-06}},
// {"PENTAX K-5 II iso 10000", "PENTAX", "PENTAX K-5 II", 10000, {0.000265783300999659, 0.000122639168605949, 0.000235362486100454}, {2.7840320043002e-06, 3.39217524427858e-07, -7.54233630657402e-06}},
// {"PENTAX K-5 II iso 10000", "PENTAX", "PENTAX K-5 II", 10000, {0.000281043944127854, 0.000117536047727737, 0.000195844177145093}, {2.52452403233629e-06, 2.18737141265951e-07, -2.13828531308629e-07}},
// {"PENTAX K-5 II iso 10000", "PENTAX", "PENTAX K-5 II", 10000, {0.000279002567859344, 0.000117370137207154, 0.00019806613968224}, {2.72419222838177e-06, 2.81942183526845e-07, -4.86824874203399e-07}},
// {"PENTAX K-5 II iso 10000", "PENTAX", "PENTAX K-5 II", 10000, {0.000287771581299932, 0.000120091245555375, 0.000199201058068374}, {1.83210690458156e-06, -5.448707989096e-07, -8.44274649546031e-07}},
  {"PENTAX K-5 II iso 12800", "PENTAX", "PENTAX K-5 II", 12800, {0.000332068583378671, 0.000155449914292439, 0.00029687381010605}, {5.25762289929232e-06, 1.24647752368732e-06, -8.39490854637923e-06}},
// {"PENTAX K-5 II iso 12800", "PENTAX", "PENTAX K-5 II", 12800, {0.000334179825180717, 0.000156325455409393, 0.000303206748265869}, {5.31424998016092e-06, 1.15058046694789e-06, -9.35460007309005e-06}},
// {"PENTAX K-5 II iso 12800", "PENTAX", "PENTAX K-5 II", 12800, {0.00035741528125264, 0.000151267625690585, 0.000248316051427446}, {5.35923751663192e-06, 7.7430006401431e-07, 2.03029984192466e-07}},
// {"PENTAX K-5 II iso 12800", "PENTAX", "PENTAX K-5 II", 12800, {0.000357326610448005, 0.000153687504119368, 0.000253090245231722}, {4.39343320932899e-06, 1.13144267164264e-08, -1.22186457863022e-08}},
// {"PENTAX K-5 II iso 12800", "PENTAX", "PENTAX K-5 II", 12800, {0.000354488824543489, 0.000152019134173981, 0.000262306175205669}, {5.1074082331854e-06, 8.66571210075711e-07, -1.12577284777392e-06}},
  {"PENTAX K-5 II iso 16000", "PENTAX", "PENTAX K-5 II", 16000, {0.00042472248826841, 0.000197292402379586, 0.000380070347522433}, {9.04681446811521e-06, 3.19656272866334e-06, -8.9065835031295e-06}},
// {"PENTAX K-5 II iso 16000", "PENTAX", "PENTAX K-5 II", 16000, {0.000414071421291993, 0.000197083804754364, 0.000373145531753749}, {1.02382420663478e-05, 3.68572859617916e-06, -8.65519633579013e-06}},
// {"PENTAX K-5 II iso 16000", "PENTAX", "PENTAX K-5 II", 16000, {0.000442019058628825, 0.000189577356470913, 0.000312218454397835}, {1.00194793248551e-05, 3.23386617096904e-06, 3.22435855948473e-06}},
// {"PENTAX K-5 II iso 16000", "PENTAX", "PENTAX K-5 II", 16000, {0.000443200532739639, 0.0001922622258033, 0.000311167952062706}, {9.96238884602876e-06, 2.53562432598562e-06, 3.26291952074925e-06}},
  {"PENTAX K-5 II iso 20000", "PENTAX", "PENTAX K-5 II", 20000, {0.000495507559017026, 0.000230320297647919, 0.000439981776493056}, {1.23207011625797e-05, 4.80339769601906e-06, -7.13041701068233e-06}},
// {"PENTAX K-5 II iso 20000", "PENTAX", "PENTAX K-5 II", 20000, {0.000495098104637634, 0.000228761635371221, 0.000445161024206274}, {1.26992607167096e-05, 4.45311361914762e-06, -8.32254107805497e-06}},
// {"PENTAX K-5 II iso 20000", "PENTAX", "PENTAX K-5 II", 20000, {0.00051781813095442, 0.000218121061696392, 0.000361726525440339}, {1.46294319414607e-05, 6.22620624675356e-06, 6.14365289819323e-06}},
// {"PENTAX K-5 II iso 20000", "PENTAX", "PENTAX K-5 II", 20000, {0.000516868040539149, 0.000219358403463935, 0.000360185860165935}, {1.40317484716805e-05, 5.76412645674918e-06, 6.36646534262202e-06}},
  {"PENTAX K-5 II iso 25600", "PENTAX", "PENTAX K-5 II", 25600, {0.000645100307511573, 0.000297628454446346, 0.000560494164579298}, {1.93164956949029e-05, 7.44564420071584e-06, -6.83878553555169e-06}},
// {"PENTAX K-5 II iso 25600", "PENTAX", "PENTAX K-5 II", 25600, {0.000668423398194173, 0.000307301572025588, 0.000580478166383699}, {1.70679261179007e-05, 5.83540004038353e-06, -9.60316405950161e-06}},
// {"PENTAX K-5 II iso 25600", "PENTAX", "PENTAX K-5 II", 25600, {0.000638934461480744, 0.000274605450383485, 0.000464266297111264}, {2.4316975325769e-05, 1.21337328383061e-05, 1.21981343844352e-05}},
// {"PENTAX K-5 II iso 25600", "PENTAX", "PENTAX K-5 II", 25600, {0.000659988182260028, 0.000280336120496216, 0.000471562474173129}, {2.23167982176962e-05, 1.07759083118875e-05, 1.07274159467832e-05}},
// {"PENTAX K-5 II iso 25600", "PENTAX", "PENTAX K-5 II", 25600, {0.0006516001996409, 0.000278211164796907, 0.000469150349661312}, {2.30760126949115e-05, 1.10025161102815e-05, 1.11085412403008e-05}},
  {"PENTAX K-5 II iso 32000", "PENTAX", "PENTAX K-5 II", 32000, {0.000848655022144304, 0.00038958134796392, 0.000720619948089309}, {3.00155342379485e-05, 1.56396858476309e-05, -5.26259355875939e-06}},
// {"PENTAX K-5 II iso 32000", "PENTAX", "PENTAX K-5 II", 32000, {0.000847938830564672, 0.000379623372060259, 0.000720170363924872}, {2.8972666887025e-05, 1.65756982588839e-05, -4.02690953090812e-06}},
  {"PENTAX K-5 II iso 40000", "PENTAX", "PENTAX K-5 II", 40000, {0.0010533643109764, 0.00045044295378891, 0.000860445509979708}, {3.5009223918748e-05, 2.57908284055978e-05, -2.71868941544628e-06}},
// {"PENTAX K-5 II iso 40000", "PENTAX", "PENTAX K-5 II", 40000, {0.00102582308308523, 0.000437186632449604, 0.000856120867344648}, {3.50841855449835e-05, 2.82607384516916e-05, -2.57659651160096e-06}},
// {"PENTAX K-5 II iso 40000", "PENTAX", "PENTAX K-5 II", 40000, {0.00102391980460037, 0.000442904108975893, 0.00085701520249954}, {3.74820780251176e-05, 2.71640899781063e-05, -2.0410427640107e-06}},
// {"PENTAX K-5 II iso 40000", "PENTAX", "PENTAX K-5 II", 40000, {0.00102955018356212, 0.000455688823418914, 0.000866086480624827}, {3.79632641879429e-05, 2.42820830224728e-05, -4.11635440122107e-06}},
  {"PENTAX K-5 II iso 51200", "PENTAX", "PENTAX K-5 II", 51200, {0.00140838403042547, 0.0006278904129486, 0.00109568099993265}, {4.77913452321424e-05, 3.05444425259522e-05, 3.08088405662888e-06}},
// {"PENTAX K-5 II iso 51200", "PENTAX", "PENTAX K-5 II", 51200, {0.00141705278553903, 0.000616600640056148, 0.00106569570813491}, {4.68340325872651e-05, 3.00804303219371e-05, 6.7204678139444e-06}},
// {"PENTAX K-5 II iso 51200", "PENTAX", "PENTAX K-5 II", 51200, {0.00147289716972574, 0.000567390503701075, 0.00107254021464673}, {2.28674576187468e-05, 3.70990620589884e-05, 4.35628607489068e-05}},

// pentax k-30 contributed by wolfgang renz
{"PENTAX K-30 iso 100",   "PENTAX", "PENTAX K-30", 100,   {6.08256734607687e-06, 2.50418453046846e-06, 4.50003715826921e-06}, {-1.0982818178987e-07, -4.73621192413376e-08, -1.43905346007396e-08}},
{"PENTAX K-30 iso 200",   "PENTAX", "PENTAX K-30", 200,   {1.13133070894958e-05, 4.54008387968996e-06, 8.20252289617633e-06}, {-1.73160646322133e-07, -3.11732033282848e-08, 3.82583686707716e-08}},
{"PENTAX K-30 iso 400",   "PENTAX", "PENTAX K-30", 400,   {2.16288619936501e-05, 8.7012759850579e-06, 1.56633360198048e-05}, {-3.17226556442234e-07, -2.54102404961372e-08, 1.58447114253642e-07}},
{"PENTAX K-30 iso 800",   "PENTAX", "PENTAX K-30", 800,   {4.27618890110052e-05, 1.71382880493923e-05, 3.08798023085163e-05}, {-4.54479661837955e-07, 2.95303487078359e-08, 3.11262608300912e-07}},
{"PENTAX K-30 iso 1600",  "PENTAX", "PENTAX K-30", 1600,  {8.38440536405075e-05, 3.37549633384513e-05, 6.1666278950546e-05}, {-7.95629802189307e-07, 1.56375822295826e-07, 6.53392619979056e-07}},
{"PENTAX K-30 iso 3200",  "PENTAX", "PENTAX K-30", 3200,  {7.08508243393009e-05, 2.88103909058781e-05, 5.3165765742642e-05}, {-6.14729853723374e-07, 4.83038202221653e-08, 5.5826308751761e-07}},
{"PENTAX K-30 iso 6400",  "PENTAX", "PENTAX K-30", 6400,  {0.000163295668053171, 6.84223327467564e-05, 0.000123081073344394}, {-8.19352031951826e-07, 1.12481459839287e-06, 2.6639047897293e-06}},
{"PENTAX K-30 iso 12800", "PENTAX", "PENTAX K-30", 12800, {0.000255902766230414, 0.000134691978757832, 0.000233584939457326}, {9.43969512665206e-06, 4.57430591595449e-06, 1.2412517499906e-05}},
{"PENTAX K-30 iso 25600", "PENTAX", "PENTAX K-30", 25600, {0.00057239516588594, 0.000278279212760814, 0.000472039645214772}, {1.77864207435382e-05, 1.60512466010676e-05, 2.98744513916705e-05}},

// pentax k10d contributed by tx0h
  {"PENTAX K10D iso 100",  "PENTAX", "PENTAX K10D", 100,  {1.02367627014916e-05, 4.9091955753185e-06, 6.04942821561138e-06}, {-1.65430085968577e-07, -2.41455429557531e-07, -1.29773775757549e-07}},
  {"PENTAX K10D iso 125",  "PENTAX", "PENTAX K10D", 125,  {1.32925111314241e-05, 6.23704365909944e-06, 7.77622928917166e-06}, {-2.15341363099805e-07, -2.77498137369662e-07, -1.39227752756895e-07}},
  {"PENTAX K10D iso 160",  "PENTAX", "PENTAX K10D", 160,  {1.44774824242514e-05, 6.85098079537835e-06, 8.9321393095887e-06}, {-1.71797480847771e-07, -2.38684243529073e-07, -1.52712105543287e-07}},
  {"PENTAX K10D iso 200",  "PENTAX", "PENTAX K10D", 200,  {1.78610074553062e-05, 8.88981958766482e-06, 1.12065844180937e-05}, {-1.18554531099998e-07, -3.28075788618915e-07, -1.3131074028536e-07}},
  {"PENTAX K10D iso 250",  "PENTAX", "PENTAX K10D", 250,  {2.28357388437652e-05, 1.10291022727514e-05, 1.40802025180412e-05}, {-2.03948118420049e-07, -3.66357454472493e-07, -1.34675901033851e-07}},
  {"PENTAX K10D iso 320",  "PENTAX", "PENTAX K10D", 320,  {2.75043791258194e-05, 1.33437246767469e-05, 1.66453015595246e-05}, {-2.26078894194755e-07, -4.347322336876e-07, -8.61474296167209e-08}},
  {"PENTAX K10D iso 400",  "PENTAX", "PENTAX K10D", 400,  {3.21011431903841e-05, 1.64512043466556e-05, 2.01250917619875e-05}, {-1.44941054937907e-08, -5.02138074076235e-07, 6.87769204291625e-08}},
  {"PENTAX K10D iso 500",  "PENTAX", "PENTAX K10D", 500,  {3.80636812250017e-05, 1.86801893195956e-05, 2.43220883869917e-05}, {-1.69737695513725e-07, -5.62215821124226e-07, -2.00209262311514e-07}},
  {"PENTAX K10D iso 640",  "PENTAX", "PENTAX K10D", 640,  {4.41596009456921e-05, 2.08649677613643e-05, 2.71361323128873e-05}, {-3.74287259842114e-07, -6.16584075639679e-07, -1.52161800076396e-07}},
  {"PENTAX K10D iso 800",  "PENTAX", "PENTAX K10D", 800,  {5.41604043484142e-05, 2.54529668520349e-05, 3.43290782919279e-05}, {-5.91127704130443e-08, -4.18903758692897e-07, 1.20024989931595e-07}},
  {"PENTAX K10D iso 1000", "PENTAX", "PENTAX K10D", 1000, {6.62850730261562e-05, 3.17771961537796e-05, 4.27137241321552e-05}, {1.41277701548127e-07, -5.03449593924017e-07, 3.51797132523966e-07}},
  {"PENTAX K10D iso 1250", "PENTAX", "PENTAX K10D", 1250, {7.51789405808404e-05, 3.58916847071261e-05, 4.73712006780775e-05}, {2.77296624961085e-08, -6.43282394959506e-07, 4.02260433128856e-07}},
  {"PENTAX K10D iso 1600", "PENTAX", "PENTAX K10D", 1600, {9.40466214217775e-05, 4.40633353160868e-05, 5.9431026273189e-05}, {5.67896354956397e-07, -2.50546497402051e-07, 1.40973645819231e-06}},

// pentax k-3 by tx0h. iso 25600+ graphs don't really match, but variance seems to still stabilise ~1.
{"PENTAX K-3 iso 100",   "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 100,   {1.05212362141226e-05, 3.94809021650713e-06, 7.84296497873154e-06}, {3.24863757063071e-09, 2.70306560025622e-09, 3.36422379205466e-09}},
{"PENTAX K-3 iso 200",   "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 200,   {1.00886456850057e-05, 3.85095373926875e-06, 7.84940704880726e-06}, {2.30392236218314e-09, 2.02513037583613e-09, 2.30379918325379e-09}},
{"PENTAX K-3 iso 400",   "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 400,   {1.95282106024963e-05, 7.07369635977605e-06, 1.41936380070267e-05}, {8.46586155002119e-09, 7.23070046715332e-09, 8.65535747943075e-09}},
{"PENTAX K-3 iso 800",   "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 800,   {3.71072664898607e-05, 1.42873655659095e-05, 2.69272266528984e-05}, {2.4662386658184e-08, 1.99068520353875e-08, 2.53153782275682e-08}},
{"PENTAX K-3 iso 1600",  "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 1600,  {7.66923132504426e-05, 2.91740299886235e-05, 5.80021550757725e-05}, {5.52664739649506e-08, 4.45832034762538e-08, 5.67444574095942e-08}},
{"PENTAX K-3 iso 3200",  "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 3200,  {0.000114399936261757, 4.61001261387223e-05, 8.46793491919569e-05}, {1.42588177773058e-07, 1.03623829839519e-07, 1.43206467059716e-07}},
{"PENTAX K-3 iso 6400",  "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 6400,  {0.000143943303396064, 9.96205678980323e-05, 0.000133443403989444}, {3.08856258390386e-07, 2.08617382626526e-07, 3.08048202097564e-07}},
{"PENTAX K-3 iso 12800", "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 12800, {0.000225150014780407, 0.000298611475032959, 0.000263279516697261}, {4.66621327765493e-07, 2.34204034466301e-07, 4.52935958198772e-07}},
{"PENTAX K-3 iso 25600", "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 25600, {0.00052468915233115, 0.000840706578816058, 0.000754318058388537}, {7.84951259958916e-07, 2.57623760531695e-07, 7.24203708279665e-07}},
{"PENTAX K-3 iso 51200", "RICOH IMAGING COMPANY, LTD.", "PENTAX K-3", 51200, {0.00133964836551631, 0.0019016742166224, 0.0016671674349937}, {1.07173308657303e-06, 2.29940172320269e-07, 9.51130321559826e-07}},

// pentax k100d contributed by elliott
  {"PENTAX K100D iso 200",  "PENTAX", "PENTAX K100D", 200,  {3.95804544331784e-06, 3.95804544331784e-06, 3.95804544331784e-06}, {-1.69302519603735e-07, -1.69302519603735e-07, -1.69302519603735e-07}},
  {"PENTAX K100D iso 400",  "PENTAX", "PENTAX K100D", 400,  {5.20488498776622e-06, 5.20488498776622e-06, 5.20488498776622e-06}, {-1.63022469587059e-07, -1.63022469587059e-07, -1.63022469587059e-07}},
  {"PENTAX K100D iso 800",  "PENTAX", "PENTAX K100D", 800,  {9.25286082752558e-06, 9.25286082752558e-06, 9.25286082752558e-06}, {-1.28100505227757e-07, -1.28100505227757e-07, -1.28100505227757e-07}},
  {"PENTAX K100D iso 1600", "PENTAX", "PENTAX K100D", 1600, {1.65249296500756e-05, 1.65249296500756e-05, 1.65249296500756e-05}, {5.07468502484445e-07, 5.07468502484445e-07, 5.07468502484445e-07}},
  {"PENTAX K100D iso 3200", "PENTAX", "PENTAX K100D", 3200, {3.31313403159836e-05, 3.31313403159836e-05, 3.31313403159836e-05}, {1.92154574341619e-06, 1.92154574341619e-06, 1.92154574341619e-06}},

// pentax k100d super contributed by bovender (copied from elliott's data)
  {"PENTAX K100D Super iso 200",  "PENTAX", "PENTAX K100D Super", 200,  {3.95804544331784e-06, 3.95804544331784e-06, 3.95804544331784e-06}, {-1.69302519603735e-07, -1.69302519603735e-07, -1.69302519603735e-07}},
  {"PENTAX K100D Super iso 400",  "PENTAX", "PENTAX K100D Super", 400,  {5.20488498776622e-06, 5.20488498776622e-06, 5.20488498776622e-06}, {-1.63022469587059e-07, -1.63022469587059e-07, -1.63022469587059e-07}},
  {"PENTAX K100D Super iso 800",  "PENTAX", "PENTAX K100D Super", 800,  {9.25286082752558e-06, 9.25286082752558e-06, 9.25286082752558e-06}, {-1.28100505227757e-07, -1.28100505227757e-07, -1.28100505227757e-07}},
  {"PENTAX K100D Super iso 1600", "PENTAX", "PENTAX K100D Super", 1600, {1.65249296500756e-05, 1.65249296500756e-05, 1.65249296500756e-05}, {5.07468502484445e-07, 5.07468502484445e-07, 5.07468502484445e-07}},
  {"PENTAX K100D Super iso 3200", "PENTAX", "PENTAX K100D Super", 3200, {3.31313403159836e-05, 3.31313403159836e-05, 3.31313403159836e-05}, {1.92154574341619e-06, 1.92154574341619e-06, 1.92154574341619e-06}},

// pentax k110d by Matthieu Volat (same as k100d with updated iso 200)
  {"PENTAX K110D iso 200", "PENTAX", "PENTAX K110D", 200, {2.06027150409162e-06, 3.54038951685638e-06, 3.77672297613284e-06}, {3.78928038552898e-08, 9.32234631575871e-09, 7.2994177239754e-08}},
  {"PENTAX K110D iso 400",  "PENTAX", "PENTAX K110D", 400,  {5.20488498776622e-06, 5.20488498776622e-06, 5.20488498776622e-06}, {-1.63022469587059e-07, -1.63022469587059e-07, -1.63022469587059e-07}},
  {"PENTAX K110D iso 800",  "PENTAX", "PENTAX K110D", 800,  {9.25286082752558e-06, 9.25286082752558e-06, 9.25286082752558e-06}, {-1.28100505227757e-07, -1.28100505227757e-07, -1.28100505227757e-07}},
  {"PENTAX K110D iso 1600", "PENTAX", "PENTAX K110D", 1600, {1.65249296500756e-05, 1.65249296500756e-05, 1.65249296500756e-05}, {5.07468502484445e-07, 5.07468502484445e-07, 5.07468502484445e-07}},
  {"PENTAX K110D iso 3200", "PENTAX", "PENTAX K110D", 3200, {3.31313403159836e-05, 3.31313403159836e-05, 3.31313403159836e-05}, {1.92154574341619e-06, 1.92154574341619e-06, 1.92154574341619e-06}},


// pentax k-m contributed by Jean-Christophe Caffe
  {"PENTAX K-m iso 100",  "PENTAX", "PENTAX K-m", 100,  {3.87011793245457e-06, 4.03322633047438e-06, 3.64311974522327e-06}, {8.69390316317425e-08, -9.83710674594626e-08, 2.90848586872562e-07}},
  {"PENTAX K-m iso 125",  "PENTAX", "PENTAX K-m", 125,  {9.39820497862007e-06, 4.45940513650294e-06, 5.5894012241424e-06}, {-4.75886637113419e-08, -9.79993907007359e-08, 2.430582641428e-08}},
  {"PENTAX K-m iso 200",  "PENTAX", "PENTAX K-m", 200,  {8.0027594239537e-06, 5.97665412632145e-06, 1.00849419032639e-05}, {3.27718864952641e-07, -1.25165177485449e-08, 1.1526746458079e-07}},
  {"PENTAX K-m iso 250",  "PENTAX", "PENTAX K-m", 250,  {1.68533011711162e-05, 7.63484692160187e-06, 8.54782686086988e-06}, {-8.23117426874891e-08, 1.60270397334301e-07, 6.5834598676889e-07}},
  {"PENTAX K-m iso 400",  "PENTAX", "PENTAX K-m", 400,  {6.62683061301946e-06, 9.87707784981921e-06, 1.19735978964325e-05}, {1.62777215214915e-06, 3.35142764636583e-07, 9.21994598739137e-07}},
  {"PENTAX K-m iso 500",  "PENTAX", "PENTAX K-m", 500,  {6.54719087651375e-06, 9.55014555455602e-06, 1.16424678153785e-05}, {9.94800238113157e-07, -1.3924992306725e-07, 3.98966198472856e-07}},
  {"PENTAX K-m iso 800",  "PENTAX", "PENTAX K-m", 800,  {3.17793968443221e-05, 1.67210136982223e-05, 2.12875577411074e-05}, {5.86977627709036e-07, -3.70990159786682e-07, 1.25737909093602e-07}},
  {"PENTAX K-m iso 1000", "PENTAX", "PENTAX K-m", 1000, {3.35272431874994e-05, 2.16524789545364e-05, 2.03945467635197e-05}, {3.78810281803938e-07, -3.61456249864358e-07, 2.01478656739782e-06}},
  {"PENTAX K-m iso 1250", "PENTAX", "PENTAX K-m", 1250, {2.74369101863684e-05, 2.40944297900712e-05, 1.9556309083575e-05}, {1.06825090605845e-06, -1.13581094450776e-07, 4.78733793440384e-06}},
  {"PENTAX K-m iso 1600", "PENTAX", "PENTAX K-m", 1600, {5.45171340107178e-05, 3.08470062241788e-05, 3.80309038084381e-05}, {1.81620674623084e-06, -4.64488174707219e-08, 2.17272222669808e-06}},
  {"PENTAX K-m iso 2000", "PENTAX", "PENTAX K-m", 2000, {4.77257364924071e-05, 3.44427137379981e-05, 3.90795319092654e-05}, {2.24539901948283e-06, 1.53912405215952e-07, 2.97588427433887e-06}},
  {"PENTAX K-m iso 2500", "PENTAX", "PENTAX K-m", 2500, {5.77832568279373e-05, 3.61121161536652e-05, 4.41370191648524e-05}, {1.53098073435018e-06, 6.24253277123082e-07, 3.92891902335599e-06}},
  {"PENTAX K-m iso 3200", "PENTAX", "PENTAX K-m", 3200, {7.80822080768476e-05, 4.49595026094475e-05, 1.5568731866553e-05}, {9.12175058243346e-07, 7.35125462445745e-07, 1.43097848611759e-05}},

// pentax k-r contributed by Enrico Hillmann
  {"PENTAX K-r iso 100",   "PENTAX", "PENTAX K-r", 100,   {6.19156302755701e-06, 3.16523510984186e-06, 4.33488249710363e-06}, {-2.56772373696417e-08, -8.05559248900957e-08, 1.87785280415261e-08}},
  {"PENTAX K-r iso 200",   "PENTAX", "PENTAX K-r", 200,   {1.0250229176481e-05, 5.12454064734191e-06, 7.1840749873889e-06}, {-3.18721153245065e-08, -9.42627636285803e-08, 3.86980081192557e-08}},
  {"PENTAX K-r iso 400",   "PENTAX", "PENTAX K-r", 400,   {2.01680195811569e-05, 9.7588183840868e-06, 1.40658524220736e-05}, {-3.80922002873133e-08, -7.41029038353914e-08, 1.03718305077219e-07}},
  {"PENTAX K-r iso 800",   "PENTAX", "PENTAX K-r", 800,   {4.05240965801942e-05, 1.88301172697162e-05, 2.84960422041034e-05}, {-3.00780147479662e-08, 5.99353245687969e-08, 2.41358707251733e-07}},
  {"PENTAX K-r iso 1600",  "PENTAX", "PENTAX K-r", 1600,  {7.78512599138544e-05, 3.80951661837849e-05, 5.60024565948438e-05}, {4.78302609408712e-07, 1.99049564161835e-07, 8.35112202282036e-07}},
  {"PENTAX K-r iso 3200",  "PENTAX", "PENTAX K-r", 3200,  {9.32147526829224e-05, 4.15792401858486e-05, 6.35952861924381e-05}, {-6.84817898763123e-07, -4.66983378834082e-07, 1.01595683678666e-07}},
  {"PENTAX K-r iso 6400",  "PENTAX", "PENTAX K-r", 6400,  {0.000179139671672974, 8.20240322777469e-05, 0.000126436079576067}, {-1.83643362768227e-06, -2.39470723347351e-06, -6.03127927129158e-07}},
  {"PENTAX K-r iso 12800", "PENTAX", "PENTAX K-r", 12800, {0.000356023968236295, 0.000156068443240422, 0.000251843393697387}, {-4.61717436588577e-06, -5.418466661142e-06, -2.29172762302178e-06}},
  {"PENTAX K-r iso 25600", "PENTAX", "PENTAX K-r", 25600, {0.000639410909049122, 0.000289341910661324, 0.000453963503048857}, {-4.33366763076952e-06, -1.05672332440916e-05, 1.37024899274193e-06}},

// pentax k-x contributed by supersubsexy
  {"PENTAX K-x iso 100",   "PENTAX", "PENTAX K-x", 100,   {6.76033046744373e-06, 3.13786529644355e-06, 4.72421413200676e-06}, {1.80245644237734e-08, -4.45550549863608e-08, 5.83376855408949e-08}},
  {"PENTAX K-x iso 100",   "PENTAX", "PENTAX K-x", 100,   {6.8608154789686e-06, 3.19957140163793e-06, 4.76268867914309e-06}, {5.89424401049698e-09, -6.7523008965824e-08, 4.80348633264404e-08}},
  {"PENTAX K-x iso 200",   "PENTAX", "PENTAX K-x", 200,   {1.17919078441947e-05, 5.52229712116575e-06, 8.14466834857774e-06}, {5.08539857565942e-08, -7.44286491740946e-08, 1.34689866497498e-07}},
  {"PENTAX K-x iso 400",   "PENTAX", "PENTAX K-x", 400,   {2.32422666175937e-05, 1.0663087786159e-05, 1.56425112185104e-05}, {9.25354928581385e-08, -1.14397212258652e-07, 3.09058950338858e-07}},
  {"PENTAX K-x iso 800",   "PENTAX", "PENTAX K-x", 800,   {4.50984884294466e-05, 2.01398787150773e-05, 3.09911879112039e-05}, {3.23969553142247e-07, 6.78171612427272e-08, 6.25272598357679e-07}},
  {"PENTAX K-x iso 1600",  "PENTAX", "PENTAX K-x", 1600,  {8.711933969702e-05, 3.93452086527508e-05, 6.06109541893504e-05}, {1.0794823832037e-06, 4.57390681907966e-07, 1.65248517451635e-06}},
  {"PENTAX K-x iso 3200",  "PENTAX", "PENTAX K-x", 3200,  {0.000112206026395514, 4.62734901808337e-05, 7.79217467241133e-05}, {-3.92579968277323e-07, -5.77225042629809e-07, 1.10981153408388e-07}},
  {"PENTAX K-x iso 6400",  "PENTAX", "PENTAX K-x", 6400,  {0.000224828833066249, 9.0372127524656e-05, 0.000148698019869859}, {-1.3754523998075e-06, -2.13009588056015e-06, 7.01772655051384e-07}},
  {"PENTAX K-x iso 12800", "PENTAX", "PENTAX K-x", 12800, {0.000436333184304212, 0.000172418953534638, 0.000290077815985367}, {-2.6555286023e-06, -5.00216442638314e-06, 1.78363603676931e-06}},

// -------------------------------------------------------------------
// Samsung.
// -------------------------------------------------------------------

// samsung wb2000 contributed by christer
{"Samsung WB2000 iso 80",   "SAMSUNG", "WB2000", 80,   {2.0160611403384e-05, 1.4707228798157e-05, 3.16913190805894e-05}, {-3.79863740782662e-09, -1.29248874817731e-08, -1.06369803618908e-08}},
{"Samsung WB2000 iso 100",  "SAMSUNG", "WB2000", 100,  {2.41301940184776e-05, 1.81195829103773e-05, 3.88247268576532e-05}, {5.79513598187703e-09, -2.30527461730876e-08, -1.11693346107315e-08}},
{"Samsung WB2000 iso 200",  "SAMSUNG", "WB2000", 200,  {4.71354961747483e-05, 3.45593589400912e-05, 7.41017484326013e-05}, {3.8004001332456e-08, -1.02562679457989e-08, 1.17358027847987e-08}},
{"Samsung WB2000 iso 400",  "SAMSUNG", "WB2000", 400,  {9.3357584405466e-05, 6.77776784640525e-05, 0.00014469902314401}, {1.52468544454589e-07, 5.73041720173694e-08, 1.27848228906766e-07}},
{"Samsung WB2000 iso 800",  "SAMSUNG", "WB2000", 800,  {0.000187755655993027, 0.000134666005121722, 0.000286691356082704}, {5.41201248752878e-07, 3.29826692216511e-07, 5.44373719972467e-07}},
{"Samsung WB2000 iso 1600", "SAMSUNG", "WB2000", 1600, {0.000187271629996756, 0.000118040444787749, 0.000280700542952258}, {-9.29308992364551e-07, -5.4806269987963e-07, -8.59651320090596e-07}},
{"Samsung WB2000 iso 3200", "SAMSUNG", "WB2000", 3200, {0.00022342803093475, 0.000161989898955004, 0.00036628293119213}, {-9.57293455172455e-07, -7.24852861277862e-07, -1.64226904032162e-06}},

// samsung nx10 contributed by Benjamin Hartmann. dark regions are 2x--3x variance.
{"Samsung NX10 iso 100",  "SAMSUNG", "NX10", 100,  {7.65735549362913e-06, 4.18741471878906e-06, 6.91666119907651e-06}, {1.04070307329532e-07, 5.59035525568829e-08, 9.65756542733997e-08}},
{"Samsung NX10 iso 200",  "SAMSUNG", "NX10", 200,  {1.49994148703597e-05, 8.07994047740921e-06, 1.34051992764919e-05}, {1.59543146310844e-07, 6.53548388731254e-08, 1.17565449136037e-07}},
{"Samsung NX10 iso 400",  "SAMSUNG", "NX10", 400,  {2.78483245307609e-05, 1.50462772059569e-05, 2.43376111339857e-05}, {4.43369696384773e-07, 2.67990879169606e-07, 4.00159340412934e-07}},
{"Samsung NX10 iso 800",  "SAMSUNG", "NX10", 800,  {6.2189516883651e-05, 3.22161023858762e-05, 5.40503479042483e-05}, {1.36125869630645e-06, 7.44485259653261e-07, 1.13880659521414e-06}},
{"Samsung NX10 iso 1600", "SAMSUNG", "NX10", 1600, {9.21346053521494e-05, 4.89527138033317e-05, 8.23199072877294e-05}, {3.11814678712925e-06, 2.16863169936236e-06, 2.90389148342322e-06}},
{"Samsung NX10 iso 3200", "SAMSUNG", "NX10", 3200, {0.000177960506523153, 9.08869401870464e-05, 0.000155735138542919}, {1.08363769753609e-05, 7.89650015794136e-06, 1.02035701747638e-05}},

// samsung nx100, contributed by pascal de bruijn, variance doesn't seem to stabilize well in dark areas
  {"Samsung NX100 iso 100",  "SAMSUNG", "NX100", 100,  {4.99270904164219e-06, 4.92373089527128e-06, 1.27096268204169e-05}, {8.3199766769219e-08, -1.45782668222198e-08, -3.10677261396051e-07}},
  {"Samsung NX100 iso 125",  "SAMSUNG", "NX100", 125,  {5.80152123811272e-06, 6.67401815502262e-06, 1.52458740580584e-05}, {8.92982644623542e-08, -2.21036386490348e-07, -3.93401686700479e-07}},
  {"Samsung NX100 iso 160",  "SAMSUNG", "NX100", 160,  {7.29549140065461e-06, 7.81002405051693e-06, 1.84318071962764e-05}, {1.75107099225069e-07, -1.20163429335486e-07, -3.98677626884079e-07}},
  {"Samsung NX100 iso 200",  "SAMSUNG", "NX100", 200,  {8.62464475204047e-06, 9.19017734190852e-06, 1.87781571105484e-05}, {9.32351589525802e-08, -2.41100351701182e-07, -1.10896237901791e-07}},
  {"Samsung NX100 iso 250",  "SAMSUNG", "NX100", 250,  {1.0553763649372e-05, 1.14098926690361e-05, 2.87748447171735e-05}, {1.86284833055937e-07, -2.81706204543787e-07, -8.58131550154709e-07}},
  {"Samsung NX100 iso 320",  "SAMSUNG", "NX100", 320,  {1.33906790901776e-05, 1.35704207167362e-05, 3.14681984120395e-05}, {3.45689187514433e-07, -4.16467389725744e-08, -2.71262841421208e-07}},
  {"Samsung NX100 iso 400",  "SAMSUNG", "NX100", 400,  {1.71186358563409e-05, 1.83138454297285e-05, 3.52738484665866e-05}, {3.52993225140339e-07, -4.15172922928797e-07, 3.84753280703847e-07}},
  {"Samsung NX100 iso 500",  "SAMSUNG", "NX100", 500,  {2.08975225212306e-05, 2.14947842139058e-05, 4.5359711568221e-05}, {5.64210011614828e-07, -2.55804112947131e-07, -1.57092044373335e-07}},
  {"Samsung NX100 iso 640",  "SAMSUNG", "NX100", 640,  {2.67310964037834e-05, 2.77160023311768e-05, 6.78675785215236e-05}, {9.71693846312659e-07, -9.39887996904199e-08, -1.04616255753396e-06}},
  {"Samsung NX100 iso 800",  "SAMSUNG", "NX100", 800,  {3.41290610856095e-05, 3.51603518961619e-05, 6.32845265295679e-05}, {1.49056732887989e-06, 1.02080357718622e-07, 2.69354727489016e-06}},
  {"Samsung NX100 iso 1000", "SAMSUNG", "NX100", 1000, {4.27291395882936e-05, 4.47900184942864e-05, 9.12495021111337e-05}, {2.2345336569479e-06, 2.92603533258487e-08, 1.47940552519575e-06}},
  {"Samsung NX100 iso 1250", "SAMSUNG", "NX100", 1250, {5.63463426258262e-05, 5.5736515883779e-05, 0.000123952729707145}, {3.44975906877591e-06, 1.24553830208962e-06, 1.97159115599619e-06}},
  {"Samsung NX100 iso 1600", "SAMSUNG", "NX100", 1600, {7.03948200996258e-05, 6.21988681000337e-05, 0.000128394078989594}, {5.37066431274778e-06, 4.00447117492512e-06, 6.87756363670757e-06}},
  {"Samsung NX100 iso 2000", "SAMSUNG", "NX100", 2000, {9.17926027538227e-05, 7.6975581999008e-05, 0.000152262064103023}, {7.5104875275086e-06, 6.25730389334235e-06, 1.14743733384823e-05}},
  {"Samsung NX100 iso 2500", "SAMSUNG", "NX100", 2500, {0.000115726837964689, 0.000107752653381485, 0.000179168085512152}, {1.14961768184372e-05, 7.75361929055781e-06, 1.94381720855378e-05}},
  {"Samsung NX100 iso 3200", "SAMSUNG", "NX100", 3200, {0.000146571675499991, 0.000135686656418072, 0.000215961863950127}, {1.51818462120236e-05, 9.81684615476285e-06, 2.51384339926021e-05}},
  {"Samsung NX100 iso 4000", "SAMSUNG", "NX100", 4000, {0.000181939407159497, 0.000148920964482435, 0.000271809667113892}, {2.33931860017987e-05, 2.10908848045366e-05, 3.54211629566242e-05}},
  {"Samsung NX100 iso 5000", "SAMSUNG", "NX100", 5000, {0.000258727075969553, 0.000192632042295206, 0.000356218198972838}, {3.71003290732463e-05, 3.83536056545317e-05, 6.11806111574509e-05}},
  {"Samsung NX100 iso 6400", "SAMSUNG", "NX100", 6400, {0.000357590485427811, 0.000255851428472357, 0.000406746090979271}, {5.23944994274853e-05, 5.52000614533804e-05, 9.21564514935925e-05}},

// -------------------------------------------------------------------
// Sony.
// -------------------------------------------------------------------

// sony a200, contributed by Filip Wostyn (blue channel underexposed)
  {"Sony DSLR-A200 iso 100",  "SONY", "DSLR-A200", 100,  {4.85490229535908e-06, 2.816649078717e-06,   0.0}, {-1.77163545180459e-08, 8.31302896857195e-09, 0.0}},
  {"Sony DSLR-A200 iso 200",  "SONY", "DSLR-A200", 200,  {9.18099803502629e-06, 5.22580231972482e-06, 0.0}, {-2.02408066474279e-09, 6.76804985592989e-08, 0.0}},
  {"Sony DSLR-A200 iso 400",  "SONY", "DSLR-A200", 400,  {1.57847654053797e-05, 1.01654454622836e-05, 0.0}, {3.15559289901219e-07, 1.6079816558863e-07,   0.0}},
  {"Sony DSLR-A200 iso 800",  "SONY", "DSLR-A200", 800,  {3.3947869388103e-05, 1.91625790598212e-05,  0.0}, {5.62314904245478e-07, 7.42771947065692e-07,  0.0}},
  {"Sony DSLR-A200 iso 1600", "SONY", "DSLR-A200", 1600, {5.51549095364216e-05, 3.84246982151535e-05, 0.0}, {3.32401113300663e-06, 1.71747661498869e-06,  0.0}},
  {"Sony DSLR-A200 iso 3200", "SONY", "DSLR-A200", 3200, {0.000117102014518605, 7.26025862455521e-05, 0.0}, {1.17017083365962e-05, 1.02035222669924e-05,  0.0}},

// sony a230 contributed by wolf kuehnel
  {"Sony DSLR-A230 iso 100",  "SONY", "DSLR-A230", 100,  {4.91156751642718e-06, 2.53383833010807e-06, 4.92096818578844e-06}, {1.85735808516565e-08, 1.58186038498792e-08, -4.53741722753503e-08}},
  {"Sony DSLR-A230 iso 200",  "SONY", "DSLR-A230", 200,  {9.56116133716475e-06, 4.96400825171195e-06, 9.51933373994277e-06}, {9.23381999584418e-08, 5.42270579473812e-08, -4.2826216442958e-08}},
  {"Sony DSLR-A230 iso 400",  "SONY", "DSLR-A230", 400,  {1.89145279069212e-05, 9.61998082452689e-06, 1.86200559251644e-05}, {2.31674729489946e-07, 1.64624654793155e-07, 2.5822239968504e-09}},
  {"Sony DSLR-A230 iso 800",  "SONY", "DSLR-A230", 800,  {3.77285168786398e-05, 1.86730237715715e-05, 3.61616843576951e-05}, {7.79126168873193e-07, 6.99135819397957e-07, 4.04054321171843e-07}},
  {"Sony DSLR-A230 iso 1600", "SONY", "DSLR-A230", 1600, {7.38492119237285e-05, 3.73676951943247e-05, 7.26326012675557e-05}, {2.77458216805808e-06, 2.27006455738785e-06, 2.05333700099163e-06}},
  {"Sony DSLR-A230 iso 3200", "SONY", "DSLR-A230", 3200, {0.000155984975527047, 7.51348421343849e-05, 0.000142668502959687}, {1.03329464808737e-05, 8.9782605561319e-06, 1.01191262667006e-05}},

// sony a350 contributed by Matthew Harrison
  {"DSLR-A350 iso 100",  "SONY", "DSLR-A350", 100,  {5.12532633234084e-06, 3.35861914751694e-06, 1.35044249743616e-05}, {2.11499702728152e-08, 1.59537407062459e-08, 2.67581570073902e-08}},
  {"DSLR-A350 iso 100",  "SONY", "DSLR-A350", 100,  {5.07296284558634e-06, 3.26827735610422e-06, 1.06709856622057e-05}, {2.25122960945006e-08, 1.73645738315921e-08, 3.01281009229553e-08}},
  {"DSLR-A350 iso 200",  "SONY", "DSLR-A350", 200,  {1.05551342097347e-05, 6.76169252263199e-06, 2.5833259450462e-05}, {6.31255405109755e-08, 4.83299855475263e-08, 7.62229529305851e-08}},
  {"DSLR-A350 iso 400",  "SONY", "DSLR-A350", 400,  {2.28142971924505e-05, 1.43630485420633e-05, 5.24974189254329e-05}, {1.88112801912616e-07, 1.39356851685102e-07, 2.21261328468707e-07}},
  {"DSLR-A350 iso 800",  "SONY", "DSLR-A350", 800,  {5.57713323979668e-05, 3.34401449763136e-05, 0.000125496890563015}, {5.52042018356297e-07, 3.94283499037157e-07, 6.14164575481293e-07}},
  {"DSLR-A350 iso 800",  "SONY", "DSLR-A350", 800,  {5.40916731471824e-05, 3.308223651537e-05, 0.000113510612163422}, {5.77653083834006e-07, 4.22856064501429e-07, 6.55152032165076e-07}},
  {"DSLR-A350 iso 1600", "SONY", "DSLR-A350", 1600, {5.27054561435461e-05, 2.76360488503607e-05, 0.000104674699015529}, {3.7857514189617e-07, 2.85605097872811e-07, 4.35273397932477e-07}},
  {"DSLR-A350 iso 3200", "SONY", "DSLR-A350", 3200, {0.000100789741590788, 6.19615681861067e-05, 0.000161245667610495}, {1.6732108761999e-06, 1.32272785241363e-06, 1.89669160925809e-06}},
  {"DSLR-A350 iso 3200", "SONY", "DSLR-A350", 3200, {0.000122509600355669, 6.56243636802537e-05, 0.000227603904481378}, {1.35263644218884e-06, 1.1532449646405e-06, 1.36647767163582e-06}},
  {"DSLR-A350 iso 3200", "SONY", "DSLR-A350", 3200, {0.000130963628773744, 6.53973646422326e-05, 0.000244088014328315}, {9.15176291213133e-07, 7.70338399685313e-07, 8.87047784136548e-07}},
  {"DSLR-A350 iso 3200", "SONY", "DSLR-A350", 3200, {0.000117777412065988, 6.86832931347456e-05, 0.000166799960755746}, {1.09552412561953e-06, 8.28316322789316e-07, 1.33018063895122e-06}},
  {"DSLR-A350 iso 3200", "SONY", "DSLR-A350", 3200, {0.00012830914962018, 6.73045386785144e-05, 0.0002066667877103}, {9.45668210196981e-07, 7.85856780337091e-07, 1.05435733257298e-06}},
  {"DSLR-A350 iso 3200", "SONY", "DSLR-A350", 3200, {0.000125507773707929, 6.58169081023234e-05, 0.000205504637828949}, {9.5919123599762e-07, 8.0085395125464e-07, 1.05927307057544e-06}},
  {"DSLR-A350 iso 3200", "SONY", "DSLR-A350", 3200, {0.000115870435154253, 7.70074926356764e-05, 0.000281526263221047}, {8.37709730392567e-07, 6.20341263598301e-07, 5.7906714203212e-07}},

// sony slt a57 by baptiste pierrat
{"Sony SLT-A57 iso 100",   "SONY", "SLT-A57", 100,   {8.20936498101313e-06, 3.27881919558562e-06, 6.11452583732428e-06}, {4.86719239159469e-08, -4.53855689673332e-08, -4.49960973005900e-08}},
{"Sony SLT-A57 iso 200",   "SONY", "SLT-A57", 200,   {1.46341604486813e-05, 6.07859947358060e-06, 1.12900396890618e-05}, {1.62164201348617e-07, -8.53402265000436e-08, -4.80232467691821e-08}},
{"Sony SLT-A57 iso 400",   "SONY", "SLT-A57", 400,   {2.69816910205665e-05, 1.11888156000366e-05, 2.10765537863301e-05}, {4.44865134287628e-07, -3.08156286732935e-08, 2.21618116948954e-08}},
{"Sony SLT-A57 iso 800",   "SONY", "SLT-A57", 800,   {4.97452057507446e-05, 2.17830164577161e-05, 4.01276128257118e-05}, {1.24618368543683e-06, 2.27977516300719e-08, 2.03307332197179e-07}},
{"Sony SLT-A57 iso 1600",  "SONY", "SLT-A57", 1600,  {0.000100665317743357, 4.26503651759825e-05, 8.01972387030296e-05}, {2.14848436974959e-06, 2.70630540891117e-08, 1.71235701725068e-07}},
{"Sony SLT-A57 iso 3200",  "SONY", "SLT-A57", 3200,  {0.000192122052096465, 8.56404928664031e-05, 0.000156633480695791}, {5.21479927933938e-06, 1.27158769041921e-07, 9.34455128479542e-07}},
{"Sony SLT-A57 iso 6400",  "SONY", "SLT-A57", 6400,  {0.000240165459232429, 0.000109559902738334, 0.000200584435688242}, {8.41917137016736e-06, 1.69448833481848e-06, 2.28216332076454e-06}},
{"Sony SLT-A57 iso 12800", "SONY", "SLT-A57", 12800, {0.000469798335730928, 0.000213881592540329, 0.000386489477766441}, {2.09952660143769e-05, 7.70078013047938e-06, 9.59324060253915e-06}},
{"Sony SLT-A57 iso 16000", "SONY", "SLT-A57", 16000, {0.000576214743660584, 0.000269271310162455, 0.000476873103852378}, {3.13750537681958e-05, 1.27033596473290e-05, 1.76296396196867e-05}},

// sony a55v by Jean-Marc Sarat
  {"Sony SLT-A55V iso 100",   "SONY", "SLT-A55V", 100,   {7.42262406916063e-06, 3.04135887283305e-06, 5.16661845263891e-06}, {-1.11085646065374e-07, -6.43083333655761e-08, -2.39136528952084e-08}},
  {"Sony SLT-A55V iso 200",   "SONY", "SLT-A55V", 200,   {1.43867846798e-05, 5.67472914394192e-06, 9.97322528654085e-06}, {-2.22498800097017e-07, -1.26471806954322e-07, -8.49645136886073e-08}},
  {"Sony SLT-A55V iso 400",   "SONY", "SLT-A55V", 400,   {2.65206940926404e-05, 1.05402195407223e-05, 1.86210155019393e-05}, {-3.05405060551708e-07, -1.27647438695619e-07, -2.96194648878434e-08}},
  {"Sony SLT-A55V iso 800",   "SONY", "SLT-A55V", 800,   {5.13396243719539e-05, 2.02267319858678e-05, 3.575983487841e-05}, {-4.56378623588778e-07, -8.12541345959856e-08, 1.95940094966157e-07}},
  {"Sony SLT-A55V iso 1600",  "SONY", "SLT-A55V", 1600,  {0.000101771502583976, 4.01516549091693e-05, 7.20409372326031e-05}, {-8.05204527673689e-07, -9.73100300579007e-08, 1.5002598324038e-07}},
  {"Sony SLT-A55V iso 3200",  "SONY", "SLT-A55V", 3200,  {0.000198778238116058, 7.8889901074859e-05, 0.000140553905681395}, {-1.03079767974994e-06, 3.00509134373011e-07, 1.0726402220341e-06}},
  {"Sony SLT-A55V iso 6400",  "SONY", "SLT-A55V", 6400,  {0.000402187274387211, 0.000156627523325748, 0.000281528929415211}, {-2.07750443286533e-06, 1.41537746223123e-06, 2.57815505679415e-06}},
  {"Sony SLT-A55V iso 12800", "SONY", "SLT-A55V", 12800, {0.0007573940584859, 0.000304197687611975, 0.000535567961856837}, {4.02077563288484e-06, 8.66095985562674e-06, 1.45131979463849e-05}},

// sony a550 by bruce guenther
  {"Sony DSLR-A550 iso 200",   "SONY", "DSLR-A550", 200,         {1.00786943371057e-05, 1.07224245005328e-05, 5.04716105014355e-06}, {-3.80021788048095e-08, -9.26052119715185e-07, 4.65564725630196e-08}},
  {"Sony DSLR-A550 iso 400",   "SONY", "DSLR-A550", 400,         {1.81290374445939e-05, 1.42631445146653e-05, 9.69795884947463e-06}, {5.48223528564755e-08, -5.40820312090065e-07, 9.44487048494704e-08}},
  {"Sony DSLR-A550 iso 800",   "SONY", "DSLR-A550", 800,         {3.3248480975959e-05, 2.26392837157423e-05, 1.87848007407424e-05}, {2.93374885810922e-07, -1.18297930280492e-07, 2.04765602893558e-07}},
  {"Sony DSLR-A550 iso 1600",  "SONY", "DSLR-A550", 1600,        {6.32749073011992e-05, 4.02128437503551e-05, 3.6197592974281e-05}, {7.34474590208309e-07, 4.20987387824998e-07, 5.67827888879688e-07}},
  {"Sony DSLR-A550 iso 3200",  "SONY", "DSLR-A550", 3200,        {0.000123228136182573, 7.74187248136683e-05, 7.07433923125239e-05}, {1.82714008313777e-06, 1.14577311187686e-06, 1.43740270332549e-06}},
  {"Sony DSLR-A550 iso 6400",  "SONY", "DSLR-A550", 6400,        {0.000245699806264141, 0.000148812791171945, 0.000142213207175152}, {4.68368085659923e-06, 3.87158485902286e-06, 4.04077292806587e-06}},
  {"Sony DSLR-A550 iso 12800", "SONY", "DSLR-A550", 12800,       {0.000501781300317304, 0.000293232336769259, 0.000290642564989266}, {1.26894604916349e-05, 1.17978767072351e-05, 1.27581101566565e-05}},

// sony a580, by toalaber
{"Sony DSLR-A580 iso 100",   "SONY", "DSLR-A580", 100,   {5.50867300646712e-06, 1.97638645903412e-06, 3.62462303194655e-06}, {6.48956950675451e-09, 3.08208260450343e-09, 8.9552444938696e-09}},
{"Sony DSLR-A580 iso 200",   "SONY", "DSLR-A580", 200,   {1.04740862192733e-05, 3.77971490092718e-06, 6.88988681917485e-06}, {1.39993205529287e-08, 7.77679511157941e-09, 1.9538038907662e-08}},
{"Sony DSLR-A580 iso 400",   "SONY", "DSLR-A580", 400,   {2.02924675146487e-05, 7.25706309073115e-06, 1.32674633351426e-05}, {3.17500989545691e-08, 2.24167984567661e-08, 3.9571086304305e-08}},
{"Sony DSLR-A580 iso 800",   "SONY", "DSLR-A580", 800,   {3.97185412311943e-05, 1.42450019951798e-05, 2.57543755665689e-05}, {8.1643041118416e-08, 5.11198050461235e-08, 8.76088766854113e-08}},
{"Sony DSLR-A580 iso 1600",  "SONY", "DSLR-A580", 1600,  {7.80707524770107e-05, 2.83480412958222e-05, 5.11544231148663e-05}, {1.95404844425502e-07, 1.28178974678645e-07, 2.0128169309088e-07}},
{"Sony DSLR-A580 iso 3200",  "SONY", "DSLR-A580", 3200,  {0.000154662709645752, 5.67118920492693e-05, 0.000102494994463731}, {5.11017799025018e-07, 3.59561823205766e-07, 5.09363493522508e-07}},
{"Sony DSLR-A580 iso 6400",  "SONY", "DSLR-A580", 6400,  {0.000307287074236872, 0.000114869637912316, 0.000208884357868878}, {1.37463026227533e-06, 1.10418593948196e-06, 1.38857089641668e-06}},
{"Sony DSLR-A580 iso 12800", "SONY", "DSLR-A580", 12800, {0.000642124694335334, 0.000234870187170428, 0.00042600839517458}, {2.97589926922858e-06, 2.7329364294393e-06, 3.14314991846045e-06}},

// sony a77v, contributed by markus
  {"Sony SLT-A77V iso 50",    "SONY", "SLT-A77V", 50,         {1.06869334489541e-05, 4.14233890901092e-06, 7.39824015697458e-06}, {-9.65197592273497e-08, -1.03093231654166e-07, -7.46417677208881e-08}},
  {"Sony SLT-A77V iso 64",    "SONY", "SLT-A77V", 64,         {1.30947455907638e-05, 5.11326057092141e-06, 9.24749420178712e-06}, {-9.93843651760285e-08, -1.31281742891423e-07, -9.15227756127758e-08}},
  {"Sony SLT-A77V iso 80",    "SONY", "SLT-A77V", 80,         {1.55501584523194e-05, 6.25098256336411e-06, 1.107640054148e-05}, {-7.3077153684224e-08, -1.61242344847707e-07, -7.61965923239562e-08}},
  {"Sony SLT-A77V iso 100",   "SONY", "SLT-A77V", 100,        {1.31809413632727e-05, 5.05476215491625e-06, 8.73644297644702e-06}, {-1.28015483855825e-07, -1.34011691841978e-07, -4.85250884830363e-08}},
  {"Sony SLT-A77V iso 125",   "SONY", "SLT-A77V", 125,        {1.55704676979089e-05, 6.25897874229003e-06, 1.0904787591274e-05}, {-7.25334414736029e-08, -1.61968833015564e-07, -6.73409574217066e-08}},
  {"Sony SLT-A77V iso 160",   "SONY", "SLT-A77V", 160,        {1.87667693774766e-05, 7.38038312709447e-06, 1.34460372573579e-05}, {-3.53772825678005e-08, -9.45044500078886e-08, -5.37702746048266e-08}},
  {"Sony SLT-A77V iso 200",   "SONY", "SLT-A77V", 200,        {2.38288320149453e-05, 9.06552008436082e-06, 1.61232027057604e-05}, {-1.41043857332797e-07, -1.21126303453853e-07, -1.70253718051751e-08}},
  {"Sony SLT-A77V iso 250",   "SONY", "SLT-A77V", 250,        {2.82945623674893e-05, 1.12366860730885e-05, 2.01653789079247e-05}, {4.25411399334958e-09, -1.47348259111106e-07, -2.80423691214106e-09}},
  {"Sony SLT-A77V iso 320",   "SONY", "SLT-A77V", 320,        {3.61305426836502e-05, 1.37868260264099e-05, 2.42858903561649e-05}, {-1.15209328135853e-07, -1.11707265335768e-07, 1.18464338110691e-07}},
  {"Sony SLT-A77V iso 400",   "SONY", "SLT-A77V", 400,        {4.34635439914684e-05, 1.72063913332607e-05, 3.13533956669611e-05}, {9.60401316754637e-08, -1.36037056125793e-07, 4.17116925303251e-08}},
  {"Sony SLT-A77V iso 500",   "SONY", "SLT-A77V", 500,        {5.40267835979583e-05, 2.11173204743383e-05, 3.91052277059884e-05}, {9.34995495963503e-08, -9.7594978117092e-08, 2.52734046302189e-08}},
  {"Sony SLT-A77V iso 640",   "SONY", "SLT-A77V", 640,        {6.68698215209594e-05, 2.64477642672602e-05, 4.84578920870458e-05}, {2.34981280995543e-07, -1.02799250473197e-07, 1.44434931881201e-07}},
  {"Sony SLT-A77V iso 800",   "SONY", "SLT-A77V", 800,        {8.34289019309174e-05, 3.30007976579265e-05, 6.05452269966379e-05}, {3.05771416800081e-07, -6.9619888897371e-08, 2.10142634946536e-07}},
  {"Sony SLT-A77V iso 1000",  "SONY", "SLT-A77V", 1000,       {0.000104328656642473, 4.10814551378428e-05, 7.50735394785421e-05}, {4.63130229070292e-07, 2.74797042035447e-08, 4.56886527715291e-07}},
  {"Sony SLT-A77V iso 1250",  "SONY", "SLT-A77V", 1250,       {0.000138487665101736, 5.19140957380518e-05, 8.63871939506887e-05}, {1.21525756002003e-06, 2.70831612109747e-08, 5.21066545439143e-07}},
  {"Sony SLT-A77V iso 1600",  "SONY", "SLT-A77V", 1600,       {0.000167698857704204, 6.41862982068095e-05, 0.000114329833043498}, {3.07215892324366e-07, 2.89432040927619e-07, 1.29612275899272e-06}},
  {"Sony SLT-A77V iso 2000",  "SONY", "SLT-A77V", 2000,       {0.000210744328326873, 8.11268389161478e-05, 0.000143802066080685}, {5.56475316378298e-07, 3.14013732294904e-07, 1.7561046983726e-06}},
  {"Sony SLT-A77V iso 2500",  "SONY", "SLT-A77V", 2500,       {0.000254850408877982, 0.000101056388449369, 0.000186051652530665}, {2.29868409729015e-06, 8.24742159672447e-07, 1.8972750991215e-06}},
  {"Sony SLT-A77V iso 3200",  "SONY", "SLT-A77V", 3200,       {0.000322363481515409, 0.000127241759865442, 0.000233220810438255}, {3.21683653668368e-06, 1.25236771357409e-06, 3.16563998646218e-06}},
  {"Sony SLT-A77V iso 4000",  "SONY", "SLT-A77V", 4000,       {0.000402104709926253, 0.000159616170230224, 0.000284045195551788}, {4.98767974403167e-06, 2.66977260500526e-06, 5.85840190013217e-06}},
  {"Sony SLT-A77V iso 5000",  "SONY", "SLT-A77V", 5000,       {0.000510271081306834, 0.000197087649695034, 0.000363430347936814}, {7.5155505637508e-06, 5.18203575684096e-06, 8.28413254559644e-06}},
  {"Sony SLT-A77V iso 6400",  "SONY", "SLT-A77V", 6400,       {0.000408679090175249, 0.000164225969295612, 0.000293750486277163}, {7.95831683253149e-06, 4.79457365934888e-06, 8.18895058964914e-06}},
  {"Sony SLT-A77V iso 8000",  "SONY", "SLT-A77V", 8000,       {0.000533765422923509, 0.000208173784975916, 0.000376677870455242}, {9.71292834686994e-06, 7.37094755735181e-06, 1.15530821941359e-05}},
  {"Sony SLT-A77V iso 10000", "SONY", "SLT-A77V", 10000,      {0.000655341337358078, 0.000264112424452394, 0.00045616906637917}, {1.63503379245944e-05, 1.2145597953482e-05, 1.95597171469294e-05}},
  {"Sony SLT-A77V iso 12800", "SONY", "SLT-A77V", 12800,      {0.000855372868363557, 0.000342792401386444, 0.000619369386944913}, {2.02328782627859e-05, 1.51917856155466e-05, 2.20074548399319e-05}},
  {"Sony SLT-A77V iso 16000", "SONY", "SLT-A77V", 16000,      {0.00109689287020687, 0.000443396498924019, 0.000772335388155414}, {2.7102241242098e-05, 2.19871484843344e-05, 3.34934514247222e-05}},

// sony a700 contributed by wolf kuehnel
  {"Sony DSLR-A700 iso 100",  "SONY", "DSLR-A700", 100,  {5.33211796567828e-06, 5.33211796567828e-06, 5.33211796567828e-06}, {-8.06059627840588e-08, -8.06059627840588e-08, -8.06059627840588e-08}},
  {"Sony DSLR-A700 iso 125",  "SONY", "DSLR-A700", 125,  {6.56375123512217e-06, 6.56375123512217e-06, 6.56375123512217e-06}, {-2.55641591259403e-08, -2.55641591259403e-08, -2.55641591259403e-08}},
  {"Sony DSLR-A700 iso 160",  "SONY", "DSLR-A700", 160,  {7.14628696578123e-06, 7.14628696578123e-06, 7.14628696578123e-06}, {1.35993082026043e-08, 1.35993082026043e-08, 1.35993082026043e-08}},
  {"Sony DSLR-A700 iso 200",  "SONY", "DSLR-A700", 200,  {6.35083595400887e-06, 6.35083595400887e-06, 6.35083595400887e-06}, {-1.62812666388806e-07, -1.62812666388806e-07, -1.62812666388806e-07}},
  {"Sony DSLR-A700 iso 250",  "SONY", "DSLR-A700", 250,  {7.01839295035388e-06, 7.01839295035388e-06, 7.01839295035388e-06}, {5.3500142820507e-08, 5.3500142820507e-08, 5.3500142820507e-08}},
  {"Sony DSLR-A700 iso 320",  "SONY", "DSLR-A700", 320,  {9.65036774717935e-06, 9.65036774717935e-06, 9.65036774717935e-06}, {2.45331447348584e-07, 2.45331447348584e-07, 2.45331447348584e-07}},
  {"Sony DSLR-A700 iso 400",  "SONY", "DSLR-A700", 400,  {1.20060675549046e-05, 1.20060675549046e-05, 1.20060675549046e-05}, {3.96465116898102e-07, 3.96465116898102e-07, 3.96465116898102e-07}},
  {"Sony DSLR-A700 iso 500",  "SONY", "DSLR-A700", 500,  {1.38033057254465e-05, 1.38033057254465e-05, 1.38033057254465e-05}, {3.55894028715277e-07, 3.55894028715277e-07, 3.55894028715277e-07}},
  {"Sony DSLR-A700 iso 640",  "SONY", "DSLR-A700", 640,  {1.6784463838112e-05, 1.6784463838112e-05, 1.6784463838112e-05}, {2.28180575432325e-07, 2.28180575432325e-07, 2.28180575432325e-07}},
  {"Sony DSLR-A700 iso 800",  "SONY", "DSLR-A700", 800,  {1.96000303340923e-05, 1.96000303340923e-05, 1.96000303340923e-05}, {1.84656393940007e-07, 1.84656393940007e-07, 1.84656393940007e-07}},
  {"Sony DSLR-A700 iso 1000", "SONY", "DSLR-A700", 1000, {2.39942978637023e-05, 2.39942978637023e-05, 2.39942978637023e-05}, {1.30423876463074e-07, 1.30423876463074e-07, 1.30423876463074e-07}},
  {"Sony DSLR-A700 iso 1250", "SONY", "DSLR-A700", 1250, {2.99414647800315e-05, 2.99414647800315e-05, 2.99414647800315e-05}, {1.25851775185069e-07, 1.25851775185069e-07, 1.25851775185069e-07}},
  {"Sony DSLR-A700 iso 1600", "SONY", "DSLR-A700", 1600, {2.79523297751563e-05, 2.79523297751563e-05, 2.79523297751563e-05}, {-6.95836889595576e-07, -6.95836889595576e-07, -6.95836889595576e-07}},
  {"Sony DSLR-A700 iso 2000", "SONY", "DSLR-A700", 2000, {3.43885587172716e-05, 3.43885587172716e-05, 3.43885587172716e-05}, {-1.18312709804912e-06, -1.18312709804912e-06, -1.18312709804912e-06}},
  {"Sony DSLR-A700 iso 2500", "SONY", "DSLR-A700", 2500, {4.243083672372e-05, 4.243083672372e-05, 4.243083672372e-05}, {-7.21932721245921e-07, -7.21932721245921e-07, -7.21932721245921e-07}},
  {"Sony DSLR-A700 iso 3200", "SONY", "DSLR-A700", 3200, {5.30312547388233e-05, 5.30312547388233e-05, 5.30312547388233e-05}, {-1.66425315126866e-07, -1.66425315126866e-07, -1.66425315126866e-07}},
  {"Sony DSLR-A700 iso 4000", "SONY", "DSLR-A700", 4000, {4.12115852399434e-05, 4.12115852399434e-05, 4.12115852399434e-05}, {-1.48982485243939e-06, -1.48982485243939e-06, -1.48982485243939e-06}},
  {"Sony DSLR-A700 iso 5000", "SONY", "DSLR-A700", 5000, {6.68026492277847e-05, 6.68026492277847e-05, 6.68026492277847e-05}, {-3.95313049198595e-06, -3.95313049198595e-06, -3.95313049198595e-06}},
  {"Sony DSLR-A700 iso 6400", "SONY", "DSLR-A700", 6400, {9.75808014924563e-05, 9.75808014924563e-05, 9.75808014924563e-05}, {-4.76703830629088e-06, -4.76703830629088e-06, -4.76703830629088e-06}},

  //sony a850 contributed by ben robbins
  {"Sony DSLR-A850 iso 100",       "SONY",      "DSLR-A850",              100,         {8.25699463069653e-06, 3.91126678645273e-06, 6.55120985640887e-06}, {-1.59018296223033e-07, -1.56989751925506e-07, 1.53342310731794e-07}},
  {"Sony DSLR-A850 iso 125",       "SONY",      "DSLR-A850",              125,         {1.05872808089519e-05, 5.07545912146522e-06, 8.56576247205075e-06}, {-1.49605047226957e-07, -1.72114992984391e-07, 2.20605143576061e-07}},
  {"Sony DSLR-A850 iso 160",       "SONY",      "DSLR-A850",              160,         {1.28119366043084e-05, 5.99658506222075e-06, 9.55132770735817e-06}, {-1.739748208547e-07, -1.77270568456809e-07, 4.00680814881945e-07}},
  {"Sony DSLR-A850 iso 200",       "SONY",      "DSLR-A850",              200,         {1.00840025279489e-05, 4.82157432184899e-06, 7.94285873724355e-06}, {-8.49505084351588e-08, -1.12861446582006e-07, 3.07499428271694e-07}},
  {"Sony DSLR-A850 iso 250",       "SONY",      "DSLR-A850",              250,         {1.26817233651916e-05, 5.85148543962174e-06, 8.77221090475886e-06}, {-1.35355117210407e-07, -1.27534715267785e-07, 5.13491008909241e-07}},
  {"Sony DSLR-A850 iso 320",       "SONY",      "DSLR-A850",              320,         {1.53951642671946e-05, 7.26971977357975e-06, 1.14402387260519e-05}, {-7.19086335578946e-08, -9.94658928343374e-08, 6.26955676102515e-07}},
  {"Sony DSLR-A850 iso 400",       "SONY",      "DSLR-A850",              400,         {2.02624223796714e-05, 9.57834434797659e-06, 1.39569987929561e-05}, {-6.83030107884948e-08, -1.09231315954154e-07, 1.01354203023742e-06}},
  {"Sony DSLR-A850 iso 500",       "SONY",      "DSLR-A850",              500,         {2.29795698185462e-05, 1.08087959929279e-05, 1.59775298984423e-05}, {7.95471496727003e-08, 8.70166506856567e-09, 1.28702035982608e-06}},
  {"Sony DSLR-A850 iso 640",       "SONY",      "DSLR-A850",              640,         {3.03540578864712e-05, 1.42702243912902e-05, 2.06123445203878e-05}, {1.1420888729045e-08, -9.67262461429989e-08, 1.68476481030001e-06}},
  {"Sony DSLR-A850 iso 800",       "SONY",      "DSLR-A850",              800,         {3.7639630788085e-05, 1.77841900996972e-05, 2.53571581146633e-05}, {1.70389659248582e-07, 6.90370057641215e-08, 2.32125819318808e-06}},
  {"Sony DSLR-A850 iso 1000",      "SONY",      "DSLR-A850",              1000,        {4.56250242276957e-05, 2.12815494210054e-05, 2.97138355738333e-05}, {2.34878175241155e-07, 1.39543978486238e-07, 2.86156923899171e-06}},
  {"Sony DSLR-A850 iso 1250",      "SONY",      "DSLR-A850",              1250,        {5.85215970426083e-05, 2.69684821848485e-05, 3.7973119431936e-05}, {2.83307838291734e-07, 2.62572126060244e-07, 3.88434755923413e-06}},
  {"Sony DSLR-A850 iso 1600",      "SONY",      "DSLR-A850",              1600,        {7.39971679736878e-05, 3.52279913601585e-05, 4.95129790940977e-05}, {1.21977349139068e-06, 6.55854348052585e-07, 5.42054665031628e-06}},
  {"Sony DSLR-A850 iso 2000",      "SONY",      "DSLR-A850",              2000,        {9.1302060516722e-05, 4.15600990054158e-05, 5.68845982420946e-05}, {1.05861475862399e-06, 9.03991128124208e-07, 6.82424773748911e-06}},
  {"Sony DSLR-A850 iso 2500",      "SONY",      "DSLR-A850",              2500,        {0.000117600626116785, 5.28166864087072e-05, 7.09155486334467e-05}, {1.63395062766912e-06, 1.50641726639289e-06, 9.26413633170439e-06}},
  {"Sony DSLR-A850 iso 3200",      "SONY",      "DSLR-A850",              3200,        {0.000161191942490328, 7.06299178984888e-05, 9.48579069462842e-05}, {1.95407596984656e-06, 2.05515958334302e-06, 1.18874876740248e-05}},
  {"Sony DSLR-A850 iso 4000",      "SONY",      "DSLR-A850",              4000,        {0.000187439552117757, 8.53799594823098e-05, 0.000114349838984479}, {3.20554061523482e-06, 2.78672823612179e-06, 1.49505067770576e-05}},
  {"Sony DSLR-A850 iso 5000",      "SONY",      "DSLR-A850",              5000,        {0.000242576861803702, 0.000108789166926406, 0.000143515511697319}, {5.23648797679808e-06, 4.83098521698916e-06, 2.11302071617619e-05}},
  {"Sony DSLR-A850 iso 6400",      "SONY",      "DSLR-A850",              6400,        {0.000322459422202308, 0.000147836850843186, 0.00018682176773794}, {8.42502696364615e-06, 6.23249233077622e-06, 2.92004841913428e-05}},

// sony a99v by Wiebe Bierma
{"Sony SLT-A99V iso 50",    "SONY", "SLT-A99V", 50,    {3.65549979880484e-06, 1.40105286464966e-06, 1.92810652964636e-06}, {-5.36500252669615e-09, -3.48418346565663e-08, 9.08974507143714e-09}},
// {"Sony SLT-A99V iso 50",    "SONY", "SLT-A99V", 50,    {3.637165004921e-06, 1.44491632983933e-06, 2.00630487048374e-06}, {-7.5682869372329e-10, -4.02743701100181e-08, -4.89027722402962e-09}},
{"Sony SLT-A99V iso 64",    "SONY", "SLT-A99V", 64,    {5.02882914657062e-06, 1.78268963769324e-06, 2.63945509966055e-06}, {-6.13349025430313e-08, -5.46844235420211e-08, -2.59401987566088e-08}},
// {"Sony SLT-A99V iso 64",    "SONY", "SLT-A99V", 64,    {4.96608621495819e-06, 1.76470045074347e-06, 2.53480401945617e-06}, {-5.4856568749775e-08, -5.22169479322759e-08, -9.87240883796342e-09}},
{"Sony SLT-A99V iso 80",    "SONY", "SLT-A99V", 80,    {3.07489802706582e-06, 8.66054167705384e-07, 3.01662145793843e-06}, {2.44830590540107e-07, 2.67542544536091e-07, 6.95419798858375e-08}},
// {"Sony SLT-A99V iso 80",    "SONY", "SLT-A99V", 80,    {2.87408018151637e-06, 9.27360925043607e-07, 3.0780225231659e-06}, {2.64263477662232e-07, 2.52388252389167e-07, 6.12765612610866e-08}},
{"Sony SLT-A99V iso 100",   "SONY", "SLT-A99V", 100,   {1.43363607249646e-06, 1.56249605803851e-06, 2.57533939391566e-06}, {3.27025291080325e-07, 7.44119919120102e-09, -2.77405570886066e-08}},
// {"Sony SLT-A99V iso 100",   "SONY", "SLT-A99V", 100,   {1.33493680348679e-06, 1.63232590765928e-06, 2.6092818480025e-06}, {3.36561936741923e-07, -1.0839275927153e-08, -3.27140845175999e-08}},
{"Sony SLT-A99V iso 125",   "SONY", "SLT-A99V", 125,   {1.52628542100174e-06, 1.78569279453577e-06, 3.23486201131121e-06}, {4.29917041884351e-07, 5.23836124761297e-08, -3.67362909076892e-08}},
// {"Sony SLT-A99V iso 125",   "SONY", "SLT-A99V", 125,   {1.51646514295835e-06, 1.56951750676638e-06, 3.24368211696532e-06}, {4.31398779221191e-07, 1.0622891845526e-07, -4.1322820632474e-08}},
{"Sony SLT-A99V iso 160",   "SONY", "SLT-A99V", 160,   {7.37807056870937e-06, 2.78026579196862e-06, 3.95971867585528e-06}, {-3.68588589581115e-08, -7.62416819326854e-08, -3.29928676397445e-08}},
// {"Sony SLT-A99V iso 160",   "SONY", "SLT-A99V", 160,   {7.39421997211837e-06, 2.80829598759127e-06, 3.92676263134345e-06}, {-4.22949668209108e-08, -8.90734116340483e-08, -3.05007704190611e-08}},
{"Sony SLT-A99V iso 200",   "SONY", "SLT-A99V", 200,   {9.02435094971006e-06, 3.2134963865061e-06, 4.71260839775169e-06}, {-4.97518218658821e-08, -4.1691023482117e-08, -1.34565299020197e-08}},
// {"Sony SLT-A99V iso 200",   "SONY", "SLT-A99V", 200,   {8.96430041293435e-06, 3.21238426275607e-06, 4.70829833273476e-06}, {-4.32632137729354e-08, -4.23299872601304e-08, -1.18703249058611e-08}},
{"Sony SLT-A99V iso 250",   "SONY", "SLT-A99V", 250,   {1.10906146712533e-05, 3.83383758507048e-06, 5.67167427741373e-06}, {-5.99508127523777e-08, -3.53473244846302e-08, 3.44016226991299e-08}},
// {"Sony SLT-A99V iso 250",   "SONY", "SLT-A99V", 250,   {1.13643100127743e-05, 3.84516051665032e-06, 5.64984712538696e-06}, {-9.04506767637854e-08, -4.04153767846374e-08, 3.70187683217084e-08}},
{"Sony SLT-A99V iso 320",   "SONY", "SLT-A99V", 320,   {1.39217213788624e-05, 4.92300177402749e-06, 7.16621956811215e-06}, {-7.74194582884942e-08, -7.99453984076374e-08, 4.3794770223202e-08}},
// {"Sony SLT-A99V iso 320",   "SONY", "SLT-A99V", 320,   {1.3857739659116e-05, 4.80511520306495e-06, 7.13540983336429e-06}, {-8.2301657326459e-08, -6.18272347528958e-08, 4.09406476187755e-08}},
{"Sony SLT-A99V iso 400",   "SONY", "SLT-A99V", 400,   {1.30157168833324e-05, 4.8433561717754e-06, 7.61094270814464e-06}, {2.62651658617957e-07, 2.30630303073832e-07, 4.10343501771413e-07}},
{"Sony SLT-A99V iso 500",   "SONY", "SLT-A99V", 500,   {1.65576156960172e-05, 6.13112569173218e-06, 8.51134441609508e-06}, {2.76787102247103e-07, 2.74225926626671e-07, 6.59050730412936e-07}},
{"Sony SLT-A99V iso 640",   "SONY", "SLT-A99V", 640,   {2.85143914703534e-05, 6.35610747434602e-06, 8.25662769198497e-06}, {-6.29902148827959e-07, 5.97675823539628e-07, 1.13238076574513e-06}},
// {"Sony SLT-A99V iso 640",   "SONY", "SLT-A99V", 640,   {2.16190186880649e-05, 6.90751181664985e-06, 1.39057335018538e-05}, {1.98147210714457e-07, 4.81463816908514e-07, 2.6740747221827e-07}},
{"Sony SLT-A99V iso 800",   "SONY", "SLT-A99V", 800,   {2.63310017706233e-05, 1.0321343513389e-05, 2.27483273161852e-05}, {2.48607775040229e-07, 1.0722940792174e-07, -5.31898166791106e-07}},
{"Sony SLT-A99V iso 1000",  "SONY", "SLT-A99V", 1000,  {3.28961649003385e-05, 1.41962637282192e-05, 2.57239222895527e-05}, {3.27010566962727e-07, -1.3668004315489e-07, -2.17066916759445e-07}},
// {"Sony SLT-A99V iso 1000",  "SONY", "SLT-A99V", 1000,  {3.11300894768422e-05, 1.29120409413954e-05, 2.32743628359793e-05}, {5.63177275508448e-07, 1.67504552902093e-07, 1.28648774450632e-07}},
// {"Sony SLT-A99V iso 1000",  "SONY", "SLT-A99V", 1000,  {3.29355569596589e-05, 1.40486735991332e-05, 2.65293278958653e-05}, {3.31441635826892e-07, -1.17669991519604e-07, -3.91074689199344e-07}},
{"Sony SLT-A99V iso 1250",  "SONY", "SLT-A99V", 1250,  {4.099128199989e-05, 1.67650997428554e-05, 2.96876964904731e-05}, {4.89838810390202e-07, 6.72075053460358e-08, 1.42708334334994e-07}},
// {"Sony SLT-A99V iso 1250",  "SONY", "SLT-A99V", 1250,  {3.83511592420526e-05, 1.65452445823787e-05, 3.07886359707115e-05}, {6.73609321313772e-07, 9.4616449206151e-08, -7.18415158585883e-08}},
{"Sony SLT-A99V iso 1600",  "SONY", "SLT-A99V", 1600,  {5.1120417083476e-05, 2.08470457014776e-05, 3.38578623283221e-05}, {7.67282727366463e-07, 1.46526690361417e-07, 6.47366596727232e-07}},
// {"Sony SLT-A99V iso 1600",  "SONY", "SLT-A99V", 1600,  {4.87272523876591e-05, 2.03932823125932e-05, 3.25805833832426e-05}, {1.056581945361e-06, 2.35588614381848e-07, 9.41419266771463e-07}},
{"Sony SLT-A99V iso 2000",  "SONY", "SLT-A99V", 2000,  {6.24408141268579e-05, 2.58702403355454e-05, 4.40273955215723e-05}, {1.13088555609574e-06, 2.50652243375364e-07, 7.58596523580091e-07}},
// {"Sony SLT-A99V iso 2000",  "SONY", "SLT-A99V", 2000,  {6.3707871726411e-05, 2.65791291700258e-05, 5.15236270160766e-05}, {8.0133746673955e-07, 5.20955187101269e-08, -4.30464071899934e-07}},
{"Sony SLT-A99V iso 2500",  "SONY", "SLT-A99V", 2500,  {7.75232642394524e-05, 3.21744377809407e-05, 5.9655135731338e-05}, {1.27439231469351e-06, 4.18641597167509e-07, 1.51820848620658e-07}},
// {"Sony SLT-A99V iso 2500",  "SONY", "SLT-A99V", 2500,  {8.12494838993552e-05, 3.36417776100578e-05, 4.82184891783629e-05}, {1.37530755345541e-06, 1.27605487096942e-07, 2.16781322196502e-06}},
{"Sony SLT-A99V iso 3200",  "SONY", "SLT-A99V", 3200,  {0.000112641232541059, 4.37080589412173e-05, 7.27922890682123e-05}, {4.93424258138145e-07, -1.16680012169657e-07, 5.2588499784782e-07}},
// {"Sony SLT-A99V iso 3200",  "SONY", "SLT-A99V", 3200,  {0.000114376156921559, 4.22066831839155e-05, 7.56977044007406e-05}, {4.41911604758481e-07, 3.07792216060875e-07, 1.44719475970204e-07}},
{"Sony SLT-A99V iso 4000",  "SONY", "SLT-A99V", 4000,  {0.000145251819416401, 5.35407119062237e-05, 8.55232518677131e-05}, {5.81109856840517e-07, 3.59693859917185e-07, 2.11014872843522e-06}},
// {"Sony SLT-A99V iso 4000",  "SONY", "SLT-A99V", 4000,  {0.000141426177506496, 5.38624412209036e-05, 7.97917799870945e-05}, {1.04612799295974e-06, 2.58750723256819e-07, 3.19172138070941e-06}},
{"Sony SLT-A99V iso 5000",  "SONY", "SLT-A99V", 5000,  {0.000111236687510301, 7.41920051309241e-05, 3.71461185629658e-05}, {2.26657415738805e-06, -1.06416274531768e-06, 2.2247579716874e-05}},
// {"Sony SLT-A99V iso 5000",  "SONY", "SLT-A99V", 5000,  {0.00011842467114375, 7.45269857091031e-05, 2.89478113626437e-05}, {1.47716468191251e-06, -9.81096096918866e-07, 2.28523407686102e-05}},
// {"Sony SLT-A99V iso 5000",  "SONY", "SLT-A99V", 5000,  {0.000180449523979157, 6.69826627339248e-05, 0.000105025784104389}, {4.57792784755193e-07, 5.56143043359499e-07, 3.60249913946547e-06}},
// {"Sony SLT-A99V iso 5000",  "SONY", "SLT-A99V", 5000,  {0.00021066512864858, 6.73168425069236e-05, 0.000102695178974684}, {-2.71253482482804e-06, 5.54700022126668e-07, 4.02733864705692e-06}},
{"Sony SLT-A99V iso 6400",  "SONY", "SLT-A99V", 6400,  {0.000149273781732163, 9.35026508765124e-05, 2.51109849153462e-05}, {2.14583366856323e-06, -1.26270432986176e-06, 2.97218516519577e-05}},
// {"Sony SLT-A99V iso 6400",  "SONY", "SLT-A99V", 6400,  {0.000135058925285655, 9.4953214611601e-05, 2.3744331046267e-05}, {3.4990346882114e-06, -1.87673225127924e-06, 3.05578057417447e-05}},
// {"Sony SLT-A99V iso 6400",  "SONY", "SLT-A99V", 6400,  {0.000244821281697665, 8.41672098156532e-05, 0.000129477008295143}, {-1.12300868389575e-06, 9.62785820626575e-07, 5.31975568118068e-06}},
// {"Sony SLT-A99V iso 6400",  "SONY", "SLT-A99V", 6400,  {0.000257365868047472, 8.46689452353835e-05, 0.000132465627915025}, {-2.48622545548508e-06, 8.48055939041919e-07, 4.78072451793209e-06}},
{"Sony SLT-A99V iso 8000",  "SONY", "SLT-A99V", 8000,  {0.000180271571911152, 0.000119594736450976, 9.00804341661919e-06}, {3.96987876459854e-06, -2.26955010321124e-06, 3.85545601723348e-05}},
// {"Sony SLT-A99V iso 8000",  "SONY", "SLT-A99V", 8000,  {0.000165913655544096, 0.000113199973526842, 1.01311005230367e-05}, {5.30283215713576e-06, -4.71822182525008e-07, 3.9101658889108e-05}},
// {"Sony SLT-A99V iso 8000",  "SONY", "SLT-A99V", 8000,  {0.000464109509393142, 0.00010681135456587, 0.000162035904943231}, {-1.79119055589005e-05, 1.29707215705487e-06, 6.87402358538333e-06}},
// {"Sony SLT-A99V iso 8000",  "SONY", "SLT-A99V", 8000,  {0.000451785374246284, 0.00010688476224788, 0.000160858500363835}, {-1.56105385780126e-05, 1.24554313819218e-06, 6.8159965646697e-06}},
{"Sony SLT-A99V iso 10000", "SONY", "SLT-A99V", 10000, {0.000345165245981186, 0.000137187560341786, 0.000193606244617899}, {4.52996571743685e-06, 1.67606572220736e-06, 1.11975633223465e-05}},
// {"Sony SLT-A99V iso 10000", "SONY", "SLT-A99V", 10000, {0.000339975062881932, 0.000137944544669164, 0.000193159312523945}, {5.0924806843355e-06, 1.5476347298456e-06, 1.08802543060995e-05}},
{"Sony SLT-A99V iso 12800", "SONY", "SLT-A99V", 12800, {0.000459316160190123, 0.000168382735992912, 0.000257604319227403}, {2.71669001389203e-06, 3.63092924183959e-06, 1.31382910342086e-05}},
// {"Sony SLT-A99V iso 12800", "SONY", "SLT-A99V", 12800, {0.000519647997216212, 0.00016973738945782, 0.000249035369933578}, {-3.41563589020247e-06, 3.29377034550418e-06, 1.36402850132935e-05}},
{"Sony SLT-A99V iso 16000", "SONY", "SLT-A99V", 16000, {0.000533102341403761, 0.00019280742303651, 0.000326491338988787}, {9.58955976376947e-06, 1.10158164202499e-05, 1.39591858885643e-05}},
{"Sony SLT-A99V iso 20000", "SONY", "SLT-A99V", 20000, {0.000677111177714143, 0.000307697963256919, 0.000392921099365228}, {1.52773661075724e-05, 2.14094551303812e-08, 2.25660960659765e-05}},
// {"Sony SLT-A99V iso 20000", "SONY", "SLT-A99V", 20000, {0.000674624199073345, 0.00023849001282811, 0.000431233016275474}, {1.52344288980014e-05, 1.76333246197936e-05, 1.60087946993446e-05}},
{"Sony SLT-A99V iso 25600", "SONY", "SLT-A99V", 25600, {0.000737326229317385, 0.000348048804828957, 0.000570742164093176}, {3.23217230486979e-05, 1.24886230061761e-05, 1.94331357714058e-05}},
// {"Sony SLT-A99V iso 25600", "SONY", "SLT-A99V", 25600, {0.000755035204997949, 0.000333628609161742, 0.000537508995512912}, {3.07703253612089e-05, 1.59081615635191e-05, 3.49475725257455e-05}},
// {"Sony SLT-A99V iso 25600", "SONY", "SLT-A99V", 25600, {0.000727023003102683, 0.000332680010779511, 0.000550858764725579}, {3.32672366834365e-05, 1.62812481778224e-05, 3.36833358782068e-05}},

// sony nex-3 contributed by wolf kuehnel
  {"Sony NEX-3 iso 200",   "SONY", "NEX-3", 200,   {1.4672503408991e-05, 5.84425408801505e-06, 8.51567050090639e-06}, {-1.40558077705532e-07, -1.79634870972196e-07, -5.82154575649645e-09}},
  {"Sony NEX-3 iso 400",   "SONY", "NEX-3", 400,   {2.85729500727844e-05, 1.12075456534149e-05, 1.64240651900713e-05}, {-2.80095894819118e-07, -3.2585492411101e-07, 1.52692521502707e-08}},
  {"Sony NEX-3 iso 800",   "SONY", "NEX-3", 800,   {5.64018950241466e-05, 2.18440717962903e-05, 3.19271650360552e-05}, {-5.51440242439692e-07, -5.46716315709677e-07, 8.56264363501028e-08}},
  {"Sony NEX-3 iso 1600",  "SONY", "NEX-3", 1600,  {0.000110606038030576, 4.29559287707001e-05, 6.82182930869616e-05}, {-6.71739627054785e-07, -7.52691101796091e-07, -5.32592920747352e-07}},
  {"Sony NEX-3 iso 3200",  "SONY", "NEX-3", 3200,  {0.000182717941876504, 6.97233934810826e-05, 0.000117553774459971}, {5.09186888587266e-07, 6.96349058974406e-07, 2.76512590304654e-07}},
  {"Sony NEX-3 iso 6400",  "SONY", "NEX-3", 6400,  {0.000360401971525734, 0.00013774697745779, 0.00023878775486719}, {2.72892083204603e-06, 3.10010867166969e-06, 2.21995484611605e-06}},
  {"Sony NEX-3 iso 12800", "SONY", "NEX-3", 12800, {0.000753756914907995, 0.000273472039349551, 0.000480830708576147}, {6.58727332505663e-06, 1.27237676174246e-05, 9.43075645199121e-06}},

// sony nex-c3 contributed by dennis gnad - the sensor should be similar/same as: Nikon D7000, Sony SLT-A35, Sony Nex-5N
  {"Sony NEX-C3 iso 200",   "SONY", "NEX-C3", 200,   {1.1337620851509e-05, 3.99049305799728e-06, 6.82229177247925e-06}, {-7.30723848631144e-08, -8.63065550655657e-08, -6.62345259016437e-08}},
  {"Sony NEX-C3 iso 400",   "SONY", "NEX-C3", 400,   {2.11480201302754e-05, 7.2650189129159e-06, 1.25432425132794e-05}, {5.65741564406398e-08, -4.37629425464589e-08, -1.07472171366622e-07}},
  {"Sony NEX-C3 iso 800",   "SONY", "NEX-C3", 800,   {3.5943971137657e-05, 1.33463905807797e-05, 2.38594290631523e-05}, {5.19092588639411e-07, 3.84015338520807e-08, -1.44821398239112e-07}},
  {"Sony NEX-C3 iso 1600",  "SONY", "NEX-C3", 1600,  {7.18786159941266e-05, 2.58241923585725e-05, 4.24042651982938e-05}, {7.78894710455171e-07, 2.05323822406154e-07, 3.84018978571853e-07}},
  {"Sony NEX-C3 iso 3200",  "SONY", "NEX-C3", 3200,  {0.00014051748495203, 5.13333793109814e-05, 9.16550598756885e-05}, {2.14290681263339e-06, 5.94476955815332e-07, -2.17460313294154e-07}},
  {"Sony NEX-C3 iso 6400",  "SONY", "NEX-C3", 6400,  {0.000274870719002775, 0.000104515238705256, 0.000182759472464313}, {5.54238235375329e-06, 1.58508135754313e-06, 7.69486976636118e-07}},
  {"Sony NEX-C3 iso 12800", "SONY", "NEX-C3", 12800, {0.000576335303431959, 0.000208842331949395, 0.000361086091669198}, {1.15611448710996e-05, 7.79335765229272e-06, 7.10490981780029e-06}},
// second set of exposures, underexposed blue channel(?)
//{"Sony NEX-C3 iso 200",   "SONY", "NEX-C3", 200,   {6.29675637391445e-06, 3.94853430067636e-06, -3.91773599676108e-06}, {-9.38474279364356e-09, -4.73816050461171e-08, 1.84055345614349e-06}},
//{"Sony NEX-C3 iso 400",   "SONY", "NEX-C3", 400,   {1.200349303506e-05, 7.39177841878958e-06, -7.45585264334457e-06}, {9.8848414995636e-09, -3.92261113336415e-08, 3.45821062441188e-06}},
//{"Sony NEX-C3 iso 800",   "SONY", "NEX-C3", 800,   {2.31370771270764e-05, 1.433508965108e-05, -1.47508377793459e-05}, {9.29486537050227e-08, -5.27746106970043e-09, 6.75755367000162e-06}},
//{"Sony NEX-C3 iso 1600",  "SONY", "NEX-C3", 1600,  {4.43983961851617e-05, 2.74425555821729e-05, -2.91515438190989e-05}, {3.4412025249595e-07, 1.41623368095983e-07, 1.31236687296088e-05}},
//{"Sony NEX-C3 iso 3200",  "SONY", "NEX-C3", 3200,  {8.86753387156914e-05, 5.39449927257143e-05, -5.61734962923041e-05}, {9.22374988898576e-07, 7.29396350919297e-07, 2.60956580465855e-05}},
//{"Sony NEX-C3 iso 6400",  "SONY", "NEX-C3", 6400,  {0.000176875263851587, 0.000108912530806839, -0.000118038844258225}, {2.61219242973953e-06, 2.10092669042785e-06, 5.31243696672049e-05}},
//{"Sony NEX-C3 iso 12800", "SONY", "NEX-C3", 12800, {0.000357867270809397, 0.00022096513727134, -0.000235779703214882}, {8.11275941967016e-06, 5.94548736442806e-06, 0.000108825576744117}},

// sony nex-5n, contributed by richard tollerton
  {"Sony NEX-5N iso 100",  "SONY","NEX-5N",   100, {5.23654750640590e-06, 2.16336531359531e-06, 4.05438567111174e-06}, {-3.76222098538270e-08, -4.55499179692451e-08, -5.24504376459165e-08}},
  {"Sony NEX-5N iso 200",  "SONY","NEX-5N",   200, {9.89787336958434e-06, 3.85507188905301e-06, 7.49692257247810e-06}, {-5.54481887402481e-08, -2.83747348743495e-08, -6.93849808682778e-08}},
  {"Sony NEX-5N iso 400",  "SONY","NEX-5N",   400, {1.84576752382422e-05, 7.55269911607910e-06, 1.39750567698195e-05}, {-2.70397610319952e-08, -6.12470862656113e-08, -3.28834706007868e-08}},
  {"Sony NEX-5N iso 800",  "SONY","NEX-5N",   800, {3.58232242940675e-05, 1.44456516660733e-05, 2.71361763136004e-05}, {-2.98964307135405e-08, -4.67581021138567e-08, -2.00092108475525e-08}},
  {"Sony NEX-5N iso 1600", "SONY","NEX-5N",  1600, {6.99954660206428e-05, 2.76663517551725e-05, 5.23694992682511e-05}, {-6.74618555057466e-08,  1.14696808304538e-07,  2.23994745851368e-07}},
  {"Sony NEX-5N iso 3200", "SONY","NEX-5N",  3200, {0.000134410000854507, 5.47759915449270e-05, 0.000104284170800690}, { 4.01630359888795e-07,  2.85597954643281e-07,  4.06299399165519e-07}},
  {"Sony NEX-5N iso 6400", "SONY","NEX-5N",  6400, {0.000287683129736784, 0.000110667011977831, 0.000208735922363851}, { 4.52454314920345e-07,  9.38727586461764e-07,  6.14812382546788e-07}},
  {"Sony NEX-5N iso 12800","SONY","NEX-5N", 12800, {0.000361829919106923, 0.000147322393107788, 0.000268295239617557}, { 1.74216303364011e-06,  1.81842912617960e-06,  2.65974180422040e-06}},
  {"Sony NEX-5N iso 25600","SONY","NEX-5N", 25600, {0.000697045115805751, 0.000272317396165591, 0.000483759056444745}, { 1.37810086962527e-05,  1.48330264051315e-05,  1.86724331206012e-05}},

// sony nex-5n contributed by Lars Kotthoff, committed as a cross-device comparison with the above
// {"Sony NEX-5N iso 100",   "SONY", "NEX-5N", 100,   {5.36019981349515e-06, 2.39161879881635e-06, 3.90437328013475e-06}, {9.26489571603524e-08, -1.00028685501653e-07, -6.29469293056305e-08}},
// {"Sony NEX-5N iso 200",   "SONY", "NEX-5N", 200,   {8.20959183035892e-06, 4.12044219533825e-06, 6.9388472433381e-06}, {3.73732910812885e-07, -7.5943633322028e-08, -4.55455194442e-08}},
// {"Sony NEX-5N iso 400",   "SONY", "NEX-5N", 400,   {1.43838589725234e-05, 7.73476618196846e-06, 1.32714099547777e-05}, {8.55468679802068e-07, -7.92020827459565e-08, -4.17343628669633e-08}},
// {"Sony NEX-5N iso 800",   "SONY", "NEX-5N", 800,   {2.72181093249944e-05, 1.51318945965584e-05, 2.65693049665937e-05}, {1.84578475491198e-06, -1.0068285107146e-07, -1.21710665490067e-07}},
// {"Sony NEX-5N iso 1600",  "SONY", "NEX-5N", 1600,  {4.72533112402627e-05, 2.95011015682024e-05, 5.04169980117211e-05}, {4.29792258011708e-06, -4.0023963379376e-08, 1.25645152697591e-07}},
// {"Sony NEX-5N iso 3200",  "SONY", "NEX-5N", 3200,  {9.584719723552e-05,   5.72618563397311e-05, 9.83315857258553e-05}, {8.13355062528892e-06, 1.49434347924996e-07, 4.82352932774002e-07}},
// {"Sony NEX-5N iso 6400",  "SONY", "NEX-5N", 6400,  {0.000191755723737552, 0.000115090156026419, 0.00019748377586531}, {1.63916255629036e-05, 6.48507108105109e-07, 1.15239914625567e-06}},
// {"Sony NEX-5N iso 12800", "SONY", "NEX-5N", 12800, {0.000241408282856131, 0.000149598711330745, 0.000248493533516753}, {2.24558412643211e-05, 2.03445385359335e-06, 3.31983658769912e-06}},
// {"Sony NEX-5N iso 25600", "SONY", "NEX-5N", 25600, {0.000651679201516031, 0.0002763638409919, 0.000458492544865695}, {2.96826015365659e-05, 1.51556566613307e-05, 2.05518566445034e-05}},

// sony nex 5r contributed by Matthias Weiss
// {"NEX-5R iso 100",       "SONY",      "NEX-5R",              100,         {6.85699339118397e-06, 2.42003349692509e-06, 3.96996457866612e-06}, {4.90136161516416e-09, 4.92700461914475e-09, 5.685034241437e-09}},
// {"NEX-5R iso 200",       "SONY",      "NEX-5R",              200,         {1.3132937652888e-05, 4.44041074471188e-06, 7.70470006870031e-06}, {1.32977815238454e-08, 1.08125795326979e-08, 1.47702834702206e-08}},
// {"NEX-5R iso 400",       "SONY",      "NEX-5R",              400,         {2.33718113062321e-05, 7.89641810075419e-06, 1.38641154948761e-05}, {1.98811927165727e-08, 1.6859693734491e-08, 2.14152116430129e-08}},
// {"NEX-5R iso 800",       "SONY",      "NEX-5R",              800,         {4.52288774955966e-05, 1.53290498632095e-05, 2.72036278718568e-05}, {3.75878876657179e-08, 3.15065976890198e-08, 4.09093148165382e-08}},
// {"NEX-5R iso 1600",       "SONY",      "NEX-5R",              1600,         {8.7740688366134e-05, 3.01256969229918e-05, 5.36334512973292e-05}, {1.26176810281378e-07, 9.10688271372503e-08, 1.29244900326199e-07}},
// {"NEX-5R iso 3200",       "SONY",      "NEX-5R",              3200,         {0.000180132114681175, 6.22504773597143e-05, 0.000110042441988373}, {2.70021152279181e-07, 1.74166631611966e-07, 2.69076779272191e-07}},
// {"NEX-5R iso 6400",       "SONY",      "NEX-5R",              6400,         {0.000361863374356428, 0.000127051699469749, 0.000223622406130105}, {9.06593644722766e-07, 5.65572323378951e-07, 8.94515586031125e-07}},
// {"NEX-5R iso 12800",       "SONY",      "NEX-5R",              12800,         {0.000723330352260339, 0.000268209165011333, 0.000470265140612054}, {2.72033141757973e-06, 1.47466282842992e-06, 2.50550953251447e-06}},
// {"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00150624165236013, 0.000535366854555623, 0.000953191044433931}, {6.31261159574463e-06, 4.60337377798384e-06, 6.7524068883851e-06}},
// additional more detailed data by Sander de Kievit
{"NEX-5R iso 100",       "SONY",      "NEX-5R",              100,         {6.02485919232659e-06, 1.99515561940902e-06, 3.81697488600802e-06}, {1.48986734057879e-08, 2.93821845994222e-09, 6.68779936020226e-09}},
{"NEX-5R iso 200",       "SONY",      "NEX-5R",              200,         {1.08718118153076e-05, 3.89744768951814e-06, 8.35427475667458e-06}, {1.5486855252504e-08, 1.18927287840774e-08, 1.69326889581658e-08}},
{"NEX-5R iso 200",       "SONY",      "NEX-5R",              200,         {1.25205792855948e-05, 4.25622766859516e-06, 8.41719823145527e-06}, {-1.42988756388898e-08, 2.27201877069381e-09, 1.34131005680558e-08}},
{"NEX-5R iso 200",       "SONY",      "NEX-5R",              200,         {1.05446036712131e-05, 3.85811107718936e-06, 8.43529294783715e-06}, {1.24976121669483e-08, 1.04646653521109e-08, 1.33992474206116e-08}},
{"NEX-5R iso 200",       "SONY",      "NEX-5R",              200,         {1.04114976965283e-05, 3.84044905798123e-06, 8.44888654140757e-06}, {1.47937087884816e-08, 1.20676795672371e-08, 1.58434761912745e-08}},
{"NEX-5R iso 400",       "SONY",      "NEX-5R",              400,         {2.01168729503325e-05, 7.57331070627187e-06, 1.63918619364794e-05}, {6.89348059790613e-08, 3.16250520681659e-08, 6.06961020696675e-08}},
{"NEX-5R iso 800",       "SONY",      "NEX-5R",              800,         {3.9792621992322e-05, 1.48481099641258e-05, 3.10436971867835e-05}, {1.53304873530239e-07, 8.76795733383133e-08, 1.53231346763569e-07}},
{"NEX-5R iso 800",       "SONY",      "NEX-5R",              800,         {3.97623968381111e-05, 1.46273595629822e-05, 3.04360218949995e-05}, {1.73640127484285e-07, 8.7678284758333e-08, 1.22614035673929e-07}},
{"NEX-5R iso 800",       "SONY",      "NEX-5R",              800,         {4.06385944542438e-05, 1.4848779781432e-05, 3.10759672299108e-05}, {1.15402867932768e-07, 7.11000466424874e-08, 1.06435117027562e-07}},
{"NEX-5R iso 1600",       "SONY",      "NEX-5R",              1600,         {8.42380495226913e-05, 2.91513770236161e-05, 5.41939222736747e-05}, {3.48294800737678e-07, 1.83834233841216e-07, 2.7840883757442e-07}},
{"NEX-5R iso 1600",       "SONY",      "NEX-5R",              1600,         {8.46785470419774e-05, 2.91975651251207e-05, 5.43273340184832e-05}, {3.3325404632954e-07, 1.78995042384467e-07, 2.71228320845313e-07}},
{"NEX-5R iso 1600",       "SONY",      "NEX-5R",              1600,         {8.27145879160434e-05, 2.8616478825917e-05, 5.32227004608661e-05}, {3.77362046217619e-07, 2.09053610343093e-07, 3.04212863743953e-07}},
{"NEX-5R iso 3200",       "SONY",      "NEX-5R",              3200,         {0.000100379166697429, 6.13129062366958e-05, 0.000113038314522595}, {2.90220003828133e-06, 7.67099975354998e-07, 1.22766416784323e-06}},
{"NEX-5R iso 3200",       "SONY",      "NEX-5R",              3200,         {9.85156062831792e-05, 6.12648663237129e-05, 0.000111801758588506}, {2.53311876467194e-06, 7.47326374428894e-07, 1.233502807548e-06}},
{"NEX-5R iso 6400",       "SONY",      "NEX-5R",              6400,         {0.000304117795874784, 0.000120247187589003, 0.000238014721246512}, {3.41296224987465e-06, 1.81788657123299e-06, 2.75898911065903e-06}},
{"NEX-5R iso 6400",       "SONY",      "NEX-5R",              6400,         {0.000272392733093938, 0.000111742455948241, 0.000187519706021076}, {8.20438936687009e-06, 3.2007484154086e-06, 8.52772215396266e-06}},
{"NEX-5R iso 6400",       "SONY",      "NEX-5R",              6400,         {0.000277973330874745, 0.000115308797525497, 0.000138034362814444}, {8.74059766201192e-06, 2.58038155405014e-06, 1.23682419201392e-05}},
{"NEX-5R iso 12800",       "SONY",      "NEX-5R",              12800,         {0.000575712217974728, 0.000231329330569908, 0.000445802478628886}, {1.2455281258724e-05, 7.66337979111034e-06, 1.1370023986731e-05}},
{"NEX-5R iso 12800",       "SONY",      "NEX-5R",              12800,         {0.000744454130512046, 0.00023656900413947, 0.000340648399705639}, {1.29121381576951e-05, 6.9176421518136e-06, 1.05055711987106e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00117038515484425, 0.00047760421895329, 0.000926059719309672}, {4.67068705419709e-05, 2.81341288817852e-05, 4.1515768296262e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00113930439341478, 0.000476472984289941, 0.000684753198380883}, {7.51114140816841e-05, 2.72166743333767e-05, 4.7272490850106e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00119442821936961, 0.000495268914877653, 0.000712571119816733}, {6.49811554970102e-05, 2.11281728903373e-05, 3.79824115120472e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00076949495359946, 0.000473955429713638, 0.000721589550607485}, {4.51417448887637e-05, 2.0791736061219e-05, 2.90717664169715e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00105166493483302, 0.000486518496348877, 0.000652588527707736}, {2.98577367300248e-05, 1.88202790576441e-05, 2.75300443995204e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00115563609720706, 0.000487714748542938, 0.000744663042999171}, {2.98452364132238e-05, 1.80257611170821e-05, 2.45050426824148e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.000993145139923179, 0.000477561137159706, 0.000795064375975271}, {4.22921235770175e-05, 2.11724791468673e-05, 2.69181749665351e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00114641742731989, 0.000453946631700971, 0.000848233157587786}, {3.54937020433472e-05, 2.2589165967833e-05, 2.5757412725888e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00131884672589541, 0.000486063130912338, 0.000888444314101783}, {4.69148275219977e-05, 2.4095719351449e-05, 2.88430495416493e-05}},
{"NEX-5R iso 25600",       "SONY",      "NEX-5R",              25600,         {0.00115877074425174, 0.000501196911878513, 0.00080127575702011}, {3.28742448285116e-05, 1.91805934747242e-05, 2.51579433492529e-05}},

// sony nex 6, contributed by christophe vescovi (same sensor: nex-5r nex-f3)
  {"Sony NEX-6 iso 100",   "SONY", "NEX-6", 100,   {2.83070523378633e-06, 2.83070523378633e-06, 2.83070523378633e-06}, {-8.99579441370004e-08, -8.99579441370004e-08, -8.99579441370004e-08}},
  {"Sony NEX-6 iso 200",   "SONY", "NEX-6", 200,   {4.41699217331605e-06, 4.41699217331605e-06, 4.41699217331605e-06}, {-5.73326494093182e-08, -5.73326494093182e-08, -5.73326494093182e-08}},
  {"Sony NEX-6 iso 400",   "SONY", "NEX-6", 400,   {7.95016906656091e-06, 7.95016906656091e-06, 7.95016906656091e-06}, {3.77445636379495e-09, 3.77445636379495e-09, 3.77445636379495e-09}},
  {"Sony NEX-6 iso 800",   "SONY", "NEX-6", 800,   {1.55977208319835e-05, 1.55977208319835e-05, 1.55977208319835e-05}, {-6.32291419219102e-08, -6.32291419219102e-08, -6.32291419219102e-08}},
  {"Sony NEX-6 iso 1600",  "SONY", "NEX-6", 1600,  {2.98162473611846e-05, 2.98162473611846e-05, 2.98162473611846e-05}, {4.43033418340148e-08, 4.43033418340148e-08, 4.43033418340148e-08}},
  {"Sony NEX-6 iso 3200",  "SONY", "NEX-6", 3200,  {5.84783565575537e-05, 5.84783565575537e-05, 5.84783565575537e-05}, {2.75198472414243e-07, 2.75198472414243e-07, 2.75198472414243e-07}},
  {"Sony NEX-6 iso 6400",  "SONY", "NEX-6", 6400,  {0.000117797177674452, 0.000117797177674452, 0.000117797177674452}, {9.27480448891856e-07, 9.27480448891856e-07, 9.27480448891856e-07}},
  {"Sony NEX-6 iso 12800", "SONY", "NEX-6", 12800, {0.000234877764709295, 0.000234877764709295, 0.000234877764709295}, {4.23945931130405e-06, 4.23945931130405e-06, 4.23945931130405e-06}},
  {"Sony NEX-6 iso 25600", "SONY", "NEX-6", 25600, {0.000427503080952746, 0.000427503080952746, 0.000427503080952746}, {2.89795299259573e-05, 2.89795299259573e-05, 2.89795299259573e-05}},

// sony nex-7, contributed by thorsten bronger
  {"Sony NEX-7 iso 100",   "SONY", "NEX-7",    100,   {9.52572013800499e-06, 3.86531317829194e-06, 7.08489524460427e-06}, {-5.79938071697469e-08, -1.13275594438032e-07, -1.47705731681281e-07}},
// {"Sony NEX-7 iso 100",   "SONY", "NEX-7",    100,   {1.0149421166248e-05, 3.91912673392486e-06, 6.49296757568039e-06}, {-8.62037040997744e-08, -1.52532679712677e-07, -1.07006470079441e-07}},
// {"Sony NEX-7 iso 100",   "SONY", "NEX-7",    100,   {1.07546793335227e-05, 3.84627597103129e-06, 6.31300284998407e-06}, {-1.59297515448314e-07, -1.36580456336747e-07, -7.38034939534284e-08}},
  {"Sony NEX-7 iso 200",   "SONY", "NEX-7",    200,   {1.77917567040153e-05, 7.07382496836488e-06, 1.27472480403923e-05}, {-7.77866590516443e-08, -1.7846323945424e-07, -2.05853963740862e-07}},
// {"Sony NEX-7 iso 200",   "SONY", "NEX-7",    200,   {1.76842119105416e-05, 6.42211581983839e-06, 1.15661528536084e-05}, {-1.10716423973862e-07, -3.52916576726673e-08, -3.15504253230607e-08}},
  {"Sony NEX-7 iso 400",   "SONY", "NEX-7",    400,   {3.19439760027425e-05, 1.25405055000036e-05, 2.26691930368754e-05}, {4.39444447968611e-08, -1.09265375370735e-07, -6.90713519345709e-08}},
// {"Sony NEX-7 iso 400",   "SONY", "NEX-7",    400,   {3.54330283303158e-05, 1.25643123762239e-05, 2.20372770293359e-05}, {-3.57812957811644e-07, -1.50068772253003e-07, -8.05398930960629e-08}},
  {"Sony NEX-7 iso 800",   "SONY", "NEX-7",    800,   {5.96599779821847e-05, 2.35985630409358e-05, 4.37998472077054e-05}, {1.45275768205654e-07, -1.47167632574697e-08, -6.0915583384487e-08}},
  {"Sony NEX-7 iso 1600",  "SONY", "NEX-7",    1600,  {0.000119860114446891, 4.60888654190478e-05, 8.36518903394813e-05}, {3.61396264240833e-07, 3.33116130999835e-08, 1.43509885327547e-07}},
// {"Sony NEX-7 iso 1600",  "SONY", "NEX-7",    1600,  {0.000115746879917472, 4.58680168931902e-05, 8.41519121473174e-05}, {5.79003788021017e-07, 1.60415648984468e-07, 6.50418638322168e-08}},
  {"Sony NEX-7 iso 3200",  "SONY", "NEX-7",    3200,  {0.000230428880907035, 8.87943130832338e-05, 0.000165863911502419}, {2.67439166201749e-06, 1.74859821955235e-06, -1.63648955084254e-08}},
// {"Sony NEX-7 iso 3200",  "SONY", "NEX-7",    3200,  {0.000228120765179111, 8.98721845504593e-05, 0.00016712686162849}, {1.95703819742258e-06, 1.15367417000547e-06, 6.83023308107306e-07}},
  {"Sony NEX-7 iso 6400",  "SONY", "NEX-7",    6400,  {0.000278667743106855, 0.000119427613876046, 0.000194826828332879}, {1.18039444751624e-05, 2.26985603622252e-06, 2.31361370485939e-06}},
// {"Sony NEX-7 iso 6400",  "SONY", "NEX-7",    6400,  {0.000348254977662911, 0.000140526931240904, 0.000232965715373809}, {-9.4686126082727e-07, -2.22799463405143e-06, 3.00129567143297e-07}},
  {"Sony NEX-7 iso 12800", "SONY", "NEX-7",    12800, {0.000639772901145468, 0.000243336342788034, 0.000421216394792856}, {6.68185306368635e-06, 7.55338728412877e-06, 1.25896465865543e-05}},
// {"Sony NEX-7 iso 12800", "SONY", "NEX-7",    12800, {0.000612280404483272, 0.000234704423227497, 0.000430372703331731}, {1.00601312342883e-05, 1.18031954443056e-05, 1.19733961935564e-05}},
// {"Sony NEX-7 iso 12800", "SONY", "NEX-7",    12800, {0.000647232862925275, 0.000255657920251736, 0.000441937159437112}, {5.71082006354221e-06, 5.86232543284394e-06, 1.07573625976752e-05}},
  {"Sony NEX-7 iso 16000", "SONY", "NEX-7",    16000, {0.000934571878504653, 0.000375159280407057, 0.000610563030948953}, {-1.59189786098775e-06, -1.80697489164933e-06, 8.84267528925131e-06}},
// {"Sony NEX-7 iso 16000", "SONY", "NEX-7",    16000, {0.000854062339078667, 0.000311263971434844, 0.000578029046786113}, {8.11611241594421e-06, 1.471946768089e-05, 1.29945921312847e-05}},

// sony a65 contributed by maigl
  {"Sony SLT-A65V iso 100",   "SONY", "SLT-A65V", 100,   {1.10038047780812e-05, 7.35712752324848e-06, 1.53293269200751e-05}, {1.34263343081115e-07, -4.14986333801341e-07, -3.29364975052324e-07}},
  {"Sony SLT-A65V iso 200",   "SONY", "SLT-A65V", 200,   {2.11442238333175e-05, 1.21611818640914e-05, 2.26452590606463e-05}, {2.57824470672561e-07, -4.94646528878113e-07, 1.81784943447172e-07}},
  {"Sony SLT-A65V iso 400",   "SONY", "SLT-A65V", 400,   {4.23238159168085e-05, 2.05389112044514e-05, 3.51205328425759e-05}, {2.32963365495563e-07, -4.02684078903633e-07, 1.35680255756746e-06}},
  {"Sony SLT-A65V iso 800",   "SONY", "SLT-A65V", 800,   {6.59313895571544e-05, 4.2281151168149e-05, 7.77802295314829e-05}, {1.30637747843506e-06, -1.70559096662665e-06, -7.53929987496757e-08}},
  {"Sony SLT-A65V iso 1600",  "SONY", "SLT-A65V", 1600,  {0.000135419572151958, 7.51611658497752e-05, 0.000128755825848952}, {1.67169166585383e-06, -1.00769094236019e-06, 1.29081161723159e-06}},
  {"Sony SLT-A65V iso 3200",  "SONY", "SLT-A65V", 3200,  {0.000267763192874051, 0.000140152972848944, 0.000237405859164511}, {5.05177030363853e-06, 4.25182701367377e-07, 5.83377085787843e-06}},
  {"Sony SLT-A65V iso 6400",  "SONY", "SLT-A65V", 6400,  {0.000345679018476196, 0.000179514241609738, 0.000309749042075687}, {8.92982895533874e-06, 4.01688240107948e-06, 1.0375397931409e-05}},
  {"Sony SLT-A65V iso 12800", "SONY", "SLT-A65V", 12800, {0.000680018758313125, 0.000362367634655054, 0.000646883923648539}, {2.56719064084042e-05, 1.47430518221999e-05, 2.54526596905135e-05}},
  {"Sony SLT-A65V iso 16000", "SONY", "SLT-A65V", 16000, {0.000825945891669277, 0.00045034729324987, 0.000802228280759472}, {4.0141545995607e-05, 2.34540854176319e-05, 4.08241723929137e-05}},

// sony rx100, contributed by josef wells, blue underexposed
  {"Sony DSC-RX100 iso 80",   "SONY", "DSC-RX100", 80,   {2.9306143923251e-05,  1.29004570140481e-05, 0.0f}, {-2.08931092235626e-07, -8.42712205571748e-08, 0.0f}},
  {"Sony DSC-RX100 iso 100",  "SONY", "DSC-RX100", 100,  {3.84082681437922e-05, 1.64740520421434e-05, 0.0f}, {-3.46694378000144e-07, -1.22384816857406e-07, 0.0f}},
  {"Sony DSC-RX100 iso 125",  "SONY", "DSC-RX100", 125,  {3.56945823812817e-05, 1.57458803321307e-05, 0.0f}, {-5.41045077108107e-07, -5.47870439906304e-07, 0.0f}},
  {"Sony DSC-RX100 iso 200",  "SONY", "DSC-RX100", 200,  {6.43253286884155e-05, 2.69593062015314e-05, 0.0f}, {-1.41916025632816e-06, -9.74729455716012e-07, 0.0f}},
  {"Sony DSC-RX100 iso 400",  "SONY", "DSC-RX100", 400,  {0.000119731304071307, 5.46009794892234e-05, 0.0f}, {-1.87693216531519e-06, -1.63381772514978e-06, 0.0f}},
  {"Sony DSC-RX100 iso 800",  "SONY", "DSC-RX100", 800,  {0.000209806271340403, 0.000100161009389223, 0.0f}, {-2.8322006166524e-06, -2.75378413743182e-06,  0.0f}},
  {"Sony DSC-RX100 iso 1600", "SONY", "DSC-RX100", 1600, {0.000348109835480264, 0.000160433656655396, 0.0f}, {-2.67415388829755e-06, -1.4922244339415e-06,  0.0f}},
  {"Sony DSC-RX100 iso 3200", "SONY", "DSC-RX100", 3200, {0.000567545502853282, 0.00026163539211963,  0.0f}, {7.79325667096793e-08, 2.49369181196474e-07,   0.0f}},
  {"Sony DSC-RX100 iso 6400", "SONY", "DSC-RX100", 6400, {0.000712223949201973, 0.00031169464275467,  0.0f}, {-1.81030385563045e-06, 2.8494616823593e-06,   0.0f}},

// sony dxc rx100m2 by Pawel Zogal
// {"DSC-RX100M2 iso 100",       "SONY",      "DSC-RX100M2",              100,         {1.51005852620076e-05, 8.45367987189378e-06, 1.93403939438192e-05}, {2.86546798561033e-08, 6.20391954422978e-09, 1.03240767431775e-07}},
// {"DSC-RX100M2 iso 125",       "SONY",      "DSC-RX100M2",              125,         {1.81640073201845e-05, 1.02831319455153e-05, 2.60796080433429e-05}, {3.99779861583934e-08, 1.72300179334189e-08, 1.00355458680747e-07}},
// {"DSC-RX100M2 iso 160",       "SONY",      "DSC-RX100M2",              160,         {1.44836429344127e-05, 8.11610248272229e-06, 2.33408330995237e-05}, {3.16452092793383e-08, 1.75606157983148e-08, 5.46807892801445e-08}},
// {"DSC-RX100M2 iso 200",       "SONY",      "DSC-RX100M2",              200,         {1.80872807363147e-05, 1.01142750698661e-05, 2.73958625757565e-05}, {4.10236435944178e-08, 2.30513081457102e-08, 8.13689391188151e-08}},
// {"DSC-RX100M2 iso 400",       "SONY",      "DSC-RX100M2",              400,         {3.45914683839806e-05, 1.91664411790998e-05, 5.23992058356036e-05}, {8.58967562499212e-08, 5.6663792374734e-08, 1.62174111925665e-07}},
// {"DSC-RX100M2 iso 800",       "SONY",      "DSC-RX100M2",              800,         {6.78489044619687e-05, 3.75481822964575e-05, 0.00010298360828447}, {1.85274510876454e-07, 1.38762022948901e-07, 3.28311622445962e-07}},
// {"DSC-RX100M2 iso 1600",       "SONY",      "DSC-RX100M2",              1600,         {0.000136281339608826, 7.43983278494936e-05, 0.000207536318607906}, {4.20154902064743e-07, 3.15264727038035e-07, 6.78936458519558e-07}},
// {"DSC-RX100M2 iso 3200",       "SONY",      "DSC-RX100M2",              3200,         {0.00027013229124777, 0.000147164083812162, 0.000415544910296959}, {9.26084473744043e-07, 7.49398828423131e-07, 1.42303882805464e-06}},
// {"DSC-RX100M2 iso 6400",       "SONY",      "DSC-RX100M2",              6400,         {0.000358557156729082, 0.00019799491833517, 0.000529160695951543}, {1.44853445739595e-06, 1.19390604223894e-06, 2.11247358791108e-06}},
// {"DSC-RX100M2 iso 12800",       "SONY",      "DSC-RX100M2",              12800,         {0.000744190949578655, 0.000415675293717899, 0.00107203922638601}, {3.91259276874732e-06, 3.51351503022162e-06, 5.21108809296857e-06}},
// additional data, more detailed by Matthew Harrison
{"DSC-RX100M2 iso 100",       "SONY",      "DSC-RX100M2",              100,         {1.85773203178648e-05, 9.20214194146827e-06, 3.00044293337357e-05}, {3.87192985022596e-08, 2.14928103987894e-08, 6.06596137919109e-08}},
{"DSC-RX100M2 iso 100",       "SONY",      "DSC-RX100M2",              100,         {1.93035394961304e-05, 9.25492021945514e-06, 2.91865315625456e-05}, {3.87689682497588e-08, 2.24615704883103e-08, 6.19034009520404e-08}},
{"DSC-RX100M2 iso 100",       "SONY",      "DSC-RX100M2",              100,         {1.49636760016543e-05, 9.36567110754016e-06, 4.38462243784758e-05}, {3.25015498758195e-08, 2.11705483941356e-08, 5.27420897555967e-08}},
{"DSC-RX100M2 iso 100",       "SONY",      "DSC-RX100M2",              100,         {1.50115629334246e-05, 9.2944261441928e-06, 4.56190795909072e-05}, {3.23378352412732e-08, 2.11946168699988e-08, 5.09497470651087e-08}},
{"DSC-RX100M2 iso 125",       "SONY",      "DSC-RX100M2",              125,         {2.4004803934483e-05, 1.13790364361947e-05, 3.50503789372008e-05}, {5.42517607964646e-08, 3.22694986886407e-08, 8.15345591548649e-08}},
{"DSC-RX100M2 iso 125",       "SONY",      "DSC-RX100M2",              125,         {2.43785296744819e-05, 1.14816235992525e-05, 3.73092052818991e-05}, {5.25716560008825e-08, 3.27651298359803e-08, 7.24142047490761e-08}},
{"DSC-RX100M2 iso 125",       "SONY",      "DSC-RX100M2",              125,         {1.89921026015214e-05, 1.17689507088729e-05, 6.00905816928449e-05}, {3.86751058143333e-08, 2.88594490874073e-08, 6.03834143321437e-08}},
{"DSC-RX100M2 iso 125",       "SONY",      "DSC-RX100M2",              125,         {1.85671955199106e-05, 1.14867346832906e-05, 5.48509747984742e-05}, {4.08644057641105e-08, 2.96910155980268e-08, 6.78765605125857e-08}},
{"DSC-RX100M2 iso 160",       "SONY",      "DSC-RX100M2",              160,         {1.49659409880319e-05, 9.39094639288209e-06, 4.57809985573049e-05}, {3.18982357045068e-08, 2.08787776973788e-08, 4.71315766478956e-08}},
{"DSC-RX100M2 iso 160",       "SONY",      "DSC-RX100M2",              160,         {1.96334072832439e-05, 9.27828929798193e-06, 2.97897577820238e-05}, {3.86487020755246e-08, 2.28598805134114e-08, 5.58448988879007e-08}},
{"DSC-RX100M2 iso 160",       "SONY",      "DSC-RX100M2",              160,         {1.96665414702849e-05, 9.37968810957902e-06, 3.07352527556406e-05}, {3.87938050228674e-08, 2.37501595616258e-08, 5.17812523933118e-08}},
{"DSC-RX100M2 iso 160",       "SONY",      "DSC-RX100M2",              160,         {1.53916440548314e-05, 9.51081070641674e-06, 5.02211937502741e-05}, {3.03719312750893e-08, 2.03036524565827e-08, 4.17225665024146e-08}},
{"DSC-RX100M2 iso 200",       "SONY",      "DSC-RX100M2",              200,         {1.6066869108553e-05, 9.93595570876719e-06, 4.84309347798526e-05}, {3.32746087162105e-08, 2.21225892450143e-08, 4.84363045526134e-08}},
{"DSC-RX100M2 iso 200",       "SONY",      "DSC-RX100M2",              200,         {2.0661554920483e-05, 9.80020715232414e-06, 3.11615936592612e-05}, {4.11265645856808e-08, 2.48426777537071e-08, 6.03502092444664e-08}},
{"DSC-RX100M2 iso 200",       "SONY",      "DSC-RX100M2",              200,         {2.10856772682343e-05, 9.88713229398563e-06, 3.24070624841743e-05}, {4.23847684683281e-08, 2.66710474564133e-08, 5.47959849432627e-08}},
{"DSC-RX100M2 iso 200",       "SONY",      "DSC-RX100M2",              200,         {1.60696104570474e-05, 1.01135339129237e-05, 5.01882121914778e-05}, {3.25274558530725e-08, 2.21522799517325e-08, 4.66880838310766e-08}},
{"DSC-RX100M2 iso 400",       "SONY",      "DSC-RX100M2",              400,         {3.14048834638661e-05, 1.95934600411529e-05, 8.89122556099453e-05}, {8.42919910586204e-08, 6.08162894462173e-08, 1.29233538252884e-07}},
{"DSC-RX100M2 iso 400",       "SONY",      "DSC-RX100M2",              400,         {4.05338241187418e-05, 1.92374306541821e-05, 6.10349846214919e-05}, {1.0299088849195e-07, 7.12663918797589e-08, 1.4065395704694e-07}},
{"DSC-RX100M2 iso 400",       "SONY",      "DSC-RX100M2",              400,         {3.19308151447444e-05, 1.99431426303462e-05, 9.87528443752187e-05}, {7.53665857081337e-08, 5.38467460756251e-08, 1.11958359196862e-07}},
{"DSC-RX100M2 iso 800",       "SONY",      "DSC-RX100M2",              800,         {6.47666562620208e-05, 4.0449355740272e-05, 0.000192343360979797}, {1.50278554820093e-07, 1.10780145439202e-07, 2.24870376017636e-07}},
{"DSC-RX100M2 iso 800",       "SONY",      "DSC-RX100M2",              800,         {8.01124092934463e-05, 3.79234167852619e-05, 0.000116871860539502}, {2.38388591295865e-07, 1.67433185030773e-07, 3.00219857029347e-07}},
{"DSC-RX100M2 iso 800",       "SONY",      "DSC-RX100M2",              800,         {8.02426563510224e-05, 3.75852857372669e-05, 0.000112744788466049}, {2.56068926428967e-07, 1.77427900552478e-07, 3.29485873064672e-07}},
{"DSC-RX100M2 iso 800",       "SONY",      "DSC-RX100M2",              800,         {6.33769349372639e-05, 3.9246591782193e-05, 0.000181878932788834}, {1.70736659608573e-07, 1.27884489677659e-07, 2.54165466460459e-07}},
{"DSC-RX100M2 iso 1600",       "SONY",      "DSC-RX100M2",              1600,         {0.000155752685065629, 7.41962234102097e-05, 0.000211343026430522}, {6.60852484098367e-07, 4.80241837408903e-07, 8.65859392930397e-07}},
{"DSC-RX100M2 iso 1600",       "SONY",      "DSC-RX100M2",              1600,         {0.000127639056506783, 7.92381354173455e-05, 0.000336152407928541}, {4.13087674859208e-07, 3.01701636222001e-07, 5.84388104677915e-07}},
{"DSC-RX100M2 iso 1600",       "SONY",      "DSC-RX100M2",              1600,         {0.000150746012445614, 7.61774806472063e-05, 0.000239918403078749}, {5.60193486432469e-07, 3.97401795746089e-07, 7.60426499356771e-07}},
{"DSC-RX100M2 iso 1600",       "SONY",      "DSC-RX100M2",              1600,         {0.000130643493640201, 7.91587336469587e-05, 0.000332291591274722}, {4.13003720019168e-07, 3.06423161659995e-07, 5.96352390582416e-07}},
{"DSC-RX100M2 iso 3200",       "SONY",      "DSC-RX100M2",              3200,         {0.000248371717595201, 0.000156607870798168, 0.000670981740794729}, {6.88754534889814e-07, 4.90637602387912e-07, 8.84794034331105e-07}},
{"DSC-RX100M2 iso 3200",       "SONY",      "DSC-RX100M2",              3200,         {0.000244476276739692, 0.000156018119137236, 0.000734227451216444}, {7.7919708578884e-07, 5.67692896768729e-07, 1.02196015676879e-06}},
{"DSC-RX100M2 iso 3200",       "SONY",      "DSC-RX100M2",              3200,         {0.00025793743201915, 0.000157276739368141, 0.000700490504394726}, {8.8323119923606e-07, 6.46597185172727e-07, 1.07992388747959e-06}},
{"DSC-RX100M2 iso 3200",       "SONY",      "DSC-RX100M2",              3200,         {0.000327562104265559, 0.000151707311608791, 0.000494164148301862}, {7.49186282414761e-07, 7.03861692387647e-07, 1.20531303111371e-06}},
{"DSC-RX100M2 iso 6400",       "SONY",      "DSC-RX100M2",              6400,         {0.000282243583986077, 0.000177154909988169, 0.000785909575575165}, {9.98490044967435e-07, 7.83432003538856e-07, 1.26891231707655e-06}},
{"DSC-RX100M2 iso 6400",       "SONY",      "DSC-RX100M2",              6400,         {0.000287296207112913, 0.000174970461200106, 0.000629367771087476}, {9.94025542929451e-07, 7.7359371017456e-07, 1.52763162379276e-06}},
{"DSC-RX100M2 iso 6400",       "SONY",      "DSC-RX100M2",              6400,         {0.000279455276092958, 0.000171400957256357, 0.000673267909083585}, {1.17102743390169e-06, 9.21994737651285e-07, 1.73464526888e-06}},
{"DSC-RX100M2 iso 6400",       "SONY",      "DSC-RX100M2",              6400,         {0.000278892342644737, 0.000170330296134838, 0.000712125494613637}, {1.32566613391264e-06, 1.04579143883652e-06, 1.83711895868086e-06}},
{"DSC-RX100M2 iso 12800",       "SONY",      "DSC-RX100M2",              12800,         {0.000555491230654247, 0.000340543556893093, 0.00110123943761198}, {4.6072788898486e-06, 3.99185764440027e-06, 8.02918354957642e-06}},
{"DSC-RX100M2 iso 12800",       "SONY",      "DSC-RX100M2",              12800,         {0.000554596373980117, 0.000341087695281971, 0.00129312401050058}, {3.89189721286739e-06, 3.49820815732941e-06, 5.70035735761427e-06}},
{"DSC-RX100M2 iso 12800",       "SONY",      "DSC-RX100M2",              12800,         {0.000564186623030703, 0.000346213617503058, 0.00135898182391578}, {3.41161943608584e-06, 3.06375387341698e-06, 4.41807534426888e-06}},
{"DSC-RX100M2 iso 12800",       "SONY",      "DSC-RX100M2",              12800,         {0.000574516129026528, 0.000359986006961321, 0.00144710372955629}, {2.80470402816349e-06, 2.45401965606868e-06, 3.07261931811361e-06}},

// sony ilce-7 by Per Östlund
{"ILCE-7 iso 50",       "SONY",      "ILCE-7",              50,         {4.63991628581115e-06, 1.60744267583856e-06, 3.39566030970468e-06}, {5.18656156070525e-09, 2.38681585462356e-09, 4.067165322867e-09}},
{"ILCE-7 iso 64",       "SONY",      "ILCE-7",              64,         {5.68195014539981e-06, 1.99487568120822e-06, 4.17840136471588e-06}, {6.72611746564801e-09, 3.10103175421657e-09, 5.40997919026921e-09}},
{"ILCE-7 iso 80",       "SONY",      "ILCE-7",              80,         {6.90945642756909e-06, 2.47240013379263e-06, 5.0461451271741e-06}, {8.48453572093157e-09, 4.11249335975865e-09, 7.23476878346299e-09}},
{"ILCE-7 iso 100",       "SONY",      "ILCE-7",              100,         {4.63725253257195e-06, 1.62309048579019e-06, 3.43907810380205e-06}, {2.85957261339619e-09, 2.39099940768725e-09, 3.44210388346535e-09}},
{"ILCE-7 iso 125",       "SONY",      "ILCE-7",              125,         {5.70453942043346e-06, 2.03407212617358e-06, 4.23595173970405e-06}, {3.78466452601855e-09, 2.99154544969413e-09, 4.60505183260023e-09}},
{"ILCE-7 iso 160",       "SONY",      "ILCE-7",              160,         {7.01039994321805e-06, 2.52584294289655e-06, 5.16196980212733e-06}, {5.00201234499623e-09, 4.03701707857158e-09, 6.13350149528209e-09}},
{"ILCE-7 iso 200",       "SONY",      "ILCE-7",              200,         {8.57368179105694e-06, 3.08662038998965e-06, 6.34521104137121e-06}, {6.54671324772749e-09, 5.43942616248109e-09, 7.86263027476475e-09}},
{"ILCE-7 iso 250",       "SONY",      "ILCE-7",              250,         {1.04312927358978e-05, 3.69748176108507e-06, 7.7484944557276e-06}, {8.49238010275873e-09, 7.10961249937888e-09, 1.00087664383718e-08}},
{"ILCE-7 iso 320",       "SONY",      "ILCE-7",              320,         {1.28910676167577e-05, 4.53689091268534e-06, 9.55931948315597e-06}, {1.09952347841243e-08, 8.87836881056096e-09, 1.30247324266455e-08}},
{"ILCE-7 iso 400",       "SONY",      "ILCE-7",              400,         {1.59848959345648e-05, 5.70017033039826e-06, 1.18246009633647e-05}, {1.46643554015071e-08, 1.16547137613918e-08, 1.72822583001932e-08}},
{"ILCE-7 iso 500",       "SONY",      "ILCE-7",              500,         {1.97672719628199e-05, 7.04613272184653e-06, 1.46328420464336e-05}, {1.91306562556368e-08, 1.5920166416504e-08, 2.24838059902434e-08}},
{"ILCE-7 iso 640",       "SONY",      "ILCE-7",              640,         {2.44018029663742e-05, 8.66141445518577e-06, 1.8151746104538e-05}, {2.6011271755615e-08, 2.08394403568906e-08, 2.99387302486595e-08}},
{"ILCE-7 iso 800",       "SONY",      "ILCE-7",              800,         {2.97493428621698e-05, 1.05248107384537e-05, 2.20337472124025e-05}, {3.39193747620006e-08, 2.73632250942671e-08, 3.86590375457153e-08}},
{"ILCE-7 iso 1000",       "SONY",      "ILCE-7",              1000,         {3.69534263962844e-05, 1.31566750301468e-05, 2.74833708017638e-05}, {4.55394749154068e-08, 3.62352396453189e-08, 5.18238822471394e-08}},
{"ILCE-7 iso 1250",       "SONY",      "ILCE-7",              1250,         {4.62443207930939e-05, 1.65433102521861e-05, 3.43443085418911e-05}, {6.11294496864831e-08, 4.98271538823171e-08, 6.94499917061774e-08}},
{"ILCE-7 iso 1600",       "SONY",      "ILCE-7",              1600,         {5.79453253999304e-05, 2.05734306263726e-05, 4.27662577605388e-05}, {8.31029458030184e-08, 6.74368087970348e-08, 9.49475097722939e-08}},
{"ILCE-7 iso 2000",       "SONY",      "ILCE-7",              2000,         {7.25255941275435e-05, 2.58235106814721e-05, 5.37818029234196e-05}, {1.13858566472873e-07, 9.04142038993618e-08, 1.27884830909206e-07}},
{"ILCE-7 iso 2500",       "SONY",      "ILCE-7",              2500,         {9.05294818894737e-05, 3.22933890440199e-05, 6.71615153352591e-05}, {1.57238772449711e-07, 1.21456663805758e-07, 1.7356560470236e-07}},
{"ILCE-7 iso 3200",       "SONY",      "ILCE-7",              3200,         {0.000112810861612709, 4.02036684944057e-05, 8.34244234477027e-05}, {2.14440714967228e-07, 1.6408536689858e-07, 2.33204192333045e-07}},
{"ILCE-7 iso 4000",       "SONY",      "ILCE-7",              4000,         {0.000142121802104032, 5.07440054235767e-05, 0.000105479289223911}, {2.90646449690934e-07, 2.19494939606944e-07, 3.15238270465113e-07}},
{"ILCE-7 iso 5000",       "SONY",      "ILCE-7",              5000,         {0.000178923495420793, 6.44726487699475e-05, 0.000132967975571087}, {3.79958305056563e-07, 2.81057384942221e-07, 4.09098020015022e-07}},
{"ILCE-7 iso 6400",       "SONY",      "ILCE-7",              6400,         {0.000223136479386788, 8.1001340418702e-05, 0.000166706912229932}, {5.06736639960396e-07, 3.7101093472181e-07, 5.46595860767199e-07}},
{"ILCE-7 iso 8000",       "SONY",      "ILCE-7",              8000,         {0.000283537692072077, 0.000102524397935184, 0.000211275742034512}, {6.82978576582224e-07, 5.0345466879561e-07, 7.43636638139591e-07}},
{"ILCE-7 iso 10000",       "SONY",      "ILCE-7",              10000,         {0.000357554178873544, 0.000129984963164276, 0.000267110536750977}, {9.28362843203383e-07, 7.03645856407898e-07, 1.01970301519657e-06}},
{"ILCE-7 iso 12800",       "SONY",      "ILCE-7",              12800,         {0.000448140011859292, 0.000164095571726624, 0.000334221929462091}, {1.30269168257319e-06, 1.00853418463818e-06, 1.45480708353137e-06}},
{"ILCE-7 iso 16000",       "SONY",      "ILCE-7",              16000,         {0.000570903573469483, 0.000210512359935446, 0.000426033906744408}, {1.78935113639203e-06, 1.43176961826847e-06, 1.99815100756104e-06}},
{"ILCE-7 iso 20000",       "SONY",      "ILCE-7",              20000,         {0.000720139358021734, 0.000267399049202392, 0.000539967488403604}, {2.47704374460509e-06, 1.99247101742069e-06, 2.77620803909013e-06}},
{"ILCE-7 iso 25600",       "SONY",      "ILCE-7",              25600,         {0.000595853208876894, 0.000224169788985352, 0.000444762712391776}, {2.53443094896798e-06, 2.23195679471506e-06, 2.86461770163049e-06}},

// sony ilce-7r by carsten liersch
{"ILCE-7R iso 50",       "SONY",      "ILCE-7R",              50,         {2.92397533888255e-06, 1.82640525863349e-06, 5.76697674073052e-06}, {2.70674433517928e-09, 1.59829719556368e-09, 4.04476426546008e-09}},
{"ILCE-7R iso 64",       "SONY",      "ILCE-7R",              64,         {3.61172263356785e-06, 2.26156130247458e-06, 7.20023986230913e-06}, {3.52598715191553e-09, 2.09721618472379e-09, 4.89350064589038e-09}},
{"ILCE-7R iso 80",       "SONY",      "ILCE-7R",              80,         {4.50904034563779e-06, 2.82207366747848e-06, 8.91740564896761e-06}, {5.09901815786339e-09, 3.06705891069034e-09, 7.52651564481865e-09}},
{"ILCE-7R iso 100",       "SONY",      "ILCE-7R",              100,         {2.84290320911723e-06, 1.75453054693271e-06, 5.64310142044487e-06}, {2.65559738764787e-09, 1.87382674870993e-09, 3.59681101113079e-09}},
{"ILCE-7R iso 125",       "SONY",      "ILCE-7R",              125,         {3.51926428383213e-06, 2.17140815533469e-06, 7.01298572765787e-06}, {3.83761030480133e-09, 2.56996672540392e-09, 5.06603392030444e-09}},
{"ILCE-7R iso 160",       "SONY",      "ILCE-7R",              160,         {4.40237095170321e-06, 2.69958114195913e-06, 8.75617599607246e-06}, {4.46948157264682e-09, 3.24114080638311e-09, 5.61401392574545e-09}},
{"ILCE-7R iso 200",       "SONY",      "ILCE-7R",              200,         {5.42816820647171e-06, 3.36921666877941e-06, 1.12463995554775e-05}, {5.54631805152682e-09, 3.94510756230898e-09, 7.11518714038678e-09}},
{"ILCE-7R iso 250",       "SONY",      "ILCE-7R",              250,         {6.91214995205936e-06, 4.21875777954619e-06, 1.38340432137642e-05}, {6.07709612321797e-09, 4.26946989469806e-09, 7.52929492358314e-09}},
{"ILCE-7R iso 320",       "SONY",      "ILCE-7R",              320,         {8.61035099839008e-06, 5.29449790933485e-06, 1.72038589288558e-05}, {8.91389240315564e-09, 6.52270236130737e-09, 1.13295068004631e-08}},
{"ILCE-7R iso 400",       "SONY",      "ILCE-7R",              400,         {1.05926329905339e-05, 6.57465042549709e-06, 2.19952810380339e-05}, {1.21641010195461e-08, 9.0660241429726e-09, 1.55055473912932e-08}},
{"ILCE-7R iso 500",       "SONY",      "ILCE-7R",              500,         {1.3474023186117e-05, 8.28739397140017e-06, 2.67464996823414e-05}, {1.62235781237691e-08, 1.16402216757711e-08, 2.04915155923018e-08}},
{"ILCE-7R iso 640",       "SONY",      "ILCE-7R",              640,         {1.65648151053918e-05, 1.03495061527709e-05, 3.42658701301925e-05}, {2.29440161424792e-08, 1.69502294170086e-08, 2.90016753376772e-08}},
{"ILCE-7R iso 800",       "SONY",      "ILCE-7R",              800,         {2.05664805036986e-05, 1.27028476187087e-05, 4.08336900220599e-05}, {2.81546271402257e-08, 2.02203913755332e-08, 3.58912262476045e-08}},
{"ILCE-7R iso 1000",       "SONY",      "ILCE-7R",              1000,         {2.54578023686761e-05, 1.59494911208786e-05, 5.26575550162981e-05}, {3.81319284080336e-08, 2.77374905433762e-08, 4.78004756329119e-08}},
{"ILCE-7R iso 1250",       "SONY",      "ILCE-7R",              1250,         {3.20041498552015e-05, 2.01161612033618e-05, 6.60186160342816e-05}, {4.8665660089461e-08, 3.57693462889145e-08, 5.99523257411758e-08}},
{"ILCE-7R iso 1600",       "SONY",      "ILCE-7R",              1600,         {4.0236544528733e-05, 2.50412161953293e-05, 7.97748079072629e-05}, {7.33013121441556e-08, 5.30701586514035e-08, 8.77493619763371e-08}},
{"ILCE-7R iso 2000",       "SONY",      "ILCE-7R",              2000,         {5.12232891492823e-05, 3.22261063645589e-05, 0.000103622980688767}, {9.10424916710011e-08, 6.57734835361781e-08, 1.10890417238593e-07}},
{"ILCE-7R iso 2500",       "SONY",      "ILCE-7R",              2500,         {6.69950015523343e-05, 4.1419935020369e-05, 0.00012858032741765}, {1.22588055884508e-07, 8.67420431879925e-08, 1.45201068888771e-07}},
{"ILCE-7R iso 3200",       "SONY",      "ILCE-7R",              3200,         {8.32672487531717e-05, 5.26896849866377e-05, 0.000164436229715415}, {1.78414467481687e-07, 1.25785196468811e-07, 2.11828759708181e-07}},
{"ILCE-7R iso 4000",       "SONY",      "ILCE-7R",              4000,         {0.000107511668258488, 6.75468828342726e-05, 0.000208422321194688}, {2.45131331494242e-07, 1.73825773597458e-07, 2.88443269656923e-07}},
{"ILCE-7R iso 5000",       "SONY",      "ILCE-7R",              5000,         {0.000137509122506857, 8.77695705440026e-05, 0.000265910064724367}, {3.30900445405307e-07, 2.3295036797295e-07, 3.8612226323736e-07}},
{"ILCE-7R iso 6400",       "SONY",      "ILCE-7R",              6400,         {0.000176803364276931, 0.000115003244635173, 0.000341371149905904}, {4.58388223850909e-07, 3.20926977189709e-07, 5.28133668065798e-07}},
{"ILCE-7R iso 8000",       "SONY",      "ILCE-7R",              8000,         {0.000228532148015455, 0.00014929861046149, 0.000429036367268219}, {6.51409337375814e-07, 4.63733479908799e-07, 7.24443674262258e-07}},
{"ILCE-7R iso 10000",       "SONY",      "ILCE-7R",              10000,         {0.000291420055727011, 0.0001928554520997, 0.000536755983843512}, {9.2455181804104e-07, 6.68668503369849e-07, 9.95485850415904e-07}},
{"ILCE-7R iso 12800",       "SONY",      "ILCE-7R",              12800,         {0.000361079044799295, 0.000228884530008187, 0.000645665956127134}, {1.35260557625985e-06, 1.02094450614731e-06, 1.36485556477423e-06}},
{"ILCE-7R iso 16000",       "SONY",      "ILCE-7R",              16000,         {0.000507260300168278, 0.00036228381176268, 0.000827977288890692}, {1.54097570577424e-06, 1.06769740955828e-06, 1.55120853601078e-06}},
{"ILCE-7R iso 20000",       "SONY",      "ILCE-7R",              20000,         {0.000651251070020254, 0.000510293553176933, 0.00102156367805789}, {1.99121178332322e-06, 1.30078628026495e-06, 1.98697663791486e-06}},
{"ILCE-7R iso 25600",       "SONY",      "ILCE-7R",              25600,         {0.00050678243246595, 0.000636084182088242, 0.00064626458320255}, {1.97865744779501e-06, 9.96446223242457e-07, 2.03965206316811e-06}},

// ilce-7s by Malte Cornils, bright regions underrepresented in image, slightly more noise there (like +10% at most)
{"ILCE-7S iso 100",       "SONY",      "ILCE-7S",              100,         {1.16780000359953e-06, 4.84004616669359e-07, 1.17038983168053e-06}, {1.71274713330852e-09, 9.88048230966181e-10, 1.78437533821158e-09}},
{"ILCE-7S iso 125",       "SONY",      "ILCE-7S",              125,         {1.47534438046078e-06, 5.97740506081295e-07, 1.51725608200498e-06}, {1.99164061806511e-09, 1.06983371279016e-09, 1.99699898843653e-09}},
{"ILCE-7S iso 160",       "SONY",      "ILCE-7S",              160,         {1.81410718751483e-06, 7.7715956459058e-07, 1.78383545650228e-06}, {2.72946178270093e-09, 1.55467305579143e-09, 2.84229513025242e-09}},
{"ILCE-7S iso 200",       "SONY",      "ILCE-7S",              200,         {2.40364563548709e-06, 9.60163343878738e-07, 2.43813052820538e-06}, {3.29831704416346e-09, 1.92336118905733e-09, 3.50051770541025e-09}},
{"ILCE-7S iso 250",       "SONY",      "ILCE-7S",              250,         {2.63317846858685e-06, 1.2221486485563e-06, 2.6900269765644e-06}, {4.67349669600427e-09, 2.38794222218928e-09, 4.69386294224911e-09}},
{"ILCE-7S iso 320",       "SONY",      "ILCE-7S",              320,         {3.75681949693426e-06, 1.54599471286984e-06, 3.71218265609825e-06}, {5.71605760672412e-09, 3.05108082035022e-09, 6.08365524530749e-09}},
{"ILCE-7S iso 400",       "SONY",      "ILCE-7S",              400,         {4.13058270089208e-06, 1.92677298960251e-06, 4.04771240632653e-06}, {7.7984308882552e-09, 3.77307569721948e-09, 8.02166144410237e-09}},
{"ILCE-7S iso 500",       "SONY",      "ILCE-7S",              500,         {6.00839195459986e-06, 2.38651031035838e-06, 5.63651911998999e-06}, {8.82066622074231e-09, 4.67288046058807e-09, 8.6433594163676e-09}},
{"ILCE-7S iso 640",       "SONY",      "ILCE-7S",              640,         {6.54924569814586e-06, 3.07780884465505e-06, 6.21404684145291e-06}, {1.26937391466551e-08, 6.92969159286105e-09, 1.27477591370805e-08}},
{"ILCE-7S iso 800",       "SONY",      "ILCE-7S",              800,         {7.45347316270917e-06, 3.8205485704841e-06, 7.53185015454969e-06}, {1.66376310646506e-08, 8.53479433292828e-09, 1.70071042201277e-08}},
{"ILCE-7S iso 1000",       "SONY",      "ILCE-7S",              1000,         {9.72996152601117e-06, 4.80408267118039e-06, 9.43571757216118e-06}, {2.17983246041539e-08, 1.20809263261793e-08, 2.18411986333698e-08}},
{"ILCE-7S iso 1250",       "SONY",      "ILCE-7S",              1250,         {1.07256731191449e-05, 5.72334505753267e-06, 1.06601254571726e-05}, {2.92587573463407e-08, 1.65441397783659e-08, 3.00066869256779e-08}},
{"ILCE-7S iso 1600",       "SONY",      "ILCE-7S",              1600,         {1.32255745192978e-05, 7.88206961016538e-06, 1.32804942242684e-05}, {3.40566658367017e-08, 1.97186900849105e-08, 3.48745787587247e-08}},
{"ILCE-7S iso 2000",       "SONY",      "ILCE-7S",              2000,         {2.50254887594989e-05, 1.03660750714191e-05, 2.28904095511643e-05}, {1.53758353594181e-08, 1.34527319759957e-08, 1.5129166324695e-08}},
{"ILCE-7S iso 2500",       "SONY",      "ILCE-7S",              2500,         {3.08124178994474e-05, 1.20674564930473e-05, 2.130636143976e-05}, {2.49611071964523e-08, 2.12967522983227e-08, 2.49699732544829e-08}},
{"ILCE-7S iso 3200",       "SONY",      "ILCE-7S",              3200,         {3.83257254924481e-05, 1.23180428242841e-05, 2.71011097602097e-05}, {3.8799506249347e-08, 3.04533667277501e-08, 3.80408911387986e-08}},
{"ILCE-7S iso 4000",       "SONY",      "ILCE-7S",              4000,         {4.46645428966226e-05, 1.3945928053869e-05, 3.68333647950979e-05}, {5.75821169144654e-08, 3.88136433428111e-08, 5.76250479997227e-08}},
{"ILCE-7S iso 5000",       "SONY",      "ILCE-7S",              5000,         {4.78382802230303e-05, 2.03258948497844e-05, 4.73683201446666e-05}, {8.87019724845304e-08, 4.77978063677061e-08, 8.87842236246807e-08}},
{"ILCE-7S iso 6400",       "SONY",      "ILCE-7S",              6400,         {4.71136219723628e-05, 2.97820544081156e-05, 5.05549005167215e-05}, {1.25758086747406e-07, 6.06173768214416e-08, 1.26547302253357e-07}},
{"ILCE-7S iso 8000",       "SONY",      "ILCE-7S",              8000,         {5.69330340134482e-05, 3.45266262981522e-05, 5.97347940727381e-05}, {1.74657221103781e-07, 9.01084035070753e-08, 1.765574294915e-07}},
{"ILCE-7S iso 10000",       "SONY",      "ILCE-7S",              10000,         {8.03310519769028e-05, 4.6296480421819e-05, 8.07776586584465e-05}, {2.32055348604686e-07, 1.10004550013063e-07, 2.41835505794222e-07}},
{"ILCE-7S iso 12800",       "SONY",      "ILCE-7S",              12800,         {0.000115548920207358, 5.81288816853769e-05, 0.000117803159919638}, {2.95712238083193e-07, 1.52097681323342e-07, 3.11085975333089e-07}},
{"ILCE-7S iso 25600",       "SONY",      "ILCE-7S",              25600,         {0.000273999621319339, 0.000123659180985631, 0.00028905299555926}, {4.12089109009503e-07, 2.38317710095906e-07, 4.33136812728134e-07}},
{"ILCE-7S iso 51200",       "SONY",      "ILCE-7S",              51200,         {0.000204455382165867, 0.000280778161454386, 0.000246634849589833}, {3.93129104724402e-07, 2.4564058617506e-07, 3.89112751219536e-07}},

// sony ilce-6000 by Matti Juvonen
{"ILCE-6000 iso 100",       "SONY",      "ILCE-6000",              100,         {9.49742203143181e-06, 3.33879097017568e-06, 4.77553313054781e-06}, {1.00626535555952e-08, 8.1507880633816e-09, 1.03824662599485e-08}},
{"ILCE-6000 iso 125",       "SONY",      "ILCE-6000",              125,         {1.11226285978416e-05, 3.95368150805264e-06, 5.81202303290319e-06}, {1.26153714538164e-08, 1.01796057568264e-08, 1.34253471770897e-08}},
{"ILCE-6000 iso 160",       "SONY",      "ILCE-6000",              160,         {1.35921451564948e-05, 4.88292532257157e-06, 7.31541808706321e-06}, {1.88094166591242e-08, 1.49568703403943e-08, 1.92806057017391e-08}},
{"ILCE-6000 iso 200",       "SONY",      "ILCE-6000",              200,         {1.63723757518916e-05, 6.04667121239389e-06, 9.0649100695859e-06}, {1.92055956417553e-08, 1.556296292031e-08, 2.01120286196708e-08}},
{"ILCE-6000 iso 250",       "SONY",      "ILCE-6000",              250,         {1.98872286883095e-05, 7.21743356237109e-06, 1.09637967055059e-05}, {2.29431899573483e-08, 1.77620904108033e-08, 2.36979683876537e-08}},
{"ILCE-6000 iso 320",       "SONY",      "ILCE-6000",              320,         {2.41023304324492e-05, 8.67089612940265e-06, 1.32491602575667e-05}, {3.21440055759117e-08, 2.65766483886782e-08, 3.3450130954933e-08}},
{"ILCE-6000 iso 400",       "SONY",      "ILCE-6000",              400,         {2.95336437119605e-05, 1.05786301273409e-05, 1.63335588580201e-05}, {3.71909208542432e-08, 2.89027386668128e-08, 3.87609645617982e-08}},
{"ILCE-6000 iso 500",       "SONY",      "ILCE-6000",              500,         {3.64573824477224e-05, 1.30325647440625e-05, 2.02606610926071e-05}, {5.37372564497618e-08, 4.06567859121507e-08, 5.56489839595908e-08}},
{"ILCE-6000 iso 640",       "SONY",      "ILCE-6000",              640,         {4.47792537566543e-05, 1.61035202464429e-05, 2.49282810978883e-05}, {7.07088802187384e-08, 5.20591779030258e-08, 7.21113109565922e-08}},
{"ILCE-6000 iso 800",       "SONY",      "ILCE-6000",              800,         {5.62549602902515e-05, 2.00972020470045e-05, 3.13496408983179e-05}, {8.92766699630282e-08, 6.65169504863811e-08, 9.21897377797765e-08}},
{"ILCE-6000 iso 1000",       "SONY",      "ILCE-6000",              1000,         {6.95554676621271e-05, 2.51420734978495e-05, 3.89579632902036e-05}, {1.20422815515713e-07, 8.87190001470654e-08, 1.22828511666634e-07}},
{"ILCE-6000 iso 1250",       "SONY",      "ILCE-6000",              1250,         {8.6710136845826e-05, 3.11746178823161e-05, 4.84844152492519e-05}, {1.52556894196602e-07, 1.14976530616013e-07, 1.55016004179249e-07}},
{"ILCE-6000 iso 1600",       "SONY",      "ILCE-6000",              1600,         {0.000107765589637389, 3.8674436461357e-05, 6.02070411943497e-05}, {2.02592141196408e-07, 1.47896363093806e-07, 2.02120015318214e-07}},
{"ILCE-6000 iso 2000",       "SONY",      "ILCE-6000",              2000,         {0.000134468798449785, 4.87082876534147e-05, 7.54828944803209e-05}, {2.57303676398664e-07, 1.90753542893289e-07, 2.58280466203778e-07}},
{"ILCE-6000 iso 2500",       "SONY",      "ILCE-6000",              2500,         {0.00016747968642166, 6.04304763439871e-05, 9.37854201638391e-05}, {3.27145792192946e-07, 2.43660470178109e-07, 3.30027304618182e-07}},
{"ILCE-6000 iso 3200",       "SONY",      "ILCE-6000",              3200,         {0.000208076054703475, 7.58524159451898e-05, 0.000118019878315744}, {4.18822454909447e-07, 3.11432790425357e-07, 4.12653134943774e-07}},
{"ILCE-6000 iso 4000",       "SONY",      "ILCE-6000",              4000,         {0.000257473547045592, 9.45617257200866e-05, 0.000147551273314195}, {5.21124723214108e-07, 4.00552714598523e-07, 5.19848763026984e-07}},
{"ILCE-6000 iso 5000",       "SONY",      "ILCE-6000",              5000,         {0.000320269533741765, 0.000118142857679619, 0.000184337425983007}, {6.34385022740921e-07, 5.05680296726015e-07, 6.4311112119269e-07}},
{"ILCE-6000 iso 6400",       "SONY",      "ILCE-6000",              6400,         {0.000399437428737323, 0.000149410335782994, 0.000232351941982894}, {7.96200789203555e-07, 6.54804611037195e-07, 8.33499130646211e-07}},
{"ILCE-6000 iso 8000",       "SONY",      "ILCE-6000",              8000,         {0.000498593507061273, 0.000188973608916726, 0.000293349720412477}, {1.01876677903293e-06, 8.7819184046477e-07, 1.10476660681968e-06}},
{"ILCE-6000 iso 10000",       "SONY",      "ILCE-6000",              10000,         {0.000617550215914806, 0.000239649304647913, 0.000373269543204376}, {1.34565274259752e-06, 1.19257997639166e-06, 1.48235346354954e-06}},
{"ILCE-6000 iso 12800",       "SONY",      "ILCE-6000",              12800,         {0.000513673184387624, 0.000196000293291409, 0.000300893162155235}, {1.36769259544917e-06, 1.29739639987722e-06, 1.52303024539103e-06}},
{"ILCE-6000 iso 16000",       "SONY",      "ILCE-6000",              16000,         {0.000636209080088239, 0.000251446906069856, 0.000381329229519294}, {1.83968851262639e-06, 1.81258939932108e-06, 2.08632639241959e-06}},
{"ILCE-6000 iso 20000",       "SONY",      "ILCE-6000",              20000,         {0.000774425990363394, 0.000316236625948286, 0.00048386949341565}, {2.53581924675283e-06, 2.61110877952971e-06, 2.90036295396829e-06}},
{"ILCE-6000 iso 25600",       "SONY",      "ILCE-6000",              25600,         {0.000958549119029686, 0.000404073507896682, 0.000627190876742582}, {3.53018984603028e-06, 3.80151544959246e-06, 3.99328572269875e-06}},

// sony ilce-5100 contributed by Mat
  {"ILCE-5100 iso 100",       "SONY",      "ILCE-5100",              100,         {7.49892719852375e-06, 3.1167955863864e-06, 6.04026075175609e-06}, {1.26327589823999e-08, 7.23824440963383e-09, 1.38366591946571e-08}},
  {"ILCE-5100 iso 125",       "SONY",      "ILCE-5100",              125,         {8.31157223509572e-06, 3.8308815143438e-06, 7.49528606856059e-06}, {1.74694134988507e-08, 9.06734907377784e-09, 1.73245573448545e-08}},
  {"ILCE-5100 iso 160",       "SONY",      "ILCE-5100",              160,         {9.90729282050676e-06, 4.71834235314889e-06, 9.26193179834325e-06}, {2.265467727286e-08, 1.14832092447625e-08, 2.20205911185764e-08}},
  {"ILCE-5100 iso 200",       "SONY",      "ILCE-5100",              200,         {1.21780076065907e-05, 5.7916742301351e-06, 1.13391523019714e-05}, {2.85110651688313e-08, 1.53895373146607e-08, 2.78475878629151e-08}},
  {"ILCE-5100 iso 250",       "SONY",      "ILCE-5100",              250,         {1.42613832091896e-05, 7.13671268123463e-06, 1.41073855565748e-05}, {3.81912361203788e-08, 2.0185571481048e-08, 3.56100797514975e-08}},
  {"ILCE-5100 iso 320",       "SONY",      "ILCE-5100",              320,         {1.66087179690462e-05, 8.86016488971504e-06, 1.76037399208698e-05}, {5.34343169818498e-08, 2.78716128647683e-08, 4.75929982536155e-08}},
  {"ILCE-5100 iso 400",       "SONY",      "ILCE-5100",              400,         {2.21770562761272e-05, 1.0984908182264e-05, 2.18740880599176e-05}, {6.79246692134295e-08, 3.62541845658057e-08, 6.18451722856456e-08}},
  {"ILCE-5100 iso 500",       "SONY",      "ILCE-5100",              500,         {3.0574707007313e-05, 1.38544228886168e-05, 2.7462696185582e-05}, {8.29507169664312e-08, 4.68348054478874e-08, 7.98622046990885e-08}},
  {"ILCE-5100 iso 640",       "SONY",      "ILCE-5100",              640,         {5.07226027917771e-05, 1.65730524419899e-05, 2.93617461902417e-05}, {4.42339397662554e-08, 3.20635008590472e-08, 4.47995841243798e-08}},
  {"ILCE-5100 iso 800",       "SONY",      "ILCE-5100",              800,         {5.62441673653945e-05, 2.12394287908329e-05, 3.83479344312903e-05}, {7.65721886123826e-08, 5.58864291222653e-08, 7.36814781789984e-08}},
  {"ILCE-5100 iso 1000",       "SONY",      "ILCE-5100",              1000,         {7.43097262144747e-05, 2.65469081336242e-05, 4.78840832685497e-05}, {9.93098594034474e-08, 7.44054405719075e-08, 9.83820354789933e-08}},
  {"ILCE-5100 iso 1250",       "SONY",      "ILCE-5100",              1250,         {9.15876960306288e-05, 3.33743675770957e-05, 6.07577653191689e-05}, {1.43391140368743e-07, 1.07889737558674e-07, 1.41847382689606e-07}},
  {"ILCE-5100 iso 1600",       "SONY",      "ILCE-5100",              1600,         {0.000111940075262127, 4.17857589919798e-05, 7.82777692510145e-05}, {2.0264456498497e-07, 1.51128570584216e-07, 1.99762223364744e-07}},
  {"ILCE-5100 iso 2000",       "SONY",      "ILCE-5100",              2000,         {0.000139519487674355, 5.2275709866881e-05, 9.77014809726257e-05}, {2.54502478721868e-07, 1.85917930704317e-07, 2.50209185521318e-07}},
  {"ILCE-5100 iso 2500",       "SONY",      "ILCE-5100",              2500,         {0.000176768017437114, 6.49604939848781e-05, 0.000123009029840579}, {3.90756870935572e-07, 2.82594252106848e-07, 3.83470492559162e-07}},
  {"ILCE-5100 iso 3200",       "SONY",      "ILCE-5100",              3200,         {0.000222901332631808, 8.1033165178216e-05, 0.00015492562194634}, {5.49525638424632e-07, 3.99481747055808e-07, 5.40246096323643e-07}},
  {"ILCE-5100 iso 4000",       "SONY",      "ILCE-5100",              4000,         {0.000272702860409206, 9.95690556414945e-05, 0.000190254035467568}, {7.23443424887309e-07, 5.41783754901496e-07, 7.15742858739861e-07}},
  {"ILCE-5100 iso 5000",       "SONY",      "ILCE-5100",              5000,         {0.000329156784401798, 0.000123267093736456, 0.000239195760323569}, {1.02786770747103e-06, 7.95191062031751e-07, 1.01515992778859e-06}},
  {"ILCE-5100 iso 6400",       "SONY",      "ILCE-5100",              6400,         {0.000396805485782619, 0.000148098705329425, 0.000291184252673775}, {1.13660706457223e-06, 8.12461934339301e-07, 1.08225634659118e-06}},
  {"ILCE-5100 iso 8000",       "SONY",      "ILCE-5100",              8000,         {0.000484635140286118, 0.000184677233903701, 0.000366073232243667}, {1.55280232417787e-06, 1.16908761662752e-06, 1.46923077132447e-06}},
  {"ILCE-5100 iso 10000",       "SONY",      "ILCE-5100",              10000,         {0.000574094976090249, 0.000233807831293575, 0.00044511245802353}, {2.34761033722574e-06, 1.70793026016389e-06, 2.20263805693606e-06}},
  {"ILCE-5100 iso 12800",       "SONY",      "ILCE-5100",              12800,         {0.000468336010941953, 0.000191633800223251, 0.000366286740621362}, {2.29474520884781e-06, 1.78941070380447e-06, 2.11785535352908e-06}},
  {"ILCE-5100 iso 16000",       "SONY",      "ILCE-5100",              16000,         {0.00057941054774327, 0.000240807718861733, 0.000469462090689246}, {3.63724359328439e-06, 2.7781961473651e-06, 3.18926095047669e-06}},
  {"ILCE-5100 iso 20000",       "SONY",      "ILCE-5100",              20000,         {0.000711782577622493, 0.000301805713360154, 0.000597717245273993}, {5.6489963993159e-06, 4.34721144735838e-06, 4.74256722737488e-06}},
  {"ILCE-5100 iso 25600",       "SONY",      "ILCE-5100",              25600,         {0.000874295653382787, 0.000380605701230177, 0.000751858523681735}, {9.29147118003536e-06, 6.84274628665347e-06, 7.37266036463866e-06}},

// fujifilm x10 by rob
  {"Fuji X10 iso 100",  "FUJIFILM", "X10", 100,  {2.32144895406858e-05, 1.10403725718086e-05, 2.27471181936408e-05}, {1.37499138625802e-08, 2.44848953581136e-08, 6.66886186121249e-08}},
  {"Fuji X10 iso 200",  "FUJIFILM", "X10", 200,  {2.49255304236662e-05, 1.15980212460838e-05, 2.32072035328905e-05}, {1.32605645545644e-08, 1.99784295154837e-08, 5.67224395232843e-08}},
  {"Fuji X10 iso 250",  "FUJIFILM", "X10", 250,  {2.84363668221069e-05, 1.32269493172388e-05, 2.66312030136466e-05}, {2.68583759267478e-08, 2.35265613383566e-08, 6.99317484699719e-08}},
  {"Fuji X10 iso 320",  "FUJIFILM", "X10", 320,  {3.54151155953606e-05, 1.65247634928859e-05, 3.33336144694953e-05}, {3.68616017951439e-08, 3.58001954034872e-08, 9.24636845642425e-08}},
  {"Fuji X10 iso 400",  "FUJIFILM", "X10", 400,  {2.38209368677238e-05, 1.12102637868409e-05, 2.27628053782582e-05}, {3.91633724867466e-08, 2.86577437960836e-08, 5.6703167783222e-08}},
  {"Fuji X10 iso 500",  "FUJIFILM", "X10", 500,  {3.00315853344797e-05, 1.39859014766729e-05, 2.80707357531215e-05}, {4.92491988417534e-08, 4.37014262052915e-08, 7.49774462691605e-08}},
  {"Fuji X10 iso 640",  "FUJIFILM", "X10", 640,  {3.17399977684284e-05, 1.46932633577131e-05, 2.94370263871516e-05}, {4.66221843386263e-08, 3.5540863062326e-08, 6.9872620333414e-08}},
  {"Fuji X10 iso 800",  "FUJIFILM", "X10", 800,  {3.95678108509452e-05, 1.82643130822075e-05, 3.68627640362496e-05}, {6.0350501197447e-08, 4.59024355924827e-08, 9.75684516412839e-08}},
  {"Fuji X10 iso 1000", "FUJIFILM", "X10", 1000, {4.9716214104893e-05, 2.26361421128087e-05, 4.5865761362122e-05}, {8.0637722073172e-08, 6.24754713928627e-08, 1.25699358928857e-07}},
  {"Fuji X10 iso 1250", "FUJIFILM", "X10", 1250, {6.16734976964275e-05, 2.86540458317097e-05, 5.76728154753275e-05}, {1.15329360737943e-07, 8.29019374710041e-08, 1.69671297993566e-07}},
  {"Fuji X10 iso 1600", "FUJIFILM", "X10", 1600, {7.6348869831024e-05, 3.52922618536065e-05, 7.23394105134079e-05}, {1.72240590684159e-07, 1.32158967951784e-07, 2.31407539679491e-07}},
  {"Fuji X10 iso 2000", "FUJIFILM", "X10", 2000, {0.000101424824311967, 4.69549531992267e-05, 9.65096292458063e-05}, {2.82090938451258e-07, 2.24401447503124e-07, 3.72236786480327e-07}},
  {"Fuji X10 iso 2500", "FUJIFILM", "X10", 2500, {0.000118274955536686, 5.47594867992304e-05, 0.000112338431227319}, {3.5612760523187e-07, 2.8916105825585e-07, 4.56956986025887e-07}},
  {"Fuji X10 iso 3200", "FUJIFILM", "X10", 3200, {0.000150110326718357, 6.97438914601073e-05, 0.000143916487244602}, {5.14719897054068e-07, 4.2483341129698e-07, 6.40774228530887e-07}},

// fujifilm x-t1 by ulrich
  {"Fuji X-T1 iso 200",  "FUJIFILM", "X-T1",  200, {1.31252992122382e-05, 4.61898555600132e-06, 8.89313720419275e-06}, {2.97890009041094e-08, 7.87670143708163e-09, 8.81502606490751e-09}},
  {"Fuji X-T1 iso 250",  "FUJIFILM", "X-T1",  250, {1.56008586696921e-05, 5.5459310933268e-06, 1.06900973738884e-05}, {2.78859458456187e-08, 6.76223783192493e-09, 2.44164987976321e-08}},
  {"Fuji X-T1 iso 320",  "FUJIFILM", "X-T1",  320, {2.05808431754735e-05, 7.0341176876662e-06, 1.31073011524834e-05}, {3.53239895598638e-08, 8.84720010556284e-09, 3.07049476960542e-08}},
  {"Fuji X-T1 iso 400",  "FUJIFILM", "X-T1",  400, {2.58971620949888e-05, 8.7241248955326e-06, 1.61772237050522e-05}, {3.72542829496922e-08, 1.67295058605305e-08, 4.77493155703809e-08}},
  {"Fuji X-T1 iso 500",  "FUJIFILM", "X-T1",  500, {3.21982826058474e-05, 1.09183600335607e-05, 2.02813595156363e-05}, {5.40284467382025e-08, 1.60372211168724e-08, 5.71858183577399e-08}},
  {"Fuji X-T1 iso 640",  "FUJIFILM", "X-T1",  640, {4.02836932944995e-05, 1.3639301157256e-05, 2.55271109756073e-05}, {5.76148090285642e-08, 2.38725870241332e-08, 8.44852158476115e-08}},
  {"Fuji X-T1 iso 800",  "FUJIFILM", "X-T1",  800, {5.1723185395914e-05, 1.71298172275452e-05, 3.10783077098765e-05}, {9.43379168547906e-08, 3.39071724000471e-08, 9.69451012360005e-08}},
  {"Fuji X-T1 iso 1000", "FUJIFILM", "X-T1", 1000, {6.0964375366835e-05, 2.11630553576901e-05, 4.07841809818507e-05}, {1.1476055925777e-07, 5.57021940446684e-08, 1.3640060755648e-07}},
  {"Fuji X-T1 iso 1250", "FUJIFILM", "X-T1", 1250, {7.66254281759912e-05, 2.65379732855013e-05, 5.1474624902721e-05}, {2.06888235167867e-07, 9.20548010392009e-08, 1.32109178406552e-07}},
  {"Fuji X-T1 iso 1600", "FUJIFILM", "X-T1", 1600, {0.00010087094937336, 3.28187865061588e-05, 6.14091154162569e-05}, {1.08263263667744e-07, 1.35888709159059e-07, 2.581034647269e-07}},
  {"Fuji X-T1 iso 2000", "FUJIFILM", "X-T1", 2000, {0.000122538866794544, 4.20897186612317e-05, 8.05625532190959e-05}, {2.86595854345368e-07, 1.59396731639574e-07, 2.75401902644433e-07}},
  {"Fuji X-T1 iso 2500", "FUJIFILM", "X-T1", 2500, {0.000123825394623677, 4.18877894504511e-05, 7.94264917019512e-05}, {2.98533684517939e-07, 1.54382865244716e-07, 2.3455895454894e-07}},
  {"Fuji X-T1 iso 3200", "FUJIFILM", "X-T1", 3200, {0.000133291746437379, 4.20425073041664e-05, 7.72132129416139e-05}, {1.13881438148889e-07, 1.50292616683618e-07, 2.81296856357845e-07}},
  {"Fuji X-T1 iso 4000", "FUJIFILM", "X-T1", 4000, {0.000131116009998063, 4.16787258944441e-05, 7.77936909642969e-05}, {-3.05271971049083e-08, 1.29551966212407e-07, 3.12908465971114e-07}},
  {"Fuji X-T1 iso 5000", "FUJIFILM", "X-T1", 5000, {0.000134043829792419, 4.2357402979867e-05, 7.70059152480817e-05}, {1.36264418240426e-07, 1.47293613952519e-07, 2.38894850008169e-07}},
  {"Fuji X-T1 iso 6400", "FUJIFILM", "X-T1", 6400, {0.000127923776194175, 4.152080368723e-05, 7.78754619809228e-05}, {1.70205481352036e-07, 1.57222947637482e-07, 2.54804994234622e-07}},

// fujifilm x-e2 by ulrich (measured on the x-t1, it's generally believed that the x-e2 has the same sensor)
  {"Fuji X-E2 iso 200",  "FUJIFILM", "X-E2",  200, {1.31252992122382e-05, 4.61898555600132e-06, 8.89313720419275e-06}, {2.97890009041094e-08, 7.87670143708163e-09, 8.81502606490751e-09}},
  {"Fuji X-E2 iso 250",  "FUJIFILM", "X-E2",  250, {1.56008586696921e-05, 5.5459310933268e-06, 1.06900973738884e-05}, {2.78859458456187e-08, 6.76223783192493e-09, 2.44164987976321e-08}},
  {"Fuji X-E2 iso 320",  "FUJIFILM", "X-E2",  320, {2.05808431754735e-05, 7.0341176876662e-06, 1.31073011524834e-05}, {3.53239895598638e-08, 8.84720010556284e-09, 3.07049476960542e-08}},
  {"Fuji X-E2 iso 400",  "FUJIFILM", "X-E2",  400, {2.58971620949888e-05, 8.7241248955326e-06, 1.61772237050522e-05}, {3.72542829496922e-08, 1.67295058605305e-08, 4.77493155703809e-08}},
  {"Fuji X-E2 iso 500",  "FUJIFILM", "X-E2",  500, {3.21982826058474e-05, 1.09183600335607e-05, 2.02813595156363e-05}, {5.40284467382025e-08, 1.60372211168724e-08, 5.71858183577399e-08}},
  {"Fuji X-E2 iso 640",  "FUJIFILM", "X-E2",  640, {4.02836932944995e-05, 1.3639301157256e-05, 2.55271109756073e-05}, {5.76148090285642e-08, 2.38725870241332e-08, 8.44852158476115e-08}},
  {"Fuji X-E2 iso 800",  "FUJIFILM", "X-E2",  800, {5.1723185395914e-05, 1.71298172275452e-05, 3.10783077098765e-05}, {9.43379168547906e-08, 3.39071724000471e-08, 9.69451012360005e-08}},
  {"Fuji X-E2 iso 1000", "FUJIFILM", "X-E2", 1000, {6.0964375366835e-05, 2.11630553576901e-05, 4.07841809818507e-05}, {1.1476055925777e-07, 5.57021940446684e-08, 1.3640060755648e-07}},
  {"Fuji X-E2 iso 1250", "FUJIFILM", "X-E2", 1250, {7.66254281759912e-05, 2.65379732855013e-05, 5.1474624902721e-05}, {2.06888235167867e-07, 9.20548010392009e-08, 1.32109178406552e-07}},
  {"Fuji X-E2 iso 1600", "FUJIFILM", "X-E2", 1600, {0.00010087094937336, 3.28187865061588e-05, 6.14091154162569e-05}, {1.08263263667744e-07, 1.35888709159059e-07, 2.581034647269e-07}},
  {"Fuji X-E2 iso 2000", "FUJIFILM", "X-E2", 2000, {0.000122538866794544, 4.20897186612317e-05, 8.05625532190959e-05}, {2.86595854345368e-07, 1.59396731639574e-07, 2.75401902644433e-07}},
  {"Fuji X-E2 iso 2500", "FUJIFILM", "X-E2", 2500, {0.000123825394623677, 4.18877894504511e-05, 7.94264917019512e-05}, {2.98533684517939e-07, 1.54382865244716e-07, 2.3455895454894e-07}},
  {"Fuji X-E2 iso 3200", "FUJIFILM", "X-E2", 3200, {0.000133291746437379, 4.20425073041664e-05, 7.72132129416139e-05}, {1.13881438148889e-07, 1.50292616683618e-07, 2.81296856357845e-07}},
  {"Fuji X-E2 iso 4000", "FUJIFILM", "X-E2", 4000, {0.000131116009998063, 4.16787258944441e-05, 7.77936909642969e-05}, {-3.05271971049083e-08, 1.29551966212407e-07, 3.12908465971114e-07}},
  {"Fuji X-E2 iso 5000", "FUJIFILM", "X-E2", 5000, {0.000134043829792419, 4.2357402979867e-05, 7.70059152480817e-05}, {1.36264418240426e-07, 1.47293613952519e-07, 2.38894850008169e-07}},
  {"Fuji X-E2 iso 6400", "FUJIFILM", "X-E2", 6400, {0.000127923776194175, 4.152080368723e-05, 7.78754619809228e-05}, {1.70205481352036e-07, 1.57222947637482e-07, 2.54804994234622e-07}},

// fuji X100 by captain siff
{"FinePix X100 iso 200", "FUJIFILM", "FinePix X100", 200, {3.5724537414653e-06,3.3536449775885e-06,1.251835540749e-05},{7.7055775233355e-09,6.3668361387545e-09,2.6543912028777e-08}},
{"FinePix X100 iso 250", "FUJIFILM", "FinePix X100", 250, {4.4154897653462e-06,4.1615957308614e-06,1.527765127711e-05},{1.0510463554552e-08,8.6365458977992e-09,3.4802252693212e-08}},
{"FinePix X100 iso 320", "FUJIFILM", "FinePix X100", 320, {5.5090786982417e-06,5.1729811855743e-06,1.9200145670976e-05},{1.4197868825734e-08,1.0896554231451e-08,3.9367192863767e-08}},
{"FinePix X100 iso 400", "FUJIFILM", "FinePix X100", 400, {6.8927519137105e-06,6.418782327969e-06,2.3973312470863e-05},{1.9997591655623e-08,1.4271657293905e-08,5.599866445685e-08}},
{"FinePix X100 iso 500", "FUJIFILM", "FinePix X100", 500, {8.9650180925027e-06,8.0545481291147e-06,3.0014234018317e-05},{2.4755049226292e-08,1.8388853159883e-08,6.854663357072e-08}},
{"FinePix X100 iso 640", "FUJIFILM", "FinePix X100", 640, {1.1325299628523e-05,1.025725010781e-05,3.804866371012e-05},{3.6292344277935e-08,2.1383482472932e-08,8.8330708933903e-08}},
{"FinePix X100 iso 800", "FUJIFILM", "FinePix X100", 800, {1.4190941918902e-05,1.2827759603117e-05,4.7321221210447e-05},{5.411490636522e-08,2.679704907762e-08,1.2181387952941e-07}},
{"FinePix X100 iso 1000", "FUJIFILM", "FinePix X100", 1000, {1.7869531637315e-05,1.6077063464829e-05,5.929471938027e-05},{7.6161658352907e-08,3.1994226796575e-08,1.5133183139171e-07}},
{"FinePix X100 iso 1250", "FUJIFILM", "FinePix X100", 1250, {2.2688910090174e-05,2.0330042131661e-05,7.4905288019158e-05},{1.1096113321987e-07,4.772540696447e-08,2.1097878747413e-07}},
{"FinePix X100 iso 1600", "FUJIFILM", "FinePix X100", 1600, {2.8782862809228e-05,2.5390498936206e-05,9.3669528173218e-05},{1.5753160745851e-07,6.0920831796774e-08,2.8813405442468e-07}},
{"FinePix X100 iso 2000", "FUJIFILM", "FinePix X100", 2000, {1.8143075916772e-05,1.5937883748877e-05,5.7551730582682e-05},{8.066129860442e-08,3.3498550975588e-08,1.332790883886e-07}},
{"FinePix X100 iso 2500", "FUJIFILM", "FinePix X100", 2500, {2.3090459194896e-05,2.0554017780478e-05,7.2480171806546e-05},{1.1461168294294e-07,3.6789200573494e-08,1.8367242339803e-07}},
{"FinePix X100 iso 3200", "FUJIFILM", "FinePix X100", 3200, {2.927083228016e-05,2.5786625870525e-05,9.2151710411743e-05},{1.6161321597347e-07,5.804482913208e-08,2.3946451269453e-07}},
{"FinePix X100 iso 4000", "FUJIFILM", "FinePix X100", 4000, {1.8786792896982e-05,1.5973897936559e-05,5.0049278459554e-05},{7.6430962945328e-08,3.461563594215e-08,9.9859376185126e-08}},
{"FinePix X100 iso 5000", "FUJIFILM", "FinePix X100", 5000, {2.3878186213768e-05,2.0363149494993e-05,6.3434940467872e-05},{1.0857130796469e-07,3.5375332270018e-08,1.5076720788396e-07}},
{"FinePix X100 iso 6400", "FUJIFILM", "FinePix X100", 6400, {3.0148100772718e-05,2.6433538187136e-05,8.0881151590032e-05},{1.4893507148005e-07,4.7822027957224e-08,2.0760029937754e-07}},

// Fujifilm X100S contributed by Dan Torop
{"X100S iso 200",       "FUJIFILM",      "X100S",              200,         {1.42104894051167e-05, 5.14993580459437e-06, 1.02754164527409e-05}, {8.95775368462702e-09, 5.75516517203673e-09, 9.55025298481869e-09}},
{"X100S iso 250",       "FUJIFILM",      "X100S",              250,         {1.61694305095309e-05, 5.75283194643454e-06, 1.16727675414652e-05}, {1.23850699615317e-08, 8.08982331415591e-09, 1.34207536646236e-08}},
{"X100S iso 320",       "FUJIFILM",      "X100S",              320,         {1.95521195947406e-05, 6.95663733767418e-06, 1.41969866831426e-05}, {1.68788029211529e-08, 1.1264899369223e-08, 1.78675811199628e-08}},
{"X100S iso 400",       "FUJIFILM",      "X100S",              400,         {2.43943674899235e-05, 8.65951020680453e-06, 1.76715948684952e-05}, {2.17065783208271e-08, 1.45958864954406e-08, 2.31105124644302e-08}},
{"X100S iso 500",       "FUJIFILM",      "X100S",              500,         {3.00911977695808e-05, 1.06877430612416e-05, 2.18281923288743e-05}, {2.81388521660289e-08, 1.94236939138495e-08, 2.96366998243926e-08}},
{"X100S iso 640",       "FUJIFILM",      "X100S",              640,         {3.87123988651168e-05, 1.38926650325104e-05, 2.61007801193065e-05}, {5.9280859461377e-08, 1.65629365701816e-08, 4.1976436776427e-08}},
{"X100S iso 800",       "FUJIFILM",      "X100S",              800,         {4.53318423717586e-05, 1.62689788237248e-05, 3.05510800238765e-05}, {7.3078914615607e-08, 2.39051697837648e-08, 5.23778953388085e-08}},
{"X100S iso 1000",       "FUJIFILM",      "X100S",              1000,         {5.70099549557356e-05, 2.05912913441767e-05, 3.82779169471831e-05}, {9.79095387812446e-08, 3.68828032357362e-08, 7.542665735223e-08}},
{"X100S iso 1250",       "FUJIFILM",      "X100S",              1250,         {7.43065518049967e-05, 2.70816626903298e-05, 5.01860231499309e-05}, {1.14458442715727e-07, 3.57408625874479e-08, 8.11045227471575e-08}},
{"X100S iso 1600",       "FUJIFILM",      "X100S",              1600,         {8.819189582515445e-05, 3.1631560477445904e-05, 5.94334360277765e-05}, {1.674870906290695e-07, 7.649222165449635e-08, 1.4112813692962698e-07}},
{"X100S iso 2000",       "FUJIFILM",      "X100S",              2000,         {0.00010421715937694, 3.6945016934419e-05, 7.1086371756495e-05}, {2.2341351415983e-07, 1.13417882432606e-07, 1.88117561849865e-07}},
{"X100S iso 2500",       "FUJIFILM",      "X100S",              2500,         {0.00013108357412016, 4.69137140370566e-05, 8.81399031355613e-05}, {2.5703890351259e-07, 1.30436595883075e-07, 2.20844561957094e-07}},
{"X100S iso 3200",       "FUJIFILM",      "X100S",              3200,         {0.0001291937229155835, 4.607888768089825e-05, 8.72921132754699e-05}, {2.45962729009841e-07, 1.42262459226602e-07, 2.2271465553342102e-07}},
{"X100S iso 4000",       "FUJIFILM",      "X100S",              4000,         {0.000124723832298109, 4.38595024657058e-05, 9.12208053399433e-05}, {1.78504020588315e-07, 1.37660752397007e-07, 1.88111094098724e-07}},
{"X100S iso 5000",       "FUJIFILM",      "X100S",              5000,         {0.000124986845209115, 4.37010904895783e-05, 9.16247494413428e-05}, {1.76411750774797e-07, 1.36319413715772e-07, 1.86926272703825e-07}},
{"X100S iso 6400",       "FUJIFILM",      "X100S",              6400,         {0.00012579517040419198, 4.422836018589326e-05, 9.183407428326638e-05}, {1.588587495193e-07, 1.2486482232114292e-07, 1.690787692086013e-07}},

// fujifilm finepix hs50exr contributed by Mat
{"FinePix HS50EXR iso 100",       "FUJIFILM",      "FinePix HS50EXR",              100,         {3.81219111529217e-05, 2.24003684485457e-05, 5.36976587332853e-05}, {8.71599562961107e-08, 5.32545337051316e-08, 1.05644709361919e-07}},
{"FinePix HS50EXR iso 200",       "FUJIFILM",      "FinePix HS50EXR",              200,         {3.79124160669571e-05, 2.20590553915043e-05, 5.33590189048668e-05}, {9.13312740732644e-08, 6.03115545479695e-08, 9.81060352616303e-08}},
{"FinePix HS50EXR iso 400",       "FUJIFILM",      "FinePix HS50EXR",              400,         {4.6367962865374e-05, 2.45677760166136e-05, 5.34281866946719e-05}, {4.81799004090481e-08, 3.87357267415957e-08, 6.32611334062915e-08}},
{"FinePix HS50EXR iso 800",       "FUJIFILM",      "FinePix HS50EXR",              800,         {6.07040032679925e-05, 3.44120596863503e-05, 8.56963926638951e-05}, {1.14634362062682e-07, 8.97954233456405e-08, 1.26181404543635e-07}},
{"FinePix HS50EXR iso 1600",       "FUJIFILM",      "FinePix HS50EXR",              1600,         {0.000120913318553405, 6.87787846275694e-05, 0.000169601624598948}, {2.89174837886409e-07, 2.13692845938706e-07, 3.14871897980861e-07}},
{"FinePix HS50EXR iso 3200",       "FUJIFILM",      "FinePix HS50EXR",              3200,         {0.000237855703717424, 0.000138807631643134, 0.000332549464301646}, {8.17076984710827e-07, 6.32449105202442e-07, 8.59645392843928e-07}},

  // clang-format on
};
const int dt_noiseprofile_cnt = sizeof(dt_noiseprofiles)/sizeof(dt_noiseprofile_t);

/*
 * fill the given buffer buf of pointers
 * to the noiseprofile array, if they are
 * found to match the given image's exif data.
 * don't exceed the bufsize, and return the count.
 */
int dt_noiseprofile_get_matching(
  const dt_image_t *cimg,
  const dt_noiseprofile_t **buf,
  const int bufsize)
{
  int cnt = 0;
  if(bufsize < 1) return cnt;
  // collect all matching entries (skip generic [0])
  for(int i=1; i<dt_noiseprofile_cnt; i++)
  {
    if(strstr(cimg->exif_maker, dt_noiseprofiles[i].maker) &&
        // we want this to be an exact match
        !strcmp(cimg->exif_model, dt_noiseprofiles[i].model))
    {
      buf[cnt++] = dt_noiseprofiles + i;
    }
    if(cnt >= bufsize) break;
  }
  return cnt;
}

/*
 * interpolate values from p1 and p2 into out.
 */
void dt_noiseprofile_interpolate(
  const dt_noiseprofile_t *const p1,  // the smaller iso
  const dt_noiseprofile_t *const p2,  // the larger iso (can't be == iso1)
  dt_noiseprofile_t *out)             // has iso initialized
{
  // stupid linear interpolation.
  // to be confirmed for gaussian part.
  const float t = CLAMP(
                    (float)(out->iso - p1->iso) / (float)(p2->iso - p1->iso),
                    0.0f, 1.0f);
  for(int k=0; k<3; k++)
  {
    out->a[k] = (1.0f-t)*p1->a[k] + t*p2->a[k];
    out->b[k] = (1.0f-t)*p1->b[k] + t*p2->b[k];
  }
}

