#include "DDSImage.h"

#include "itextstream.h"
#include "BasicTexture2D.h"

TexturePtr DDSImage::bindTexture(const std::string& name) const
{
    GLuint textureNum;

    debug::assertNoGlErrors();

    // Allocate a new texture number and store it into the Texture structure
    glGenTextures(1, &textureNum);
    glBindTexture(GL_TEXTURE_2D, textureNum);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

    glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_FALSE);

    for (std::size_t i = 0; i < _mipMapInfo.size(); ++i)
    {
        const MipMapInfo& mipMap = _mipMapInfo[i];

        glCompressedTexImage2D(
            GL_TEXTURE_2D,
            static_cast<GLint>(i),
            _format,
            static_cast<GLsizei>(mipMap.width),
            static_cast<GLsizei>(mipMap.height),
            0,
            static_cast<GLsizei>(mipMap.size),
            _pixelData + mipMap.offset
        );

        // Handle unsupported format error
        if (glGetError() == GL_INVALID_ENUM)
        {
            rConsoleError() << "[DDSImage] Unable to bind texture '"
                      << name << "'; unsupported texture format"
                      << std::endl;

            return TexturePtr();
        }

        debug::assertNoGlErrors();
    }

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_LEVEL, static_cast<GLint>(_mipMapInfo.size() - 1));

    // Un-bind the texture
    glBindTexture(GL_TEXTURE_2D, 0);

    // Create and return texture object
    BasicTexture2DPtr texObj(new BasicTexture2D(textureNum, name));
    texObj->setWidth(getWidth(0));
    texObj->setHeight(getHeight(0));

    debug::assertNoGlErrors();

    return texObj;
}

void DDSImage::addMipMap(std::size_t mipWidth,
                         std::size_t mipHeight,
                         std::size_t mipSize,
                         std::size_t mipOffset)
{
    MipMapInfo info;

    info.size = mipSize;
    info.width = mipWidth;
    info.height = mipHeight;
    info.offset = mipOffset;

    _mipMapInfo.push_back(info);
}

