;;;; -*- mode: lisp; indent-tabs-mode: nil -*-
#+(and sbcl (or x86 x86-64))
(defpackage :ironclad-vm
  ;; more recent SBCL exports various symbols making this package
  ;; definition more concise. This is the backward-compatible way.
  (:use #:common-lisp
        #:sb-c     ; for DEFINE-VOP and SC-IS
        #:sb-assem ; for INST, GEN-LABEL
        #:sb-vm)   ; for primtype names, SCs, constants
  (:shadow #:ea) ; in case SB-VM exports it
  (:import-from #:sb-vm
                #:positive-fixnum #:unsigned-num
                #:descriptor-reg #:unsigned-reg #:double-reg #:immediate
                #:simple-array-unsigned-byte-8
                #:simple-array-unsigned-byte-32
                #+x86-64 #:simple-array-unsigned-byte-64
                #+x86-64 #:rax-offset #+x86-64 #:rcx-offset))

#+(and sbcl (or x86 x86-64))
(in-package :ironclad-vm)

#+(and sbcl x86)
(eval-when (:compile-toplevel :load-toplevel :execute)
  (defun ea (displacement &optional base index (scale 1))
    (sb-vm::make-ea :dword
                    :base base
                    :index index
                    :scale scale
                    :disp (or displacement 0)))
  (setf (fdefinition 'dword-ea) (fdefinition 'ea))
  (defmacro dword-inst (name &rest operands)
    `(inst ,name ,@operands)))

#+(and sbcl x86-64)
(eval-when (:compile-toplevel :load-toplevel :execute)
  (if (fboundp 'sb-vm::ea)
      (progn ; Newer SBCL (>= 1.4.11)
        (setf (fdefinition 'ea) (fdefinition 'sb-vm::ea))
        (setf (fdefinition 'dword-ea) (fdefinition 'ea))
        (defmacro dword-inst (name &rest operands)
          `(inst ,name :dword ,@operands)))

      (progn ; Older SBCL (< 1.4.11)
        (defun ea (displacement &optional base index (scale 1))
          (sb-vm::make-ea :qword
                          :base base
                          :index index
                          :scale scale
                          :disp (or displacement 0)))
        (defun dword-ea (displacement &optional base index (scale 1))
          (sb-vm::make-ea :dword
                          :base base
                          :index index
                          :scale scale
                          :disp (or displacement 0)))
        (defmacro dword-inst (name &rest operands)
          `(inst ,name ,@(mapcar (lambda (operand)
                                   `(if (tn-p ,operand)
                                        (sb-vm::reg-in-size ,operand :dword)
                                        ,operand))
                                 operands))))))

#+(and sbcl (or x86 x86-64))
(progn
(define-vop (fill-block-ub8)
  (:policy :fast-safe)
  (:args (block :scs (descriptor-reg))
         (buffer :scs (descriptor-reg))
         (offset :scs (unsigned-reg immediate) :target buffer-index))
  (:variant-vars big-endian-p bytes-to-copy 64-bit-p)
  (:temporary (:sc unsigned-reg) temp buffer-index block-index)
  (:generator 50
    (let* ((data-offset (- (* n-word-bytes vector-data-offset)
                             other-pointer-lowtag))
           (block-disp (+ data-offset bytes-to-copy))
           (immediate-offset (sc-is offset immediate))
           (unroll (if immediate-offset 2 1))
           (loop (gen-label)))
      (flet ((ea-for-buffer (&optional (offset 0))
               (if immediate-offset
                   (ea (+ block-disp offset) buffer block-index n-word-bytes)
                   (ea data-offset buffer buffer-index)))
             (ea-for-block (&optional (offset 0))
               (ea (+ block-disp offset) block block-index n-word-bytes))
             (handle-endianness (x)
               (when big-endian-p
                 (inst bswap x)
                 #+x86-64
                 (unless 64-bit-p
                   (inst rol x 32)))))
        (unless immediate-offset
          (move buffer-index offset))
        (inst mov block-index (- (truncate bytes-to-copy n-word-bytes)))
        (emit-label loop)
        (inst mov temp (ea-for-buffer 0))
        (when immediate-offset
          (inst mov buffer-index (ea-for-buffer n-word-bytes)))
        (handle-endianness temp)
        (when immediate-offset
          (handle-endianness buffer-index))
        (inst mov (ea-for-block) temp)
        (when immediate-offset
          (inst mov (ea-for-block n-word-bytes) buffer-index))
        (unless immediate-offset
          (inst add buffer-index n-word-bytes))
        (inst add block-index unroll)
        (inst jmp :nz loop)))))

(define-vop (fill-block-ub8-le fill-block-ub8)
  (:translate ironclad::fill-block-ub8-le)
  (:arg-types simple-array-unsigned-byte-32
              simple-array-unsigned-byte-8
              positive-fixnum)
  (:variant nil 64 nil))

(define-vop (fill-block-ub8-be fill-block-ub8)
  (:translate ironclad::fill-block-ub8-be)
  (:arg-types simple-array-unsigned-byte-32
              simple-array-unsigned-byte-8
              positive-fixnum)
  (:variant t 64 nil))

#+x86-64
(define-vop (fill-block-ub8-le/64 fill-block-ub8)
  (:translate ironclad::fill-block-ub8-le/64)
  (:arg-types simple-array-unsigned-byte-64
              simple-array-unsigned-byte-8
              positive-fixnum)
  ;; Yes.  Really.  Inconsistent naming FTW.
  (:variant nil 64 t))

#+x86-64
(define-vop (fill-block-ub8-be/64 fill-block-ub8)
  (:translate ironclad::fill-block-ub8-be/64)
  (:arg-types simple-array-unsigned-byte-64
              simple-array-unsigned-byte-8
              positive-fixnum)
  (:variant t 128 t))

(define-vop (expand-block)
  (:translate ironclad::expand-block)
  (:policy :fast-safe)
  (:args (block :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-32)
  (:temporary (:sc unsigned-reg) temp count)
  (:generator 100
    (flet ((block-word (elem-offset)
             (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                               other-pointer-lowtag)
                            (* 4 elem-offset))))
               (dword-ea disp block count 4))))
      (let ((loop (gen-label)))
        (inst mov count 16)
        (emit-label loop)
        (dword-inst mov temp (block-word -3))
        (dword-inst xor temp (block-word -8))
        (dword-inst xor temp (block-word -14))
        (dword-inst xor temp (block-word -16))
        (dword-inst rol temp 1)
        (dword-inst mov (block-word 0) temp)
        (inst add count 1)
        (inst cmp count 79)
        (inst jmp :le loop)))))

(define-vop (sha256-expand-block)
  (:translate ironclad::sha256-expand-block)
  (:policy :fast-safe)
  (:args (block :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-32)
  (:temporary (:sc unsigned-reg) t1 t2 t3 t4 count)
  (:generator 100
    (flet ((block-word (elem-offset)
             (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                               other-pointer-lowtag)
                            (* 4 elem-offset))))
               (dword-ea disp block count 4))))
      (let ((loop (gen-label)))
        ;; The code could definitely be improved for x86-64 by using
        ;; more temporaries, but this version works on both 32- and
        ;; 64-bit and eliminates many of the stupidities in the modular
        ;; arithmetic version (mostly on 64-bit, but some lameness in
        ;; the 32-bit version as well).
        (inst mov count 16)
        (emit-label loop)
        (dword-inst mov t1 (block-word -2))
        ;; sigma1
        (dword-inst mov t2 t1)
        (dword-inst rol t2 15)
        (dword-inst mov t3 t1)
        (dword-inst rol t3 13)
        (dword-inst xor t2 t3)
        (dword-inst shr t1 10)
        (dword-inst xor t1 t2)
        (dword-inst mov t2 (block-word -15))
        ;; sigma0
        (dword-inst mov t3 t2)
        (dword-inst rol t3 25)
        (dword-inst mov t4 t2)
        (dword-inst rol t4 14)
        (dword-inst xor t3 t4)
        (dword-inst shr t2 3)
        (dword-inst xor t2 t3)
        (dword-inst add t1 (block-word -7))
        (dword-inst add t2 (block-word -16))
        (dword-inst add t1 t2)
        (dword-inst mov (block-word 0) t1)
        (inst add count 1)
        (inst cmp count 63)
        (inst jmp :le loop)))))

;;; Implementing this for x86 would require nasty hacks with
;;; pseudo-atomic.  Might just be worth it for the speed increase,
;;; though.  The code is also probably not scheduled optimally.
#+x86-64
(define-vop (update-sha1-block)
  (:translate ironclad::%update-sha1-block)
  (:policy :fast-safe)
  (:args (regs :scs (descriptor-reg) :target result)
         (block :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-32 simple-array-unsigned-byte-32)
  (:results (result :scs (descriptor-reg)))
  (:result-types simple-array-unsigned-byte-32)
  (:temporary (:sc unsigned-reg) a b c d e t1 t2)
  (:generator 1000
    (let ((k1 #x5a827999)
          (k2 #x6ed9eba1)
          (k3 #x-70e44324)
          (k4 #x-359d3e2a))
      (labels ((block/reg-ea (base index)
                 (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                                   other-pointer-lowtag)
                                (* 4 index))))
                   (dword-ea disp base)))
               (f1 (a b c d e n)
                 (dword-inst mov t2 a)
                 (dword-inst mov t1 c)
                 (dword-inst rol t2 5)
                 (dword-inst xor t1 d)
                 (dword-inst add t2 (block/reg-ea block n))
                 (dword-inst and t1 b)
                 (dword-inst xor t1 d)
                 (dword-inst lea e (ea k1 t1 e))
                 (dword-inst rol b 30)
                 (dword-inst add e t2))
               (f2/4 (a b c d e n k)
                 (dword-inst mov t2 a)
                 (dword-inst mov t1 d)
                 (dword-inst rol t2 5)
                 (dword-inst xor t1 c)
                 (dword-inst add t2 (block/reg-ea block n))
                 (dword-inst xor t1 b)
                 (dword-inst lea e (ea k t1 e))
                 (dword-inst rol b 30)
                 (dword-inst add e t2))
               (f2 (a b c d e n)
                 (f2/4 a b c d e n k2))
               (f4 (a b c d e n)
                 (f2/4 a b c d e n k4))
               (f3 (a b c d e n)
                 (dword-inst mov t2 c)
                 (dword-inst mov t1 c)
                 (dword-inst and t2 b)
                 (dword-inst or t1 b)
                 (dword-inst and t1 d)
                 (dword-inst or t1 t2)
                 (dword-inst mov t2 a)
                 (dword-inst rol t2 5)
                 (dword-inst add t2 (block/reg-ea block n))
                 (dword-inst rol b 30)
                 (dword-inst lea e (ea k3 t1 e))
                 (dword-inst add e t2))
               (sha1-rounds (start end f)
                 (let ((xvars (ironclad::make-circular-list a b c d e)))
                   (loop for i from start upto end
                         for vars on xvars by #'cddddr
                         do (multiple-value-bind (a b c d e)
                                (apply #'values (ironclad::circular-list-subseq vars 0 5))
                              (funcall f a b c d e i))))))
        (dword-inst mov a (block/reg-ea regs 0))
        (dword-inst mov b (block/reg-ea regs 1))
        (dword-inst mov c (block/reg-ea regs 2))
        (dword-inst mov d (block/reg-ea regs 3))
        (dword-inst mov e (block/reg-ea regs 4))
        (sha1-rounds 0 19 #'f1)
        (sha1-rounds 20 39 #'f2)
        (sha1-rounds 40 59 #'f3)
        (sha1-rounds 60 79 #'f4)
        (dword-inst add (block/reg-ea regs 0) a)
        (dword-inst add (block/reg-ea regs 1) b)
        (dword-inst add (block/reg-ea regs 2) c)
        (dword-inst add (block/reg-ea regs 3) d)
        (dword-inst add (block/reg-ea regs 4) e)
        (move result regs)))))

#+x86-64
(define-vop (salsa-core-fast)
  (:translate ironclad::x-salsa-core)
  (:policy :fast-safe)
  (:args (buffer :scs (descriptor-reg))
         (state :scs (descriptor-reg)))
  (:info n-rounds)
  (:arg-types (:constant (signed-byte 61))
              simple-array-unsigned-byte-8
              simple-array-unsigned-byte-32)
  (:temporary (:sc double-reg) x0 x1 x2 x3)
  (:temporary (:sc unsigned-reg) r0 r1 r2 r3 temp count)
  (:generator 1000
    (labels ((nth-xmm-mem (base i)
               (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                                 other-pointer-lowtag)
                              (* 16 i))))
                 (ea disp base)))
             (nth-buffer-dword (i)
               (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                                 other-pointer-lowtag)
                              (* 4 i))))
                 (dword-ea disp buffer)))
             (quarter-round (y0 y1 y2 y3)
               ;; x[y0] = XOR(x[y0],ROTATE(PLUS(x[y3],x[y2]), 7));
               ;; x[y1] = XOR(x[y1],ROTATE(PLUS(x[y0],x[y3]), 9));
               ;; x[y2] = XOR(x[y2],ROTATE(PLUS(x[y1],x[y0]),13));
               ;; x[y3] = XOR(x[y3],ROTATE(PLUS(x[y2],x[y1]),18));
               (dword-inst mov r2 (nth-buffer-dword y2))
               (dword-inst mov r3 (nth-buffer-dword y3))

               (dword-inst lea r0 (ea 0 r3 r2))
               (dword-inst rol r0 7)
               (dword-inst xor r0 (nth-buffer-dword y0))

               (dword-inst lea r1 (ea 0 r0 r3))
               (dword-inst rol r1 9)
               (dword-inst xor r1 (nth-buffer-dword y1))

               (dword-inst lea temp (ea 0 r1 r0))
               (dword-inst rol temp 13)
               (dword-inst xor r2 temp)

               (dword-inst lea temp (ea 0 r2 r1))
               (dword-inst rol temp 18)
               (dword-inst xor r3 temp)

               (dword-inst mov (nth-buffer-dword y0) r0)
               (dword-inst mov (nth-buffer-dword y1) r1)
               (dword-inst mov (nth-buffer-dword y2) r2)
               (dword-inst mov (nth-buffer-dword y3) r3)))
      ;; copy state to the output buffer
      (inst movdqu x0 (nth-xmm-mem state 0))
      (inst movdqu x1 (nth-xmm-mem state 1))
      (inst movdqu x2 (nth-xmm-mem state 2))
      (inst movdqu x3 (nth-xmm-mem state 3))
      (inst movdqu (nth-xmm-mem buffer 0) x0)
      (inst movdqu (nth-xmm-mem buffer 1) x1)
      (inst movdqu (nth-xmm-mem buffer 2) x2)
      (inst movdqu (nth-xmm-mem buffer 3) x3)

      (let ((repeat (gen-label)))
        (inst mov count n-rounds)
        (emit-label repeat)
        (quarter-round 4 8 12 0)
        (quarter-round 9 13 1 5)
        (quarter-round 14 2 6 10)
        (quarter-round 3 7 11 15)

        (quarter-round 1 2 3 0)
        (quarter-round 6 7 4 5)
        (quarter-round 11 8 9 10)
        (quarter-round 12 13 14 15)
        (inst sub count 1)
        (inst jmp :nz repeat))

      (inst paddd x0 (nth-xmm-mem buffer 0))
      (inst paddd x1 (nth-xmm-mem buffer 1))
      (inst paddd x2 (nth-xmm-mem buffer 2))
      (inst paddd x3 (nth-xmm-mem buffer 3))
      (inst movdqu (nth-xmm-mem buffer 0) x0)
      (inst movdqu (nth-xmm-mem buffer 1) x1)
      (inst movdqu (nth-xmm-mem buffer 2) x2)
      (inst movdqu (nth-xmm-mem buffer 3) x3))))

#+x86-64
(define-vop (chacha-core-fast)
  (:translate ironclad::x-chacha-core)
  (:policy :fast-safe)
  (:args (buffer :scs (descriptor-reg))
         (state :scs (descriptor-reg)))
  (:info n-rounds)
  (:arg-types (:constant (signed-byte 61))
              simple-array-unsigned-byte-8
              simple-array-unsigned-byte-32)
  (:temporary (:sc double-reg) x0 x1 x2 x3 y0 y1 y2 y3 t0)
  (:temporary (:sc unsigned-reg) count)
  (:generator 1000
    (flet ((nth-xmm-mem (base i)
             (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                               other-pointer-lowtag)
                            (* 16 i))))
               (ea disp base)))
           (chacha-round ()
             (inst paddd y0 y1)
             (inst pxor y3 y0)
             (inst movdqa t0 y3)
             (inst pslld-imm y3 16)
             (inst psrld-imm t0 16)
             (inst por y3 t0)

             (inst paddd y2 y3)
             (inst pxor y1 y2)
             (inst movdqa t0 y1)
             (inst pslld-imm y1 12)
             (inst psrld-imm t0 20)
             (inst por y1 t0)

             (inst paddd y0 y1)
             (inst pxor y3 y0)
             (inst movdqa t0 y3)
             (inst pslld-imm y3 8)
             (inst psrld-imm t0 24)
             (inst por y3 t0)

             (inst paddd y2 y3)
             (inst pxor y1 y2)
             (inst movdqa t0 y1)
             (inst pslld-imm y1 7)
             (inst psrld-imm t0 25)
             (inst por y1 t0)))
      (inst movdqu x0 (nth-xmm-mem state 0))
      (inst movdqu x1 (nth-xmm-mem state 1))
      (inst movdqu x2 (nth-xmm-mem state 2))
      (inst movdqu x3 (nth-xmm-mem state 3))
      (inst movdqa y0 x0)
      (inst movdqa y1 x1)
      (inst movdqa y2 x2)
      (inst movdqa y3 x3)

      (let ((repeat (gen-label)))
        (inst mov count n-rounds)
        (emit-label repeat)

        (chacha-round)
        (inst pshufd y1 y1 #b00111001)
        (inst pshufd y2 y2 #b01001110)
        (inst pshufd y3 y3 #b10010011)

        (chacha-round)
        (inst pshufd y1 y1 #b10010011)
        (inst pshufd y2 y2 #b01001110)
        (inst pshufd y3 y3 #b00111001)

        (inst sub count 1)
        (inst jmp :nz repeat))

      (inst paddd x0 y0)
      (inst paddd x1 y1)
      (inst paddd x2 y2)
      (inst paddd x3 y3)
      (inst movdqu (nth-xmm-mem buffer 0) x0)
      (inst movdqu (nth-xmm-mem buffer 1) x1)
      (inst movdqu (nth-xmm-mem buffer 2) x2)
      (inst movdqu (nth-xmm-mem buffer 3) x3))))

#+x86-64
(define-vop (pclmulqdq-support-p)
  (:translate ironclad::pclmulqdq-support-p)
  (:policy :fast-safe)
  (:conditional :c)
  (:temporary (:sc unsigned-reg :offset rax-offset) eax)
  (:temporary (:sc unsigned-reg :offset rcx-offset) ecx)
  (:generator 10
    (inst mov eax 1)
    (inst cpuid)
    (inst bt ecx 1)))

#+x86-64
(define-vop (aes-ni-support-p)
  (:translate ironclad::aes-ni-support-p)
  (:policy :fast-safe)
  (:conditional :c)
  (:temporary (:sc unsigned-reg :offset rax-offset) eax)
  (:temporary (:sc unsigned-reg :offset rcx-offset) ecx)
  (:generator 10
    (inst mov eax 1)
    (inst cpuid)
    (inst bt ecx 25)))

#+x86-64
(define-vop (aes-ni-generate-round-keys)
  (:translate ironclad::aes-ni-generate-round-keys)
  (:policy :fast-safe)
  (:args (key :scs (descriptor-reg))
         (key-length :scs (unsigned-reg))
         (encryption-keys :scs (descriptor-reg))
         (decryption-keys :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-32
              simple-array-unsigned-byte-32)
  (:temporary (:sc double-reg) x0 x1 x2 x3 x4 x5 x6)
  (:generator 1000
    (labels ((buffer-mem (base i)
               (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                                 other-pointer-lowtag)
                              (* 16 i))))
                 (ea disp base)))
             (expand-key-128 ()
               (inst pshufd x1 x1 #b11111111)
               (inst shufps x2 x0 #b00010000)
               (inst pxor x0 x2)
               (inst shufps x2 x0 #b10001100)
               (inst pxor x0 x2)
               (inst pxor x0 x1))
             (expand-key-192a ()
               (inst pshufd x1 x1 #b01010101)
               (inst shufps x2 x0 #b00010000)
               (inst pxor x0 x2)
               (inst shufps x2 x0 #b10001100)
               (inst pxor x0 x2)
               (inst pxor x0 x1)
               (inst movdqa x4 x3)
               (inst movdqa x5 x3)
               (inst pslldq x4 4)
               (inst pshufd x6 x0 #b11111111)
               (inst pxor x3 x6)
               (inst pxor x3 x4)
               (inst movdqa x1 x0)
               (inst shufps x5 x0 #b01000100)
               (inst shufps x1 x3 #b01001110))
             (expand-key-192b ()
               (inst pshufd x1 x1 #b01010101)
               (inst shufps x2 x0 #b00010000)
               (inst pxor x0 x2)
               (inst shufps x2 x0 #b10001100)
               (inst pxor x0 x2)
               (inst pxor x0 x1)
               (inst movdqa x4 x3)
               (inst pslldq x4 4)
               (inst pshufd x5 x0 #b11111111)
               (inst pxor x3 x5)
               (inst pxor x3 x4))
             (expand-key-256a ()
               (expand-key-128))
             (expand-key-256b ()
               (inst pshufd x1 x1 #b10101010)
               (inst shufps x2 x3 #b00010000)
               (inst pxor x3 x2)
               (inst shufps x2 x3 #b10001100)
               (inst pxor x3 x2)
               (inst pxor x3 x1)))
      (let ((key-192 (gen-label))
            (key-128 (gen-label))
            (end (gen-label)))
        (inst pxor x2 x2)
        (inst movdqu x0 (buffer-mem key 0))
        (inst cmp key-length 24)
        (inst jmp :b key-128)
        (inst jmp :z key-192)

        (inst movdqu x3 (buffer-mem key 1))
        (inst movdqu (buffer-mem encryption-keys 0) x0)
        (inst movdqu (buffer-mem decryption-keys 14) x0)
        (inst movdqu (buffer-mem encryption-keys 1) x3)
        (inst aesimc x6 x3)
        (inst movdqu (buffer-mem decryption-keys 13) x6)
        (inst aeskeygenassist x1 x3 1)
        (expand-key-256a)
        (inst movdqu (buffer-mem encryption-keys 2) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 12) x6)
        (inst aeskeygenassist x1 x0 1)
        (expand-key-256b)
        (inst movdqu (buffer-mem encryption-keys 3) x3)
        (inst aesimc x6 x3)
        (inst movdqu (buffer-mem decryption-keys 11) x6)
        (inst aeskeygenassist x1 x3 2)
        (expand-key-256a)
        (inst movdqu (buffer-mem encryption-keys 4) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 10) x6)
        (inst aeskeygenassist x1 x0 2)
        (expand-key-256b)
        (inst movdqu (buffer-mem encryption-keys 5) x3)
        (inst aesimc x6 x3)
        (inst movdqu (buffer-mem decryption-keys 9) x6)
        (inst aeskeygenassist x1 x3 4)
        (expand-key-256a)
        (inst movdqu (buffer-mem encryption-keys 6) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 8) x6)
        (inst aeskeygenassist x1 x0 4)
        (expand-key-256b)
        (inst movdqu (buffer-mem encryption-keys 7) x3)
        (inst aesimc x6 x3)
        (inst movdqu (buffer-mem decryption-keys 7) x6)
        (inst aeskeygenassist x1 x3 8)
        (expand-key-256a)
        (inst movdqu (buffer-mem encryption-keys 8) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 6) x6)
        (inst aeskeygenassist x1 x0 8)
        (expand-key-256b)
        (inst movdqu (buffer-mem encryption-keys 9) x3)
        (inst aesimc x6 x3)
        (inst movdqu (buffer-mem decryption-keys 5) x6)
        (inst aeskeygenassist x1 x3 16)
        (expand-key-256a)
        (inst movdqu (buffer-mem encryption-keys 10) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 4) x6)
        (inst aeskeygenassist x1 x0 16)
        (expand-key-256b)
        (inst movdqu (buffer-mem encryption-keys 11) x3)
        (inst aesimc x6 x3)
        (inst movdqu (buffer-mem decryption-keys 3) x6)
        (inst aeskeygenassist x1 x3 32)
        (expand-key-256a)
        (inst movdqu (buffer-mem encryption-keys 12) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 2) x6)
        (inst aeskeygenassist x1 x0 32)
        (expand-key-256b)
        (inst movdqu (buffer-mem encryption-keys 13) x3)
        (inst aesimc x6 x3)
        (inst movdqu (buffer-mem decryption-keys 1) x6)
        (inst aeskeygenassist x1 x3 64)
        (expand-key-256a)
        (inst movdqu (buffer-mem encryption-keys 14) x0)
        (inst movdqu (buffer-mem decryption-keys 0) x0)
        (inst jmp end)

        (emit-label key-192)
        (inst movdqu x3 (buffer-mem key 1))
        (inst movdqu (buffer-mem encryption-keys 0) x0)
        (inst movdqu (buffer-mem decryption-keys 12) x0)
        (inst aeskeygenassist x1 x3 1)
        (expand-key-192a)
        (inst movdqu (buffer-mem encryption-keys 1) x5)
        (inst aesimc x6 x5)
        (inst movdqu (buffer-mem decryption-keys 11) x6)
        (inst movdqu (buffer-mem encryption-keys 2) x1)
        (inst aesimc x6 x1)
        (inst movdqu (buffer-mem decryption-keys 10) x6)
        (inst aeskeygenassist x1 x3 2)
        (expand-key-192b)
        (inst movdqu (buffer-mem encryption-keys 3) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 9) x6)
        (inst aeskeygenassist x1 x3 4)
        (expand-key-192a)
        (inst movdqu (buffer-mem encryption-keys 4) x5)
        (inst aesimc x6 x5)
        (inst movdqu (buffer-mem decryption-keys 8) x6)
        (inst movdqu (buffer-mem encryption-keys 5) x1)
        (inst aesimc x6 x1)
        (inst movdqu (buffer-mem decryption-keys 7) x6)
        (inst aeskeygenassist x1 x3 8)
        (expand-key-192b)
        (inst movdqu (buffer-mem encryption-keys 6) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 6) x6)
        (inst aeskeygenassist x1 x3 16)
        (expand-key-192a)
        (inst movdqu (buffer-mem encryption-keys 7) x5)
        (inst aesimc x6 x5)
        (inst movdqu (buffer-mem decryption-keys 5) x6)
        (inst movdqu (buffer-mem encryption-keys 8) x1)
        (inst aesimc x6 x1)
        (inst movdqu (buffer-mem decryption-keys 4) x6)
        (inst aeskeygenassist x1 x3 32)
        (expand-key-192b)
        (inst movdqu (buffer-mem encryption-keys 9) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 3) x6)
        (inst aeskeygenassist x1 x3 64)
        (expand-key-192a)
        (inst movdqu (buffer-mem encryption-keys 10) x5)
        (inst aesimc x6 x5)
        (inst movdqu (buffer-mem decryption-keys 2) x6)
        (inst movdqu (buffer-mem encryption-keys 11) x1)
        (inst aesimc x6 x1)
        (inst movdqu (buffer-mem decryption-keys 1) x6)
        (inst aeskeygenassist x1 x3 128)
        (expand-key-192b)
        (inst movdqu (buffer-mem encryption-keys 12) x0)
        (inst movdqu (buffer-mem decryption-keys 0) x0)
        (inst jmp end)

        (emit-label key-128)
        (inst movdqu (buffer-mem encryption-keys 0) x0)
        (inst movdqu (buffer-mem decryption-keys 10) x0)
        (inst aeskeygenassist x1 x0 1)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 1) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 9) x6)
        (inst aeskeygenassist x1 x0 2)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 2) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 8) x6)
        (inst aeskeygenassist x1 x0 4)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 3) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 7) x6)
        (inst aeskeygenassist x1 x0 8)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 4) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 6) x6)
        (inst aeskeygenassist x1 x0 16)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 5) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 5) x6)
        (inst aeskeygenassist x1 x0 32)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 6) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 4) x6)
        (inst aeskeygenassist x1 x0 64)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 7) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 3) x6)
        (inst aeskeygenassist x1 x0 128)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 8) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 2) x6)
        (inst aeskeygenassist x1 x0 27)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 9) x0)
        (inst aesimc x6 x0)
        (inst movdqu (buffer-mem decryption-keys 1) x6)
        (inst aeskeygenassist x1 x0 54)
        (expand-key-128)
        (inst movdqu (buffer-mem encryption-keys 10) x0)
        (inst movdqu (buffer-mem decryption-keys 0) x0)
        (emit-label end)))))

#+x86-64
(define-vop (aes-ni-encrypt)
  (:translate ironclad::aes-ni-encrypt)
  (:policy :fast-safe)
  (:args (plaintext :scs (descriptor-reg))
         (plaintext-start :scs (unsigned-reg))
         (ciphertext :scs (descriptor-reg))
         (ciphertext-start :scs (unsigned-reg))
         (round-keys :scs (descriptor-reg))
         (n-rounds :scs (unsigned-reg)))
  (:arg-types simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-32
              unsigned-num)
  (:temporary (:sc double-reg) x0 x1)
  (:generator 1000
    (flet ((buffer-mem (base offset)
             (let ((disp (- (* n-word-bytes vector-data-offset)
                            other-pointer-lowtag)))
               (ea disp base offset)))
           (round-key (i)
             (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                               other-pointer-lowtag)
                            (* 16 i))))
               (ea disp round-keys))))
      (let ((last-round (gen-label)))
        (inst movdqu x0 (buffer-mem plaintext plaintext-start))
        (inst movdqu x1 (round-key 0))
        (inst pxor x0 x1)
        (inst movdqu x1 (round-key 1))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 2))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 3))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 4))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 5))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 6))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 7))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 8))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 9))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 10))
        (inst cmp n-rounds 10)
        (inst jmp :z last-round)
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 11))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 12))
        (inst cmp n-rounds 12)
        (inst jmp :z last-round)
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 13))
        (inst aesenc x0 x1)
        (inst movdqu x1 (round-key 14))
        (emit-label last-round)
        (inst aesenclast x0 x1)
        (inst movdqu (buffer-mem ciphertext ciphertext-start) x0)))))

#+x86-64
(define-vop (aes-ni-decrypt)
  (:translate ironclad::aes-ni-decrypt)
  (:policy :fast-safe)
  (:args (ciphertext :scs (descriptor-reg))
         (ciphertext-start :scs (unsigned-reg))
         (plaintext :scs (descriptor-reg))
         (plaintext-start :scs (unsigned-reg))
         (round-keys :scs (descriptor-reg))
         (n-rounds :scs (unsigned-reg)))
  (:arg-types simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-32
              unsigned-num)
  (:temporary (:sc double-reg) x0 x1)
  (:generator 1000
    (flet ((buffer-mem (base offset)
             (let ((disp (- (* n-word-bytes vector-data-offset)
                            other-pointer-lowtag)))
               (ea disp base offset)))
           (round-key (i)
             (let ((disp (+ (- (* n-word-bytes vector-data-offset)
                               other-pointer-lowtag)
                            (* 16 i))))
               (ea disp round-keys))))
      (let ((last-round (gen-label)))
        (inst movdqu x0 (buffer-mem ciphertext ciphertext-start))
        (inst movdqu x1 (round-key 0))
        (inst pxor x0 x1)
        (inst movdqu x1 (round-key 1))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 2))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 3))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 4))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 5))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 6))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 7))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 8))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 9))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 10))
        (inst cmp n-rounds 10)
        (inst jmp :z last-round)
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 11))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 12))
        (inst cmp n-rounds 12)
        (inst jmp :z last-round)
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 13))
        (inst aesdec x0 x1)
        (inst movdqu x1 (round-key 14))
        (emit-label last-round)
        (inst aesdeclast x0 x1)
        (inst movdqu (buffer-mem plaintext plaintext-start) x0)))))

#+x86-64
(define-vop (fast-blake2s-mixing)
  (:translate ironclad::fast-blake2s-mixing)
  (:policy :fast-safe)
  (:args (va :scs (unsigned-reg) :target ra)
         (vb :scs (unsigned-reg) :target rb)
         (vc :scs (unsigned-reg) :target rc)
         (vd :scs (unsigned-reg) :target rd)
         (x :scs (unsigned-reg))
         (y :scs (unsigned-reg)))
  (:arg-types unsigned-num
              unsigned-num
              unsigned-num
              unsigned-num
              unsigned-num
              unsigned-num)
  (:results (ra :scs (unsigned-reg) :from (:argument 0))
            (rb :scs (unsigned-reg) :from (:argument 1))
            (rc :scs (unsigned-reg) :from (:argument 2))
            (rd :scs (unsigned-reg) :from (:argument 3)))
  (:result-types unsigned-num
                 unsigned-num
                 unsigned-num
                 unsigned-num)
  (:generator 1000
    (dword-inst mov ra va)
    (dword-inst mov rb vb)
    (dword-inst mov rc vc)
    (dword-inst mov rd vd)
    (dword-inst add ra rb)
    (dword-inst add ra x)
    (dword-inst xor rd ra)
    (dword-inst ror rd 16)
    (dword-inst add rc rd)
    (dword-inst xor rb rc)
    (dword-inst ror rb 12)
    (dword-inst add ra rb)
    (dword-inst add ra y)
    (dword-inst xor rd ra)
    (dword-inst ror rd 8)
    (dword-inst add rc rd)
    (dword-inst xor rb rc)
    (dword-inst ror rb 7)))

#+x86-64
(define-vop (gmac-mul-fast)
  (:translate ironclad::gmac-mul-fast)
  (:policy :fast-safe)
  (:args (accumulator :scs (descriptor-reg))
         (key :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-8
              simple-array-unsigned-byte-8)
  (:temporary (:sc double-reg) x0 x1 x2 x3 x4 x5 x6 x7 x8 x9)
  (:generator 1000
    (flet ((buffer-mem (base)
             (let ((disp (- (* n-word-bytes vector-data-offset)
                            other-pointer-lowtag)))
               (ea disp base))))
      (inst movdqu x0 (buffer-mem accumulator))
      (inst movdqu x1 (buffer-mem key))
      (inst movdqa x3 x0)
      (inst pclmulqdq x3 x1 0)
      (inst movdqa x4 x0)
      (inst pclmulqdq x4 x1 16)
      (inst movdqa x5 x0)
      (inst pclmulqdq x5 x1 1)
      (inst movdqa x6 x0)
      (inst pclmulqdq x6 x1 17)
      (inst pxor x4 x5)
      (inst movdqa x5 x4)
      (inst psrldq x4 8)
      (inst pslldq x5 8)
      (inst pxor x3 x5)
      (inst pxor x6 x4)
      (inst movdqa x7 x3)
      (inst movdqa x8 x6)
      (inst pslld-imm x3 1)
      (inst pslld-imm x6 1)
      (inst psrld-imm x7 31)
      (inst psrld-imm x8 31)
      (inst movdqa x9 x7)
      (inst pslldq x8 4)
      (inst pslldq x7 4)
      (inst psrldq x9 12)
      (inst por x3 x7)
      (inst por x6 x8)
      (inst por x6 x9)
      (inst movdqa x7 x3)
      (inst movdqa x8 x3)
      (inst movdqa x9 x3)
      (inst pslld-imm x7 31)
      (inst pslld-imm x8 30)
      (inst pslld-imm x9 25)
      (inst pxor x7 x8)
      (inst pxor x7 x9)
      (inst movdqa x8 x7)
      (inst pslldq x7 12)
      (inst psrldq x8 4)
      (inst pxor x3 x7)
      (inst movdqa x2 x3)
      (inst movdqa x4 x3)
      (inst movdqa x5 x3)
      (inst psrld-imm x2 1)
      (inst psrld-imm x4 2)
      (inst psrld-imm x5 7)
      (inst pxor x2 x4)
      (inst pxor x2 x5)
      (inst pxor x2 x8)
      (inst pxor x3 x2)
      (inst pxor x6 x3)
      (inst movdqu (buffer-mem accumulator) x6))))

#+x86-64
(define-vop (xor128)
  (:translate ironclad::xor128)
  (:policy :fast-safe)
  (:args (in1 :scs (descriptor-reg))
         (start-in1 :scs (unsigned-reg))
         (in2 :scs (descriptor-reg))
         (start-in2 :scs (unsigned-reg))
         (out :scs (descriptor-reg))
         (start-out :scs (unsigned-reg)))
  (:arg-types simple-array-unsigned-byte-8
              positive-fixnum
              simple-array-unsigned-byte-8
              positive-fixnum
              simple-array-unsigned-byte-8
              positive-fixnum)
  (:temporary (:sc double-reg) x0 x1)
  (:generator 1000
    (flet ((buffer-mem (base offset)
             (let ((disp (- (* n-word-bytes vector-data-offset)
                            other-pointer-lowtag)))
               (ea disp base offset))))
      (inst movdqu x0 (buffer-mem in1 start-in1))
      (inst movdqu x1 (buffer-mem in2 start-in2))
      (inst pxor x0 x1)
      (inst movdqu (buffer-mem out start-out) x0))))

#+x86-64
(define-vop (mov128)
  (:translate ironclad::mov128)
  (:policy :fast-safe)
  (:args (in :scs (descriptor-reg))
         (start-in :scs (unsigned-reg))
         (out :scs (descriptor-reg))
         (start-out :scs (unsigned-reg)))
  (:arg-types simple-array-unsigned-byte-8
              positive-fixnum
              simple-array-unsigned-byte-8
              positive-fixnum)
  (:temporary (:sc double-reg) x0)
  (:generator 1000
    (flet ((buffer-mem (base offset)
             (let ((disp (- (* n-word-bytes vector-data-offset)
                            other-pointer-lowtag)))
               (ea disp base offset))))
      (inst movdqu x0 (buffer-mem in start-in))
      (inst movdqu (buffer-mem out start-out) x0))))

(define-vop (inc-counter-block)
  (:translate ironclad::inc-counter-block)
  (:policy :fast-safe)
  (:args (size :scs (unsigned-reg) :target idx)
         (counter :scs (descriptor-reg)))
  (:arg-types positive-fixnum
              simple-array-unsigned-byte-8)
  (:temporary (:sc unsigned-reg) idx)
  (:generator 1000
    (let ((start (gen-label))
          (end (gen-label)))
      (move idx size)
      (inst stc)
      (emit-label start)
      #.(let ((disp '(- (* n-word-bytes vector-data-offset)
                        other-pointer-lowtag 1)))
          (if (and (member :x86-64 *features*) (fboundp 'sb-vm::ea))
              `(inst adc :byte (ea ,disp counter idx) 0)
              `(inst adc (sb-vm::make-ea :byte :base counter :index idx :disp ,disp) 0)))
      (inst jmp :nc end)
      (inst dec idx)
      (inst jmp :nz start)
      (emit-label end))))
) ; PROGN
