// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_MEDIA_ROUTER_PROVIDERS_DIAL_DIAL_MEDIA_ROUTE_PROVIDER_H_
#define CHROME_BROWSER_MEDIA_ROUTER_PROVIDERS_DIAL_DIAL_MEDIA_ROUTE_PROVIDER_H_

#include <memory>
#include <string>
#include <vector>

#include "base/containers/flat_map.h"
#include "base/containers/flat_set.h"
#include "base/gtest_prod_util.h"
#include "base/macros.h"
#include "base/memory/weak_ptr.h"
#include "base/sequence_checker.h"
#include "chrome/browser/media/router/discovery/dial/dial_media_sink_service_impl.h"
#include "chrome/browser/media/router/providers/common/buffered_message_sender.h"
#include "chrome/browser/media/router/providers/dial/dial_activity_manager.h"
#include "chrome/browser/media/router/providers/dial/dial_internal_message_util.h"
#include "chrome/common/media_router/mojom/media_router.mojom.h"
#include "mojo/public/cpp/bindings/binding.h"

namespace url {
class Origin;
}

namespace media_router {

class DataDecoder;

// MediaRouteProvider for DIAL sinks.
// DialMediaRouteProvider supports custom DIAL launch, which is a
// way for websites that uses Cast SDK to launch apps on DIAL devices.
// The life of a custom DIAL launch workflow is as follows:
// 1) The user initiates a custom DIAL launch by selecting a DIAL device
//    to cast to. This becomes a CreateRoute request to the
//    DialMediaRouteProvider.
// 2) The DialMediaRouteProvider sends NEW_SESSION / RECEIVER_ACTION messages to
//    the Cast SDK to inform of a new Cast session. In addition, a
//    CUSTOM_DIAL_LAUNCH request is sent to the Cast SDK.
// 3) The Cast SDK sends back a CUSTOM_DIAL_LAUNCH response. Depending on the
//    response, either the page have already handled the app launch, or the
//    DialMediaRouteProvider will initiate the app launch on the device.
// 4) Once the app is launched, the workflow is complete. The webpage will then
//    communicate with the app on the device via its own mechanism.
class DialMediaRouteProvider : public mojom::MediaRouteProvider,
                               public MediaSinkServiceBase::Observer {
 public:
  // |request|: Request to bind to |this|.
  // |media_router|: Pointer to MediaRouter.
  // |media_sink_service|: DIAL MediaSinkService providing information on sinks.
  // |connector|: Connector object for accessing data_decoder services.
  // |hash_token|: A per-profile value used to hash sink IDs.
  // |task_runner|: The task runner on which |this| runs.
  DialMediaRouteProvider(
      mojom::MediaRouteProviderRequest request,
      mojom::MediaRouterPtrInfo media_router,
      DialMediaSinkServiceImpl* media_sink_service,
      service_manager::Connector* connector,
      const std::string& hash_token,
      const scoped_refptr<base::SequencedTaskRunner>& task_runner);
  ~DialMediaRouteProvider() override;

  // mojom::MediaRouteProvider:
  void CreateRoute(const std::string& media_source,
                   const std::string& sink_id,
                   const std::string& presentation_id,
                   const url::Origin& origin,
                   int32_t tab_id,
                   base::TimeDelta timeout,
                   bool incognito,
                   CreateRouteCallback callback) override;
  void JoinRoute(const std::string& media_source,
                 const std::string& presentation_id,
                 const url::Origin& origin,
                 int32_t tab_id,
                 base::TimeDelta timeout,
                 bool incognito,
                 JoinRouteCallback callback) override;
  void ConnectRouteByRouteId(const std::string& media_source,
                             const std::string& route_id,
                             const std::string& presentation_id,
                             const url::Origin& origin,
                             int32_t tab_id,
                             base::TimeDelta timeout,
                             bool incognito,
                             ConnectRouteByRouteIdCallback callback) override;
  void TerminateRoute(const std::string& route_id,
                      TerminateRouteCallback callback) override;
  void SendRouteMessage(const std::string& media_route_id,
                        const std::string& message) override;
  void SendRouteBinaryMessage(const std::string& media_route_id,
                              const std::vector<uint8_t>& data) override;
  void StartObservingMediaSinks(const std::string& media_source) override;
  void StopObservingMediaSinks(const std::string& media_source) override;
  void StartObservingMediaRoutes(const std::string& media_source) override;
  void StopObservingMediaRoutes(const std::string& media_source) override;
  void StartListeningForRouteMessages(const std::string& route_id) override;
  void StopListeningForRouteMessages(const std::string& route_id) override;
  void DetachRoute(const std::string& route_id) override;
  void EnableMdnsDiscovery() override;
  void UpdateMediaSinks(const std::string& media_source) override;
  void SearchSinks(const std::string& sink_id,
                   const std::string& media_source,
                   mojom::SinkSearchCriteriaPtr search_criteria,
                   SearchSinksCallback callback) override;
  void ProvideSinks(
      const std::string& provider_name,
      const std::vector<media_router::MediaSinkInternal>& sinks) override;
  void CreateMediaRouteController(
      const std::string& route_id,
      mojom::MediaControllerRequest media_controller,
      mojom::MediaStatusObserverPtr observer,
      CreateMediaRouteControllerCallback callback) override;

  void SetActivityManagerForTest(
      std::unique_ptr<DialActivityManager> activity_manager);

 private:
  FRIEND_TEST_ALL_PREFIXES(DialMediaRouteProviderTest, AddRemoveSinkQuery);
  FRIEND_TEST_ALL_PREFIXES(DialMediaRouteProviderTest,
                           AddSinkQuerySameMediaSource);
  FRIEND_TEST_ALL_PREFIXES(DialMediaRouteProviderTest,
                           AddSinkQuerySameAppDifferentMediaSources);
  FRIEND_TEST_ALL_PREFIXES(DialMediaRouteProviderTest,
                           AddSinkQueryDifferentApps);
  FRIEND_TEST_ALL_PREFIXES(DialMediaRouteProviderTest, ListenForRouteMessages);

  struct MediaSinkQuery {
    MediaSinkQuery();
    ~MediaSinkQuery();

    // Set of registered media sources for current sink query.
    base::flat_set<MediaSource> media_sources;
    DialMediaSinkServiceImpl::SinkQueryByAppSubscription subscription;

    DISALLOW_COPY_AND_ASSIGN(MediaSinkQuery);
  };

  // MediaSinkServiceBase::Observer:
  void OnSinksDiscovered(const std::vector<MediaSinkInternal>& sinks) override;

  // Binds the message pipes |request| and |media_router| to |this|.
  void Init(mojom::MediaRouteProviderRequest request,
            mojom::MediaRouterPtrInfo media_router);

  void OnAvailableSinksUpdated(const std::string& app_name);

  void NotifyOnSinksReceived(const MediaSource::Id& source_id,
                             const std::vector<MediaSinkInternal>& sinks,
                             const std::vector<url::Origin>& origins);

  void HandleParsedRouteMessage(const MediaRoute::Id& route_id,
                                base::Value message);
  void HandleClientConnect(const DialActivity& activity,
                           const MediaSinkInternal& sink);
  void SendCustomDialLaunchMessage(const MediaRoute::Id& route_id,
                                   const MediaSink::Id& sink_id,
                                   const std::string& app_name,
                                   DialAppInfoResult result);
  void HandleCustomDialLaunchResponse(const DialActivity& activity,
                                      const DialInternalMessage& message);
  void HandleAppLaunchResult(const MediaRoute::Id& route_id, bool success);
  void DoTerminateRoute(const DialActivity& activity,
                        const MediaSinkInternal& sink,
                        TerminateRouteCallback callback);
  void HandleStopAppResult(const MediaRoute::Id& route_id,
                           TerminateRouteCallback callback,
                           const base::Optional<std::string>& message,
                           RouteRequestResult::ResultCode result_code);
  void NotifyAllOnRoutesUpdated();
  void NotifyOnRoutesUpdated(const MediaSource::Id& source_id,
                             const std::vector<MediaRoute>& routes);

  // Returns a list of valid origins for |app_name|. Returns an empty list if
  // all origins are valid.
  std::vector<url::Origin> GetOrigins(const std::string& app_name);

  // Binds |this| to the Mojo request passed into the ctor.
  mojo::Binding<mojom::MediaRouteProvider> binding_;

  // Mojo pointer to the Media Router.
  mojom::MediaRouterPtr media_router_;

  // Non-owned pointer to the DialMediaSinkServiceImpl instance.
  DialMediaSinkServiceImpl* const media_sink_service_;

  // Map of media sink queries, keyed by app name.
  base::flat_map<std::string, std::unique_ptr<MediaSinkQuery>>
      media_sink_queries_;

  // Set of route queries by MediaSource ID.
  base::flat_set<MediaSource::Id> media_route_queries_;

  // Set of pending DIAL launches by sequence number. The max number of pending
  // launches is capped, and oldest entries (smallest number) will be evicted.
  base::flat_set<int> pending_dial_launches_;

  std::unique_ptr<DialActivityManager> activity_manager_;
  std::unique_ptr<BufferedMessageSender> message_sender_;

  // Used for parsing Custom DIAL launch JSON messages.
  std::unique_ptr<DataDecoder> data_decoder_;

  DialInternalMessageUtil internal_message_util_;

  SEQUENCE_CHECKER(sequence_checker_);
  base::WeakPtrFactory<DialMediaRouteProvider> weak_ptr_factory_{this};
  DISALLOW_COPY_AND_ASSIGN(DialMediaRouteProvider);
};

}  // namespace media_router

#endif  // CHROME_BROWSER_MEDIA_ROUTER_PROVIDERS_DIAL_DIAL_MEDIA_ROUTE_PROVIDER_H_
