// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <stddef.h>

#include <string>
#include <vector>

#include "base/base_switches.h"
#include "base/files/file_path.h"
#include "base/path_service.h"
#include "base/strings/stringprintf.h"
#include "base/strings/utf_string_conversions.h"
#include "base/test/scoped_feature_list.h"
#include "base/threading/platform_thread.h"
#include "base/time/time.h"
#include "build/build_config.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/tabs/tab_strip_model.h"
#include "chrome/common/chrome_paths.h"
#include "chrome/common/chrome_switches.h"
#include "chrome/test/base/in_process_browser_test.h"
#include "chrome/test/base/test_launcher_utils.h"
#include "chrome/test/base/ui_test_utils.h"
#include "content/public/test/browser_test_utils.h"
#include "media/base/media_switches.h"
#include "media/base/test_data_util.h"
#include "media/media_buildflags.h"
#include "net/test/embedded_test_server/embedded_test_server.h"
#include "third_party/widevine/cdm/buildflags.h"
#include "url/gurl.h"

#if BUILDFLAG(ENABLE_LIBRARY_CDMS)
#include "chrome/browser/media/library_cdm_test_helper.h"
#include "media/cdm/cdm_paths.h"
#endif

#if defined(OS_ANDROID)
#error This file needs to be updated to run on Android.
#endif

namespace {

const char kClearKey[] = "org.w3.clearkey";
const char kExternalClearKey[] = "org.chromium.externalclearkey";
const char kWidevine[] = "com.widevine.alpha";

const char kAudioWebMMimeType[] = "audio/webm";
const char kVideoWebMMimeType[] = "video/webm";
const char kAudioMP4MimeType[] = "audio/mp4";
const char kVideoMP4MimeType[] = "video/mp4";

// These are the expected titles set by checkKeySystemWithMediaMimeType()
// in test_key_system_instantiation.html. Other titles are possible, but
// they are unexpected and will be logged with the failure. kUnsupportedResult
// is the actual message generated by navigator.requestMediaKeySystemAccess(),
// and will need to be updated if that code is modified.
const char kSuccessResult[] = "success";
const char kUnsupportedResult[] =
    "Unsupported keySystem or supportedConfigurations.";
const char kUnexpectedResult[] = "unexpected result";
const char kTypeErrorResult[] = "TypeError";

#define EXPECT_SUCCESS(test) EXPECT_EQ(kSuccessResult, test)
#define EXPECT_UNSUPPORTED(test) EXPECT_EQ(kUnsupportedResult, test)
#define EXPECT_TYPEERROR(test) EXPECT_EQ(kTypeErrorResult, test)

#if BUILDFLAG(ENABLE_AV1_DECODER)
#define EXPECT_AV1 EXPECT_SUCCESS
#else
#define EXPECT_AV1 EXPECT_UNSUPPORTED
#endif

#if BUILDFLAG(USE_PROPRIETARY_CODECS)
#define EXPECT_PROPRIETARY EXPECT_SUCCESS
#else
#define EXPECT_PROPRIETARY EXPECT_UNSUPPORTED
#endif

// Expectations for External Clear Key.
#if BUILDFLAG(ENABLE_LIBRARY_CDMS)
#define EXPECT_ECK EXPECT_SUCCESS
#define EXPECT_ECK_AV1 EXPECT_AV1
#define EXPECT_ECK_PROPRIETARY EXPECT_PROPRIETARY
#else
#define EXPECT_ECK EXPECT_UNSUPPORTED
#define EXPECT_ECK_AV1 EXPECT_UNSUPPORTED
#define EXPECT_ECK_PROPRIETARY EXPECT_UNSUPPORTED
#endif  // BUILDFLAG(ENABLE_LIBRARY_CDMS)

// Expectations for Widevine.
#if BUILDFLAG(BUNDLE_WIDEVINE_CDM)
#define EXPECT_WV_SUCCESS EXPECT_SUCCESS
#define EXPECT_WV_AV1 EXPECT_AV1
#define EXPECT_WV_PROPRIETARY EXPECT_PROPRIETARY
#else
#define EXPECT_WV_SUCCESS EXPECT_UNSUPPORTED
#define EXPECT_WV_AV1 EXPECT_UNSUPPORTED
#define EXPECT_WV_PROPRIETARY EXPECT_UNSUPPORTED
#endif  // BUILDFLAG(BUNDLE_WIDEVINE_CDM)

}  // namespace

class EncryptedMediaSupportedTypesTest : public InProcessBrowserTest {
 protected:
  EncryptedMediaSupportedTypesTest() {
    audio_webm_codecs_.push_back("opus");
    audio_webm_codecs_.push_back("vorbis");

    video_webm_codecs_.push_back("vp8");
    video_webm_codecs_.push_back("vp8.0");
    video_webm_codecs_.push_back("vp9");    // Legacy VP9 only supported in WebM
    video_webm_codecs_.push_back("vp9.0");  // Legacy VP9 only supported in WebM

    audio_mp4_codecs_.push_back("mp4a.40.2");

    audio_mp4_flac_codecs_.push_back("flac");

    video_mp4_codecs_.push_back("avc1.42001E");  // Baseline profile.
    video_mp4_codecs_.push_back("avc1.4D000C");  // Main profile.
    video_mp4_codecs_.push_back("avc3.64001F");  // High profile.

    video_mp4_hi10p_codecs_.push_back("avc1.6E001E");  // Hi10P profile

#if BUILDFLAG(ENABLE_HEVC_DEMUXING)
    video_mp4_codecs_.push_back("hvc1.1.6.L93.B0");
    video_mp4_codecs_.push_back("hev1.1.6.L93.B0");
#else
    invalid_codecs_.push_back("hvc1.1.6.L93.B0");
    invalid_codecs_.push_back("hev1.1.6.L93.B0");
#endif

    // New multi-part VP9 codecs are allowed in both MP4 and WebM.
    // For vp9 codec string format, see https://www.webmproject.org/vp9/mp4/
    vp9_profile0_codecs_.push_back("vp09.00.10.08");

    // VP9 profile 2 is supported in WebM/MP4 with ClearKey/ExternalClearKey.
    // TODO(crbug.com/707128): Add support in Widevine CDM.
    vp9_profile2_codecs_.push_back("vp09.02.10.10");

    // AV1 codec string: https://aomediacodec.github.io/av1-isobmff/#codecsparam
    av1_codecs_.push_back("av01.0.04M.08");        // 8 bit
    av1_codecs_.push_back("av01.0.00M.10.0.112");  // 10 bit

    // Extended codecs are used, so make sure generic ones fail. These will be
    // tested against all init data types as they should always fail to be
    // supported.
    invalid_codecs_.push_back("avc1");
    invalid_codecs_.push_back("avc1.");
    invalid_codecs_.push_back("avc3");

    // Other invalid codecs.
    invalid_codecs_.push_back("vp8.1");
    invalid_codecs_.push_back("mp4a");
    invalid_codecs_.push_back("avc2");
    invalid_codecs_.push_back("foo");

    // We only support proper long-form HEVC codec ids.
    invalid_codecs_.push_back("hev1");
    invalid_codecs_.push_back("hev1.");
    invalid_codecs_.push_back("hvc1");
    invalid_codecs_.push_back("hvc1.");

    // Profile 1 and 3 are not supported by EME. See https://crbug.com/898298.
    invalid_codecs_.push_back("vp09.01.10.08");
    invalid_codecs_.push_back("vp09.03.10.10");
  }

  typedef std::vector<std::string> CodecVector;

  const CodecVector& no_codecs() const { return no_codecs_; }
  const CodecVector& audio_webm_codecs() const { return audio_webm_codecs_; }
  const CodecVector& video_webm_codecs() const { return video_webm_codecs_; }
  const CodecVector& audio_mp4_codecs() const { return audio_mp4_codecs_; }
  const CodecVector& audio_mp4_flac_codecs() const {
    return audio_mp4_flac_codecs_;
  }
  const CodecVector& video_mp4_codecs() const { return video_mp4_codecs_; }
  const CodecVector& video_mp4_hi10p_codecs() const {
    return video_mp4_hi10p_codecs_;
  }
  const CodecVector& vp9_profile0_codecs() const {
    return vp9_profile0_codecs_;
  }
  const CodecVector& vp9_profile2_codecs() const {
    return vp9_profile2_codecs_;
  }
  const CodecVector& av1_codecs() const { return av1_codecs_; }
  const CodecVector& invalid_codecs() const { return invalid_codecs_; }

  void SetUpDefaultCommandLine(base::CommandLine* command_line) override {
#if BUILDFLAG(ENABLE_LIBRARY_CDMS)
    base::CommandLine default_command_line(base::CommandLine::NO_PROGRAM);
    InProcessBrowserTest::SetUpDefaultCommandLine(&default_command_line);
    test_launcher_utils::RemoveCommandLineSwitch(
        default_command_line, switches::kDisableComponentUpdate, command_line);
#endif  // BUILDFLAG(ENABLE_LIBRARY_CDMS)
    command_line->AppendSwitchASCII(
        "enable-blink-features",
        "EncryptedMediaEncryptionSchemeQuery,"
        "EncryptedMediaPersistentUsageRecordSession");
  }

  void SetUpOnMainThread() override {
    // Load the test page needed so that checkKeySystemWithMediaMimeType()
    // is available.
    net::EmbeddedTestServer* http_test_server = embedded_test_server();
    http_test_server->ServeFilesFromSourceDirectory(media::GetTestDataPath());
    CHECK(http_test_server->Start());
    GURL gurl = http_test_server->GetURL("/test_key_system_instantiation.html");
    ui_test_utils::NavigateToURL(browser(), gurl);
  }

  // Create a valid JavaScript string for the content type. Format is
  // |mime_type|; codecs="|codec|", where codecs= is omitted if there
  // is no codec.
  static std::string MakeContentType(const std::string& mime_type,
                                     const std::string& codec) {
    std::string content_type(mime_type);
    if (!codec.empty())
      content_type += base::StringPrintf("; codecs=\"%s\"", codec.c_str());
    return content_type;
  }

  // Format: {contentType: |content_type|, encryptionScheme:
  // |encryption_scheme|, robustness: |robustness|}. encryptionScheme and
  // robustness will not be included if |encryption_scheme| or |robustness|
  // is null, respectively.
  static std::string MakeMediaCapability(const std::string& content_type,
                                         const char* robustness,
                                         const char* encryption_scheme) {
    std::string capability =
        base::StringPrintf("{contentType: '%s'", content_type.c_str());
    if (encryption_scheme) {
      base::StringAppendF(&capability, ", encryptionScheme: '%s'",
                          encryption_scheme);
    }
    if (robustness) {
      base::StringAppendF(&capability, ", robustness: '%s'", robustness);
    }
    base::StringAppendF(&capability, "}");
    return capability;
  }

  static std::string MakeMediaCapabilities(const std::string& mime_type,
                                           const CodecVector& codecs,
                                           const char* robustness,
                                           const char* encryption_scheme) {
    std::string capabilities("[");
    if (codecs.empty()) {
      capabilities +=
          MakeMediaCapability(MakeContentType(mime_type, std::string()),
                              robustness, encryption_scheme);
    } else {
      for (auto codec : codecs) {
        capabilities += MakeMediaCapability(MakeContentType(mime_type, codec),
                                            robustness, encryption_scheme) +
                        ",";
      }
      // Remove trailing comma.
      capabilities.erase(capabilities.length() - 1);
    }
    capabilities += ("]");
    return capabilities;
  }

  static std::string ExecuteCommand(content::WebContents* contents,
                                    const std::string& command) {
    content::TitleWatcher title_watcher(contents,
                                        base::ASCIIToUTF16(kSuccessResult));
    title_watcher.AlsoWaitForTitle(base::ASCIIToUTF16(kUnsupportedResult));
    title_watcher.AlsoWaitForTitle(base::ASCIIToUTF16(kUnexpectedResult));
    title_watcher.AlsoWaitForTitle(base::ASCIIToUTF16(kTypeErrorResult));
    EXPECT_TRUE(content::ExecuteScript(contents, command));
    base::string16 result = title_watcher.WaitAndGetTitle();
    return base::UTF16ToASCII(result);
  }

  enum class SessionType {
    kTemporary,             // Temporary session
    kPersistentLicense,     // Persistent license session
    kPersistentUsageRecord  // Persistent usage record session
  };

  std::string GetSessionTypeString(SessionType session_type) {
    switch (session_type) {
      case SessionType::kTemporary:
        return "temporary";
      case SessionType::kPersistentLicense:
        return "persistent-license";
      case SessionType::kPersistentUsageRecord:
        return "persistent-usage-record";
    }

    NOTREACHED();
    return "";
  }

  std::string IsSupportedByKeySystem(
      const std::string& key_system,
      const std::string& mime_type,
      const CodecVector& codecs,
      SessionType session_type = SessionType::kTemporary,
      const char* robustness = nullptr,
      const char* encryption_scheme = nullptr) {
    // Choose the appropriate init data type for the sub type.
    size_t pos = mime_type.find('/');
    DCHECK(pos > 0);
    std::string sub_type(mime_type.substr(pos + 1));
    std::string init_data_type;
    if (sub_type == "mp4") {
      init_data_type = "cenc";
    } else {
      DCHECK(sub_type == "webm");
      init_data_type = "webm";
    }

    bool is_audio = mime_type.compare(0, 5, "audio") == 0;
    DCHECK(is_audio || mime_type.compare(0, 5, "video") == 0);
    auto capabilities =
        MakeMediaCapabilities(mime_type, codecs, robustness, encryption_scheme);
    auto audio_capabilities = is_audio ? capabilities : "null";
    auto video_capabilities = !is_audio ? capabilities : "null";
    auto session_type_string = GetSessionTypeString(session_type);

    std::string command = base::StringPrintf(
        "checkKeySystemWithMediaMimeType('%s', '%s', %s, %s, '%s')",
        key_system.c_str(), init_data_type.c_str(), audio_capabilities.c_str(),
        video_capabilities.c_str(), session_type_string.c_str());
    DVLOG(1) << "command: " << command;

    auto result = ExecuteCommand(
        browser()->tab_strip_model()->GetActiveWebContents(), command);
    DVLOG(1) << "result: " << result;

    return result;
  }

  std::string IsSessionTypeSupported(const std::string& key_system,
                                     SessionType session_type) {
    return IsSupportedByKeySystem(key_system, kVideoWebMMimeType,
                                  video_webm_codecs(), session_type);
  }

  std::string IsAudioRobustnessSupported(const std::string& key_system,
                                         const char* robustness) {
    return IsSupportedByKeySystem(key_system, kAudioWebMMimeType,
                                  audio_webm_codecs(), SessionType::kTemporary,
                                  robustness);
  }

  std::string IsVideoRobustnessSupported(const std::string& key_system,
                                         const char* robustness) {
    return IsSupportedByKeySystem(key_system, kVideoWebMMimeType,
                                  video_webm_codecs(), SessionType::kTemporary,
                                  robustness);
  }

  std::string IsVideoMp4RobustnessSupported(const std::string& key_system,
                                            const char* robustness) {
    return IsSupportedByKeySystem(key_system, kVideoMP4MimeType,
                                  video_mp4_codecs(), SessionType::kTemporary,
                                  robustness);
  }

  std::string IsAudioMp4RobustnessSupported(const std::string& key_system,
                                            const char* robustness) {
    return IsSupportedByKeySystem(key_system, kAudioMP4MimeType,
                                  audio_mp4_codecs(), SessionType::kTemporary,
                                  robustness);
  }

  std::string IsAudioEncryptionSchemeSupported(
      const std::string& key_system,
      const char* encryption_scheme,
      const char* robustness = nullptr) {
    return IsSupportedByKeySystem(key_system, kAudioWebMMimeType,
                                  audio_webm_codecs(), SessionType::kTemporary,
                                  robustness, encryption_scheme);
  }

  std::string IsVideoEncryptionSchemeSupported(
      const std::string& key_system,
      const char* encryption_scheme,
      const char* robustness = nullptr) {
    return IsSupportedByKeySystem(key_system, kVideoWebMMimeType,
                                  video_webm_codecs(), SessionType::kTemporary,
                                  robustness, encryption_scheme);
  }

 private:
  const CodecVector no_codecs_;
  CodecVector audio_webm_codecs_;
  CodecVector video_webm_codecs_;
  CodecVector audio_mp4_codecs_;
  CodecVector audio_mp4_flac_codecs_;
  CodecVector video_mp4_codecs_;
  CodecVector video_mp4_hi10p_codecs_;
  CodecVector vp9_profile0_codecs_;
  CodecVector vp9_profile2_codecs_;
  CodecVector av1_codecs_;
  CodecVector invalid_codecs_;
};

// For ClearKey, nothing additional is required.
class EncryptedMediaSupportedTypesClearKeyTest
    : public EncryptedMediaSupportedTypesTest {};

// For ExternalClearKey tests, ensure that the ClearKey adapter is loaded.
class EncryptedMediaSupportedTypesExternalClearKeyTest
    : public EncryptedMediaSupportedTypesTest {
#if BUILDFLAG(ENABLE_LIBRARY_CDMS)
 protected:
  EncryptedMediaSupportedTypesExternalClearKeyTest() {
    scoped_feature_list_.InitAndEnableFeature(
        media::kExternalClearKeyForTesting);
  }

  ~EncryptedMediaSupportedTypesExternalClearKeyTest() override {}

  void SetUpCommandLine(base::CommandLine* command_line) override {
    EncryptedMediaSupportedTypesTest::SetUpCommandLine(command_line);
    RegisterClearKeyCdm(command_line);
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;
#endif  // BUILDFLAG(ENABLE_LIBRARY_CDMS)

  DISALLOW_COPY_AND_ASSIGN(EncryptedMediaSupportedTypesExternalClearKeyTest);
};

// By default, the External Clear Key (ECK) key system is not supported even if
// present. This test case tests this behavior by not enabling
// kExternalClearKeyForTesting. Even registering the Clear Key CDM does not
// enable the ECK key system support.
class EncryptedMediaSupportedTypesExternalClearKeyNotEnabledTest
    : public EncryptedMediaSupportedTypesTest {
 protected:
  void SetUpCommandLine(base::CommandLine* command_line) override {
    EncryptedMediaSupportedTypesTest::SetUpCommandLine(command_line);
#if BUILDFLAG(ENABLE_LIBRARY_CDMS)
    RegisterClearKeyCdm(command_line);
#endif  // BUILDFLAG(ENABLE_LIBRARY_CDMS)
  }
};

class EncryptedMediaSupportedTypesWidevineTest
    : public EncryptedMediaSupportedTypesTest {
 protected:
  EncryptedMediaSupportedTypesWidevineTest() {
    scoped_feature_list_.InitAndEnableFeature(media::kWidevineAv1);
  }

  void SetUpCommandLine(base::CommandLine* command_line) override {
    EncryptedMediaSupportedTypesTest::SetUpCommandLine(command_line);
    // Persistent license is supported on ChromeOS when protected media
    // identifier is allowed which involves a user action. Use this switch to
    // always allow the identifier for testing purpose. Note that the test page
    // is hosted on "127.0.0.1". See net::EmbeddedTestServer for details.
    command_line->AppendSwitchASCII(
        switches::kUnsafelyAllowProtectedMediaIdentifierForDomain, "127.0.0.1");
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;

  DISALLOW_COPY_AND_ASSIGN(EncryptedMediaSupportedTypesWidevineTest);
};

class EncryptedMediaSupportedTypesWidevineHwSecureTest
    : public EncryptedMediaSupportedTypesWidevineTest {
 protected:
  EncryptedMediaSupportedTypesWidevineHwSecureTest() {
    scoped_feature_list_.InitAndEnableFeature(media::kHardwareSecureDecryption);
  }

  void SetUpCommandLine(base::CommandLine* command_line) override {
    EncryptedMediaSupportedTypesWidevineTest::SetUpCommandLine(command_line);
    // Pretend that we support hardware secure decryption for vp8 and vp9, but
    // not for avc1.
    command_line->AppendSwitchASCII(
        switches::kOverrideHardwareSecureCodecsForTesting, "vp8,vp9");
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;

  DISALLOW_COPY_AND_ASSIGN(EncryptedMediaSupportedTypesWidevineHwSecureTest);
};

#if BUILDFLAG(ENABLE_LIBRARY_CDMS)
// Registers ClearKey CDM with the wrong path (filename).
class EncryptedMediaSupportedTypesClearKeyCdmRegisteredWithWrongPathTest
    : public EncryptedMediaSupportedTypesTest {
 protected:
  EncryptedMediaSupportedTypesClearKeyCdmRegisteredWithWrongPathTest() {
    scoped_feature_list_.InitAndEnableFeature(
        media::kExternalClearKeyForTesting);
  }

  ~EncryptedMediaSupportedTypesClearKeyCdmRegisteredWithWrongPathTest()
      override {}

  void SetUpCommandLine(base::CommandLine* command_line) override {
    EncryptedMediaSupportedTypesTest::SetUpCommandLine(command_line);
    RegisterClearKeyCdm(command_line, true /* use_wrong_cdm_path */);
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;

  DISALLOW_COPY_AND_ASSIGN(
      EncryptedMediaSupportedTypesClearKeyCdmRegisteredWithWrongPathTest);
};
#endif  // BUILDFLAG(ENABLE_LIBRARY_CDMS)

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest, Basic) {
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                        video_webm_codecs()));
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kAudioWebMMimeType,
                                        audio_webm_codecs()));
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType,
                                        audio_mp4_flac_codecs()));
  EXPECT_PROPRIETARY(
      IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType, video_mp4_codecs()));
  EXPECT_PROPRIETARY(
      IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType, audio_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest, NoCodecs) {
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType, no_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kAudioWebMMimeType, no_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType, no_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType, no_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest,
                       InvalidKeySystems) {
  // Case sensitive.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "org.w3.ClEaRkEy", kVideoWebMMimeType, video_webm_codecs()));

  // Prefixed Clear Key key system.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "webkit-org.w3.clearkey", kVideoWebMMimeType, video_webm_codecs()));

  // TLDs are not allowed.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem("org.", kVideoWebMMimeType, video_webm_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem("org", kVideoWebMMimeType, video_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("org.w3.", kVideoWebMMimeType,
                                            video_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("org.w3", kVideoWebMMimeType,
                                            video_webm_codecs()));

  // Incomplete.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "org.w3.clearke", kVideoWebMMimeType, video_webm_codecs()));

  // Extra character.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "org.w3.clearkeyz", kVideoWebMMimeType, video_webm_codecs()));

  // There are no child key systems for Clear Key.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "org.w3.clearkey.foo", kVideoWebMMimeType, video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest, Video_WebM) {
  // Valid video types.
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                        video_webm_codecs()));
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                        vp9_profile0_codecs()));
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                        vp9_profile2_codecs()));
  EXPECT_AV1(
      IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType, av1_codecs()));

  // Non-video WebM codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                            audio_webm_codecs()));

  // Invalid or non-Webm video codecs.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                            audio_mp4_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                            video_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest, Audio_WebM) {
  // Valid audio types.
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kAudioWebMMimeType,
                                        audio_webm_codecs()));

  // Non-audio WebM codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kAudioWebMMimeType,
                                            video_webm_codecs()));

  // Invalid or Non-Webm codecs.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kAudioWebMMimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kAudioWebMMimeType,
                                            audio_mp4_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kAudioWebMMimeType,
                                            audio_mp4_flac_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kAudioWebMMimeType,
                                            video_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest, Video_MP4) {
  // Valid video types.
  EXPECT_PROPRIETARY(
      IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType, video_mp4_codecs()));
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType,
                                        vp9_profile0_codecs()));
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType,
                                        vp9_profile2_codecs()));
  EXPECT_AV1(
      IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType, av1_codecs()));

// High 10-bit Profile is supported when using ClearKey if it is supported for
// clear content on this platform.
#if BUILDFLAG(ENABLE_FFMPEG_VIDEO_DECODERS)
  EXPECT_PROPRIETARY(IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType,
                                            video_mp4_hi10p_codecs()));
#else
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType,
                                            video_mp4_hi10p_codecs()));
#endif

  // Non-video MP4 codecs.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType, audio_mp4_codecs()));

  // Invalid or non-MP4 codecs.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType,
                                            audio_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kVideoMP4MimeType,
                                            video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest, Audio_MP4) {
  // Valid audio types.
  EXPECT_PROPRIETARY(
      IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType, audio_mp4_codecs()));
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType,
                                        audio_mp4_flac_codecs()));

  // Non-audio MP4 codecs.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType, video_mp4_codecs()));

  // Invalid or non-MP4 codec.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType,
                                            audio_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kClearKey, kAudioMP4MimeType,
                                            video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest, SessionType) {
  // Temporary session always supported.
  EXPECT_SUCCESS(IsSessionTypeSupported(kClearKey, SessionType::kTemporary));

  // Persistent sessions not supported by Clear Key key system.
  EXPECT_UNSUPPORTED(
      IsSessionTypeSupported(kClearKey, SessionType::kPersistentLicense));
  EXPECT_UNSUPPORTED(
      IsSessionTypeSupported(kClearKey, SessionType::kPersistentUsageRecord));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest, Robustness) {
  // External Clear Key doesn't require a robustness string.
  EXPECT_SUCCESS(IsVideoRobustnessSupported(kClearKey, nullptr));
  EXPECT_SUCCESS(IsVideoRobustnessSupported(kClearKey, ""));
  EXPECT_SUCCESS(IsAudioRobustnessSupported(kClearKey, nullptr));
  EXPECT_SUCCESS(IsAudioRobustnessSupported(kClearKey, ""));

  // Non-empty robustness string will be rejected, including valid Widevine
  // robustness strings.
  EXPECT_UNSUPPORTED(IsVideoRobustnessSupported(kClearKey, "Invalid String"));
  EXPECT_UNSUPPORTED(IsVideoRobustnessSupported(kClearKey, "SW_SECURE_CRYPTO"));
  EXPECT_UNSUPPORTED(IsAudioRobustnessSupported(kClearKey, "Invalid String"));
  EXPECT_UNSUPPORTED(IsAudioRobustnessSupported(kClearKey, "SW_SECURE_CRYPTO"));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesClearKeyTest,
                       EncryptionScheme) {
  EXPECT_SUCCESS(IsAudioEncryptionSchemeSupported(kClearKey, nullptr));
  EXPECT_SUCCESS(IsAudioEncryptionSchemeSupported(kClearKey, "cenc"));
  EXPECT_SUCCESS(IsAudioEncryptionSchemeSupported(kClearKey, "cbcs"));
  EXPECT_SUCCESS(IsVideoEncryptionSchemeSupported(kClearKey, nullptr));
  EXPECT_SUCCESS(IsVideoEncryptionSchemeSupported(kClearKey, "cenc"));
  EXPECT_SUCCESS(IsVideoEncryptionSchemeSupported(kClearKey, "cbcs"));

  // Invalid encryption schemes will be rejected. However, invalid values
  // generate a TypeError (The provided value '...' is not a valid enum value
  // of type EncryptionScheme), which is not handled by the test page.
  EXPECT_TYPEERROR(IsAudioEncryptionSchemeSupported(kClearKey, "Invalid"));
  EXPECT_TYPEERROR(IsVideoEncryptionSchemeSupported(kClearKey, "Invalid"));
  EXPECT_TYPEERROR(IsAudioEncryptionSchemeSupported(kClearKey, ""));
  EXPECT_TYPEERROR(IsVideoEncryptionSchemeSupported(kClearKey, ""));
}

//
// External Clear Key
//

// When BUILDFLAG(ENABLE_LIBRARY_CDMS), this also tests the Pepper CDM check.
IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       Basic) {
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kVideoWebMMimeType,
                                    video_webm_codecs()));
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kAudioWebMMimeType,
                                    audio_webm_codecs()));
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kAudioMP4MimeType,
                                    audio_mp4_flac_codecs()));
  EXPECT_ECK_PROPRIETARY(IsSupportedByKeySystem(
      kExternalClearKey, kVideoMP4MimeType, video_mp4_codecs()));
  EXPECT_ECK_PROPRIETARY(IsSupportedByKeySystem(
      kExternalClearKey, kAudioMP4MimeType, audio_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       NoCodecs) {
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kExternalClearKey,
                                            kVideoWebMMimeType, no_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kExternalClearKey,
                                            kAudioWebMMimeType, no_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kExternalClearKey,
                                            kVideoMP4MimeType, no_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kExternalClearKey,
                                            kAudioMP4MimeType, no_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       InvalidKeySystems) {
  // Case sensitive.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("org.chromium.ExTeRnAlClEaRkEy",
                                            kVideoWebMMimeType,
                                            video_webm_codecs()));

  // TLDs are not allowed.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem("org.", kVideoWebMMimeType, video_webm_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem("org", kVideoWebMMimeType, video_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("org.chromium", kVideoWebMMimeType,
                                            video_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("org.chromium.", kVideoWebMMimeType,
                                            video_webm_codecs()));

  // Incomplete.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "org.chromium.externalclearke", kVideoWebMMimeType, video_webm_codecs()));

  // Extra character.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("org.chromium.externalclearkeyz",
                                            kVideoWebMMimeType,
                                            video_webm_codecs()));

  // There are no child key systems for External Clear Key.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("org.chromium.externalclearkey.foo",
                                            kVideoWebMMimeType,
                                            video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       Video_WebM) {
  // Valid video types.
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kVideoWebMMimeType,
                                    video_webm_codecs()));
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kVideoWebMMimeType,
                                    vp9_profile0_codecs()));
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kVideoWebMMimeType,
                                    vp9_profile2_codecs()));
  EXPECT_ECK_AV1(IsSupportedByKeySystem(kExternalClearKey, kVideoWebMMimeType,
                                        av1_codecs()));

  // Non-video WebM codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoWebMMimeType, audio_webm_codecs()));

  // Invalid or non-Webm codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoWebMMimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoWebMMimeType, audio_mp4_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoWebMMimeType, video_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       Audio_WebM) {
  // Valid audio types.
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kAudioWebMMimeType,
                                    audio_webm_codecs()));

  // Non-audio WebM codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioWebMMimeType, video_webm_codecs()));

  // Invalid or non-Webm codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioWebMMimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioWebMMimeType, audio_mp4_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioWebMMimeType, audio_mp4_flac_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioWebMMimeType, video_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       Video_MP4) {
  // Valid video types.
  EXPECT_ECK_PROPRIETARY(IsSupportedByKeySystem(
      kExternalClearKey, kVideoMP4MimeType, video_mp4_codecs()));
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kVideoMP4MimeType,
                                    vp9_profile0_codecs()));
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kVideoMP4MimeType,
                                    vp9_profile2_codecs()));
  EXPECT_ECK_AV1(IsSupportedByKeySystem(kExternalClearKey, kVideoWebMMimeType,
                                        av1_codecs()));

  // High 10-bit Profile is not supported when using ExternalClearKey.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoMP4MimeType, video_mp4_hi10p_codecs()));

  // Non-video MP4 codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoMP4MimeType, audio_mp4_codecs()));

  // Invalid or non-MP4 codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoMP4MimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoMP4MimeType, audio_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoMP4MimeType, video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       Audio_MP4) {
  // Valid audio types.
  EXPECT_ECK_PROPRIETARY(IsSupportedByKeySystem(
      kExternalClearKey, kAudioMP4MimeType, audio_mp4_codecs()));
  EXPECT_ECK(IsSupportedByKeySystem(kExternalClearKey, kAudioMP4MimeType,
                                    audio_mp4_flac_codecs()));

  // Non-audio MP4 codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioMP4MimeType, video_mp4_codecs()));

  // Invalid or Non-MP4 codec.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioMP4MimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioMP4MimeType, audio_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kAudioMP4MimeType, video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       SessionType) {
  // Temporary session always supported.
  EXPECT_SUCCESS(
      IsSessionTypeSupported(kExternalClearKey, SessionType::kTemporary));

  // Persistent sessions always supported by External Clear Key.
  EXPECT_SUCCESS(IsSessionTypeSupported(kExternalClearKey,
                                        SessionType::kPersistentLicense));
  EXPECT_SUCCESS(IsSessionTypeSupported(kExternalClearKey,
                                        SessionType::kPersistentUsageRecord));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       Robustness) {
  // External Clear Key doesn't require a robustness string.
  EXPECT_SUCCESS(IsVideoRobustnessSupported(kExternalClearKey, nullptr));
  EXPECT_SUCCESS(IsVideoRobustnessSupported(kExternalClearKey, ""));
  EXPECT_SUCCESS(IsAudioRobustnessSupported(kExternalClearKey, nullptr));
  EXPECT_SUCCESS(IsAudioRobustnessSupported(kExternalClearKey, ""));

  // Non-empty robustness string will be rejected, including valid Widevine
  // robustness strings.
  EXPECT_UNSUPPORTED(
      IsVideoRobustnessSupported(kExternalClearKey, "Invalid String"));
  EXPECT_UNSUPPORTED(
      IsVideoRobustnessSupported(kExternalClearKey, "SW_SECURE_CRYPTO"));
  EXPECT_UNSUPPORTED(
      IsAudioRobustnessSupported(kExternalClearKey, "Invalid String"));
  EXPECT_UNSUPPORTED(
      IsAudioRobustnessSupported(kExternalClearKey, "SW_SECURE_CRYPTO"));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesExternalClearKeyTest,
                       EncryptionScheme) {
  EXPECT_SUCCESS(IsAudioEncryptionSchemeSupported(kExternalClearKey, nullptr));
  EXPECT_SUCCESS(IsAudioEncryptionSchemeSupported(kExternalClearKey, "cenc"));
  EXPECT_SUCCESS(IsAudioEncryptionSchemeSupported(kExternalClearKey, "cbcs"));
  EXPECT_SUCCESS(IsVideoEncryptionSchemeSupported(kExternalClearKey, nullptr));
  EXPECT_SUCCESS(IsVideoEncryptionSchemeSupported(kExternalClearKey, "cenc"));
  EXPECT_SUCCESS(IsVideoEncryptionSchemeSupported(kExternalClearKey, "cbcs"));

  // Invalid encryption schemes will be rejected. However, invalid values
  // generate a TypeError (The provided value '...' is not a valid enum value
  // of type EncryptionScheme), which is not handled by the test page.
  EXPECT_TYPEERROR(
      IsAudioEncryptionSchemeSupported(kExternalClearKey, "Invalid"));
  EXPECT_TYPEERROR(
      IsVideoEncryptionSchemeSupported(kExternalClearKey, "Invalid"));
  EXPECT_TYPEERROR(IsAudioEncryptionSchemeSupported(kExternalClearKey, ""));
  EXPECT_TYPEERROR(IsVideoEncryptionSchemeSupported(kExternalClearKey, ""));
}

// External Clear Key is disabled by default.
IN_PROC_BROWSER_TEST_F(
    EncryptedMediaSupportedTypesExternalClearKeyNotEnabledTest,
    Basic) {
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoWebMMimeType, video_webm_codecs()));

  // Clear Key should still be registered.
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                        video_webm_codecs()));
}

//
// Widevine
//

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest, Basic) {
  EXPECT_WV_SUCCESS(IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType,
                                           video_webm_codecs()));
  EXPECT_WV_SUCCESS(IsSupportedByKeySystem(kWidevine, kAudioWebMMimeType,
                                           audio_webm_codecs()));
  EXPECT_WV_SUCCESS(IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType,
                                           audio_mp4_flac_codecs()));
  EXPECT_WV_PROPRIETARY(
      IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType, video_mp4_codecs()));
  EXPECT_WV_PROPRIETARY(
      IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType, audio_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest,
                       InvalidKeySystems) {
  // Case sensitive.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "com.widevine.WideVine", kVideoWebMMimeType, video_webm_codecs()));

  // TLDs are not allowed.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem("com.", kVideoWebMMimeType, video_webm_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem("com", kVideoWebMMimeType, video_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("com.widevine.", kVideoWebMMimeType,
                                            video_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem("com.widevine", kVideoWebMMimeType,
                                            video_webm_codecs()));

  // Incomplete.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "com.widevine.alp", kVideoWebMMimeType, video_webm_codecs()));

  // Extra character.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "com.widevine.alphab", kVideoWebMMimeType, video_webm_codecs()));

  // There are no child key systems for Widevine.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      "com.widevine.alpha.child", kVideoWebMMimeType, video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest, NoCodecs) {
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType, no_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kAudioWebMMimeType, no_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType, no_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType, no_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest, Video_WebM) {
  // Valid video types.
  EXPECT_WV_SUCCESS(IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType,
                                           video_webm_codecs()));
  EXPECT_WV_SUCCESS(IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType,
                                           vp9_profile0_codecs()));
  EXPECT_WV_AV1(
      IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType, av1_codecs()));

  // Non-video WebM codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType,
                                            audio_webm_codecs()));

  // Invalid or non-Webm codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType,
                                            vp9_profile2_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType,
                                            audio_mp4_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType,
                                            audio_mp4_flac_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoWebMMimeType,
                                            video_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest, Audio_WebM) {
  // Valid audio types.
  EXPECT_WV_SUCCESS(IsSupportedByKeySystem(kWidevine, kAudioWebMMimeType,
                                           audio_webm_codecs()));

  // Non-audio WebM codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kAudioWebMMimeType,
                                            video_webm_codecs()));

  // Invalid or non-Webm codecs.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kAudioWebMMimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kAudioWebMMimeType,
                                            audio_mp4_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kAudioWebMMimeType,
                                            audio_mp4_flac_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kAudioWebMMimeType,
                                            video_mp4_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest, Video_MP4) {
  // Valid video types.
  EXPECT_WV_PROPRIETARY(
      IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType, video_mp4_codecs()));
  EXPECT_WV_SUCCESS(IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType,
                                           vp9_profile0_codecs()));
  EXPECT_WV_AV1(
      IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType, av1_codecs()));

  // High 10-bit Profile is not supported when using Widevine.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType,
                                            video_mp4_hi10p_codecs()));

  // Non-video MP4 codecs.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType, audio_mp4_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType,
                                            audio_mp4_flac_codecs()));

  // Invalid or non-MP4 codecs.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType,
                                            vp9_profile2_codecs()));
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType,
                                            audio_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kVideoMP4MimeType,
                                            video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest, Audio_MP4) {
  // Valid audio types.
  EXPECT_WV_PROPRIETARY(
      IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType, audio_mp4_codecs()));
  EXPECT_WV_SUCCESS(IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType,
                                           audio_mp4_flac_codecs()));

  // Non-audio MP4 codecs.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType, video_mp4_codecs()));

  // Invalid or Non-MP4 codec.
  EXPECT_UNSUPPORTED(
      IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType, invalid_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType,
                                            audio_webm_codecs()));
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(kWidevine, kAudioMP4MimeType,
                                            video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest, SessionType) {
  // Temporary session always supported.
  EXPECT_WV_SUCCESS(IsSessionTypeSupported(kWidevine, SessionType::kTemporary));

  // Persistent license session support varies by platform.
  auto result =
      IsSessionTypeSupported(kWidevine, SessionType::kPersistentLicense);

#if defined(OS_CHROMEOS) || defined(OS_WIN) || defined(OS_MACOSX)
  // Persistent license session supported by Widevine key system on Windows and
  // Mac. On ChromeOS, it is supported when the protected media identifier
  // permission is allowed. See kUnsafelyAllowProtectedMediaIdentifierForDomain
  // used above.
  EXPECT_WV_SUCCESS(result);
#else
  EXPECT_UNSUPPORTED(result);
#endif

  // Persistent usage record session not supported on any platform.
  EXPECT_UNSUPPORTED(
      IsSessionTypeSupported(kWidevine, SessionType::kPersistentUsageRecord));
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest, Robustness) {
  // Robustness is recommended but not required.
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, nullptr));
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, ""));

  EXPECT_UNSUPPORTED(IsVideoRobustnessSupported(kWidevine, "Invalid String"));
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, "SW_SECURE_CRYPTO"));

#if defined(OS_CHROMEOS)
  // "HW_SECURE_ALL" supported on ChromeOS when the protected media identifier
  // permission is allowed. See kUnsafelyAllowProtectedMediaIdentifierForDomain
  // used above.
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#else
  EXPECT_UNSUPPORTED(IsVideoRobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#endif

  // Robustness is recommended but not required.
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, nullptr));
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, ""));

  EXPECT_UNSUPPORTED(IsAudioRobustnessSupported(kWidevine, "Invalid String"));
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, "SW_SECURE_CRYPTO"));

#if defined(OS_CHROMEOS)
  // "SW_SECURE_DECODE" and "HW_SECURE_ALL" supported on ChromeOS when the
  // protected media identifier permission is allowed. See
  // kUnsafelyAllowProtectedMediaIdentifierForDomain used above.
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#else
  EXPECT_UNSUPPORTED(IsAudioRobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
  EXPECT_UNSUPPORTED(IsAudioRobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#endif
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineTest,
                       EncryptionScheme) {
  EXPECT_WV_SUCCESS(IsAudioEncryptionSchemeSupported(kWidevine, nullptr));
  EXPECT_WV_SUCCESS(IsAudioEncryptionSchemeSupported(kWidevine, "cenc"));
  EXPECT_WV_SUCCESS(IsAudioEncryptionSchemeSupported(kWidevine, "cbcs"));
  EXPECT_WV_SUCCESS(IsVideoEncryptionSchemeSupported(kWidevine, nullptr));
  EXPECT_WV_SUCCESS(IsVideoEncryptionSchemeSupported(kWidevine, "cenc"));
  EXPECT_WV_SUCCESS(IsVideoEncryptionSchemeSupported(kWidevine, "cbcs"));

  // Invalid encryption schemes will be rejected. However, invalid values
  // generate a TypeError (The provided value '...' is not a valid enum value
  // of type EncryptionScheme), which is not handled by the test page.
  EXPECT_TYPEERROR(IsAudioEncryptionSchemeSupported(kWidevine, "Invalid"));
  EXPECT_TYPEERROR(IsVideoEncryptionSchemeSupported(kWidevine, "Invalid"));
  EXPECT_TYPEERROR(IsAudioEncryptionSchemeSupported(kWidevine, ""));
  EXPECT_TYPEERROR(IsVideoEncryptionSchemeSupported(kWidevine, ""));
}

//
// EncryptedMediaSupportedTypesWidevineHwSecureTest tests Widevine with hardware
// secure decryption support.
// - ChromeOS: HW_SECURE_ALL are supported by default which is not affected by
// feature media::kHardwareSecureDecryption.
// - Linux/Mac/Windows: Feature media::kHardwareSecureDecryption is enabled, and
// command line switch kOverrideHardwareSecureCodecsForTesting is used to always
// enable vp8 and vp9, and disable avc1; always enable 'cenc' and disable
// 'cbcs', for HW_SECURE* robustness levels. With the switch, real hardware
// capabilities are not checked for the stability of tests.

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineHwSecureTest,
                       Robustness) {
  // Robustness is recommended but not required.
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, nullptr));
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, ""));

  // Video robustness.
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, "SW_SECURE_CRYPTO"));
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, "HW_SECURE_CRYPTO"));
  EXPECT_WV_SUCCESS(IsVideoRobustnessSupported(kWidevine, "HW_SECURE_ALL"));

  // Audio robustness.
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, "SW_SECURE_CRYPTO"));
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, "HW_SECURE_CRYPTO"));
#if defined(OS_CHROMEOS)
  // "SW_SECURE_DECODE" and "HW_SECURE_ALL" supported on ChromeOS when the
  // protected media identifier permission is allowed. See
  // kUnsafelyAllowProtectedMediaIdentifierForDomain used above.
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
  EXPECT_WV_SUCCESS(IsAudioRobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#else
  EXPECT_UNSUPPORTED(IsAudioRobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
  EXPECT_UNSUPPORTED(IsAudioRobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#endif

  // Video proprietary codecs.
  EXPECT_WV_PROPRIETARY(
      IsVideoMp4RobustnessSupported(kWidevine, "SW_SECURE_CRYPTO"));
  EXPECT_WV_PROPRIETARY(
      IsVideoMp4RobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
#if defined(OS_CHROMEOS)
  // "SW_SECURE_DECODE" and "HW_SECURE_ALL" supported on ChromeOS when the
  // protected media identifier permission is allowed. See
  // kUnsafelyAllowProtectedMediaIdentifierForDomain used above.
  EXPECT_WV_PROPRIETARY(
      IsVideoMp4RobustnessSupported(kWidevine, "HW_SECURE_CRYPTO"));
  EXPECT_WV_PROPRIETARY(
      IsVideoMp4RobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#else
  // Not supported because hardware secure avc1 is not supported.
  EXPECT_UNSUPPORTED(
      IsVideoMp4RobustnessSupported(kWidevine, "HW_SECURE_CRYPTO"));
  EXPECT_UNSUPPORTED(IsVideoMp4RobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#endif

  // Audio proprietary codecs.
  // Note that "hardware secure audio" is still supported since hardware secure
  // decryption is supported (because hardware vp8 and vp9 are supported), and
  // we only do decrypt-only for audio.
  EXPECT_WV_PROPRIETARY(
      IsAudioMp4RobustnessSupported(kWidevine, "SW_SECURE_CRYPTO"));
  EXPECT_WV_PROPRIETARY(
      IsAudioMp4RobustnessSupported(kWidevine, "HW_SECURE_CRYPTO"));
#if defined(OS_CHROMEOS)
  // "SW_SECURE_DECODE" and "HW_SECURE_ALL" supported on ChromeOS when the
  // protected media identifier permission is allowed. See
  // kUnsafelyAllowProtectedMediaIdentifierForDomain used above.
  EXPECT_WV_PROPRIETARY(
      IsAudioMp4RobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
  EXPECT_WV_PROPRIETARY(
      IsAudioMp4RobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#else
  EXPECT_UNSUPPORTED(
      IsAudioMp4RobustnessSupported(kWidevine, "SW_SECURE_DECODE"));
  EXPECT_UNSUPPORTED(IsAudioMp4RobustnessSupported(kWidevine, "HW_SECURE_ALL"));
#endif
}

IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesWidevineHwSecureTest,
                       EncryptionScheme) {
  // Both encryption schemes are supported when no robustness is specified.
  EXPECT_WV_SUCCESS(IsAudioEncryptionSchemeSupported(kWidevine, "cenc"));
  EXPECT_WV_SUCCESS(IsAudioEncryptionSchemeSupported(kWidevine, "cbcs"));
  EXPECT_WV_SUCCESS(IsVideoEncryptionSchemeSupported(kWidevine, "cenc"));
  EXPECT_WV_SUCCESS(IsVideoEncryptionSchemeSupported(kWidevine, "cbcs"));

  // Both encryption schemes are supported when SW_SECURE* robustness is
  // specified.
  EXPECT_WV_SUCCESS(
      IsAudioEncryptionSchemeSupported(kWidevine, "cenc", "SW_SECURE_CRYPTO"));
  EXPECT_WV_SUCCESS(
      IsAudioEncryptionSchemeSupported(kWidevine, "cbcs", "SW_SECURE_CRYPTO"));
  EXPECT_WV_SUCCESS(
      IsVideoEncryptionSchemeSupported(kWidevine, "cenc", "SW_SECURE_DECODE"));
  EXPECT_WV_SUCCESS(
      IsVideoEncryptionSchemeSupported(kWidevine, "cbcs", "SW_SECURE_DECODE"));

  // For HW_SECURE* robustness levels. 'cenc' is always supported. 'cbcs' is
  // supported on ChromeOS, but not on other platforms.
  EXPECT_WV_SUCCESS(
      IsAudioEncryptionSchemeSupported(kWidevine, "cenc", "HW_SECURE_CRYPTO"));
  EXPECT_WV_SUCCESS(
      IsVideoEncryptionSchemeSupported(kWidevine, "cenc", "HW_SECURE_ALL"));
#if defined(OS_CHROMEOS)
  EXPECT_WV_SUCCESS(
      IsAudioEncryptionSchemeSupported(kWidevine, "cbcs", "HW_SECURE_CRYPTO"));
  EXPECT_WV_SUCCESS(
      IsVideoEncryptionSchemeSupported(kWidevine, "cbcs", "HW_SECURE_ALL"));
#else
  EXPECT_UNSUPPORTED(
      IsAudioEncryptionSchemeSupported(kWidevine, "cbcs", "HW_SECURE_CRYPTO"));
  EXPECT_UNSUPPORTED(
      IsVideoEncryptionSchemeSupported(kWidevine, "cbcs", "HW_SECURE_ALL"));
#endif
}

//
// Misc failure test cases.
//

#if BUILDFLAG(ENABLE_LIBRARY_CDMS)
IN_PROC_BROWSER_TEST_F(EncryptedMediaSupportedTypesTest,
                       ClearKeyCdmNotRegistered) {
  // External Clear Key will not be supported because Clear Key CDM is not
  // registered on the command line.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoWebMMimeType, video_webm_codecs()));

  // Clear Key should always be supported.
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                        video_webm_codecs()));
}

IN_PROC_BROWSER_TEST_F(
    EncryptedMediaSupportedTypesClearKeyCdmRegisteredWithWrongPathTest,
    Basic) {
  // External Clear Key will not be supported because Clear Key CDM is
  // registered with the wrong path.
  EXPECT_UNSUPPORTED(IsSupportedByKeySystem(
      kExternalClearKey, kVideoWebMMimeType, video_webm_codecs()));

  // Clear Key should always be supported.
  EXPECT_SUCCESS(IsSupportedByKeySystem(kClearKey, kVideoWebMMimeType,
                                        video_webm_codecs()));
}

#endif  // BUILDFLAG(ENABLE_LIBRARY_CDMS)
