// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.share;

import android.app.Activity;
import android.app.PendingIntent;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.content.pm.ResolveInfo;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.Icon;
import android.os.Parcelable;
import android.text.TextUtils;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.OptIn;
import androidx.annotation.VisibleForTesting;
import androidx.core.os.BuildCompat;

import org.chromium.base.ContextUtils;
import org.chromium.base.IntentUtils;
import org.chromium.base.Log;
import org.chromium.base.PackageManagerUtils;
import org.chromium.base.StrictModeContext;
import org.chromium.base.metrics.RecordHistogram;
import org.chromium.chrome.browser.flags.ChromeFeatureList;
import org.chromium.chrome.browser.preferences.ChromePreferenceKeys;
import org.chromium.chrome.browser.preferences.SharedPreferencesManager;
import org.chromium.chrome.browser.profiles.Profile;
import org.chromium.components.browser_ui.share.ShareParams;
import org.chromium.components.browser_ui.share.ShareParams.TargetChosenCallback;
import org.chromium.ui.base.WindowAndroid;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A helper class that provides additional Chrome-specific share functionality.
 */
public class ShareHelper extends org.chromium.components.browser_ui.share.ShareHelper {
    private static final String TAG = "AndroidShare";
    // TODO(https://crbug.com/1420388): Remove when Android OS provides this string.
    private static final String INTENT_EXTRA_CHOOSER_CUSTOM_ACTIONS =
            "android.intent.extra.CHOOSER_CUSTOM_ACTIONS";
    // A generous number used to allocate requestCode for pending intent used by custom actions.
    private static final int MAX_CUSTOM_ACTION_SUPPORTED = 10;
    private static final int CUSTOM_ACTION_REQUEST_CODE_BASE = 112;
    @VisibleForTesting
    static final String EXTRA_SHARE_CUSTOM_ACTION = "EXTRA_SHARE_CUSTOM_ACTION";

    private ShareHelper() {}

    /**
     * Shares the params using the system share sheet.
     * @param params The share parameters.
     * @param profile The profile last shared component will be saved to, if |saveLastUsed| is set.
     * @param saveLastUsed True if the chosen share component should be saved for future reuse.
     */
    // TODO(crbug/1022172): Should be package-protected once modularization is complete.
    public static void shareWithSystemShareSheetUi(
            ShareParams params, @Nullable Profile profile, boolean saveLastUsed) {
        shareWithSystemShareSheetUi(params, profile, saveLastUsed, null);
    }

    /**
     * Shares the params using the system share sheet with custom actinos.
     * @param params The share parameters.
     * @param profile The profile last shared component will be saved to, if |saveLastUsed| is set.
     * @param saveLastUsed True if the chosen share component should be saved for future reuse.
     * @param customActionProvider List of custom actions for Android share sheet.
     */
    @OptIn(markerClass = BuildCompat.PrereleaseSdkCheck.class)
    public static void shareWithSystemShareSheetUi(ShareParams params, @Nullable Profile profile,
            boolean saveLastUsed, @Nullable ChromeCustomShareAction.Provider customActionProvider) {
        assert (customActionProvider == null || ChooserActionHelper.isSupported())
            : "Custom action is not supported.";

        recordShareSource(ShareSourceAndroid.ANDROID_SHARE_SHEET);
        if (saveLastUsed) {
            params.setCallback(new SaveComponentCallback(profile, params.getCallback()));
        }
        Intent intent = getShareIntent(params);

        sendChooserIntent(params.getWindow(), intent, params.getCallback(), customActionProvider);
    }

    /**
     * Share directly with the provided share target.
     * @param params The container holding the share parameters.
     * @param component The component to share to, bypassing any UI.
     * @param profile The profile last shared component will be saved to, if |saveLastUsed| is set.
     * @param saveLastUsed True if the chosen share component should be saved for future reuse.
     */
    // TODO(crbug/1022172): Should be package-protected once modularization is complete.
    public static void shareDirectly(@NonNull ShareParams params, @NonNull ComponentName component,
            @Nullable Profile profile, boolean saveLastUsed) {
        // Save the component directly without using a SaveComponentCallback.
        if (saveLastUsed) {
            setLastShareComponentName(profile, component);
        }
        Intent intent = getShareIntent(params);
        intent.addFlags(Intent.FLAG_ACTIVITY_FORWARD_RESULT | Intent.FLAG_ACTIVITY_PREVIOUS_IS_TOP);
        if (ChromeFeatureList.isEnabled(ChromeFeatureList.CHROME_SHARING_HUB_LAUNCH_ADJACENT)) {
            intent.addFlags(Intent.FLAG_ACTIVITY_LAUNCH_ADJACENT);
        }
        intent.setComponent(component);
        fireIntent(params.getWindow(), intent, null);
    }

    /**
     * Convenience method to create an Intent to retrieve all the apps that support sharing text.
     */
    public static Intent getShareTextAppCompatibilityIntent() {
        Intent intent = new Intent(Intent.ACTION_SEND);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_DOCUMENT);
        intent.putExtra(Intent.EXTRA_SUBJECT, "");
        intent.putExtra(Intent.EXTRA_TEXT, "");
        intent.setType("text/plain");
        return intent;
    }

    /**
     * Convenience method to create an Intent to retrieve all the apps that support sharing image.
     */
    public static Intent getShareImageAppCompatibilityIntent() {
        return getShareFileAppCompatibilityIntent("image/jpeg");
    }

    /**
     * Convenience method to create an Intent to retrieve all the apps that support sharing {@code
     * fileContentType}.
     */
    public static Intent getShareFileAppCompatibilityIntent(String fileContentType) {
        Intent intent = new Intent(Intent.ACTION_SEND);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_DOCUMENT);
        intent.setType(fileContentType);
        return intent;
    }

    /**
     * Gets the {@link ComponentName} of the app that was used to last share.
     */
    @Nullable
    public static ComponentName getLastShareComponentName() {
        SharedPreferencesManager preferencesManager = SharedPreferencesManager.getInstance();
        String name = preferencesManager.readString(
                ChromePreferenceKeys.SHARING_LAST_SHARED_COMPONENT_NAME, null);
        if (name == null) {
            return null;
        }
        return ComponentName.unflattenFromString(name);
    }

    /**
     * Get the icon and name of the most recently shared app by certain app.
     * @param shareIntent Intent used to get list of apps support sharing.
     * @return The Image and the String of the recently shared Icon.
     */
    public static Pair<Drawable, CharSequence> getShareableIconAndName(Intent shareIntent) {
        Drawable directShareIcon = null;
        CharSequence directShareTitle = null;

        final ComponentName component = getLastShareComponentName();
        boolean isComponentValid = false;
        if (component != null) {
            shareIntent.setPackage(component.getPackageName());
            List<ResolveInfo> resolveInfoList =
                    PackageManagerUtils.queryIntentActivities(shareIntent, 0);
            for (ResolveInfo info : resolveInfoList) {
                ActivityInfo ai = info.activityInfo;
                if (component.equals(new ComponentName(ai.applicationInfo.packageName, ai.name))) {
                    isComponentValid = true;
                    break;
                }
            }
        }
        if (isComponentValid) {
            boolean retrieved = false;
            final PackageManager pm = ContextUtils.getApplicationContext().getPackageManager();
            try {
                // TODO(dtrainor): Make asynchronous and have a callback to update the menu.
                // https://crbug.com/729737
                try (StrictModeContext ignored = StrictModeContext.allowDiskReads()) {
                    directShareIcon = pm.getActivityIcon(component);
                    directShareTitle = pm.getActivityInfo(component, 0).loadLabel(pm);
                }
                retrieved = true;
            } catch (NameNotFoundException exception) {
                // Use the default null values.
            }
            RecordHistogram.recordBooleanHistogram(
                    "Android.IsLastSharedAppInfoRetrieved", retrieved);
        }

        return new Pair<>(directShareIcon, directShareTitle);
    }

    /**
     * Share directly with the last used share target, and record its share source.
     * @param params The container holding the share parameters.
     */
    static void shareWithLastUsedComponent(@NonNull ShareParams params) {
        ComponentName component = getLastShareComponentName();
        if (component == null) return;
        assert params.getCallback() == null;
        recordShareSource(ShareSourceAndroid.DIRECT_SHARE);
        shareDirectly(params, component, null, false);
    }

    /**
     * Stores the component selected for sharing last time share was called by certain app.
     *
     * This method is public since it is used in tests to avoid creating share dialog.
     * @param component The {@link ComponentName} of the app selected for sharing.
     */
    @VisibleForTesting
    public static void setLastShareComponentName(Profile profile, ComponentName component) {
        SharedPreferencesManager.getInstance().writeString(
                ChromePreferenceKeys.SHARING_LAST_SHARED_COMPONENT_NAME,
                component.flattenToString());
        if (profile != null) {
            ShareHistoryBridge.addShareEntry(profile, component.flattenToString());
        }
    }

    private static void sendChooserIntent(WindowAndroid window, Intent sharingIntent,
            @Nullable TargetChosenCallback callback,
            ChromeCustomShareAction.Provider customActions) {
        new CustomActionChosenReceiver(callback, customActions)
                .sendChooserIntent(window, sharingIntent);
    }

    /**
     * Helper class for injecting extras into the sharing intents.
     */
    private static class CustomActionChosenReceiver extends TargetChosenReceiver {
        private final ChromeCustomShareAction.Provider mCustomActionProvider;
        private final Map<String, Runnable> mActionsMap = new HashMap<>();

        protected CustomActionChosenReceiver(@Nullable TargetChosenCallback callback,
                @Nullable ChromeCustomShareAction.Provider customActionProvider) {
            super(callback);
            mCustomActionProvider = customActionProvider;
        }

        // Override so this file can have access to call this protected method.
        @Override
        protected void sendChooserIntent(WindowAndroid windowAndroid, Intent sharingIntent) {
            super.sendChooserIntent(windowAndroid, sharingIntent);
        }

        @Override
        protected Intent getChooserIntent(WindowAndroid window, Intent sharingIntent) {
            Intent chooserIntent = super.getChooserIntent(window, sharingIntent);
            if (mCustomActionProvider == null) return chooserIntent;

            List<ChromeCustomShareAction> chromeCustomShareActions =
                    mCustomActionProvider.getCustomActions();
            assert chromeCustomShareActions.size() <= MAX_CUSTOM_ACTION_SUPPORTED
                : "Max number of actions supported:"
                  + MAX_CUSTOM_ACTION_SUPPORTED;

            List<Parcelable> chooserActions = new ArrayList<>();
            Activity activity = window.getActivity().get();

            // Use different request code to avoid pending intent don't collision.
            int requestCode = activity.getTaskId() * MAX_CUSTOM_ACTION_SUPPORTED
                    + CUSTOM_ACTION_REQUEST_CODE_BASE;
            for (var action : chromeCustomShareActions) {
                Intent sendBackIntent = createSendBackIntentWithFilteredAction();
                sendBackIntent.putExtra(EXTRA_SHARE_CUSTOM_ACTION, action.key);
                // Make custom action immutable, since it doesn't need change any chooser component.
                PendingIntent pendingIntent =
                        PendingIntent.getBroadcast(activity, requestCode++, sendBackIntent,
                                PendingIntent.FLAG_CANCEL_CURRENT | PendingIntent.FLAG_ONE_SHOT
                                        | PendingIntent.FLAG_IMMUTABLE);

                Parcelable chooserAction = ChooserActionHelper.newChooserAction(
                        action.icon, action.label, pendingIntent);
                mActionsMap.put(action.key, action.runnable);

                chooserActions.add(chooserAction);
            }

            Parcelable[] customActions = chooserActions.toArray(new Parcelable[0]);
            chooserIntent.putExtra(INTENT_EXTRA_CHOOSER_CUSTOM_ACTIONS, customActions);

            return chooserIntent;
        }

        @Override
        protected void onReceiveInternal(Context context, Intent intent) {
            String action = IntentUtils.safeGetStringExtra(intent, EXTRA_SHARE_CUSTOM_ACTION);
            if (!TextUtils.isEmpty(action)) {
                assert mActionsMap.get(action) != null : "Action <" + action + "> does not exists.";
                mActionsMap.get(action).run();
            }
        }
    }

    /**
     * A {@link TargetChosenCallback} that wraps another callback, forwarding calls to it, and
     * saving the chosen component.
     */
    private static class SaveComponentCallback implements TargetChosenCallback {
        private TargetChosenCallback mOriginalCallback;
        private Profile mProfile;

        public SaveComponentCallback(
                @Nullable Profile profile, @Nullable TargetChosenCallback originalCallback) {
            mOriginalCallback = originalCallback;
            mProfile = profile;
        }

        @Override
        public void onTargetChosen(ComponentName chosenComponent) {
            if (chosenComponent != null) {
                setLastShareComponentName(mProfile, chosenComponent);
            }
            if (mOriginalCallback != null) mOriginalCallback.onTargetChosen(chosenComponent);
        }

        @Override
        public void onCancel() {
            if (mOriginalCallback != null) mOriginalCallback.onCancel();
        }
    }

    /**
     * Helper class used to build Android custom action.
     */
    // TODO(https://crbug.com/1420388): Replace calls with Android OS chooser actions.
    @VisibleForTesting
    public static class ChooserActionHelper {
        /**
         * Try to query if the builder class exists.
         */
        @SuppressWarnings("PrivateApi")
        static boolean isSupported() {
            try {
                Class.forName("android.service.chooser.ChooserAction$Builder");
                return true;
            } catch (ClassNotFoundException e) {
                return false;
            }
        }

        @SuppressWarnings("PrivateApi")
        static Parcelable newChooserAction(Icon icon, String name, PendingIntent action) {
            Parcelable parcelable = null;
            try {
                Class<?> chooserActionBuilderClass =
                        Class.forName("android.service.chooser.ChooserAction$Builder");
                Constructor<?> ctor = chooserActionBuilderClass.getConstructor(
                        Icon.class, CharSequence.class, PendingIntent.class);
                Object builder = ctor.newInstance(icon, name, action);
                parcelable =
                        (Parcelable) chooserActionBuilderClass.getMethod("build").invoke(builder);
            } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException
                    | InvocationTargetException | InstantiationException e) {
                Log.w(TAG, "Building ChooserAction failed.", e);
            }
            return parcelable;
        }
    }
}
