// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_WEBUI_SHORTCUT_CUSTOMIZATION_UI_BACKEND_ACCELERATOR_LAYOUT_TABLE_H_
#define ASH_WEBUI_SHORTCUT_CUSTOMIZATION_UI_BACKEND_ACCELERATOR_LAYOUT_TABLE_H_

#include <cstdint>
#include <functional>
#include <map>
#include <string>

#include "ash/public/cpp/accelerators.h"
#include "ash/public/mojom/accelerator_info.mojom.h"
#include "ash/strings/grit/ash_strings.h"
#include "third_party/abseil-cpp/absl/types/optional.h"
#include "ui/base/accelerators/accelerator.h"
#include "ui/events/event_constants.h"
#include "ui/events/keycodes/keyboard_codes_posix.h"

namespace ash {

// non-ash accelerator action Id. Contains browser action ids and ambient action
// ids.
enum NonConfigurableActions {
  // Browser action ids:
  kBrowserCloseTab,
  kBrowserCloseWindow,
  kBrowserSelectLastTab,
  kBrowserOpenFile,
  kBrowserNewIncognitoWindow,
  kBrowserNewTab,
  kBrowserNewWindow,
  kBrowserRestoreTab,
  kBrowserTabSearch,
  kBrowserClearBrowsingData,
  kBrowserCloseFindOrStop,
  kBrowserFocusBookmarks,
  kBrowserBack,
  kBrowserForward,
  kBrowserFind,
  kBrowserFindNext,
  kBrowserFindPrevious,
  kBrowserHome,
  kBrowserShowDownloads,
  kBrowserShowHistory,
  kBrowserFocusSearch,
  kBrowserFocusMenuBar,
  kBrowserPrint,
  kBrowserReload,
  kBrowserReloadBypassingCache,
  kBrowserZoomNormal,
  kBrowserBookmarkAllTabs,
  kBrowserSavePage,
  kBrowserBookmarkThisTab,
  kBrowserShowAppMenu,
  kBrowserShowBookmarkManager,
  kBrowserDevToolsConsole,
  kBrowserDevToolsInspect,
  kBrowserDevTools,
  kBrowserShowBookmarkBar,
  kBrowserViewSource,
  kBrowserZoomPlus,
  kBrowserZoomMinus,
  kBrowserFocusLocation,
  kBrowserFocusToolbar,
  kBrowserFocusInactivePopupForAccessibility,
  kBrowserSelectTabByIndex,
  kBrowserBottomPage,
  kBrowserTopPage,
  kBrowserNextPane,
  kBrowserRightClick,
  kBrowserAutoComplete,
  kBrowserStopDragTab,
  kBrowserSelectNextTab,
  kBrowserSelectPreviousTab,
  kBrowserPageUp,
  kBrowserPageDown,
  // Ambient action ids:
  kAmbientDragLinkInSameTab,
  kAmbientCycleForwardMRU,
  kAmbientDragLinkInNewTab,
  kAmbientOpenLinkInTab,
  kAmbientOpenLinkInTabBackground,
  kAmbientOpenLinkInWindow,
  kAmbientOpenPageInNewTab,
  kAmbientCycleBackwardMRU,
  kAmbientRightClick,
  kAmbientSaveLinkAsBookmark,
  kAmbientLaunchNumberedApp,
  kAmbientOpenFile,
  kAmbientOpenHighlightedItemOnShelf,
  kAmbientHighlightNextItemOnShelf,
  kAmbientHighlightPreviousItemOnShelf,
  kAmbientMoveAppsInGrid,
  kAmbientMoveAppsInOutFolder,
  kAmbientRemoveHighlightOnShelf,
  kAmbientActivateIndexedDesk,
  kAmbientLaunchAppByIndex,
  kAmbientDisplayHiddenFiles,
  kAmbientOpenRightClickMenu,
  kAmbientCaretBrowsing,
  kAmbientSwitchFocus,
  kAmbientCopy,
  kAmbientCut,
  kAmbientPaste,
  kAmbientPastePlainText,
  kAmbientDeletePreviousWord,
  kAmbientUndo,
  kAmbientRedo,
  kAmbientContentContextSelectAll,
  kAmbientSelectTextToBeginning,
  kAmbientSelectTextToEndOfLine,
  kAmbientSelectPreviousWord,
  kAMbientSelectNextWord,
  kAmbientDeleteNextWord,
  kAmbientGoToBeginningOfDocument,
  kAmbientGoToEndOfDocument,
  kAmbientGoToBeginningOfLine,
  kAmbientGoToEndOfLine,
  kAmbientMoveStartOfPreviousWord,
  kAmbientMoveToEndOfWord,
};

// Used to separate text accelerator parts in the UI e.g ctrl + 1.
enum TextAcceleratorDelimiter {
  kPlusSign,
};

// Contains details for UI styling of an accelerator.
struct AcceleratorLayoutDetails {
  // The accelerator action id associated for a source. Concat `source` and
  // `action_id` to get a unique identifier for an accelerator action.
  uint32_t action_id;

  // String id of the accelerator's description.
  int description_string_id;

  // Category of the accelerator.
  mojom::AcceleratorCategory category;

  // Subcategory of the accelerator.
  mojom::AcceleratorSubcategory sub_category;

  // True if the accelerator cannot be modified through customization.
  // False if the accelerator can be modified through customization.
  bool locked;

  // The layout style of the accelerator, this provides additional context
  // on how to accelerator should be represented in the UI.
  mojom::AcceleratorLayoutStyle layout_style;

  // The source of which the accelerator is from.
  mojom::AcceleratorSource source;
};

// Represents a replacement for part of a non-configurable accelerator.
// Contains the text to display as well as its type (Modifier, Key, Plain Text)
// which is needed to determine how to display the text in the shortcut
// customization app.
class TextAcceleratorPart : public mojom::TextAcceleratorPart {
 public:
  explicit TextAcceleratorPart(ui::EventFlags modifier);
  explicit TextAcceleratorPart(ui::KeyboardCode key_code);
  explicit TextAcceleratorPart(const std::u16string& plain_text);
  explicit TextAcceleratorPart(TextAcceleratorDelimiter delimiter);
  TextAcceleratorPart(const TextAcceleratorPart&);
  TextAcceleratorPart& operator=(const TextAcceleratorPart&);
  ~TextAcceleratorPart();
};

// Contains info related to a non-configurable accelerator. A non-configurable
// accelerator can contain either a standard or text-based accelerator. The
// message_id and list of replacements will be provided when dealing
// with text-based accelerators; otherwise, accelerators will be provided
// and message_id/replacements should not have any value set.
// AcceleratorConfigurationProvider uses this struct to create a list of
// AcceleratorInfo struct's for each non-configurable action.
struct NonConfigurableAcceleratorDetails {
  NonConfigurableAcceleratorDetails(
      int message_id,
      std::vector<TextAcceleratorPart> replacements);
  explicit NonConfigurableAcceleratorDetails(int resource_id);
  explicit NonConfigurableAcceleratorDetails(
      std::vector<ui::Accelerator> accels);
  NonConfigurableAcceleratorDetails(const NonConfigurableAcceleratorDetails&);
  NonConfigurableAcceleratorDetails& operator=(
      const NonConfigurableAcceleratorDetails&);
  ~NonConfigurableAcceleratorDetails();

 public:
  bool IsStandardAccelerator() const { return accelerators.has_value(); }

  // These members are used for the Ambient action ids contained in
  // the NonConfigurableActions enum.
  absl::optional<int> message_id;
  absl::optional<std::vector<TextAcceleratorPart>> replacements;
  // This member is used for the Browser action ids contained in
  // the NonConfigurableActions enum.
  absl::optional<std::vector<ui::Accelerator>> accelerators;
};

using NonConfigurableActionsMap =
    std::map<NonConfigurableActions, NonConfigurableAcceleratorDetails>;

const NonConfigurableActionsMap& GetNonConfigurableActionsMap();

std::u16string GetKeyDisplay(ui::KeyboardCode key_code);

// A fixed array of accelerator layouts used for categorization and styling of
// accelerator actions. The ordering of the array is important and is used
// 1:1 for displaying shortcuts in the shortcut customization app.
// Adding an accelerator layout in this array will create a new entry in the
// app.
constexpr AcceleratorLayoutDetails kAcceleratorLayouts[] = {
    // General
    {TOGGLE_APP_LIST, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_APP_LIST,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_OVERVIEW, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_OVERVIEW,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_SYSTEM_TRAY_BUBBLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_SYSTEM_TRAY_BUBBLE,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_CALENDAR, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_CALENDAR,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_MESSAGE_CENTER_BUBBLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MESSAGE_CENTER_BUBBLE,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TAKE_SCREENSHOT, IDS_ASH_ACCELERATOR_DESCRIPTION_TAKE_SCREENSHOT,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TAKE_PARTIAL_SCREENSHOT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TAKE_PARTIAL_SCREENSHOT,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TAKE_WINDOW_SCREENSHOT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TAKE_WINDOW_SCREENSHOT,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {LOCK_SCREEN, IDS_ASH_ACCELERATOR_DESCRIPTION_LOCK_SCREEN,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SUSPEND, IDS_ASH_ACCELERATOR_DESCRIPTION_SUSPEND,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {EXIT, IDS_ASH_ACCELERATOR_DESCRIPTION_EXIT,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWITCH_TO_NEXT_USER, IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_NEXT_USER,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWITCH_TO_PREVIOUS_USER,
     IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_PREVIOUS_USER,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {FOCUS_CAMERA_PREVIEW, IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_CAMERA_PREVIEW,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_DICTATION, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_DICTATION,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {START_ASSISTANT, IDS_ASH_ACCELERATOR_DESCRIPTION_START_ASSISTANT,
     mojom::AcceleratorCategory::kGeneral,
     mojom::AcceleratorSubcategory::kGeneralControls,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_FILE_MANAGER, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_FILE_MANAGER,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientOpenFile,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_FILE,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientDisplayHiddenFiles,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DISPLAY_HIDDEN_FILES,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {SHOW_SHORTCUT_VIEWER, IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_SHORTCUT_VIEWER,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_CALCULATOR, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_CALCULATOR,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_DIAGNOSTICS, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_DIAGNOSTICS,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_GET_HELP, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_GET_HELP,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_FEEDBACK_PAGE, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_FEEDBACK_PAGE,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientLaunchNumberedApp,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {LAUNCH_LAST_APP, IDS_ASH_ACCELERATOR_DESCRIPTION_LAUNCH_LAST_APP,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {UNPIN, IDS_ASH_ACCELERATOR_DESCRIPTION_UNPIN,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_RESIZE_LOCK_MENU,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_RESIZE_LOCK_MENU,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SHOW_TASK_MANAGER, IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_TASK_MANAGER,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_CROSH, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_CROSH,
     mojom::AcceleratorCategory::kGeneral, mojom::AcceleratorSubcategory::kApps,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},

    // Device
    {VOLUME_UP, IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_UP,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {VOLUME_DOWN, IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_DOWN,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {VOLUME_MUTE, IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_MUTE,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MICROPHONE_MUTE_TOGGLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MICROPHONE_MUTE,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MEDIA_PLAY, IDS_ASH_ACCELERATOR_DESCRIPTION_PLAY_MEDIA,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MEDIA_PAUSE, IDS_ASH_ACCELERATOR_DESCRIPTION_PAUSE_MEDIA,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MEDIA_PLAY_PAUSE, IDS_ASH_ACCELERATOR_DESCRIPTION_PLAY_PAUSE_MEDIA,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MEDIA_NEXT_TRACK, IDS_ASH_ACCELERATOR_DESCRIPTION_NEXT_TRACK,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MEDIA_PREV_TRACK, IDS_ASH_ACCELERATOR_DESCRIPTION_PREVIOUS_TRACK,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MEDIA_FAST_FORWARD, IDS_ASH_ACCELERATOR_DESCRIPTION_FAST_FORWARD_MEDIA,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {FOCUS_PIP, IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_PIP,
     mojom::AcceleratorCategory::kDevice, mojom::AcceleratorSubcategory::kMedia,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {KEYBOARD_BACKLIGHT_TOGGLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_KEYBOARD_BACKLIGHT,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kInputs,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {KEYBOARD_BRIGHTNESS_UP,
     IDS_ASH_ACCELERATOR_DESCRIPTION_KEYBOARD_BRIGHTNESS_UP,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kInputs,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {KEYBOARD_BRIGHTNESS_DOWN,
     IDS_ASH_ACCELERATOR_DESCRIPTION_KEYBOARD_BRIGHTNESS_DOWN,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kInputs,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_IME_MENU_BUBBLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_IME_MENU_BUBBLE,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kInputs,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWITCH_TO_NEXT_IME, IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_NEXT_IME,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kInputs,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWITCH_TO_LAST_USED_IME,
     IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_LAST_USED_IME,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kInputs,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserRightClick,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_RIGHT_CLICK,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kInputs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {SHOW_STYLUS_TOOLS, IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_STYLUS_TOOLS,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kInputs,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {BRIGHTNESS_UP, IDS_ASH_ACCELERATOR_DESCRIPTION_BRIGHTNESS_UP,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {BRIGHTNESS_DOWN, IDS_ASH_ACCELERATOR_DESCRIPTION_BRIGHTNESS_DOWN,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SCALE_UI_UP, IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_UP,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SCALE_UI_DOWN, IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_DOWN,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SCALE_UI_RESET, IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_RESET,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {PRIVACY_SCREEN_TOGGLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_PRIVACY_SCREEN,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_MIRROR_MODE, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MIRROR_MODE,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWAP_PRIMARY_DISPLAY, IDS_ASH_ACCELERATOR_DESCRIPTION_SWAP_PRIMARY_DISPLAY,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {ROTATE_SCREEN, IDS_ASH_ACCELERATOR_DESCRIPTION_ROTATE_SCREEN,
     mojom::AcceleratorCategory::kDevice,
     mojom::AcceleratorSubcategory::kDisplay,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},

    // Browser
    {NonConfigurableActions::kBrowserPrint,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_PRINT,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowAppMenu,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_APP_MENU,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowDownloads,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_DOWNLOADS,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowHistory,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_HISTORY,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserClearBrowsingData,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLEAR_BROWSING_DATA,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserOpenFile,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_OPEN_FILE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusSearch,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_SEARCH,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBrowserNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserAutoComplete,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_AUTO_COMPLETE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBrowserNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenPageInNewTab,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_PAGE_IN_NEW_TAB,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBrowserNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusMenuBar,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_MENU_BAR,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBrowserNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusBookmarks,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_BOOKMARKS,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBrowserNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserNextPane,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEXT_PANE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBrowserNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {FOCUS_PREVIOUS_PANE, IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_PREVIOUS_PANE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBrowserNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserBack,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_BACK,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserForward,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FORWARD,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserHome,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_HOME,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserReload,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_RELOAD,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserReloadBypassingCache,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_RELOAD_BYPASSING_CACHE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserPageUp,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_PAGE_UP,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserPageDown,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_PAGE_DOWN,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserTopPage,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_TOP_PAGE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserBottomPage,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOTTOM_PAGE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserZoomPlus,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_PLUS,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserZoomMinus,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_MINUS,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserZoomNormal,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_NORMAL,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFind,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserSavePage,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SAVE_PAGE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFindNext,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND_NEXT,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFindPrevious,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND_PREVIOUS,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kPages,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserNewTab,
     IDS_ASH_ACCELERATOR_DESCRIPTION_NEW_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserSelectNextTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SELECT_NEXT_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserSelectPreviousTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SELECT_PREVIOU_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserNewWindow,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEW_WINDOW,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserNewIncognitoWindow,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEW_INCOGNITO_WINDOW,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserTabSearch,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_TAB_SEARCH,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserCloseTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLOSE_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserRestoreTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_RESTORE_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserSelectLastTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SELECT_LAST_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserSelectTabByIndex,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientDragLinkInSameTab,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DRAG_LINK_IN_SAME_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientDragLinkInNewTab,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DRAG_LINK_IN_NEW_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenLinkInWindow,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_LINK_IN_WINDOW,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenLinkInTab,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_LINK_IN_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenLinkInTabBackground,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_LINK_IN_TAB_BACKGROUND,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserStopDragTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_STOP_DRAG_TAB,
     mojom::AcceleratorCategory::kBrowser, mojom::AcceleratorSubcategory::kTabs,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserBookmarkThisTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOOKMARK_THIS_TAB,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBookmarks,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientSaveLinkAsBookmark,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SAVE_LINK_AS_BOOKMARK,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBookmarks,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserBookmarkAllTabs,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOOKMARK_ALL_TABS,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBookmarks,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowBookmarkBar,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_BOOKMARK_BAR,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBookmarks,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowBookmarkManager,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_BOOKMARK_MANAGER,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kBookmarks,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserDevToolsConsole,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS_CONSOLE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kDeveloperTools,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserDevToolsInspect,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS_INSPECT,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kDeveloperTools,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserDevTools,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kDeveloperTools,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserViewSource,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_VIEW_SOURCE,
     mojom::AcceleratorCategory::kBrowser,
     mojom::AcceleratorSubcategory::kDeveloperTools,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},

    // Text
    {NonConfigurableActions::kAmbientGoToBeginningOfDocument,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_BEGINNING_OF_DOCUMENT,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientGoToEndOfDocument,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_END_OF_DOCUMENT,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    // TODO(longbowei): Re-enable these shortcuts. These conflict with
    // the kBrowserTopPage and kBrowserBottomPage.
    // {NonConfigurableActions::kAmbientGoToBeginningOfLine,
    //  IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_BEGINNING_OF_LINE,
    //  mojom::AcceleratorCategory::kText,
    //  mojom::AcceleratorSubcategory::kTextNavigation,
    //  /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
    //  mojom::AcceleratorSource::kAmbient},
    // {NonConfigurableActions::kAmbientGoToEndOfLine,
    //  IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_END_OF_LINE,
    //  mojom::AcceleratorCategory::kText,
    //  mojom::AcceleratorSubcategory::kTextNavigation,
    //  /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
    //  mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientMoveStartOfPreviousWord,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_MOVE_TO_START_OF_PREVIOUS_WORD,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientMoveToEndOfWord,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_MOVE_TO_END_OF_WORD,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {TOGGLE_CAPS_LOCK, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_CAPS_LOCK,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SHOW_EMOJI_PICKER, IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_EMOJI_PICKER,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientCopy,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_COPY,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientCut,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CUT, mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientPaste,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_PASTE,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientPastePlainText,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_PASTE_PLAIN_TEXT,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {TOGGLE_CLIPBOARD_HISTORY,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_CLIPBOARD_HISTORY,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientDeletePreviousWord,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DELETE_PREVIOUS_WORD,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientUndo,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_UNDO,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientRedo,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_REDO,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientContentContextSelectAll,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CONTENT_CONTEXT_SELECTALL,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientSelectTextToBeginning,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SELECT_TEXT_TO_BEGINNING,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientSelectTextToEndOfLine,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SELECT_TEXT_TO_END_OF_LINE,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientSelectPreviousWord,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SELECT_PREVIOUS_WORD,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAMbientSelectNextWord,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SELECT_NEXT_WORD,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientDeleteNextWord,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DELETE_NEXT_WORD,
     mojom::AcceleratorCategory::kText,
     mojom::AcceleratorSubcategory::kTextEditing,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},

    // Windows and Desks
    {CYCLE_FORWARD_MRU, IDS_ASH_ACCELERATOR_DESCRIPTION_CYCLE_FORWARD_MRU,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientCycleBackwardMRU,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CYCLE_BACKWARD_MRU,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {TOGGLE_MAXIMIZED, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MAXIMIZED,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {WINDOW_MINIMIZE, IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_MINIMIZE,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_FULLSCREEN, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_FULLSCREEN,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserCloseWindow,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLOSE_WINDOW,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {WINDOW_CYCLE_SNAP_LEFT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_CYCLE_SNAP_LEFT,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {WINDOW_CYCLE_SNAP_RIGHT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_CYCLE_SNAP_RIGHT,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MOVE_ACTIVE_WINDOW_BETWEEN_DISPLAYS,
     IDS_ASH_ACCELERATOR_DESCRIPTION_MOVE_ACTIVE_WINDOW_BETWEEN_DISPLAYS,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MINIMIZE_TOP_WINDOW_ON_BACK,
     IDS_ASH_ACCELERATOR_DESCRIPTION_MINIMIZE_TOP_WINDOW_ON_BACK,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_FLOATING, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_FLOATING,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kWindows,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_NEW_DESK, IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_NEW_DESK,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kDesks,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_REMOVE_CURRENT_DESK,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_REMOVE_CURRENT_DESK,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kDesks,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_ACTIVATE_DESK_LEFT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_ACTIVATE_DESK_LEFT,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kDesks,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_ACTIVATE_DESK_RIGHT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_ACTIVATE_DESK_RIGHT,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kDesks,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_MOVE_ACTIVE_ITEM_LEFT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_MOVE_ACTIVE_ITEM_LEFT,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kDesks,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_MOVE_ACTIVE_ITEM_RIGHT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_MOVE_ACTIVE_ITEM_RIGHT,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kDesks,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientActivateIndexedDesk,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_ACTIVATE_INDEXED_DESK,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kDesks,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {DESKS_TOGGLE_ASSIGN_TO_ALL_DESKS,
     IDS_ASH_ACCELERATOR_ACTIONDESKS_TOGGLE_ASSIGN_TO_ALL_DESKS,
     mojom::AcceleratorCategory::kWindowsAndDesks,
     mojom::AcceleratorSubcategory::kDesks,
     /*locked=*/false, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},

    // Accessibility
    {TOGGLE_SPOKEN_FEEDBACK,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_SPOKEN_FEEDBACK,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kChromeVox,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_HIGH_CONTRAST, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_HIGH_CONTRAST,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kVisibility,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_DOCKED_MAGNIFIER,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_DOCKED_MAGNIFIER,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kVisibility,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_FULLSCREEN_MAGNIFIER,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_FULLSCREEN_MAGNIFIER,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kVisibility,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MAGNIFIER_ZOOM_IN, IDS_ASH_ACCELERATOR_DESCRIPTION_MAGNIFIER_ZOOM_IN,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kVisibility,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MAGNIFIER_ZOOM_OUT, IDS_ASH_ACCELERATOR_DESCRIPTION_MAGNIFIER_ZOOM_OUT,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kVisibility,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientSwitchFocus,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SWITCH_FOCUS,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientCaretBrowsing,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CARET_BROWSING,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {FOCUS_SHELF, IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientHighlightNextItemOnShelf,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_HIGHLIGHT_NEXT_ITEM_ON_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientHighlightPreviousItemOnShelf,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_HIGHTLIGHT_PREVIOUS_ITEM_ON_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenHighlightedItemOnShelf,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_HIGHLIGHTED_ITEM_ON_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientRemoveHighlightOnShelf,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_REMOVE_HIGHLIGHT_ON_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenRightClickMenu,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_RIGHT_CLICK_MENU,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusInactivePopupForAccessibility,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_INACTIVE_POPUP_FOR_ACCESSIBILITY,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusToolbar,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_TOOLBAR,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientMoveAppsInGrid,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_MOVE_APPS_IN_GRID,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientMoveAppsInOutFolder,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_MOVE_APPS_IN_OUT_FOLDER,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kAccessibilityNavigation,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
};
}  // namespace ash

#endif  // ASH_WEBUI_SHORTCUT_CUSTOMIZATION_UI_BACKEND_ACCELERATOR_LAYOUT_TABLE_H_
