// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/system/privacy/privacy_indicators_tray_item_view.h"

#include <string>

#include "ash/constants/ash_features.h"
#include "ash/root_window_controller.h"
#include "ash/shelf/shelf.h"
#include "ash/shell.h"
#include "ash/strings/grit/ash_strings.h"
#include "ash/system/status_area_widget.h"
#include "ash/system/unified/unified_system_tray.h"
#include "ash/test/ash_test_base.h"
#include "base/test/metrics/histogram_tester.h"
#include "base/test/scoped_feature_list.h"
#include "base/test/task_environment.h"
#include "base/time/time.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/gfx/animation/linear_animation.h"
#include "ui/views/controls/image_view.h"
#include "ui/views/layout/box_layout.h"

namespace {

const int kPrivacyIndicatorsViewExpandedShorterSideSize = 24;
const int kPrivacyIndicatorsViewExpandedLongerSideSize = 50;
const int kPrivacyIndicatorsViewSize = 8;

constexpr char kPrivacyIndicatorsShowTypeHistogramName[] =
    "Ash.PrivacyIndicators.ShowType";
constexpr char kPrivacyIndicatorsShowPerSessionHistogramName[] =
    "Ash.PrivacyIndicators.NumberOfShowsPerSession";
constexpr char kCountAppsAccessCameraHistogramName[] =
    "Ash.PrivacyIndicators.NumberOfAppsAccessingCamera";
constexpr char kCountAppsAccessMicrophoneHistogramName[] =
    "Ash.PrivacyIndicators.NumberOfAppsAccessingMicrophone";
constexpr char kRepeatedShowsHistogramName[] =
    "Ash.PrivacyIndicators.NumberOfRepeatedShows";

// Get the expected size in expand animation, given the animation value.
int GetExpectedSizeInExpandAnimation(double progress) {
  return kPrivacyIndicatorsViewExpandedLongerSideSize *
         gfx::Tween::CalculateValue(gfx::Tween::ACCEL_20_DECEL_100, progress);
}

// Get the expected size in shrink animation, given the animation value.
int GetExpectedSizeInShrinkAnimation(bool for_longer_side, double progress) {
  double animation_value =
      gfx::Tween::CalculateValue(gfx::Tween::ACCEL_20_DECEL_100, progress);
  int begin_size = for_longer_side
                       ? kPrivacyIndicatorsViewExpandedLongerSideSize
                       : kPrivacyIndicatorsViewExpandedShorterSideSize;
  return begin_size -
         (begin_size - kPrivacyIndicatorsViewSize) * animation_value;
}

// Get the expected tooltip text, given the string for camera/mic access and
// screen share.
std::u16string GetExpectedTooltipText(std::u16string cam_mic_status,
                                      std::u16string screen_share_status) {
  if (cam_mic_status.empty())
    return screen_share_status;

  if (screen_share_status.empty())
    return cam_mic_status;

  return l10n_util::GetStringFUTF16(IDS_PRIVACY_INDICATORS_VIEW_TOOLTIP,
                                    {cam_mic_status, screen_share_status},
                                    /*offsets=*/nullptr);
}

}  // namespace

namespace ash {

class PrivacyIndicatorsTrayItemViewTest : public AshTestBase {
 public:
  PrivacyIndicatorsTrayItemViewTest()
      : AshTestBase(base::test::TaskEnvironment::TimeSource::MOCK_TIME) {}
  PrivacyIndicatorsTrayItemViewTest(const PrivacyIndicatorsTrayItemViewTest&) =
      delete;
  PrivacyIndicatorsTrayItemViewTest& operator=(
      const PrivacyIndicatorsTrayItemViewTest&) = delete;
  ~PrivacyIndicatorsTrayItemViewTest() override = default;

  // AshTestBase:
  void SetUp() override {
    scoped_feature_list_.InitAndEnableFeature(features::kPrivacyIndicators);

    AshTestBase::SetUp();
    privacy_indicators_view_ =
        std::make_unique<PrivacyIndicatorsTrayItemView>(GetPrimaryShelf());
  }

  void TearDown() override {
    privacy_indicators_view_.reset();
    AshTestBase::TearDown();
  }

  std::u16string GetTooltipText() {
    return privacy_indicators_view_->GetTooltipText(gfx::Point());
  }

  views::BoxLayout* GetLayoutManager(
      PrivacyIndicatorsTrayItemView* privacy_indicators_view) {
    return privacy_indicators_view->layout_manager_;
  }

  void AnimateToValue(gfx::LinearAnimation* animation, double animation_value) {
    EXPECT_TRUE(animation->is_animating());
    animation->SetCurrentValue(animation_value);
    privacy_indicators_view_->AnimationProgressed(animation);
  }

  // Set `privacy_indicators_view_` to be visible and perform animation.
  void SetViewVisibleWithAnimation() {
    privacy_indicators_view()->SetVisible(true);
    privacy_indicators_view_->PerformAnimation();
  }

  // Simulates completing the animation.
  void SimulateAnimationEnded() {
    privacy_indicators_view_->AnimationEnded(
        privacy_indicators_view_->shorter_side_shrink_animation_.get());
  }

 protected:
  PrivacyIndicatorsTrayItemView* privacy_indicators_view() {
    return privacy_indicators_view_.get();
  }

  views::ImageView* camera_icon() {
    return privacy_indicators_view_->camera_icon_;
  }
  views::ImageView* microphone_icon() {
    return privacy_indicators_view_->microphone_icon_;
  }
  views::ImageView* screen_share_icon() {
    return privacy_indicators_view_->screen_share_icon_;
  }

  gfx::LinearAnimation* expand_animation() {
    return privacy_indicators_view_->expand_animation_.get();
  }

  PrivacyIndicatorsTrayItemView::AnimationState animation_state() {
    return privacy_indicators_view_->animation_state_;
  }

  gfx::LinearAnimation* longer_side_shrink_animation() {
    return privacy_indicators_view_->longer_side_shrink_animation_.get();
  }

  gfx::LinearAnimation* shorter_side_shrink_animation() {
    return privacy_indicators_view_->shorter_side_shrink_animation_.get();
  }

 private:
  std::unique_ptr<PrivacyIndicatorsTrayItemView> privacy_indicators_view_;

  base::test::ScopedFeatureList scoped_feature_list_;
};

TEST_F(PrivacyIndicatorsTrayItemViewTest, IconsVisibility) {
  EXPECT_FALSE(privacy_indicators_view()->GetVisible());

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/false);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_TRUE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/true);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_TRUE(microphone_icon()->GetVisible());

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/true);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_TRUE(camera_icon()->GetVisible());
  EXPECT_TRUE(microphone_icon()->GetVisible());

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/false);
  EXPECT_FALSE(privacy_indicators_view()->GetVisible());
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, IconsVisibilityAfterAnimation) {
  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/true);
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kExpand,
            animation_state());
  ASSERT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_TRUE(camera_icon()->GetVisible());
  EXPECT_TRUE(microphone_icon()->GetVisible());

  // No icons shown after the animation.
  SimulateAnimationEnded();
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kIdle,
            animation_state());
  ASSERT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());

  // Since there's no new sensor and new media stream added, no icons should be
  // visible and animation should not be triggered.
  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/false);
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kIdle,
            animation_state());
  ASSERT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());

  // New sensor is accessed (microphone), so we show all icons accessing that
  // particular app. Animation should start.
  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/true);
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kExpand,
            animation_state());
  ASSERT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_TRUE(camera_icon()->GetVisible());
  EXPECT_TRUE(microphone_icon()->GetVisible());

  SimulateAnimationEnded();

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/false);

  // New app accessed, show the indicator according to that app.
  privacy_indicators_view()->Update(/*app_id=*/"app_id2",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/true);
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kExpand,
            animation_state());
  ASSERT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_TRUE(microphone_icon()->GetVisible());

  SimulateAnimationEnded();

  // Updates the previous app. However, since no new sensor is accessed
  // (microphone is already accessed by the second app), the indicator should
  // remain the same with no animation.
  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/true);
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kIdle,
            animation_state());
  ASSERT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, ScreenShareIconsVisibility) {
  EXPECT_FALSE(privacy_indicators_view()->GetVisible());

  privacy_indicators_view()->UpdateScreenShareStatus(
      /*is_screen_sharing=*/true);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_TRUE(screen_share_icon()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());

  privacy_indicators_view()->UpdateScreenShareStatus(
      /*is_screen_sharing=*/false);
  EXPECT_FALSE(privacy_indicators_view()->GetVisible());

  // Test screen share showing up with other icons.
  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/true);
  privacy_indicators_view()->UpdateScreenShareStatus(
      /*is_screen_sharing=*/true);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_TRUE(microphone_icon()->GetVisible());
  EXPECT_TRUE(screen_share_icon()->GetVisible());

  privacy_indicators_view()->UpdateScreenShareStatus(
      /*is_screen_sharing=*/false);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_TRUE(microphone_icon()->GetVisible());
  EXPECT_FALSE(screen_share_icon()->GetVisible());
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, TooltipText) {
  EXPECT_EQ(GetExpectedTooltipText(/*cam_mic_status=*/std::u16string(),
                                   /*screen_share_status=*/std::u16string()),
            GetTooltipText());

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/false);
  EXPECT_EQ(GetExpectedTooltipText(/*cam_mic_status=*/l10n_util::GetStringUTF16(
                                       IDS_PRIVACY_NOTIFICATION_TITLE_CAMERA),
                                   /*screen_share_status=*/std::u16string()),
            GetTooltipText());

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/true);
  EXPECT_EQ(GetExpectedTooltipText(/*cam_mic_status=*/l10n_util::GetStringUTF16(
                                       IDS_PRIVACY_NOTIFICATION_TITLE_MIC),
                                   /*screen_share_status=*/std::u16string()),
            GetTooltipText());

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/true);
  EXPECT_EQ(
      GetExpectedTooltipText(/*cam_mic_status=*/l10n_util::GetStringUTF16(
                                 IDS_PRIVACY_NOTIFICATION_TITLE_CAMERA_AND_MIC),
                             /*screen_share_status=*/std::u16string()),
      GetTooltipText());

  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/false);
  EXPECT_EQ(GetExpectedTooltipText(/*cam_mic_status=*/std::u16string(),
                                   /*screen_share_status=*/std::u16string()),
            GetTooltipText());

  privacy_indicators_view()->UpdateScreenShareStatus(
      /*is_screen_sharing=*/true);
  EXPECT_EQ(GetExpectedTooltipText(
                /*cam_mic_status=*/std::u16string(),
                /*screen_share_status=*/l10n_util::GetStringUTF16(
                    IDS_ASH_STATUS_TRAY_SCREEN_SHARE_TITLE)),
            GetTooltipText());
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, ShelfAlignmentChanged) {
  auto* privacy_indicators_view =
      GetPrimaryUnifiedSystemTray()->privacy_indicators_view();

  GetPrimaryShelf()->SetAlignment(ShelfAlignment::kLeft);
  EXPECT_EQ(views::BoxLayout::Orientation::kVertical,
            GetLayoutManager(privacy_indicators_view)->GetOrientation());

  GetPrimaryShelf()->SetAlignment(ShelfAlignment::kBottom);
  EXPECT_EQ(views::BoxLayout::Orientation::kHorizontal,
            GetLayoutManager(privacy_indicators_view)->GetOrientation());

  GetPrimaryShelf()->SetAlignment(ShelfAlignment::kRight);
  EXPECT_EQ(views::BoxLayout::Orientation::kVertical,
            GetLayoutManager(privacy_indicators_view)->GetOrientation());

  GetPrimaryShelf()->SetAlignment(ShelfAlignment::kBottomLocked);
  EXPECT_EQ(views::BoxLayout::Orientation::kHorizontal,
            GetLayoutManager(privacy_indicators_view)->GetOrientation());
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, VisibilityAnimation) {
  GetPrimaryShelf()->SetAlignment(ShelfAlignment::kBottom);

  EXPECT_FALSE(privacy_indicators_view()->GetVisible());
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kIdle,
            animation_state());

  SetViewVisibleWithAnimation();
  double progress = 0.5;

  // Firstly, expand animation will be performed.
  AnimateToValue(expand_animation(), progress);
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kExpand,
            animation_state());
  EXPECT_EQ(kPrivacyIndicatorsViewExpandedShorterSideSize,
            privacy_indicators_view()->GetPreferredSize().height());
  EXPECT_EQ(GetExpectedSizeInExpandAnimation(progress),
            privacy_indicators_view()->GetPreferredSize().width());

  expand_animation()->End();

  // When expand animation ends, the view will be in `kDwellInExpand` state.
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kDwellInExpand,
            animation_state());
  EXPECT_EQ(kPrivacyIndicatorsViewExpandedShorterSideSize,
            privacy_indicators_view()->GetPreferredSize().height());
  EXPECT_EQ(kPrivacyIndicatorsViewExpandedLongerSideSize,
            privacy_indicators_view()->GetPreferredSize().width());

  // After that shrink animations will be started.
  longer_side_shrink_animation()->Start();
  AnimateToValue(longer_side_shrink_animation(), progress);

  EXPECT_EQ(
      PrivacyIndicatorsTrayItemView::AnimationState::kOnlyLongerSideShrink,
      animation_state());
  EXPECT_EQ(kPrivacyIndicatorsViewExpandedShorterSideSize,
            privacy_indicators_view()->GetPreferredSize().height());
  EXPECT_EQ(
      GetExpectedSizeInShrinkAnimation(/*for_longer_side=*/true, progress),
      privacy_indicators_view()->GetPreferredSize().width());

  shorter_side_shrink_animation()->Start();
  AnimateToValue(shorter_side_shrink_animation(), progress);

  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kBothSideShrink,
            animation_state());
  EXPECT_EQ(
      GetExpectedSizeInShrinkAnimation(/*for_longer_side=*/false, progress),
      privacy_indicators_view()->GetPreferredSize().height());
  EXPECT_EQ(
      GetExpectedSizeInShrinkAnimation(/*for_longer_side=*/true, progress),
      privacy_indicators_view()->GetPreferredSize().width());

  longer_side_shrink_animation()->End();
  shorter_side_shrink_animation()->End();

  // When finish, the view should have the size of a dot.
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kIdle,
            animation_state());
  EXPECT_EQ(kPrivacyIndicatorsViewSize,
            privacy_indicators_view()->GetPreferredSize().height());
  EXPECT_EQ(kPrivacyIndicatorsViewSize,
            privacy_indicators_view()->GetPreferredSize().width());

  // All icon should not be visible.
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());
  EXPECT_FALSE(screen_share_icon()->GetVisible());
}

// Same test as above, but with the side shelf (the longer and shorter side will
// be flipped).
TEST_F(PrivacyIndicatorsTrayItemViewTest, SideShelfVisibilityAnimation) {
  GetPrimaryShelf()->SetAlignment(ShelfAlignment::kLeft);

  EXPECT_FALSE(privacy_indicators_view()->GetVisible());
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kIdle,
            animation_state());

  SetViewVisibleWithAnimation();
  double progress = 0.5;

  // Firstly, expand animation will be performed.
  AnimateToValue(expand_animation(), progress);
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kExpand,
            animation_state());
  EXPECT_EQ(kPrivacyIndicatorsViewExpandedShorterSideSize,
            privacy_indicators_view()->GetPreferredSize().width());
  EXPECT_EQ(GetExpectedSizeInExpandAnimation(progress),
            privacy_indicators_view()->GetPreferredSize().height());

  expand_animation()->End();

  // When expand animation ends, the view will be in `kDwellInExpand` state.
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kDwellInExpand,
            animation_state());
  EXPECT_EQ(kPrivacyIndicatorsViewExpandedShorterSideSize,
            privacy_indicators_view()->GetPreferredSize().width());
  EXPECT_EQ(kPrivacyIndicatorsViewExpandedLongerSideSize,
            privacy_indicators_view()->GetPreferredSize().height());

  // After that shrink animations will be started.
  longer_side_shrink_animation()->Start();
  AnimateToValue(longer_side_shrink_animation(), progress);

  EXPECT_EQ(
      PrivacyIndicatorsTrayItemView::AnimationState::kOnlyLongerSideShrink,
      animation_state());
  EXPECT_EQ(kPrivacyIndicatorsViewExpandedShorterSideSize,
            privacy_indicators_view()->GetPreferredSize().width());
  EXPECT_EQ(
      GetExpectedSizeInShrinkAnimation(/*for_longer_side=*/true, progress),
      privacy_indicators_view()->GetPreferredSize().height());

  shorter_side_shrink_animation()->Start();
  AnimateToValue(shorter_side_shrink_animation(), progress);

  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kBothSideShrink,
            animation_state());
  EXPECT_EQ(
      GetExpectedSizeInShrinkAnimation(/*for_longer_side=*/false, progress),
      privacy_indicators_view()->GetPreferredSize().width());
  EXPECT_EQ(
      GetExpectedSizeInShrinkAnimation(/*for_longer_side=*/true, progress),
      privacy_indicators_view()->GetPreferredSize().height());

  longer_side_shrink_animation()->End();
  shorter_side_shrink_animation()->End();

  // When finish, the view should have the size of a dot.
  EXPECT_EQ(PrivacyIndicatorsTrayItemView::AnimationState::kIdle,
            animation_state());
  EXPECT_EQ(kPrivacyIndicatorsViewSize,
            privacy_indicators_view()->GetPreferredSize().width());
  EXPECT_EQ(kPrivacyIndicatorsViewSize,
            privacy_indicators_view()->GetPreferredSize().height());

  // All icon should not be visible.
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());
  EXPECT_FALSE(screen_share_icon()->GetVisible());
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, StateChangeDuringAnimation) {
  SetViewVisibleWithAnimation();
  double progress = 0.5;

  // Firstly, expand animation will be performed.
  AnimateToValue(expand_animation(), progress);

  // Update state in mid animation, shouldn't crash anything.
  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/false);

  expand_animation()->End();

  // After that shrink animations will be started.
  longer_side_shrink_animation()->Start();
  AnimateToValue(longer_side_shrink_animation(), progress);

  // Update the state again, no crash expected.
  privacy_indicators_view()->UpdateScreenShareStatus(
      /*is_screen_sharing=*/true);

  shorter_side_shrink_animation()->Start();
  AnimateToValue(shorter_side_shrink_animation(), progress);

  // The view should become invisible immediately after setting these states.
  privacy_indicators_view()->Update(/*app_id=*/"app_id",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/false);
  privacy_indicators_view()->UpdateScreenShareStatus(
      /*is_screen_sharing=*/false);
  EXPECT_FALSE(privacy_indicators_view()->GetVisible());

  // Clean up.
  longer_side_shrink_animation()->End();
  shorter_side_shrink_animation()->End();
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, MultipleAppsAccess) {
  EXPECT_FALSE(privacy_indicators_view()->GetVisible());

  privacy_indicators_view()->Update(/*app_id=*/"app_id1",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/false);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_TRUE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());

  // When a new app accessing mic/cam, we will show the icons according to the
  // access state of that particular app.
  privacy_indicators_view()->Update(/*app_id=*/"app_id2",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/false);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_TRUE(camera_icon()->GetVisible());
  EXPECT_FALSE(microphone_icon()->GetVisible());

  privacy_indicators_view()->Update(/*app_id=*/"app_id3",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/true);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());
  EXPECT_FALSE(camera_icon()->GetVisible());
  EXPECT_TRUE(microphone_icon()->GetVisible());

  // Indicator should still show when removing 1 and 2 app(s).
  privacy_indicators_view()->Update(/*app_id=*/"app_id2",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/false);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());

  privacy_indicators_view()->Update(/*app_id=*/"app_id3",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/false);
  EXPECT_TRUE(privacy_indicators_view()->GetVisible());

  // Indicator should hide when removing all apps.
  privacy_indicators_view()->Update(/*app_id=*/"app_id1",
                                    /*is_camera_used=*/false,
                                    /*is_microphone_used=*/false);
  EXPECT_FALSE(privacy_indicators_view()->GetVisible());
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, RecordShowTypeMetrics) {
  auto check_histogram_record = [](bool is_camera_used, bool is_microphone_used,
                                   bool is_screen_sharing,
                                   PrivacyIndicatorsTrayItemView* view,
                                   PrivacyIndicatorsTrayItemView::Type type) {
    base::HistogramTester histograms;
    view->Update(/*app_id=*/"app_id", is_camera_used, is_microphone_used);
    view->UpdateScreenShareStatus(is_screen_sharing);
    histograms.ExpectBucketCount(kPrivacyIndicatorsShowTypeHistogramName, type,
                                 1);
  };

  check_histogram_record(/*is_camera_used=*/true, /*is_microphone_used=*/false,
                         /*is_screen_sharing=*/false, privacy_indicators_view(),
                         PrivacyIndicatorsTrayItemView::Type::kCamera);

  check_histogram_record(/*is_camera_used=*/false, /*is_microphone_used=*/true,
                         /*is_screen_sharing=*/false, privacy_indicators_view(),
                         PrivacyIndicatorsTrayItemView::Type::kMicrophone);

  check_histogram_record(/*is_camera_used=*/false, /*is_microphone_used=*/false,
                         /*is_screen_sharing=*/true, privacy_indicators_view(),
                         PrivacyIndicatorsTrayItemView::Type::kScreenSharing);

  check_histogram_record(
      /*is_camera_used=*/true, /*is_microphone_used=*/true,
      /*is_screen_sharing=*/false, privacy_indicators_view(),
      PrivacyIndicatorsTrayItemView::Type::kCameraMicrophone);

  check_histogram_record(
      /*is_camera_used=*/true, /*is_microphone_used=*/false,
      /*is_screen_sharing=*/true, privacy_indicators_view(),
      PrivacyIndicatorsTrayItemView::Type::kCameraScreenSharing);

  check_histogram_record(
      /*is_camera_used=*/false, /*is_microphone_used=*/true,
      /*is_screen_sharing=*/true, privacy_indicators_view(),
      PrivacyIndicatorsTrayItemView::Type::kMicrophoneScreenSharing);

  check_histogram_record(
      /*is_camera_used=*/true, /*is_microphone_used=*/true,
      /*is_screen_sharing=*/true, privacy_indicators_view(),
      PrivacyIndicatorsTrayItemView::Type::kAllUsed);
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, RecordShowPerSessionMetrics) {
  // Set up 2 displays. Note that only one instance should be recorded for the
  // primary display when session changes.
  UpdateDisplay("100x200,300x400");
  int expected_count = 1;

  // Show the indicator in the given `show_count` number of times.
  auto trigger_show_indicator = [](int show_count) {
    // Update the state of camera/microphone access so that the indicators on
    // all displays show, then hide for `show_count` times.
    for (auto i = 0; i < show_count; i++) {
      for (auto* controller : Shell::Get()->GetAllRootWindowControllers()) {
        auto* indicator_view = controller->GetStatusAreaWidget()
                                   ->unified_system_tray()
                                   ->privacy_indicators_view();
        indicator_view->Update(/*app_id=*/"app_id", /*is_camera_used=*/true,
                               /*is_microphone_used=*/true);
        indicator_view->Update(/*app_id=*/"app_id", /*is_camera_used=*/false,
                               /*is_microphone_used=*/false);
      }
    }
  };

  base::HistogramTester histograms;

  GetSessionControllerClient()->SetSessionState(
      session_manager::SessionState::ACTIVE);

  int expected_sample = 1;
  trigger_show_indicator(expected_sample);

  // After session changed, metrics should be recorded.
  GetSessionControllerClient()->SetSessionState(
      session_manager::SessionState::LOCKED);
  histograms.ExpectBucketCount(kPrivacyIndicatorsShowPerSessionHistogramName,
                               expected_sample, expected_count);

  expected_sample = 6;
  trigger_show_indicator(expected_sample);

  // After session changed, metrics should be recorded.
  GetSessionControllerClient()->SetSessionState(
      session_manager::SessionState::ACTIVE);
  histograms.ExpectBucketCount(kPrivacyIndicatorsShowPerSessionHistogramName,
                               expected_sample, expected_count);

  expected_sample = 10;
  trigger_show_indicator(expected_sample);

  // After session changed, metrics should be recorded.
  GetSessionControllerClient()->SetSessionState(
      session_manager::SessionState::LOGIN_PRIMARY);
  histograms.ExpectBucketCount(kPrivacyIndicatorsShowPerSessionHistogramName,
                               expected_sample, expected_count);
}

// When multiple apps access camera and microphone, their histograms should
// update accordingly.
TEST_F(PrivacyIndicatorsTrayItemViewTest, RecordAppAccessSimultaneously) {
  base::HistogramTester histograms;

  privacy_indicators_view()->Update(/*app_id=*/"app_id1",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/false);
  histograms.ExpectBucketCount(kCountAppsAccessCameraHistogramName, 1, 1);
  histograms.ExpectBucketCount(kCountAppsAccessMicrophoneHistogramName, 1, 0);

  privacy_indicators_view()->Update(/*app_id=*/"app_id2",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/true);
  histograms.ExpectBucketCount(kCountAppsAccessCameraHistogramName, 2, 1);
  histograms.ExpectBucketCount(kCountAppsAccessMicrophoneHistogramName, 1, 1);

  privacy_indicators_view()->Update(/*app_id=*/"app_id3",
                                    /*is_camera_used=*/true,
                                    /*is_microphone_used=*/true);
  histograms.ExpectBucketCount(kCountAppsAccessCameraHistogramName, 3, 1);
  histograms.ExpectBucketCount(kCountAppsAccessMicrophoneHistogramName, 2, 1);
}

TEST_F(PrivacyIndicatorsTrayItemViewTest, RecordRepeatedShows) {
  // Set up 2 displays. Note that only one instance should be recorded for the
  // primary display when session changes.
  UpdateDisplay("100x200,300x400");

  base::HistogramTester histograms;

  int expected_sample = 6;

  // Makes the view flicker (show then hide) for `expected_sample` of times.
  // Metric should be recorded for this repeated shows.
  for (auto i = 0; i < expected_sample; i++) {
    for (auto* controller : Shell::Get()->GetAllRootWindowControllers()) {
      auto* indicator_view = controller->GetStatusAreaWidget()
                                 ->unified_system_tray()
                                 ->privacy_indicators_view();
      indicator_view->Update(/*app_id=*/"app_id", /*is_camera_used=*/true,
                             /*is_microphone_used=*/true);
      indicator_view->Update(/*app_id=*/"app_id", /*is_camera_used=*/false,
                             /*is_microphone_used=*/false);
    }
    task_environment()->FastForwardBy(base::Milliseconds(80));
  }

  task_environment()->FastForwardBy(base::Milliseconds(100));
  histograms.ExpectBucketCount(kRepeatedShowsHistogramName, expected_sample, 1);

  // Makes one more flickering after 100ms. This flicker should not count
  // towards the previous ones.
  for (auto* controller : Shell::Get()->GetAllRootWindowControllers()) {
    auto* indicator_view = controller->GetStatusAreaWidget()
                               ->unified_system_tray()
                               ->privacy_indicators_view();
    indicator_view->Update(/*app_id=*/"app_id", /*is_camera_used=*/true,
                           /*is_microphone_used=*/true);
    indicator_view->Update(/*app_id=*/"app_id", /*is_camera_used=*/false,
                           /*is_microphone_used=*/false);
  }
  task_environment()->FastForwardBy(base::Milliseconds(100));
  histograms.ExpectBucketCount(kRepeatedShowsHistogramName, expected_sample + 1,
                               0);

  // Make sure it works again.
  expected_sample = 8;

  // Makes the view flicker (show then hide) for `expected_sample` of times.
  // Metric should be recorded for this repeated shows.
  for (auto i = 0; i < expected_sample; i++) {
    for (auto* controller : Shell::Get()->GetAllRootWindowControllers()) {
      auto* indicator_view = controller->GetStatusAreaWidget()
                                 ->unified_system_tray()
                                 ->privacy_indicators_view();
      indicator_view->Update(/*app_id=*/"app_id", /*is_camera_used=*/true,
                             /*is_microphone_used=*/true);
      indicator_view->Update(/*app_id=*/"app_id", /*is_camera_used=*/false,
                             /*is_microphone_used=*/false);
    }
    task_environment()->FastForwardBy(base::Milliseconds(80));
  }

  task_environment()->FastForwardBy(base::Milliseconds(100));
  histograms.ExpectBucketCount(kRepeatedShowsHistogramName, expected_sample, 1);
}

}  // namespace ash
