// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_WEBUI_SHORTCUT_CUSTOMIZATION_UI_BACKEND_ACCELERATOR_LAYOUT_TABLE_H_
#define ASH_WEBUI_SHORTCUT_CUSTOMIZATION_UI_BACKEND_ACCELERATOR_LAYOUT_TABLE_H_

#include <cstdint>
#include <functional>
#include <map>
#include <string>

#include "ash/public/cpp/accelerators.h"
#include "ash/public/mojom/accelerator_info.mojom.h"
#include "ash/strings/grit/ash_strings.h"
#include "base/containers/fixed_flat_map.h"
#include "base/no_destructor.h"
#include "base/strings/string_piece_forward.h"
#include "third_party/abseil-cpp/absl/types/optional.h"
#include "ui/base/accelerators/accelerator.h"
#include "ui/events/event_constants.h"
#include "ui/events/keycodes/keyboard_codes_posix.h"

namespace ash {

// non-ash accelerator action Id. Contains browser action ids and ambient action
// ids.
enum NonConfigurableActions {
  // Browser action ids:
  kBrowserCloseTab,
  kBrowserCloseWindow,
  kBrowserSelectLastTab,
  kBrowserOpenFile,
  kBrowserNewIncognitoWindow,
  kBrowserNewTab,
  kBrowserNewWindow,
  kBrowserRestoreTab,
  kBrowserTabSearch,
  kBrowserClearBrowsingData,
  kBrowserCloseFindOrStop,
  kBrowserFocusBookmarks,
  kBrowserBack,
  kBrowserForward,
  kBrowserFind,
  kBrowserFindNext,
  kBrowserFindPrevious,
  kBrowserHome,
  kBrowserShowDownloads,
  kBrowserShowHistory,
  kBrowserFocusSearch,
  kBrowserFocusMenuBar,
  kBrowserPrint,
  kBrowserReload,
  kBrowserReloadBypassingCache,
  kBrowserZoomNormal,
  kBrowserBookmarkAllTabs,
  kBrowserSavePage,
  kBrowserBookmarkThisTab,
  kBrowserShowAppMenu,
  kBrowserShowBookmarkManager,
  kBrowserDevToolsConsole,
  kBrowserDevToolsInspect,
  kBrowserDevTools,
  kBrowserShowBookmarkBar,
  kBrowserViewSource,
  kBrowserZoomPlus,
  kBrowserZoomMinus,
  kBrowserFocusLocation,
  kBrowserFocusToolbar,
  kBrowserFocusInactivePopupForAccessibility,
  kBrowserSelectTabByIndex,
  kBrowserBottomPage,
  kBrowserTopPage,
  kBrowserNextPane,
  kBrowserPageUp,
  kBrowserPageDown,
  kBrowserRightClick,
  // Ambient action ids:
  kAmbientDragLinkInSameTab,
  kAmbientCycleForwardMRU,
  kAmbientDragLinkInNewTab,
  kAmbientOpenLinkInTab,
  kAmbientOpenLinkInTabBackground,
  kAmbientOpenLinkInWindow,
  kAmbientOpenPageInNewTab,
  kAmbientCycleBackwardMRU,
  kAmbientRightClick,
  kAmbientSaveLinkAsBookmark,
  kAmbientLaunchNumberedApp,
  kAmbientOpenFile,
  kAmbientOpenHighlightedItemOnShelf,
  kAmbientHighlightNextItemOnShelf,
  kAmbientHighlightPreviousItemOnShelf,
  kAmbientMoveAppsInGrid,
  kAmbientMoveAppsInOutFolder,
  kAmbientRemoveHighlightOnShelf,
  kAmbientActivateIndexedDesk,
  kAmbientLaunchAppByIndex,
  kAmbientDisplayHiddenFiles,
  kAmbientOpenRightClickMenu,
  kAmbientCaretBrowsing,
  // Six-pack-key action ids:
  kSixPackDelete,
  kSixPackHome,
  kSixPackPageUp,
  kSixPackEnd,
  kSixPackPageDown,
  kSixPackInsert,
};

// Used to separate text accelerator parts in the UI e.g ctrl + 1.
enum TextAcceleratorDelimiter {
  kPlusSign,
};

// Contains details for UI styling of an accelerator.
struct AcceleratorLayoutDetails {
  // The accelerator action id associated for a source. Concat `source` and
  // `action_id` to get a unique identifier for an accelerator action.
  uint32_t action_id;

  // String id of the accelerator's description.
  int description_string_id;

  // Category of the accelerator.
  mojom::AcceleratorCategory category;

  // Subcategory of the accelerator.
  mojom::AcceleratorSubcategory sub_category;

  // True if the accelerator cannot be modified through customization.
  // False if the accelerator can be modified through customization.
  bool locked;

  // The layout style of the accelerator, this provides additional context
  // on how to accelerator should be represented in the UI.
  mojom::AcceleratorLayoutStyle layout_style;

  // The source of which the accelerator is from.
  mojom::AcceleratorSource source;
};

// Represents a replacement for part of a non-configurable accelerator.
// Contains the text to display as well as its type (Modifier, Key, Plain Text)
// which is needed to determine how to display the text in the shortcut
// customization app.
class TextAcceleratorPart : public mojom::TextAcceleratorPart {
 public:
  explicit TextAcceleratorPart(ui::EventFlags modifier);
  explicit TextAcceleratorPart(ui::KeyboardCode key_code);
  explicit TextAcceleratorPart(const std::u16string& plain_text);
  explicit TextAcceleratorPart(TextAcceleratorDelimiter delimiter);
  TextAcceleratorPart(const TextAcceleratorPart&);
  TextAcceleratorPart& operator=(const TextAcceleratorPart&);
  ~TextAcceleratorPart();
};

// Contains info related to a non-configurable accelerator. A non-configurable
// accelerator can contain either a standard or text-based accelerator. The
// message_id and list of replacements will be provided when dealing
// with text-based accelerators; otherwise, accelerators will be provided
// and message_id/replacements should not have any value set.
// AcceleratorConfigurationProvider uses this struct to create a list of
// AcceleratorInfo struct's for each non-configurable action.
struct NonConfigurableAcceleratorDetails {
  NonConfigurableAcceleratorDetails(
      int message_id,
      std::vector<TextAcceleratorPart> replacements);
  explicit NonConfigurableAcceleratorDetails(int resource_id);
  explicit NonConfigurableAcceleratorDetails(
      std::vector<ui::Accelerator> accels);
  NonConfigurableAcceleratorDetails(const NonConfigurableAcceleratorDetails&);
  NonConfigurableAcceleratorDetails& operator=(
      const NonConfigurableAcceleratorDetails&);
  ~NonConfigurableAcceleratorDetails();

 public:
  bool IsStandardAccelerator() const { return accelerators.has_value(); }

  // These members are used for the Ambient action ids contained in
  // the NonConfigurableActions enum.
  absl::optional<int> message_id;
  absl::optional<std::vector<TextAcceleratorPart>> replacements;
  // This member is used for the Browser action ids contained in
  // the NonConfigurableActions enum.
  absl::optional<std::vector<ui::Accelerator>> accelerators;
};

using NonConfigurableActionsMap =
    std::map<NonConfigurableActions, NonConfigurableAcceleratorDetails>;

const NonConfigurableActionsMap& GetNonConfigurableActionsMap();

// A fixed array of accelerator layouts used for categorization and styling of
// accelerator actions. The ordering of the array is important and is used
// 1:1 for displaying shortcuts in the shortcut customization app.
// Adding an accelerator layout in this array will create a new entry in the
// app.
// TODO(jimmyxgong): This is a stub map with stub details, replace with real
// one when categorization is available.
constexpr AcceleratorLayoutDetails kAcceleratorLayouts[] = {
    // Tab & Windows.
    {DESKS_ACTIVATE_DESK_LEFT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_ACTIVATE_DESK_LEFT,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_ACTIVATE_DESK_RIGHT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_ACTIVATE_DESK_RIGHT,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_TOGGLE_ASSIGN_TO_ALL_DESKS,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_ACTIVATE_DESK_RIGHT,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserCloseTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLOSE_TAB,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserCloseWindow,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLOSE_WINDOW,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {DESKS_NEW_DESK, IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_NEW_DESK,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {WINDOW_CYCLE_SNAP_LEFT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_CYCLE_SNAP_LEFT,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {WINDOW_CYCLE_SNAP_RIGHT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_CYCLE_SNAP_RIGHT,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserSelectTabByIndex,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_GO_TO_TAB_IN_RANGE,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserSelectLastTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SELECT_LAST_TAB,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {TOGGLE_MAXIMIZED, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MAXIMIZED,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {WINDOW_MINIMIZE, IDS_ASH_ACCELERATOR_DESCRIPTION_WINDOW_MINIMIZE,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MOVE_ACTIVE_WINDOW_BETWEEN_DISPLAYS,
     IDS_ASH_ACCELERATOR_DESCRIPTION_MOVE_ACTIVE_WINDOW_BETWEEN_DISPLAYS,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_MOVE_ACTIVE_ITEM_LEFT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_MOVE_ACTIVE_ITEM_LEFT,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_MOVE_ACTIVE_ITEM_RIGHT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_MOVE_ACTIVE_ITEM_RIGHT,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_CROSH, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_CROSH,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserOpenFile,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_OPEN_FILE,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserNewIncognitoWindow,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEW_INCOGNITO_WINDOW,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserNewTab,
     IDS_ASH_ACCELERATOR_DESCRIPTION_NEW_TAB,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserNewWindow,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEW_WINDOW,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientDragLinkInNewTab,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DRAG_LINK_IN_NEW_TAB,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenLinkInTab,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_LINK_IN_TAB,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientDragLinkInSameTab,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DRAG_LINK_IN_SAME_TAB,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenLinkInWindow,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_LINK_IN_WINDOW,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenPageInNewTab,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_PAGE_IN_NEW_TAB,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientCycleBackwardMRU,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CYCLE_BACKWARD_MRU,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserRestoreTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_RESTORE_TAB,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {ROTATE_WINDOW, IDS_ASH_ACCELERATOR_DESCRIPTION_ROTATE_WINDOW,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserTabSearch,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_TAB_SEARCH,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {CYCLE_FORWARD_MRU, IDS_ASH_ACCELERATOR_DESCRIPTION_CYCLE_FORWARD_MRU,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TAKE_WINDOW_SCREENSHOT,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TAKE_WINDOW_SCREENSHOT,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_FULLSCREEN, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_FULLSCREEN,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_RESIZE_LOCK_MENU,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_RESIZE_LOCK_MENU,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {DESKS_REMOVE_CURRENT_DESK,
     IDS_ASH_ACCELERATOR_DESCRIPTION_DESKS_REMOVE_CURRENT_DESK,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {UNPIN, IDS_ASH_ACCELERATOR_DESCRIPTION_UNPIN,
     mojom::AcceleratorCategory::kTabsAndWindows,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},

    // Page and Web Browser.
    {NonConfigurableActions::kBrowserClearBrowsingData,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLEAR_BROWSING_DATA,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserCloseFindOrStop,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_CLOSE_FIND_OR_STOP,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusBookmarks,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_BOOKMARKS,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserBack,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_BACK,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserForward,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FORWARD,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserBottomPage,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOTTOM_PAGE,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFindNext,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND_NEXT,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFindPrevious,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND_PREVIOUS,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserTopPage,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_TOP_PAGE,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserNextPane,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_NEXT_PANE,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserPageDown,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_PAGE_DOWN,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserPageUp,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_PAGE_UP,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusSearch,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_SEARCH,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},

    {NonConfigurableActions::kBrowserShowDownloads,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_DOWNLOADS,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowHistory,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_HISTORY,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusMenuBar,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_MENU_BAR,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {FOCUS_PREVIOUS_PANE, IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_PREVIOUS_PANE,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserPrint,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_PRINT,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserReload,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_RELOAD,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserReloadBypassingCache,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_RELOAD_BYPASSING_CACHE,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserZoomNormal,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_NORMAL,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserRightClick,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_RIGHT_CLICK,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientSaveLinkAsBookmark,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_SAVE_LINK_AS_BOOKMARK,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserBookmarkAllTabs,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOOKMARK_ALL_TABS,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserSavePage,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SAVE_PAGE,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserBookmarkThisTab,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_BOOKMARK_THIS_TAB,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFind,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FIND,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowAppMenu,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_APP_MENU,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowBookmarkManager,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_BOOKMARK_MANAGER,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserDevToolsConsole,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS_CONSOLE,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserDevToolsInspect,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS_INSPECT,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserDevTools,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_DEV_TOOLS,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserShowBookmarkBar,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_SHOW_BOOKMARK_BAR,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserViewSource,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_VIEW_SOURCE,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserZoomPlus,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_PLUS,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserZoomMinus,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_ZOOM_MINUS,
     mojom::AcceleratorCategory::kPageAndWebBrowser,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},

    // System and display settings.
    {SHOW_TASK_MANAGER, IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_TASK_MANAGER,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientLaunchNumberedApp,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_LAUNCH_NUMBERED_APP,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {LAUNCH_LAST_APP, IDS_ASH_ACCELERATOR_DESCRIPTION_LAUNCH_LAST_APP,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SCALE_UI_DOWN, IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_DOWN,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SCALE_UI_UP, IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_UP,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientDisplayHiddenFiles,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_DISPLAY_HIDDEN_FILES,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {TOGGLE_MIRROR_MODE, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MIRROR_MODE,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {VOLUME_MUTE, IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_MUTE,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_DIAGNOSTICS, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_DIAGNOSTICS,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_GET_HELP, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_GET_HELP,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_FILE_MANAGER, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_FILE_MANAGER,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_SYSTEM_TRAY_BUBBLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_SYSTEM_TRAY_BUBBLE,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_APP_LIST, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_APP_LIST,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientOpenFile,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_FILE,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {SUSPEND, IDS_ASH_ACCELERATOR_DESCRIPTION_SUSPEND,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SCALE_UI_RESET, IDS_ASH_ACCELERATOR_DESCRIPTION_SCALE_UI_RESET,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {ROTATE_SCREEN, IDS_ASH_ACCELERATOR_DESCRIPTION_ROTATE_SCREEN,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_MESSAGE_CENTER_BUBBLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_MESSAGE_CENTER_BUBBLE,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SHOW_STYLUS_TOOLS, IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_STYLUS_TOOLS,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_IME_MENU_BUBBLE,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_IME_MENU_BUBBLE,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {EXIT, IDS_ASH_ACCELERATOR_DESCRIPTION_EXIT,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {OPEN_FEEDBACK_PAGE, IDS_ASH_ACCELERATOR_DESCRIPTION_OPEN_FEEDBACK_PAGE,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWAP_PRIMARY_DISPLAY, IDS_ASH_ACCELERATOR_DESCRIPTION_SWAP_PRIMARY_DISPLAY,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWITCH_TO_LAST_USED_IME,
     IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_LAST_USED_IME,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWITCH_TO_NEXT_IME, IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_NEXT_IME,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWITCH_TO_NEXT_USER, IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_NEXT_USER,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SWITCH_TO_PREVIOUS_USER,
     IDS_ASH_ACCELERATOR_DESCRIPTION_SWITCH_TO_PREVIOUS_USER,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {BRIGHTNESS_DOWN, IDS_ASH_ACCELERATOR_DESCRIPTION_BRIGHTNESS_DOWN,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {BRIGHTNESS_UP, IDS_ASH_ACCELERATOR_DESCRIPTION_BRIGHTNESS_UP,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {VOLUME_DOWN, IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_DOWN,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {VOLUME_UP, IDS_ASH_ACCELERATOR_DESCRIPTION_VOLUME_UP,
     mojom::AcceleratorCategory::kSystemAndDisplaySettings,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},

    // Text Editing.
    {KEYBOARD_BRIGHTNESS_DOWN,
     IDS_ASH_ACCELERATOR_DESCRIPTION_KEYBOARD_BRIGHTNESS_DOWN,
     mojom::AcceleratorCategory::kTextEditing,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {KEYBOARD_BRIGHTNESS_UP,
     IDS_ASH_ACCELERATOR_DESCRIPTION_KEYBOARD_BRIGHTNESS_UP,
     mojom::AcceleratorCategory::kTextEditing,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_CLIPBOARD_HISTORY,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_CLIPBOARD_HISTORY,
     mojom::AcceleratorCategory::kTextEditing,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {SHOW_EMOJI_PICKER, IDS_ASH_ACCELERATOR_DESCRIPTION_SHOW_EMOJI_PICKER,
     mojom::AcceleratorCategory::kTextEditing,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_CAPS_LOCK, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_CAPS_LOCK,
     mojom::AcceleratorCategory::kTextEditing,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},

    // Accessibility
    {FOCUS_SHELF, IDS_ASH_ACCELERATOR_DESCRIPTION_FOCUS_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kBrowserFocusToolbar,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_TOOLBAR,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kBrowserFocusInactivePopupForAccessibility,
     IDS_BROWSER_ACCELERATOR_DESCRIPTION_FOCUS_INACTIVE_POPUP_FOR_ACCESSIBILITY,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {TOGGLE_SPOKEN_FEEDBACK,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_SPOKEN_FEEDBACK,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_HIGH_CONTRAST, IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_HIGH_CONTRAST,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_DOCKED_MAGNIFIER,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_DOCKED_MAGNIFIER,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {TOGGLE_FULLSCREEN_MAGNIFIER,
     IDS_ASH_ACCELERATOR_DESCRIPTION_TOGGLE_FULLSCREEN_MAGNIFIER,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemControls,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MAGNIFIER_ZOOM_IN, IDS_ASH_ACCELERATOR_DESCRIPTION_MAGNIFIER_ZOOM_IN,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {MAGNIFIER_ZOOM_OUT, IDS_ASH_ACCELERATOR_DESCRIPTION_MAGNIFIER_ZOOM_OUT,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAsh},
    {NonConfigurableActions::kAmbientHighlightNextItemOnShelf,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_HIGHLIGHT_NEXT_ITEM_ON_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientHighlightPreviousItemOnShelf,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_HIGHTLIGHT_PREVIOUS_ITEM_ON_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kGeneral,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenRightClickMenu,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_RIGHT_CLICK_MENU,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientOpenHighlightedItemOnShelf,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_OPEN_HIGHLIGHTED_ITEM_ON_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientRemoveHighlightOnShelf,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_REMOVE_HIGHLIGHT_ON_SHELF,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kText,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kAmbientCaretBrowsing,
     IDS_AMBIENT_ACCELERATOR_DESCRIPTION_CARET_BROWSING,
     mojom::AcceleratorCategory::kAccessibility,
     mojom::AcceleratorSubcategory::kSystemApps,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kSixPackDelete,
     IDS_EVENT_REWRITER_ACCELERATOR_SIX_PACK_DELETE_DESCRIPTION,
     mojom::AcceleratorCategory::kEventRewriter,
     mojom::AcceleratorSubcategory::kSixPackKeys,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kSixPackHome,
     IDS_EVENT_REWRITER_ACCELERATOR_SIX_PACK_HOME_DESCRIPTION,
     mojom::AcceleratorCategory::kEventRewriter,
     mojom::AcceleratorSubcategory::kSixPackKeys,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kSixPackPageUp,
     IDS_EVENT_REWRITER_ACCELERATOR_SIX_PACK_PAGE_UP_DESCRIPTION,
     mojom::AcceleratorCategory::kEventRewriter,
     mojom::AcceleratorSubcategory::kSixPackKeys,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kSixPackEnd,
     IDS_EVENT_REWRITER_ACCELERATOR_SIX_PACK_END_DESCRIPTION,
     mojom::AcceleratorCategory::kEventRewriter,
     mojom::AcceleratorSubcategory::kSixPackKeys,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kSixPackPageDown,
     IDS_EVENT_REWRITER_ACCELERATOR_SIX_PACK_PAGE_DOWN_DESCRIPTION,
     mojom::AcceleratorCategory::kEventRewriter,
     mojom::AcceleratorSubcategory::kSixPackKeys,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
    {NonConfigurableActions::kSixPackInsert,
     IDS_EVENT_REWRITER_ACCELERATOR_SIX_PACK_INSERT_DESCRIPTION,
     mojom::AcceleratorCategory::kEventRewriter,
     mojom::AcceleratorSubcategory::kSixPackKeys,
     /*locked=*/true, mojom::AcceleratorLayoutStyle::kDefault,
     mojom::AcceleratorSource::kAmbient},
};
}  // namespace ash

#endif  // ASH_WEBUI_SHORTCUT_CUSTOMIZATION_UI_BACKEND_ACCELERATOR_LAYOUT_TABLE_H_
