// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.browserservices.intents;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Map;

/**
 * Stores information about a web app and the previous information of the web app, and can be
 * conditionally configured to return either information. This is useful because when updating a
 * webapp, it is possible that the new WebAppInfo object created for the update contains a request
 * to alter element(s) of the app identity (name and/or icons) that are not permitted to change.
 * This object therefore keeps track of the previous values for the currently installed webapp, so
 * that they can be returned to callers in order for the rest of the (non-App Identity) update
 * requests to be fulfilled.
 */
public class MergedWebappInfo extends WebappInfo {
    // The old WebappInfo values (typically the current values for a webapp being upgraded).
    private WebappInfo mOldWebappInfo;

    // Whether to override the name and shortName information returned to callers.
    private boolean mUseOldName;

    // Whether to override the icon information returned to callers.
    private boolean mUseOldIcons;

    public static MergedWebappInfo create(@NonNull WebappInfo oldWebappInfo,
            @Nullable BrowserServicesIntentDataProvider provider) {
        return (provider == null) ? null : new MergedWebappInfo(oldWebappInfo, provider);
    }

    public static MergedWebappInfo createForTesting(
            @NonNull WebappInfo oldWebappInfo, @NonNull WebappInfo newWebappInfo) {
        return new MergedWebappInfo(oldWebappInfo, newWebappInfo.getProvider());
    }

    private MergedWebappInfo(@NonNull WebappInfo oldWebappInfo,
            @NonNull BrowserServicesIntentDataProvider provider) {
        super(provider);
        mOldWebappInfo = oldWebappInfo;
    }

    /**
     * Sets whether to use the override values when webapp name/shortName are requested.
     */
    public void setUseOldName(boolean enabled) {
        mUseOldName = enabled;
    }

    /**
     * Sets whether to use the override values when webapp icon information is requested.
     */
    public void setUseOldIcon(boolean enabled) {
        mUseOldIcons = enabled;
    }

    @Override
    public String name() {
        if (mUseOldName) {
            return mOldWebappInfo.getWebappExtras().name;
        }
        return super.name();
    }

    @Override
    public String shortName() {
        if (mUseOldName) {
            return mOldWebappInfo.getWebappExtras().shortName;
        }
        return super.shortName();
    }

    @Override
    @NonNull
    public WebappIcon icon() {
        if (mUseOldIcons) {
            return mOldWebappInfo.getWebappExtras().icon;
        }
        return super.icon();
    }

    /**
     * Returns whether the {@link #icon} should be used as an Android Adaptive Icon.
     */
    @Override
    public boolean isIconAdaptive() {
        if (mUseOldIcons) {
            return mOldWebappInfo.getWebappExtras().isIconAdaptive;
        }
        return super.isIconAdaptive();
    }

    /**
     * Returns whether the icon was generated by Chromium.
     */
    @Override
    public boolean isIconGenerated() {
        if (mUseOldIcons) {
            return mOldWebappInfo.getWebappExtras().isIconGenerated;
        }
        return super.isIconGenerated();
    }

    @Override
    @NonNull
    public Map<String, String> iconUrlToMurmur2HashMap() {
        if (mUseOldIcons) {
            return mOldWebappInfo.getWebApkExtras().iconUrlToMurmur2HashMap;
        }
        return super.iconUrlToMurmur2HashMap();
    }

    @Override
    public String appKey() {
        return mOldWebappInfo.appKey();
    }
}
