// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/wm/desks/templates/saved_desk_util.h"

#include "ash/constants/ash_features.h"
#include "ash/constants/ash_pref_names.h"
#include "ash/public/cpp/session/session_controller.h"
#include "ash/public/cpp/session/session_types.h"
#include "ash/session/session_controller_impl.h"
#include "ash/shell.h"
#include "ash/wm/overview/overview_controller.h"
#include "ash/wm/overview/overview_session.h"
#include "components/prefs/pref_registry_simple.h"
#include "components/prefs/pref_service.h"

namespace ash {
namespace {

PrefService* GetPrimaryUserPrefService() {
  return Shell::Get()->session_controller()->GetPrimaryUserPrefService();
}

bool IsGuestSession() {
  // User 0 is the current user.
  const UserIndex user_index = 0;
  const UserSession* const user_session =
      Shell::Get()->session_controller()->GetUserSession(user_index);
  return user_session->user_info.type == user_manager::USER_TYPE_GUEST;
}

}  // namespace

namespace saved_desk_util {

void RegisterProfilePrefs(PrefRegistrySimple* registry) {
  registry->RegisterBooleanPref(prefs::kDeskTemplatesEnabled, false);
}

bool AreDesksTemplatesEnabled() {
  if (IsGuestSession())
    return false;

  if (PrefService* pref_service = GetPrimaryUserPrefService()) {
    const PrefService::Preference* desk_templates_pref =
        pref_service->FindPreference(prefs::kDeskTemplatesEnabled);
    DCHECK(desk_templates_pref);

    if (desk_templates_pref->IsManaged()) {
      // Let policy settings override flags configuration.
      return pref_service->GetBoolean(prefs::kDeskTemplatesEnabled);
    }
  }

  // Allow the feature to be enabled by user when there is not explicit policy.
  return features::AreDesksTemplatesEnabled();
}

bool IsDeskSaveAndRecallEnabled() {
  if (IsGuestSession())
    return false;

  return features::IsSavedDesksEnabled();
}

bool IsSavedDesksEnabled() {
  return AreDesksTemplatesEnabled() || IsDeskSaveAndRecallEnabled();
}

SavedDeskDialogController* GetSavedDeskDialogController() {
  auto* overview_session =
      Shell::Get()->overview_controller()->overview_session();
  DCHECK(overview_session);
  SavedDeskDialogController* controller =
      overview_session->saved_desk_dialog_controller();
  DCHECK(controller);
  return controller;
}

SavedDeskPresenter* GetSavedDeskPresenter() {
  auto* overview_session =
      Shell::Get()->overview_controller()->overview_session();
  DCHECK(overview_session);
  SavedDeskPresenter* presenter = overview_session->saved_desk_presenter();
  DCHECK(presenter);
  return presenter;
}

}  // namespace saved_desk_util
}  // namespace ash
