// Copyright (c) 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_CHROME_CLEANER_HTTP_INTERNET_HELPERS_H_
#define CHROME_CHROME_CLEANER_HTTP_INTERNET_HELPERS_H_

#include <Windows.h>  // NOLINT

#include <stdint.h>

#include <map>
#include <string>
#include "base/strings/string16.h"

namespace chrome_cleaner {

// Parses the value of a Content-Type header.
// @param content_type_str The value of a Content-Type HTTP header (without the
//     label or ':').
// @param mime_type Receives the specified mime-type, if any, in lower-case.
//     Unmodified otherwise.
// @param charset Receives the specified charset, if any, in lower-case.
//     Unmodified otherwise.
// @param boundary Optional. Receives the (quoted) value of the boundary
//     parameter, if any. Unmodified otherwise.
void ParseContentType(const base::string16& content_type_str,
                      base::string16* mime_type,
                      base::string16* charset,
                      bool* had_charset,
                      base::string16* boundary);

// Parses an URL.
// @param url The URL to parse.
// @param scheme Receives the parsed scheme.
// @param host Receives the parsed host.
// @param port Receives the parsed port (or the implicit default port).
// @param path Receives the parsed path.
// @returns true if the URL is successfully parsed.
bool DecomposeUrl(const base::string16& url,
                  base::string16* scheme,
                  base::string16* host,
                  uint16_t* port,
                  base::string16* path);

// Composes an HTTP or HTTPS URL.
// @param host The URL host component.
// @param port The URL port component.
// @param path The URL path component.
// @returns The composed URL.
base::string16 ComposeUrl(const base::string16& host,
                          uint16_t port,
                          const base::string16& path,
                          bool secure);

// @returns A random string to be used as a multipart MIME message boundary.
base::string16 GenerateMultipartHttpRequestBoundary();

// Generates an appropriate Content-Type header (starting with "Content-Type:")
// for a multi-part HTTP message.
// @param boundary The MIME boundary to use.
// @returns An HTTP Content-Type header suitable for the multipart message
//     generated by GenerateMultipartHttpRequestBody.
base::string16 GenerateMultipartHttpRequestContentTypeHeader(
    const base::string16 boundary);

// Generates a multipart HTTP message body.
// @param parameters HTTP request parameters to be encoded in the body.
// @param upload_file File contents to be encoded in the body.
// @param file_part_name The parameter name to be assigned to the file part.
// @param boundary The MIME boundary to use.
// @returns A multipart HTTP message body.
std::string GenerateMultipartHttpRequestBody(
    const std::map<base::string16, base::string16>& parameters,
    const std::string& upload_file,
    const base::string16& file_part_name,
    const base::string16& boundary);

}  // namespace chrome_cleaner

#endif  // CHROME_CHROME_CLEANER_HTTP_INTERNET_HELPERS_H_
