// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/ui/focus_ring_layer.h"

#include "ui/aura/window.h"
#include "ui/compositor/compositor_animation_observer.h"
#include "ui/compositor/layer.h"
#include "ui/compositor/paint_recorder.h"
#include "ui/display/display.h"
#include "ui/display/screen.h"
#include "ui/gfx/canvas.h"

namespace ui {
class Compositor;
}

namespace chromeos {

namespace {

const int kShadowRadius = 10;
const int kShadowAlpha = 90;
const SkColor kShadowColor = SkColorSetRGB(77, 144, 254);

}  // namespace

FocusRingLayerDelegate::~FocusRingLayerDelegate() {}

FocusRingLayer::FocusRingLayer(FocusRingLayerDelegate* delegate)
    : delegate_(delegate), root_window_(nullptr), compositor_(nullptr) {}

FocusRingLayer::~FocusRingLayer() {
  if (compositor_ && compositor_->HasAnimationObserver(this))
    compositor_->RemoveAnimationObserver(this);
}

void FocusRingLayer::Set(aura::Window* root_window, const gfx::Rect& bounds) {
  focus_ring_ = bounds;
  gfx::Rect layer_bounds = bounds;
  int inset = -(kShadowRadius + 2);
  layer_bounds.Inset(inset, inset, inset, inset);
  CreateOrUpdateLayer(root_window, "FocusRing", layer_bounds);
}

bool FocusRingLayer::CanAnimate() const {
  return compositor_ && compositor_->HasAnimationObserver(this);
}

void FocusRingLayer::SetOpacity(float opacity) {
  layer()->SetOpacity(opacity);
}

void FocusRingLayer::CreateOrUpdateLayer(aura::Window* root_window,
                                         const char* layer_name,
                                         const gfx::Rect& bounds) {
  if (!layer_ || root_window != root_window_) {
    root_window_ = root_window;
    ui::Layer* root_layer = root_window->layer();
    layer_.reset(new ui::Layer(ui::LAYER_TEXTURED));
    layer_->set_name(layer_name);
    layer_->set_delegate(this);
    layer_->SetFillsBoundsOpaquely(false);
    root_layer->Add(layer_.get());
  }

  // Keep moving it to the top in case new layers have been added
  // since we created this layer.
  layer_->parent()->StackAtTop(layer_.get());

  layer_->SetBounds(bounds);

  // Update the animation observer.
  display::Display display =
      display::Screen::GetScreen()->GetDisplayMatching(bounds);
  ui::Compositor* compositor = root_window->layer()->GetCompositor();
  if (compositor != compositor_) {
    if (compositor_ && compositor_->HasAnimationObserver(this))
      compositor_->RemoveAnimationObserver(this);
    compositor_ = compositor;
    if (compositor_ && !compositor_->HasAnimationObserver(this))
      compositor_->AddAnimationObserver(this);
  }
}

void FocusRingLayer::OnPaintLayer(const ui::PaintContext& context) {
  if (!root_window_ || focus_ring_.IsEmpty())
    return;

  ui::PaintRecorder recorder(context, layer_->size());

  SkPaint paint;
  paint.setColor(kShadowColor);
  paint.setFlags(SkPaint::kAntiAlias_Flag);
  paint.setStyle(SkPaint::kStroke_Style);
  paint.setStrokeWidth(2);

  gfx::Rect bounds = focus_ring_ - layer_->bounds().OffsetFromOrigin();
  int r = kShadowRadius;
  for (int i = 0; i < r; i++) {
    // Fade out alpha quadratically.
    paint.setAlpha((kShadowAlpha * (r - i) * (r - i)) / (r * r));
    gfx::Rect outsetRect = bounds;
    outsetRect.Inset(-i, -i, -i, -i);
    recorder.canvas()->DrawRect(outsetRect, paint);
  }
}

void FocusRingLayer::OnDelegatedFrameDamage(
    const gfx::Rect& damage_rect_in_dip) {
}

void FocusRingLayer::OnDeviceScaleFactorChanged(float device_scale_factor) {
  if (delegate_)
    delegate_->OnDeviceScaleFactorChanged();
}

void FocusRingLayer::OnAnimationStep(base::TimeTicks timestamp) {
  delegate_->OnAnimationStep(timestamp);
}

void FocusRingLayer::OnCompositingShuttingDown(ui::Compositor* compositor) {
  if (compositor == compositor_) {
    compositor->RemoveAnimationObserver(this);
    compositor_ = nullptr;
  }
}

}  // namespace chromeos
