// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.customtabs;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.test.suitebuilder.annotation.LargeTest;
import android.test.suitebuilder.annotation.MediumTest;
import android.util.Base64;

import org.chromium.base.ActivityState;
import org.chromium.base.ApplicationStatus;
import org.chromium.base.ThreadUtils;
import org.chromium.base.test.util.CommandLineFlags;
import org.chromium.base.test.util.Feature;
import org.chromium.chrome.browser.ChromeSwitches;
import org.chromium.chrome.browser.customtabs.CustomTabDelegateFactory.CustomTabNavigationDelegate;
import org.chromium.chrome.browser.document.ChromeLauncherActivity;
import org.chromium.chrome.browser.externalnav.ExternalNavigationHandler.OverrideUrlLoadingResult;
import org.chromium.chrome.browser.preferences.ChromePreferenceManager;
import org.chromium.chrome.browser.tab.InterceptNavigationDelegateImpl;
import org.chromium.chrome.browser.tab.Tab;
import org.chromium.chrome.browser.tab.TabDelegateFactory;
import org.chromium.content.browser.test.util.Criteria;
import org.chromium.content.browser.test.util.CriteriaHelper;
import org.chromium.net.test.EmbeddedTestServer;

import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Tests for external navigation handling of Custom Tabs generated by Chrome.
 */
@CommandLineFlags.Add(ChromeSwitches.HERB_FLAVOR_ELDERBERRY_SWITCH)
public class CustomTabFromChromeExternalNavigationTest extends CustomTabActivityTestBase {

    private EmbeddedTestServer mTestServer;

    @Override
    public void setUp() throws Exception {
        mTestServer = EmbeddedTestServer.createAndStartServer(getInstrumentation().getContext());
        super.setUp();

        ChromePreferenceManager.getInstance(getInstrumentation().getTargetContext())
                .setCachedHerbFlavor(ChromeSwitches.HERB_FLAVOR_ELDERBERRY);
    }

    @Override
    public void tearDown() throws Exception {
        mTestServer.stopAndDestroyServer();
        super.tearDown();
    }

    @Override
    protected void startActivityCompletely(Intent intent) {
        Activity activity = getInstrumentation().startActivitySync(intent);
        assertTrue(activity instanceof CustomTabActivity);
        setActivity(activity);
    }

    private Intent getCustomTabFromChromeIntent(final String url) {
        return ThreadUtils.runOnUiThreadBlockingNoException(new Callable<Intent>() {
            @Override
            public Intent call() throws Exception {
                Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                return ChromeLauncherActivity.createCustomTabActivityIntent(
                        getInstrumentation().getTargetContext(), intent, true);
            }
        });

    }

    private void startCustomTabFromChrome(String url) throws InterruptedException {
        startCustomTabActivityWithIntent(getCustomTabFromChromeIntent(url));
    }

    @Feature("CustomTabFromChrome")
    @MediumTest
    public void testUsingStandardExternalNavigationHandler() throws Exception {
        startCustomTabFromChrome("about:blank");

        Tab tab = getActivity().getActivityTab();
        TabDelegateFactory delegateFactory = tab.getDelegateFactory();
        assertTrue(delegateFactory instanceof CustomTabDelegateFactory);
        CustomTabDelegateFactory customTabDelegateFactory =
                ((CustomTabDelegateFactory) delegateFactory);
        assertFalse(customTabDelegateFactory.getExternalNavigationDelegate()
                instanceof CustomTabNavigationDelegate);
    }

    @Feature("CustomTabFromChrome")
    @LargeTest
    public void testIntentWithRedirectToApp() throws Exception {
        final String redirectUrl = "https://maps.google.com/maps?q=1600+amphitheatre+parkway";
        final String initialUrl = mTestServer.getURL(
                "/chrome/test/data/android/redirect/js_redirect.html"
                + "?replace_text="
                + Base64.encodeToString("PARAM_URL".getBytes("utf-8"), Base64.URL_SAFE) + ":"
                + Base64.encodeToString(redirectUrl.getBytes("utf-8"), Base64.URL_SAFE));

        startActivityCompletely(getCustomTabFromChromeIntent(initialUrl));

        final AtomicReference<InterceptNavigationDelegateImpl> navigationDelegate =
                new AtomicReference<>();
        CriteriaHelper.pollUiThread(new Criteria("Tab never selected/initialized.") {
            @Override
            public boolean isSatisfied() {
                Tab tab = getActivity().getActivityTab();
                if (tab == null || tab.getInterceptNavigationDelegate() == null) return false;
                navigationDelegate.set(tab.getInterceptNavigationDelegate());
                return true;
            }
        });

        CriteriaHelper.pollUiThread(Criteria.equals(
                OverrideUrlLoadingResult.OVERRIDE_WITH_EXTERNAL_INTENT,
                new Callable<OverrideUrlLoadingResult>() {
                    @Override
                    public OverrideUrlLoadingResult call() throws Exception {
                        return navigationDelegate.get().getLastOverrideUrlLoadingResultForTests();
                    }
                }));

        CriteriaHelper.pollUiThread(new Criteria() {
            @Override
            public boolean isSatisfied() {
                int activityState = ApplicationStatus.getStateForActivity(getActivity());
                return activityState == ActivityState.STOPPED
                        || activityState == ActivityState.DESTROYED;
            }
        });
    }
}
