/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Ajax.org Code Editor (ACE).
 *
 * The Initial Developer of the Original Code is
 * Ajax.org B.V.
 * Portions created by the Initial Developer are Copyright (C) 2010
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *      Fabian Jakobs <fabian AT ajax DOT org>
 *      Julian Viereck <julian.viereck@gmail.com>
 *      Harutyun Amirjanyan <amirjanyan@gmail.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

define(function(require, exports, module) {
"use strict";

var keyUtil  = require("../lib/keys");
var event = require("../lib/event");
require("../commands/default_commands");

var KeyBinding = function(editor) {
    this.$editor = editor;
    this.$data = { };
    this.$handlers = [this];
};

(function() {
    this.setKeyboardHandler = function(keyboardHandler) {
        if (this.$handlers[this.$handlers.length - 1] == keyboardHandler)
            return;
        this.$data = { };
        this.$handlers = keyboardHandler ? [this, keyboardHandler] : [this];
    };

    this.addKeyboardHandler = function(keyboardHandler) {
        this.removeKeyboardHandler(keyboardHandler);
        this.$handlers.push(keyboardHandler);
    };

    this.removeKeyboardHandler = function(keyboardHandler) {
        var i = this.$handlers.indexOf(keyboardHandler);
        if (i == -1)
            return false;
        this.$handlers.splice(i, 1);
        return true;
    };

    this.getKeyboardHandler = function() {
        return this.$handlers[this.$handlers.length - 1];
    };

    this.$callKeyboardHandlers = function (hashId, keyString, keyCode, e) {
        var toExecute;
        for (var i = this.$handlers.length; i--;) {
            toExecute = this.$handlers[i].handleKeyboard(
                this.$data, hashId, keyString, keyCode, e
            );
            if (toExecute && toExecute.command)
                break;
        }

        if (!toExecute || !toExecute.command)
            return false;

        var success = false;
        var commands = this.$editor.commands;

        // allow keyboardHandler to consume keys
        if (toExecute.command != "null")
            success = commands.exec(toExecute.command, this.$editor, toExecute.args);
        else
            success = true;

        if (success && e)
            event.stopEvent(e);

        return success;
    };

    this.handleKeyboard = function(data, hashId, keyString) {
        return {
            command: this.$editor.commands.findKeyCommand(hashId, keyString)
        };
    };

    this.onCommandKey = function(e, hashId, keyCode) {
        var keyString = keyUtil.keyCodeToString(keyCode);
        this.$callKeyboardHandlers(hashId, keyString, keyCode, e);
    };

    this.onTextInput = function(text, pasted) {
        var success = false;
        if (!pasted && text.length == 1)
            success = this.$callKeyboardHandlers(0, text);
        if (!success)
            this.$editor.commands.exec("insertstring", this.$editor, text);
    };

}).call(KeyBinding.prototype);

exports.KeyBinding = KeyBinding;
});
