// SPDX-License-Identifier: LicenseRef-GPL-3.0-or-later-OpenSSL

#include <munit.h>

#include <chiaki/ecdh.h>
#include <chiaki/gkcrypt.h>

static MunitResult test_ecdh(const MunitParameter params[], void *user)
{
	static const uint8_t handshake_key[] = { 0xfc, 0x5d, 0x4b, 0xa0, 0x3a, 0x35, 0x3a, 0xbb, 0x6a, 0x7f, 0xac, 0x79, 0x1b, 0x17, 0xbb, 0x34 };
	static const uint8_t local_private_key[] = { 0x16, 0xe7, 0x5d, 0xcb, 0xda, 0x98, 0x55, 0xfb, 0x6b, 0xef, 0xdd, 0x8a, 0xa5, 0xf1, 0x6e, 0x7f, 0x46, 0xfd, 0xe1, 0xd2, 0x27, 0x97, 0x3, 0x60, 0x18, 0x72, 0xd8, 0x4b, 0x15, 0x38, 0xd9, 0x0 };
	static const uint8_t local_public_key[] = { 0x4, 0xf4, 0xa, 0xf1, 0x35, 0xa4, 0x88, 0x94, 0x36, 0xce, 0xe5, 0x2b, 0x5c, 0x73, 0xa3, 0x3e, 0xc5, 0xad, 0xb, 0xe0, 0x95, 0x2f, 0x57, 0xf4, 0xf0, 0xed, 0xc, 0x80, 0xb0, 0xbe, 0xda, 0x7c, 0xa6, 0x43, 0x78, 0x93, 0x93, 0xa5, 0x94, 0x7e, 0x9f, 0xaa, 0x3f, 0x67, 0x95, 0xc9, 0xaa, 0x9, 0xa9, 0x63, 0x25, 0xdf, 0xe8, 0x50, 0xbf, 0xc3, 0xf1, 0xdb, 0x62, 0xa5, 0xa, 0xbf, 0xb0, 0xff, 0xf7 };
	static const uint8_t local_public_key_sig[] = { 0x99, 0xb5, 0xcb, 0xb5, 0x37, 0x18, 0xb, 0xfc, 0x55, 0xda, 0x43, 0x7f, 0x44, 0x76, 0xa8, 0x17, 0xc9, 0x37, 0xfe, 0x56, 0x1b, 0x8a, 0xbe, 0xc, 0x41, 0x12, 0xab, 0x71, 0xf5, 0xa6, 0x8d, 0x29 };
	static const uint8_t remote_public_key[] = { 0x4, 0xdf, 0xef, 0x8, 0xbb, 0xa8, 0x56, 0xf2, 0xb4, 0x4b, 0x8a, 0xe, 0x4f, 0x44, 0x20, 0x3f, 0x8e, 0x49, 0x3f, 0xee, 0xd4, 0x3c, 0xe9, 0x3a, 0xfe, 0x5c, 0x64, 0x67, 0x77, 0x20, 0x15, 0x7c, 0x59, 0x10, 0x15, 0x67, 0x94, 0xae, 0x5f, 0x2, 0x4a, 0xad, 0xc, 0xce, 0xfa, 0x14, 0x15, 0xa, 0xab, 0xee, 0x8, 0xb, 0x14, 0x12, 0x76, 0xea, 0x3e, 0xc0, 0xd5, 0x65, 0xf4, 0x68, 0x77, 0xa3, 0xca };
	static const uint8_t remote_public_key_sig[] = { 0x13, 0xc5, 0x89, 0xe2, 0x3b, 0x72, 0x85, 0x24, 0xa9, 0x9f, 0x96, 0x80, 0x3, 0xa1, 0x81, 0x30, 0x59, 0x68, 0xf1, 0xbb, 0xb6, 0x4d, 0xc4, 0xa7, 0x6c, 0xce, 0xf6, 0x79, 0x4c, 0xeb, 0x2d, 0x98 };
	static const uint8_t secret[] = { 0xb8, 0x1c, 0x61, 0x46, 0xe7, 0x49, 0x73, 0x8c, 0x96, 0x30, 0xca, 0x13, 0xff, 0x71, 0xe5, 0x9b, 0x3b, 0xf9, 0x41, 0x98, 0xd4, 0x67, 0xa5, 0xa2, 0xbc, 0x78, 0x4, 0x92, 0x81, 0x43, 0xec, 0x1d };

	ChiakiECDH ecdh;
	ChiakiErrorCode err = chiaki_ecdh_init(&ecdh);
	if(err != CHIAKI_ERR_SUCCESS)
		return MUNIT_ERROR;

	err = chiaki_ecdh_set_local_key(&ecdh, local_private_key, sizeof(local_private_key), local_public_key, sizeof(local_public_key));
	if(err != CHIAKI_ERR_SUCCESS)
		return MUNIT_ERROR;

	uint8_t local_public_key_result[128];
	size_t local_public_key_result_size = sizeof(local_public_key_result);
	uint8_t local_public_key_sig_result[32];
	size_t local_public_key_sig_result_size = sizeof(local_public_key_sig_result);
	chiaki_ecdh_get_local_pub_key(&ecdh, local_public_key_result, &local_public_key_result_size, handshake_key, local_public_key_sig_result, &local_public_key_sig_result_size);

	munit_assert_size(local_public_key_result_size, ==, sizeof(local_public_key));
	munit_assert_memory_equal(sizeof(local_public_key), local_public_key_result, local_public_key);

	munit_assert_size(local_public_key_sig_result_size, ==, sizeof(local_public_key_sig));
	munit_assert_memory_equal(sizeof(local_public_key_sig), local_public_key_sig_result, local_public_key_sig);

	uint8_t secret_result[128];
	chiaki_ecdh_derive_secret(&ecdh, secret_result,
			remote_public_key, sizeof(remote_public_key),
			handshake_key,
			remote_public_key_sig, sizeof(remote_public_key_sig));

	munit_assert_size(CHIAKI_ECDH_SECRET_SIZE, ==, sizeof(secret));
	munit_assert_memory_equal(sizeof(secret), secret_result, secret);

	chiaki_ecdh_fini(&ecdh);

	return MUNIT_OK;
}


static MunitResult test_key_stream(const MunitParameter params[], void *user)
{
	static const uint8_t handshake_key[] = { 0x83, 0xcf, 0x93, 0x1a, 0x6a, 0xa7, 0x69, 0xa6, 0xc4, 0x48, 0x5d, 0x19, 0xc1, 0x5c, 0xcc, 0x52 };
	static const uint8_t ecdh_secret[] = { 0x73, 0xc8, 0xd5, 0x49, 0xc4, 0xd9, 0xdb, 0x50, 0x2e, 0xc0, 0x44, 0xea, 0x33, 0x64, 0x8c, 0x6a, 0xc9, 0xf3, 0x6c, 0x41, 0xb6, 0xa0, 0x50, 0x4f, 0xe0, 0x93, 0xde, 0xfb, 0x61, 0x9b, 0x9, 0x73 };
	static const uint8_t gkcrypt_key[] = { 0x8, 0x81, 0x6f, 0xa2, 0xe5, 0x55, 0x89, 0x61, 0xd5, 0xa2, 0x86, 0xd9, 0xe, 0xec, 0x5b, 0x8c };
	static const uint8_t gkcrypt_iv[] = { 0x2a, 0xe1, 0xbb, 0x3d, 0x84, 0xdc, 0x9a, 0xa9, 0xc3, 0x52, 0xa4, 0xcf, 0x3f, 0xfb, 0x8b, 0x72 };
	static const uint8_t key_stream[] = { 0xf, 0x6d, 0x89, 0x85, 0x5b, 0xa7, 0x86, 0x74, 0x5b, 0xa1, 0xfe, 0x5c, 0x81, 0x19, 0x6c, 0xd5, 0x54, 0xc4, 0x1c, 0xca, 0xf6, 0xe9, 0x34, 0xa4, 0x89, 0x26, 0x98, 0xb0, 0x62, 0x12, 0xb3, 0x1a };

	ChiakiLog log;

	ChiakiGKCrypt gkcrypt;
	ChiakiErrorCode err = chiaki_gkcrypt_init(&gkcrypt, &log, 0, 42, handshake_key, ecdh_secret);
	if(err != CHIAKI_ERR_SUCCESS)
		return MUNIT_ERROR;

	munit_assert_memory_equal(sizeof(gkcrypt_key), gkcrypt.key_base, gkcrypt_key);
	munit_assert_memory_equal(sizeof(gkcrypt_iv), gkcrypt.iv, gkcrypt_iv);

	uint8_t key_stream_result[0x20];
	err = chiaki_gkcrypt_gen_key_stream(&gkcrypt, 0x30, key_stream_result, sizeof(key_stream_result));
	if(err != CHIAKI_ERR_SUCCESS)
	{
		chiaki_gkcrypt_fini(&gkcrypt);
		return MUNIT_ERROR;
	}

	munit_assert_memory_equal(sizeof(key_stream), key_stream_result, key_stream);

	chiaki_gkcrypt_fini(&gkcrypt);
	return MUNIT_OK;
}


static MunitResult test_endecrypt(const MunitParameter params[], void *user)
{
	static const uint8_t handshake_key[] = { 0x14, 0xf1, 0xe6, 0x94, 0x6c, 0x5d, 0xce, 0xa8, 0xb7, 0xaa, 0x48, 0x50, 0xf6, 0x4d, 0x21, 0xac };
	static const uint8_t ecdh_secret[] = { 0xc, 0xeb, 0x77, 0x9, 0x83, 0x4d, 0x7a, 0xfc, 0x50, 0xb8, 0x46, 0x8c, 0xc6, 0x3c, 0x1e, 0x7c, 0x4e, 0x4a, 0x88, 0x93, 0x42, 0x80, 0xc1, 0x28, 0xe6, 0x1e, 0xe9, 0xd4, 0x1b, 0x8c, 0x69, 0x36 };
	static const uint8_t gkcrypt_key[] = { 0x27, 0x7a, 0xa5, 0x1d, 0xac, 0xd1, 0x5f, 0xe, 0x54, 0x12, 0xfa, 0xce, 0xd, 0xc4, 0x63, 0x6a };
	static const uint8_t gkcrypt_iv[] = { 0xef, 0x20, 0x40, 0xc2, 0x15, 0x3c, 0x2, 0x66, 0x32, 0x1f, 0x42, 0xbb, 0xf4, 0x50, 0x34, 0x4d };
	static const uint8_t clear_data[] = { 0x4e, 0x61, 0x9f, 0x94, 0x5d, 0x4b, 0x8e, 0xbd, 0x2a, 0x15, 0x4d, 0x3, 0x6a, 0xcd, 0x49, 0x56, 0x9c, 0xc7, 0x5c, 0xe3, 0xe7, 0x0, 0x17, 0x9a, 0x38, 0xd9, 0x69, 0x53, 0x45, 0xf9, 0xc, 0xb5, 0x8c, 0x5, 0x65, 0xf, 0x70 };
	static const uint8_t enc_data[] = { 0x23, 0xf4, 0x8d, 0xd8, 0xaa, 0xf9, 0x58, 0x9b, 0xb1, 0x94, 0x4f, 0xad, 0x2b, 0x8d, 0xaa, 0x8d, 0x25, 0x88, 0xfa, 0xf8, 0xb6, 0xd4, 0x17, 0xf4, 0x5f, 0x78, 0xec, 0xf5, 0x4e, 0x37, 0x20, 0xb0, 0x76, 0x81, 0x7, 0x67, 0x9a };

	ChiakiLog log;

	ChiakiGKCrypt gkcrypt;
	ChiakiErrorCode err = chiaki_gkcrypt_init(&gkcrypt, &log, 0, 42, handshake_key, ecdh_secret);
	if(err != CHIAKI_ERR_SUCCESS)
		return MUNIT_ERROR;

	munit_assert_memory_equal(sizeof(gkcrypt_key), gkcrypt.key_base, gkcrypt_key);
	munit_assert_memory_equal(sizeof(gkcrypt_iv), gkcrypt.iv, gkcrypt_iv);

	uint8_t key_stream_result[0x20];
	err = chiaki_gkcrypt_gen_key_stream(&gkcrypt, 0x30, key_stream_result, sizeof(key_stream_result));
	if(err != CHIAKI_ERR_SUCCESS)
	{
		chiaki_gkcrypt_fini(&gkcrypt);
		return MUNIT_ERROR;
	}

	uint8_t buf[0x25];

	memcpy(buf, clear_data, sizeof(buf));
	err = chiaki_gkcrypt_encrypt(&gkcrypt, 0x11, buf, sizeof(buf));
	if(err != CHIAKI_ERR_SUCCESS)
	{
		chiaki_gkcrypt_fini(&gkcrypt);
		return MUNIT_ERROR;
	}
	munit_assert_memory_equal(sizeof(buf), buf, enc_data);

	memcpy(buf, clear_data, sizeof(buf));
	err = chiaki_gkcrypt_decrypt(&gkcrypt, 0x11, buf, sizeof(buf));
	if(err != CHIAKI_ERR_SUCCESS)
	{
		chiaki_gkcrypt_fini(&gkcrypt);
		return MUNIT_ERROR;
	}
	munit_assert_memory_equal(sizeof(buf), buf, enc_data);

	chiaki_gkcrypt_fini(&gkcrypt);
	return MUNIT_OK;
}

static MunitResult test_gmac(const MunitParameter params[], void *user)
{
	static const uint8_t gkcrypt_key[] = {	0xb6, 0x4b, 0x1e, 0x65, 0x3f, 0xbb, 0xa7, 0xab, 0x80, 0xb3, 0x1e, 0x5a, 0x32, 0x4d, 0xec, 0xc0 };
	static const uint8_t gkcrypt_iv[] = {	0x7c, 0xc8, 0xd0, 0x19, 0x9e, 0xdf, 0xd8, 0xc3, 0xb5, 0x0f, 0x32, 0xee, 0x36, 0x33, 0x6a, 0x5a };

	static const uint8_t buf[] = {	0x03, 0x00, 0x18, 0x00, 0x19, 0x00, 0x02, 0x50, 0x21, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x69, 0xa0, 0x00, 0xf9, 0xf8, 0x98, 0x24, 0xda, 0x56, 0x0b, 0x74, 0xec, 0x68, 0x15, 0xd3, 0x1f,
									0x85, 0x43, 0xaa, 0xd3, 0xcd, 0xb8, 0x50, 0x2e, 0x90, 0xa4, 0xb0, 0x85, 0xf3, 0xbe, 0x9a, 0x47,
									0x97, 0xc1, 0xab, 0x5f, 0xa5, 0xfd, 0x62, 0x5c, 0x26, 0x1a, 0x54, 0x36, 0x72, 0x30, 0xc3, 0x2b,
									0xe0, 0x2c, 0x5a, 0x86, 0x07, 0xf5, 0xdc, 0x12, 0x55, 0xb4, 0x8e, 0x17, 0x5b, 0xe4, 0x7f, 0x96,
									0x55, 0x86, 0xf0, 0xfb, 0x45, 0xe2, 0xf8, 0xe8, 0x7d, 0x5f, 0x3a, 0x8b, 0xb0, 0x64, 0x6c, 0x54,
									0xf6, 0x8c, 0x4e, 0xa4, 0x12, 0xa3, 0x26, 0x7f, 0x22, 0x5b, 0x19, 0x98, 0xba, 0xd4, 0xad, 0x9f,
									0x5f, 0x63, 0xaf, 0x1f, 0x97, 0x8c, 0x17, 0x49, 0xa3, 0x30, 0x71, 0xd2, 0xb0, 0x49, 0x2c, 0x75,
									0xa2, 0x89, 0xda, 0x50, 0xfd, 0x9b, 0x08, 0x41, 0x98, 0x2d, 0x8c, 0x0b, 0x92, 0x79, 0xae, 0x60,
									0x23, 0x83, 0xef, 0x5c, 0xda, 0xce, 0x71, 0x08, 0x2d, 0x0b, 0x17, 0x62, 0x56, 0x48, 0xc5, 0xe2,
									0x0f, 0x22, 0x1d, 0xdc, 0xcc, 0xc2, 0x82, 0xa6, 0x8e, 0xec, 0x1e, 0x04, 0x0c, 0x08, 0xaa, 0xa2,
									0x80, 0x6c, 0x4f, 0xd3, 0x6f, 0xc1, 0x9b, 0x76, 0x9a, 0x58, 0xaa, 0x25, 0x56, 0xef, 0xf2, 0xfd,
									0x77, 0xf0, 0x32, 0x4e, 0x83, 0xe2, 0x30, 0x25, 0xa5, 0x5a, 0xf1, 0xdf, 0xc8, 0x75, 0xa2, 0xb6,
									0x2b, 0xfd, 0xac, 0xbc, 0x63, 0x09, 0xa1, 0xcc, 0x2b, 0xf5, 0x46, 0xef, 0x49, 0x83, 0x8f, 0x0b,
									0xc1, 0x47, 0xac, 0x5a, 0x00, 0x2e, 0xec, 0x1c, 0xd0, 0x0a, 0x66, 0xb5, 0xee, 0x9e, 0xad, 0xc9,
									0x64, 0x2d, 0xcb, 0x98, 0x0f, 0x43, 0xa6, 0xe1, 0x4c, 0xe7, 0x9f, 0x2d, 0xab, 0x94, 0x01, 0xd7,
									0x52, 0x84, 0x19 };

	static const size_t key_pos = 0x69a0;

	static const uint8_t gmac_expected[] = { 0x6f, 0x81, 0x10, 0x97 };

	ChiakiGKCrypt gkcrypt;
	memset(&gkcrypt, 0, sizeof(gkcrypt));

	memcpy(gkcrypt.key_gmac_current, gkcrypt_key, sizeof(gkcrypt.key_gmac_current));
	memcpy(gkcrypt.iv, gkcrypt_iv, sizeof(gkcrypt.iv));
	gkcrypt.key_buf = NULL;
	gkcrypt.key_buf_size = 0;
	gkcrypt.key_gmac_index_current = 0;

	uint8_t gmac[CHIAKI_GKCRYPT_GMAC_SIZE];
	ChiakiErrorCode err = chiaki_gkcrypt_gmac(&gkcrypt, key_pos, buf, sizeof(buf), gmac);
	if(err != CHIAKI_ERR_SUCCESS)
		return MUNIT_ERROR;

	munit_assert_memory_equal(sizeof(gmac), gmac, gmac_expected);

	return MUNIT_OK;
}

static MunitResult test_gen_gmac_key(const MunitParameter params[], void *user)
{
	static const uint8_t key_initial[] = {	0xbe, 0xeb, 0xa0, 0xf0, 0x3d, 0x05, 0x70, 0x7d, 0x3a, 0xc7, 0x3c, 0xd7, 0x32, 0xb9, 0x48, 0x01 };
	static const uint8_t iv[] = {			0xe8, 0x71, 0x87, 0xe7, 0x63, 0xe0, 0xdf, 0x46, 0x3d, 0xc2, 0x02, 0x4a, 0x2c, 0xd2, 0x9c, 0x45 };
	static const uint8_t key_result[] = {	0xe3, 0xdb, 0x92, 0xd9, 0xdd, 0xd3, 0x68, 0x99, 0xae, 0xfd, 0x9b, 0x15, 0xe1, 0xa6, 0x87, 0x8b };

	ChiakiGKCrypt gkcrypt;
	memset(&gkcrypt, 0, sizeof(gkcrypt));
	memcpy(gkcrypt.key_gmac_base, key_initial, sizeof(gkcrypt.key_gmac_base));
	memcpy(gkcrypt.iv, iv, sizeof(gkcrypt.iv));

	uint8_t key_out[CHIAKI_GKCRYPT_BLOCK_SIZE];
	chiaki_gkcrypt_gen_gmac_key(1, key_initial, iv, key_out);

	munit_assert_memory_equal(sizeof(key_out), key_out, key_result);

	return MUNIT_OK;
}


static MunitResult test_gmac_split(const MunitParameter params[], void *user)
{
	static const uint8_t gkcrypt_key[] = {	0x35, 0x95, 0x81, 0x87, 0xeb, 0xcb, 0x63, 0x54, 0x90, 0x00, 0xd7, 0xd5, 0xbd, 0xf4, 0xc5, 0xf4 };
	static const uint8_t gkcrypt_iv[] = {	0xb8, 0xde, 0xfc, 0xa4, 0x48, 0xf0, 0x4c, 0x3f, 0xd5, 0x94, 0x0e, 0x1c, 0x04, 0x64, 0x63, 0xb1 };

	static const uint8_t buf[] = { 0x2, 0x0, 0x1e, 0x0, 0x9, 0x0, 0x20, 0x8, 0x1, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xad,
								   0xf0, 0x2, 0x4e, 0x0, 0x2c, 0x6f, 0x3b, 0x6c, 0x2e, 0x4a, 0x27, 0x9c, 0x50, 0x34, 0x84, 0xea,
								   0x37, 0xc7, 0x76, 0xab, 0x92, 0x30, 0x43, 0x61, 0xc7, 0x1d, 0xcb, 0xc4, 0x35, 0x94, 0xcc, 0x44,
								   0xe2, 0xfa, 0x89, 0x86, 0x47, 0x5e, 0xc2, 0x1, 0x39, 0x4c, 0x9f, 0x67, 0xb3, 0x6e, 0x35, 0x44,
								   0x97, 0xf4, 0x28, 0xe2, 0x51, 0x12, 0xda, 0xf5, 0x2c, 0x43, 0x1c, 0x49, 0x46, 0x2c, 0x53, 0xe3,
								   0xe9, 0x76, 0x61, 0x87, 0x2b, 0x7a, 0xe3, 0x2f, 0xbb, 0x9f, 0x4d, 0x66, 0x8a, 0x31, 0xbc, 0x64,
								   0x9d, 0x6, 0xd, 0x4a, 0xdd, 0x42, 0x21, 0x73, 0x3c, 0x97, 0x21, 0x84, 0x53, 0x65, 0x53, 0x7a,
								   0xf4, 0x57, 0x77, 0x61, 0xae, 0xc, 0x7a, 0x42, 0x90, 0xfc, 0x6b, 0xbb, 0x38, 0x6e, 0x52, 0xb6,
								   0xe5, 0x26, 0xf, 0xd2, 0x2a, 0xf1, 0x56, 0x50, 0xdd, 0x65, 0x14, 0x9e, 0xe3, 0xb5, 0x70, 0x67,
								   0x55, 0x47, 0xcd, 0x8e, 0xc1, 0xba, 0xf6, 0x72, 0xb1, 0xb4, 0x11, 0x1b, 0x25, 0xd4, 0x3d, 0x19,
								   0xb5, 0xf5, 0xdd, 0x2f, 0xef, 0xd6, 0xd1, 0xaf, 0xb4, 0xdc, 0x93, 0xe5, 0x3c, 0xc3, 0xa4, 0x64,
								   0xb3, 0xad, 0xd3, 0xfc, 0x23, 0x84, 0x86, 0xf7, 0xf9, 0x10, 0xdf, 0x15, 0x6e, 0xc, 0x6e, 0x43,
								   0x9a, 0x63, 0x3b, 0xb5, 0x38, 0x5, 0xf9, 0x4a, 0x68, 0x4a, 0xfb, 0x95, 0x94, 0x9d, 0x31, 0xfd,
								   0x84, 0x31, 0x43, 0x85, 0xd, 0x8e, 0xbb, 0x66, 0xca, 0x4d, 0x3b, 0xa, 0x75, 0xe5, 0xeb, 0xde,
								   0xd5, 0xf8, 0x82, 0x85, 0xb2, 0xcc, 0xdb, 0x70, 0x9e, 0xa9, 0xe2, 0xb0, 0xac, 0x76, 0x1, 0x28,
								   0xea, 0x10, 0x39, 0xb7, 0x47, 0x78, 0x82, 0xc9, 0xa4, 0x24, 0x16, 0xc3, 0xe5, 0x16, 0x2, 0x46,
								   0x92, 0x7, 0xdb, 0x3a, 0xb0, 0xd0, 0xdd, 0xc0, 0x15, 0xdc, 0x5, 0xc7, 0x61, 0xe5, 0x7d, 0x82,
								   0xd0, 0x4c, 0x17, 0x44, 0xb4, 0xfa, 0x28, 0xd8, 0x94, 0xc1, 0xf4, 0xd0, 0xb3, 0x50, 0x6a, 0x25,
								   0x81, 0x98, 0x7b, 0x49, 0x67, 0xa2, 0x19, 0xb2, 0x41, 0xe5, 0xd2, 0xdf, 0x9a, 0x4f, 0xc9, 0x60,
								   0xff, 0xb2, 0x91, 0xe8, 0x5f, 0x77, 0x67, 0xff, 0xc0, 0xa1, 0x76, 0x57, 0xeb, 0x61, 0x34, 0x57,
								   0x80, 0x75, 0xdb, 0x3, 0xc5, 0x53, 0xf5, 0xb3, 0xa8, 0x61, 0xdf, 0xf, 0xb6, 0xe6, 0xc4, 0x39,
								   0xa0, 0xcd, 0xe9, 0xa6, 0x32, 0xb8, 0xba, 0x46, 0x9c, 0x43, 0x16, 0x49, 0x65, 0x9d, 0x71, 0x87,
								   0xf3, 0x37, 0xe5, 0xe0, 0xc6, 0xdf, 0x5e, 0x4c, 0xc5, 0x20, 0x9b, 0x16, 0xed, 0x13, 0x2d, 0x63,
								   0x79, 0x2b, 0xd0, 0x51, 0x44, 0x0, 0x44, 0x8c, 0x7c, 0x2c, 0xc6, 0x82, 0x20, 0xca, 0xca, 0x1f,
								   0x7f, 0xea, 0x71, 0xb7, 0x84, 0x5b, 0x8c, 0x94, 0x5b, 0x92, 0xb6, 0xef, 0xbe, 0x5f, 0x8, 0x3d,
								   0x5, 0xb2, 0xfb, 0xcb, 0x5a, 0x1, 0xd4, 0x16, 0x71, 0x8d, 0x9e, 0xf5, 0x8d, 0xae, 0x80, 0x40,
								   0x90, 0xfe, 0xef, 0x5d, 0xa4, 0x74, 0xf3, 0xe3, 0x4a, 0x9a, 0x67, 0x8b, 0xa6, 0xb5, 0x59, 0x18,
								   0x61, 0x4f, 0x42, 0x5f, 0x2b, 0x74, 0x39, 0xa8, 0x30, 0x5d, 0x8d, 0xf1, 0xad, 0x25, 0x4f, 0x41,
								   0xca, 0x27, 0xda, 0x7, 0x75, 0x49, 0x46, 0xd7, 0x81, 0xaa, 0x27, 0x61, 0x56, 0x9c, 0x29, 0x40,
								   0x90, 0x22, 0xa, 0xcd, 0x3a, 0x7b, 0xb4, 0x85, 0x96, 0x7, 0x80, 0x13, 0x12, 0x3f, 0x25, 0x27,
								   0xa0, 0xf6, 0x26, 0xbd, 0xd1, 0xb0, 0x95, 0xbd, 0x26, 0xba, 0x49, 0xa6, 0x81, 0x7, 0x31, 0xd3,
								   0xe1, 0x25, 0xda, 0xab, 0x71, 0xb9, 0x65, 0x1d, 0x13, 0xab, 0x82, 0x52, 0x95, 0xb3, 0x62, 0xa6,
								   0x48, 0x48, 0xb8, 0xd8, 0x55, 0x26, 0x98, 0xa1, 0xc9, 0x57, 0x66, 0x1c, 0xac, 0xb1, 0xc7, 0x5f,
								   0xfa, 0x44, 0xc6, 0xd8, 0x19, 0x1, 0x8d, 0x41, 0x1f, 0xc7, 0xf, 0x89, 0x1d, 0xef, 0x34, 0x9a,
								   0x68, 0x77, 0x8c };

	static const size_t key_pos = 0xadf0;

	static const uint8_t gmac_expected[] = { 0xd6, 0x6a, 0x07, 0xb7 };

	ChiakiGKCrypt gkcrypt;
	memset(&gkcrypt, 0, sizeof(gkcrypt));

	memcpy(gkcrypt.key_gmac_current, gkcrypt_key, sizeof(gkcrypt.key_gmac_current));
	memcpy(gkcrypt.iv, gkcrypt_iv, sizeof(gkcrypt.iv));
	gkcrypt.key_buf = NULL;
	gkcrypt.key_buf_size = 0;
	gkcrypt.key_gmac_index_current = 0;

	uint8_t gmac[CHIAKI_GKCRYPT_GMAC_SIZE];
	ChiakiErrorCode err = chiaki_gkcrypt_gmac(&gkcrypt, key_pos, buf, sizeof(buf), gmac);
	if(err != CHIAKI_ERR_SUCCESS)
		return MUNIT_ERROR;

	munit_assert_memory_equal(sizeof(gmac), gmac, gmac_expected);

	return MUNIT_OK;
}

static MunitResult test_gmac_multiple_of_key_refresh(const MunitParameter params[], void *user)
{
	static const uint8_t handshake_key[] = { 0x70, 0x58, 0x37, 0x50, 0x91, 0xea, 0xd1, 0x37, 0x71, 0x58, 0xec, 0xb3, 0xb, 0xea, 0x23, 0x87 };
	static const uint8_t ecdh_secret[] = { 0x3c, 0x3a, 0xf0, 0xec, 0xd6, 0x33, 0x1b, 0xb1, 0x6d, 0x24, 0x4f, 0x48, 0x19, 0xde, 0x6, 0x3d,
										0xc7, 0xe, 0xac, 0x95, 0x70, 0xac, 0x24, 0x92, 0x86, 0xa7, 0x24, 0xd0, 0x7a, 0x37, 0x55, 0x52 };

	static const uint8_t data[] = { 0x3, 0x11, 0xa4, 0x11, 0xa5, 00, 0x2, 0x50, 0x21, 0x5, 00, 00, 00, 00, 00, 0x6b,
									0x1d, 0xe0, 00, 0x83, 0xf1, 0xc4, 0x79, 0x71, 0xe4, 0x67, 0x7c, 0xcc, 0xb7, 0x92, 0x7, 0x8b,
									0xba, 0x8a, 0x8, 0xd6, 0x62, 0xf5, 0x7, 0x5d, 0x4a, 0x20, 0x14, 0xb3, 0xe7, 0xe, 0x5c, 0x77,
									0xfd, 0x6d, 0x60, 0xa3, 0xc5, 0x48, 0xa9, 0xc0, 0x61, 0xf9, 0x84, 0xf6, 0x37, 0x62, 0xac, 0xbc,
									0x1f, 0x5d, 0xe4, 0x40, 0x90, 0x54, 0x71, 0x37, 0xab, 0x33, 0x7e, 0xaa, 0xa5, 0xf4, 0xc4, 0x9,
									0x3, 0xf4, 0xb6, 0xa1, 0x2f, 0x2, 0x27, 0x53, 0xc6, 0xa5, 0xb1, 0x9c, 0x8f, 0xb4, 0x69, 0xd6,
									0x5c, 0x7f, 0x79, 0x5f, 0x14, 0x47, 0xcc, 0xa8, 0x8b, 0x37, 0xf7, 0x77, 0x6d, 0x1, 0xf4, 0x5b,
									0x59, 0xca, 0x96, 0x73, 0xa, 0x20, 0xf3, 0x7a, 0xd1, 0xd2, 0x7f, 0xfe, 0xb6, 0x66, 0xf4, 0xdb,
									0x10, 0xaf, 0xf3, 0x7, 0xee, 0x49, 0x8a, 0xa1, 0xa5, 0x73, 0xd8, 0x23, 0x83, 0xe, 0xc, 0xc,
									0x3a, 0x49, 0xdc, 0xd8, 0xc, 0xc8, 0xe4, 0xb6, 0xf2, 0x3c, 0x6d, 0x25, 0x6a, 0x7d, 0xa0, 0x1c,
									0x3a, 0x8e, 0xf, 0x3f, 0x25, 0x47, 0x11, 0x89, 0x67, 0xc4, 0xf9, 0xac, 0xc7, 0xe2, 0x93, 0x74,
									0xc6, 0xfb, 0xdd, 0x48, 0x57, 0xae, 0x53, 0xe2, 0xf4, 0x17, 0x80, 0x62, 0xf3, 0x9, 0xd8, 0xd4,
									0x7a, 0x46, 0x6b, 0x8, 0x76, 0x9c, 0x29, 0x81, 0x91, 0x4e, 0x57, 0xaf, 0x70, 0xb6, 0x32, 0x78,
									0x98, 0x9e, 0x85, 0xf2, 0xe3, 0x69, 0x15, 0xaf, 0xd4, 0x32, 0x54, 0xf, 0xd4, 0xa2, 0xfc, 0xf5,
									0x36, 0xdc, 0x6, 0x45, 0xf4, 0x6, 0xa3, 0x38, 0x21, 0x2, 0xfd, 0xf0, 0x51, 0x75, 0xea, 0xb1,
									0xf6, 0x8e, 0x39, 0x35, 0xfd, 0x75, 0x74, 0xad, 0xbc, 0x3e, 0xfb, 0xee, 0x7d, 0xf, 0x2d, 0x76,
									0xbe, 0xfd, 0xee };

	static const uint8_t crypt_index = 3;
	static const size_t key_pos = 0x6b1de0; // % CHIAKI_GKCRYPT_GMAC_KEY_REFRESH_KEY_POS == 0
	static const uint8_t gmac_expected[] = { 0x20, 0xcc, 0xa5, 0xf1 };

	ChiakiLog log;
	ChiakiGKCrypt gkcrypt;
	chiaki_gkcrypt_init(&gkcrypt, &log, 0, crypt_index, handshake_key, ecdh_secret);

	uint8_t gmac[CHIAKI_GKCRYPT_GMAC_SIZE];
	ChiakiErrorCode err = chiaki_gkcrypt_gmac(&gkcrypt, key_pos, data, sizeof(data), gmac);
	if(err != CHIAKI_ERR_SUCCESS)
		return MUNIT_ERROR;

	munit_assert_memory_equal(sizeof(gmac), gmac, gmac_expected);

	chiaki_gkcrypt_fini(&gkcrypt);

	return MUNIT_OK;
}


MunitTest tests_gkcrypt[] = {
	{
		"/ecdh",
		test_ecdh,
		NULL,
		NULL,
		MUNIT_TEST_OPTION_NONE,
		NULL
	},
	{
		"/key_stream",
		test_key_stream,
		NULL,
		NULL,
		MUNIT_TEST_OPTION_NONE,
		NULL
	},
	{
		"/en_decrypt",
		test_endecrypt,
		NULL,
		NULL,
		MUNIT_TEST_OPTION_NONE,
		NULL
	},
	{
		"/gmac",
		test_gmac,
		NULL,
		NULL,
		MUNIT_TEST_OPTION_NONE,
		NULL
	},
	{
		"/gen_gmac_key",
		test_gen_gmac_key,
		NULL,
		NULL,
		MUNIT_TEST_OPTION_NONE,
		NULL
	},
	{
		"/gmac_split",
		test_gmac_split,
		NULL,
		NULL,
		MUNIT_TEST_OPTION_NONE,
		NULL
	},
	{
		"/gmac_multiple_of_key_refresh",
		test_gmac_multiple_of_key_refresh,
		NULL,
		NULL,
		MUNIT_TEST_OPTION_NONE,
		NULL
	},
	{ NULL, NULL, NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL }
};