/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "cdw_config.h"
#include "gettext.h"
#include "cdw_logging.h"
#include "cdw_text_file_viewer.h"
#include "cdw_widgets.h"
#include "cdw_fs.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_utils.h"

#include "config_cdw.h"



/**
   \file cdw_logging.c
   \brief Implementation of logging for cdw

   File implements logging module: function that perform initialization of
   log file when application is started, functions that can be used to
   write to log file when application runs, function that closes log file
   and does necessary clean up before application exits.
*/




extern cdw_config_t global_config;

/* Default name of application log file. */
const char *cdw_log_file_name = "cdw.log";

/* Application's log file. */
static FILE *log_file = (FILE *) NULL;


static cdw_rv_t cdw_logging_search_for_default_full_path(void);
static cdw_rv_t cdw_logging_ask_for_dir_path(void);

static cdw_rv_t cdw_logging_check_available_space(void);
static void cdw_logging_path_reset_message(void);
static void cdw_logging_no_space_message(void);

static void cdw_logging_handle_old_log_file(void);



/**
   \brief Initialize logging module: find and open log file

   Try to open log file using file path set in configuration. If opening
   file with path set in configuration fails then try opening file with
   some default path. If all of above fails then ask user for path to
   given file.

   When correct full path to log file is known to this function, the
   function opens the file for reading and writing.

   If there is no place left on device for writing to the file, function
   closes old file and asks user for new path to log file.

   If after calling the function learns that current path is empty or invalid,
   it tries to silently set path to some default value. It does so silently,
   but in case of success user is informed about the fact of (re)setting
   the path to new value.

   If log file already existed then file is truncated to zero at opening.
   Correct full path to log file is stored in config.general.log_fullpath

   \return CDW_OK if path is set correctly
   \return CDW_NO if user cancels entering new path or user didn't enter correct path in n tries
*/
cdw_rv_t cdw_logging_module_init(void)
{
	log_file = (FILE *) NULL;

	cdw_logging_handle_old_log_file();

	if (!global_config.general.log_fullpath
	    || !strlen(global_config.general.log_fullpath)) {

		cdw_rv_t crv = cdw_logging_search_for_default_full_path();
		if (crv != CDW_OK) {
			return CDW_NO;
		}

		cdw_logging_path_reset_message();
	}

	/* At this point we should have a non-empty path. The code
	   below will attempt to use it. */

	int n = 5;
	for (int i = 0; i < n; i++) {

		errno = 0;
		log_file = fopen(global_config.general.log_fullpath, "w+");
		int e = errno;
		if (log_file) {
			cdw_rv_t crv = cdw_logging_check_available_space();
				if (crv == CDW_OK) {
					rewind(log_file);
					return CDW_OK;
				} else {
					fclose(log_file);
					log_file = (FILE *) NULL;

					cdw_logging_no_space_message();

					cdw_vdm ("ERROR: no space available for this log file: \"%s\"\n", global_config.general.log_fullpath);
					e = ENOSPC;
				}
		} else {
			cdw_vdm ("ERROR: failed to open log file: \"%s\"\n", strerror(e));
		}

		if (e) {
			cdw_fs_errno_handler(e);
		}

		if (i == n - 1) {
			return CDW_ERROR;
		}

		cdw_rv_t crv = cdw_logging_ask_for_dir_path();
		if (crv != CDW_OK) {
			return CDW_NO;
		}
	}

	return CDW_NO;
}





/**
   \brief Deallocate all resources opened by logging module

   Close log file, clean up all other logging resources.
 */
void cdw_logging_module_clean(void)
{
	if (log_file) {
		fflush(log_file);
		fclose(log_file);
		log_file = (FILE *) NULL;
	}

	/* Note that config.general.log_fullpath is free()d somewhere
	   else. */

	return;
}





/**
   \brief Print message to log file

   printf()-like function that writes given format string and all
   following arguments to log file.

   Function created to hide FILE *log_file from user and to provide
   convenient way of writing to log.

   Function calls fflush() for file.

   \param format - format string of message that you want to print to log
 */
void cdw_logging_write(const char *format, ...)
{
	cdw_assert (log_file, "ERROR: log file is not open when trying to write to it\n");

	va_list ap;
	va_start(ap, format);

	vfprintf(log_file, format, ap);
	fflush(log_file);

	va_end(ap);

	return;
}





/**
   \brief Write some separator between parts of log file

   Current implementation of log file appends more and more data for
   every task performed during one cdw session. This can make the log
   file very long and hard to browse. Inserting some separator between
   log file parts can make it easier to browse log file.
 */
void cdw_logging_write_separator(void)
{
	cdw_assert (log_file, "ERROR: log file is not open when trying to write to it\n");

	fprintf(log_file, "\n\n");
	/* 2TRANS: this is header printed in log file;
	   first %s is program name, second %s is program version */
	fprintf(log_file, _("   ::: Log file for %s %s :::   "), PACKAGE, VERSION);
	fprintf(log_file, "\n\n\n");

	return;
}





/**
   \brief Search for valid full path to log file in some default locations

   Search for correct full path to log file in user's home directory, and
   if this fails, in tmp directory. The tmp directory is tmp_dir set
   in cdw_graftpoints.c. Therefore you have to call cdw_graftpoints_init()
   before initializing logging module.

   The function is non-interactive: user is not informed about new path
   to log file.

   Log file name is CDW_LOG_FILE_NAME ("cdw.log"). If function succeeds
   then full path to log file is copied to config.general.log_fullpath. Otherwise
   config.general.log_fullpath is set to NULL.

   The function does not try to open the file, it just tries to set
   config.general.log_fullpath;

   \return CDW_ERROR if malloc() fails
   \return CDW_NO if function failed to set path to log file
   \return CDW_OK if path to file is set correctly.
*/
cdw_rv_t cdw_logging_search_for_default_full_path(void)
{
#if 0
	/* log file should be in user home dir by default */
	const char *dir_path = cdw_fs_get_home_dir_fullpath();
	if (!dir_path) {
		dir_path = cdw_fs_get_tmp_dir_fullpath();
		/* log_init() should be called after
		   tmp dir is set, but let's check just in case */
		if (!dir_path) {
			/* there was no HOME dir, and now it turned
			    out that there is no tmp dir :( */
			return CDW_NO;
		} else {
			; /* pass to concatenating full path */
		}

	} else {
		; /* pass to concatenating full path */
	}
#endif
	const char *dir = cdw_config_get_config_dir();
	char *tmp = cdw_string_concat(dir, cdw_log_file_name, (char *) NULL);
	if (!tmp) {
		return CDW_ERROR;
	} else {
		if (global_config.general.log_fullpath) {
			free(global_config.general.log_fullpath);
			global_config.general.log_fullpath = (char *) NULL;
		}
		global_config.general.log_fullpath = tmp;
		return CDW_OK;
	}
}





/**
   \brief Display information about resetting log file path

   This function displays dialog box informing about resetting full
   path to log file in cdw configuration. It should be used when
   during search for log file the log file path was reset from it's
   previous to new value, or when the path was set form empty path to
   non-empty path.

   Reason for using this function is that log file path is important
   parameter and user must be informed about any changes.

   The function was created to avoid shifting code too much to the
   right side ;)
 */
void cdw_logging_path_reset_message(void)
{
	/* 2TRANS: this is title of dialog window */
	cdw_buttons_dialog(_("Message"),
			   /* 2TRANS: this is message in dialog window */
			   _("Path to CDW log file has been reset. You may want to check current path in Configuration -> Log."),
			   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);

	return;
}





void cdw_logging_no_space_message(void)
{
	/* 2TRANS: this is title of dialog window */
	cdw_buttons_dialog(_("Error"),
		/* 2TRANS: this is message in dialog window */
			   _("No space left on current device for cdw log file. Select directory on another device."),
			   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

	return;
	}





/**
   \brief Display dialog window in which user can enter path to directory where log file will be put

   Display dialog box where user can enter new path to directory.

   If user entered new path, the function appends log file name to it
   and puts result to config.general.log_fullpath.  Otherwise
   config.general.log_fullpath is not modified in any way.

   \return CDW_OK if user entered valid path
   \return CDW_CANCEL if user canceled entering the path, or the path was invalid in n tries
   \return CDW_ERROR on malloc errors
*/
cdw_rv_t cdw_logging_ask_for_dir_path(void)
{
	char *buffer = strdup(cdw_fs_get_tmp_dir_fullpath());
	/* 2TRANS: this is title of dialog window */
	cdw_rv_t crv = cdw_file_picker(_("Configuring cdw..."),
				       _("Enter valid path to directory on non-empty device."),
				       &(buffer),
				       CDW_FS_DIR, R_OK | W_OK, CDW_FS_EXISTING);

	if (crv == CDW_OK) {
		char *tmp = cdw_string_concat(buffer, "/", cdw_log_file_name, (char *) NULL);
		free(buffer);
		buffer = (char *) NULL;
		if (!tmp) {
			return CDW_ERROR;
		} else {
			if (global_config.general.log_fullpath) {
				free(global_config.general.log_fullpath);
				global_config.general.log_fullpath = (char *) NULL;
			}
			global_config.general.log_fullpath = tmp;
			return CDW_OK;
		}
	} else {
		/* cdw_fs_ui_file_picker() does not allocate path if
		   user pressed ESCAPE, but it allocates memory for
		   empty string when user clears input field */
		if (buffer) {
			free(buffer);
			buffer = (char *) NULL;
		}
		return crv;
	}
}





/**
 * \brief Check if there is enough space on a device to store log file
 *
 * Try to write to log file some amount of data. If writing fails then return
 * error - caller will interpret this as not enough space on device.
 *
 * Global variable log_file must be valid (FILE *) pointer referring to
 * file open for writing.
 *
 * The function clears open file before returning.
 *
 * Current size of test data is 50k chars. When creating 2GB iso image
 * log file grows to ~7kB.
 *
 * \return CDW_ERROR on errors
 * \return CDW_OK if there is some space left on a device and using the file as log file is probably safe
 */
cdw_rv_t cdw_logging_check_available_space(void)
{
	bool failed = false;
	if (!log_file) {
		return CDW_ERROR;
	}

#define BUFFER_SIZE 1000
	char buffer[BUFFER_SIZE];
	for (int i = 0; i < BUFFER_SIZE; i++) {
		buffer[i] = 'a';
	}
	buffer[BUFFER_SIZE - 1] = '\0';

	for (int i = 0; i < 50; i++) {
		int a = fprintf(log_file, "%s\n", buffer);
		int b = fflush(log_file);
		if (a < 0) {
			failed = true;
			break;
		}
		if (b == EOF) {
			failed = true;
			break;
		}
	}

	freopen(global_config.general.log_fullpath, "w+", log_file); /* reset file size to zero */

	if (failed) {
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Display log file

   Wrapper using file_display() and local FILE *log_file variable to
   simplify displaying log file: callers don't have to use
   config.general.log_fullpath anymore.

   \p title may be empty string or NULL.

   \param title - title of window in which log file will be displayed
 */
cdw_rv_t cdw_logging_display_log(const char *title)
{
	rewind(log_file);
	cdw_rv_t crv = cdw_text_file_viewer(log_file, title);
	fseek(log_file, 0L, SEEK_END);

	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot show log file. Unknown error occurred."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Display log file if "show log after write" checkbox is checked

   Wrapper using file_display() and local FILE *log_file variable
   to simplify displaying log file: callers don't have to use
   config.general.log_fullpath anymore. "conditional" means that the log
   is displayed only if "show log after write" checkbox in configuration
   window is checked.

   \p title may be empty string or NULL.

   \param title - title of window in which log file will be displayed
*/
cdw_rv_t cdw_logging_display_log_conditional(const char *title)
{
	if (global_config.general.show_log) {
		return cdw_logging_display_log(title);
	} else {
		return CDW_OK;
	}
}





/**
   \brief Remove log file from old location

   cdw is migrating from log file in old location: ~/.cdw.log,
   into new location: ~/.cdw/cdw.log. This function removes the log
   file from old location.
*/
void cdw_logging_handle_old_log_file(void)
{
	const char *dir = cdw_fs_get_home_dir_fullpath();
	if (!dir) {
		return;
	}

	char *fullpath = cdw_string_concat(dir, ".cdw.log", (char *) NULL);
	if (!fullpath) {
		cdw_vdm ("ERROR: failed to concatenate path\n");
		return;
	}

	if (access(fullpath, F_OK)) {
		free(fullpath);
		fullpath = (char *) NULL;
		return;
	}


	FILE *file = fopen(fullpath, "r");
	if (!file) {
		cdw_vdm ("ERROR: failed to open file \"%s\"\n", fullpath);
		free(fullpath);
		fullpath = (char *) NULL;
		return;
	}
	bool found = false;
	for (int i = 0; i < 100; i++) {
		char *line = my_readline_10k(file);
		if (line) {
			char *ptr = strstr(line, "   ::: Log file for cdw");
			free(line);
			if (ptr) {
				found = true;
				break;
			}
		} else {
			/* EOF ? */
			break;
		}
	}

	fclose(file);
	file = (FILE *) NULL;
	if (!found) {
		; /* the file doesn't appear to be cdw log file, despite its name */
	} else {
		unlink(fullpath);
	}
	free(fullpath);
	fullpath = (char *) NULL;

	return;
}
