#ifndef H_CDW_CDIO
#define H_CDW_CDIO

#include <stdbool.h>

#include <cdio/cdio.h>
#include <cdio/cd_types.h>
#include <cdio/iso9660.h>

#include "config_cdw_undefine.h"

#include "main.h"
#include "cdw_ofs.h"


/* how many blocks of data (sectors) to read/write at a time */
#define CDW_CDIO_RW_N_SECTORS 20
#define CDW_CDIO_VOLUME_ID_LEN_MAX ISO_MAX_VOLUME_ID



/* CD, DVD or UNKNOWN - sometimes knowing this is enough */
typedef enum {
	CDW_DISC_SIMPLE_TYPE_UNKNOWN,
	CDW_DISC_SIMPLE_TYPE_CD,
	CDW_DISC_SIMPLE_TYPE_DVD,
} cdw_disc_simple_type_t;



typedef enum cdw_track_format {
	CDW_TRACK_BLACK_BOOK_UNKNOWN = 0,     /* no information collected yet, or track has unsupported format */
	CDW_TRACK_RED_BOOK_AUDIO,             /* all 2352 bytes are useful data */
	CDW_TRACK_YELLOW_BOOK_MODE1,          /* sync + header + 2048 bytes of useful data + some error magic */
	CDW_TRACK_YELLOW_BOOK_MODE2_FORMLESS, /* sync + header + subheader + 2336 bytes of useful data + some error magic */
	CDW_TRACK_YELLOW_BOOK_MODE2_FORM1,    /* sync + header + subheader + 2048 bytes of useful data + some error magic */
	CDW_TRACK_YELLOW_BOOK_MODE2_FORM2,    /* sync + header + subheader + 2324 bytes of useful data + some error magic */
	CDW_TRACK_MIXED,                      /* "I'm puzzled, but ioctl() returns CDS_MIXED, which means data + audio" */
	CDW_TRACK_DVD                         /* DVD disc: all DVD tracks have the same size of useful data in sector (2048) */
} cdw_track_format_t;



typedef struct cdw_track {
	track_format_t     cdio_track_format; /* AUDIO, DATA, XA, CDI, PSX, ERROR - this is track format provided by libcdio */
	cdw_track_format_t cdw_track_format;  /* this is track format defined in cdw, set by cdw code - it has more direct relation to size of useful data in sector */
	lsn_t    first_sector;
	lsn_t    last_sector;
	lsn_t    n_sectors;               /* (last_sector - first_sector + 1) this value may include padding zeros, which you may not want to read */
	lsn_t    n_sectors_to_read;       /* real number of sectors with real data, may be smaller than total number of sectors in track */
	uint16_t sector_size;             /* size of sectors in given track, using uint16_t type as in cdio library */
	struct {
		cdio_fs_t    cdio_fs;   /* cdio file system type */
		unsigned int size;      /* size in multiple of blocks (block size can be e.g. 2048 bytes) */
	} fs; /* file system */
} cdw_track_t;


typedef struct {
	CdIo_t *p_cdio;       /* main libcdio data structure */

	discmode_t mode;      /* disc mode as defined in cdio/disc.h */

	cdw_disc_simple_type_t simple_type; /* CD / DVD / UNKNOWN */
	char simple_type_label[9 + 1];

      	bool open;            /* is the disc opened already */
	bool blank;           /* is disc in drive blank? */

	track_t first_track;  /* first track on disc, most probably always equals to 1 */
	track_t last_track;   /* last track on disc */
	track_t n_tracks;     /* number of all user tracks of all kinds */
	track_t unknown_tracks; /* total amount of tracks of unknown type */

	cdw_track_t tracks[100]; /* CD disc has no more than 99 tracks, track numbering by cdio is from 1 */

	cdw_ofs_t *ofs;
} cdw_cdio_t;


/* functions initializing and cleaning up cdio module */
void cdw_cdio_init(void);
void cdw_cdio_clean(void);


cdw_cdio_t *cdw_cdio_new(void);
cdw_rv_t    cdw_cdio_get(cdw_cdio_t *disc, const char *drive_fullpath);
void        cdw_cdio_delete(cdw_cdio_t **disc);


bool cdw_cdio_is_audio_track(cdw_cdio_t *disc, track_t t);
bool cdw_cdio_is_data_track(cdw_cdio_t *disc, track_t t);


int  cdw_cdio_copy_track(cdw_cdio_t *disc, track_t track, int fd, long long int sectors);
void cdw_cdio_copy_print_debug_on_error(int error);


#endif /* H_CDW_CDIO */
