/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ensstat    ensrange        Ensemble range
      Ensstat    ensmin          Ensemble minimum
      Ensstat    ensmax          Ensemble maximum
      Ensstat    enssum          Ensemble sum
      Ensstat    ensmean         Ensemble mean
      Ensstat    ensavg          Ensemble average
      Ensstat    ensstd          Ensemble standard deviation
      Ensstat    ensstd1         Ensemble standard deviation
      Ensstat    ensvar          Ensemble variance
      Ensstat    ensvar1         Ensemble variance
      Ensstat    enspctl         Ensemble percentiles
*/

#include <cdi.h>

#include "cdo_rlimit.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "param_conversion.h"
#include "cdo_task.h"
#include "cdo_options.h"
#include "util_files.h"
#include "cimdOmp.h"

struct ens_file_t
{
  CdoStreamID streamID;
  int vlistID;
  size_t nmiss[2];
  double missval[2];
  Varray<double> array[2];
};

struct ensstat_arg_t
{
  int t;
  int varID[2];
  int levelID[2];
  int vlistID1;
  CdoStreamID streamID2;
  int nfiles;
  ens_file_t *efData;
  double *array2Data;
  double *count2Data;
  Field *fieldsData;
  int operfunc;
  double pn;
  bool lpctl;
  bool withCountData;
  int nvars;
};

static void *
ensstat_func(void *ensarg)
{
  if (Options::CDO_task) cdo_omp_set_num_threads(Threading::ompNumThreads);

  const auto arg = (ensstat_arg_t *) ensarg;
  const auto t = arg->t;
  const auto  nfiles = arg->nfiles;
  auto ef = arg->efData;
  auto fields = arg->fieldsData;
  auto array2 = arg->array2Data;
  auto count2 = arg->count2Data;
  const auto withCountData = arg->withCountData;

  bool lmiss = false;
  for (int fileID = 0; fileID < nfiles; fileID++)
    if (ef[fileID].nmiss[t] > 0) lmiss = true;

  const auto gridID = vlistInqVarGrid(arg->vlistID1, arg->varID[t]);
  const auto gridsize = gridInqSize(gridID);
  const auto missval = vlistInqVarMissval(arg->vlistID1, arg->varID[t]);

  size_t nmiss = 0;
#ifdef HAVE_OPENMP4
#pragma omp parallel for default(shared) reduction(+ : nmiss)
#endif
  for (size_t i = 0; i < gridsize; ++i)
    {
      const auto ompthID = cdo_omp_get_thread_num();

      auto &field = fields[ompthID];
      field.missval = missval;
      field.nmiss = 0;
      for (int fileID = 0; fileID < nfiles; fileID++)
        {
          field.vec_d[fileID] = ef[fileID].array[t][i];
          if (lmiss && DBL_IS_EQUAL(field.vec_d[fileID], ef[fileID].missval[t]))
            {
              field.vec_d[fileID] = missval;
              field.nmiss++;
            }
        }

      array2[i] = arg->lpctl ? fieldPctl(field, arg->pn) : fieldFunction(field, arg->operfunc);

      if (DBL_IS_EQUAL(array2[i], field.missval)) nmiss++;

      if (withCountData) count2[i] = nfiles - field.nmiss;
    }

  cdoDefRecord(arg->streamID2, arg->varID[t], arg->levelID[t]);
  cdoWriteRecord(arg->streamID2, array2, nmiss);

  if (withCountData)
    {
      cdoDefRecord(arg->streamID2, arg->varID[t] + arg->nvars, arg->levelID[t]);
      cdoWriteRecord(arg->streamID2, count2, 0);
    }

  return nullptr;
}

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("ensrange", func_range, 0, nullptr);
  cdoOperatorAdd("ensmin",   func_min,   0, nullptr);
  cdoOperatorAdd("ensmax",   func_max,   0, nullptr);
  cdoOperatorAdd("enssum",   func_sum,   0, nullptr);
  cdoOperatorAdd("ensmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("ensavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("ensstd",   func_std,   0, nullptr);
  cdoOperatorAdd("ensstd1",  func_std1,  0, nullptr);
  cdoOperatorAdd("ensvar",   func_var,   0, nullptr);
  cdoOperatorAdd("ensvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("enspctl",  func_pctl,  0, nullptr);
  cdoOperatorAdd("ensskew",  func_skew,  0, nullptr);
  cdoOperatorAdd("enskurt",  func_kurt,  0, nullptr);
  // clang-format on
}

void *
Ensstat(void *process)
{
  cdo::Task *task = Options::CDO_task ? new cdo::Task : nullptr;
  int nrecs0;

  cdoInitialize(process);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);

  const bool lpctl = operfunc == func_pctl;

  auto argc = operatorArgc();
  const auto nargc = argc;

  double pn = 0;
  if (operfunc == func_pctl)
    {
      operatorInputArg("percentile number");
      pn = parameter2double(cdoOperatorArgv(0));
      argc--;
    }

  bool withCountData = false;
  if (argc == 1)
    {
      if (cdoOperatorArgv(nargc - 1) == "count")
        withCountData = true;
      else
        cdoAbort("Unknown parameter: >%s<", cdoOperatorArgv(nargc - 1).c_str());
    }

  const auto nfiles = cdoStreamCnt() - 1;

  if (Options::cdoVerbose) cdoPrint("Ensemble over %d files.", nfiles);

  cdo::set_numfiles(nfiles + 8);

  const auto ofilename = cdoGetStreamName(nfiles);

  if (!Options::cdoOverwriteMode && fileExists(ofilename) && !userFileOverwrite(ofilename))
    cdoAbort("Outputfile %s already exists!", ofilename);

  std::vector<ens_file_t> ef(nfiles);

  FieldVector fields(Threading::ompNumThreads);
  for (int i = 0; i < Threading::ompNumThreads; i++) fields[i].resize(nfiles);

  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      ef[fileID].streamID = cdoOpenRead(fileID);
      ef[fileID].vlistID = cdoStreamInqVlist(ef[fileID].streamID);
    }

  // check that the contents is always the same
  for (int fileID = 1; fileID < nfiles; fileID++) vlistCompare(ef[0].vlistID, ef[fileID].vlistID, CMP_ALL);

  const auto vlistID1 = ef[0].vlistID;
  const auto vlistID2 = vlistDuplicate(vlistID1);
  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      ef[fileID].array[0].resize(gridsizemax);
      if (Options::CDO_task) ef[fileID].array[1].resize(gridsizemax);
    }

  Varray<double> array2(gridsizemax);

  const auto nvars = vlistNvars(vlistID2);
  Varray<double> count2;
  if (withCountData)
    {
      count2.resize(gridsizemax);
      for (int varID = 0; varID < nvars; ++varID)
        {
          char name[CDI_MAX_NAME];
          vlistInqVarName(vlistID2, varID, name);
          strcat(name, "_count");
          const auto gridID = vlistInqVarGrid(vlistID2, varID);
          const auto zaxisID = vlistInqVarZaxis(vlistID2, varID);
          const auto timetype = vlistInqVarTimetype(vlistID2, varID);
          const auto cvarID = vlistDefVar(vlistID2, gridID, zaxisID, timetype);
          cdiDefKeyString(vlistID2, cvarID, CDI_KEY_NAME, name);
          vlistDefVarDatatype(vlistID2, cvarID, CDI_DATATYPE_INT16);
          if (cvarID != (varID + nvars)) cdoAbort("Internal error, varIDs do not match!");
        }
    }

  const auto streamID2 = cdoOpenWrite(nfiles);
  cdoDefVlist(streamID2, vlistID2);

  ensstat_arg_t ensstat_arg;
  ensstat_arg.vlistID1 = vlistID1;
  ensstat_arg.streamID2 = streamID2;
  ensstat_arg.nfiles = nfiles;
  ensstat_arg.array2Data = array2.data();
  ensstat_arg.count2Data = count2.data();
  ensstat_arg.fieldsData = fields.data();
  ensstat_arg.operfunc = operfunc;
  ensstat_arg.pn = pn;
  ensstat_arg.lpctl = lpctl;
  ensstat_arg.withCountData = withCountData;
  ensstat_arg.nvars = nvars;
  ensstat_arg.t = 0;

  bool lwarning = false;
  bool lerror = false;
  int t = 0;
  int tsID = 0;
  do
    {
      nrecs0 = cdoStreamInqTimestep(ef[0].streamID, tsID);
      for (int fileID = 1; fileID < nfiles; fileID++)
        {
          const auto streamID = ef[fileID].streamID;
          const auto nrecs = cdoStreamInqTimestep(streamID, tsID);
          if (nrecs != nrecs0)
            {
              if (nrecs == 0)
                {
                  lwarning = true;
                  cdoWarning("Inconsistent ensemble file, too few time steps in %s!", cdoGetStreamName(fileID));
                }
              else if (nrecs0 == 0)
                {
                  lwarning = true;
                  cdoWarning("Inconsistent ensemble file, too few time steps in %s!", cdoGetStreamName(0));
                }
              else
                {
                  lerror = true;
                  cdoWarning("Inconsistent ensemble file, number of records at time step %d of %s and %s differ!", tsID + 1,
                             cdoGetStreamName(0), cdoGetStreamName(fileID));
                }
              goto CLEANUP;
            }
        }

      if (nrecs0 > 0)
        {
          taxisCopyTimestep(taxisID2, taxisID1);
          cdoDefTimestep(streamID2, tsID);
        }

      for (int recID = 0; recID < nrecs0; recID++)
        {
          int varID = -1, levelID = -1;

          for (int fileID = 0; fileID < nfiles; fileID++)
            {
              cdoInqRecord(ef[fileID].streamID, &varID, &levelID);
              ef[fileID].missval[t] = vlistInqVarMissval(ef[fileID].vlistID, varID);
            }
          //#pragma omp parallel for default(none) shared(ef, t)
          for (int fileID = 0; fileID < nfiles; ++fileID)
            {
              cdoReadRecord(ef[fileID].streamID, ef[fileID].array[t].data(), &ef[fileID].nmiss[t]);
            }

          ensstat_arg.efData = ef.data();
          ensstat_arg.varID[t] = varID;
          ensstat_arg.levelID[t] = levelID;
          if (Options::CDO_task)
            {
              task->start(ensstat_func, &ensstat_arg);
              task->wait();
              // t = !t;
            }
          else
            {
              ensstat_func(&ensstat_arg);
            }
        }

      tsID++;
    }
  while (nrecs0 > 0);

CLEANUP:

  if (lwarning) cdoWarning("Inconsistent ensemble, processed only the first %d timesteps!", tsID);
  if (lerror) cdoAbort("Inconsistent ensemble, processed only the first %d timesteps!", tsID);

  for (int fileID = 0; fileID < nfiles; fileID++) cdoStreamClose(ef[fileID].streamID);

  cdoStreamClose(streamID2);

  if (task) delete task;

  cdoFinish();

  return nullptr;
}
