/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2018 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Info       info            Dataset information
      Info       map             Dataset information and simple map
*/

#include <cfloat>

#include <cdi.h>

#include "cdo_int.h"
#include "pstream_int.h"
#include "text.h"
#include "array.h"
#include "datetime.h"
#include "printinfo.h"

static void
printMap(const int nlon, const int nlat, const double *array, const double missval, const double min, const double max)
{
  /* source code from PINGO */
  int ilon, ilat, i;
  double x, a, b;
  double level[10];
  int min_n, max_n;
  int bmin = 1, bmax = 1;
  unsigned char c;

  double step = (max - min) / 10;

  if (IS_NOT_EQUAL(step, 0))
    {
      a = pow(10, floor(log(step) / M_LN10));
      b = step / a;

      if (b > 5)
        b = 0.5 * ceil(b / 0.5);
      else if (b > 2)
        b = 0.2 * ceil(b / 0.2);
      else if (b > 1)
        b = 0.1 * ceil(b / 0.1);
      else
        b = 1;

      step = b * a;

      if (min < 0 && max > 0)
        {
          min_n = (int) floor(10 * (-min) / (max - min) - 0.5);
          max_n = (int) ceil(10 * (-min) / (max - min) - 0.5);
          level[min_n] = 0;
          for (i = min_n - 1; i >= 0; i--) level[i] = level[i + 1] - step;
          for (i = max_n; i < 9; i++) level[i] = level[i - 1] + step;
        }
      else
        {
          level[0] = step * ceil(min / step + 0.5);
          for (i = 1; i < 9; i++) level[i] = level[i - 1] + step;
        }
    }
  else
    for (i = 0; i < 9; i++) level[i] = min;

  fputc('\n', stdout);
  fflush(stdout);

  if (nlon >= 1000)
    {
      printf("%.*s", nlat < 10 ? 2 : nlat < 100 ? 3 : nlat < 1000 ? 4 : 5, "     ");
      for (ilon = 0; ilon < nlon; ilon++) printf("%d", ((ilon + 1) / 1000) % 10);
      putchar('\n');
      fflush(stdout);
    }

  if (nlon >= 100)
    {
      printf("%.*s", nlat < 10 ? 2 : nlat < 100 ? 3 : nlat < 1000 ? 4 : 5, "     ");
      for (ilon = 0; ilon < nlon; ilon++) printf("%d", ((ilon + 1) / 100) % 10);
      putchar('\n');
      fflush(stdout);
    }

  if (nlon >= 10)
    {
      printf("%.*s", nlat < 10 ? 2 : nlat < 100 ? 3 : nlat < 1000 ? 4 : 5, "     ");
      for (ilon = 0; ilon < nlon; ilon++) printf("%d", ((ilon + 1) / 10) % 10);
      putchar('\n');
      fflush(stdout);
    }

  printf("%.*s", nlat < 10 ? 2 : nlat < 100 ? 3 : nlat < 1000 ? 4 : 5, "     ");

  for (ilon = 0; ilon < nlon; ilon++) printf("%d", (ilon + 1) % 10);
  putchar('\n');
  fflush(stdout);
  putchar('\n');
  fflush(stdout);

  for (ilat = 0; ilat < nlat; ilat++)
    {
      printf("%0*d ", nlat < 10 ? 1 : nlat < 100 ? 2 : nlat < 1000 ? 3 : 4, ilat + 1);
      for (ilon = 0; ilon < nlon; ilon++)
        {
          x = array[ilat * nlon + ilon];
          if (DBL_IS_EQUAL(x, missval))
            c = '.';
          else if (DBL_IS_EQUAL(x, min) && !DBL_IS_EQUAL(min, max))
            c = 'm';
          else if (DBL_IS_EQUAL(x, max) && !DBL_IS_EQUAL(min, max))
            c = 'M';
          else if (DBL_IS_EQUAL(x, 0.))
            c = '*';
          else if (x < 0)
            {
              c = '9';
              for (i = 0; i < 9; i++)
                if (level[i] > x)
                  {
                    c = i + '0';
                    break;
                  }
            }
          else
            {
              c = '0';
              for (i = 8; i >= 0; i--)
                if (level[i] < x)
                  {
                    c = i + 1 + '0';
                    break;
                  }
            }

          if (c == '0') set_text_color(stdout, BRIGHT, BLUE);
          else if (c == '1') set_text_color(stdout, RESET, BLUE);
          else if (c == '2') set_text_color(stdout, BRIGHT, CYAN);
          else if (c == '3') set_text_color(stdout, RESET, CYAN);
          else if (c == '4') set_text_color(stdout, RESET, GREEN);
          else if (c == '5') set_text_color(stdout, RESET, YELLOW);
          else if (c == '6') set_text_color(stdout, RESET, RED);
          else if (c == '7') set_text_color(stdout, BRIGHT, RED);
          else if (c == '8') set_text_color(stdout, RESET, MAGENTA);
          else if (c == '9') set_text_color(stdout, BRIGHT, MAGENTA);
          else if (c == 'M')
            {
              set_text_color(stdout, bmax ? BLINK : RESET, BLACK);
              if (bmax) bmax = 0;
            }
          else if (c == 'm')
            {
              set_text_color(stdout, bmin ? BLINK : RESET, BLACK);
              if (bmin) bmin = 0;
            }
          putchar(c);
          reset_text_color(stdout);
        }
      printf(" %0*d\n", nlat < 10 ? 1 : nlat < 100 ? 2 : nlat < 1000 ? 3 : 4, ilat + 1);
      fflush(stdout);
    }
  putchar('\n');
  fflush(stdout);

  if (nlon >= 1000)
    {
      printf("%.*s", nlat < 10 ? 2 : nlat < 100 ? 3 : nlat < 1000 ? 4 : 5, "     ");
      for (ilon = 0; ilon < nlon; ilon++) printf("%d", ((ilon + 1) / 1000) % 10);
      putchar('\n');
      fflush(stdout);
    }

  if (nlon >= 100)
    {
      printf("%.*s", nlat < 10 ? 2 : nlat < 100 ? 3 : nlat < 1000 ? 4 : 5, "     ");
      for (ilon = 0; ilon < nlon; ilon++) printf("%d", ((ilon + 1) / 100) % 10);
      putchar('\n');
      fflush(stdout);
    }

  if (nlon >= 10)
    {
      printf("%.*s", nlat < 10 ? 2 : nlat < 100 ? 3 : nlat < 1000 ? 4 : 5, "     ");
      for (ilon = 0; ilon < nlon; ilon++) printf("%d", ((ilon + 1) / 10) % 10);
      putchar('\n');
      fflush(stdout);
    }

  printf("%.*s", nlat < 10 ? 2 : nlat < 100 ? 3 : nlat < 1000 ? 4 : 5, "     ");
  for (ilon = 0; ilon < nlon; ilon++) printf("%d", (ilon + 1) % 10);
  putchar('\n');
  fflush(stdout);
  putchar('\n');
  fflush(stdout);

  for (i = 0; i < 10; i++)
    {
      printf("%d=%c%+9.3e,%+9.3e%c%s", (int) i, i == 0 || level[i - 1] >= 0 ? '[' : '[', i == 0 ? min : level[i - 1],
             i == 9 ? max : level[i], i == 9 || level[i] <= 0 ? ']' : ']', i != 2 && i != 5 && i != 8 ? "  " : "");

      if (i == 2 || i == 5 || i == 8)
        {
          fputc('\n', stdout);
          fflush(stdout);
        }
    }

  printf("*=0  .=miss  m=min=%+9.3e  M=max=%+9.3e\n", min, max);
  fflush(stdout);
  putchar('\n');
  fflush(stdout);
}

struct Infostat
{
  double min, max, sum, sumi;
  size_t nvals, nmiss, nlevs;
};

static void
infostatInit(Infostat &infostat)
{
  infostat.nvals = 0;
  infostat.nmiss = 0;
  infostat.nlevs = 0;
  infostat.min = DBL_MAX;
  infostat.max = -DBL_MAX;
  infostat.sum = 0;
  infostat.sumi = 0;
}

void *
Info(void *process)
{
  enum
  {
    E_NAME,
    E_CODE,
    E_PARAM
  };
  int fpeRaised = 0;
  int varID, levelID;
  int nrecs;
  size_t nmiss;
  size_t imiss = 0;
  char varname[CDI_MAX_NAME];
  char paramstr[32];
  char vdatestr[32], vtimestr[32];

  cdoInitialize(process);

  // clang-format off
  const int INFO   = cdoOperatorAdd("info",   E_PARAM,  0, NULL);
  const int INFOP  = cdoOperatorAdd("infop",  E_PARAM,  0, NULL);
  const int INFON  = cdoOperatorAdd("infon",  E_NAME,   0, NULL);
  const int INFOC  = cdoOperatorAdd("infoc",  E_CODE,   0, NULL);
  const int XINFON = cdoOperatorAdd("xinfon", E_NAME,   0, NULL);
  const int MAP    = cdoOperatorAdd("map",    E_PARAM,  0, NULL);
  // clang-format on

  UNUSED(INFO);
  UNUSED(INFOP);
  UNUSED(INFON);
  UNUSED(INFOC);
  UNUSED(XINFON);

  const int operatorID = cdoOperatorID();

  const int operfunc = cdoOperatorF1(operatorID);

  DateTimeList dtlist;

  for (int indf = 0; indf < cdoStreamCnt(); indf++)
    {
      const int streamID = cdoStreamOpenRead(indf);
      const int vlistID = cdoStreamInqVlist(streamID);
      const int taxisID = vlistInqTaxis(vlistID);

      const int nvars = vlistNvars(vlistID);
      if (nvars == 0) continue;

      std::vector<Infostat> infostat(nvars);

      size_t gridsizemax = vlistGridsizeMax(vlistID);
      if (vlistNumber(vlistID) != CDI_REAL) gridsizemax *= 2;

      std::vector<float> arrayf;
      std::vector<double> array;
      if (CDO_Memtype == MEMTYPE_FLOAT)
        arrayf.resize(gridsizemax);
      else
        array.resize(gridsizemax);
        

      int indg = 0;
      int tsID = 0;
      while ((nrecs = cdoStreamInqTimestep(streamID, tsID)))
        {
          dtlist.taxisInqTimestep(taxisID, 0);
          const int64_t vdate = dtlist.getVdate(0);
          const int vtime = dtlist.getVtime(0);

          date2str(vdate, vdatestr, sizeof(vdatestr));
          time2str(vtime, vtimestr, sizeof(vtimestr));

          for (varID = 0; varID < nvars; ++varID) infostatInit(infostat[varID]);

          for (int recID = 0; recID < nrecs; ++recID)
            {
              if ((tsID == 0 && recID == 0) || operatorID == MAP)
                {
                  set_text_color(stdout, BRIGHT, BLACK);
                  fprintf(stdout,
                          "%6d :       Date     Time   %s Gridsize    Miss :     Minimum        Mean     Maximum : ", -(indf + 1),
                          operatorID == XINFON ? "Nlevs" : "Level");

                  if (operfunc == E_NAME)
                    fprintf(stdout, "Parameter name");
                  else if (operfunc == E_CODE)
                    fprintf(stdout, "Code number");
                  else
                    fprintf(stdout, "Parameter ID");

                  if (Options::cdoVerbose) fprintf(stdout, " : Extra");
                  reset_text_color(stdout);
                  fprintf(stdout, "\n");
                }

              pstreamInqRecord(streamID, &varID, &levelID);
              if (CDO_Memtype == MEMTYPE_FLOAT)
                pstreamReadRecordF(streamID, arrayf.data(), &nmiss);
              else
                pstreamReadRecord(streamID, array.data(), &nmiss);

              indg = (operatorID == XINFON) ? varID + 1 : indg + 1;

              const int param = vlistInqVarParam(vlistID, varID);
              const int code = vlistInqVarCode(vlistID, varID);
              const int gridID = vlistInqVarGrid(vlistID, varID);
              const int zaxisID = vlistInqVarZaxis(vlistID, varID);
              const int number = vlistInqVarNumber(vlistID, varID);
              const size_t gridsize = gridInqSize(gridID);
              const size_t nlevs = zaxisInqSize(zaxisID);
              const double level = cdoZaxisInqLevel(zaxisID, levelID);
              const double missval = vlistInqVarMissval(vlistID, varID);

              bool loutput = (operatorID != XINFON);

              if (loutput) infostatInit(infostat[varID]);

              Infostat &infostatr = infostat[varID];
              infostatr.nlevs += 1;
              infostatr.nmiss += nmiss;

              if (nlevs == infostatr.nlevs) loutput = true;

              if (loutput)
                {
                  cdiParamToString(param, paramstr, sizeof(paramstr));

                  if (operfunc == E_NAME) vlistInqVarName(vlistID, varID, varname);

                  set_text_color(stdout, BRIGHT, BLACK);
                  fprintf(stdout, "%6d ", indg);
                  reset_text_color(stdout);
                  set_text_color(stdout, RESET, BLACK);
                  fprintf(stdout, ":");
                  reset_text_color(stdout);

                  set_text_color(stdout, RESET, MAGENTA);
                  fprintf(stdout, "%s %s ", vdatestr, vtimestr);
                  reset_text_color(stdout);

                  set_text_color(stdout, RESET, GREEN);
                  if (operatorID == XINFON)
                    fprintf(stdout, "%7zu ", nlevs);
                  else
                    fprintf(stdout, "%7g ", level);

                  fprintf(stdout, "%8zu %7zu ", gridsize, infostatr.nmiss);

                  set_text_color(stdout, RESET, BLACK);
                  fprintf(stdout, ":");
                  reset_text_color(stdout);

                  set_text_color(stdout, RESET, BLUE);
                }

              if (number == CDI_REAL)
                {
                  fpeRaised = 0;

                  if (infostatr.nmiss > 0)
                    {
                      size_t nvals;
                      if (CDO_Memtype == MEMTYPE_FLOAT)
                        nvals = arrayMinMaxSumMV_f(gridsize, arrayf.data(), missval, &infostatr.min, &infostatr.max, &infostatr.sum);
                      else
                        nvals = arrayMinMaxSumMV(gridsize, array.data(), missval, &infostatr.min, &infostatr.max, &infostatr.sum);
                      imiss = gridsize - nvals;
                      infostatr.nvals += nvals;
                    }
                  else if (gridsize == 1)
                    {
                      const double val = (CDO_Memtype == MEMTYPE_FLOAT) ? arrayf[0] : array[0];
                      if (infostatr.nvals == 0)
                        infostatr.sum = val;
                      else
                        infostatr.sum += val;
                      infostatr.nvals += 1;
                    }
                  else
                    {
                      if (CDO_Memtype == MEMTYPE_FLOAT)
                        arrayMinMaxSum_f(gridsize, arrayf.data(), &infostatr.min, &infostatr.max, &infostatr.sum);
                      else
                        arrayMinMaxSum(gridsize, array.data(), &infostatr.min, &infostatr.max, &infostatr.sum);
                      infostatr.nvals += gridsize;
                    }

                  if (loutput)
                    {
                      if (infostatr.nvals)
                        {
                          if (infostatr.nvals == 1)
                            {
                              fprintf(stdout, "            %#12.5g            ", infostatr.sum);
                            }
                          else
                            {
                              const double mean = infostatr.sum / (double) infostatr.nvals;
                              fprintf(stdout, "%#12.5g%#12.5g%#12.5g", infostatr.min, mean, infostatr.max);
                            }
                        }
                      else
                        {
                          fprintf(stdout, "                     nan            ");
                        }
                    }
                }
              else
                {
                  size_t nvals = 0;
                  if (CDO_Memtype == MEMTYPE_FLOAT)
                    {
                      for (size_t i = 0; i < gridsize; i++)
                        {
                          if (!DBL_IS_EQUAL(arrayf[i * 2], missval) && !DBL_IS_EQUAL(arrayf[i * 2 + 1], missval))
                            {
                              infostatr.sum += arrayf[i * 2];
                              infostatr.sumi += arrayf[i * 2 + 1];
                              nvals++;
                            }
                        }
                    }
                  else
                    {
                      for (size_t i = 0; i < gridsize; i++)
                        {
                          if (!DBL_IS_EQUAL(array[i * 2], missval) && !DBL_IS_EQUAL(array[i * 2 + 1], missval))
                            {
                              infostatr.sum += array[i * 2];
                              infostatr.sumi += array[i * 2 + 1];
                              nvals++;
                            }
                        }
                    }
                  
                  fpeRaised = 0;

                  imiss = gridsize - nvals;
                  infostatr.nvals += nvals;

                  if (loutput)
                    {
                      const double arrmean_r = (infostatr.nvals > 0) ? infostatr.sum / infostatr.nvals : 0;
                      const double arrmean_i = (infostatr.nvals > 0) ? infostatr.sumi / infostatr.nvals : 0;
                      fprintf(stdout, "   -  (%#12.5g,%#12.5g)  -", arrmean_r, arrmean_i);
                    }
                }

              if (loutput)
                {
                  reset_text_color(stdout);

                  set_text_color(stdout, RESET, BLACK);
                  fprintf(stdout, " : ");
                  reset_text_color(stdout);

                  set_text_color(stdout, BRIGHT, GREEN);
                  if (operfunc == E_NAME)
                    fprintf(stdout, "%-14s", varname);
                  else if (operfunc == E_CODE)
                    fprintf(stdout, "%4d   ", code);
                  else
                    fprintf(stdout, "%-14s", paramstr);
                  reset_text_color(stdout);

                  if (Options::cdoVerbose)
                    {
                      char varextra[CDI_MAX_NAME];
                      vlistInqVarExtra(vlistID, varID, varextra);
                      fprintf(stdout, " : %s", varextra);
                    }

                  fprintf(stdout, "\n");
                }

              if (imiss != nmiss && nmiss > 0) cdoPrint("Found %zu of %zu missing values!", imiss, nmiss);

              if (fpeRaised > 0) cdoWarning("floating-point exception reported: %s!", fpe_errstr(fpeRaised));

              if (operatorID == MAP)
                {
                  const size_t nlon = gridInqXsize(gridID);
                  const size_t nlat = gridInqYsize(gridID);

                  if (gridInqType(gridID) == GRID_GAUSSIAN || gridInqType(gridID) == GRID_LONLAT
                      || gridInqType(gridID) == GRID_CURVILINEAR
                      || (gridInqType(gridID) == GRID_GENERIC && nlon * nlat == gridInqSize(gridID) && nlon < 1024))
                    {
                      printMap(nlon, nlat, array.data(), missval, infostatr.min, infostatr.max);
                    }
                }
            }

          tsID++;
        }

      cdoStreamClose(streamID);
    }

  cdoFinish();

  return 0;
}
